<?php
// Exit if accessed directly.
if (! defined('ABSPATH')) {
    exit; // Prevent direct access.
}

// Hook for General settings save.
add_action('admin_post_integrazo_wz_save_general_settings', 'integrazo_wz_save_general_settings');

function integrazo_wz_save_general_settings()
{

    // Verify nonce.
    if (
        ! isset($_POST['integrazo_wz_settings_nonce']) ||
        ! wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['integrazo_wz_settings_nonce'])), 'integrazo_wz_save_settings')
    ) {
        wp_die(esc_html__('Nonce verification failed.', 'integrazo-integration-for-zoho-apps'));
    }
    if (! current_user_can('manage_options')) {
        wp_die(esc_html__('Insufficient permissions.', 'integrazo-integration-for-zoho-apps'));
    }
    // Sanitize & update "Delete data" option.
    $delete_data_on_uninstall = isset($_POST['delete_data_on_uninstall']) ? '1' : '0';
    update_option('integrazo_wz_delete_data', $delete_data_on_uninstall);

    // Sanitize & update "Enable log cleanup" option.
    $enable_log_cleanup = isset($_POST['enable_log_cleanup']) ? '1' : '0';
    update_option('integrazo_wz_enable_log_cleanup', $enable_log_cleanup);

    // Sanitize & update "Log retention days" option.
    $log_retention_days = isset($_POST['log_retention_days']) ? absint($_POST['log_retention_days']) : 30;
    update_option('integrazo_wz_log_retention_days', $log_retention_days);

    // Add success message.
    set_transient('integrazo_wz_success', __('General settings saved successfully.', 'integrazo-integration-for-zoho-apps'), 30);

    // Redirect back to General subtab.
    $redirect_url = admin_url('admin.php?page=integrazo-integration-for-zoho-apps&tab=settings&subtab=general');
    wp_safe_redirect($redirect_url);
    exit;
}

function integrazo_wz_maybe_cleanup_logs()
{
    // Check if log cleanup is enabled.
    $is_cleanup_enabled = get_option('integrazo_wz_enable_log_cleanup', '0');

    if ($is_cleanup_enabled !== '1') {
        return;
    }

    // Get log retention days.
    $retention_days = absint(get_option('integrazo_wz_log_retention_days', 30));

    // Call the function to delete old logs.
    $logs_model = new INTEGRAZO_WZ_Logs_Model();
    $logs_model->delete_old_logs($retention_days);
}


add_action('admin_post_integrazo_wz_save_notifications_settings', 'integrazo_wz_save_notifications_settings');

function integrazo_wz_save_notifications_settings()
{
    // Nonce check
    if (
        ! isset($_POST['integrazo_wz_save_notifications_settings_nonce']) ||
        ! wp_verify_nonce(
            sanitize_text_field(wp_unslash($_POST['integrazo_wz_save_notifications_settings_nonce'])),
            'integrazo_wz_save_notifications_settings'
        )
    ) {
        wp_die(esc_html__('Nonce verification failed.', 'integrazo-integration-for-zoho-apps'));
    }
    if (! current_user_can('manage_options')) {
        wp_die(esc_html__('Insufficient permissions.', 'integrazo-integration-for-zoho-apps'));
    }
    // Checkbox: enabled or not
    $error_notification = isset($_POST['integrazo_wz_error_notification']) && $_POST['integrazo_wz_error_notification'] === 'yes' ? 'yes' : 'no';
    update_option('integrazo_wz_error_notification', $error_notification);

    // Get raw fields
    $error_notification_email   = isset($_POST['integrazo_wz_error_notification_email']) ? sanitize_email(wp_unslash($_POST['integrazo_wz_error_notification_email'])) : '';
    $error_notification_subject = isset($_POST['integrazo_wz_error_notification_subject']) ? sanitize_text_field(wp_unslash($_POST['integrazo_wz_error_notification_subject'])) : '';

    $valid = true;

    if ($error_notification === 'yes') {
        // BOTH must not be empty
        if ($error_notification_email === '' || $error_notification_subject === '') {
            $valid = false;
            set_transient('integrazo_wz_fail', __('Notification email and subject are required when error notifications are enabled.', 'integrazo-integration-for-zoho-apps'), 30);
        } else {
            // Validate emails, must be valid
            $emails = array_map('trim', explode(',', strtolower($error_notification_email)));
            foreach ($emails as $email) {
                if (!is_email($email)) {
                    $valid = false;
                    set_transient('integrazo_wz_fail', __('Invalid email(s). Please enter only valid, comma-separated email addresses.', 'integrazo-integration-for-zoho-apps'), 30);
                    break;
                }
            }
        }

        if ($valid) {
            // Save lowercase email(s) and subject
            update_option('integrazo_wz_error_notification_email', strtolower($error_notification_email));
            update_option('integrazo_wz_error_notification_subject', $error_notification_subject);
            set_transient('integrazo_wz_success', __('Notification settings saved successfully.', 'integrazo-integration-for-zoho-apps'), 30);
        }
    } else {
        // Notifications OFF — OK to clear fields
        update_option('integrazo_wz_error_notification_email', '');
        update_option('integrazo_wz_error_notification_subject', '');
        set_transient('integrazo_wz_success', __('Notification settings saved successfully.', 'integrazo-integration-for-zoho-apps'), 30);
    }

    // Redirect back
    $redirect_url = admin_url('admin.php?page=integrazo-integration-for-zoho-apps&tab=settings&subtab=notifications');
    wp_safe_redirect($redirect_url);
    exit;
}



function integrazo_wz_send_error_log_details_email_by_log_id($log_id)
{
    try {
        if (get_option('integrazo_wz_error_notification', '') !== "yes") return;

        $error_model = new INTEGRAZO_WZ_Logs_Model();
        $integration_model = new INTEGRAZO_WZ_Integration_Model();

        $log = $error_model->get_error_log($log_id);
        if (!$log) return;

        $integration = $integration_model->get_integration_by_id($log->integration_id);
        $integration_name = $integration['integration_name'] ?? 'N/A';

        $submitted_data = maybe_unserialize($log->submitted_data);
        $response_data = maybe_unserialize($log->response_data);
        $api_response = $response_data['data'][0] ?? [];

        $recipient_email = get_option('integrazo_wz_error_notification_email', '');
        $email_subject = get_option('integrazo_wz_error_notification_subject', 'Zoho Integration – Failed Sync Alert');
        if (empty($recipient_email)) return;

        ob_start(); ?>
        <html>

        <body style="font-family: Arial, sans-serif; font-size: 14px; color: #333; background-color: #fff; padding: 20px;">

            <h2 style="color: #444;">Integration Summary</h2>
            <table style="width: 100%; border-collapse: collapse; margin-bottom: 30px;">
                <tbody>
                    <tr>
                        <td style="width: 30%; padding: 8px; background: #f9f9f9; border: 1px solid #ddd;"><strong>Integration Name</strong></td>
                        <td style="padding: 8px; border: 1px solid #ddd;"><?php echo esc_html($integration_name); ?></td>
                    </tr>
                    <tr>
                        <td style="padding: 8px; background: #f9f9f9; border: 1px solid #ddd;"><strong>Error Type</strong></td>
                        <td style="padding: 8px; border: 1px solid #ddd;"><?php echo esc_html($log->category); ?></td>
                    </tr>
                    <tr>
                        <td style="padding: 8px; background: #f9f9f9; border: 1px solid #ddd;"><strong>Message</strong></td>
                        <td style="padding: 8px; border: 1px solid #ddd;"><?php echo esc_html($log->message); ?></td>
                    </tr>
                    <tr>
                        <td style="padding: 8px; background: #f9f9f9; border: 1px solid #ddd;"><strong>Created At</strong></td>
                        <td style="padding: 8px; border: 1px solid #ddd;"><?php echo esc_html($log->created_at); ?></td>
                    </tr>
                </tbody>
            </table>

            <h3 style="color: #444;">Captured Details</h3>
            <table style="width: 100%; border-collapse: collapse; border: 1px solid #ccc; margin-bottom: 30px;">
                <?php if (!empty($submitted_data) && is_array($submitted_data)) : ?>
                    <?php foreach ($submitted_data as $key => $val) : ?>
                        <tr>
                            <td style="width: 30%; padding: 8px; background: #f9f9f9; border: 1px solid #ddd;"><strong><?php echo esc_html($key); ?></strong></td>
                            <td style="padding: 8px; border: 1px solid #ddd;"><?php echo esc_html(is_array($val) ? implode(', ', $val) : $val); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php else : ?>
                    <tr>
                        <td colspan="2" style="padding: 8px;">No form data available.</td>
                    </tr>
                <?php endif; ?>
            </table>

            <h3 style="color: #444;">API Response</h3>
            <?php if (!empty($api_response)) : ?>
                <pre style="background: #f6f6f6; padding: 12px; border: 1px solid #ccc; border-radius: 4px; font-family: monospace; overflow-x: auto;">
<?php echo esc_html(wp_json_encode($api_response, JSON_PRETTY_PRINT)); ?>
                </pre>
            <?php else : ?>
                <p>No API response data available.</p>
            <?php endif; ?>
        </body>

        </html>
<?php
        $content = ob_get_clean();
        $headers = ['Content-Type: text/html; charset=UTF-8'];

        wp_mail($recipient_email, $email_subject, $content, $headers);
    } catch (Throwable $e) {
        //
    }
}
