<?php

/**
 * Plugin Name: Integration for CF7 and Mollie
 * Description: Integrates Mollie payment gateway with Contact Form 7 
 * Version: 1.1
 * Author: SRX Webdesign
 * Author URI: https://srxwebdesign.com/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires at least: 6.3
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) exit;

class CF7_Mollie_Integration
{

    public function __construct()
    {
        // Check if Contact Form 7 is active
        include_once(ABSPATH . 'wp-admin/includes/plugin.php');
        if (!is_plugin_active('contact-form-7/wp-contact-form-7.php')) {
            add_action('admin_notices', array($this, 'cf7_not_installed_notice'));
            deactivate_plugins(plugin_basename(__FILE__));
            return;
        }

        // Register settings page
        add_action('admin_menu', array($this, 'register_settings_page'));
        add_action('admin_init', array($this, 'register_plugin_settings'));

        // Uninstall hook
        register_uninstall_hook(__FILE__, 'cf7_mollie_uninstall');

        // Enqueue jQuery and custom script
        function cf7_mollie_enqueue_custom_scripts()
        {
            // Enqueue jQuery
            wp_enqueue_script('jquery');

            // Enqueue custom redirect script
            wp_enqueue_script('cf7-mollie-custom', plugins_url('/js/cf7-mollie-custom.js', __FILE__), array('jquery'), null, true);

            // Localize the script with nonce and AJAX URL
            wp_localize_script('cf7-mollie-custom', 'cf7MollieSettings', array(
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('cf7_mollie_nonce'),
            ));
        }
        add_action('wp_enqueue_scripts', 'cf7_mollie_enqueue_custom_scripts');

        //Ajax Request
        add_action('wp_ajax_cf7_mollie_generate_payment_url', array($this, 'cf7_mollie_generate_payment_url'));
        add_action('wp_ajax_nopriv_cf7_mollie_generate_payment_url', array($this, 'cf7_mollie_generate_payment_url'));
    }


    function cf7_mollie_generate_payment_url()
    {
        // Check the nonce for security
        check_ajax_referer('cf7_mollie_nonce', 'nonce');

        // Get the form ID and any other necessary fields from the request
        $form_id = isset($_POST['form_id']) ? sanitize_text_field(wp_unslash($_POST['form_id'])) : 0;

        $selected_form_ids = get_option('cf7_mollie_form_ids', []);
        $form_currencies = get_option('cf7_mollie_form_currencies', []);

        if (!in_array($form_id, $selected_form_ids)) {
            wp_send_json_error([
                'message' => 'not for mollie',
                'code' => 'not_cf7_mollie'
            ]);
            return;
        }
        // Get other fields from the form submission if needed
        // For example, let's assume we're receiving an 'email' and 'amount'
        $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
        $amount = isset($_POST['amount']) ? sanitize_text_field(wp_unslash($_POST['amount'])) : '';

        if ($email == '' || $amount == '') {
            wp_send_json_error([
                'message' => 'email or amount field required for mollie payment.',
                'code' => 'cf7_mollie'
            ]);
            return;
        }

        $currency = isset($form_currencies[$form_id]) ? $form_currencies[$form_id] : 'EUR';  // Set currency, or fetch it from settings

        // Get the API key and test mode from settings
        $api_key = get_option('cf7_mollie_api_key');
        $redirect_url = get_option('cf7_mollie_redirect_url');

        // Call Mollie API to generate the payment
        $mollie_payment_url = ''; // The URL will be populated after Mollie API call

        try {
            require_once 'vendor/autoload.php'; // Mollie API client
            $mollie = new \Mollie\Api\MollieApiClient();
            $mollie->setApiKey($api_key);

            $payment = $mollie->payments->create([
                "amount" => [
                    "currency" => $currency,
                    "value"    => number_format($amount, 2, '.', '') // Ensure it’s properly formatted
                ],
                "description" => "Payment by " . $email,
                "redirectUrl" => $redirect_url ? $redirect_url : site_url('/'), // Replace with a success page URL
                "metadata" => [
                    "form_id" => $form_id,
                    "email"   => $email,
                ],
            ]);

            // Get the Mollie payment URL
            $mollie_payment_url = $payment->getCheckoutUrl();
        } catch (Exception $e) {
            wp_send_json_error(['message' => $e->getMessage()]);
        }

        // Return the payment URL to the frontend
        wp_send_json_success(['payment_url' => $mollie_payment_url]);
    }

    // Notify the admin if CF7 is not installed
    public function cf7_not_installed_notice()
    {
        echo '<div class="error"><p><strong>Contact Form 7 Mollie Integration</strong> requires <strong>Contact Form 7</strong> to be installed and activated.</p></div>';
    }


    // Register the settings page in the admin menu
    public function register_settings_page()
    {
        add_options_page(
            'CF7 Mollie Integration Settings',
            'CF7 Mollie Settings',
            'manage_options',
            'cf7-mollie-settings',
            array($this, 'settings_page_html')
        );
    }

    // Display settings page HTML
    public function settings_page_html()
    {
?>
        <div class="wrap">
            <h1>Mollie Payment Settings</h1>
            <form method="post" action="options.php">
                <?php settings_fields('cf7_mollie_options_group'); ?>
                <?php do_settings_sections('cf7_mollie_options_group'); ?>

                <table class="form-table">
                    <tr valign="top">
                        <th scope="row">Mollie API Key</th>
                        <td><input type="text" name="cf7_mollie_api_key" value="<?php echo esc_attr(get_option('cf7_mollie_api_key')); ?>" /></td>
                    </tr>
                    <?php
                    // Fetch all CF7 forms
                    $forms = get_posts([
                        'post_type'   => 'wpcf7_contact_form',
                        'numberposts' => -1
                    ]);
                    ?>
                    <tr valign="top">
                        <th scope="row">Select CF7 Form IDs</th>
                        <td>
                            <?php if ($forms) { ?>
                                <select name="cf7_mollie_form_ids[]" multiple>
                                    <?php

                                    // Get the saved selected form IDs
                                    $selected_forms = get_option('cf7_mollie_form_ids', []);

                                    $selected = '';
                                    // Loop through each CF7 form and display as an option


                                    foreach ($forms as $form) {
                                        if ($selected_forms != null) {
                                            $selected = in_array($form->ID, $selected_forms) ? 'selected="selected"' : '';
                                        }

                                        echo '<option value="' . esc_attr($form->ID) . '" ' . esc_html($selected) . '>' . esc_html($form->post_title) . ' (ID: ' . esc_attr($form->ID) . ')</option>';
                                    }
                                    ?>
                                </select>
                                <p>Select multiple forms using Ctrl (Windows) or Command (Mac)</p>
                            <?php
                            } else {
                                echo "No forms found";
                            }
                            ?>
                        </td>
                    </tr>

                    <tr valign="top">
                        <th scope="row">Currencies for Selected Forms</th>
                        <td>
                            <?php if ($forms) { ?>
                                <?php
                                $form_currencies = get_option('cf7_mollie_form_currencies', []);

                                // Display currency input for each form
                                foreach ($forms as $form) {
                                    $currency = isset($form_currencies[$form->ID]) ? $form_currencies[$form->ID] : 'EUR';
                                    echo '<p><label for="currency_' . esc_attr($form->ID) . '">' . esc_html($form->post_title) . ' Currency:</label>';
                                    echo '<input type="text" name="cf7_mollie_form_currencies[' . esc_attr($form->ID) . ']" value="' . esc_attr($currency) . '" /></p>';
                                }
                                ?>
                                <p>Enter the currency code (e.g., EUR, USD, GBP, AUD, CAD, CZK, DKK, HUF, NOK, PLN, SEK, CHF, USD) for each form.</p>
                            <?php
                            } else {
                                echo "No forms found";
                            }
                            ?>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Redirect URL</th>
                        <td><input type="text" name="cf7_mollie_redirect_url" value="<?php echo esc_attr(get_option('cf7_mollie_redirect_url')); ?>" /></td>
                    </tr>

                </table>

                <?php submit_button(); ?>
            </form>
        </div>
<?php
    }
    function register_plugin_settings()
    {
        register_setting('cf7_mollie_options_group', 'cf7_mollie_api_key');
        register_setting('cf7_mollie_options_group', 'cf7_mollie_form_ids'); // Store selected form IDs
        register_setting('cf7_mollie_options_group', 'cf7_mollie_form_currencies'); // Store currencies per form
        register_setting('cf7_mollie_options_group', 'cf7_mollie_redirect_url');
    }
}

// Register plugin settings

// Uninstall function to drop table if the user selects that option
function cf7_mollie_uninstall()
{

    // Optionally, remove stored options
    delete_option('cf7_mollie_api_key');
    delete_option('cf7_mollie_form_ids');
    delete_option('cf7_mollie_redirect_url');
    delete_option('cf7_mollie_form_currencies');
}

new CF7_Mollie_Integration();
