<?php

if ( ! defined( 'ABSPATH' ) ) exit;

GFForms::include_feed_addon_framework();

class OYOIPGF_GFSImpleFeedAddonPipeDrive extends GFFeedAddOn {
    protected $_version = INTEGRATE_PIPEDRIVE_AND_GRAVITY_FORMS;
    protected $_min_gravityforms_version = '1.9';
    protected $_slug = 'integrate-pipedrive-and-gravity-forms';
    protected $_path = 'integrate-pipedrive-and-gravity-forms/integrate-pipedrive-and-gravity-forms.php';
    protected $_full_path = __FILE__;
    protected $_title = 'Integrate Pipedrive and Gravity Forms';
    protected $_short_title = 'Pipedrive';
    protected $_multiple_feeds = true;
    private static $_instance = null;

	/**
	 * Get an instance of this class.
	 */
	public static function get_instance() {
		if ( self::$_instance == null ) {
			self::$_instance = new OYOIPGF_GFSImpleFeedAddonPipeDrive();
		}
		return self::$_instance;
    }
    
    /**
	 * Plugin starting point.
	 */
	public function init() {
		parent::init();
    }

    /**
	 * Register needed styles.
	 */
	public function styles() {
		$styles = array(
			array(
				'handle'  => $this->_slug . '_form_settings',
				'src'     => $this->get_base_url() . "/css/form_settings.css",
				'version' => $this->_version,
				'enqueue' => array(
					array( 'admin_page' => array( 'form_settings' ) ),
				),
			),
		);
		return array_merge( parent::styles(), $styles );
    }

    /**
	 * Configures the settings which should be rendered on the add-on settings tab.
	 */
	public function plugin_settings_fields() {
		return array(
            array(
                'fields' => array(
                    array(
                        'name'              => 'api_key',
                        'label'             => esc_html__('Pipedrive API Key', 'integrate-pipedrive-and-gravity-forms'),
                        'type'              => 'text',
                        'class'             => 'medium',
                        'feedback_callback' => array($this, 'is_api_valid_callback'),
                    )
                ),
            ),
        );
    }

	/**
	 * Configures the settings which should be rendered on the feed edit page.
	 */
    public function feed_settings_fields() {        
		return array(
            array(
                'title'         => esc_html__('Pipedrive Feed Settings', 'integrate-pipedrive-and-gravity-forms'),
                'fields'        => array(
                    array(
                        'label'         => esc_html__('Feed Name', 'integrate-pipedrive-and-gravity-forms'),
                        'type'          => 'text',
                        'name'          => 'feed_name',
                        'placeholder'   => '',
                        'required'      => 1,
                        'class'         => 'medium',
                        'tooltip'       => esc_html__('Enter a feed name to uniquely identify this setup.', 'integrate-pipedrive-and-gravity-forms'),
                    ),
                    array(
                        'label'   => esc_html__('Owner', 'integrate-pipedrive-and-gravity-forms'),
                        'name'    => 'owner',
                        'type'    => 'select',
                        'tooltip' => esc_html__('Select the owner of this lead from Pipedrive.', 'integrate-pipedrive-and-gravity-forms'),
                        'choices' => array_map(function($owner) {
                            return array(
                                'label' => $owner['name'],
                                'value' => $owner['id'],
                            );
                        }, $this->get_owner()),
                    ),
                ),
            ),
            array(
				'fields'     => array(
					array(
						'name'      => 'mappedFields',
                        'label'     => esc_html__( 'Map Fields', 'integrate-pipedrive-and-gravity-forms' ),
                        'type'      => 'field_map',
                        'tooltip'   => 'Associate your Pipedrive fields to the appropriate Gravity Form fields.',
						'field_map' => array(
							array(
								'name'          => 'email',
								'label'         => esc_html__( 'Email', 'integrate-pipedrive-and-gravity-forms' ),
								'required'      => 1,
								'field_type'    => array( 'email', 'hidden' )
							),
							array(
								'name'          => 'firstName',
								'label'         => esc_html__( 'First Name', 'integrate-pipedrive-and-gravity-forms' ),
								'required'      => 0,
                            ),
                            array(
								'name'          => 'lastName',
								'label'         => esc_html__( 'Last Name', 'integrate-pipedrive-and-gravity-forms' ),
								'required'      => 0,
							),
							array(
								'name'          => 'phone',
								'label'         => esc_html__( 'Phone', 'integrate-pipedrive-and-gravity-forms' ),
								'required'      => 0,
								'field_type'    => 'phone',
                            ),
                            array(
								'name'          => 'comments',
								'label'         => esc_html__( 'Comments', 'integrate-pipedrive-and-gravity-forms' ),
								'required'      => 0,
                            ),
						),
                    ),
                ),
            ),
        );
    }

    /**
	 * Initialize API.
	 */
    public function initialize_api() {
        // Send a request to verify credentials.
        $results = $this->pipedrive_get( 'leads?limit=1' );

        $api_response = $results['response'];

        if( $results['response']['code'] !== 200 ) {
            // If the API response code is not 200, return false.
            return false;
        }

        // If the API Key is valid, return true.
        return true;
    }

    /**
	 * Check if the API key is valid.
	 */
    public function is_api_valid_callback() {
        if( !$this->initialize_api() ) {
            echo "<p style='color: red; margin-bottom: 1rem;'><em>Invalid API Key</em></p>";
            return false;
        }

        return true;
    }

    /**
	 * Prevent feeds being listed or created if the API credentials are not valid.
	 */
	public function can_create_feed() {
        return $this->initialize_api();
	}
    
    /**
	 * Configures which columns should be displayed on the feed list page.
	 */
	public function feed_list_columns() {        
		return array(
			'feed_name' => esc_html__( 'Feed Name', 'integrate-pipedrive-and-gravity-forms' ),
		);
    }

    /**
	 * Get users from pipedrive and allow them to be selected as owners.
	 */
    public function get_owner() {
        $response = $this->pipedrive_get( 'users' );
        $response_body = json_decode($response['body'], true);
        $owners = $response_body['data'];

        $new_owners = [];

        if (is_array($owners)) {
            foreach ($owners as $owner) {
            if (!empty($owner['active_flag']) && $owner['active_flag'] === true) {
                $new_owners[] = array(
                'id'   => $owner['id'],
                'name' => $owner['name'],
                );
            }
            }
        }

        return $new_owners;
    }

    /**
	 * Check for existing contact in pipedrive. 
	 */
    public function check_for_existing_contact( $email ) {
        if( $email ) {
            $params     = "/persons/search?term={$email}&exact_match=true";
            $response   = $this->pipedrive_get( $params );
            $body       = json_decode($response['body'], true);

            if (!empty($body['data']['items'])) {
                if( $body['data']['items'][0]['item']['id'] ) {
                    return $body['data']['items'][0]['item']['id'];
                }
            }

            return false;
        }
    }

    /**
	 * Process the feed - Create lead and add to list
	 */
	public function process_feed( $feed, $entry, $form ) {
        // If API is not initialized, return false.
		if ( ! $this->initialize_api() ) {
            return false;
        }

        // Don't process the lead if it has a status of spam
        if ( $entry[ 'status' ] === 'spam' ){
            return false;
        }

        $field_map = $this->get_field_map_fields( $feed, 'mappedFields' );

		// Loop through the fields from the field map setting building an array of values to be passed to Pipedrive.
        $merge_vars = array();
        $comments_str = array();

        foreach ( $field_map as $name => $field_id ) {
            // Get the field value for the specified field id.
            $merge_vars[ $name ] = $this->get_field_value( $form, $entry, $field_id );
            if ($name === 'comments') {
                $comments_str = $merge_vars[$name];
                unset($merge_vars[$name]);
            }
        }

        // Combine firstName and lastName into a single name field.
        $merge_vars['name'] = trim(
            (isset($merge_vars['firstName']) ? $merge_vars['firstName'] : '') . ' ' .
            (isset($merge_vars['lastName']) ? $merge_vars['lastName'] : '')
        );

        // Remove individual firstName and lastName fields from the array.
        unset($merge_vars['firstName'], $merge_vars['lastName']);

        // Get the selected owner value from the feed settings.
        $owner_id = rgar($feed['meta'], 'owner');

        if( $owner_id ) {
            // Add the owner ID to the merge_vars array.
            $merge_vars['owner_id'] = $owner_id;
        }
    
        // Add the date created to the merge_vars array.
        $merge_vars['add_time'] = gmdate('Y-m-d H:i:s', strtotime(rgar($entry, 'date_created')));

        // Check to see if person exists in pipedrive, return the ID if true.
        $person_exists = $this->check_for_existing_contact( $merge_vars['email'] );

        if( !$person_exists ) {
            $person_post  = $this->pipedrive_post( $merge_vars, 'persons' );
            $person_response_body = json_decode($person_post['body'], true);
        }

        if( isset( $person_response_body['success'] ) && isset( $person_response_body['data']['id'] ) ) {
            $lead_data = [
                'person_id' => $person_response_body['data']['id'],
                'title'     => $merge_vars['name'] . ' Lead',
                'owner_id'  => intval($owner_id),
            ];
            $lead_post = $this->pipedrive_post($lead_data, 'leads');
        }

        if( $person_exists ) {
            $lead_data = [
                'person_id' => $person_exists,
                'title'     => $merge_vars['name'] . ' Lead',
                'owner_id'  => intval($owner_id),
            ];
            $lead_post = $this->pipedrive_post($lead_data, 'leads');
        }

        if( $comments_str && $lead_post ) {
            $lead_post = json_decode($lead_post['body'], true);
            $form_name = isset($form['title']) ? $form['title'] : 'Unknown Form';
            $note = "<strong>Lead created from form:</strong><br>{$form_name}<br><strong>Form comments:</strong><br>{$comments_str}";
            $comment_data = [
                'content'   => $note,
                'lead_id'   => $lead_post['data']['id'],
            ];
            $comments_post = $this->pipedrive_post($comment_data, 'notes');
        }
    }

    /**
	 * Handle all API GET requests to Pipedrive.
	 */
    public function pipedrive_get( $params = '' ) {
        $api_key     = $this->get_plugin_setting( 'api_key' );         
        $url         = "https://api.pipedrive.com/v1/" . $params;

        $args = array(
            'headers'   => array(
                'Content-Type'  => 'application/json',
                'x-api-token'   => $api_key,
            ),
        );
        
        $pipedrive_result = wp_remote_get( $url, $args );
        return $pipedrive_result;
    }

    /**
	 * Handle all API POST requests to Pipedrive.
	 */
    public function pipedrive_post( $data = null, $params = '' ) {
        $api_key     = $this->get_plugin_setting( 'api_key' );                                                                      
        $url         = "https://api.pipedrive.com/v1/" . $params;

        $args = array(
            'body'      => json_encode($data),
            'headers'   => array(
                'Content-Type'  => 'application/json',
                'x-api-token'   => $api_key,
            ),
        );

        $pipedrive_result = wp_remote_post( $url, $args );
        return $pipedrive_result;
    }
}