<?php
if ( ! defined( 'ABSPATH' ) ) exit;
class IntegratePalace_Properties_Webhooks {

    public static function init() {
        add_action('rest_api_init', array(__CLASS__, 'register_routes'));
    }

    public static function register_routes() {
        $blog_id = get_option("integrate_palace-palace-unique-id");
        $webhook_url = "/webhook/$blog_id";
        register_rest_route('integrate-palace-properties/v1', $webhook_url, array(
            'methods' => 'POST',
            'callback' => array(__CLASS__, 'handle_webhook'),
            'permission_callback' => '__return_true',
        ));
    }

    private static function get_property_id_by_code($property_code) {
        // Try to get the post ID from cache first
        $cache_key = 'property_id_' . $property_code;
        $post_id = wp_cache_get($cache_key, 'property_cache');

        if ($post_id === false) {
            global $wpdb;
			$args = [
				'post_type'  => 'post', // Change to your custom post type if needed
				'meta_key'   => 'information_property_code',
				'meta_value' => $property_code,
				'fields'     => 'ids',
				'posts_per_page' => 1,
			];
			
			$post_id = get_posts($args);
			$post_id = !empty($post_id) ? $post_id[0] : null;
            if($post_id)  wp_cache_set($cache_key, $post_id, 'property_cache', 3600);
        }

        return $post_id;
    }

    private static function delete_property($property_code){
        global $wpdb;
        $property_code = sanitize_text_field($property_code);
        if(!$property_code) return new WP_REST_Response(['status' => 'error', 'message' => 'Property code does not exist'], 404);
        
        $post_id = self::get_property_id_by_code($property_code);
        
        if ($post_id && wp_delete_post($post_id, false)) {
            // Get gallery field values (image IDs)
            $gallery_images = get_field('gallery', $post_id);

            if (!empty($gallery_images)) {
                // Delete associated images
                foreach ($gallery_images as $image_id) {
                    wp_delete_attachment($image_id['ID'], true);
                    $file_path = get_attached_file($image_id['ID']);
                    if (file_exists($file_path)) {
                        wp_delete_file($file_path);
                    }
                }
            }
            // Delete post thumbnail
            delete_post_thumbnail($post_id);
            return new WP_REST_Response(['status' => 'success', 'message' => 'Property deleted successfully'], 200);
        } else {
            return new WP_REST_Response(['status' => 'error', 'message' => $post_id ? 'Failed to delete property' : 'Property not found'], $post_id ? 500 : 404);
        }
    }

    public static function handle_webhook(WP_REST_Request $request) {
        global $wpdb;

        $property = $request->get_json_params();

        if (!empty($property['PropertyCode']) && isset($property['action']) && $property['action'] == "delete") {
            return self::delete_property($property['PropertyCode']);
        }

        // Ensure data contains the required fields
        if (!isset($property['PropertyCode'])) {
            return new WP_REST_Response(array('status' => 'error', 'message' => 'Missing required fields'), 400);
        }

        $property = apply_filters('before_insert_palace_property', $property);
        $show_on_website = apply_filters('show_on_website_palace_property', true, $property);
        if (!$show_on_website) {
            return self::delete_property($property['PropertyCode']);
        }

        $property_code = sanitize_text_field($property['PropertyCode']);
        $post_title = $property['PropertyFeatures']['PropertyHeader'] ?: 
            ($property['PropertyAddress3'] . ", " . $property['PropertyAddress1'] . " " . $property['PropertyAddress2']);
        $post_content = $property['PropertyFeatures']['PropertyAdvertText'] ?: '';

        // Check if the property already exists
        $post_id = self::get_property_id_by_code($property_code);

        if (!empty($post_id)) {
            // Update the existing property
            $post_data = array(
                'ID' => $post_id,
                'post_title' => $post_title,
                'post_content' => $post_content,
                'post_status' => 'publish',
                'post_type' => 'palace-properties',
            );
            $post_id = wp_update_post($post_data);
        } else {
            // Create a new property
            $post_data = array(
                'post_title' => $post_title,
                'post_content' => $post_content,
                'post_status' => 'publish',
                'post_type' => 'palace-properties',
            );
            $post_id = wp_insert_post($post_data);
        }

        // Update custom fields
        if ($post_id) {
            // Prepare data
            $property_address = $property['PropertyAddress1'] . ', ' . $property['PropertyAddress2'] . ', ' . $property['PropertyAddress3'];
            $property_suburb = $property['PropertySuburb'][0]['PropertySuburbName'] ?: '';
            $property_region = $property['PropertySuburb'][0]['PropertySuburbRegionOrState'] ?: 'Auckland';
            $property_rent = $property['PropertyRentAmount'];
            $property_date = $property['PropertyDateAvailable'];
            $property_agent_code = $property['PropertyAgent']['PropertyAgentCode'];
            $property_bedrooms = $property['PropertyFeatures']['PropertyBedroomsNo'] ?: '';
            $property_bathrooms = $property['PropertyFeatures']['PropertyBathroomsNo'] ?: '';
            $property_cars = $property['PropertyFeatures']['PropertyCarsNo'] ?: '';
            $property_pets = $property['PropertyFeatures']['PropertyPetsAllowed'] == 'Yes' ? 1 : 0;
            $property_parking = $property['PropertyFeatures']['PropertyParking'] ?: '';
            $property_year_built = $property['PropertyFeatures']['PropertyYearBuilt'] ?: '';
            $property_class = $property['PropertyFeatures']['PropertyClass'] ?: '';
            $property_smokers_allowed = $property['PropertyFeatures']['PropertySmokersAllowed'] == 'Yes' ? 1 : 0;
            $property_virtual_tour_url = $property['PropertyFeatures']['PropertyVirtualTourURL'] ?: '';
            $property_geographic_location = $property['PropertyFeatures']['PropertyGeographicLocation'] ?: '';

            update_post_meta($post_id, 'information_property_code', $property_code);
            if (!empty($property['PropertyFeatures']['PropertyLandAreaMSquared'])) {
                update_post_meta($post_id, 'property_area', $property['PropertyFeatures']['PropertyLandAreaMSquared']);
            } elseif (!empty($property['PropertyFeatures']['PropertyFloorArea'])) {
                update_post_meta($post_id, 'property_area', $property['PropertyFeatures']['PropertyFloorArea']);
            }

            update_post_meta($post_id, 'information_address', $property_address);
            update_post_meta($post_id, 'information_rent', $property_rent);
            update_post_meta($post_id, 'information_date', $property_date);
            update_post_meta($post_id, 'information_suburb', $property_suburb);
            update_post_meta($post_id, 'information_region', $property_region);
            update_post_meta($post_id, 'information_code', $property['PropertyChangeCode'] ?: '');
            update_post_meta($post_id, 'accommodation_bed_room', $property_bedrooms);
            update_post_meta($post_id, 'accommodation_bath_room', $property_bathrooms);
            update_post_meta($post_id, 'accommodation_automobile', $property_cars);
            update_post_meta($post_id, 'other_pets', $property_pets);
            update_post_meta($post_id, 'other_parking', $property_parking);
            update_post_meta($post_id, 'street_number', $property['PropertyAddress1'] ?: '');
            update_post_meta($post_id, 'street_name', $property['PropertyAddress2'] ?: '');
            update_post_meta($post_id, 'property_city', $property['PropertyAddress3'] ?: '');
            update_post_meta($post_id, 'zipcode', isset($property['PropertyPostCode']) ? $property['PropertyPostCode'] : '');
            update_post_meta($post_id, 'agent_code', $property_agent_code);

            // Additional fields
            update_post_meta($post_id, 'property_year_built', $property_year_built);
            update_post_meta($post_id, 'property_class', $property_class);
            update_post_meta($post_id, 'property_smokers_allowed', $property_smokers_allowed);
            update_post_meta($post_id, 'property_virtual_tour_url', $property_virtual_tour_url);
            update_post_meta($post_id, 'property_geographic_location', $property_geographic_location);
            update_post_meta($post_id, 'property_rental_period', $property['PropertyRentalPeriod'] ?: '');
            update_post_meta($post_id, 'property_status', $property['PropertyStatus'] ?: '');
            update_post_meta($post_id, 'property_management_type', $property['PropertyManagementType'] ?: '');
            update_post_meta($post_id, 'property_owner_code', $property['PropertyOwnerCode'] ?: '');

            // Property Agent Details
            update_post_meta($post_id, 'agent_full_name', $property['PropertyAgent']['PropertyAgentFullName'] ?: '');
            update_post_meta($post_id, 'agent_email1', $property['PropertyAgent']['PropertyAgentEmail1'] ?: '');
            update_post_meta($post_id, 'agent_email2', $property['PropertyAgent']['PropertyAgentEmail2'] ?: '');
            update_post_meta($post_id, 'agent_fax', $property['PropertyAgent']['PropertyAgentFax'] ?: '');
            update_post_meta($post_id, 'agent_phone_mobile', $property['PropertyAgent']['PropertyAgentPhoneMobile'] ?: '');
            update_post_meta($post_id, 'agent_phone_work', $property['PropertyAgent']['PropertyAgentPhoneWork'] ?: '');
            update_post_meta($post_id, 'agent_title', $property['PropertyAgent']['PropertyAgentTitle'] ?: '');

            // Custom fields
            if (!empty($property['PropertyCustomList'])) {
                foreach ($property['PropertyCustomList'] as $custom_field) {
                    update_post_meta($post_id, sanitize_key($custom_field['CustomFieldName']), $custom_field['CustomFieldValue']);
                }
            }

            // Property Suburb Details
            foreach ($property['PropertySuburb'] as $index => $suburb) {
                update_post_meta($post_id, 'property_suburb_code_' . $index, $suburb['PropertySuburbCode']);
                update_post_meta($post_id, 'property_suburb_district_' . $index, $suburb['PropertySuburbDistrictOrPostcode']);
                update_post_meta($post_id, 'property_suburb_name_' . $index, $suburb['PropertySuburbName']);
                update_post_meta($post_id, 'property_suburb_region_' . $index, $suburb['PropertySuburbRegionOrState']);
                update_post_meta($post_id, 'property_suburb_type_' . $index, $suburb['PropertySuburbType']);
            }

            if (!wp_next_scheduled('update_integrate_palace_gallery', array($post_id, $property_code))) {
                wp_schedule_single_event(time() + 600, 'update_integrate_palace_gallery', array($post_id, $property_code));
            }
        }

        return new WP_REST_Response(array('status' => 'success'), 200);
    }
}
