<?php
/**
 * Gutenberg Block Editor Integration
 *
 * Adds InstaRank SEO panel to the Gutenberg sidebar
 *
 * @package InstaRank
 * @since 1.4.0
 */

defined('ABSPATH') || exit;

class InstaRank_Gutenberg_Integration {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('enqueue_block_editor_assets', [$this, 'enqueue_block_editor_assets']);
        add_action('rest_api_init', [$this, 'register_meta_fields']);
    }

    /**
     * Enqueue Gutenberg editor assets
     */
    public function enqueue_block_editor_assets() {
        // Check if we're editing a post/page
        $screen = get_current_screen();
        if (!$screen || !$screen->is_block_editor()) {
            return;
        }

        // Enqueue the React component
        wp_enqueue_script(
            'instarank-gutenberg-panel',
            INSTARANK_PLUGIN_URL . 'integrations/gutenberg/build/index.js',
            ['wp-plugins', 'wp-edit-post', 'wp-element', 'wp-components', 'wp-data', 'wp-compose'],
            INSTARANK_VERSION,
            true
        );

        // Enqueue styles
        wp_enqueue_style(
            'instarank-gutenberg-panel',
            INSTARANK_PLUGIN_URL . 'integrations/gutenberg/build/style.css',
            ['wp-edit-post'],
            INSTARANK_VERSION
        );

        // Get current post data
        global $post;
        $detector = new InstaRank_SEO_Detector();

        // Get pSEO data if this is a generated page
        $pseo_fields = get_post_meta($post->ID, '_instarank_pseo_fields', true);
        $is_pseo_page = get_post_meta($post->ID, '_instarank_pseo_generated', true);
        $pseo_generated_at = get_post_meta($post->ID, '_instarank_pseo_generated_at', true);

        // Pass data to JavaScript
        wp_localize_script('instarank-gutenberg-panel', 'instarankGutenberg', [
            'postId' => $post->ID,
            'postType' => $post->post_type,
            'nonce' => wp_create_nonce('instarank_gutenberg'),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'apiUrl' => INSTARANK_API_URL,
            'connected' => (bool) get_option('instarank_connected'),
            'projectId' => get_option('instarank_project_id', ''),
            'seoPlugin' => $detector->get_active_seo_plugin(),
            'metaData' => [
                'metaTitle' => get_post_meta($post->ID, 'instarank_meta_title', true),
                'metaDescription' => get_post_meta($post->ID, 'instarank_meta_description', true),
                'focusKeyword' => get_post_meta($post->ID, 'instarank_focus_keyword', true),
                'canonical' => get_post_meta($post->ID, 'instarank_canonical', true),
                'robotsNoindex' => get_post_meta($post->ID, 'instarank_robots_noindex', true),
                'robotsMeta' => get_post_meta($post->ID, 'instarank_robots_meta', true),
                'ogTitle' => get_post_meta($post->ID, 'instarank_og_title', true),
                'ogDescription' => get_post_meta($post->ID, 'instarank_og_description', true),
                'ogImage' => get_post_meta($post->ID, 'instarank_og_image', true),
                'twitterTitle' => get_post_meta($post->ID, 'instarank_twitter_title', true),
                'twitterDescription' => get_post_meta($post->ID, 'instarank_twitter_description', true),
                'twitterImage' => get_post_meta($post->ID, 'instarank_twitter_image', true),
            ],
            'pseoData' => [
                'isPseoPage' => (bool) $is_pseo_page,
                'fields' => is_array($pseo_fields) ? $pseo_fields : [],
                'generatedAt' => $pseo_generated_at ?: '',
                'fieldCount' => is_array($pseo_fields) ? count($pseo_fields) : 0,
            ],
            'pendingChanges' => $this->get_pending_changes_count($post->ID),
            'limits' => [
                'titleMax' => 60,
                'titleOptimal' => 50,
                'descriptionMax' => 160,
                'descriptionOptimal' => 155,
            ],
        ]);
    }

    /**
     * Register meta fields for REST API access
     */
    public function register_meta_fields() {
        $post_types = get_post_types(['public' => true]);

        $meta_fields = [
            'instarank_meta_title',
            'instarank_meta_description',
            'instarank_focus_keyword',
            'instarank_canonical',
            'instarank_robots_noindex',
            'instarank_robots_meta',
            'instarank_og_title',
            'instarank_og_description',
            'instarank_og_image',
            'instarank_twitter_title',
            'instarank_twitter_description',
            'instarank_twitter_image',
        ];

        foreach ($post_types as $post_type) {
            foreach ($meta_fields as $meta_key) {
                register_post_meta($post_type, $meta_key, [
                    'show_in_rest' => true,
                    'single' => true,
                    'type' => 'string',
                    'auth_callback' => function() {
                        return current_user_can('edit_posts');
                    }
                ]);
            }
        }
    }

    /**
     * Get pending changes count for a post
     */
    private function get_pending_changes_count($post_id) {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}instarank_changes WHERE post_id = %d AND status = 'pending'",
            $post_id
        ));

        return (int) $count;
    }
}

// Initialize
InstaRank_Gutenberg_Integration::instance();
