<?php
/**
 * Plugin Name: InstaRank
 * Plugin URI: https://instarank.com/wordpress-plugin
 * Description: Connect your WordPress site to InstaRank for AI-powered SEO optimization, schema markup generation, and programmatic SEO. Create and sync custom post types, automatically apply SEO improvements, and generate structured data with InstaRank's AI engine.
 * Version: 2.0.8
 * Author: InstaRank
 * Author URI: https://instarank.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: instarank
 * Domain Path: /languages
 * Requires at least: 5.6
 * Requires PHP: 7.4
 */

defined('ABSPATH') || exit;

// Define plugin constants
define('INSTARANK_VERSION', '2.0.8');
define('INSTARANK_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('INSTARANK_PLUGIN_URL', plugin_dir_url(__FILE__));
// NOTE: This should be your InstaRank app URL (e.g., https://app.instarank.com or http://localhost:3000 for development)
// NOT the API URL - OAuth endpoints are on the main app, not the API subdomain
define('INSTARANK_API_URL', 'https://app.instarank.com');

// Include core classes
require_once INSTARANK_PLUGIN_DIR . 'includes/class-api-handler.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-auth-manager.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-seo-detector.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-change-manager.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-webhook-sender.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-schema-injector.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-schema-generator.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-sitemap-generator.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-robots-txt.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-llms-txt.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-breadcrumbs.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-classic-editor.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-page-builder-api.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-field-detector.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-acf-detector.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-spintax-engine.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-indexnow.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-related-links.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-virtual-pages.php';
require_once INSTARANK_PLUGIN_DIR . 'includes/class-multilang.php';
require_once INSTARANK_PLUGIN_DIR . 'api/endpoints.php';
require_once INSTARANK_PLUGIN_DIR . 'api/agent-endpoints.php';

// Include page builder integrations
require_once INSTARANK_PLUGIN_DIR . 'integrations/gutenberg/class-gutenberg-integration.php';

/**
 * Main InstaRank Class
 */
class InstaRank_SEO {

    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Note: Translations are handled automatically by WordPress.org for hosted plugins.
        // No need for load_plugin_textdomain() since WP 4.6+

        // REST API endpoints are registered in api/endpoints.php via class instantiation

        // Add admin menu
        add_action('admin_menu', [$this, 'admin_menu']);

        // Add settings link on plugins page
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), [$this, 'action_links']);

        // Enqueue admin scripts
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);

        // Add admin notices
        add_action('admin_notices', [$this, 'admin_notices']);

        // Output custom meta tags in head (when no SEO plugin is active)
        add_action('wp_head', [$this, 'output_custom_meta_tags'], 1);

        // Fix Kadence dynamic image content - hook into block rendering
        add_filter('render_block_kadence/image', [$this, 'fix_kadence_dynamic_image_block'], 10, 3);

        // AJAX handlers
        add_action('wp_ajax_instarank_test_connection', [$this, 'ajax_test_connection']);
        add_action('wp_ajax_instarank_check_connection_status', [$this, 'ajax_check_connection_status']);
        add_action('wp_ajax_instarank_confirm_oauth_connection', [$this, 'ajax_confirm_oauth_connection']);
        add_action('wp_ajax_instarank_sync_now', [$this, 'ajax_sync_now']);
        add_action('wp_ajax_instarank_approve_change', [$this, 'ajax_approve_change']);
        add_action('wp_ajax_instarank_reject_change', [$this, 'ajax_reject_change']);
        add_action('wp_ajax_instarank_rollback_change', [$this, 'ajax_rollback_change']);
        add_action('wp_ajax_instarank_view_details', [$this, 'ajax_view_details']);
        add_action('wp_ajax_instarank_disconnect', [$this, 'ajax_disconnect']);
        add_action('wp_ajax_instarank_reset_auth_attempts', [$this, 'ajax_reset_auth_attempts']);
        add_action('wp_ajax_instarank_clear_history', [$this, 'ajax_clear_history']);
        add_action('wp_ajax_instarank_reset_robots_txt', [$this, 'ajax_reset_robots_txt']);
        add_action('wp_ajax_instarank_save_dataset_url', [$this, 'ajax_save_dataset_url']);
    }

    /**
     * Fix Kadence dynamic image content
     *
     * Kadence Blocks Pro expects image dynamic content to return an array with
     * [url, width, height, crop, alt, attachment_id]. When using custom fields
     * that store just a URL string, Kadence treats the string as an array and
     * extracts individual characters, resulting in src="h" alt=":" etc.
     *
     * This filter intercepts the rendered block output and fixes any corrupted
     * image tags by looking up the actual custom field values.
     *
     * @param string $block_content The block content.
     * @param array  $block         The full block data.
     * @param WP_Block $instance    The block instance.
     * @return string Modified block content.
     */
    public function fix_kadence_dynamic_image_block($block_content, $block, $instance) {
        // Check if this block has kadenceDynamic with URL enabled
        if (empty($block['attrs']['kadenceDynamic']['url']['enable'])) {
            return $block_content;
        }

        // Get the custom field name
        $custom_field = $block['attrs']['kadenceDynamic']['url']['custom'] ?? '';
        if (empty($custom_field)) {
            return $block_content;
        }

        // Get current post
        $post_id = get_the_ID();
        if (!$post_id) {
            return $block_content;
        }

        // Get the actual image URL from the custom field
        $image_url = get_post_meta($post_id, $custom_field, true);
        if (empty($image_url) || !filter_var($image_url, FILTER_VALIDATE_URL)) {
            return $block_content;
        }

        // Get alt text - try the _alt suffix field
        $alt_field = $custom_field . '_alt';
        $alt_text = get_post_meta($post_id, $alt_field, true);
        if (empty($alt_text)) {
            // Fallback: use the field name as descriptive alt
            $alt_text = '';
        }

        // Fix corrupted image tags - pattern matches short src values
        // that indicate Kadence treated the URL string as an array
        $pattern = '/<img([^>]*)src="([^"]*)"([^>]*)alt="([^"]*)"([^>]*)class="kb-img([^"]*)"([^>]*)>/';

        $block_content = preg_replace_callback($pattern, function($matches) use ($image_url, $alt_text) {
            $src_value = $matches[2];

            // Only fix if src is suspiciously short (corrupted - under 10 chars)
            if (strlen($src_value) > 10) {
                return $matches[0]; // Not corrupted, return as-is
            }

            // Build attributes, cleaning up corrupted width/height
            $before_src = $matches[1];
            $after_src = $matches[3];
            $after_alt = $matches[5];
            $class_suffix = $matches[6];
            $after_class = $matches[7];

            // Remove corrupted width/height attributes
            $before_src = preg_replace('/\s*width="[^"]{1,5}"\s*/', ' ', $before_src);
            $before_src = preg_replace('/\s*height="[^"]{1,5}"\s*/', ' ', $before_src);
            $after_src = preg_replace('/\s*width="[^"]{1,5}"\s*/', ' ', $after_src);
            $after_src = preg_replace('/\s*height="[^"]{1,5}"\s*/', ' ', $after_src);
            $after_alt = preg_replace('/\s*width="[^"]{1,5}"\s*/', ' ', $after_alt);
            $after_alt = preg_replace('/\s*height="[^"]{1,5}"\s*/', ' ', $after_alt);
            $after_class = preg_replace('/\s*width="[^"]{1,5}"\s*/', ' ', $after_class);
            $after_class = preg_replace('/\s*height="[^"]{1,5}"\s*/', ' ', $after_class);

            // Clean up wp-image- with corrupted ID
            $class_suffix = preg_replace('/\s*wp-image-[^"]*/', '', $class_suffix);

            // Reconstruct the img tag with correct values
            return '<img' . $before_src . 'src="' . esc_url($image_url) . '"' . $after_src .
                   'alt="' . esc_attr($alt_text) . '"' . $after_alt . 'class="kb-img' . $class_suffix . '"' . $after_class . '>';
        }, $block_content);

        return $block_content;
    }

    /**
     * Output custom meta tags when no SEO plugin is active
     * Works for all pages and posts, not just homepage
     */
    public function output_custom_meta_tags() {
        // Check if an SEO plugin is active
        $detector = new InstaRank_SEO_Detector();
        $seo_plugin = $detector->get_active_seo_plugin();

        // If an SEO plugin is active, let it handle the meta tags
        if ($seo_plugin !== 'default' && $seo_plugin !== 'none') {
            return;
        }

        $meta_output = [];

        // Homepage meta tags
        if (is_front_page()) {
            $home_meta = get_option('instarank_home_meta', []);

            if (!empty($home_meta)) {
                if (!empty($home_meta['meta_title'])) {
                    add_filter('pre_get_document_title', function() use ($home_meta) {
                        return $home_meta['meta_title'];
                    }, 999);
                }

                if (!empty($home_meta['meta_description'])) {
                    $meta_output[] = '<meta name="description" content="' . esc_attr($home_meta['meta_description']) . '" />';
                }

                // Open Graph
                if (!empty($home_meta['og_title'])) {
                    $meta_output[] = '<meta property="og:title" content="' . esc_attr($home_meta['og_title']) . '" />';
                }
                if (!empty($home_meta['og_description'])) {
                    $meta_output[] = '<meta property="og:description" content="' . esc_attr($home_meta['og_description']) . '" />';
                }
                if (!empty($home_meta['og_image'])) {
                    $meta_output[] = '<meta property="og:image" content="' . esc_url($home_meta['og_image']) . '" />';
                }

                $meta_output[] = '<meta property="og:url" content="' . esc_url(home_url('/')) . '" />';
                $meta_output[] = '<meta property="og:type" content="website" />';
            }
        }

        // Post/Page specific meta tags
        elseif (is_singular()) {
            global $post;
            $post_id = $post->ID;

            // Meta title
            $meta_title = get_post_meta($post_id, 'instarank_meta_title', true);
            if ($meta_title) {
                add_filter('pre_get_document_title', function() use ($meta_title) {
                    return $meta_title;
                }, 999);
            }

            // Meta description
            $meta_desc = get_post_meta($post_id, 'instarank_meta_description', true);
            if ($meta_desc) {
                $meta_output[] = '<meta name="description" content="' . esc_attr($meta_desc) . '" />';
            }

            // Canonical URL
            $canonical = get_post_meta($post_id, 'instarank_canonical', true);
            if ($canonical) {
                $meta_output[] = '<link rel="canonical" href="' . esc_url($canonical) . '" />';
            }

            // Robots meta
            $robots_noindex = get_post_meta($post_id, 'instarank_robots_noindex', true);
            $robots_meta = get_post_meta($post_id, 'instarank_robots_meta', true);
            if ($robots_noindex === '1' || $robots_meta) {
                $robots_content = [];
                if ($robots_noindex === '1') {
                    $robots_content[] = 'noindex';
                }
                if ($robots_meta) {
                    $robots_content[] = $robots_meta;
                }
                $meta_output[] = '<meta name="robots" content="' . esc_attr(implode(', ', $robots_content)) . '" />';
            }

            // Open Graph tags
            $og_title = get_post_meta($post_id, 'instarank_og_title', true);
            $og_desc = get_post_meta($post_id, 'instarank_og_description', true);
            $og_image = get_post_meta($post_id, 'instarank_og_image', true);
            $og_url = get_post_meta($post_id, 'instarank_og_url', true);

            if ($og_title || $og_desc || $og_image || $og_url) {
                if ($og_title) {
                    $meta_output[] = '<meta property="og:title" content="' . esc_attr($og_title) . '" />';
                }
                if ($og_desc) {
                    $meta_output[] = '<meta property="og:description" content="' . esc_attr($og_desc) . '" />';
                }
                if ($og_image) {
                    $meta_output[] = '<meta property="og:image" content="' . esc_url($og_image) . '" />';
                }
                if ($og_url) {
                    $meta_output[] = '<meta property="og:url" content="' . esc_url($og_url) . '" />';
                } else {
                    $meta_output[] = '<meta property="og:url" content="' . esc_url(get_permalink($post_id)) . '" />';
                }
                $meta_output[] = '<meta property="og:type" content="article" />';
            }

            // Twitter Card tags
            $twitter_title = get_post_meta($post_id, 'instarank_twitter_title', true);
            $twitter_desc = get_post_meta($post_id, 'instarank_twitter_description', true);
            $twitter_image = get_post_meta($post_id, 'instarank_twitter_image', true);

            if ($twitter_title || $twitter_desc || $twitter_image) {
                $meta_output[] = '<meta name="twitter:card" content="summary_large_image" />';
                if ($twitter_title) {
                    $meta_output[] = '<meta name="twitter:title" content="' . esc_attr($twitter_title) . '" />';
                }
                if ($twitter_desc) {
                    $meta_output[] = '<meta name="twitter:description" content="' . esc_attr($twitter_desc) . '" />';
                }
                if ($twitter_image) {
                    $meta_output[] = '<meta name="twitter:image" content="' . esc_url($twitter_image) . '" />';
                }
            }

            // Schema markup
            $schema = get_post_meta($post_id, 'instarank_schema_markup', true);
            if ($schema) {
                // Output as JSON-LD
                $meta_output[] = '<script type="application/ld+json">' . wp_json_encode(json_decode($schema), JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . '</script>';
            }
        }

        // Output all meta tags
        if (!empty($meta_output)) {
            echo "\n<!-- InstaRank Meta Tags -->\n";
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- All values already escaped individually
            echo implode("\n", $meta_output);
            echo "\n<!-- / InstaRank Meta Tags -->\n\n";
        }
    }

    /**
     * Add admin menu
     */
    public function admin_menu() {
        // Load custom InstaRank icon from file
        $icon_path = plugin_dir_path(__FILE__) . 'assets/images/menu-icon.svg';
        $icon_svg = '';

        if (file_exists($icon_path)) {
            $svg_content = file_get_contents($icon_path);
            $icon_svg = 'data:image/svg+xml;base64,' . base64_encode($svg_content);
        } else {
            // Fallback to dashicons if file not found
            $icon_svg = 'dashicons-chart-line';
        }

        add_menu_page(
            __('InstaRank', 'instarank'),
            __('InstaRank', 'instarank'),
            'manage_options',
            'instarank',
            [$this, 'render_dashboard'],
            $icon_svg,
            30
        );

        add_submenu_page(
            'instarank',
            __('Dashboard', 'instarank'),
            __('Dashboard', 'instarank'),
            'manage_options',
            'instarank',
            [$this, 'render_dashboard']
        );

        add_submenu_page(
            'instarank',
            __('Pending Changes', 'instarank'),
            __('Pending Changes', 'instarank'),
            'manage_options',
            'instarank-changes',
            [$this, 'render_changes']
        );

        add_submenu_page(
            'instarank',
            __('Settings', 'instarank'),
            __('Settings', 'instarank'),
            'manage_options',
            'instarank-settings',
            [$this, 'render_settings']
        );
    }

    /**
     * Add action links
     */
    public function action_links($links) {
        $settings_link = sprintf(
            '<a href="%s">%s</a>',
            admin_url('admin.php?page=instarank-settings'),
            __('Settings', 'instarank')
        );
        array_unshift($links, $settings_link);
        return $links;
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        // InstaRank admin pages
        if (strpos($hook, 'instarank') !== false) {
            wp_enqueue_style(
                'instarank-admin',
                INSTARANK_PLUGIN_URL . 'assets/css/admin-minimal.css',
                [],
                INSTARANK_VERSION
            );

            wp_enqueue_script(
                'instarank-admin',
                INSTARANK_PLUGIN_URL . 'assets/js/admin.js',
                ['jquery'],
                INSTARANK_VERSION,
                true
            );

            wp_localize_script('instarank-admin', 'instarankAdmin', [
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('instarank_admin'),
                'apiUrl' => INSTARANK_API_URL,
                'siteUrl' => get_site_url(),
                'apiKey' => get_option('instarank_api_key', '')
            ]);
        }

        // Post/Page editor - load custom fields integration for all page builders
        if ($hook === 'post.php' || $hook === 'post-new.php') {
            $this->enqueue_custom_fields_integration();
        }
    }

    /**
     * Detect active page builder for the current post
     */
    private function detect_active_builder($post_id) {
        $post = get_post($post_id);
        if (!$post) {
            return 'gutenberg';
        }

        // Check for Kadence Blocks
        if (defined('KADENCE_BLOCKS_VERSION') || class_exists('Kadence_Blocks')) {
            if (has_blocks($post->post_content)) {
                $blocks = parse_blocks($post->post_content);
                foreach ($blocks as $block) {
                    if (strpos($block['blockName'] ?? '', 'kadence/') === 0) {
                        return 'kadence_blocks';
                    }
                }
            }
        }

        // Check for Elementor
        if (defined('ELEMENTOR_VERSION') && get_post_meta($post_id, '_elementor_edit_mode', true) === 'builder') {
            return 'elementor';
        }

        // Check for Divi
        if (get_post_meta($post_id, '_et_pb_use_builder', true) === 'on') {
            return 'divi';
        }

        // Check for Beaver Builder
        if (get_post_meta($post_id, '_fl_builder_enabled', true)) {
            return 'beaver_builder';
        }

        // Check for Bricks
        if (get_post_meta($post_id, '_bricks_page_content_2', true) || get_post_meta($post_id, '_bricks_page_content', true)) {
            return 'bricks';
        }

        // Default to Gutenberg
        return 'gutenberg';
    }

    /**
     * Enqueue universal custom fields integration script
     * Shows InstaRank dataset fields in any page builder's custom fields UI
     *
     * IMPORTANT: Only loads for:
     * 1. Template pages (pages configured as pSEO templates with field mappings)
     * 2. Generated pSEO pages (pages created by InstaRank with _instarank_pseo_generated meta)
     *
     * Does NOT load for regular static pages (Home, About, etc.)
     */
    private function enqueue_custom_fields_integration() {
        global $post;

        // Only proceed if we have a post
        if (!$post) {
            return;
        }

        $post_id = $post->ID;

        // Check if this is a pSEO generated page
        $is_pseo_page = get_post_meta($post_id, '_instarank_pseo_generated', true);

        // Get field mappings for this specific template (if configured)
        $option_name = 'instarank_field_mappings_' . $post_id;
        $saved_mappings = get_option($option_name, []);
        $field_mappings = isset($saved_mappings['mappings']) ? $saved_mappings['mappings'] : [];
        $dataset_name = isset($saved_mappings['dataset_name']) ? $saved_mappings['dataset_name'] : '';

        // Check if this post is a configured template (has field mappings)
        $is_template_page = !empty($field_mappings);

        // IMPORTANT: Only show the custom fields panel on template pages or pSEO generated pages
        // Do NOT show on regular static pages like Home, About, Contact, etc.
        if (!$is_template_page && !$is_pseo_page) {
            return;
        }

        // Extract dataset columns from template-specific mappings
        $dataset_columns = [];
        if (!empty($field_mappings)) {
            foreach ($field_mappings as $field_name => $mapping) {
                if (!empty($mapping['dataset_column'])) {
                    $dataset_columns[] = $mapping['dataset_column'];
                }
            }
        }

        // For pSEO generated pages, get their stored fields as columns
        if ($is_pseo_page) {
            $pseo_fields = get_post_meta($post_id, '_instarank_pseo_fields', true);
            if (is_array($pseo_fields)) {
                $dataset_columns = array_merge($dataset_columns, array_keys($pseo_fields));
            }
        }

        // ALSO get global dataset columns (linked on Templates page) - only for template pages
        if ($is_template_page) {
            $global_dataset = get_option('instarank_global_dataset', []);
            if (!empty($global_dataset['columns'])) {
                // Add global columns to the list
                $dataset_columns = array_merge($dataset_columns, $global_dataset['columns']);

                // Use global dataset name if no template-specific name
                if (empty($dataset_name) && !empty($global_dataset['dataset_name'])) {
                    $dataset_name = $global_dataset['dataset_name'];
                }
            }
        }

        // Deduplicate columns
        $dataset_columns = array_unique($dataset_columns);

        // Enqueue if we have columns to show
        if (empty($dataset_columns)) {
            return;
        }

        // Detect active page builder
        $active_builder = $this->detect_active_builder($post_id);

        // Enqueue universal custom fields script (works with all builders)
        wp_enqueue_script(
            'instarank-custom-fields',
            INSTARANK_PLUGIN_URL . 'assets/js/instarank-custom-fields.js',
            ['wp-blocks', 'wp-element', 'wp-editor'],
            INSTARANK_VERSION,
            true
        );

        wp_localize_script('instarank-custom-fields', 'instarankCustomFields', [
            'templateId' => $post_id,
            'fieldMappings' => $field_mappings,
            'datasetColumns' => array_values($dataset_columns),
            'datasetName' => $dataset_name,
            'globalDataset' => $is_template_page && !empty(get_option('instarank_global_dataset', [])['columns']),
            'activeBuilder' => $active_builder,
            'isPseoPage' => (bool) $is_pseo_page,
            'isTemplatePage' => $is_template_page,
        ]);
    }

    /**
     * Admin notices
     */
    public function admin_notices() {
        $api_key = get_option('instarank_api_key');
        $connected = get_option('instarank_connected');

        if (!$api_key || !$connected) {
            $screen = get_current_screen();
            if ($screen && strpos($screen->id, 'instarank') === false) {
                ?>
                <div class="notice notice-warning is-dismissible">
                    <p>
                        <strong><?php esc_html_e('InstaRank:', 'instarank'); ?></strong>
                        <?php esc_html_e('Your WordPress site is not connected to InstaRank yet.', 'instarank'); ?>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=instarank')); ?>">
                            <?php esc_html_e('Connect Now', 'instarank'); ?>
                        </a>
                    </p>
                </div>
                <?php
            }
        }
    }

    /**
     * Render dashboard page
     */
    public function render_dashboard() {
        require_once INSTARANK_PLUGIN_DIR . 'admin/dashboard-minimal.php';
    }

    /**
     * Render changes page
     */
    public function render_changes() {
        require_once INSTARANK_PLUGIN_DIR . 'admin/changes-minimal.php';
    }

    /**
     * Render settings page
     */
    public function render_settings() {
        require_once INSTARANK_PLUGIN_DIR . 'admin/settings-minimal.php';
    }

    /**
     * AJAX handler for test connection
     */
    public function ajax_test_connection() {
        check_ajax_referer('instarank_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        // Get project ID
        $project_id = get_option('instarank_project_id', '');
        if (empty($project_id)) {
            wp_send_json_error(['message' => 'Not connected to InstaRank. Please connect first.']);
        }

        // Simple connection test - just verify we have the necessary settings
        $connected = get_option('instarank_connected', false);
        $api_key = get_option('instarank_api_key', '');

        if ($connected && !empty($api_key)) {
            wp_send_json_success(['message' => 'Connection is active']);
        } else {
            wp_send_json_error(['message' => 'Connection not properly configured']);
        }
    }

    /**
     * AJAX handler for check connection status
     */
    public function ajax_check_connection_status() {
        check_ajax_referer('instarank_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        // Get the API key
        $api_key = get_option('instarank_api_key', '');
        if (empty($api_key)) {
            wp_send_json_error(['message' => 'No API key found. Please regenerate the API key and connect from InstaRank.']);
        }

        // Make a request to the plugin's own info endpoint to check if it's working
        $site_url = get_site_url();
        $info_url = $site_url . '/wp-json/instarank/v1/info';

        $response = wp_remote_get($info_url, [
            'headers' => [
                'X-WordPress-API-Key' => $api_key
            ],
            'timeout' => 15
        ]);

        if (is_wp_error($response)) {
            wp_send_json_error(['message' => 'Failed to check plugin status: ' . $response->get_error_message()]);
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!$data || !isset($data['site'])) {
            wp_send_json_error(['message' => 'Invalid response from plugin endpoint']);
        }

        // IMPORTANT: Verify with InstaRank backend if integration still exists
        // This prevents showing "connected" when integration was deleted from dashboard
        $instarank_url = defined('INSTARANK_API_URL') ? INSTARANK_API_URL : 'https://app.instarank.com';
        $verify_url = $instarank_url . '/api/wordpress-oauth/verify';

        $verify_response = wp_remote_post($verify_url, [
            'headers' => [
                'Content-Type' => 'application/json'
            ],
            'body' => wp_json_encode([
                'siteUrl' => $site_url,
                'apiKey' => $api_key
            ]),
            'timeout' => 15
        ]);

        // Check if InstaRank has connected to this site (local WordPress options)
        $connected = get_option('instarank_connected', false);
        $project_id = get_option('instarank_project_id', '');

        // If we got a response from InstaRank, verify the integration still exists
        if (!is_wp_error($verify_response)) {
            $verify_body = wp_remote_retrieve_body($verify_response);
            $verify_data = json_decode($verify_body, true);

            // If integration doesn't exist in InstaRank database, clear local connection state
            if ($verify_data && isset($verify_data['exists']) && !$verify_data['exists']) {
                // Integration was deleted from InstaRank dashboard - clear WordPress options
                update_option('instarank_connected', false);
                update_option('instarank_project_id', '');
                update_option('instarank_integration_id', '');
                delete_option('instarank_connected_at');

                wp_send_json_success([
                    'connected' => false,
                    'message' => 'Connection was removed from InstaRank dashboard. Please reconnect.'
                ]);
                return;
            }

            // If integration exists but is not connected, update local state
            if ($verify_data && isset($verify_data['connected']) && !$verify_data['connected']) {
                update_option('instarank_connected', false);

                wp_send_json_success([
                    'connected' => false,
                    'message' => 'Not fully connected. Please complete the connection from InstaRank dashboard.'
                ]);
                return;
            }
        }

        // Return current connection status
        if ($connected && !empty($project_id)) {
            wp_send_json_success([
                'connected' => true,
                'project_id' => $project_id,
                'message' => 'Connected to InstaRank'
            ]);
        } else {
            wp_send_json_success([
                'connected' => false,
                'message' => 'Not connected. Please connect from your InstaRank dashboard.'
            ]);
        }
    }

    /**
     * AJAX handler for confirming OAuth connection
     */
    public function ajax_confirm_oauth_connection() {
        check_ajax_referer('instarank_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        // Get integration data from request
        // phpcs:ignore WordPress.Security.NonceVerification.Missing,WordPress.Security.ValidatedSanitizedInput.InputNotSanitized,WordPress.Security.ValidatedSanitizedInput.MissingUnslash
        $integration_raw = isset($_POST['integration']) ? wp_unslash($_POST['integration']) : null;

        // Sanitize the integration data
        $integration = null;
        if (is_array($integration_raw)) {
            $integration = array_map('sanitize_text_field', $integration_raw);
        }

        if (!$integration || !isset($integration['project_id'])) {
            wp_send_json_error(['message' => 'Invalid integration data']);
        }

        // Update WordPress options to mark as connected
        update_option('instarank_connected', true);
        update_option('instarank_project_id', sanitize_text_field($integration['project_id']));
        update_option('instarank_integration_id', isset($integration['id']) ? sanitize_text_field($integration['id']) : '');
        update_option('instarank_connected_at', current_time('mysql'));

        wp_send_json_success([
            'message' => 'Connection confirmed successfully',
            'project_id' => sanitize_text_field($integration['project_id'])
        ]);
    }

    /**
     * AJAX handler for sync now
     */
    public function ajax_sync_now() {
        check_ajax_referer('instarank_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        // Get project ID
        $project_id = get_option('instarank_project_id', '');
        if (empty($project_id)) {
            wp_send_json_error(['message' => 'Not connected to InstaRank. Please connect first.']);
        }

        // Count posts and pages
        $posts_count = wp_count_posts('post')->publish;
        $pages_count = wp_count_posts('page')->publish;
        $total_count = $posts_count + $pages_count;

        // Update last sync time
        update_option('instarank_last_sync', current_time('mysql'));

        wp_send_json_success([
            'message' => 'Sync completed successfully',
            'count' => $total_count,
            'posts' => $posts_count,
            'pages' => $pages_count
        ]);
    }

    /**
     * AJAX handler for approving a change
     */
    public function ajax_approve_change() {
        check_ajax_referer('instarank_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $change_id = isset($_POST['change_id']) ? intval($_POST['change_id']) : 0;
        if (!$change_id) {
            wp_send_json_error(['message' => 'Invalid change ID']);
        }

        $change_manager = new InstaRank_Change_Manager();
        $result = $change_manager->approve_change($change_id);

        if ($result) {
            wp_send_json_success(['message' => 'Change approved and applied successfully']);
        } else {
            wp_send_json_error(['message' => 'Failed to approve change']);
        }
    }

    /**
     * AJAX handler for rejecting a change
     */
    public function ajax_reject_change() {
        check_ajax_referer('instarank_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $change_id = isset($_POST['change_id']) ? intval($_POST['change_id']) : 0;
        if (!$change_id) {
            wp_send_json_error(['message' => 'Invalid change ID']);
        }

        $change_manager = new InstaRank_Change_Manager();
        $result = $change_manager->reject_change($change_id);

        if ($result) {
            wp_send_json_success(['message' => 'Change rejected successfully']);
        } else {
            wp_send_json_error(['message' => 'Failed to reject change']);
        }
    }

    /**
     * AJAX handler for rolling back a change
     */
    public function ajax_rollback_change() {
        check_ajax_referer('instarank_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $change_id = isset($_POST['change_id']) ? intval($_POST['change_id']) : 0;
        if (!$change_id) {
            wp_send_json_error(['message' => 'Invalid change ID']);
        }

        $change_manager = new InstaRank_Change_Manager();
        $result = $change_manager->rollback_change($change_id);

        if ($result) {
            wp_send_json_success(['message' => 'Change rolled back successfully']);
        } else {
            wp_send_json_error(['message' => 'Failed to rollback change']);
        }
    }

    /**
     * AJAX handler for viewing change details
     */
    public function ajax_view_details() {
        check_ajax_referer('instarank_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $change_id = isset($_POST['change_id']) ? intval($_POST['change_id']) : 0;
        if (!$change_id) {
            wp_send_json_error(['message' => 'Invalid change ID']);
        }

        global $wpdb;
        $table = $wpdb->prefix . 'instarank_changes';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $change = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $change_id
        ), ARRAY_A);

        if (!$change) {
            wp_send_json_error(['message' => 'Change not found']);
        }

        // Decode metadata if it exists
        if (!empty($change['metadata'])) {
            $change['metadata'] = json_decode($change['metadata'], true);
        }

        wp_send_json_success(['change' => $change]);
    }

    /**
     * AJAX handler for disconnecting from InstaRank
     */
    public function ajax_disconnect() {
        check_ajax_referer('instarank_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        // Check if user wants to clear all programmatic SEO data
        $clear_pseo_data = isset($_POST['clear_pseo_data']) && $_POST['clear_pseo_data'] === 'true';

        // Clear all InstaRank connection options
        delete_option('instarank_connected');
        delete_option('instarank_project_id');
        delete_option('instarank_integration_id');
        delete_option('instarank_connected_at');
        delete_option('instarank_last_sync');

        // Always clear Programmatic SEO data on disconnect to prevent stale data issues
        // This ensures a fresh start when reconnecting
        $this->clear_programmatic_seo_data();

        // Send webhook notification about disconnection
        $webhook_sender = new InstaRank_Webhook_Sender();
        $webhook_sender->send_disconnected();

        wp_send_json_success(['message' => 'Disconnected successfully. Programmatic SEO data has been cleared.']);
    }

    /**
     * Clear all Programmatic SEO data (datasets, field mappings, etc.)
     */
    private function clear_programmatic_seo_data() {
        global $wpdb;

        // Clear global dataset option
        delete_option('instarank_global_dataset');

        // Clear all field mappings for all posts
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->query(
            "DELETE FROM {$wpdb->options} WHERE option_name LIKE 'instarank_field_mappings_%'"
        );

        // Clear template field mappings
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->query(
            "DELETE FROM {$wpdb->options} WHERE option_name LIKE 'instarank_template_%'"
        );

        // Clear any cached sync data
        delete_option('instarank_last_dataset_sync');
        delete_option('instarank_dataset_cache');
    }

    /**
     * AJAX handler for resetting authentication attempts
     */
    public function ajax_reset_auth_attempts() {
        check_ajax_referer('instarank_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        // Clear failed authentication attempts
        delete_option('instarank_failed_auth_attempts');
        delete_option('instarank_last_failed_auth');

        wp_send_json_success(['message' => 'Authentication counter reset successfully']);
    }

    /**
     * AJAX handler for clearing all change history
     */
    public function ajax_clear_history() {
        check_ajax_referer('instarank_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        global $wpdb;
        $table = $wpdb->prefix . 'instarank_changes';

        // Delete all change records
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $deleted = $wpdb->query("DELETE FROM {$table}");

        if ($deleted !== false) {
            wp_send_json_success(['message' => 'History cleared successfully', 'deleted' => $deleted]);
        } else {
            wp_send_json_error(['message' => 'Failed to clear history']);
        }
    }

    /**
     * AJAX handler for resetting robots.txt to default
     */
    public function ajax_reset_robots_txt() {
        check_ajax_referer('instarank_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $robots_manager = InstaRank_Robots_Txt::instance();
        $robots_manager->reset_to_default();

        wp_send_json_success([
            'message' => 'Robots.txt reset to default',
            'content' => $robots_manager->get_default_content()
        ]);
    }

    /**
     * AJAX handler to save dataset URL
     */
    public function ajax_save_dataset_url() {
        check_ajax_referer('instarank_dataset_url', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $template_id = isset($_POST['template_id']) ? intval($_POST['template_id']) : 0;
        $url = isset($_POST['url']) ? esc_url_raw(wp_unslash($_POST['url'])) : '';

        if (!$template_id || !$url) {
            wp_send_json_error(['message' => 'Invalid parameters']);
        }

        // Save the dataset URL for this template
        update_option('instarank_dataset_url_' . $template_id, $url);

        wp_send_json_success([
            'message' => 'Dataset URL saved',
            'template_id' => $template_id,
            'url' => $url
        ]);
    }
}

/**
 * Register custom post types from InstaRank on WordPress init
 */
add_action('init', function() {
    $stored_post_types = get_option('instarank_custom_post_types', []);

    foreach ($stored_post_types as $slug => $data) {
        if (isset($data['config']) && !post_type_exists($slug)) {
            $config = $data['config'];

            // Handle parent post type rewrite chains
            // If config has parent_post_type, set up rewrite with parent slug token
            if (!empty($config['parent_post_type'])) {
                $parent_slug = $config['parent_post_type'];
                // Add a rewrite tag for the parent slug so WordPress can parse it
                add_rewrite_tag('%' . $parent_slug . '%', '([^/]+)');
            }

            register_post_type($slug, $config);
        }
    }

    // Add custom rewrite rules for parent-child post type chains
    foreach ($stored_post_types as $slug => $data) {
        if (!empty($data['config']['parent_post_type'])) {
            $config = $data['config'];
            $parent_slug = $config['parent_post_type'];
            $rewrite_slug = isset($config['rewrite']['slug']) ? $config['rewrite']['slug'] : $slug;

            // Check if the rewrite slug contains the parent token pattern
            if (strpos($rewrite_slug, '%' . $parent_slug . '%') !== false) {
                // Build regex from the rewrite slug by replacing the token
                $regex_base = str_replace('%' . $parent_slug . '%', '([^/]+)', preg_quote($rewrite_slug, '#'));

                // Single post: /{parent_slug}/{parent-item}/{child_slug}/{child-item}/
                add_rewrite_rule(
                    '^' . $regex_base . '/([^/]+)/?$',
                    'index.php?post_type=' . $slug . '&' . $slug . '=$matches[2]&' . $parent_slug . '=$matches[1]',
                    'top'
                );

                // Pagination: /{parent_slug}/{parent-item}/{child_slug}/{child-item}/page/2/
                add_rewrite_rule(
                    '^' . $regex_base . '/([^/]+)/page/([0-9]+)/?$',
                    'index.php?post_type=' . $slug . '&' . $slug . '=$matches[2]&' . $parent_slug . '=$matches[1]&paged=$matches[3]',
                    'top'
                );

                // Archive: /{parent_slug}/{parent-item}/{child_slug}/
                add_rewrite_rule(
                    '^' . $regex_base . '/?$',
                    'index.php?post_type=' . $slug . '&' . $parent_slug . '=$matches[1]',
                    'top'
                );
            }
        }
    }
}, 5); // Priority 5 to run before default post types

/**
 * Add parent post type query vars so WordPress recognizes them
 */
add_filter('query_vars', function($vars) {
    $stored_post_types = get_option('instarank_custom_post_types', []);
    foreach ($stored_post_types as $slug => $data) {
        if (!empty($data['config']['parent_post_type'])) {
            $parent_slug = $data['config']['parent_post_type'];
            if (!in_array($parent_slug, $vars)) {
                $vars[] = $parent_slug;
            }
        }
    }
    return $vars;
});

/**
 * Filter post type link to replace parent slug token in URLs
 */
add_filter('post_type_link', function($post_link, $post) {
    $stored_post_types = get_option('instarank_custom_post_types', []);

    if (isset($stored_post_types[$post->post_type])) {
        $config = $stored_post_types[$post->post_type]['config'];

        if (!empty($config['parent_post_type'])) {
            $parent_slug = $config['parent_post_type'];
            $token = '%' . $parent_slug . '%';

            if (strpos($post_link, $token) !== false) {
                // Find the parent post via post meta or parent resolution
                $parent_value = get_post_meta($post->ID, '_instarank_parent_slug', true);

                if (!$parent_value && $post->post_parent) {
                    // Try to get parent post's slug
                    $parent_post = get_post($post->post_parent);
                    if ($parent_post) {
                        $parent_value = $parent_post->post_name;
                    }
                }

                // Fallback: try to find parent value from resolution config meta
                if (!$parent_value) {
                    $parent_config = isset($config['parent_resolution_config']) ? $config['parent_resolution_config'] : null;
                    if ($parent_config && !empty($parent_config['column_name'])) {
                        $col = $parent_config['column_name'];
                        $val = get_post_meta($post->ID, $col, true);
                        if ($val) {
                            $parent_value = sanitize_title($val);
                        }
                    }
                }

                // Final fallback: use "uncategorized" to avoid duplicating the post type slug in the URL
                if (!$parent_value) {
                    $parent_value = 'uncategorized';
                }

                $post_link = str_replace($token, $parent_value, $post_link);
            }
        }
    }

    return $post_link;
}, 10, 2);

/**
 * Activation hook
 */
register_activation_hook(__FILE__, 'instarank_activate');
function instarank_activate() {
    global $wpdb;

    $charset_collate = $wpdb->get_charset_collate();

    // Create changes table with SQLite compatibility
    $sql = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}instarank_changes (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        post_id bigint(20) unsigned NOT NULL,
        change_type varchar(50) NOT NULL,
        field_name varchar(100) DEFAULT NULL,
        old_value longtext,
        new_value longtext,
        status varchar(20) DEFAULT 'pending',
        applied_at datetime DEFAULT NULL,
        rolled_back_at datetime DEFAULT NULL,
        metadata longtext,
        created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY post_id (post_id),
        KEY status (status),
        KEY created_at (created_at)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

    // Try to create table, catch any errors gracefully
    $result = dbDelta($sql);

    // Log any errors for debugging (only in WP_DEBUG mode)
    // Removed error_log to comply with WordPress Plugin Check standards
    // if (defined('WP_DEBUG') && WP_DEBUG && empty($result)) {
    //     error_log('InstaRank: Table creation might have failed, but continuing activation...');
    // }

    // Generate initial API key
    if (!get_option('instarank_api_key')) {
        $api_key = 'ir_' . bin2hex(random_bytes(32));
        update_option('instarank_api_key', $api_key);
    }

    // Set default settings
    $defaults = [
        'auto_approve' => false,
        'allowed_change_types' => [
            // Meta tags
            'meta_title',
            'meta_description',
            'image_alt',
            'canonical',
            'og_title',
            'og_description',
            'og_image',
            'og_url',
            'robots_noindex',
            'focus_keyword',
            'post_title',
            // Content issues
            'low_word_count',
            'duplicate_title',
            'duplicate_description',
            'missing_h1',
            'multiple_h1',
            'broken_links',
            'slow_page',
            'missing_image_alt',
            'large_image',
            'missing_structured_data',
            // Additional SEO fields
            'robots_meta',
            'meta_keywords',
            'schema_markup'
        ],
        'webhook_enabled' => true,
        'rollback_days' => 90,
        'connected' => false,
        'project_id' => ''
    ];

    foreach ($defaults as $key => $value) {
        if (get_option("instarank_{$key}") === false) {
            update_option("instarank_{$key}", $value);
        }
    }

    // Set activation timestamp
    update_option('instarank_activated_at', current_time('mysql'));

    // Flush rewrite rules for sitemap
    flush_rewrite_rules();
}

/**
 * Deactivation hook
 */
register_deactivation_hook(__FILE__, 'instarank_deactivate');
function instarank_deactivate() {
    // Notify InstaRank that plugin was deactivated
    $webhook_sender = new InstaRank_Webhook_Sender();
    $webhook_sender->send_plugin_deactivated();
}

/**
 * Initialize plugin
 */
function instarank_init() {
    return InstaRank_SEO::instance();
}

// Start the plugin
add_action('plugins_loaded', 'instarank_init');
