<?php
/**
 * Webhook Sender
 * Sends event notifications to InstaRank platform
 */

defined('ABSPATH') || exit;

class InstaRank_Webhook_Sender {

    private $webhook_url;
    private $api_key;

    public function __construct() {
        $this->webhook_url = get_option('instarank_webhook_url', INSTARANK_API_URL . '/webhooks/wordpress');
        $this->api_key = get_option('instarank_api_key', '');
    }

    /**
     * Send webhook to InstaRank platform
     */
    private function send_webhook($event_type, $data = []) {
        // Check if webhooks are enabled
        if (!get_option('instarank_webhook_enabled', true)) {
            return false;
        }

        $payload = [
            'event' => $event_type,
            'timestamp' => current_time('mysql'),
            'site_url' => get_site_url(),
            'site_name' => get_bloginfo('name'),
            'project_id' => get_option('instarank_project_id', ''),
            'data' => $data
        ];

        $args = [
            'method' => 'POST',
            'headers' => [
                'Content-Type' => 'application/json',
                'X-WordPress-API-Key' => $this->api_key,
                'X-WordPress-Site' => get_site_url(),
                'X-Event-Type' => $event_type
            ],
            'body' => json_encode($payload),
            'timeout' => 10,
            'blocking' => false // Non-blocking to avoid slowing down WordPress
        ];

        $response = wp_remote_post($this->webhook_url, $args);

        if (is_wp_error($response)) {
            // error_log('InstaRank Webhook Error: ' . $response->get_error_message());
            return false;
        }

        return true;
    }

    /**
     * Send connection established webhook
     */
    public function send_connection_established($project_id) {
        return $this->send_webhook('connection_established', [
            'project_id' => $project_id,
            'plugin_version' => INSTARANK_VERSION,
            'wp_version' => get_bloginfo('version')
        ]);
    }

    /**
     * Send sync completed webhook
     */
    public function send_sync_completed($stats) {
        return $this->send_webhook('sync_completed', $stats);
    }

    /**
     * Send change received webhook
     */
    public function send_change_received($change_data) {
        return $this->send_webhook('change_received', $change_data);
    }

    /**
     * Send change applied webhook
     */
    public function send_change_applied($change_data) {
        return $this->send_webhook('change_applied', $change_data);
    }

    /**
     * Send change rejected webhook
     */
    public function send_change_rejected($change_data) {
        return $this->send_webhook('change_rejected', $change_data);
    }

    /**
     * Send change rolled back webhook
     */
    public function send_change_rolled_back($change_data) {
        return $this->send_webhook('change_rolled_back', $change_data);
    }

    /**
     * Send settings updated webhook
     */
    public function send_settings_updated($settings) {
        return $this->send_webhook('settings_updated', $settings);
    }

    /**
     * Send security alert webhook
     */
    public function send_security_alert($alert_data) {
        return $this->send_webhook('security_alert', $alert_data);
    }

    /**
     * Send post published webhook
     */
    public function send_post_published($post_id) {
        $post = get_post($post_id);
        $detector = new InstaRank_SEO_Detector();
        $meta = $detector->get_post_meta($post_id);

        return $this->send_webhook('post_published', [
            'post_id' => $post_id,
            'title' => $post->post_title,
            'url' => get_permalink($post_id),
            'type' => $post->post_type,
            'seo' => $meta
        ]);
    }

    /**
     * Send post updated webhook
     */
    public function send_post_updated($post_id) {
        $post = get_post($post_id);
        $detector = new InstaRank_SEO_Detector();
        $meta = $detector->get_post_meta($post_id);

        return $this->send_webhook('post_updated', [
            'post_id' => $post_id,
            'title' => $post->post_title,
            'url' => get_permalink($post_id),
            'type' => $post->post_type,
            'seo' => $meta
        ]);
    }

    /**
     * Send post deleted webhook
     */
    public function send_post_deleted($post_id) {
        return $this->send_webhook('post_deleted', [
            'post_id' => $post_id
        ]);
    }

    /**
     * Send plugin activated webhook
     */
    public function send_plugin_activated() {
        return $this->send_webhook('plugin_activated', [
            'plugin_version' => INSTARANK_VERSION,
            'wp_version' => get_bloginfo('version')
        ]);
    }

    /**
     * Send plugin deactivated webhook
     */
    public function send_plugin_deactivated() {
        return $this->send_webhook('plugin_deactivated', [
            'plugin_version' => INSTARANK_VERSION
        ]);
    }

    /**
     * Send disconnected webhook
     */
    public function send_disconnected() {
        return $this->send_webhook('disconnected', [
            'plugin_version' => INSTARANK_VERSION,
            'disconnected_at' => current_time('mysql')
        ]);
    }

    /**
     * Send error webhook
     */
    public function send_error($error_data) {
        return $this->send_webhook('error', $error_data);
    }
}
