<?php
/**
 * Virtual Pages Handler
 *
 * Intercepts URL requests and renders virtual pages from the InstaRank SaaS
 * without creating physical WordPress posts.
 *
 * This class handles:
 * - URL interception via template_redirect hook
 * - Fetching page data from InstaRank API
 * - Rendering the virtual page with theme integration
 * - Caching for performance
 */

defined('ABSPATH') || exit;

class InstaRank_Virtual_Pages {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Virtual page configuration from InstaRank
     */
    private $config = null;

    /**
     * Cache group for virtual pages
     */
    const CACHE_GROUP = 'instarank_virtual';

    /**
     * Get singleton instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor - Initialize hooks
     */
    private function __construct() {
        // Only initialize if connected to InstaRank
        if (!$this->is_connected()) {
            return;
        }

        // Check virtual pages on template redirect (priority 5 to run early)
        add_action('template_redirect', [$this, 'maybe_render_virtual_page'], 5);

        // Add virtual pages sitemap endpoint
        add_action('rest_api_init', [$this, 'register_sitemap_endpoint']);

        // Filter robots.txt to include virtual sitemap
        add_filter('robots_txt', [$this, 'add_sitemap_to_robots'], 10, 2);

        // Add rewrite rules for virtual pages
        add_action('init', [$this, 'add_rewrite_rules']);
    }

    /**
     * Check if connected to InstaRank
     */
    private function is_connected() {
        $api_key = get_option('instarank_api_key', '');
        return !empty($api_key);
    }

    /**
     * Get API key
     */
    private function get_api_key() {
        return get_option('instarank_api_key', '');
    }

    /**
     * Get InstaRank API base URL
     */
    private function get_api_url() {
        return defined('INSTARANK_API_URL') ? INSTARANK_API_URL : 'https://app.instarank.com';
    }

    /**
     * Load virtual page configuration from InstaRank
     */
    private function load_config() {
        if ($this->config !== null) {
            return $this->config;
        }

        // Try to get from transient cache first
        $cached = get_transient('instarank_virtual_config');
        if ($cached !== false) {
            $this->config = $cached;
            return $this->config;
        }

        // Fetch from API
        $response = wp_remote_get(
            $this->get_api_url() . '/api/programmatic-seo/virtual-pages',
            [
                'headers' => [
                    'X-WordPress-API-Key' => $this->get_api_key(),
                ],
                'timeout' => 10,
            ]
        );

        if (is_wp_error($response)) {
            $this->config = ['enabled' => false];
            return $this->config;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($body['success']) && !empty($body['data'])) {
            $this->config = $body['data'];
        } else {
            $this->config = ['enabled' => false];
        }

        // Cache for 5 minutes
        set_transient('instarank_virtual_config', $this->config, 5 * MINUTE_IN_SECONDS);

        return $this->config;
    }

    /**
     * Check if virtual pages are enabled
     */
    public function is_enabled() {
        $config = $this->load_config();
        return !empty($config['enabled']);
    }

    /**
     * Get URL pattern from config
     */
    private function get_url_pattern() {
        $config = $this->load_config();
        return $config['url_pattern'] ?? '/{post_type}/{slug}';
    }

    /**
     * Check if current URL matches virtual page pattern
     */
    private function match_url($request_uri) {
        $pattern = $this->get_url_pattern();

        // Convert pattern to regex
        $regex_pattern = preg_quote($pattern, '#');
        $regex_pattern = str_replace('\{post_type\}', '(?P<post_type>[^/]+)', $regex_pattern);
        $regex_pattern = str_replace('\{slug\}', '(?P<slug>[^/]+)', $regex_pattern);
        $regex_pattern = '#^' . $regex_pattern . '/?$#';

        if (preg_match($regex_pattern, $request_uri, $matches)) {
            return [
                'post_type' => $matches['post_type'] ?? null,
                'slug' => $matches['slug'] ?? null,
            ];
        }

        return null;
    }

    /**
     * Maybe render a virtual page
     */
    public function maybe_render_virtual_page() {
        // Don't run in admin
        if (is_admin()) {
            return;
        }

        // Check if virtual pages are enabled
        if (!$this->is_enabled()) {
            return;
        }

        // Get current request URI
        $request_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        $request_uri = wp_parse_url($request_uri, PHP_URL_PATH);

        // Check if URL matches pattern
        $match = $this->match_url($request_uri);
        if (!$match || empty($match['slug'])) {
            return;
        }

        // Fetch virtual page from API
        $page_data = $this->fetch_virtual_page($match['slug'], $match['post_type']);

        if (!$page_data) {
            return; // Let WordPress handle 404
        }

        // Render the virtual page
        $this->render_virtual_page($page_data);
        exit;
    }

    /**
     * Fetch virtual page from InstaRank API
     */
    private function fetch_virtual_page($slug, $post_type = null) {
        // Check cache first
        $cache_key = 'vp_' . md5($slug . '_' . ($post_type ?? ''));
        $cached = wp_cache_get($cache_key, self::CACHE_GROUP);

        if ($cached !== false) {
            return $cached;
        }

        // Build API URL
        $api_url = $this->get_api_url() . '/api/programmatic-seo/virtual-pages/' . urlencode($slug);
        $params = [];
        if ($post_type) {
            $params['post_type'] = $post_type;
        }
        if (!empty($params)) {
            $api_url .= '?' . http_build_query($params);
        }

        // Fetch from API
        $response = wp_remote_get($api_url, [
            'headers' => [
                'X-WordPress-API-Key' => $this->get_api_key(),
            ],
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            // Log error only in debug mode
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('[InstaRank Virtual Pages] API error: ' . $response->get_error_message());
            }
            return null;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            return null;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (empty($body['success']) || empty($body['data'])) {
            return null;
        }

        $page_data = $body['data'];
        $cache_duration = $body['cache_duration'] ?? 3600;

        // Cache the result
        wp_cache_set($cache_key, $page_data, self::CACHE_GROUP, $cache_duration);

        return $page_data;
    }

    /**
     * Render a virtual page
     */
    private function render_virtual_page($page_data) {
        global $wp_query;

        // Set up fake post for template
        $fake_post = new stdClass();
        $fake_post->ID = -1;
        $fake_post->post_title = $page_data['title'] ?? '';
        $fake_post->post_content = $page_data['content'] ?? '';
        $fake_post->post_excerpt = $page_data['excerpt'] ?? '';
        $fake_post->post_type = $page_data['post_type'] ?? 'page';
        $fake_post->post_status = 'publish';
        $fake_post->post_date = $page_data['published_at'] ?? current_time('mysql');
        $fake_post->post_date_gmt = $page_data['published_at'] ?? current_time('mysql', 1);
        $fake_post->post_modified = $page_data['updated_at'] ?? current_time('mysql');
        $fake_post->post_modified_gmt = $page_data['updated_at'] ?? current_time('mysql', 1);
        $fake_post->post_name = $page_data['slug'] ?? '';
        $fake_post->post_author = 1;
        $fake_post->comment_status = 'closed';
        $fake_post->ping_status = 'closed';
        $fake_post->post_parent = 0;
        $fake_post->guid = home_url('/' . $page_data['slug']);
        $fake_post->menu_order = 0;
        $fake_post->post_mime_type = '';
        $fake_post->comment_count = 0;
        $fake_post->filter = 'raw';

        // Convert to WP_Post
        $fake_post = new WP_Post($fake_post);

        // Set up WordPress query vars
        $wp_query->is_single = false;
        $wp_query->is_singular = true;
        $wp_query->is_page = true;
        $wp_query->is_404 = false;
        $wp_query->is_archive = false;
        $wp_query->is_home = false;
        $wp_query->post = $fake_post;
        $wp_query->posts = [$fake_post];
        $wp_query->post_count = 1;
        $wp_query->found_posts = 1;
        $wp_query->max_num_pages = 1;
        $wp_query->queried_object = $fake_post;
        $wp_query->queried_object_id = $fake_post->ID;

        // Set global post
        $GLOBALS['post'] = $fake_post;

        // Output SEO meta tags
        $this->output_seo_meta($page_data);

        // Output schema markup
        $this->output_schema_markup($page_data);

        // Set HTTP status
        status_header(200);

        // Use theme template
        $template = get_page_template();
        if (!$template) {
            $template = get_index_template();
        }

        if ($template) {
            include $template;
        } else {
            // Fallback: basic HTML output
            $this->render_fallback_template($page_data);
        }
    }

    /**
     * Output SEO meta tags for virtual page
     */
    private function output_seo_meta($page_data) {
        add_action('wp_head', function() use ($page_data) {
            // Title
            $title = $page_data['meta_title'] ?? $page_data['title'] ?? '';
            if ($title) {
                echo '<title>' . esc_html($title) . '</title>' . "\n";
            }

            // Meta description
            $description = $page_data['meta_description'] ?? '';
            if ($description) {
                echo '<meta name="description" content="' . esc_attr($description) . '" />' . "\n";
            }

            // Canonical URL
            $canonical = $page_data['canonical_url'] ?? '';
            if ($canonical) {
                echo '<link rel="canonical" href="' . esc_url($canonical) . '" />' . "\n";
            }

            // Open Graph tags
            $og_title = $page_data['og_title'] ?? $title;
            $og_description = $page_data['og_description'] ?? $description;
            $og_image = $page_data['og_image'] ?? '';

            if ($og_title) {
                echo '<meta property="og:title" content="' . esc_attr($og_title) . '" />' . "\n";
            }
            if ($og_description) {
                echo '<meta property="og:description" content="' . esc_attr($og_description) . '" />' . "\n";
            }
            if ($og_image) {
                echo '<meta property="og:image" content="' . esc_url($og_image) . '" />' . "\n";
            }
            echo '<meta property="og:type" content="article" />' . "\n";
            $current_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
            echo '<meta property="og:url" content="' . esc_url(home_url($current_uri)) . '" />' . "\n";

            // Twitter Card
            echo '<meta name="twitter:card" content="summary_large_image" />' . "\n";
            if ($og_title) {
                echo '<meta name="twitter:title" content="' . esc_attr($og_title) . '" />' . "\n";
            }
            if ($og_description) {
                echo '<meta name="twitter:description" content="' . esc_attr($og_description) . '" />' . "\n";
            }
            if ($og_image) {
                echo '<meta name="twitter:image" content="' . esc_url($og_image) . '" />' . "\n";
            }
        }, 1);
    }

    /**
     * Output schema markup for virtual page
     */
    private function output_schema_markup($page_data) {
        if (empty($page_data['schema_markup'])) {
            return;
        }

        add_action('wp_head', function() use ($page_data) {
            $schema = $page_data['schema_markup'];
            if (is_array($schema)) {
                $schema = wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
            }
            // Schema JSON is already validated and encoded, safe to output
            echo '<script type="application/ld+json">' . wp_kses_post($schema) . '</script>' . "\n";
        }, 2);
    }

    /**
     * Render fallback template if no theme template available
     */
    private function render_fallback_template($page_data) {
        ?>
<!DOCTYPE html>
<html <?php language_attributes(); ?>>
<head>
    <meta charset="<?php bloginfo('charset'); ?>">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <?php wp_head(); ?>
</head>
<body <?php body_class(); ?>>
    <main class="instarank-virtual-page">
        <article>
            <header>
                <h1><?php echo esc_html($page_data['title'] ?? ''); ?></h1>
            </header>
            <div class="entry-content">
                <?php echo wp_kses_post($page_data['content'] ?? ''); ?>
            </div>
        </article>
    </main>
    <?php wp_footer(); ?>
</body>
</html>
        <?php
    }

    /**
     * Register sitemap endpoint
     */
    public function register_sitemap_endpoint() {
        register_rest_route('instarank/v1', '/virtual-sitemap.xml', [
            'methods' => 'GET',
            'callback' => [$this, 'serve_sitemap'],
            'permission_callback' => '__return_true',
        ]);
    }

    /**
     * Serve virtual pages sitemap
     */
    public function serve_sitemap() {
        // Fetch sitemap from InstaRank API
        $response = wp_remote_get(
            $this->get_api_url() . '/api/programmatic-seo/virtual-pages/sitemap?api_key=' . urlencode($this->get_api_key()),
            [
                'timeout' => 30,
            ]
        );

        if (is_wp_error($response)) {
            return new WP_Error('sitemap_error', 'Failed to fetch sitemap', ['status' => 500]);
        }

        $sitemap = wp_remote_retrieve_body($response);
        $content_type = wp_remote_retrieve_header($response, 'content-type');

        return new WP_REST_Response(
            $sitemap,
            200,
            [
                'Content-Type' => $content_type ?: 'application/xml',
                'Cache-Control' => 'public, max-age=3600',
            ]
        );
    }

    /**
     * Add sitemap to robots.txt
     */
    public function add_sitemap_to_robots($output, $public) {
        if ($public && $this->is_enabled()) {
            $sitemap_url = rest_url('instarank/v1/virtual-sitemap.xml');
            $output .= "\n# InstaRank Virtual Pages Sitemap\n";
            $output .= "Sitemap: " . esc_url($sitemap_url) . "\n";
        }
        return $output;
    }

    /**
     * Add rewrite rules for virtual pages
     */
    public function add_rewrite_rules() {
        if (!$this->is_enabled()) {
            return;
        }

        // Get URL pattern and add rewrite rules
        // Note: This is handled via template_redirect instead of rewrite rules
        // because we need to intercept before WordPress tries to match posts
    }

    /**
     * Clear virtual page cache
     */
    public static function clear_cache() {
        delete_transient('instarank_virtual_config');
        wp_cache_flush_group(self::CACHE_GROUP);
    }
}

// Initialize
InstaRank_Virtual_Pages::instance();
