<?php
/**
 * Spintax Engine
 * Processes spintax syntax for content variation
 *
 * Syntax supported:
 * - Basic: {option1|option2|option3}
 * - Nested: {big {house|home}|small apt}
 * - Weighted: {common=70|rare=30}
 * - Anchors: {anchor:name}{a|b} ... {anchor:name}{a|b}
 * - Variables: {{field_name}}
 *
 * @package InstaRank
 * @since 1.3.0
 */

defined('ABSPATH') || exit;

class InstaRank_Spintax_Engine {

    /**
     * Anchor values for consistent selection
     * @var array
     */
    private $anchors = [];

    /**
     * Random seed for reproducibility
     * @var int|null
     */
    private $seed = null;

    /**
     * Whether to preserve case in output
     * @var bool
     */
    private $preserve_case = true;

    /**
     * Internal random state for seeded random generation
     * @var int|null
     */
    private $random_state = null;

    /**
     * Constructor
     *
     * @param array $options Optional settings
     */
    public function __construct($options = []) {
        if (isset($options['seed'])) {
            $this->seed = (int) $options['seed'];
        }
        if (isset($options['preserve_case'])) {
            $this->preserve_case = (bool) $options['preserve_case'];
        }
        if (isset($options['anchors'])) {
            $this->anchors = (array) $options['anchors'];
        }
    }

    /**
     * Set the random seed
     *
     * @param int $seed
     * @return self
     */
    public function set_seed($seed) {
        $this->seed = (int) $seed;
        return $this;
    }

    /**
     * Process spintax and return a single variation
     *
     * @param string $text Text containing spintax
     * @param array $variables Optional variables to replace {{var}}
     * @return string Processed text
     */
    public function spin($text, $variables = []) {
        // Initialize internal random state if seed is set
        if ($this->seed !== null) {
            $this->random_state = $this->seed;
        }

        // First replace variables
        $text = $this->replace_variables($text, $variables);

        // Then process spintax
        $text = $this->process_spintax($text);

        return $text;
    }

    /**
     * Generate multiple variations
     *
     * @param string $text Text containing spintax
     * @param int $count Number of variations to generate
     * @param array $variables Optional variables
     * @return array Array of variations
     */
    public function spin_many($text, $count = 5, $variables = []) {
        $variations = [];
        $base_seed = $this->seed ?? time();

        for ($i = 0; $i < $count; $i++) {
            $this->seed = $base_seed + $i;
            $this->anchors = []; // Reset anchors for each variation
            $variations[] = $this->spin($text, $variables);
        }

        // Reset seed
        $this->seed = $base_seed;

        return $variations;
    }

    /**
     * Generate all possible variations
     *
     * @param string $text Text containing spintax
     * @param int $max_limit Maximum variations to return
     * @param array $variables Optional variables
     * @return array Array of all variations
     */
    public function spin_all($text, $max_limit = 1000, $variables = []) {
        // Replace variables first
        $text = $this->replace_variables($text, $variables);

        // Generate all combinations
        $variations = $this->generate_all_combinations($text);

        // Limit results
        if (count($variations) > $max_limit) {
            $variations = array_slice($variations, 0, $max_limit);
        }

        return array_unique($variations);
    }

    /**
     * Validate spintax syntax
     *
     * @param string $text Text to validate
     * @return array Validation result with 'valid', 'errors', 'warnings', 'stats'
     */
    public function validate($text) {
        $errors = [];
        $warnings = [];
        $stats = [
            'total_variations' => 0,
            'choice_groups' => 0,
            'max_depth' => 0,
            'has_weights' => false,
            'has_anchors' => false,
        ];

        // Check for unclosed braces
        $open_count = substr_count($text, '{');
        $close_count = substr_count($text, '}');

        if ($open_count !== $close_count) {
            $errors[] = [
                'message' => 'Mismatched braces: ' . $open_count . ' opening, ' . $close_count . ' closing',
                'position' => 0,
            ];
        }

        // Check for empty options
        if (preg_match('/\{\s*\|/', $text) || preg_match('/\|\s*\}/', $text) || preg_match('/\|\s*\|/', $text)) {
            $warnings[] = [
                'message' => 'Empty option detected in spintax',
                'suggestion' => 'Remove empty options or add content',
            ];
        }

        // Check for single-option groups
        if (preg_match('/\{[^|{}]+\}/', $text, $matches)) {
            // Make sure it's not a variable {{var}} or anchor
            foreach ($matches as $match) {
                if (!preg_match('/^\{\{.+\}\}$/', $match) && !preg_match('/^\{anchor:/', $match)) {
                    $warnings[] = [
                        'message' => 'Single option in spintax group: ' . $match,
                        'suggestion' => 'Add more options or remove braces',
                    ];
                }
            }
        }

        // Count choice groups and calculate variations
        $stats = $this->calculate_stats($text);

        return [
            'valid' => empty($errors),
            'errors' => $errors,
            'warnings' => $warnings,
            'stats' => $stats,
        ];
    }

    /**
     * Check if text contains spintax
     *
     * @param string $text
     * @return bool
     */
    public function has_spintax($text) {
        // Must have { and } and |
        // But not just {{variable}}
        $pattern = '/\{(?!\{)[^{}]*\|[^{}]*\}/';
        return (bool) preg_match($pattern, $text);
    }

    /**
     * Replace {{variable}} placeholders AND bare HTML attribute values
     *
     * Supports:
     * - {{field_name}} - Standard double brace format
     * - src="field_name" - Bare attribute values
     * - alt="field_name" - Alt text attributes
     * - title="field_name" - Title attributes
     * - href="field_name" - Link href attributes
     * - attr="{{field_name}}" - Attributes with {{}} inside
     *
     * @param string $text
     * @param array $variables
     * @return string
     */
    private function replace_variables($text, $variables) {
        if (empty($variables) || !is_array($variables)) {
            return $text;
        }

        foreach ($variables as $key => $value) {
            // Convert null to empty string
            $value = ($value === null) ? '' : $value;
            $value_str = (string) $value;

            // 1. Replace {{field_name}} patterns
            $text = str_replace('{{' . $key . '}}', $value_str, $text);

            // 2. Replace bare HTML attribute patterns
            $attributes_to_replace = ['src', 'alt', 'title', 'href', 'data-src'];

            foreach ($attributes_to_replace as $attr) {
                $pattern = '/' . preg_quote($attr, '/') . '="' . preg_quote($key, '/') . '"/';

                if (preg_match($pattern, $text)) {
                    $escaped_value = esc_attr($value_str);
                    $replacement = $attr . '="' . $escaped_value . '"';
                    $text = preg_replace($pattern, $replacement, $text);
                }
            }

            // 3. Handle attributes with {{field}} inside
            foreach ($attributes_to_replace as $attr) {
                $pattern = '/' . preg_quote($attr, '/') . '="\\{\\{' . preg_quote($key, '/') . '\\}\\}"/';

                if (preg_match($pattern, $text)) {
                    $escaped_value = esc_attr($value_str);
                    $replacement = $attr . '="' . $escaped_value . '"';
                    $text = preg_replace($pattern, $replacement, $text);
                }
            }
        }

        return $text;
    }

    /**
     * Process spintax recursively
     *
     * @param string $text
     * @return string
     */
    private function process_spintax($text) {
        // Process from innermost to outermost
        $max_iterations = 100; // Prevent infinite loops
        $iteration = 0;

        while (preg_match('/\{([^{}]+)\}/', $text) && $iteration < $max_iterations) {
            $text = preg_replace_callback('/\{([^{}]+)\}/', function($matches) {
                return $this->select_option($matches[1]);
            }, $text);
            $iteration++;
        }

        return $text;
    }

    /**
     * Select one option from a spintax group
     *
     * @param string $options_string Pipe-separated options
     * @return string Selected option
     */
    private function select_option($options_string) {
        // Check for anchor
        $anchor = null;
        if (preg_match('/^anchor:([a-zA-Z0-9_-]+)(.*)$/s', $options_string, $anchor_match)) {
            $anchor = $anchor_match[1];
            $options_string = $anchor_match[2];

            // If we've seen this anchor before, return the same value
            if (isset($this->anchors[$anchor])) {
                return $this->anchors[$anchor];
            }
        }

        // Split by pipe, but not pipes inside nested braces
        $options = $this->split_options($options_string);

        if (empty($options)) {
            return '';
        }

        // Check for weights
        $weights = [];
        $clean_options = [];
        $has_weights = false;

        foreach ($options as $option) {
            if (preg_match('/^(.+)=(\d+)$/', trim($option), $weight_match)) {
                $clean_options[] = trim($weight_match[1]);
                $weights[] = (int) $weight_match[2];
                $has_weights = true;
            } else {
                $clean_options[] = trim($option);
                $weights[] = 1; // Default weight
            }
        }

        // Select option
        if ($has_weights) {
            $selected = $this->weighted_random($clean_options, $weights);
        } else {
            $index = $this->get_random(0, count($clean_options) - 1);
            $selected = $clean_options[$index];
        }

        // Store anchor value
        if ($anchor !== null) {
            $this->anchors[$anchor] = $selected;
        }

        return $selected;
    }

    /**
     * Split options by pipe, respecting nested braces
     *
     * @param string $text
     * @return array
     */
    private function split_options($text) {
        $options = [];
        $current = '';
        $depth = 0;

        for ($i = 0; $i < strlen($text); $i++) {
            $char = $text[$i];

            if ($char === '{') {
                $depth++;
                $current .= $char;
            } elseif ($char === '}') {
                $depth--;
                $current .= $char;
            } elseif ($char === '|' && $depth === 0) {
                $options[] = $current;
                $current = '';
            } else {
                $current .= $char;
            }
        }

        if ($current !== '') {
            $options[] = $current;
        }

        return $options;
    }

    /**
     * Weighted random selection
     *
     * @param array $options
     * @param array $weights
     * @return string
     */
    private function weighted_random($options, $weights) {
        $total = array_sum($weights);
        $random = $this->get_random(1, $total);
        $cumulative = 0;

        foreach ($options as $i => $option) {
            $cumulative += $weights[$i];
            if ($random <= $cumulative) {
                return $option;
            }
        }

        return $options[count($options) - 1];
    }

    /**
     * Get a random number, using seeded PRNG if seed is set, otherwise wp_rand()
     *
     * @param int $min Minimum value
     * @param int $max Maximum value
     * @return int Random number between min and max
     */
    private function get_random($min, $max) {
        // If no seed is set, use WordPress's wp_rand()
        if ($this->random_state === null) {
            return wp_rand($min, $max);
        }

        // Use a simple Linear Congruential Generator for seeded random
        // This provides reproducible results when a seed is set
        // LCG parameters from Numerical Recipes.
        $this->random_state = (1103515245 * $this->random_state + 12345) & 0x7fffffff;

        // Scale to the desired range.
        $range = $max - $min + 1;
        return $min + ($this->random_state % $range);
    }

    /**
     * Calculate spintax statistics
     *
     * @param string $text
     * @return array
     */
    private function calculate_stats($text) {
        $stats = [
            'total_variations' => 1,
            'choice_groups' => 0,
            'max_depth' => 0,
            'has_weights' => (bool) preg_match('/=[0-9]+[|}]/', $text),
            'has_anchors' => (bool) preg_match('/\{anchor:/', $text),
        ];

        // Calculate max depth
        $depth = 0;
        $max_depth = 0;
        for ($i = 0; $i < strlen($text); $i++) {
            if ($text[$i] === '{' && ($i === 0 || $text[$i-1] !== '{')) {
                $depth++;
                $max_depth = max($max_depth, $depth);
            } elseif ($text[$i] === '}' && ($i === strlen($text) - 1 || $text[$i+1] !== '}')) {
                $depth--;
            }
        }
        $stats['max_depth'] = $max_depth;

        // Count variations (simplified - count top-level groups)
        // For accurate count, we'd need to parse the full tree
        preg_match_all('/\{([^{}]+)\}/', $text, $matches);
        foreach ($matches[1] as $group) {
            if (strpos($group, '|') !== false) {
                $options = $this->split_options($group);
                $stats['choice_groups']++;
                $stats['total_variations'] *= count($options);
            }
        }

        // Cap at reasonable number
        if ($stats['total_variations'] > 1000000) {
            $stats['total_variations'] = 1000000;
        }

        return $stats;
    }

    /**
     * Generate all possible combinations
     *
     * @param string $text
     * @return array
     */
    private function generate_all_combinations($text) {
        // Find first spintax group
        if (!preg_match('/\{([^{}]+)\}/', $text, $match, PREG_OFFSET_MATCH)) {
            return [$text];
        }

        $full_match = $match[0][0];
        $options_string = $match[1][0];
        $position = $match[0][1];

        // Get options for this group
        $options = $this->split_options($options_string);

        // For each option, recursively generate combinations
        $results = [];
        foreach ($options as $option) {
            // Handle weighted options
            $option = preg_replace('/=\d+$/', '', trim($option));

            // Replace this group with the option
            $new_text = substr($text, 0, $position) . $option . substr($text, $position + strlen($full_match));

            // Recursively process remaining groups
            $sub_results = $this->generate_all_combinations($new_text);
            $results = array_merge($results, $sub_results);
        }

        return $results;
    }

    /**
     * Static helper to quickly process spintax
     *
     * @param string $text
     * @param array $variables
     * @param int|null $seed
     * @return string
     */
    public static function process($text, $variables = [], $seed = null) {
        $engine = new self(['seed' => $seed]);
        return $engine->spin($text, $variables);
    }

    /**
     * Static helper to validate spintax
     *
     * @param string $text
     * @return array
     */
    public static function check($text) {
        $engine = new self();
        return $engine->validate($text);
    }

    /**
     * Process conditional blocks in content
     * Syntax: {if field_name}...{else}...{endif}
     * Supports: {if field == "value"}, {if field > 10}, etc.
     *
     * @param string $text
     * @param array $context Variables/fields to evaluate against
     * @return string
     */
    public function process_conditionals($text, $context = []) {
        $output = $text;
        $max_iterations = 100;

        while ($max_iterations-- > 0) {
            // Find {if condition}
            if (!preg_match('/\{if\s+([^}]+)\}/i', $output, $match)) {
                break;
            }

            $full_match = $match[0];
            $condition_str = $match[1];
            $start_pos = strpos($output, $full_match);

            // Find matching endif
            $endif_data = $this->find_matching_endif($output, $start_pos + strlen($full_match));
            $endif_pos = $endif_data['endif'];
            $else_pos = $endif_data['else'];

            if ($endif_pos === false) {
                // Unclosed if block - leave as is
                break;
            }

            // Parse and evaluate condition
            $result = $this->evaluate_condition($condition_str, $context);

            // Extract if/else content
            if ($else_pos !== false) {
                $if_content = substr($output, $start_pos + strlen($full_match), $else_pos - ($start_pos + strlen($full_match)));
                $else_content = substr($output, $else_pos + 6, $endif_pos - ($else_pos + 6)); // 6 = strlen('{else}')
            } else {
                $if_content = substr($output, $start_pos + strlen($full_match), $endif_pos - ($start_pos + strlen($full_match)));
                $else_content = '';
            }

            // Replace block with appropriate content
            $replacement = $result ? $if_content : $else_content;
            $block_end = $endif_pos + 7; // 7 = strlen('{endif}')

            $output = substr($output, 0, $start_pos) . $replacement . substr($output, $block_end);
        }

        return $output;
    }

    /**
     * Find matching endif for an if block
     *
     * @param string $text
     * @param int $start_pos
     * @return array ['endif' => position, 'else' => position or false]
     */
    private function find_matching_endif($text, $start_pos) {
        $depth = 1;
        $else_pos = false;
        $pos = $start_pos;

        while ($pos < strlen($text) && $depth > 0) {
            // Find next control structure
            $if_pos = stripos($text, '{if ', $pos);
            $else_check = stripos($text, '{else}', $pos);
            $elseif_pos = stripos($text, '{elseif ', $pos);
            $endif_check = stripos($text, '{endif}', $pos);

            // Find earliest
            $positions = array_filter([
                'if' => $if_pos,
                'else' => $else_check,
                'elseif' => $elseif_pos,
                'endif' => $endif_check,
            ], function($p) { return $p !== false; });

            if (empty($positions)) {
                break;
            }

            $next_type = array_keys($positions, min($positions))[0];
            $next_pos = min($positions);

            switch ($next_type) {
                case 'if':
                    $depth++;
                    $pos = $next_pos + 4;
                    break;
                case 'else':
                    if ($depth === 1 && $else_pos === false) {
                        $else_pos = $next_pos;
                    }
                    $pos = $next_pos + 6;
                    break;
                case 'elseif':
                    // Treat elseif as else + if for depth tracking
                    $pos = $next_pos + 8;
                    break;
                case 'endif':
                    $depth--;
                    if ($depth === 0) {
                        return ['endif' => $next_pos, 'else' => $else_pos];
                    }
                    $pos = $next_pos + 7;
                    break;
            }
        }

        return ['endif' => false, 'else' => false];
    }

    /**
     * Evaluate a condition string
     *
     * @param string $condition_str
     * @param array $context
     * @return bool
     */
    private function evaluate_condition($condition_str, $context) {
        $condition = trim($condition_str);

        // Handle negation
        $negate = false;
        if (strpos($condition, '!') === 0) {
            $negate = true;
            $condition = trim(substr($condition, 1));
        }

        // Try different operators
        $operators = [
            '==' => function($a, $b) { return $a == $b; },
            '!=' => function($a, $b) { return $a != $b; },
            '>=' => function($a, $b) { return (float)$a >= (float)$b; },
            '<=' => function($a, $b) { return (float)$a <= (float)$b; },
            '>' => function($a, $b) { return (float)$a > (float)$b; },
            '<' => function($a, $b) { return (float)$a < (float)$b; },
        ];

        foreach ($operators as $op => $func) {
            if (strpos($condition, $op) !== false) {
                $parts = explode($op, $condition, 2);
                $field = trim($parts[0]);
                $value = $this->parse_condition_value(trim($parts[1]));
                $field_value = $this->get_context_value($context, $field);
                $result = $func($field_value, $value);
                return $negate ? !$result : $result;
            }
        }

        // Check for word operators
        $word_operators = [
            ' contains ' => function($haystack, $needle) {
                if (is_array($haystack)) return in_array($needle, $haystack);
                return stripos((string)$haystack, (string)$needle) !== false;
            },
            ' starts_with ' => function($str, $prefix) {
                return stripos((string)$str, (string)$prefix) === 0;
            },
            ' ends_with ' => function($str, $suffix) {
                $len = strlen((string)$suffix);
                return $len === 0 || substr((string)$str, -$len) === (string)$suffix;
            },
            ' is_empty' => function($val) {
                return empty($val) || $val === '' || $val === null;
            },
            ' is_not_empty' => function($val) {
                return !empty($val) && $val !== '' && $val !== null;
            },
        ];

        foreach ($word_operators as $op => $func) {
            if (stripos($condition, $op) !== false) {
                $parts = preg_split('/' . preg_quote($op, '/') . '/i', $condition, 2);
                $field = trim($parts[0]);
                $field_value = $this->get_context_value($context, $field);

                if (in_array($op, [' is_empty', ' is_not_empty'])) {
                    $result = $func($field_value);
                } else {
                    $value = $this->parse_condition_value(trim($parts[1] ?? ''));
                    $result = $func($field_value, $value);
                }
                return $negate ? !$result : $result;
            }
        }

        // Simple field existence check
        $field_value = $this->get_context_value($context, $condition);
        $result = !empty($field_value) && $field_value !== '' && $field_value !== null && $field_value !== false;

        return $negate ? !$result : $result;
    }

    /**
     * Parse a value from condition string
     *
     * @param string $value_str
     * @return mixed
     */
    private function parse_condition_value($value_str) {
        $trimmed = trim($value_str);

        // Handle quoted strings
        if ((substr($trimmed, 0, 1) === '"' && substr($trimmed, -1) === '"') ||
            (substr($trimmed, 0, 1) === "'" && substr($trimmed, -1) === "'")) {
            return substr($trimmed, 1, -1);
        }

        // Handle booleans
        if (strtolower($trimmed) === 'true') return true;
        if (strtolower($trimmed) === 'false') return false;
        if (strtolower($trimmed) === 'null') return null;

        // Handle numbers
        if (is_numeric($trimmed)) {
            return strpos($trimmed, '.') !== false ? (float)$trimmed : (int)$trimmed;
        }

        return $trimmed;
    }

    /**
     * Get a value from context, supporting dot notation
     *
     * @param array $context
     * @param string $path
     * @return mixed
     */
    private function get_context_value($context, $path) {
        $parts = explode('.', $path);
        $value = $context;

        foreach ($parts as $part) {
            if (is_array($value) && isset($value[$part])) {
                $value = $value[$part];
            } elseif (is_object($value) && isset($value->$part)) {
                $value = $value->$part;
            } else {
                return null;
            }
        }

        return $value;
    }

    /**
     * Process switch/case blocks
     * Syntax: {switch field}{case "value1"}...{case "value2"}...{default}...{endswitch}
     *
     * @param string $text
     * @param array $context
     * @return string
     */
    public function process_switch($text, $context = []) {
        $output = $text;

        while (preg_match('/\{switch\s+([^}]+)\}([\s\S]*?)\{endswitch\}/i', $output, $match)) {
            $full_match = $match[0];
            $field_name = trim($match[1]);
            $switch_content = $match[2];

            $field_value = $this->get_context_value($context, $field_name);
            $replacement = '';
            $matched = false;

            // Parse cases
            if (preg_match_all('/\{case\s+([^}]+)\}([\s\S]*?)(?=\{case|\{default\}|\{endswitch\})/i', $switch_content, $cases, PREG_SET_ORDER)) {
                foreach ($cases as $case) {
                    if ($matched) continue;

                    $case_value = $this->parse_condition_value(trim($case[1]));
                    if ($field_value == $case_value) {
                        $replacement = $case[2];
                        $matched = true;
                    }
                }
            }

            // Check for default
            if (!$matched && preg_match('/\{default\}([\s\S]*?)(?=\{endswitch\})/i', $switch_content, $default_match)) {
                $replacement = $default_match[1];
            }

            $output = str_replace($full_match, $replacement, $output);
        }

        return $output;
    }

    /**
     * Process all template features: conditionals, switch, variables, and spintax
     *
     * @param string $text
     * @param array $variables
     * @return string
     */
    public function process_all($text, $variables = []) {
        // Initialize random state
        if ($this->seed !== null) {
            $this->random_state = $this->seed;
        }

        // 1. Process switch blocks first
        $text = $this->process_switch($text, $variables);

        // 2. Process conditionals
        $text = $this->process_conditionals($text, $variables);

        // 3. Replace variables
        $text = $this->replace_variables($text, $variables);

        // 4. Process spintax
        $text = $this->process_spintax($text);

        return $text;
    }

    /**
     * Static helper to process all template features
     *
     * @param string $text
     * @param array $variables
     * @param int|null $seed
     * @return string
     */
    public static function process_template($text, $variables = [], $seed = null) {
        $engine = new self(['seed' => $seed]);
        return $engine->process_all($text, $variables);
    }
}
