<?php
/**
 * XML Sitemap Generator
 *
 * Generates XML sitemaps for posts, pages, and taxonomies
 * with caching and performance optimization
 *
 * @package InstaRank
 * @since 1.4.0
 */

defined('ABSPATH') || exit;

class InstaRank_Sitemap_Generator {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Max URLs per sitemap
     */
    const MAX_URLS_PER_SITEMAP = 2000;

    /**
     * Cache duration in seconds (24 hours)
     */
    const CACHE_DURATION = 86400;

    /**
     * Get singleton instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('init', [$this, 'add_rewrite_rules']);
        add_action('template_redirect', [$this, 'handle_sitemap_request']);
        add_action('save_post', [$this, 'invalidate_cache']);
        add_action('deleted_post', [$this, 'invalidate_cache']);
        add_action('created_term', [$this, 'invalidate_cache']);
        add_action('edited_term', [$this, 'invalidate_cache']);
        add_action('deleted_term', [$this, 'invalidate_cache']);
    }

    /**
     * Add rewrite rules for sitemaps
     */
    public function add_rewrite_rules() {
        // Main sitemap index
        add_rewrite_rule(
            '^sitemap\.xml$',
            'index.php?instarank_sitemap=index',
            'top'
        );

        // Post type sitemaps
        add_rewrite_rule(
            '^sitemap-([a-z0-9_-]+)\.xml$',
            'index.php?instarank_sitemap=post_type&post_type=$matches[1]',
            'top'
        );

        // Taxonomy sitemaps
        add_rewrite_rule(
            '^sitemap-tax-([a-z0-9_-]+)\.xml$',
            'index.php?instarank_sitemap=taxonomy&taxonomy=$matches[1]',
            'top'
        );

        // XSL stylesheet
        add_rewrite_rule(
            '^sitemap\.xsl$',
            'index.php?instarank_sitemap=xsl',
            'top'
        );

        // Add query vars
        add_filter('query_vars', function($vars) {
            $vars[] = 'instarank_sitemap';
            $vars[] = 'post_type';
            $vars[] = 'taxonomy';
            return $vars;
        });
    }

    /**
     * Handle sitemap requests
     */
    public function handle_sitemap_request() {
        $sitemap_type = get_query_var('instarank_sitemap');

        if (!$sitemap_type) {
            return;
        }

        // Check if sitemaps are enabled
        if (!$this->is_enabled()) {
            wp_die(esc_html__('Sitemaps are disabled', 'instarank'), 404);
        }

        switch ($sitemap_type) {
            case 'index':
                $this->render_sitemap_index();
                break;
            case 'post_type':
                $post_type = get_query_var('post_type');
                $this->render_post_type_sitemap($post_type);
                break;
            case 'taxonomy':
                $taxonomy = get_query_var('taxonomy');
                $this->render_taxonomy_sitemap($taxonomy);
                break;
            case 'xsl':
                $this->render_xsl_stylesheet();
                break;
        }

        exit;
    }

    /**
     * Render sitemap index
     */
    private function render_sitemap_index() {
        $cache_key = 'instarank_sitemap_index';
        $cached = get_transient($cache_key);

        if ($cached !== false) {
            $this->output_xml($cached);
            return;
        }

        $sitemaps = [];

        // Get enabled post types
        $post_types = $this->get_enabled_post_types();
        foreach ($post_types as $post_type) {
            $count = wp_count_posts($post_type)->publish;
            if ($count > 0) {
                $sitemaps[] = [
                    'loc' => home_url("/sitemap-{$post_type}.xml"),
                    'lastmod' => $this->get_post_type_lastmod($post_type),
                ];
            }
        }

        // Get enabled taxonomies
        $taxonomies = $this->get_enabled_taxonomies();
        foreach ($taxonomies as $taxonomy) {
            $count = wp_count_terms(['taxonomy' => $taxonomy, 'hide_empty' => true]);
            if ($count > 0) {
                $sitemaps[] = [
                    'loc' => home_url("/sitemap-tax-{$taxonomy}.xml"),
                    'lastmod' => $this->get_taxonomy_lastmod($taxonomy),
                ];
            }
        }

        // Build XML
        $xml = $this->build_sitemap_index($sitemaps);

        // Cache
        set_transient($cache_key, $xml, self::CACHE_DURATION);

        $this->output_xml($xml);
    }

    /**
     * Render post type sitemap
     */
    private function render_post_type_sitemap($post_type) {
        if (!post_type_exists($post_type) || !in_array($post_type, $this->get_enabled_post_types())) {
            wp_die(esc_html__('Invalid post type', 'instarank'), 404);
        }

        $cache_key = "instarank_sitemap_post_{$post_type}";
        $cached = get_transient($cache_key);

        if ($cached !== false) {
            $this->output_xml($cached);
            return;
        }

        // Get posts
        $args = [
            'post_type' => $post_type,
            'post_status' => 'publish',
            'posts_per_page' => self::MAX_URLS_PER_SITEMAP,
            'orderby' => 'modified',
            'order' => 'DESC',
            'no_found_rows' => true,
            'update_post_meta_cache' => false,
            'update_post_term_cache' => false,
        ];

        // Exclude noindex posts
        // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Necessary for filtering noindex pages
        $args['meta_query'] = [
            'relation' => 'OR',
            [
                'key' => 'instarank_robots_noindex',
                'compare' => 'NOT EXISTS',
            ],
            [
                'key' => 'instarank_robots_noindex',
                'value' => '1',
                'compare' => '!=',
            ],
        ];

        $query = new WP_Query($args);
        $urls = [];

        foreach ($query->posts as $post) {
            $urls[] = [
                'loc' => get_permalink($post->ID),
                'lastmod' => mysql2date('c', $post->post_modified_gmt, false),
                'priority' => $this->get_post_priority($post),
                'changefreq' => $this->get_post_changefreq($post),
            ];
        }

        $xml = $this->build_urlset($urls);
        set_transient($cache_key, $xml, self::CACHE_DURATION);

        $this->output_xml($xml);
    }

    /**
     * Render taxonomy sitemap
     */
    private function render_taxonomy_sitemap($taxonomy) {
        if (!taxonomy_exists($taxonomy) || !in_array($taxonomy, $this->get_enabled_taxonomies())) {
            wp_die(esc_html__('Invalid taxonomy', 'instarank'), 404);
        }

        $cache_key = "instarank_sitemap_tax_{$taxonomy}";
        $cached = get_transient($cache_key);

        if ($cached !== false) {
            $this->output_xml($cached);
            return;
        }

        // Get terms
        $terms = get_terms([
            'taxonomy' => $taxonomy,
            'hide_empty' => true,
            'number' => self::MAX_URLS_PER_SITEMAP,
        ]);

        $urls = [];
        foreach ($terms as $term) {
            $urls[] = [
                'loc' => get_term_link($term),
                'lastmod' => gmdate('c'),
                'priority' => '0.6',
                'changefreq' => 'weekly',
            ];
        }

        $xml = $this->build_urlset($urls);
        set_transient($cache_key, $xml, self::CACHE_DURATION);

        $this->output_xml($xml);
    }

    /**
     * Build sitemap index XML
     */
    private function build_sitemap_index($sitemaps) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>';
        $xml .= '<?xml-stylesheet type="text/xsl" href="' . home_url('/sitemap.xsl') . '"?>';
        $xml .= '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';

        foreach ($sitemaps as $sitemap) {
            $xml .= '<sitemap>';
            $xml .= '<loc>' . esc_url($sitemap['loc']) . '</loc>';
            if (!empty($sitemap['lastmod'])) {
                $xml .= '<lastmod>' . esc_xml($sitemap['lastmod']) . '</lastmod>';
            }
            $xml .= '</sitemap>';
        }

        $xml .= '</sitemapindex>';

        return $xml;
    }

    /**
     * Build urlset XML
     */
    private function build_urlset($urls) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>';
        $xml .= '<?xml-stylesheet type="text/xsl" href="' . home_url('/sitemap.xsl') . '"?>';
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';

        foreach ($urls as $url) {
            $xml .= '<url>';
            $xml .= '<loc>' . esc_url($url['loc']) . '</loc>';
            if (!empty($url['lastmod'])) {
                $xml .= '<lastmod>' . esc_xml($url['lastmod']) . '</lastmod>';
            }
            if (!empty($url['changefreq'])) {
                $xml .= '<changefreq>' . esc_xml($url['changefreq']) . '</changefreq>';
            }
            if (!empty($url['priority'])) {
                $xml .= '<priority>' . esc_xml($url['priority']) . '</priority>';
            }
            $xml .= '</url>';
        }

        $xml .= '</urlset>';

        return $xml;
    }

    /**
     * Render XSL stylesheet
     */
    private function render_xsl_stylesheet() {
        header('Content-Type: text/xsl; charset=UTF-8');
        echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        ?>
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns:sitemap="http://www.sitemaps.org/schemas/sitemap/0.9">
<xsl:output method="html" indent="yes" encoding="UTF-8"/>
<xsl:template match="/">
<html>
<head>
    <title>XML Sitemap - <?php echo esc_html(get_bloginfo('name')); ?></title>
    <style>
        body { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif; margin: 40px; background: #f5f5f5; }
        .container { max-width: 1200px; margin: 0 auto; background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        h1 { color: #1e3a8a; margin: 0 0 10px; }
        .description { color: #64748b; margin-bottom: 30px; }
        table { width: 100%; border-collapse: collapse; }
        th { background: #1e3a8a; color: white; padding: 12px; text-align: left; font-weight: 600; }
        td { padding: 12px; border-bottom: 1px solid #e2e8f0; }
        tr:hover { background: #f8fafc; }
        .url { color: #2563eb; text-decoration: none; }
        .url:hover { text-decoration: underline; }
        .stats { display: flex; gap: 20px; margin-bottom: 30px; }
        .stat { background: #f1f5f9; padding: 15px 20px; border-radius: 6px; flex: 1; }
        .stat-value { font-size: 24px; font-weight: bold; color: #1e3a8a; }
        .stat-label { font-size: 14px; color: #64748b; margin-top: 5px; }
    </style>
</head>
<body>
    <div class="container">
        <h1><?php echo esc_html(get_bloginfo('name')); ?> - XML Sitemap</h1>
        <p class="description">This is an XML sitemap generated by InstaRank SEO. It helps search engines discover and index your content.</p>

        <div class="stats">
            <div class="stat">
                <div class="stat-value"><xsl:value-of select="count(sitemap:urlset/sitemap:url)"/><xsl:value-of select="count(sitemap:sitemapindex/sitemap:sitemap)"/></div>
                <div class="stat-label">Total URLs</div>
            </div>
        </div>

        <xsl:choose>
            <xsl:when test="sitemap:sitemapindex">
                <table>
                    <tr>
                        <th>Sitemap</th>
                        <th>Last Modified</th>
                    </tr>
                    <xsl:for-each select="sitemap:sitemapindex/sitemap:sitemap">
                        <tr>
                            <td><a class="url" href="{sitemap:loc}"><xsl:value-of select="sitemap:loc"/></a></td>
                            <td><xsl:value-of select="sitemap:lastmod"/></td>
                        </tr>
                    </xsl:for-each>
                </table>
            </xsl:when>
            <xsl:otherwise>
                <table>
                    <tr>
                        <th>URL</th>
                        <th>Last Modified</th>
                        <th>Change Frequency</th>
                        <th>Priority</th>
                    </tr>
                    <xsl:for-each select="sitemap:urlset/sitemap:url">
                        <tr>
                            <td><a class="url" href="{sitemap:loc}"><xsl:value-of select="sitemap:loc"/></a></td>
                            <td><xsl:value-of select="sitemap:lastmod"/></td>
                            <td><xsl:value-of select="sitemap:changefreq"/></td>
                            <td><xsl:value-of select="sitemap:priority"/></td>
                        </tr>
                    </xsl:for-each>
                </table>
            </xsl:otherwise>
        </xsl:choose>
    </div>
</body>
</html>
</xsl:template>
</xsl:stylesheet>
        <?php
    }

    /**
     * Output XML with proper headers
     */
    private function output_xml($xml) {
        header('Content-Type: application/xml; charset=UTF-8');
        header('X-Robots-Tag: noindex, follow');
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- XML is generated internally and properly formatted
        echo $xml;
    }

    /**
     * Invalidate cache
     */
    public function invalidate_cache() {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_instarank_sitemap_%'");
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_instarank_sitemap_%'");

        // Note: Webhook notification removed - sitemap cache invalidation doesn't require external notification
    }

    /**
     * Check if sitemaps are enabled
     */
    private function is_enabled() {
        return (bool) get_option('instarank_sitemap_enabled', true);
    }

    /**
     * Get enabled post types
     */
    private function get_enabled_post_types() {
        $enabled = get_option('instarank_sitemap_post_types', ['post', 'page']);
        $public_post_types = get_post_types(['public' => true], 'names');

        return array_intersect($enabled, $public_post_types);
    }

    /**
     * Get enabled taxonomies
     */
    private function get_enabled_taxonomies() {
        $enabled = get_option('instarank_sitemap_taxonomies', ['category', 'post_tag']);
        $public_taxonomies = get_taxonomies(['public' => true], 'names');

        return array_intersect($enabled, $public_taxonomies);
    }

    /**
     * Get post type last modified date
     */
    private function get_post_type_lastmod($post_type) {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $lastmod = $wpdb->get_var($wpdb->prepare(
            "SELECT post_modified_gmt FROM {$wpdb->posts} WHERE post_type = %s AND post_status = 'publish' ORDER BY post_modified_gmt DESC LIMIT 1",
            $post_type
        ));

        return $lastmod ? mysql2date('c', $lastmod, false) : gmdate('c');
    }

    /**
     * Get taxonomy last modified date
     */
    private function get_taxonomy_lastmod($taxonomy) {
        return gmdate('c');
    }

    /**
     * Get post priority
     */
    private function get_post_priority($post) {
        if (is_front_page($post->ID)) {
            return '1.0';
        }
        if ($post->post_type === 'page') {
            return '0.8';
        }
        return '0.6';
    }

    /**
     * Get post change frequency
     */
    private function get_post_changefreq($post) {
        $age_days = (time() - strtotime($post->post_modified)) / DAY_IN_SECONDS;

        if ($age_days < 7) {
            return 'daily';
        } elseif ($age_days < 30) {
            return 'weekly';
        } else {
            return 'monthly';
        }
    }

    /**
     * Manual regeneration
     */
    public function regenerate() {
        $this->invalidate_cache();
        return true;
    }
}

// Initialize
InstaRank_Sitemap_Generator::instance();
