<?php
/**
 * SEO Plugin Detector
 * Detects and integrates with popular SEO plugins
 */

defined('ABSPATH') || exit;

class InstaRank_SEO_Detector {

    /**
     * Detect active SEO plugin
     *
     * @return string Plugin identifier (yoast, rankmath, aioseo, or none)
     */
    public function get_active_seo_plugin() {
        if (defined('WPSEO_VERSION')) {
            return 'yoast';
        } elseif (class_exists('RankMath')) {
            return 'rankmath';
        } elseif (defined('AIOSEO_VERSION') || class_exists('AIOSEO')) {
            return 'aioseo';
        }
        return 'none';
    }

    /**
     * Get SEO plugin name
     *
     * @return string Plugin display name
     */
    public function get_seo_plugin_name() {
        $plugin = $this->get_active_seo_plugin();

        $names = [
            'yoast' => 'Yoast SEO',
            'rankmath' => 'Rank Math',
            'aioseo' => 'All in One SEO',
            'none' => 'None'
        ];

        return $names[$plugin] ?? 'Unknown';
    }

    /**
     * Get post SEO meta based on active plugin
     *
     * @param int $post_id Post ID
     * @return array SEO meta data
     */
    public function get_post_meta($post_id) {
        $plugin = $this->get_active_seo_plugin();

        switch ($plugin) {
            case 'yoast':
                return $this->get_yoast_meta($post_id);

            case 'rankmath':
                return $this->get_rankmath_meta($post_id);

            case 'aioseo':
                return $this->get_aioseo_meta($post_id);

            default:
                return $this->get_default_meta($post_id);
        }
    }

    /**
     * Get Yoast SEO meta
     */
    private function get_yoast_meta($post_id) {
        return [
            'title' => get_post_meta($post_id, '_yoast_wpseo_title', true),
            'description' => get_post_meta($post_id, '_yoast_wpseo_metadesc', true),
            'canonical' => get_post_meta($post_id, '_yoast_wpseo_canonical', true),
            'focus_keyword' => get_post_meta($post_id, '_yoast_wpseo_focuskw', true),
            'noindex' => get_post_meta($post_id, '_yoast_wpseo_meta-robots-noindex', true),
            'nofollow' => get_post_meta($post_id, '_yoast_wpseo_meta-robots-nofollow', true),
            'og_title' => get_post_meta($post_id, '_yoast_wpseo_opengraph-title', true),
            'og_description' => get_post_meta($post_id, '_yoast_wpseo_opengraph-description', true),
            'og_image' => get_post_meta($post_id, '_yoast_wpseo_opengraph-image', true),
            'og_url' => get_post_meta($post_id, '_yoast_wpseo_opengraph-url', true)
        ];
    }

    /**
     * Get Rank Math meta
     */
    private function get_rankmath_meta($post_id) {
        return [
            'title' => get_post_meta($post_id, 'rank_math_title', true),
            'description' => get_post_meta($post_id, 'rank_math_description', true),
            'canonical' => get_post_meta($post_id, 'rank_math_canonical_url', true),
            'focus_keyword' => get_post_meta($post_id, 'rank_math_focus_keyword', true),
            'robots' => get_post_meta($post_id, 'rank_math_robots', true),
            'og_title' => get_post_meta($post_id, 'rank_math_facebook_title', true),
            'og_description' => get_post_meta($post_id, 'rank_math_facebook_description', true),
            'og_image' => get_post_meta($post_id, 'rank_math_facebook_image', true),
            'og_url' => get_post_meta($post_id, 'rank_math_facebook_url', true)
        ];
    }

    /**
     * Get All in One SEO meta
     */
    private function get_aioseo_meta($post_id) {
        return [
            'title' => get_post_meta($post_id, '_aioseo_title', true),
            'description' => get_post_meta($post_id, '_aioseo_description', true),
            'canonical' => get_post_meta($post_id, '_aioseo_canonical_url', true),
            'og_title' => get_post_meta($post_id, '_aioseo_og_title', true),
            'og_description' => get_post_meta($post_id, '_aioseo_og_description', true),
            'og_image' => get_post_meta($post_id, '_aioseo_og_image_url', true),
            'og_url' => get_post_meta($post_id, '_aioseo_og_url', true),
            'robots_noindex' => get_post_meta($post_id, '_aioseo_robots_noindex', true)
        ];
    }

    /**
     * Get default meta (no SEO plugin)
     */
    private function get_default_meta($post_id) {
        $post = get_post($post_id);

        return [
            'title' => get_the_title($post_id),
            'description' => has_excerpt($post_id) ? get_the_excerpt($post_id) : wp_trim_words($post->post_content, 30),
            'canonical' => get_permalink($post_id)
        ];
    }

    /**
     * Get meta field names for active SEO plugin
     *
     * @return array Field name mapping
     */
    public function get_meta_field_names() {
        $plugin = $this->get_active_seo_plugin();

        $fields = [
            'yoast' => [
                'title' => '_yoast_wpseo_title',
                'description' => '_yoast_wpseo_metadesc',
                'canonical' => '_yoast_wpseo_canonical',
                'focus_keyword' => '_yoast_wpseo_focuskw',
                'noindex' => '_yoast_wpseo_meta-robots-noindex',
                'nofollow' => '_yoast_wpseo_meta-robots-nofollow',
                'og_title' => '_yoast_wpseo_opengraph-title',
                'og_description' => '_yoast_wpseo_opengraph-description',
                'og_image' => '_yoast_wpseo_opengraph-image',
                'og_url' => '_yoast_wpseo_opengraph-url'
            ],
            'rankmath' => [
                'title' => 'rank_math_title',
                'description' => 'rank_math_description',
                'canonical' => 'rank_math_canonical_url',
                'focus_keyword' => 'rank_math_focus_keyword',
                'robots' => 'rank_math_robots',
                'og_title' => 'rank_math_facebook_title',
                'og_description' => 'rank_math_facebook_description',
                'og_image' => 'rank_math_facebook_image',
                'og_url' => 'rank_math_facebook_url'
            ],
            'aioseo' => [
                'title' => '_aioseo_title',
                'description' => '_aioseo_description',
                'canonical' => '_aioseo_canonical_url',
                'og_title' => '_aioseo_og_title',
                'og_description' => '_aioseo_og_description',
                'og_image' => '_aioseo_og_image_url',
                'og_url' => '_aioseo_og_url',
                'robots_noindex' => '_aioseo_robots_noindex'
            ]
        ];

        return $fields[$plugin] ?? [];
    }

    /**
     * Get term (category/tag) SEO meta based on active plugin
     *
     * @param int $term_id Term ID
     * @param string $taxonomy Taxonomy name (category, post_tag, etc.)
     * @return array SEO meta data
     */
    public function get_term_meta($term_id, $taxonomy = 'category') {
        $plugin = $this->get_active_seo_plugin();

        switch ($plugin) {
            case 'yoast':
                return $this->get_yoast_term_meta($term_id, $taxonomy);

            case 'rankmath':
                return $this->get_rankmath_term_meta($term_id, $taxonomy);

            case 'aioseo':
                return $this->get_aioseo_term_meta($term_id, $taxonomy);

            default:
                return $this->get_default_term_meta($term_id, $taxonomy);
        }
    }

    /**
     * Get Yoast SEO term meta
     */
    private function get_yoast_term_meta($term_id, $taxonomy) {
        return [
            'title' => get_term_meta($term_id, '_yoast_wpseo_title', true),
            'description' => get_term_meta($term_id, '_yoast_wpseo_desc', true),
            'canonical' => get_term_meta($term_id, '_yoast_wpseo_canonical', true),
            'noindex' => get_term_meta($term_id, '_yoast_wpseo_noindex', true)
        ];
    }

    /**
     * Get Rank Math term meta
     */
    private function get_rankmath_term_meta($term_id, $taxonomy) {
        return [
            'title' => get_term_meta($term_id, 'rank_math_title', true),
            'description' => get_term_meta($term_id, 'rank_math_description', true),
            'canonical' => get_term_meta($term_id, 'rank_math_canonical_url', true),
            'robots' => get_term_meta($term_id, 'rank_math_robots', true)
        ];
    }

    /**
     * Get All in One SEO term meta
     */
    private function get_aioseo_term_meta($term_id, $taxonomy) {
        return [
            'title' => get_term_meta($term_id, '_aioseo_title', true),
            'description' => get_term_meta($term_id, '_aioseo_description', true),
            'canonical' => get_term_meta($term_id, '_aioseo_canonical_url', true)
        ];
    }

    /**
     * Get default term meta (no SEO plugin)
     */
    private function get_default_term_meta($term_id, $taxonomy) {
        $term = get_term($term_id, $taxonomy);

        if (is_wp_error($term)) {
            return [];
        }

        return [
            'title' => $term->name,
            'description' => $term->description
        ];
    }

    /**
     * Get capabilities
     *
     * @return array Plugin capabilities
     */
    public function get_capabilities() {
        return [
            'seo_plugin' => $this->get_active_seo_plugin(),
            'seo_plugin_name' => $this->get_seo_plugin_name(),
            'can_edit_posts' => current_user_can('edit_posts'),
            'can_edit_pages' => current_user_can('edit_pages'),
            'can_upload_files' => current_user_can('upload_files'),
            'can_manage_options' => current_user_can('manage_options')
        ];
    }
}
