<?php
/**
 * Schema Injector Class
 * Injects JSON-LD schema markup into wp_head when SEO plugins don't handle it
 */

defined('ABSPATH') || exit;

class InstaRank_Schema_Injector {

    /**
     * Constructor
     */
    public function __construct() {
        // Hook into wp_head with high priority (after SEO plugins)
        add_action('wp_head', [$this, 'inject_schema'], 99);
    }

    /**
     * Inject schema markup into the page head
     */
    public function inject_schema() {
        // Only inject on singular posts/pages
        if (!is_singular()) {
            return;
        }

        $post_id = get_the_ID();
        if (!$post_id) {
            return;
        }

        // Get stored schema JSON-LD
        $schema_json = get_post_meta($post_id, 'instarank_schema_json_ld', true);

        if (empty($schema_json)) {
            return;
        }

        // Check if an SEO plugin is already handling schema
        if ($this->has_seo_plugin_schema()) {
            // SEO plugin is handling it, but we still output our custom schema
            // as additional structured data (most search engines support multiple schemas)
            // However, we should check if the schema is redundant
        }

        // Validate JSON before outputting
        $schema_data = json_decode($schema_json, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            // Invalid JSON, don't output
            return;
        }

        // Output the schema markup
        // Note: Schema JSON is already validated and doesn't need escaping as it's within a JSON-LD script tag
        echo '<!-- InstaRank Schema Markup -->' . "\n";
        echo '<script type="application/ld+json">' . "\n";
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Schema JSON is validated before output
        echo $schema_json . "\n";
        echo '</script>' . "\n";
    }

    /**
     * Check if an SEO plugin is actively outputting schema
     * Returns true if we should defer to the SEO plugin
     */
    private function has_seo_plugin_schema() {
        // For now, we always inject our schema as additional structured data
        // Search engines can handle multiple schema blocks
        // In the future, we could add logic to detect if the SEO plugin
        // is already outputting the exact same schema type

        // Check if Yoast SEO is active and handling schema
        if (defined('WPSEO_VERSION')) {
            // Yoast outputs schema, but we can add additional schema
            // return true; // Uncomment to defer to Yoast completely
        }

        // Check if Rank Math is active
        if (class_exists('RankMath')) {
            // Rank Math outputs schema, but we can add additional schema
            // return true; // Uncomment to defer to Rank Math completely
        }

        // Check if All in One SEO is active
        if (defined('AIOSEO_VERSION') || class_exists('AIOSEO\\Plugin\\AIOSEO')) {
            // AIOSEO outputs schema, but we can add additional schema
            // return true; // Uncomment to defer to AIOSEO completely
        }

        // No SEO plugin detected or we want to add additional schema
        return false;
    }

    /**
     * Get the schema type from JSON-LD
     */
    private function get_schema_type($schema_json) {
        $schema_data = json_decode($schema_json, true);
        if (is_array($schema_data) && isset($schema_data['@type'])) {
            return $schema_data['@type'];
        }
        return null;
    }

    /**
     * Check if this schema type is likely already handled by SEO plugin
     */
    private function is_schema_type_handled_by_plugin($schema_type) {
        // Common schema types that SEO plugins auto-generate
        $auto_generated_types = ['WebSite', 'WebPage', 'BreadcrumbList'];

        if (in_array($schema_type, $auto_generated_types, true)) {
            // These are commonly auto-generated, but we can still add
            // more specific versions (e.g., Article instead of WebPage)
            return false;
        }

        return false;
    }

    /**
     * Remove schema markup from a post
     * Useful for rollback functionality
     */
    public static function remove_schema($post_id) {
        delete_post_meta($post_id, 'instarank_schema_json_ld');
        return true;
    }

    /**
     * Get schema markup for a post
     * Returns the JSON-LD string or null if not set
     */
    public static function get_schema($post_id) {
        return get_post_meta($post_id, 'instarank_schema_json_ld', true);
    }

    /**
     * Update schema markup for a post
     * Validates JSON before saving
     */
    public static function update_schema($post_id, $schema_json) {
        // Validate JSON
        $schema_data = json_decode($schema_json, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('invalid_json', 'Invalid JSON-LD schema');
        }

        // Validate has @context and @type
        if (!isset($schema_data['@context']) || !isset($schema_data['@type'])) {
            return new WP_Error('invalid_schema', 'Schema must have @context and @type');
        }

        // Save the schema
        update_post_meta($post_id, 'instarank_schema_json_ld', $schema_json);
        return true;
    }
}

// Initialize the schema injector
new InstaRank_Schema_Injector();
