<?php
/**
 * Schema Generator
 *
 * Auto-generates common schema types for WordPress content
 *
 * @package InstaRank
 * @since 1.4.0
 */

defined('ABSPATH') || exit;

class InstaRank_Schema_Generator {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        // Hook into wp_head to output auto-generated schemas
        add_action('wp_head', [$this, 'output_schemas'], 2);
    }

    /**
     * Output all schemas
     */
    public function output_schemas() {
        $schemas = [];

        // Always add Organization/Website schema
        $schemas[] = $this->get_website_schema();

        // Add page-specific schemas
        if (is_singular()) {
            $schemas = array_merge($schemas, $this->get_singular_schemas());
        } elseif (is_front_page()) {
            $schemas[] = $this->get_frontpage_schema();
        }

        // Filter schemas
        $schemas = apply_filters('instarank_generated_schemas', $schemas);

        // Output schemas
        if (!empty($schemas)) {
            echo '<!-- InstaRank Auto-Generated Schema -->' . "\n";
            foreach ($schemas as $schema) {
                if (!empty($schema)) {
                    echo '<script type="application/ld+json">' . "\n";
                    echo wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . "\n";
                    echo '</script>' . "\n";
                }
            }
        }
    }

    /**
     * Get Website schema with SearchAction
     */
    public function get_website_schema() {
        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'WebSite',
            '@id' => home_url('/#website'),
            'url' => home_url('/'),
            'name' => get_bloginfo('name'),
        ];

        // Add description if available
        $description = get_bloginfo('description');
        if (!empty($description)) {
            $schema['description'] = $description;
        }

        // Add SearchAction if search is enabled
        if (get_option('instarank_schema_search_action', true)) {
            $schema['potentialAction'] = [
                '@type' => 'SearchAction',
                'target' => [
                    '@type' => 'EntryPoint',
                    'urlTemplate' => home_url('/?s={search_term_string}')
                ],
                'query-input' => 'required name=search_term_string'
            ];
        }

        return apply_filters('instarank_website_schema', $schema);
    }

    /**
     * Get frontpage schema
     */
    public function get_frontpage_schema() {
        return [
            '@context' => 'https://schema.org',
            '@type' => 'WebPage',
            '@id' => home_url('/#webpage'),
            'url' => home_url('/'),
            'name' => get_bloginfo('name'),
            'description' => get_bloginfo('description'),
            'isPartOf' => [
                '@id' => home_url('/#website')
            ]
        ];
    }

    /**
     * Get schemas for singular posts/pages
     */
    public function get_singular_schemas() {
        $schemas = [];
        $post = get_post();

        if (!$post) {
            return $schemas;
        }

        // Add WebPage schema
        $schemas[] = $this->get_webpage_schema($post);

        // Add Article schema for posts
        if ($post->post_type === 'post') {
            $schemas[] = $this->get_article_schema($post);
        }

        // Add Product/Offer schema for WooCommerce
        if ($post->post_type === 'product' && class_exists('WooCommerce')) {
            $product_schema = $this->get_product_schema($post);
            if ($product_schema) {
                $schemas[] = $product_schema;
            }
        }

        return $schemas;
    }

    /**
     * Get WebPage schema
     */
    public function get_webpage_schema($post) {
        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'WebPage',
            '@id' => get_permalink($post) . '#webpage',
            'url' => get_permalink($post),
            'name' => get_the_title($post),
            'isPartOf' => [
                '@id' => home_url('/#website')
            ],
            'datePublished' => get_the_date('c', $post),
            'dateModified' => get_the_modified_date('c', $post)
        ];

        // Add description
        $description = get_post_meta($post->ID, '_instarank_meta_description', true);
        if (empty($description)) {
            $description = wp_trim_words($post->post_excerpt ?: $post->post_content, 30);
        }
        if (!empty($description)) {
            $schema['description'] = $description;
        }

        // Add breadcrumb if available
        if (get_option('instarank_breadcrumb_schema_enabled', true)) {
            $schema['breadcrumb'] = [
                '@id' => get_permalink($post) . '#breadcrumb'
            ];
        }

        // Add primary image
        $thumbnail_id = get_post_thumbnail_id($post);
        if ($thumbnail_id) {
            $image = wp_get_attachment_image_src($thumbnail_id, 'full');
            if ($image) {
                $schema['primaryImageOfPage'] = [
                    '@type' => 'ImageObject',
                    '@id' => get_permalink($post) . '#primaryimage',
                    'url' => $image[0],
                    'width' => $image[1],
                    'height' => $image[2]
                ];
            }
        }

        return apply_filters('instarank_webpage_schema', $schema, $post);
    }

    /**
     * Get Article schema
     */
    public function get_article_schema($post) {
        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'Article',
            '@id' => get_permalink($post) . '#article',
            'headline' => get_the_title($post),
            'url' => get_permalink($post),
            'datePublished' => get_the_date('c', $post),
            'dateModified' => get_the_modified_date('c', $post),
            'author' => [
                '@type' => 'Person',
                'name' => get_the_author_meta('display_name', $post->post_author),
                'url' => get_author_posts_url($post->post_author)
            ],
            'publisher' => [
                '@type' => 'Organization',
                'name' => get_bloginfo('name'),
                'url' => home_url('/'),
                'logo' => [
                    '@type' => 'ImageObject',
                    'url' => $this->get_site_logo_url()
                ]
            ],
            'mainEntityOfPage' => [
                '@type' => 'WebPage',
                '@id' => get_permalink($post) . '#webpage'
            ]
        ];

        // Add description
        $description = get_post_meta($post->ID, '_instarank_meta_description', true);
        if (empty($description)) {
            $description = wp_trim_words($post->post_excerpt ?: $post->post_content, 30);
        }
        if (!empty($description)) {
            $schema['description'] = $description;
        }

        // Add article section (category)
        $categories = get_the_category($post->ID);
        if (!empty($categories)) {
            $schema['articleSection'] = $categories[0]->name;
        }

        // Add featured image
        $thumbnail_id = get_post_thumbnail_id($post);
        if ($thumbnail_id) {
            $image = wp_get_attachment_image_src($thumbnail_id, 'full');
            if ($image) {
                $schema['image'] = [
                    '@type' => 'ImageObject',
                    'url' => $image[0],
                    'width' => $image[1],
                    'height' => $image[2]
                ];
            }
        }

        // Add word count
        $word_count = str_word_count(wp_strip_all_tags($post->post_content));
        if ($word_count > 0) {
            $schema['wordCount'] = $word_count;
        }

        return apply_filters('instarank_article_schema', $schema, $post);
    }

    /**
     * Get Product schema for WooCommerce
     */
    public function get_product_schema($post) {
        if (!class_exists('WooCommerce')) {
            return null;
        }

        $product = wc_get_product($post->ID);
        if (!$product) {
            return null;
        }

        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'Product',
            '@id' => get_permalink($post) . '#product',
            'name' => $product->get_name(),
            'url' => get_permalink($post),
            'description' => wp_trim_words($product->get_description(), 50),
            'sku' => $product->get_sku()
        ];

        // Add image
        $image_id = $product->get_image_id();
        if ($image_id) {
            $image = wp_get_attachment_image_src($image_id, 'full');
            if ($image) {
                $schema['image'] = $image[0];
            }
        }

        // Add brand if available
        $brand = get_post_meta($post->ID, '_product_brand', true);
        if (!empty($brand)) {
            $schema['brand'] = [
                '@type' => 'Brand',
                'name' => $brand
            ];
        }

        // Add offers
        $schema['offers'] = $this->get_offer_schema($product, $post);

        // Add aggregate rating if available
        if ($product->get_rating_count() > 0) {
            $schema['aggregateRating'] = [
                '@type' => 'AggregateRating',
                'ratingValue' => $product->get_average_rating(),
                'reviewCount' => $product->get_rating_count()
            ];
        }

        return apply_filters('instarank_product_schema', $schema, $product, $post);
    }

    /**
     * Get Offer schema for WooCommerce product
     */
    public function get_offer_schema($product, $post) {
        $offer = [
            '@type' => 'Offer',
            'url' => get_permalink($post),
            'priceCurrency' => get_woocommerce_currency(),
            'price' => $product->get_price(),
            'priceValidUntil' => gmdate('Y-12-31'), // End of current year
        ];

        // Add availability
        $stock_status = $product->get_stock_status();
        $availability_map = [
            'instock' => 'https://schema.org/InStock',
            'outofstock' => 'https://schema.org/OutOfStock',
            'onbackorder' => 'https://schema.org/PreOrder'
        ];
        $offer['availability'] = $availability_map[$stock_status] ?? 'https://schema.org/InStock';

        // Add seller
        $offer['seller'] = [
            '@type' => 'Organization',
            'name' => get_bloginfo('name'),
            'url' => home_url('/')
        ];

        // Variable product - add multiple offers
        if ($product->is_type('variable')) {
            $variations = $product->get_available_variations();
            if (!empty($variations)) {
                $offers = [];
                foreach ($variations as $variation_data) {
                    $variation = wc_get_product($variation_data['variation_id']);
                    if ($variation) {
                        $offers[] = [
                            '@type' => 'Offer',
                            'url' => get_permalink($post),
                            'priceCurrency' => get_woocommerce_currency(),
                            'price' => $variation->get_price(),
                            'availability' => $availability_map[$variation->get_stock_status()] ?? 'https://schema.org/InStock',
                            'sku' => $variation->get_sku()
                        ];
                    }
                }
                return $offers;
            }
        }

        return $offer;
    }

    /**
     * Get site logo URL
     */
    private function get_site_logo_url() {
        $custom_logo_id = get_theme_mod('custom_logo');
        if ($custom_logo_id) {
            $logo = wp_get_attachment_image_src($custom_logo_id, 'full');
            if ($logo) {
                return $logo[0];
            }
        }

        // Fallback to site icon
        $site_icon = get_site_icon_url();
        if ($site_icon) {
            return $site_icon;
        }

        // Default placeholder
        return home_url('/wp-content/plugins/instarank-seo/assets/images/logo.png');
    }

    /**
     * Generate schema for a specific post (manual generation)
     */
    public function generate_for_post($post_id) {
        $post = get_post($post_id);
        if (!$post) {
            return null;
        }

        $schemas = [];

        // WebPage
        $schemas[] = $this->get_webpage_schema($post);

        // Article for posts
        if ($post->post_type === 'post') {
            $schemas[] = $this->get_article_schema($post);
        }

        // Product for WooCommerce
        if ($post->post_type === 'product' && class_exists('WooCommerce')) {
            $product_schema = $this->get_product_schema($post);
            if ($product_schema) {
                $schemas[] = $product_schema;
            }
        }

        return $schemas;
    }
}

// Initialize
InstaRank_Schema_Generator::instance();
