<?php
/**
 * Robots.txt Editor
 *
 * Allows editing robots.txt content from WordPress admin
 *
 * @package InstaRank
 * @since 1.4.0
 */

defined('ABSPATH') || exit;

class InstaRank_Robots_Txt {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_filter('robots_txt', [$this, 'custom_robots_txt'], 10, 2);
        add_action('do_robotstxt', [$this, 'add_sitemap_to_robots'], 0);
    }

    /**
     * Filter robots.txt output
     */
    public function custom_robots_txt($output, $public) {
        // Get custom robots.txt content
        $custom_content = get_option('instarank_robots_txt', '');

        if (!empty($custom_content)) {
            return $custom_content;
        }

        // Return default WordPress output if no custom content
        return $output;
    }

    /**
     * Add sitemap to robots.txt
     */
    public function add_sitemap_to_robots() {
        // Only add if sitemaps are enabled
        $sitemap_enabled = get_option('instarank_sitemap_enabled', true);

        if ($sitemap_enabled) {
            $sitemap_url = home_url('/sitemap.xml');
            echo "Sitemap: " . esc_url($sitemap_url) . "\n";
        }
    }

    /**
     * Get default robots.txt content
     */
    public function get_default_content() {
        $site_url = wp_parse_url(home_url(), PHP_URL_PATH);

        $default = "User-agent: *\n";
        $default .= "Disallow: /wp-admin/\n";
        $default .= "Allow: /wp-admin/admin-ajax.php\n";
        $default .= "Disallow: /wp-includes/\n";
        $default .= "Disallow: /wp-content/plugins/\n";
        $default .= "Disallow: /wp-content/themes/\n";
        $default .= "Disallow: /trackback/\n";
        $default .= "Disallow: /feed/\n";
        $default .= "Disallow: /comments/\n";
        $default .= "Disallow: /xmlrpc.php\n";
        $default .= "Disallow: *?s=\n";
        $default .= "Disallow: *?attachment_id=\n";
        $default .= "Disallow: */embed$\n\n";

        // Add sitemap reference
        if (get_option('instarank_sitemap_enabled', true)) {
            $default .= "Sitemap: " . home_url('/sitemap.xml') . "\n";
        }

        return apply_filters('instarank_default_robots_txt', $default);
    }

    /**
     * Save robots.txt content
     */
    public function save_content($content) {
        // Sanitize content
        $content = sanitize_textarea_field($content);

        // Validate
        if ($this->validate_content($content)) {
            update_option('instarank_robots_txt', $content);

            // Send webhook notification
            $webhook = new InstaRank_Webhook_Sender();
            $webhook->send_settings_updated([
                'robots_txt_updated' => true,
                'message' => 'Robots.txt content updated'
            ]);

            return true;
        }

        return false;
    }

    /**
     * Reset to default
     */
    public function reset_to_default() {
        delete_option('instarank_robots_txt');
        return true;
    }

    /**
     * Validate robots.txt content
     */
    private function validate_content($content) {
        // Basic validation
        if (strlen($content) > 100000) { // Max 100KB
            return false;
        }

        // Check for common directives
        $has_user_agent = stripos($content, 'user-agent:') !== false;

        return true; // Allow any content, even without User-agent
    }

    /**
     * Get current robots.txt content
     */
    public function get_content() {
        $custom = get_option('instarank_robots_txt', '');

        if (!empty($custom)) {
            return $custom;
        }

        return $this->get_default_content();
    }

    /**
     * Test if robots.txt is accessible
     */
    public function is_accessible() {
        $url = home_url('/robots.txt');
        $response = wp_remote_get($url, ['timeout' => 5]);

        if (is_wp_error($response)) {
            return false;
        }

        return wp_remote_retrieve_response_code($response) === 200;
    }
}

// Initialize
InstaRank_Robots_Txt::instance();
