<?php
/**
 * Related Links / Internal Linking System
 *
 * Automatically create internal links between generated pages.
 * Critical for SEO - helps spread link equity and improves crawlability.
 *
 * @package InstaRank
 */

if (!defined('ABSPATH')) {
    exit;
}

class InstaRank_Related_Links {

    /**
     * Instance of this class
     */
    private static $instance = null;

    /**
     * Default configuration
     */
    private $default_config = array(
        'max_links' => 5,
        'display_style' => 'list', // list, grid, cards, inline-text
        'position' => 'bottom', // top, bottom, inline, sidebar
        'sort_by' => 'relevance', // relevance, date, random, alphabetical
        'anchor_text_type' => 'title', // title, custom, varied
        'same_category_weight' => 3,
        'same_tag_weight' => 2,
        'title_similarity_weight' => 2,
        'auto_insert' => false,
        'max_auto_links' => 3,
    );

    /**
     * Get instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        // Register shortcode
        add_shortcode('instarank_related', array($this, 'shortcode_related_links'));

        // REST API
        add_action('rest_api_init', array($this, 'register_rest_routes'));

        // Auto-insert filter (if enabled)
        add_filter('the_content', array($this, 'maybe_auto_insert_links'), 20);

        // Admin settings
        add_action('admin_init', array($this, 'register_settings'));
    }

    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        register_rest_route('instarank/v1', '/related-links', array(
            'methods' => 'GET',
            'callback' => array($this, 'rest_get_related'),
            'permission_callback' => array($this, 'check_api_permission'),
            'args' => array(
                'post_id' => array(
                    'required' => true,
                    'type' => 'integer',
                ),
                'max' => array(
                    'default' => 5,
                    'type' => 'integer',
                ),
                'style' => array(
                    'default' => 'list',
                    'type' => 'string',
                ),
                'format' => array(
                    'default' => 'json',
                    'type' => 'string',
                ),
            ),
        ));

        register_rest_route('instarank/v1', '/related-links/auto-insert', array(
            'methods' => 'POST',
            'callback' => array($this, 'rest_auto_insert'),
            'permission_callback' => array($this, 'check_api_permission'),
        ));
    }

    /**
     * Check API permission
     */
    public function check_api_permission($request) {
        $api_key = $request->get_header('X-WordPress-API-Key');
        if ($api_key) {
            $stored_key = get_option('instarank_api_key');
            if ($api_key === $stored_key) {
                return true;
            }
        }
        return current_user_can('edit_posts');
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('instarank_related_links', 'instarank_related_links_config', array(
            'type' => 'array',
            'default' => $this->default_config,
            'sanitize_callback' => array($this, 'sanitize_config'),
        ));
    }

    /**
     * Sanitize configuration
     */
    public function sanitize_config($config) {
        $sanitized = array();
        $sanitized['max_links'] = absint($config['max_links'] ?? 5);
        $sanitized['display_style'] = sanitize_text_field($config['display_style'] ?? 'list');
        $sanitized['position'] = sanitize_text_field($config['position'] ?? 'bottom');
        $sanitized['sort_by'] = sanitize_text_field($config['sort_by'] ?? 'relevance');
        $sanitized['anchor_text_type'] = sanitize_text_field($config['anchor_text_type'] ?? 'title');
        $sanitized['same_category_weight'] = floatval($config['same_category_weight'] ?? 3);
        $sanitized['same_tag_weight'] = floatval($config['same_tag_weight'] ?? 2);
        $sanitized['title_similarity_weight'] = floatval($config['title_similarity_weight'] ?? 2);
        $sanitized['auto_insert'] = !empty($config['auto_insert']);
        $sanitized['max_auto_links'] = absint($config['max_auto_links'] ?? 3);
        return $sanitized;
    }

    /**
     * Get configuration
     */
    public function get_config() {
        return array_merge(
            $this->default_config,
            get_option('instarank_related_links_config', array())
        );
    }

    /**
     * Get related posts for a given post
     */
    public function get_related_posts($post_id, $args = array()) {
        $config = $this->get_config();

        $defaults = array(
            'max' => $config['max_links'],
            'post_type' => null,
            'categories' => array(),
            'tags' => array(),
            'exclude_ids' => array($post_id), // IDs to filter out after query (VIP performance)
            'sort_by' => $config['sort_by'],
        );

        $args = wp_parse_args($args, $defaults);

        $post = get_post($post_id);
        if (!$post) {
            return array();
        }

        // Build query - avoiding post__not_in for VIP performance
        // See: https://wpvip.com/documentation/performance-improvements-by-removing-usage-of-post__not_in/
        $query_args = array(
            'post_type' => $args['post_type'] ?: $post->post_type,
            'post_status' => 'publish',
            'posts_per_page' => ($args['max'] * 3) + count($args['exclude_ids']), // Get extra to account for filtering
            'orderby' => 'date',
            'order' => 'DESC',
        );

        // Filter by categories
        if (!empty($args['categories'])) {
            $query_args['category__in'] = $args['categories'];
        }

        // Filter by tags
        if (!empty($args['tags'])) {
            $query_args['tag__in'] = $args['tags'];
        }

        $query = new WP_Query($query_args);

        // Filter out excluded IDs after query (better performance than post__not_in)
        $exclude_ids = array_map('absint', $args['exclude_ids']);
        $candidates = array_filter($query->posts, function($candidate) use ($exclude_ids) {
            return !in_array($candidate->ID, $exclude_ids, true);
        });

        if (empty($candidates)) {
            return array();
        }

        // Get source post data for scoring
        $source_categories = wp_get_post_categories($post_id);
        $source_tags = wp_get_post_tags($post_id, array('fields' => 'ids'));
        $source_title = $post->post_title;

        // Score and rank candidates
        $scored = array();
        foreach ($candidates as $candidate) {
            $score = $this->calculate_relevance_score(
                $source_categories,
                $source_tags,
                $source_title,
                $candidate,
                $config
            );

            $scored[] = array(
                'post' => $candidate,
                'score' => $score,
            );
        }

        // Sort by score
        if ($args['sort_by'] === 'relevance') {
            usort($scored, function($a, $b) {
                return $b['score'] - $a['score'];
            });
        } elseif ($args['sort_by'] === 'random') {
            shuffle($scored);
        } elseif ($args['sort_by'] === 'alphabetical') {
            usort($scored, function($a, $b) {
                return strcasecmp($a['post']->post_title, $b['post']->post_title);
            });
        }
        // 'date' is already sorted by query

        // Take only max needed
        $scored = array_slice($scored, 0, $args['max']);

        // Format results
        return array_map(function($item) {
            $post = $item['post'];
            return array(
                'id' => $post->ID,
                'title' => $post->post_title,
                'url' => get_permalink($post->ID),
                'slug' => $post->post_name,
                'excerpt' => wp_trim_words($post->post_excerpt ?: $post->post_content, 20),
                'post_type' => $post->post_type,
                'date' => $post->post_date,
                'score' => $item['score'],
            );
        }, $scored);
    }

    /**
     * Calculate relevance score between source and candidate
     */
    private function calculate_relevance_score($source_cats, $source_tags, $source_title, $candidate, $config) {
        $score = 0;

        // Category match
        $candidate_cats = wp_get_post_categories($candidate->ID);
        $common_cats = array_intersect($source_cats, $candidate_cats);
        $score += count($common_cats) * $config['same_category_weight'];

        // Tag match
        $candidate_tags = wp_get_post_tags($candidate->ID, array('fields' => 'ids'));
        $common_tags = array_intersect($source_tags, $candidate_tags);
        $score += count($common_tags) * $config['same_tag_weight'];

        // Title similarity (Jaccard)
        $similarity = $this->calculate_text_similarity($source_title, $candidate->post_title);
        $score += $similarity * $config['title_similarity_weight'] * 10;

        return $score;
    }

    /**
     * Calculate text similarity using Jaccard index
     */
    private function calculate_text_similarity($text1, $text2) {
        $words1 = array_filter(preg_split('/\s+/', strtolower($text1)), function($w) {
            return strlen($w) > 2;
        });
        $words2 = array_filter(preg_split('/\s+/', strtolower($text2)), function($w) {
            return strlen($w) > 2;
        });

        if (empty($words1) || empty($words2)) {
            return 0;
        }

        $intersection = array_intersect($words1, $words2);
        $union = array_unique(array_merge($words1, $words2));

        return count($intersection) / count($union);
    }

    /**
     * Generate anchor text
     */
    private function generate_anchor_text($related, $config, $index = 0) {
        $type = $config['anchor_text_type'] ?? 'title';

        switch ($type) {
            case 'varied':
                $variations = array(
                    $related['title'],
                    /* translators: %s: related page title */
                    sprintf(__('Read more about %s', 'instarank'), $related['title']),
                    /* translators: %s: related page title */
                    sprintf(__('Learn about %s', 'instarank'), $related['title']),
                    /* translators: %s: related page title */
                    sprintf(__('Discover %s', 'instarank'), $related['title']),
                );
                return $variations[$index % count($variations)];

            case 'title':
            default:
                return $related['title'];
        }
    }

    /**
     * Generate HTML for related links
     */
    public function generate_html($related_posts, $args = array()) {
        $config = $this->get_config();

        $defaults = array(
            'style' => $config['display_style'],
            'position' => $config['position'],
            'heading' => __('Related Pages', 'instarank'),
        );

        $args = wp_parse_args($args, $defaults);

        if (empty($related_posts)) {
            return '';
        }

        $style = $args['style'];
        $position = $args['position'];
        $heading = esc_html($args['heading']);

        ob_start();

        switch ($style) {
            case 'grid':
                ?>
                <div class="instarank-related-links instarank-related-links--grid position-<?php echo esc_attr($position); ?>">
                    <h3><?php echo esc_html($heading); ?></h3>
                    <div class="related-links-grid">
                        <?php foreach ($related_posts as $i => $post): ?>
                            <div class="related-link-item">
                                <a href="<?php echo esc_url($post['url']); ?>">
                                    <?php echo esc_html($this->generate_anchor_text($post, $config, $i)); ?>
                                </a>
                                <?php if (!empty($post['excerpt'])): ?>
                                    <p class="excerpt"><?php echo esc_html($post['excerpt']); ?></p>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php
                break;

            case 'cards':
                ?>
                <div class="instarank-related-links instarank-related-links--cards position-<?php echo esc_attr($position); ?>">
                    <h3><?php echo esc_html($heading); ?></h3>
                    <div class="related-links-cards">
                        <?php foreach ($related_posts as $i => $post): ?>
                            <article class="related-link-card">
                                <h4>
                                    <a href="<?php echo esc_url($post['url']); ?>">
                                        <?php echo esc_html($this->generate_anchor_text($post, $config, $i)); ?>
                                    </a>
                                </h4>
                                <?php if (!empty($post['excerpt'])): ?>
                                    <p><?php echo esc_html($post['excerpt']); ?></p>
                                <?php endif; ?>
                            </article>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php
                break;

            case 'inline-text':
                $links = array();
                foreach ($related_posts as $i => $post) {
                    $links[] = sprintf(
                        '<a href="%s">%s</a>',
                        esc_url($post['url']),
                        esc_html($this->generate_anchor_text($post, $config, $i))
                    );
                }
                ?>
                <p class="instarank-related-links instarank-related-links--inline position-<?php echo esc_attr($position); ?>">
                    <?php
                    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- $links contains pre-escaped HTML anchor tags
                    echo esc_html__('Related:', 'instarank') . ' ' . implode(', ', $links);
                    ?>
                </p>
                <?php
                break;

            case 'list':
            default:
                ?>
                <div class="instarank-related-links instarank-related-links--list position-<?php echo esc_attr($position); ?>">
                    <h3><?php echo esc_html($heading); ?></h3>
                    <ul>
                        <?php foreach ($related_posts as $i => $post): ?>
                            <li>
                                <a href="<?php echo esc_url($post['url']); ?>">
                                    <?php echo esc_html($this->generate_anchor_text($post, $config, $i)); ?>
                                </a>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php
                break;
        }

        return ob_get_clean();
    }

    /**
     * Shortcode: [instarank_related]
     */
    public function shortcode_related_links($atts) {
        $atts = shortcode_atts(array(
            'max' => 5,
            'style' => 'list',
            'sort' => 'relevance',
            'post_type' => '',
            'categories' => '',
            'tags' => '',
            'heading' => __('Related Pages', 'instarank'),
            'anchor' => 'title',
        ), $atts, 'instarank_related');

        $post_id = get_the_ID();
        if (!$post_id) {
            return '';
        }

        // Parse categories and tags
        $categories = $atts['categories'] ? array_map('absint', explode(',', $atts['categories'])) : array();
        $tags = $atts['tags'] ? array_map('absint', explode(',', $atts['tags'])) : array();

        $related = $this->get_related_posts($post_id, array(
            'max' => absint($atts['max']),
            'post_type' => $atts['post_type'] ?: null,
            'categories' => $categories,
            'tags' => $tags,
            'sort_by' => $atts['sort'],
        ));

        // Temporarily override anchor text type
        $config = $this->get_config();
        $config['anchor_text_type'] = $atts['anchor'];
        update_option('instarank_related_links_config', $config);

        $html = $this->generate_html($related, array(
            'style' => $atts['style'],
            'heading' => $atts['heading'],
        ));

        return $html;
    }

    /**
     * Auto-insert internal links into content
     */
    public function auto_insert_internal_links($content, $args = array()) {
        $defaults = array(
            'max_links' => 3,
            'post_id' => null,
            'exclude_phrases' => array(),
            'min_phrase_length' => 3,
        );

        $args = wp_parse_args($args, $defaults);
        $post_id = $args['post_id'] ?: get_the_ID();

        if (!$post_id) {
            return $content;
        }

        // Get related posts
        $related = $this->get_related_posts($post_id, array(
            'max' => $args['max_links'] * 2, // Get extra for matching
        ));

        if (empty($related)) {
            return $content;
        }

        // Sort by title length (longest first)
        usort($related, function($a, $b) {
            return strlen($b['title']) - strlen($a['title']);
        });

        $links_inserted = 0;
        $linked_content = $content;

        foreach ($related as $post) {
            if ($links_inserted >= $args['max_links']) {
                break;
            }

            $title = $post['title'];
            if (strlen($title) < $args['min_phrase_length']) {
                continue;
            }
            if (in_array(strtolower($title), array_map('strtolower', $args['exclude_phrases']))) {
                continue;
            }

            // Escape regex special characters
            $escaped_title = preg_quote($title, '/');

            // Pattern: word boundary, not inside existing link or tag
            $pattern = '/(?<![<\/a-zA-Z])\b(' . $escaped_title . ')\b(?![^<]*>|[^<>]*<\/a>)/i';

            // Check if match exists
            if (preg_match($pattern, $linked_content)) {
                $replacement = '<a href="' . esc_url($post['url']) . '" class="instarank-auto-link">$1</a>';
                $linked_content = preg_replace($pattern, $replacement, $linked_content, 1);
                $links_inserted++;
            }
        }

        return $linked_content;
    }

    /**
     * Maybe auto-insert links in content (filter hook)
     */
    public function maybe_auto_insert_links($content) {
        $config = $this->get_config();

        if (empty($config['auto_insert'])) {
            return $content;
        }

        if (!is_singular() || !in_the_loop() || !is_main_query()) {
            return $content;
        }

        return $this->auto_insert_internal_links($content, array(
            'max_links' => $config['max_auto_links'],
        ));
    }

    /**
     * REST API: Get related posts
     */
    public function rest_get_related($request) {
        $post_id = $request->get_param('post_id');
        $max = $request->get_param('max');
        $style = $request->get_param('style');
        $format = $request->get_param('format');

        $post = get_post($post_id);
        if (!$post) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => 'Post not found',
            ), 404);
        }

        $related = $this->get_related_posts($post_id, array(
            'max' => $max,
        ));

        if ($format === 'html') {
            return new WP_REST_Response(array(
                'success' => true,
                'post_id' => $post_id,
                'count' => count($related),
                'html' => $this->generate_html($related, array('style' => $style)),
            ));
        }

        return new WP_REST_Response(array(
            'success' => true,
            'post_id' => $post_id,
            'count' => count($related),
            'related_posts' => $related,
        ));
    }

    /**
     * REST API: Auto-insert links
     */
    public function rest_auto_insert($request) {
        $params = $request->get_json_params();

        $content = $params['content'] ?? '';
        $post_id = $params['post_id'] ?? null;
        $max_links = $params['max_links'] ?? 3;

        if (empty($content)) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => 'content is required',
            ), 400);
        }

        $linked_content = $this->auto_insert_internal_links($content, array(
            'max_links' => $max_links,
            'post_id' => $post_id,
        ));

        preg_match_all('/class="instarank-auto-link"/', $linked_content, $matches);
        $links_count = count($matches[0]);

        return new WP_REST_Response(array(
            'success' => true,
            'original_content' => $content,
            'linked_content' => $linked_content,
            'links_inserted' => $links_count,
        ));
    }
}

// Initialize
InstaRank_Related_Links::get_instance();
