<?php
/**
 * Page Builder API Handler
 *
 * Provides REST API endpoints for page builder content extraction and manipulation
 *
 * @package InstaRank
 * @since 1.3.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class InstaRank_Page_Builder_API {
    /**
     * Initialize the API handler
     */
    public static function init() {
        add_action('rest_api_init', [__CLASS__, 'register_routes']);
    }

    /**
     * Register REST API routes
     */
    public static function register_routes() {
        // Get page builder data for a post
        register_rest_route('instarank/v1', '/page-builder/(?P<id>\d+)', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'get_page_builder_data'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
            'args' => [
                'id' => [
                    'required' => true,
                    'validate_callback' => function($param) {
                        return is_numeric($param);
                    }
                ]
            ]
        ]);

        // Update page builder data for a post
        register_rest_route('instarank/v1', '/page-builder/(?P<id>\d+)', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'update_page_builder_data'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
            'args' => [
                'id' => [
                    'required' => true,
                    'validate_callback' => function($param) {
                        return is_numeric($param);
                    }
                ],
                'builder' => [
                    'required' => true,
                    'type' => 'string'
                ],
                'data' => [
                    'required' => true,
                    'type' => 'string'
                ]
            ]
        ]);

        // Detect page builder for multiple posts
        register_rest_route('instarank/v1', '/page-builder/detect', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'detect_page_builders'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
            'args' => [
                'post_ids' => [
                    'required' => true,
                    'type' => 'array'
                ]
            ]
        ]);

        // Get all posts with page builder info
        register_rest_route('instarank/v1', '/page-builder/posts', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'get_posts_with_builders'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
            'args' => [
                'per_page' => [
                    'default' => 100,
                    'type' => 'integer'
                ],
                'page' => [
                    'default' => 1,
                    'type' => 'integer'
                ],
                'builder' => [
                    'default' => '',
                    'type' => 'string'
                ]
            ]
        ]);

        // NEW: Get all installed/active page builders on the site
        register_rest_route('instarank/v1', '/page-builders', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'get_installed_builders'],
            'permission_callback' => [__CLASS__, 'check_permissions']
        ]);

        // NEW: Scan for existing templates in page builders
        register_rest_route('instarank/v1', '/templates/scan', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'scan_existing_templates'],
            'permission_callback' => [__CLASS__, 'check_permissions']
        ]);

        // NEW: Import and parse template content
        register_rest_route('instarank/v1', '/templates/import', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'import_template'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
            'args' => [
                'template_content' => [
                    'required' => true,
                    'type' => ['object', 'array', 'string']
                ],
                'builder' => [
                    'required' => true,
                    'type' => 'string'
                ]
            ]
        ]);
    }

    /**
     * Get page builder data for a post
     */
    public static function get_page_builder_data($request) {
        $post_id = $request['id'];
        $post = get_post($post_id);

        if (!$post) {
            return new WP_Error('post_not_found', 'Post not found', ['status' => 404]);
        }

        $detection = self::detect_page_builder($post_id);
        $data = null;
        $meta_fields = [];

        // Extract builder-specific data and meta fields
        switch ($detection['builder']) {
            case 'elementor':
                $data = get_post_meta($post_id, '_elementor_data', true);
                $meta_fields = [
                    '_elementor_edit_mode' => get_post_meta($post_id, '_elementor_edit_mode', true),
                    '_elementor_version' => get_post_meta($post_id, '_elementor_version', true),
                    '_elementor_pro_version' => get_post_meta($post_id, '_elementor_pro_version', true),
                    '_elementor_template_type' => get_post_meta($post_id, '_elementor_template_type', true),
                    '_elementor_page_settings' => get_post_meta($post_id, '_elementor_page_settings', true),
                    '_elementor_css' => get_post_meta($post_id, '_elementor_css', true),
                ];
                break;

            case 'divi':
                $data = $post->post_content; // Divi stores shortcodes in post_content
                $meta_fields = [
                    '_et_pb_use_builder' => get_post_meta($post_id, '_et_pb_use_builder', true),
                    '_et_pb_old_content' => get_post_meta($post_id, '_et_pb_old_content', true),
                    '_et_pb_page_layout' => get_post_meta($post_id, '_et_pb_page_layout', true),
                    '_et_pb_side_nav' => get_post_meta($post_id, '_et_pb_side_nav', true),
                    '_et_pb_post_hide_nav' => get_post_meta($post_id, '_et_pb_post_hide_nav', true),
                    '_et_pb_built_for_post_type' => get_post_meta($post_id, '_et_pb_built_for_post_type', true),
                ];
                break;

            case 'beaver_builder':
                $data = get_post_meta($post_id, '_fl_builder_data', true);
                $meta_fields = [
                    '_fl_builder_enabled' => get_post_meta($post_id, '_fl_builder_enabled', true),
                    '_fl_builder_draft' => get_post_meta($post_id, '_fl_builder_draft', true),
                    '_fl_builder_data_settings' => get_post_meta($post_id, '_fl_builder_data_settings', true),
                    '_fl_builder_css' => get_post_meta($post_id, '_fl_builder_css', true),
                    '_fl_builder_js' => get_post_meta($post_id, '_fl_builder_js', true),
                ];
                break;

            case 'bricks':
                $data = get_post_meta($post_id, '_bricks_page_content_2', true);
                if (!$data) {
                    $data = get_post_meta($post_id, '_bricks_page_content', true);
                }
                $meta_fields = [
                    '_bricks_page_content_2' => get_post_meta($post_id, '_bricks_page_content_2', true),
                    '_bricks_page_content' => get_post_meta($post_id, '_bricks_page_content', true),
                    '_bricks_editor_mode' => get_post_meta($post_id, '_bricks_editor_mode', true),
                    '_bricks_page_header' => get_post_meta($post_id, '_bricks_page_header', true),
                    '_bricks_page_footer' => get_post_meta($post_id, '_bricks_page_footer', true),
                ];
                break;

            case 'oxygen':
                $data = get_post_meta($post_id, 'ct_builder_shortcodes', true);
                $meta_fields = [
                    'ct_builder_shortcodes' => get_post_meta($post_id, 'ct_builder_shortcodes', true),
                    'ct_builder_json' => get_post_meta($post_id, 'ct_builder_json', true),
                    'ct_other_template' => get_post_meta($post_id, 'ct_other_template', true),
                ];
                break;

            case 'kadence_blocks':
            case 'gutenberg':
                $data = $post->post_content; // Gutenberg stores blocks in post_content

                // Get all Kadence-specific meta fields
                // Check cache first
                $cache_key = 'instarank_kadence_meta_' . $post_id;
                $kadence_meta = wp_cache_get($cache_key, 'instarank');

                if (false === $kadence_meta) {
                    // Get all post meta and filter for Kadence keys
                    // Using get_post_meta() without a key to get all meta
                    $all_meta = get_post_meta($post_id);
                    $kadence_meta = array();

                    // Filter for Kadence-specific meta keys
                    foreach ($all_meta as $key => $values) {
                        if (strpos($key, '_kad') === 0 || strpos($key, 'kt_') === 0) {
                            // Store as array with field_key and field_value
                            $kadence_meta[] = array(
                                'field_key' => $key,
                                'field_value' => is_array($values) ? $values[0] : $values
                            );
                        }
                    }

                    // Cache for 1 hour
                    wp_cache_set($cache_key, $kadence_meta, 'instarank', HOUR_IN_SECONDS);
                }

                // Convert to associative array
                foreach ($kadence_meta as $meta_row) {
                    $meta_fields[$meta_row['field_key']] = $meta_row['field_value'];
                }

                // Add common Gutenberg meta
                $meta_fields['_wp_page_template'] = get_post_meta($post_id, '_wp_page_template', true);
                break;

            case 'wpbakery':
                $data = $post->post_content; // WPBakery stores shortcodes in post_content
                $meta_fields = [
                    '_wpb_vc_js_status' => get_post_meta($post_id, '_wpb_vc_js_status', true),
                    '_wpb_shortcodes_custom_css' => get_post_meta($post_id, '_wpb_shortcodes_custom_css', true),
                    '_wpb_post_custom_css' => get_post_meta($post_id, '_wpb_post_custom_css', true),
                ];
                break;

            case 'brizy':
                $data = get_post_meta($post_id, 'brizy', true);
                $meta_fields = [
                    'brizy' => get_post_meta($post_id, 'brizy', true),
                    'brizy-post-uid' => get_post_meta($post_id, 'brizy-post-uid', true),
                    'brizy-meta' => get_post_meta($post_id, 'brizy-meta', true),
                ];
                break;

            default:
                // Classic editor or unknown builder
                $data = $post->post_content;
                break;
        }

        // Remove empty meta fields for cleaner response
        $meta_fields = array_filter($meta_fields, function($value) {
            return !empty($value);
        });

        return [
            'success' => true,
            'post_id' => $post_id,
            'post_title' => $post->post_title,
            'post_type' => $post->post_type,
            'post_status' => $post->post_status,
            'builder' => $detection['builder'],
            'version' => $detection['version'],
            'data' => $data,
            'meta_fields' => $meta_fields,
            'metadata' => $detection['metadata']
        ];
    }

    /**
     * Update page builder data for a post
     */
    public static function update_page_builder_data($request) {
        $post_id = $request['id'];
        $builder = $request['builder'];
        $data = $request['data'];

        $post = get_post($post_id);
        if (!$post) {
            return new WP_Error('post_not_found', 'Post not found', ['status' => 404]);
        }

        $success = false;

        switch ($builder) {
            case 'elementor':
                $success = update_post_meta($post_id, '_elementor_data', $data);
                // Clear Elementor cache
                if (class_exists('Elementor\Plugin')) {
                    Elementor\Plugin::$instance->files_manager->clear_cache();
                }
                break;

            case 'bricks':
                $success = update_post_meta($post_id, '_bricks_page_content_2', $data);
                // Clear Bricks cache if available
                if (function_exists('bricks_clear_cache')) {
                    bricks_clear_cache();
                }
                break;

            default:
                return new WP_Error('unsupported_builder', 'Unsupported page builder', ['status' => 400]);
        }

        if ($success) {
            return [
                'success' => true,
                'post_id' => $post_id,
                'builder' => $builder,
                'message' => 'Page builder data updated successfully'
            ];
        }

        return new WP_Error('update_failed', 'Failed to update page builder data', ['status' => 500]);
    }

    /**
     * Detect page builders for multiple posts
     */
    public static function detect_page_builders($request) {
        $post_ids = $request['post_ids'];

        if (empty($post_ids) || !is_array($post_ids)) {
            return new WP_Error('invalid_post_ids', 'Invalid post IDs', ['status' => 400]);
        }

        $results = [];

        foreach ($post_ids as $post_id) {
            $detection = self::detect_page_builder($post_id);
            $results[] = [
                'id' => $post_id,
                'builder' => $detection['builder'],
                'version' => $detection['version'],
                'confidence' => $detection['confidence']
            ];
        }

        return [
            'success' => true,
            'total' => count($results),
            'posts' => $results
        ];
    }

    /**
     * Get all posts with page builder information
     */
    public static function get_posts_with_builders($request) {
        $per_page = $request['per_page'];
        $page = $request['page'];
        $builder_filter = $request['builder'];

        $args = [
            'post_type' => 'any',
            'posts_per_page' => $per_page,
            'paged' => $page,
            'post_status' => 'publish'
        ];

        $query = new WP_Query($args);
        $posts = [];

        foreach ($query->posts as $post) {
            $detection = self::detect_page_builder($post->ID);

            // Apply builder filter if specified
            if ($builder_filter && $detection['builder'] !== $builder_filter) {
                continue;
            }

            $posts[] = [
                'id' => $post->ID,
                'title' => $post->post_title,
                'type' => $post->post_type,
                'builder' => $detection['builder'],
                'version' => $detection['version'],
                'url' => get_permalink($post->ID),
                'modified' => $post->post_modified
            ];
        }

        return [
            'success' => true,
            'posts' => $posts,
            'total' => $query->found_posts,
            'pages' => $query->max_num_pages,
            'current_page' => $page
        ];
    }

    /**
     * Detect which page builder is used for a post
     */
    private static function detect_page_builder($post_id) {
        // Check for Elementor
        $elementor_mode = get_post_meta($post_id, '_elementor_edit_mode', true);
        if ($elementor_mode === 'builder') {
            $version = get_post_meta($post_id, '_elementor_version', true);
            $pro_version = get_post_meta($post_id, '_elementor_pro_version', true);

            return [
                'builder' => 'elementor',
                'version' => $pro_version ?: $version,
                'confidence' => 'high',
                'metadata' => [
                    'has_pro' => !empty($pro_version),
                    'edit_mode' => $elementor_mode
                ]
            ];
        }

        // Check for Divi
        if (get_post_meta($post_id, '_et_pb_use_builder', true) === 'on') {
            return [
                'builder' => 'divi',
                'version' => defined('ET_BUILDER_VERSION') ? ET_BUILDER_VERSION : 'unknown',
                'confidence' => 'high',
                'metadata' => [
                    'builder_enabled' => true
                ]
            ];
        }

        // Check for Beaver Builder
        if (get_post_meta($post_id, '_fl_builder_enabled', true)) {
            return [
                'builder' => 'beaver_builder',
                'version' => defined('FL_BUILDER_VERSION') ? FL_BUILDER_VERSION : 'unknown',
                'confidence' => 'high',
                'metadata' => [
                    'draft' => get_post_meta($post_id, '_fl_builder_draft', true)
                ]
            ];
        }

        // Check for WPBakery (Visual Composer)
        $post = get_post($post_id);
        if ($post && (strpos($post->post_content, '[vc_') !== false || get_post_meta($post_id, '_wpb_vc_js_status', true))) {
            return [
                'builder' => 'wpbakery',
                'version' => defined('WPB_VC_VERSION') ? WPB_VC_VERSION : 'unknown',
                'confidence' => 'high',
                'metadata' => [
                    'js_composer' => true
                ]
            ];
        }

        // Check for Oxygen Builder
        if (get_post_meta($post_id, 'ct_builder_shortcodes', true)) {
            return [
                'builder' => 'oxygen',
                'version' => defined('CT_VERSION') ? CT_VERSION : 'unknown',
                'confidence' => 'high',
                'metadata' => [
                    'has_shortcodes' => true
                ]
            ];
        }

        // Check for Bricks
        $bricks_data_v2 = get_post_meta($post_id, '_bricks_page_content_2', true);
        if ($bricks_data_v2) {
            return [
                'builder' => 'bricks',
                'version' => defined('BRICKS_VERSION') ? BRICKS_VERSION : '1.5+',
                'confidence' => 'high',
                'metadata' => [
                    'format' => 'v2',
                    'has_header' => !empty(get_post_meta($post_id, '_bricks_page_header', true)),
                    'has_footer' => !empty(get_post_meta($post_id, '_bricks_page_footer', true))
                ]
            ];
        }

        $bricks_data_v1 = get_post_meta($post_id, '_bricks_page_content', true);
        if ($bricks_data_v1) {
            return [
                'builder' => 'bricks',
                'version' => '<1.5',
                'confidence' => 'high',
                'metadata' => [
                    'format' => 'v1'
                ]
            ];
        }

        // Check for Kadence Blocks (Gutenberg extension)
        if ($post && has_blocks($post->post_content)) {
            $blocks = parse_blocks($post->post_content);
            $has_kadence = false;
            foreach ($blocks as $block) {
                if (strpos($block['blockName'], 'kadence/') === 0) {
                    $has_kadence = true;
                    break;
                }
            }
            if ($has_kadence) {
                return [
                    'builder' => 'kadence_blocks',
                    'version' => defined('KADENCE_BLOCKS_VERSION') ? KADENCE_BLOCKS_VERSION : 'unknown',
                    'confidence' => 'high',
                    'metadata' => [
                        'has_kadence_blocks' => true
                    ]
                ];
            }
        }

        // Check for GenerateBlocks (Gutenberg extension)
        if ($post && has_blocks($post->post_content)) {
            $blocks = parse_blocks($post->post_content);
            $has_generateblocks = false;
            foreach ($blocks as $block) {
                if (strpos($block['blockName'], 'generateblocks/') === 0) {
                    $has_generateblocks = true;
                    break;
                }
            }
            if ($has_generateblocks) {
                return [
                    'builder' => 'generateblocks',
                    'version' => defined('GENERATEBLOCKS_VERSION') ? GENERATEBLOCKS_VERSION : 'unknown',
                    'confidence' => 'high',
                    'metadata' => [
                        'has_generateblocks' => true
                    ]
                ];
            }
        }

        // Check for Brizy
        if (get_post_meta($post_id, 'brizy', true) || get_post_meta($post_id, 'brizy-post-uid', true)) {
            return [
                'builder' => 'brizy',
                'version' => defined('BRIZY_VERSION') ? BRIZY_VERSION : 'unknown',
                'confidence' => 'high',
                'metadata' => [
                    'post_uid' => get_post_meta($post_id, 'brizy-post-uid', true)
                ]
            ];
        }

        // Check for Gutenberg (standard blocks)
        if ($post && has_blocks($post->post_content)) {
            return [
                'builder' => 'gutenberg',
                'version' => get_bloginfo('version'),
                'confidence' => 'high',
                'metadata' => [
                    'has_blocks' => true
                ]
            ];
        }

        // Default to classic
        return [
            'builder' => 'classic',
            'version' => null,
            'confidence' => 'medium',
            'metadata' => []
        ];
    }

    /**
     * Get all installed/active page builders on the site
     */
    public static function get_installed_builders($request) {
        $builders = [];

        // Check Gutenberg (always available in modern WP)
        if (function_exists('has_blocks')) {
            $builders[] = [
                'name' => 'Gutenberg',
                'slug' => 'gutenberg',
                'active' => true,
                'version' => get_bloginfo('version'),
                'is_native' => true
            ];
        }

        // Check Elementor
        if (defined('ELEMENTOR_VERSION')) {
            $builders[] = [
                'name' => 'Elementor',
                'slug' => 'elementor',
                'active' => class_exists('Elementor\Plugin'),
                'version' => ELEMENTOR_VERSION,
                'has_pro' => defined('ELEMENTOR_PRO_VERSION'),
                'pro_version' => defined('ELEMENTOR_PRO_VERSION') ? ELEMENTOR_PRO_VERSION : null
            ];
        }

        // Check Divi
        if (function_exists('et_divi_builder_init') || defined('ET_BUILDER_VERSION')) {
            $builders[] = [
                'name' => 'Divi Builder',
                'slug' => 'divi',
                'active' => true,
                'version' => defined('ET_BUILDER_VERSION') ? ET_BUILDER_VERSION : 'unknown'
            ];
        }

        // Check Beaver Builder
        if (class_exists('FLBuilder') || defined('FL_BUILDER_VERSION')) {
            $builders[] = [
                'name' => 'Beaver Builder',
                'slug' => 'beaver_builder',
                'active' => class_exists('FLBuilder'),
                'version' => defined('FL_BUILDER_VERSION') ? FL_BUILDER_VERSION : 'unknown'
            ];
        }

        // Check WPBakery
        if (defined('WPB_VC_VERSION')) {
            $builders[] = [
                'name' => 'WPBakery Page Builder',
                'slug' => 'wpbakery',
                'active' => true,
                'version' => WPB_VC_VERSION
            ];
        }

        // Check Oxygen
        if (defined('CT_VERSION')) {
            $builders[] = [
                'name' => 'Oxygen Builder',
                'slug' => 'oxygen',
                'active' => true,
                'version' => CT_VERSION
            ];
        }

        // Check Bricks
        if (defined('BRICKS_VERSION')) {
            $builders[] = [
                'name' => 'Bricks Builder',
                'slug' => 'bricks',
                'active' => true,
                'version' => BRICKS_VERSION
            ];
        }

        // Check Kadence Blocks
        if (defined('KADENCE_BLOCKS_VERSION')) {
            $builders[] = [
                'name' => 'Kadence Blocks',
                'slug' => 'kadence_blocks',
                'active' => true,
                'version' => KADENCE_BLOCKS_VERSION
            ];
        }

        // Check GenerateBlocks
        if (defined('GENERATEBLOCKS_VERSION')) {
            $builders[] = [
                'name' => 'GenerateBlocks',
                'slug' => 'generateblocks',
                'active' => true,
                'version' => GENERATEBLOCKS_VERSION
            ];
        }

        // Check Brizy
        if (defined('BRIZY_VERSION')) {
            $builders[] = [
                'name' => 'Brizy',
                'slug' => 'brizy',
                'active' => true,
                'version' => BRIZY_VERSION
            ];
        }

        return [
            'success' => true,
            'builders' => $builders,
            'total' => count($builders)
        ];
    }

    /**
     * Scan for existing templates in page builders
     */
    public static function scan_existing_templates($request) {
        $templates = [];

        // Scan Elementor templates (including drafts)
        if (class_exists('Elementor\Plugin')) {
            $elementor_templates = get_posts([
                'post_type' => 'elementor_library',
                'posts_per_page' => -1,
                'post_status' => ['publish', 'draft', 'private']
            ]);

            foreach ($elementor_templates as $template) {
                $template_type = get_post_meta($template->ID, '_elementor_template_type', true);

                // Get the actual Elementor content data
                $elementor_data = get_post_meta($template->ID, '_elementor_data', true);

                // Parse the content if it's a JSON string
                if (is_string($elementor_data)) {
                    $elementor_data = json_decode($elementor_data, true);
                }

                $templates[] = [
                    'id' => $template->ID,
                    'name' => $template->post_title,
                    'builder' => 'elementor',
                    'type' => $template_type ?: 'page',
                    'modified' => $template->post_modified,
                    'status' => $template->post_status,
                    'url' => get_permalink($template->ID),
                    'edit_url' => admin_url('post.php?post=' . $template->ID . '&action=edit'),
                    'content' => $elementor_data // Include the actual template content
                ];
            }
        }

        // Scan Beaver Builder templates (including drafts)
        if (class_exists('FLBuilder')) {
            $fl_templates = get_posts([
                'post_type' => 'fl-builder-template',
                'posts_per_page' => -1,
                'post_status' => ['publish', 'draft', 'private']
            ]);

            foreach ($fl_templates as $template) {
                // Get Beaver Builder data
                $fl_data = get_post_meta($template->ID, '_fl_builder_data', true);

                $templates[] = [
                    'id' => $template->ID,
                    'name' => $template->post_title,
                    'builder' => 'beaver_builder',
                    'type' => 'layout',
                    'modified' => $template->post_modified,
                    'status' => $template->post_status,
                    'url' => get_permalink($template->ID),
                    'edit_url' => admin_url('post.php?post=' . $template->ID . '&action=edit'),
                    'content' => $fl_data
                ];
            }
        }

        // Scan Divi Library (including drafts)
        if (function_exists('et_divi_builder_init')) {
            $divi_layouts = get_posts([
                'post_type' => 'et_pb_layout',
                'posts_per_page' => -1,
                'post_status' => ['publish', 'draft', 'private']
            ]);

            foreach ($divi_layouts as $layout) {
                // Get Divi content (stored in post_content with shortcodes)
                $divi_content = $layout->post_content;

                $templates[] = [
                    'id' => $layout->ID,
                    'name' => $layout->post_title,
                    'builder' => 'divi',
                    'type' => 'layout',
                    'modified' => $layout->post_modified,
                    'status' => $layout->post_status,
                    'url' => get_permalink($layout->ID),
                    'edit_url' => admin_url('post.php?post=' . $layout->ID . '&action=edit'),
                    'content' => $divi_content
                ];
            }
        }

        // Scan Bricks templates
        if (defined('BRICKS_VERSION')) {
            global $wpdb;

            // Try to get from cache first
            $cache_key = 'instarank_bricks_templates';
            $bricks_templates = wp_cache_get($cache_key, 'instarank');

            if (false === $bricks_templates) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
                $bricks_templates = $wpdb->get_results(
                    $wpdb->prepare(
                        "SELECT post_id
                        FROM {$wpdb->postmeta}
                        WHERE meta_key = %s
                           OR meta_key = %s
                        GROUP BY post_id
                        LIMIT %d",
                        '_bricks_page_content_2',
                        '_bricks_page_content',
                        100
                    )
                );

                // Cache for 1 hour
                wp_cache_set($cache_key, $bricks_templates, 'instarank', 3600);
            }

            foreach ($bricks_templates as $template) {
                $post = get_post($template->post_id);
                if ($post) {
                    // Get Bricks content (v2 format preferred)
                    $bricks_data = get_post_meta($post->ID, '_bricks_page_content_2', true);
                    if (!$bricks_data) {
                        $bricks_data = get_post_meta($post->ID, '_bricks_page_content', true);
                    }

                    // Parse if JSON string
                    if (is_string($bricks_data)) {
                        $bricks_data = json_decode($bricks_data, true);
                    }

                    $templates[] = [
                        'id' => $post->ID,
                        'name' => $post->post_title,
                        'builder' => 'bricks',
                        'type' => $post->post_type,
                        'modified' => $post->post_modified,
                        'url' => get_permalink($post->ID),
                        'content' => $bricks_data
                    ];
                }
            }
        }

        // Scan Kadence elements (NEW in v1.4.2)
        if (class_exists('Kadence_Blocks_Frontend')) {
            $kadence_elements = get_posts([
                'post_type' => 'kadence_element',
                'posts_per_page' => -1,
                'post_status' => ['publish', 'draft']
            ]);

            foreach ($kadence_elements as $element) {
                $element_type = get_post_meta($element->ID, '_kad_element_type', true);
                $placement = get_post_meta($element->ID, '_kad_element_placement', true);

                // Get Kadence content (Gutenberg blocks)
                $kadence_content = $element->post_content;

                $templates[] = [
                    'id' => $element->ID,
                    'name' => $element->post_title,
                    'builder' => 'kadence',
                    'type' => $element_type ?: 'element',
                    'placement' => $placement,
                    'modified' => $element->post_modified,
                    'status' => $element->post_status,
                    'url' => admin_url('post.php?post=' . $element->ID . '&action=edit'),
                    'content' => $kadence_content
                ];
            }
        }

        // Scan Gutenberg reusable blocks (including drafts)
        $reusable_blocks = get_posts([
            'post_type' => 'wp_block',
            'posts_per_page' => -1,
            'post_status' => ['publish', 'draft', 'private']
        ]);

        foreach ($reusable_blocks as $block) {
            // Get Gutenberg content (block markup)
            $gutenberg_content = $block->post_content;

            $templates[] = [
                'id' => $block->ID,
                'name' => $block->post_title,
                'builder' => 'gutenberg',
                'type' => 'reusable_block',
                'modified' => $block->post_modified,
                'status' => $block->post_status,
                'url' => admin_url('post.php?post=' . $block->ID . '&action=edit'),
                'edit_url' => admin_url('post.php?post=' . $block->ID . '&action=edit'),
                'content' => $gutenberg_content
            ];
        }

        // Scan regular pages and posts that have page builder content (NEW in v1.4.3)
        // This includes any pages built with Kadence Blocks or other Gutenberg-based builders
        $regular_pages = get_posts([
            'post_type' => ['page', 'post'],
            'posts_per_page' => 100, // Limit to prevent overwhelming the system
            'post_status' => ['publish', 'draft'],
            'orderby' => 'modified',
            'order' => 'DESC'
        ]);

        foreach ($regular_pages as $page) {
            // Skip if already in templates list (e.g., from Bricks scan)
            $already_included = false;
            foreach ($templates as $existing_template) {
                if ($existing_template['id'] === $page->ID) {
                    $already_included = true;
                    break;
                }
            }
            if ($already_included) {
                continue;
            }

            // Detect page builder for this page
            $detection = self::detect_page_builder($page->ID);

            // Only include if it has a page builder (not classic editor)
            if ($detection['builder'] !== 'classic') {
                $page_content = $page->post_content;

                // For non-Gutenberg builders, get the meta data
                if ($detection['builder'] === 'kadence_blocks') {
                    // It's already in post_content
                } elseif ($detection['builder'] === 'elementor') {
                    $page_content = get_post_meta($page->ID, '_elementor_data', true);
                } elseif ($detection['builder'] === 'bricks') {
                    $page_content = get_post_meta($page->ID, '_bricks_page_content_2', true);
                    if (!$page_content) {
                        $page_content = get_post_meta($page->ID, '_bricks_page_content', true);
                    }
                }

                $templates[] = [
                    'id' => $page->ID,
                    'name' => $page->post_title,
                    'builder' => $detection['builder'],
                    'type' => $page->post_type,
                    'modified' => $page->post_modified,
                    'status' => $page->post_status,
                    'url' => get_permalink($page->ID),
                    'edit_url' => admin_url('post.php?post=' . $page->ID . '&action=edit'),
                    'content' => $page_content
                ];
            }
        }

        return [
            'success' => true,
            'templates' => $templates,
            'total' => count($templates),
            'scanned_at' => current_time('mysql')
        ];
    }

    /**
     * Import and parse template content
     */
    public static function import_template($request) {
        $params = $request->get_json_params();
        $template_content = $params['template_content'];
        $builder = $params['builder'];

        if (!$template_content || !$builder) {
            return new WP_Error('missing_params', 'Missing template_content or builder', ['status' => 400]);
        }

        // Parse the template based on builder type
        $template_schema = [];
        $widget_count = 0;

        switch ($builder) {
            case 'elementor':
                $result = self::parse_elementor_template($template_content);
                $template_schema = $result['schema'];
                $widget_count = $result['widget_count'];
                break;

            case 'kadence':
            case 'gutenberg':
                $result = self::parse_gutenberg_template($template_content);
                $template_schema = $result['schema'];
                $widget_count = $result['widget_count'];
                break;

            case 'bricks':
                $result = self::parse_bricks_template($template_content);
                $template_schema = $result['schema'];
                $widget_count = $result['widget_count'];
                break;

            default:
                return new WP_Error('unsupported_builder', "Builder '$builder' is not supported yet", ['status' => 400]);
        }

        return [
            'success' => true,
            'builder' => $builder,
            'widget_count' => $widget_count,
            'template_schema' => $template_schema,
            'parsed_at' => current_time('mysql')
        ];
    }

    /**
     * Parse Elementor template and extract schema
     */
    private static function parse_elementor_template($template_content) {
        $schema = [];
        $widget_count = 0;

        // Ensure content is an array
        if (is_string($template_content)) {
            $template_content = json_decode($template_content, true);
        }

        if (!is_array($template_content)) {
            return ['schema' => [], 'widget_count' => 0];
        }

        // Recursively parse Elementor elements
        $position_index = 0;
        self::parse_elementor_elements($template_content, $schema, $widget_count, $position_index, null, 0);

        return [
            'schema' => $schema,
            'widget_count' => $widget_count
        ];
    }

    /**
     * Recursively parse Elementor elements
     */
    private static function parse_elementor_elements($elements, &$schema, &$widget_count, &$position_index, $parent_id = null, $level = 0) {
        foreach ($elements as $element) {
            $element_type = $element['elType'] ?? null;
            $widget_type = $element['widgetType'] ?? null;

            // Add to schema if it's a widget
            if ($element_type === 'widget' && $widget_type) {
                $widget_count++;

                $schema[] = [
                    'widget_id' => $element['id'] ?? uniqid('widget_'),
                    'widget_type' => $widget_type,
                    'widget_label' => self::get_elementor_widget_label($widget_type),
                    'section_name' => $parent_id,
                    'section_type' => 'elementor_section',
                    'available_properties' => self::get_elementor_widget_properties($element),
                    'parent_widget_id' => $parent_id,
                    'nesting_level' => $level,
                    'position_index' => $position_index++,
                    'is_repeatable' => false,
                    'is_conditional' => false
                ];
            }

            // Recursively parse child elements
            if (isset($element['elements']) && is_array($element['elements'])) {
                self::parse_elementor_elements(
                    $element['elements'],
                    $schema,
                    $widget_count,
                    $position_index,
                    $element['id'] ?? $parent_id,
                    $level + 1
                );
            }
        }
    }

    /**
     * Get human-readable label for Elementor widget
     */
    private static function get_elementor_widget_label($widget_type) {
        $labels = [
            'heading' => 'Heading',
            'text-editor' => 'Text Editor',
            'image' => 'Image',
            'button' => 'Button',
            'divider' => 'Divider',
            'spacer' => 'Spacer',
            'icon' => 'Icon',
            'image-box' => 'Image Box',
            'testimonial' => 'Testimonial',
            'tabs' => 'Tabs',
            'accordion' => 'Accordion',
            'toggle' => 'Toggle',
            'social-icons' => 'Social Icons',
            'alert' => 'Alert',
            'html' => 'HTML',
            'shortcode' => 'Shortcode',
        ];

        return $labels[$widget_type] ?? ucwords(str_replace('-', ' ', $widget_type));
    }

    /**
     * Get available properties for Elementor widget
     */
    private static function get_elementor_widget_properties($element) {
        $settings = $element['settings'] ?? [];
        $properties = [];

        foreach ($settings as $key => $value) {
            if (is_string($value) || is_numeric($value)) {
                $properties[$key] = [
                    'type' => gettype($value),
                    'path' => "settings.{$key}", // Add property path
                    'value_type' => is_numeric($value) ? 'number' : 'string'
                ];
            }
        }

        return $properties;
    }

    /**
     * Parse Gutenberg/Kadence template
     */
    private static function parse_gutenberg_template($template_content) {
        // For now, return basic structure
        // TODO: Implement proper Gutenberg block parsing
        return [
            'schema' => [],
            'widget_count' => 0
        ];
    }

    /**
     * Parse Bricks template
     */
    private static function parse_bricks_template($template_content) {
        // For now, return basic structure
        // TODO: Implement proper Bricks parsing
        return [
            'schema' => [],
            'widget_count' => 0
        ];
    }

    /**
     * Check API permissions
     */
    public static function check_permissions($request) {
        // Check for API key in header
        $api_key = $request->get_header('X-WordPress-API-Key');

        if (!$api_key) {
            return false;
        }

        // Get stored API key from options
        $stored_key = get_option('instarank_api_key');

        if (!$stored_key) {
            return false;
        }

        // Verify API key matches
        return hash_equals($stored_key, $api_key);
    }
}

// Initialize the API
InstaRank_Page_Builder_API::init();
