<?php
/**
 * Multi-Language Integration
 *
 * Handles WPML and Polylang integration for programmatic SEO pages.
 * Sets language for created posts and links translations.
 *
 * This is backend functionality only - all configuration is done in the SaaS.
 */

defined('ABSPATH') || exit;

class InstaRank_MultiLang {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Detected translation plugin
     */
    private $plugin = null;

    /**
     * Get singleton instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        // Detect translation plugin
        $this->detect_plugin();
    }

    /**
     * Detect which translation plugin is active
     */
    private function detect_plugin() {
        if (defined('ICL_SITEPRESS_VERSION')) {
            $this->plugin = 'wpml';
        } elseif (function_exists('pll_current_language') || defined('POLYLANG_VERSION')) {
            $this->plugin = 'polylang';
        } else {
            $this->plugin = null;
        }
    }

    /**
     * Check if a translation plugin is active
     */
    public function is_active() {
        return !is_null($this->plugin);
    }

    /**
     * Get the detected plugin name
     */
    public function get_plugin_name() {
        return $this->plugin;
    }

    /**
     * Get available languages
     */
    public function get_languages() {
        if ($this->plugin === 'wpml') {
            return $this->get_wpml_languages();
        } elseif ($this->plugin === 'polylang') {
            return $this->get_polylang_languages();
        }
        return [];
    }

    /**
     * Get WPML languages
     */
    private function get_wpml_languages() {
        if (!function_exists('icl_get_languages')) {
            return [];
        }

        $languages = icl_get_languages('skip_missing=0');
        $result = [];

        foreach ($languages as $code => $lang) {
            $result[] = [
                'code' => $code,
                'name' => $lang['native_name'],
                'default' => $lang['active'] == 1,
            ];
        }

        return $result;
    }

    /**
     * Get Polylang languages
     */
    private function get_polylang_languages() {
        if (!function_exists('pll_languages_list')) {
            return [];
        }

        $languages = pll_languages_list(['fields' => []]);
        $default_lang = pll_default_language();
        $result = [];

        foreach ($languages as $lang) {
            $result[] = [
                'code' => $lang->slug,
                'name' => $lang->name,
                'default' => $lang->slug === $default_lang,
            ];
        }

        return $result;
    }

    /**
     * Set language for a post
     *
     * @param int    $post_id  The post ID
     * @param string $lang     The language code (e.g., 'en', 'es')
     * @return bool  Success status
     */
    public function set_post_language($post_id, $lang) {
        if (!$post_id || !$lang) {
            return false;
        }

        if ($this->plugin === 'wpml') {
            return $this->set_wpml_language($post_id, $lang);
        } elseif ($this->plugin === 'polylang') {
            return $this->set_polylang_language($post_id, $lang);
        }

        return false;
    }

    /**
     * Set WPML language for a post
     */
    private function set_wpml_language($post_id, $lang) {
        global $sitepress;

        if (!$sitepress || !method_exists($sitepress, 'set_element_language_details')) {
            return false;
        }

        $post_type = get_post_type($post_id);

        // Get the translation element type
        $element_type = 'post_' . $post_type;

        // Get default language for trid
        $default_lang = $sitepress->get_default_language();

        // Set language details
        $sitepress->set_element_language_details(
            $post_id,
            $element_type,
            null, // trid - null for new, or existing trid for translations
            $lang,
            $default_lang
        );

        return true;
    }

    /**
     * Set Polylang language for a post
     */
    private function set_polylang_language($post_id, $lang) {
        if (!function_exists('pll_set_post_language')) {
            return false;
        }

        pll_set_post_language($post_id, $lang);
        return true;
    }

    /**
     * Link posts as translations of each other
     *
     * @param array $translations Array of [lang => post_id] pairs
     * @return bool Success status
     */
    public function link_translations($translations) {
        if (empty($translations) || count($translations) < 2) {
            return false;
        }

        if ($this->plugin === 'wpml') {
            return $this->link_wpml_translations($translations);
        } elseif ($this->plugin === 'polylang') {
            return $this->link_polylang_translations($translations);
        }

        return false;
    }

    /**
     * Link WPML translations
     */
    private function link_wpml_translations($translations) {
        global $sitepress, $wpdb;

        if (!$sitepress) {
            return false;
        }

        // Get the first post to determine element type and get/create trid
        $first_post_id = reset($translations);
        $first_lang = key($translations);
        $post_type = get_post_type($first_post_id);
        $element_type = 'post_' . $post_type;

        // Get existing trid or create new one
        $trid = $sitepress->get_element_trid($first_post_id, $element_type);

        if (!$trid) {
            // Create new translation set - direct query needed for WPML's custom table
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $wpdb->insert(
                $wpdb->prefix . 'icl_translations',
                [
                    'element_type' => $element_type,
                    'element_id' => $first_post_id,
                    'language_code' => $first_lang,
                    'source_language_code' => null,
                ]
            );
            $trid = $wpdb->insert_id;
        }

        // Link all translations to this trid
        foreach ($translations as $lang => $post_id) {
            if ($post_id === $first_post_id) {
                continue;
            }

            $sitepress->set_element_language_details(
                $post_id,
                $element_type,
                $trid,
                $lang,
                $first_lang // source language
            );
        }

        return true;
    }

    /**
     * Link Polylang translations
     */
    private function link_polylang_translations($translations) {
        if (!function_exists('pll_save_post_translations')) {
            return false;
        }

        pll_save_post_translations($translations);
        return true;
    }

    /**
     * Get translation of a post
     *
     * @param int    $post_id The post ID
     * @param string $lang    The target language code
     * @return int|null The translated post ID or null
     */
    public function get_translation($post_id, $lang) {
        if ($this->plugin === 'wpml') {
            return $this->get_wpml_translation($post_id, $lang);
        } elseif ($this->plugin === 'polylang') {
            return $this->get_polylang_translation($post_id, $lang);
        }
        return null;
    }

    /**
     * Get WPML translation
     */
    private function get_wpml_translation($post_id, $lang) {
        if (!function_exists('icl_object_id')) {
            return null;
        }

        $translated_id = icl_object_id($post_id, get_post_type($post_id), false, $lang);
        return $translated_id ?: null;
    }

    /**
     * Get Polylang translation
     */
    private function get_polylang_translation($post_id, $lang) {
        if (!function_exists('pll_get_post')) {
            return null;
        }

        $translated_id = pll_get_post($post_id, $lang);
        return $translated_id ?: null;
    }

    /**
     * Get post language
     *
     * @param int $post_id The post ID
     * @return string|null The language code or null
     */
    public function get_post_language($post_id) {
        if ($this->plugin === 'wpml') {
            return $this->get_wpml_post_language($post_id);
        } elseif ($this->plugin === 'polylang') {
            return $this->get_polylang_post_language($post_id);
        }
        return null;
    }

    /**
     * Get WPML post language
     */
    private function get_wpml_post_language($post_id) {
        global $sitepress;

        if (!$sitepress) {
            return null;
        }

        $lang_info = $sitepress->get_element_language_details(
            $post_id,
            'post_' . get_post_type($post_id)
        );

        return $lang_info ? $lang_info->language_code : null;
    }

    /**
     * Get Polylang post language
     */
    private function get_polylang_post_language($post_id) {
        if (!function_exists('pll_get_post_language')) {
            return null;
        }

        return pll_get_post_language($post_id);
    }

    /**
     * Get all translations of a post
     *
     * @param int $post_id The post ID
     * @return array [lang => post_id] pairs
     */
    public function get_all_translations($post_id) {
        if ($this->plugin === 'wpml') {
            return $this->get_wpml_all_translations($post_id);
        } elseif ($this->plugin === 'polylang') {
            return $this->get_polylang_all_translations($post_id);
        }
        return [];
    }

    /**
     * Get all WPML translations
     */
    private function get_wpml_all_translations($post_id) {
        global $sitepress;

        if (!$sitepress) {
            return [];
        }

        $post_type = get_post_type($post_id);
        $trid = $sitepress->get_element_trid($post_id, 'post_' . $post_type);

        if (!$trid) {
            return [];
        }

        $translations = $sitepress->get_element_translations($trid, 'post_' . $post_type);
        $result = [];

        foreach ($translations as $lang => $translation) {
            if (!empty($translation->element_id)) {
                $result[$lang] = (int) $translation->element_id;
            }
        }

        return $result;
    }

    /**
     * Get all Polylang translations
     */
    private function get_polylang_all_translations($post_id) {
        if (!function_exists('pll_get_post_translations')) {
            return [];
        }

        return pll_get_post_translations($post_id);
    }

    /**
     * Get plugin info for API response
     */
    public function get_info() {
        return [
            'active' => $this->is_active(),
            'plugin' => $this->plugin,
            'languages' => $this->get_languages(),
        ];
    }
}

// Initialize
InstaRank_MultiLang::instance();
