<?php
/**
 * LLMs.txt Manager
 *
 * Manages llms.txt file for AI/LLM crawler guidance
 * Following the llmstxt.org specification
 *
 * @package InstaRank
 * @since 2.1.0
 */

defined('ABSPATH') || exit;

class InstaRank_LLMs_Txt {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('init', [$this, 'add_rewrite_rules']);
        add_action('template_redirect', [$this, 'handle_llms_txt_request']);
        add_filter('query_vars', [$this, 'add_query_vars']);
    }

    /**
     * Add rewrite rules for /llms.txt
     */
    public function add_rewrite_rules() {
        add_rewrite_rule('^llms\.txt$', 'index.php?instarank_llms_txt=1', 'top');
    }

    /**
     * Add query vars
     */
    public function add_query_vars($vars) {
        $vars[] = 'instarank_llms_txt';
        return $vars;
    }

    /**
     * Handle /llms.txt requests
     */
    public function handle_llms_txt_request() {
        $is_llms_request = get_query_var('instarank_llms_txt');

        if (!$is_llms_request) {
            return;
        }

        // Check if LLMs.txt is enabled
        if (!$this->is_enabled()) {
            status_header(404);
            exit;
        }

        // Set headers
        header('Content-Type: text/plain; charset=utf-8');
        header('X-Robots-Tag: noindex, follow');
        header('Cache-Control: public, max-age=86400');

        // Output content - using esc_html for security
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Content is plain text, properly sanitized
        echo esc_html( $this->get_content() );
        exit;
    }

    /**
     * Check if LLMs.txt is enabled
     */
    public function is_enabled() {
        return (bool) get_option('instarank_llms_txt_enabled', true);
    }

    /**
     * Enable LLMs.txt
     */
    public function enable() {
        update_option('instarank_llms_txt_enabled', true);
        flush_rewrite_rules();
    }

    /**
     * Disable LLMs.txt
     */
    public function disable() {
        update_option('instarank_llms_txt_enabled', false);
    }

    /**
     * Get current LLMs.txt content
     */
    public function get_content() {
        $custom = get_option('instarank_llms_txt', '');

        if (!empty($custom)) {
            return $custom;
        }

        return $this->get_default_content();
    }

    /**
     * Save LLMs.txt content
     */
    public function save_content($content) {
        // Sanitize content
        $content = sanitize_textarea_field($content);

        // Validate
        if (!$this->validate_content($content)) {
            return false;
        }

        update_option('instarank_llms_txt', $content);

        // Send webhook notification if class exists
        if (class_exists('InstaRank_Webhook_Sender')) {
            $webhook = new InstaRank_Webhook_Sender();
            $webhook->send_settings_updated([
                'llms_txt_updated' => true,
                'message' => 'LLMs.txt content updated'
            ]);
        }

        return true;
    }

    /**
     * Reset to default content
     */
    public function reset_to_default() {
        delete_option('instarank_llms_txt');
        return true;
    }

    /**
     * Validate LLMs.txt content
     */
    private function validate_content($content) {
        // Max size: 100KB
        if (strlen($content) > 100000) {
            return false;
        }

        return true;
    }

    /**
     * Get default LLMs.txt content following llmstxt.org specification
     */
    public function get_default_content() {
        $site_name = get_bloginfo('name');
        $site_description = get_bloginfo('description');
        $site_url = home_url();
        $admin_email = get_option('admin_email');

        $default = "# {$site_name}\n\n";

        if (!empty($site_description)) {
            $default .= "> {$site_description}\n\n";
        }

        $default .= "## About\n";
        $default .= "This is the official website for {$site_name}.\n";
        $default .= "URL: {$site_url}\n\n";

        $default .= "## Content\n";
        $default .= "- Blog posts and articles\n";
        $default .= "- Product/service information\n";
        $default .= "- Company updates\n\n";

        $default .= "## Contact\n";
        if (!empty($admin_email)) {
            $default .= "Email: {$admin_email}\n";
        }
        $default .= "Website: {$site_url}\n\n";

        $default .= "## Policies\n";
        $default .= "- Content may be used to answer questions about {$site_name}\n";
        $default .= "- Attribution is appreciated when quoting directly\n";
        $default .= "- For training purposes, please contact us first\n";

        return apply_filters('instarank_default_llms_txt', $default);
    }

    /**
     * Test if llms.txt is accessible
     */
    public function is_accessible() {
        $url = home_url('/llms.txt');
        $response = wp_remote_get($url, ['timeout' => 5]);

        if (is_wp_error($response)) {
            return false;
        }

        return wp_remote_retrieve_response_code($response) === 200;
    }

    /**
     * Get the URL to llms.txt
     */
    public function get_url() {
        return home_url('/llms.txt');
    }
}

// Initialize
InstaRank_LLMs_Txt::instance();
