<?php
/**
 * IndexNow Integration for WordPress
 *
 * Automatically submits URLs to search engines when content is published or updated.
 * IndexNow is supported by Bing, Yandex, Seznam, and Naver.
 *
 * @package InstaRank
 * @since 1.6.0
 */

defined('ABSPATH') || exit;

class InstaRank_IndexNow {

    /**
     * IndexNow API endpoints
     */
    private const ENDPOINTS = [
        'bing'   => 'https://www.bing.com/indexnow',
        'yandex' => 'https://yandex.com/indexnow',
        'seznam' => 'https://search.seznam.cz/indexnow',
        'naver'  => 'https://searchadvisor.naver.com/indexnow',
    ];

    /**
     * Option name for API key
     */
    private const OPTION_API_KEY = 'instarank_indexnow_api_key';

    /**
     * Option name for enabled status
     */
    private const OPTION_ENABLED = 'instarank_indexnow_enabled';

    /**
     * Option name for search engines
     */
    private const OPTION_ENGINES = 'instarank_indexnow_engines';

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Auto-submit on post publish
        add_action('publish_post', [$this, 'on_post_publish'], 10, 2);
        add_action('publish_page', [$this, 'on_post_publish'], 10, 2);

        // Hook into custom post types
        add_action('wp_insert_post', [$this, 'on_post_insert'], 10, 3);

        // Add settings section
        add_action('admin_init', [$this, 'register_settings']);

        // Add REST API endpoint for manual submissions
        add_action('rest_api_init', [$this, 'register_rest_routes']);

        // Add rewrite rule for key file
        add_action('init', [$this, 'add_key_file_rewrite']);
        add_action('template_redirect', [$this, 'serve_key_file']);
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('instarank_indexnow', self::OPTION_API_KEY, [
            'type'              => 'string',
            'sanitize_callback' => [$this, 'sanitize_api_key'],
            'default'           => '',
        ]);

        register_setting('instarank_indexnow', self::OPTION_ENABLED, [
            'type'              => 'boolean',
            'sanitize_callback' => 'rest_sanitize_boolean',
            'default'           => false,
        ]);

        register_setting('instarank_indexnow', self::OPTION_ENGINES, [
            'type'              => 'array',
            'sanitize_callback' => [$this, 'sanitize_engines'],
            'default'           => ['bing'],
        ]);
    }

    /**
     * Sanitize API key
     */
    public function sanitize_api_key($key) {
        $key = sanitize_text_field($key);
        // Validate hex format (8-128 chars)
        if (!empty($key) && !preg_match('/^[0-9a-fA-F]{8,128}$/', $key)) {
            add_settings_error(
                self::OPTION_API_KEY,
                'invalid_key',
                __('IndexNow API key must be 8-128 hexadecimal characters.', 'instarank')
            );
            return get_option(self::OPTION_API_KEY, '');
        }
        return strtolower($key);
    }

    /**
     * Sanitize engines array
     */
    public function sanitize_engines($engines) {
        if (!is_array($engines)) {
            return ['bing'];
        }
        return array_intersect($engines, array_keys(self::ENDPOINTS));
    }

    /**
     * Generate a new API key
     */
    public static function generate_api_key() {
        $length = 32;
        $chars = '0123456789abcdef';
        $key = '';
        for ($i = 0; $i < $length; $i++) {
            $key .= $chars[wp_rand(0, 15)];
        }
        return $key;
    }

    /**
     * Get the current API key
     */
    public function get_api_key() {
        return get_option(self::OPTION_API_KEY, '');
    }

    /**
     * Set the API key
     */
    public function set_api_key($key) {
        return update_option(self::OPTION_API_KEY, $this->sanitize_api_key($key));
    }

    /**
     * Check if IndexNow is enabled
     */
    public function is_enabled() {
        return (bool) get_option(self::OPTION_ENABLED, false);
    }

    /**
     * Enable/disable IndexNow
     */
    public function set_enabled($enabled) {
        return update_option(self::OPTION_ENABLED, (bool) $enabled);
    }

    /**
     * Get enabled search engines
     */
    public function get_engines() {
        return get_option(self::OPTION_ENGINES, ['bing']);
    }

    /**
     * Get the key file URL
     */
    public function get_key_file_url() {
        $key = $this->get_api_key();
        if (empty($key)) {
            return '';
        }
        return home_url($key . '.txt');
    }

    /**
     * Add rewrite rule for serving the key file
     */
    public function add_key_file_rewrite() {
        $key = $this->get_api_key();
        if (!empty($key)) {
            add_rewrite_rule(
                '^' . preg_quote($key, '/') . '\.txt$',
                'index.php?instarank_indexnow_key=1',
                'top'
            );
            add_rewrite_tag('%instarank_indexnow_key%', '1');
        }
    }

    /**
     * Serve the key file content
     */
    public function serve_key_file() {
        if (get_query_var('instarank_indexnow_key') === '1') {
            $key = $this->get_api_key();
            if (!empty($key)) {
                header('Content-Type: text/plain; charset=utf-8');
                header('Cache-Control: public, max-age=86400');
                echo esc_html($key);
                exit;
            }
        }

        // Also check for direct file request
        $request_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        $key = $this->get_api_key();
        if (!empty($key) && preg_match('/\/' . preg_quote($key, '/') . '\.txt$/', $request_uri)) {
            header('Content-Type: text/plain; charset=utf-8');
            header('Cache-Control: public, max-age=86400');
            echo esc_html($key);
            exit;
        }
    }

    /**
     * Handle post publish event
     */
    public function on_post_publish($post_id, $post) {
        if (!$this->is_enabled()) {
            return;
        }

        // Don't process revisions
        if (wp_is_post_revision($post_id)) {
            return;
        }

        // Get the post URL
        $url = get_permalink($post_id);
        if (!$url) {
            return;
        }

        // Submit to IndexNow
        $this->submit_url($url);
    }

    /**
     * Handle post insert event (for custom post types)
     */
    public function on_post_insert($post_id, $post, $update) {
        if (!$this->is_enabled()) {
            return;
        }

        // Only process on publish
        if ($post->post_status !== 'publish') {
            return;
        }

        // Don't process revisions
        if (wp_is_post_revision($post_id)) {
            return;
        }

        // Skip built-in post types (handled by specific hooks)
        if (in_array($post->post_type, ['post', 'page'], true)) {
            return;
        }

        // Get the post URL
        $url = get_permalink($post_id);
        if (!$url) {
            return;
        }

        // Submit to IndexNow
        $this->submit_url($url);
    }

    /**
     * Submit a single URL to IndexNow
     */
    public function submit_url($url, $engine = null) {
        $key = $this->get_api_key();
        if (empty($key)) {
            return new WP_Error('no_api_key', __('IndexNow API key not configured.', 'instarank'));
        }

        $host = wp_parse_url(home_url(), PHP_URL_HOST);
        $engines = $engine ? [$engine] : $this->get_engines();
        $results = [];

        foreach ($engines as $engine_name) {
            if (!isset(self::ENDPOINTS[$engine_name])) {
                continue;
            }

            $endpoint = self::ENDPOINTS[$engine_name];
            $request_url = add_query_arg([
                'url' => $url,
                'key' => $key,
            ], $endpoint);

            $response = wp_remote_get($request_url, [
                'timeout'    => 10,
                'user-agent' => 'InstaRank-WordPress/' . INSTARANK_VERSION,
            ]);

            if (is_wp_error($response)) {
                $results[$engine_name] = [
                    'success' => false,
                    'error'   => $response->get_error_message(),
                ];
                continue;
            }

            $status_code = wp_remote_retrieve_response_code($response);

            if ($status_code === 200 || $status_code === 202) {
                $results[$engine_name] = [
                    'success'     => true,
                    'status_code' => $status_code,
                    'message'     => $status_code === 200
                        ? __('URL submitted successfully.', 'instarank')
                        : __('URL received, key validation pending.', 'instarank'),
                ];

                // Log successful submission
                $this->log_submission($url, $engine_name, true);
            } else {
                $results[$engine_name] = [
                    'success'     => false,
                    'status_code' => $status_code,
                    'error'       => $this->get_error_message($status_code),
                ];

                // Log failed submission
                $this->log_submission($url, $engine_name, false, $status_code);
            }
        }

        return $results;
    }

    /**
     * Submit multiple URLs to IndexNow (batch)
     */
    public function submit_urls($urls, $engine = 'bing') {
        $key = $this->get_api_key();
        if (empty($key)) {
            return new WP_Error('no_api_key', __('IndexNow API key not configured.', 'instarank'));
        }

        if (!isset(self::ENDPOINTS[$engine])) {
            return new WP_Error('invalid_engine', __('Invalid search engine specified.', 'instarank'));
        }

        $host = wp_parse_url(home_url(), PHP_URL_HOST);

        // Filter URLs to only include those from this host
        $valid_urls = array_filter($urls, function($url) use ($host) {
            return wp_parse_url($url, PHP_URL_HOST) === $host;
        });

        if (empty($valid_urls)) {
            return new WP_Error('no_valid_urls', __('No valid URLs to submit.', 'instarank'));
        }

        // Split into batches of 10,000 (IndexNow limit)
        $batches = array_chunk($valid_urls, 10000);
        $total_submitted = 0;
        $total_failed = 0;

        foreach ($batches as $batch) {
            $response = wp_remote_post(self::ENDPOINTS[$engine], [
                'timeout'    => 30,
                'user-agent' => 'InstaRank-WordPress/' . INSTARANK_VERSION,
                'headers'    => [
                    'Content-Type' => 'application/json; charset=utf-8',
                ],
                'body'       => wp_json_encode([
                    'host'        => $host,
                    'key'         => $key,
                    'keyLocation' => $this->get_key_file_url(),
                    'urlList'     => array_values($batch),
                ]),
            ]);

            if (is_wp_error($response)) {
                $total_failed += count($batch);
                continue;
            }

            $status_code = wp_remote_retrieve_response_code($response);

            if ($status_code === 200 || $status_code === 202) {
                $total_submitted += count($batch);
            } else {
                $total_failed += count($batch);
            }
        }

        return [
            'total_urls' => count($urls),
            'submitted'  => $total_submitted,
            'failed'     => $total_failed + (count($urls) - count($valid_urls)),
        ];
    }

    /**
     * Get human-readable error message
     */
    private function get_error_message($status_code) {
        switch ($status_code) {
            case 400:
                return __('Invalid request - check URL format.', 'instarank');
            case 403:
                return __('API key not valid - ensure key file is accessible.', 'instarank');
            case 422:
                return __('URL does not belong to the specified host.', 'instarank');
            case 429:
                return __('Too many requests - rate limit exceeded.', 'instarank');
            default:
                /* translators: %d: HTTP status code */
                return sprintf(__('Unexpected status code: %d', 'instarank'), $status_code);
        }
    }

    /**
     * Log submission attempt
     */
    private function log_submission($url, $engine, $success, $status_code = null) {
        $log = get_option('instarank_indexnow_log', []);

        // Keep only last 100 entries
        if (count($log) >= 100) {
            $log = array_slice($log, -99);
        }

        $log[] = [
            'url'         => $url,
            'engine'      => $engine,
            'success'     => $success,
            'status_code' => $status_code,
            'timestamp'   => current_time('mysql'),
        ];

        update_option('instarank_indexnow_log', $log, false);
    }

    /**
     * Get submission log
     */
    public function get_log($limit = 20) {
        $log = get_option('instarank_indexnow_log', []);
        return array_slice(array_reverse($log), 0, $limit);
    }

    /**
     * Clear submission log
     */
    public function clear_log() {
        return delete_option('instarank_indexnow_log');
    }

    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        register_rest_route('instarank/v1', '/indexnow/submit', [
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => [$this, 'rest_submit_urls'],
            'permission_callback' => [$this, 'rest_permission_check'],
            'args'                => [
                'urls' => [
                    'required'          => true,
                    'type'              => 'array',
                    'items'             => ['type' => 'string'],
                    'sanitize_callback' => function($urls) {
                        return array_map('esc_url_raw', $urls);
                    },
                ],
                'engine' => [
                    'required' => false,
                    'type'     => 'string',
                    'default'  => 'bing',
                    'enum'     => array_keys(self::ENDPOINTS),
                ],
            ],
        ]);

        register_rest_route('instarank/v1', '/indexnow/config', [
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => [$this, 'rest_get_config'],
            'permission_callback' => [$this, 'rest_permission_check'],
        ]);

        register_rest_route('instarank/v1', '/indexnow/config', [
            'methods'             => WP_REST_Server::EDITABLE,
            'callback'            => [$this, 'rest_update_config'],
            'permission_callback' => [$this, 'rest_permission_check'],
            'args'                => [
                'enabled'      => ['type' => 'boolean'],
                'api_key'      => ['type' => 'string'],
                'generate_key' => ['type' => 'boolean'],
                'engines'      => ['type' => 'array'],
            ],
        ]);

        register_rest_route('instarank/v1', '/indexnow/log', [
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => [$this, 'rest_get_log'],
            'permission_callback' => [$this, 'rest_permission_check'],
            'args'                => [
                'limit' => [
                    'type'    => 'integer',
                    'default' => 20,
                    'minimum' => 1,
                    'maximum' => 100,
                ],
            ],
        ]);
    }

    /**
     * REST API permission check
     */
    public function rest_permission_check($request) {
        // Check for API key in header
        $api_key = $request->get_header('X-WordPress-API-Key');
        if (!empty($api_key)) {
            $stored_key = get_option('instarank_api_key', '');
            if ($api_key === $stored_key) {
                return true;
            }
        }

        // Fall back to capability check
        return current_user_can('manage_options');
    }

    /**
     * REST API: Submit URLs
     */
    public function rest_submit_urls($request) {
        $urls = $request->get_param('urls');
        $engine = $request->get_param('engine');

        if (count($urls) === 1) {
            $result = $this->submit_url($urls[0], $engine);
        } else {
            $result = $this->submit_urls($urls, $engine);
        }

        if (is_wp_error($result)) {
            return new WP_REST_Response([
                'success' => false,
                'error'   => $result->get_error_message(),
            ], 400);
        }

        return new WP_REST_Response([
            'success' => true,
            'data'    => $result,
        ], 200);
    }

    /**
     * REST API: Get config
     */
    public function rest_get_config($request) {
        return new WP_REST_Response([
            'success' => true,
            'data'    => [
                'enabled'      => $this->is_enabled(),
                'api_key'      => $this->get_api_key(),
                'key_file_url' => $this->get_key_file_url(),
                'engines'      => $this->get_engines(),
                'host'         => wp_parse_url(home_url(), PHP_URL_HOST),
            ],
        ], 200);
    }

    /**
     * REST API: Update config
     */
    public function rest_update_config($request) {
        if ($request->has_param('generate_key') && $request->get_param('generate_key')) {
            $new_key = self::generate_api_key();
            $this->set_api_key($new_key);

            // Flush rewrite rules for new key file
            flush_rewrite_rules();
        } elseif ($request->has_param('api_key')) {
            $this->set_api_key($request->get_param('api_key'));
            flush_rewrite_rules();
        }

        if ($request->has_param('enabled')) {
            $this->set_enabled($request->get_param('enabled'));
        }

        if ($request->has_param('engines')) {
            update_option(self::OPTION_ENGINES, $this->sanitize_engines($request->get_param('engines')));
        }

        return $this->rest_get_config($request);
    }

    /**
     * REST API: Get log
     */
    public function rest_get_log($request) {
        $limit = $request->get_param('limit');
        return new WP_REST_Response([
            'success' => true,
            'data'    => $this->get_log($limit),
        ], 200);
    }
}

// Initialize
InstaRank_IndexNow::instance();
