<?php
/**
 * InstaRank Field Detector
 *
 * Detects custom fields used in page builder templates for programmatic SEO
 * Supports: Kadence, Elementor, Bricks, Oxygen, Gutenberg
 *
 * @package InstaRank
 * @since 1.0.0
 */

defined('ABSPATH') || exit;

class InstaRank_Field_Detector {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Detect custom fields in a post/template
     *
     * @param int $post_id Post ID
     * @return array Detected fields with metadata
     */
    public function detect_fields($post_id) {
        $post = get_post($post_id);

        if (!$post) {
            return [
                'success' => false,
                'error' => 'Post not found',
                'fields' => []
            ];
        }

        // Detect page builder
        $page_builder = $this->detect_page_builder($post_id);

        // Get post content
        $content = $post->post_content;

        // Detect fields based on page builder
        $fields = [];

        switch ($page_builder) {
            case 'kadence':
                $fields = $this->detect_kadence_fields($post_id, $content);
                break;

            case 'elementor':
                $fields = $this->detect_elementor_fields($post_id);
                break;

            case 'bricks':
                $fields = $this->detect_bricks_fields($post_id);
                break;

            case 'oxygen':
                $fields = $this->detect_oxygen_fields($post_id, $content);
                break;

            case 'gutenberg':
            default:
                $fields = $this->detect_gutenberg_fields($content);
                break;
        }

        // Deduplicate fields by name
        $unique_fields = [];
        foreach ($fields as $field) {
            $field_name = $field['field_name'];
            if (!isset($unique_fields[$field_name])) {
                $unique_fields[$field_name] = $field;
            } else {
                // Merge locations if field appears in multiple places
                $unique_fields[$field_name]['locations'] = array_unique(
                    array_merge(
                        $unique_fields[$field_name]['locations'],
                        $field['locations']
                    )
                );
            }
        }

        return [
            'success' => true,
            'template_id' => $post_id,
            'page_builder' => $page_builder,
            'post_type' => $post->post_type,
            'title' => $post->post_title,
            'fields' => array_values($unique_fields),
            'total_fields' => count($unique_fields)
        ];
    }

    /**
     * Detect which page builder is being used
     *
     * @param int $post_id Post ID
     * @return string Page builder name
     */
    private function detect_page_builder($post_id) {
        // Check for Elementor
        if (get_post_meta($post_id, '_elementor_edit_mode', true) === 'builder') {
            return 'elementor';
        }

        // Check for Bricks
        if (get_post_meta($post_id, '_bricks_page_content_2', true)) {
            return 'bricks';
        }

        // Check for Oxygen
        if (get_post_meta($post_id, 'ct_builder_shortcodes', true)) {
            return 'oxygen';
        }

        // Check for Kadence blocks in content
        $content = get_post_field('post_content', $post_id);
        if (strpos($content, 'wp:kadence/') !== false) {
            return 'kadence';
        }

        // Default to Gutenberg
        if (strpos($content, '<!-- wp:') !== false) {
            return 'gutenberg';
        }

        return 'unknown';
    }

    /**
     * Detect Kadence block custom fields
     *
     * @param int $post_id Post ID
     * @param string $content Post content
     * @return array Detected fields
     */
    private function detect_kadence_fields($post_id, $content) {
        $fields = [];

        // Pattern to match Kadence block comments with JSON
        // Use brace counting to handle nested JSON properly
        preg_match_all('/<!-- wp:kadence\/([a-z0-9-]+)\s+(.+?)\s*(\/)?-->/is', $content, $blocks, PREG_SET_ORDER);

        foreach ($blocks as $block) {
            $block_type = $block[1];
            $json_candidate = $block[2];

            // Find the opening brace
            $start_pos = strpos($json_candidate, '{');
            if ($start_pos === false) continue;

            // Count braces to find matching closing brace
            $depth = 0;
            $end_pos = $start_pos;
            $length = strlen($json_candidate);

            for ($i = $start_pos; $i < $length; $i++) {
                if ($json_candidate[$i] === '{') $depth++;
                if ($json_candidate[$i] === '}') {
                    $depth--;
                    if ($depth === 0) {
                        $end_pos = $i;
                        break;
                    }
                }
            }

            // Extract the balanced JSON string
            $json_string = substr($json_candidate, $start_pos, $end_pos - $start_pos + 1);

            // Parse JSON
            $attributes = json_decode($json_string, true);

            if ($attributes && isset($attributes['kadenceDynamic'])) {
                $dynamic_fields = $attributes['kadenceDynamic'];

                // Extract custom field references from kadenceDynamic
                foreach ($dynamic_fields as $property => $config) {
                    if (is_array($config) && isset($config['field'])) {
                        // Check if it's a custom field reference
                        if ($config['field'] === 'post|post_custom_field' && isset($config['custom'])) {
                            $field_name = $config['custom'];

                            $fields[] = [
                                'field_name' => $field_name,
                                'field_type' => $this->guess_field_type($field_name, $property),
                                'locations' => ["kadence/{$block_type}/{$property}"],
                                'required' => !isset($config['fallback']) || empty($config['fallback']),
                                'page_builder' => 'kadence',
                                'block_type' => $block_type,
                                'property' => $property
                            ];
                        }
                    }
                }
            }
        }

        // IMPORTANT: Check for data-custom="field_name" attributes in HTML content
        // This catches Kadence dynamic fields used in core blocks (headings, paragraphs, etc.)
        // Pattern: <span ... data-custom="field_name" ... class="kb-inline-dynamic">
        preg_match_all('/data-custom="([^"]+)"/', $content, $data_custom_matches);

        if (!empty($data_custom_matches[1])) {
            foreach ($data_custom_matches[1] as $field_name) {
                // Skip empty or placeholder values
                if (empty($field_name) || $field_name === 'No Content') {
                    continue;
                }

                $fields[] = [
                    'field_name' => $field_name,
                    'field_type' => $this->guess_field_type($field_name, 'content'),
                    'locations' => ['kadence/inline-dynamic/content'],
                    'required' => true,
                    'page_builder' => 'kadence',
                    'block_type' => 'inline_dynamic',
                    'property' => 'content'
                ];
            }
        }

        // Also check for fallback values in kadenceDynamic image blocks
        // Pattern: "fallback":"field_name" or "fallback":"{{field_name}}"
        preg_match_all('/"fallback":"([^"]+)"/', $content, $fallback_matches);

        if (!empty($fallback_matches[1])) {
            foreach ($fallback_matches[1] as $field_name) {
                // Strip {{ }} if present
                $field_name = preg_replace('/^\{\{|\}\}$/', '', $field_name);

                // Skip empty values or URLs
                if (empty($field_name) || strpos($field_name, 'http') === 0) {
                    continue;
                }

                $fields[] = [
                    'field_name' => $field_name,
                    'field_type' => 'image',
                    'locations' => ['kadence/image/fallback'],
                    'required' => false,
                    'page_builder' => 'kadence',
                    'block_type' => 'image_fallback',
                    'property' => 'url'
                ];
            }
        }

        // Also check for {{placeholder}} syntax (legacy/fallback)
        preg_match_all('/\{\{([a-zA-Z0-9_-]+)\}\}/', $content, $placeholder_matches);

        if (!empty($placeholder_matches[1])) {
            foreach ($placeholder_matches[1] as $field_name) {
                $fields[] = [
                    'field_name' => $field_name,
                    'field_type' => 'text',
                    'locations' => ['content'],
                    'required' => true,
                    'page_builder' => 'kadence',
                    'block_type' => 'legacy_placeholder',
                    'property' => 'content'
                ];
            }
        }

        // Check for bare HTML attribute patterns: src="field", alt="field", title="field"
        $attribute_patterns = [
            'src' => 'image',
            'alt' => 'text',
            'title' => 'text',
            'href' => 'url'
        ];

        foreach ($attribute_patterns as $attr => $field_type) {
            preg_match_all('/' . preg_quote($attr, '/') . '="([a-zA-Z][a-zA-Z0-9_-]+)"/', $content, $attr_matches);

            if (!empty($attr_matches[1])) {
                foreach ($attr_matches[1] as $field_name) {
                    // Skip if it looks like a URL or path
                    if (strpos($field_name, '/') !== false ||
                        strpos($field_name, '.') !== false ||
                        strpos($field_name, 'http') === 0) {
                        continue;
                    }

                    $fields[] = [
                        'field_name' => $field_name,
                        'field_type' => $field_type,
                        'locations' => ['content/' . $attr],
                        'required' => false,
                        'page_builder' => 'kadence',
                        'block_type' => 'html_attribute',
                        'property' => $attr
                    ];
                }
            }
        }

        return $fields;
    }

    /**
     * Detect Elementor custom fields
     *
     * @param int $post_id Post ID
     * @return array Detected fields
     */
    private function detect_elementor_fields($post_id) {
        $fields = [];

        // Get Elementor data
        $elementor_data = get_post_meta($post_id, '_elementor_data', true);

        if (empty($elementor_data)) {
            return $fields;
        }

        // Parse JSON
        $data = json_decode($elementor_data, true);

        if (!$data || !is_array($data)) {
            return $fields;
        }

        // Recursively search for dynamic tags
        $this->search_elementor_dynamic_tags($data, $fields);

        return $fields;
    }

    /**
     * Recursively search Elementor data for dynamic tags
     *
     * @param array $data Elementor data structure
     * @param array &$fields Fields array (passed by reference)
     */
    private function search_elementor_dynamic_tags($data, &$fields) {
        if (!is_array($data)) {
            return;
        }

        foreach ($data as $key => $value) {
            // Check for dynamic settings
            if ($key === '__dynamic__' && is_array($value)) {
                foreach ($value as $property => $dynamic_config) {
                    if (is_string($dynamic_config) && strpos($dynamic_config, 'custom-field') !== false) {
                        // Parse dynamic tag: [custom-field key="field_name"]
                        if (preg_match('/key="([^"]+)"/', $dynamic_config, $matches)) {
                            $field_name = $matches[1];

                            $fields[] = [
                                'field_name' => $field_name,
                                'field_type' => $this->guess_field_type($field_name, $property),
                                'locations' => ["elementor/{$property}"],
                                'required' => true,
                                'page_builder' => 'elementor',
                                'block_type' => 'dynamic_tag',
                                'property' => $property
                            ];
                        }
                    }
                }
            }

            // Recurse into nested arrays
            if (is_array($value)) {
                $this->search_elementor_dynamic_tags($value, $fields);
            }
        }
    }

    /**
     * Detect Bricks custom fields
     *
     * @param int $post_id Post ID
     * @return array Detected fields
     */
    private function detect_bricks_fields($post_id) {
        $fields = [];

        // Get Bricks data
        $bricks_data = get_post_meta($post_id, '_bricks_page_content_2', true);

        if (empty($bricks_data)) {
            return $fields;
        }

        // Parse JSON
        $data = json_decode($bricks_data, true);

        if (!$data || !is_array($data)) {
            return $fields;
        }

        // Recursively search for custom field references
        $this->search_bricks_dynamic_data($data, $fields);

        return $fields;
    }

    /**
     * Recursively search Bricks data for dynamic data
     *
     * @param array $data Bricks data structure
     * @param array &$fields Fields array (passed by reference)
     */
    private function search_bricks_dynamic_data($data, &$fields) {
        if (!is_array($data)) {
            return;
        }

        foreach ($data as $key => $value) {
            // Check for dynamic data patterns: {echo:field_name} or {custom_field:field_name}
            if (is_string($value)) {
                if (preg_match('/\{echo:([a-zA-Z0-9_-]+)\}/', $value, $matches)) {
                    $field_name = $matches[1];

                    $fields[] = [
                        'field_name' => $field_name,
                        'field_type' => $this->guess_field_type($field_name, $key),
                        'locations' => ["bricks/{$key}"],
                        'required' => true,
                        'page_builder' => 'bricks',
                        'block_type' => 'echo',
                        'property' => $key
                    ];
                } elseif (preg_match('/\{custom_field:([a-zA-Z0-9_-]+)\}/', $value, $matches)) {
                    $field_name = $matches[1];

                    $fields[] = [
                        'field_name' => $field_name,
                        'field_type' => $this->guess_field_type($field_name, $key),
                        'locations' => ["bricks/{$key}"],
                        'required' => true,
                        'page_builder' => 'bricks',
                        'block_type' => 'custom_field',
                        'property' => $key
                    ];
                }
            }

            // Recurse into nested arrays
            if (is_array($value)) {
                $this->search_bricks_dynamic_data($value, $fields);
            }
        }
    }

    /**
     * Detect Oxygen custom fields
     *
     * @param int $post_id Post ID
     * @param string $content Post content
     * @return array Detected fields
     */
    private function detect_oxygen_fields($post_id, $content) {
        $fields = [];

        // Get Oxygen shortcodes
        $oxygen_data = get_post_meta($post_id, 'ct_builder_shortcodes', true);

        if (!empty($oxygen_data)) {
            $content = $oxygen_data;
        }

        // Search for dynamic data attributes
        if (preg_match_all('/data=[\'"]custom_field:([a-zA-Z0-9_-]+)[\'"]/', $content, $matches)) {
            foreach ($matches[1] as $field_name) {
                $fields[] = [
                    'field_name' => $field_name,
                    'field_type' => 'text',
                    'locations' => ['oxygen/custom_field'],
                    'required' => true,
                    'page_builder' => 'oxygen',
                    'block_type' => 'custom_field',
                    'property' => 'data'
                ];
            }
        }

        return $fields;
    }

    /**
     * Detect Gutenberg custom fields
     *
     * @param string $content Post content
     * @return array Detected fields
     */
    private function detect_gutenberg_fields($content) {
        $fields = [];

        // Check for {{placeholder}} syntax
        preg_match_all('/\{\{([a-zA-Z0-9_-]+)\}\}/', $content, $matches);

        if (!empty($matches[1])) {
            foreach ($matches[1] as $field_name) {
                $fields[] = [
                    'field_name' => $field_name,
                    'field_type' => 'text',
                    'locations' => ['content'],
                    'required' => true,
                    'page_builder' => 'gutenberg',
                    'block_type' => 'placeholder',
                    'property' => 'content'
                ];
            }
        }

        return $fields;
    }

    /**
     * Get standard WordPress page/post fields that can be mapped
     *
     * @param string $post_type Post type (page, post, etc.)
     * @return array Standard fields
     */
    public function get_standard_fields($post_type = 'page') {
        $fields = [];

        // Core WordPress fields
        $core_fields = [
            [
                'field_name' => 'Page_Title',
                'field_type' => 'text',
                'locations' => ['wordpress/post_title'],
                'required' => true,
                'page_builder' => 'wordpress',
                'block_type' => 'core',
                'property' => 'post_title',
                'description' => 'WordPress page/post title'
            ],
            [
                'field_name' => 'Slug',
                'field_type' => 'text',
                'locations' => ['wordpress/post_name'],
                'required' => false,
                'page_builder' => 'wordpress',
                'block_type' => 'core',
                'property' => 'post_name',
                'description' => 'URL slug'
            ],
            [
                'field_name' => 'status',
                'field_type' => 'text',
                'locations' => ['wordpress/post_status'],
                'required' => false,
                'page_builder' => 'wordpress',
                'block_type' => 'core',
                'property' => 'post_status',
                'description' => 'publish, draft, pending, private'
            ],
            [
                'field_name' => 'excerpt',
                'field_type' => 'text',
                'locations' => ['wordpress/post_excerpt'],
                'required' => false,
                'page_builder' => 'wordpress',
                'block_type' => 'core',
                'property' => 'post_excerpt',
                'description' => 'Post excerpt/summary'
            ],
            [
                'field_name' => 'featured_image',
                'field_type' => 'image',
                'locations' => ['wordpress/_thumbnail_id'],
                'required' => false,
                'page_builder' => 'wordpress',
                'block_type' => 'core',
                'property' => '_thumbnail_id',
                'description' => 'Featured image URL'
            ],
            [
                'field_name' => 'publish_date',
                'field_type' => 'date',
                'locations' => ['wordpress/post_date'],
                'required' => false,
                'page_builder' => 'wordpress',
                'block_type' => 'core',
                'property' => 'post_date',
                'description' => 'Publication date (YYYY-MM-DD HH:MM:SS)'
            ],
            [
                'field_name' => 'author',
                'field_type' => 'text',
                'locations' => ['wordpress/post_author'],
                'required' => false,
                'page_builder' => 'wordpress',
                'block_type' => 'core',
                'property' => 'post_author',
                'description' => 'Author username or ID'
            ],
            [
                'field_name' => 'parent',
                'field_type' => 'text',
                'locations' => ['wordpress/post_parent'],
                'required' => false,
                'page_builder' => 'wordpress',
                'block_type' => 'core',
                'property' => 'post_parent',
                'description' => 'Parent page ID or slug'
            ],
            [
                'field_name' => 'menu_order',
                'field_type' => 'number',
                'locations' => ['wordpress/menu_order'],
                'required' => false,
                'page_builder' => 'wordpress',
                'block_type' => 'core',
                'property' => 'menu_order',
                'description' => 'Menu order (for sorting)'
            ],
            [
                'field_name' => 'comment_status',
                'field_type' => 'text',
                'locations' => ['wordpress/comment_status'],
                'required' => false,
                'page_builder' => 'wordpress',
                'block_type' => 'core',
                'property' => 'comment_status',
                'description' => 'open or closed'
            ],
            [
                'field_name' => 'ping_status',
                'field_type' => 'text',
                'locations' => ['wordpress/ping_status'],
                'required' => false,
                'page_builder' => 'wordpress',
                'block_type' => 'core',
                'property' => 'ping_status',
                'description' => 'open or closed'
            ],
        ];

        // Add category/tag fields for posts
        if ($post_type === 'post') {
            $core_fields[] = [
                'field_name' => 'category',
                'field_type' => 'text',
                'locations' => ['wordpress/category'],
                'required' => false,
                'page_builder' => 'wordpress',
                'block_type' => 'taxonomy',
                'property' => 'category',
                'description' => 'Category name(s), comma-separated'
            ];
            $core_fields[] = [
                'field_name' => 'tags',
                'field_type' => 'text',
                'locations' => ['wordpress/post_tag'],
                'required' => false,
                'page_builder' => 'wordpress',
                'block_type' => 'taxonomy',
                'property' => 'post_tag',
                'description' => 'Tag name(s), comma-separated'
            ];
        }

        // Page template field for pages
        if ($post_type === 'page') {
            $core_fields[] = [
                'field_name' => 'page_template',
                'field_type' => 'text',
                'locations' => ['wordpress/_wp_page_template'],
                'required' => false,
                'page_builder' => 'wordpress',
                'block_type' => 'core',
                'property' => '_wp_page_template',
                'description' => 'Page template file name'
            ];
        }

        return $core_fields;
    }

    /**
     * Get SEO plugin fields based on active SEO plugin
     *
     * @return array SEO fields
     */
    public function get_seo_fields() {
        $fields = [];

        // Detect active SEO plugin
        $seo_plugin = 'none';
        if (defined('WPSEO_VERSION')) {
            $seo_plugin = 'yoast';
        } elseif (class_exists('RankMath')) {
            $seo_plugin = 'rankmath';
        } elseif (defined('AIOSEO_VERSION') || class_exists('AIOSEO')) {
            $seo_plugin = 'aioseo';
        }

        // Common SEO fields (available for all SEO plugins)
        $seo_fields = [
            [
                'field_name' => 'SEO_Title',
                'field_type' => 'text',
                'locations' => ["seo/{$seo_plugin}/title"],
                'required' => false,
                'page_builder' => 'seo',
                'block_type' => $seo_plugin,
                'property' => 'title',
                'description' => 'SEO meta title'
            ],
            [
                'field_name' => 'SEO_Meta_Description',
                'field_type' => 'text',
                'locations' => ["seo/{$seo_plugin}/description"],
                'required' => false,
                'page_builder' => 'seo',
                'block_type' => $seo_plugin,
                'property' => 'description',
                'description' => 'SEO meta description'
            ],
            [
                'field_name' => 'SEO_Focus_Keyword',
                'field_type' => 'text',
                'locations' => ["seo/{$seo_plugin}/focus_keyword"],
                'required' => false,
                'page_builder' => 'seo',
                'block_type' => $seo_plugin,
                'property' => 'focus_keyword',
                'description' => 'Focus keyphrase for SEO analysis'
            ],
            [
                'field_name' => 'canonical_url',
                'field_type' => 'url',
                'locations' => ["seo/{$seo_plugin}/canonical"],
                'required' => false,
                'page_builder' => 'seo',
                'block_type' => $seo_plugin,
                'property' => 'canonical',
                'description' => 'Canonical URL'
            ],
            [
                'field_name' => 'robots_index',
                'field_type' => 'text',
                'locations' => ["seo/{$seo_plugin}/robots_index"],
                'required' => false,
                'page_builder' => 'seo',
                'block_type' => $seo_plugin,
                'property' => 'robots_index',
                'description' => 'index or noindex'
            ],
            [
                'field_name' => 'robots_follow',
                'field_type' => 'text',
                'locations' => ["seo/{$seo_plugin}/robots_follow"],
                'required' => false,
                'page_builder' => 'seo',
                'block_type' => $seo_plugin,
                'property' => 'robots_follow',
                'description' => 'follow or nofollow'
            ],
            [
                'field_name' => 'og_title',
                'field_type' => 'text',
                'locations' => ["seo/{$seo_plugin}/og_title"],
                'required' => false,
                'page_builder' => 'seo',
                'block_type' => $seo_plugin,
                'property' => 'og_title',
                'description' => 'Open Graph title for social sharing'
            ],
            [
                'field_name' => 'og_description',
                'field_type' => 'text',
                'locations' => ["seo/{$seo_plugin}/og_description"],
                'required' => false,
                'page_builder' => 'seo',
                'block_type' => $seo_plugin,
                'property' => 'og_description',
                'description' => 'Open Graph description for social sharing'
            ],
            [
                'field_name' => 'og_image',
                'field_type' => 'image',
                'locations' => ["seo/{$seo_plugin}/og_image"],
                'required' => false,
                'page_builder' => 'seo',
                'block_type' => $seo_plugin,
                'property' => 'og_image',
                'description' => 'Open Graph image URL for social sharing'
            ],
        ];

        return $seo_fields;
    }

    /**
     * Detect all fields including standard and SEO fields
     *
     * @param int $post_id Post ID
     * @param bool $include_standard Include standard WordPress fields
     * @param bool $include_seo Include SEO plugin fields
     * @return array All detected fields
     */
    public function detect_all_fields($post_id, $include_standard = true, $include_seo = true) {
        // Get custom fields from template
        $result = $this->detect_fields($post_id);

        if (!$result['success']) {
            return $result;
        }

        $post = get_post($post_id);
        $all_fields = $result['fields'];

        // Add standard WordPress fields
        if ($include_standard) {
            $standard_fields = $this->get_standard_fields($post->post_type);
            $result['standard_fields'] = $standard_fields;
        }

        // Add SEO fields
        if ($include_seo) {
            $seo_fields = $this->get_seo_fields();
            $result['seo_fields'] = $seo_fields;
        }

        return $result;
    }

    /**
     * Guess field type based on field name and context
     *
     * @param string $field_name Field name
     * @param string $context Context (property name or location)
     * @return string Field type
     */
    private function guess_field_type($field_name, $context = '') {
        $field_lower = strtolower($field_name);
        $context_lower = strtolower($context);

        // Image fields
        if (preg_match('/(image|img|photo|picture|thumbnail|banner|hero|logo|icon)/', $field_lower) ||
            preg_match('/(src|url|background)/', $context_lower)) {
            return 'image';
        }

        // URL fields
        if (preg_match('/(url|link|href)/', $field_lower)) {
            return 'url';
        }

        // Number fields
        if (preg_match('/(price|cost|amount|number|count|quantity|rating|score)/', $field_lower)) {
            return 'number';
        }

        // Date fields
        if (preg_match('/(date|time|publish|created|updated)/', $field_lower)) {
            return 'date';
        }

        // Email fields
        if (preg_match('/(email|mail)/', $field_lower)) {
            return 'email';
        }

        // Default to text
        return 'text';
    }
}
