<?php
/**
 * Classic Editor Integration
 *
 * Adds SEO metabox to Classic Editor with SERP preview and social cards
 *
 * @package InstaRank
 * @since 1.4.0
 */

defined('ABSPATH') || exit;

class InstaRank_Classic_Editor {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('add_meta_boxes', [$this, 'add_meta_box']);
        add_action('save_post', [$this, 'save_meta_box']);
        add_action('save_post', [$this, 'save_pseo_fields'], 20); // Higher priority to run after save_meta_box
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);

        // AJAX handler for Gutenberg sidebar pSEO field saving
        add_action('wp_ajax_instarank_save_pseo_fields', [$this, 'ajax_save_pseo_fields']);

        // AJAX handler for importing external images to Media Library
        add_action('wp_ajax_instarank_import_external_image', [$this, 'ajax_import_external_image']);
    }

    /**
     * Add meta box
     */
    public function add_meta_box() {
        $post_types = get_post_types(['public' => true], 'names');

        foreach ($post_types as $post_type) {
            add_meta_box(
                'instarank_seo_metabox',
                'InstaRank SEO',
                [$this, 'render_meta_box'],
                $post_type,
                'normal',
                'high'
            );
        }
    }

    /**
     * Enqueue assets
     */
    public function enqueue_assets($hook) {
        // Only load on post edit screens
        if (!in_array($hook, ['post.php', 'post-new.php'])) {
            return;
        }

        // Check if Gutenberg is active for this post type
        if (function_exists('use_block_editor_for_post')) {
            global $post;
            if (use_block_editor_for_post($post)) {
                return; // Gutenberg is active, don't load Classic Editor assets
            }
        }

        wp_enqueue_style(
            'instarank-classic-editor',
            INSTARANK_PLUGIN_URL . 'assets/css/classic-editor.css',
            [],
            INSTARANK_VERSION
        );

        wp_enqueue_script(
            'instarank-classic-editor',
            INSTARANK_PLUGIN_URL . 'assets/js/classic-editor.js',
            ['jquery'],
            INSTARANK_VERSION,
            true
        );

        // Get post meta
        global $post;
        $meta_data = [
            'title' => get_post_meta($post->ID, '_instarank_meta_title', true),
            'description' => get_post_meta($post->ID, '_instarank_meta_description', true),
            'focus_keyword' => get_post_meta($post->ID, '_instarank_focus_keyword', true),
            'canonical_url' => get_post_meta($post->ID, '_instarank_canonical_url', true),
            'og_title' => get_post_meta($post->ID, '_instarank_og_title', true),
            'og_description' => get_post_meta($post->ID, '_instarank_og_description', true),
            'og_image' => get_post_meta($post->ID, '_instarank_og_image', true),
            'twitter_title' => get_post_meta($post->ID, '_instarank_twitter_title', true),
            'twitter_description' => get_post_meta($post->ID, '_instarank_twitter_description', true),
            'twitter_image' => get_post_meta($post->ID, '_instarank_twitter_image', true),
            'noindex' => get_post_meta($post->ID, '_instarank_noindex', true),
            'nofollow' => get_post_meta($post->ID, '_instarank_nofollow', true),
        ];

        // Get pending changes count
        global $wpdb;
        $changes_table = $wpdb->prefix . 'instarank_changes';
        $pending_count = 0;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $changes_table)) === $changes_table) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $pending_count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}instarank_changes WHERE post_id = %d AND status = 'pending'",
                $post->ID
            ));
        }

        wp_localize_script('instarank-classic-editor', 'instarankClassicEditor', [
            'postId' => $post->ID,
            'postTitle' => $post->post_title,
            'postExcerpt' => $post->post_excerpt,
            'postUrl' => get_permalink($post->ID),
            'siteName' => get_bloginfo('name'),
            'metaData' => $meta_data,
            'pendingChanges' => (int) $pending_count,
            'nonce' => wp_create_nonce('instarank_classic_editor')
        ]);
    }

    /**
     * Render meta box
     */
    public function render_meta_box($post) {
        // Check if this is a pSEO generated page and display field values
        $pseo_fields = get_post_meta($post->ID, '_instarank_pseo_fields', true);
        $is_pseo_page = get_post_meta($post->ID, '_instarank_pseo_generated', true);
        $pseo_generated_at = get_post_meta($post->ID, '_instarank_pseo_generated_at', true);

        if ($is_pseo_page && !empty($pseo_fields) && is_array($pseo_fields)) {
            $this->render_pseo_fields_section($pseo_fields, $pseo_generated_at);
        }

        // Check if Gutenberg is active for this post type
        if (function_exists('use_block_editor_for_post')) {
            if (use_block_editor_for_post($post)) {
                if (!$is_pseo_page) {
                    echo '<p>' . esc_html__('This post uses the Block Editor. SEO settings are available in the InstaRank sidebar panel.', 'instarank') . '</p>';
                }
                return;
            }
        }

        wp_nonce_field('instarank_classic_editor_save', 'instarank_classic_editor_nonce');

        // Get current values
        $meta_title = get_post_meta($post->ID, '_instarank_meta_title', true);
        $meta_description = get_post_meta($post->ID, '_instarank_meta_description', true);
        $focus_keyword = get_post_meta($post->ID, '_instarank_focus_keyword', true);
        $canonical_url = get_post_meta($post->ID, '_instarank_canonical_url', true);
        $og_title = get_post_meta($post->ID, '_instarank_og_title', true);
        $og_description = get_post_meta($post->ID, '_instarank_og_description', true);
        $og_image = get_post_meta($post->ID, '_instarank_og_image', true);
        $twitter_title = get_post_meta($post->ID, '_instarank_twitter_title', true);
        $twitter_description = get_post_meta($post->ID, '_instarank_twitter_description', true);
        $twitter_image = get_post_meta($post->ID, '_instarank_twitter_image', true);
        $noindex = get_post_meta($post->ID, '_instarank_noindex', true);
        $nofollow = get_post_meta($post->ID, '_instarank_nofollow', true);

        // Get pending changes
        global $wpdb;
        $changes_table = $wpdb->prefix . 'instarank_changes';
        $pending_count = 0;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $changes_table)) === $changes_table) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $pending_count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}instarank_changes WHERE post_id = %d AND status = 'pending'",
                $post->ID
            ));
        }

        ?>
        <div class="instarank-classic-metabox">
            <?php if ($pending_count > 0): ?>
            <div class="instarank-pending-notice">
                <span class="dashicons dashicons-info"></span>
                <span>
                    <?php
                    // translators: %d is the number of pending changes
                    echo esc_html(sprintf(_n('%d pending change for this post.', '%d pending changes for this post.', $pending_count, 'instarank'), absint($pending_count)));
                    ?>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=instarank-changes')); ?>">
                        <?php esc_html_e('Review changes', 'instarank'); ?>
                    </a>
                </span>
            </div>
            <?php endif; ?>

            <!-- Tab Navigation -->
            <div class="instarank-tabs">
                <button type="button" class="instarank-tab active" data-tab="general">
                    <span class="dashicons dashicons-search"></span>
                    <?php esc_html_e('General SEO', 'instarank'); ?>
                </button>
                <button type="button" class="instarank-tab" data-tab="social">
                    <span class="dashicons dashicons-share"></span>
                    <?php esc_html_e('Social', 'instarank'); ?>
                </button>
                <button type="button" class="instarank-tab" data-tab="advanced">
                    <span class="dashicons dashicons-admin-settings"></span>
                    <?php esc_html_e('Advanced', 'instarank'); ?>
                </button>
            </div>

            <!-- Tab Content -->
            <div class="instarank-tab-content active" data-content="general">
                <!-- SERP Preview -->
                <div class="instarank-serp-preview">
                    <h4><?php esc_html_e('Google Search Preview', 'instarank'); ?></h4>
                    <div class="serp-preview-container">
                        <div class="serp-preview-url"></div>
                        <div class="serp-preview-title"></div>
                        <div class="serp-preview-description"></div>
                    </div>
                </div>

                <!-- Meta Title -->
                <div class="instarank-field">
                    <label for="instarank_meta_title">
                        <?php esc_html_e('SEO Title', 'instarank'); ?>
                    </label>
                    <input type="text"
                           id="instarank_meta_title"
                           name="instarank_meta_title"
                           class="widefat"
                           value="<?php echo esc_attr($meta_title); ?>"
                           maxlength="70"
                           placeholder="<?php echo esc_attr($post->post_title); ?>">
                    <div class="instarank-char-counter" data-max="60" data-optimal="50">
                        <span class="count">0 / 60</span>
                        <div class="bar"><div class="fill"></div></div>
                    </div>
                    <p class="description">
                        <?php esc_html_e('Optimal length: 50-60 characters', 'instarank'); ?>
                    </p>
                </div>

                <!-- Meta Description -->
                <div class="instarank-field">
                    <label for="instarank_meta_description">
                        <?php esc_html_e('Meta Description', 'instarank'); ?>
                    </label>
                    <textarea id="instarank_meta_description"
                              name="instarank_meta_description"
                              class="widefat"
                              rows="3"
                              maxlength="160"
                              placeholder="<?php echo esc_attr(wp_trim_words($post->post_excerpt ?: $post->post_content, 20)); ?>"><?php echo esc_textarea($meta_description); ?></textarea>
                    <div class="instarank-char-counter" data-max="160" data-optimal="120">
                        <span class="count">0 / 160</span>
                        <div class="bar"><div class="fill"></div></div>
                    </div>
                    <p class="description">
                        <?php esc_html_e('Optimal length: 120-160 characters', 'instarank'); ?>
                    </p>
                </div>

                <!-- Focus Keyword -->
                <div class="instarank-field">
                    <label for="instarank_focus_keyword">
                        <?php esc_html_e('Focus Keyword', 'instarank'); ?>
                    </label>
                    <input type="text"
                           id="instarank_focus_keyword"
                           name="instarank_focus_keyword"
                           class="widefat"
                           value="<?php echo esc_attr($focus_keyword); ?>"
                           placeholder="<?php esc_attr_e('e.g., best SEO practices', 'instarank'); ?>">
                    <div class="instarank-keyword-analysis">
                        <div class="keyword-check" data-check="title">
                            <span class="dashicons dashicons-minus"></span>
                            <?php esc_html_e('Keyword in title', 'instarank'); ?>
                        </div>
                        <div class="keyword-check" data-check="description">
                            <span class="dashicons dashicons-minus"></span>
                            <?php esc_html_e('Keyword in description', 'instarank'); ?>
                        </div>
                        <div class="keyword-check" data-check="content">
                            <span class="dashicons dashicons-minus"></span>
                            <?php esc_html_e('Keyword in content', 'instarank'); ?>
                        </div>
                    </div>
                </div>

                <!-- Canonical URL -->
                <div class="instarank-field">
                    <label for="instarank_canonical_url">
                        <?php esc_html_e('Canonical URL', 'instarank'); ?>
                    </label>
                    <input type="url"
                           id="instarank_canonical_url"
                           name="instarank_canonical_url"
                           class="widefat"
                           value="<?php echo esc_attr($canonical_url); ?>"
                           placeholder="<?php echo esc_url(get_permalink($post->ID)); ?>">
                    <p class="description">
                        <?php esc_html_e('Leave empty to use the default permalink', 'instarank'); ?>
                    </p>
                </div>
            </div>

            <!-- Social Tab -->
            <div class="instarank-tab-content" data-content="social">
                <h4><?php esc_html_e('Facebook / Open Graph', 'instarank'); ?></h4>

                <!-- Facebook Preview -->
                <div class="instarank-social-preview facebook-preview">
                    <div class="social-preview-image"></div>
                    <div class="social-preview-content">
                        <div class="social-preview-title"></div>
                        <div class="social-preview-description"></div>
                        <div class="social-preview-url"></div>
                    </div>
                </div>

                <div class="instarank-field">
                    <label for="instarank_og_title">
                        <?php esc_html_e('Facebook Title', 'instarank'); ?>
                    </label>
                    <input type="text"
                           id="instarank_og_title"
                           name="instarank_og_title"
                           class="widefat"
                           value="<?php echo esc_attr($og_title); ?>"
                           placeholder="<?php echo esc_attr($meta_title ?: $post->post_title); ?>">
                </div>

                <div class="instarank-field">
                    <label for="instarank_og_description">
                        <?php esc_html_e('Facebook Description', 'instarank'); ?>
                    </label>
                    <textarea id="instarank_og_description"
                              name="instarank_og_description"
                              class="widefat"
                              rows="3"
                              placeholder="<?php echo esc_attr($meta_description ?: wp_trim_words($post->post_content, 20)); ?>"><?php echo esc_textarea($og_description); ?></textarea>
                </div>

                <div class="instarank-field">
                    <label for="instarank_og_image">
                        <?php esc_html_e('Facebook Image', 'instarank'); ?>
                    </label>
                    <div class="instarank-image-selector">
                        <input type="hidden"
                               id="instarank_og_image"
                               name="instarank_og_image"
                               value="<?php echo esc_attr($og_image); ?>">
                        <button type="button" class="button instarank-select-image" data-target="og_image">
                            <?php esc_html_e('Select Image', 'instarank'); ?>
                        </button>
                        <button type="button" class="button instarank-remove-image" data-target="og_image" <?php echo empty($og_image) ? 'style="display:none;"' : ''; ?>>
                            <?php esc_html_e('Remove', 'instarank'); ?>
                        </button>
                        <div class="image-preview" <?php echo empty($og_image) ? 'style="display:none;"' : ''; ?>>
                            <?php if ($og_image): ?>
                                <img src="<?php echo esc_url($og_image); ?>" alt="">
                            <?php endif; ?>
                        </div>
                    </div>
                    <p class="description">
                        <?php esc_html_e('Recommended: 1200x630px', 'instarank'); ?>
                    </p>
                </div>

                <hr>

                <h4><?php esc_html_e('Twitter Card', 'instarank'); ?></h4>

                <!-- Twitter Preview -->
                <div class="instarank-social-preview twitter-preview">
                    <div class="social-preview-image"></div>
                    <div class="social-preview-content">
                        <div class="social-preview-title"></div>
                        <div class="social-preview-description"></div>
                        <div class="social-preview-url"></div>
                    </div>
                </div>

                <div class="instarank-field">
                    <label for="instarank_twitter_title">
                        <?php esc_html_e('Twitter Title', 'instarank'); ?>
                    </label>
                    <input type="text"
                           id="instarank_twitter_title"
                           name="instarank_twitter_title"
                           class="widefat"
                           value="<?php echo esc_attr($twitter_title); ?>"
                           placeholder="<?php echo esc_attr($og_title ?: $meta_title ?: $post->post_title); ?>">
                </div>

                <div class="instarank-field">
                    <label for="instarank_twitter_description">
                        <?php esc_html_e('Twitter Description', 'instarank'); ?>
                    </label>
                    <textarea id="instarank_twitter_description"
                              name="instarank_twitter_description"
                              class="widefat"
                              rows="3"
                              placeholder="<?php echo esc_attr($og_description ?: $meta_description ?: wp_trim_words($post->post_content, 20)); ?>"><?php echo esc_textarea($twitter_description); ?></textarea>
                </div>

                <div class="instarank-field">
                    <label for="instarank_twitter_image">
                        <?php esc_html_e('Twitter Image', 'instarank'); ?>
                    </label>
                    <div class="instarank-image-selector">
                        <input type="hidden"
                               id="instarank_twitter_image"
                               name="instarank_twitter_image"
                               value="<?php echo esc_attr($twitter_image); ?>">
                        <button type="button" class="button instarank-select-image" data-target="twitter_image">
                            <?php esc_html_e('Select Image', 'instarank'); ?>
                        </button>
                        <button type="button" class="button instarank-remove-image" data-target="twitter_image" <?php echo empty($twitter_image) ? 'style="display:none;"' : ''; ?>>
                            <?php esc_html_e('Remove', 'instarank'); ?>
                        </button>
                        <div class="image-preview" <?php echo empty($twitter_image) ? 'style="display:none;"' : ''; ?>>
                            <?php if ($twitter_image): ?>
                                <img src="<?php echo esc_url($twitter_image); ?>" alt="">
                            <?php endif; ?>
                        </div>
                    </div>
                    <p class="description">
                        <?php esc_html_e('Recommended: 1200x600px', 'instarank'); ?>
                    </p>
                </div>
            </div>

            <!-- Advanced Tab -->
            <div class="instarank-tab-content" data-content="advanced">
                <h4><?php esc_html_e('Robots Meta', 'instarank'); ?></h4>

                <div class="instarank-field">
                    <label>
                        <input type="checkbox"
                               id="instarank_noindex"
                               name="instarank_noindex"
                               value="1"
                               <?php checked($noindex, '1'); ?>>
                        <?php esc_html_e('No Index', 'instarank'); ?>
                    </label>
                    <p class="description">
                        <?php esc_html_e('Prevent search engines from indexing this page', 'instarank'); ?>
                    </p>
                </div>

                <div class="instarank-field">
                    <label>
                        <input type="checkbox"
                               id="instarank_nofollow"
                               name="instarank_nofollow"
                               value="1"
                               <?php checked($nofollow, '1'); ?>>
                        <?php esc_html_e('No Follow', 'instarank'); ?>
                    </label>
                    <p class="description">
                        <?php esc_html_e('Prevent search engines from following links on this page', 'instarank'); ?>
                    </p>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Save meta box
     */
    public function save_meta_box($post_id) {
        // Check nonce
        if (!isset($_POST['instarank_classic_editor_nonce']) ||
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['instarank_classic_editor_nonce'])), 'instarank_classic_editor_save')) {
            return;
        }

        // Check autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        // Check permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Save meta fields
        $fields = [
            '_instarank_meta_title' => 'sanitize_text_field',
            '_instarank_meta_description' => 'sanitize_textarea_field',
            '_instarank_focus_keyword' => 'sanitize_text_field',
            '_instarank_canonical_url' => 'esc_url_raw',
            '_instarank_og_title' => 'sanitize_text_field',
            '_instarank_og_description' => 'sanitize_textarea_field',
            '_instarank_og_image' => 'esc_url_raw',
            '_instarank_twitter_title' => 'sanitize_text_field',
            '_instarank_twitter_description' => 'sanitize_textarea_field',
            '_instarank_twitter_image' => 'esc_url_raw',
        ];

        foreach ($fields as $field => $sanitize_callback) {
            $post_field = str_replace('_instarank_', 'instarank_', $field);
            if (isset($_POST[$post_field])) {
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Input is sanitized by the callback function
                $raw_value = wp_unslash($_POST[$post_field]);
                $sanitized_value = $sanitize_callback($raw_value);
                update_post_meta($post_id, $field, $sanitized_value);
            } else {
                delete_post_meta($post_id, $field);
            }
        }

        // Save checkboxes
        update_post_meta($post_id, '_instarank_noindex', isset($_POST['instarank_noindex']) ? '1' : '0');
        update_post_meta($post_id, '_instarank_nofollow', isset($_POST['instarank_nofollow']) ? '1' : '0');

        // Send webhook notification
        $webhook = new InstaRank_Webhook_Sender();
        $webhook->send_event('seo_meta_updated', [
            'post_id' => $post_id,
            'post_title' => get_the_title($post_id),
            'post_url' => get_permalink($post_id)
        ]);
    }

    /**
     * Check if a field name suggests it's an image field
     *
     * @param string $field_key The field key/name.
     * @return bool True if this appears to be an image field.
     */
    private function is_image_field_name($field_key) {
        $image_patterns = ['image', 'img', 'photo', 'picture', 'thumbnail', 'avatar', 'logo', 'icon', 'banner', 'hero'];
        $field_lower = strtolower($field_key);

        foreach ($image_patterns as $pattern) {
            if (strpos($field_lower, $pattern) !== false) {
                return true;
            }
        }
        return false;
    }

    /**
     * Check if a URL points to an image
     *
     * @param string $url The URL to check.
     * @return bool True if this appears to be an image URL.
     */
    private function is_image_url($url) {
        if (empty($url) || !filter_var($url, FILTER_VALIDATE_URL)) {
            return false;
        }

        // Check file extension
        $image_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg', 'bmp', 'ico'];
        $path = wp_parse_url($url, PHP_URL_PATH);
        if ($path) {
            $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION));
            if (in_array($ext, $image_extensions, true)) {
                return true;
            }
        }

        // Check for common image CDN patterns
        $image_cdn_patterns = ['unsplash.com', 'pexels.com', 'pixabay.com', 'cloudinary.com', 'imgix.net', 'wp-content/uploads'];
        foreach ($image_cdn_patterns as $pattern) {
            if (strpos($url, $pattern) !== false) {
                return true;
            }
        }

        return false;
    }

    /**
     * Render pSEO fields section for generated pages - EDITABLE VERSION
     *
     * @param array  $pseo_fields     The pSEO field values from the dataset.
     * @param string $generated_at    When the page was generated.
     */
    private function render_pseo_fields_section($pseo_fields, $generated_at) {
        global $post;
        $post_id = $post->ID;

        // Enqueue media library scripts for image uploads
        wp_enqueue_media();

        // Filter out internal fields and categorize fields
        $editable_fields = [];
        $url_fields = [];
        $html_fields = [];
        $image_fields = [];

        foreach ($pseo_fields as $field_key => $field_value) {
            // Skip internal/system fields
            if (strpos($field_key, '_') === 0) {
                continue;
            }

            $display_value = is_array($field_value) ? wp_json_encode($field_value) : (string) $field_value;

            // Categorize fields - check for images first
            if ($this->is_image_field_name($field_key) || $this->is_image_url($display_value)) {
                $image_fields[$field_key] = $display_value;
            } elseif (filter_var($display_value, FILTER_VALIDATE_URL)) {
                $url_fields[$field_key] = $display_value;
            } elseif (preg_match('/<[^>]+>/', $display_value)) {
                $html_fields[$field_key] = $display_value;
            } else {
                $editable_fields[$field_key] = $display_value;
            }
        }
        ?>
        <div class="instarank-pseo-fields" style="margin-bottom: 20px; padding: 15px; background: linear-gradient(135deg, #fef7ed 0%, #fff7ed 100%); border: 1px solid #fed7aa; border-radius: 8px;">
            <div style="display: flex; align-items: center; justify-content: space-between; margin-bottom: 15px;">
                <h3 style="margin: 0; color: #1d2327; font-size: 14px; font-weight: 600; display: flex; align-items: center; gap: 8px;">
                    <span class="dashicons dashicons-database" style="color: #f97316;"></span>
                    <?php esc_html_e('Programmatic SEO Content', 'instarank'); ?>
                    <span style="background: #f97316; color: white; padding: 2px 8px; border-radius: 10px; font-size: 11px; font-weight: 600;">
                        <?php esc_html_e('Editable', 'instarank'); ?>
                    </span>
                </h3>
                <?php if ($generated_at) : ?>
                    <span style="font-size: 11px; color: #757575;">
                        <?php
                        /* translators: %s: date/time when the page was generated */
                        echo esc_html(sprintf(__('Generated: %s', 'instarank'), $generated_at));
                        ?>
                    </span>
                <?php endif; ?>
            </div>

            <p style="margin: 0 0 12px 0; font-size: 12px; color: #50575e; background: #fef3c7; padding: 8px 12px; border-radius: 4px; border-left: 3px solid #f59e0b;">
                <span class="dashicons dashicons-edit" style="font-size: 14px; vertical-align: middle; margin-right: 4px;"></span>
                <?php esc_html_e('Edit field values below. Changes will update the page content when you save/update the post.', 'instarank'); ?>
            </p>

            <div class="pseo-fields-list" style="max-height: 500px; overflow-y: auto;">
                <?php if (!empty($image_fields)) : ?>
                    <!-- Image Fields with Preview and Upload -->
                    <div style="margin-bottom: 16px;">
                        <h4 style="margin: 0 0 10px 0; font-size: 12px; text-transform: uppercase; color: #6b7280; border-bottom: 1px solid #e5e7eb; padding-bottom: 6px;">
                            <span class="dashicons dashicons-format-image" style="font-size: 14px; width: 14px; height: 14px; vertical-align: middle; margin-right: 4px; color: #f97316;"></span>
                            <?php esc_html_e('Image Fields', 'instarank'); ?>
                        </h4>
                        <?php
                        $site_url = get_site_url();
                        foreach ($image_fields as $field_key => $field_value) :
                            $display_key = ucfirst(str_replace(['_', '-'], ' ', $field_key));
                            $has_image = !empty($field_value) && filter_var($field_value, FILTER_VALIDATE_URL);
                            $is_external = $has_image && strpos($field_value, $site_url) !== 0;
                            ?>
                            <div class="pseo-field-row pseo-image-field" style="margin-bottom: 12px; background: white; padding: 12px; border-radius: 6px; border: 1px solid #e2e4e7;">
                                <label for="pseo_field_<?php echo esc_attr($field_key); ?>" style="display: block; font-weight: 600; color: #2271b1; font-size: 12px; margin-bottom: 8px;">
                                    <?php echo esc_html($display_key); ?>
                                    <?php if ($is_external) : ?>
                                        <span class="pseo-external-badge" style="background: #fef3c7; color: #92400e; font-size: 10px; padding: 2px 6px; border-radius: 10px; font-weight: 500; margin-left: 8px;">
                                            <span class="dashicons dashicons-cloud" style="font-size: 12px; width: 12px; height: 12px; vertical-align: middle;"></span>
                                            <?php esc_html_e('External', 'instarank'); ?>
                                        </span>
                                    <?php endif; ?>
                                </label>

                                <!-- Image URL Display -->
                                <?php if ($has_image) : ?>
                                    <div style="margin-bottom: 8px; padding: 8px 10px; background: #f3f4f6; border-radius: 4px; border: 1px solid #e5e7eb;">
                                        <div style="display: flex; align-items: center; justify-content: space-between; gap: 8px;">
                                            <span style="font-size: 11px; color: #6b7280; font-weight: 500;">
                                                <?php esc_html_e('Current URL:', 'instarank'); ?>
                                            </span>
                                            <button
                                                type="button"
                                                class="button button-small"
                                                onclick="navigator.clipboard.writeText('<?php echo esc_js($field_value); ?>'); this.innerHTML='<span class=\'dashicons dashicons-yes\' style=\'font-size: 14px; vertical-align: middle;\'></span> Copied!';"
                                                style="height: 22px; padding: 0 8px; font-size: 11px; line-height: 22px;"
                                                title="<?php esc_attr_e('Copy URL to clipboard', 'instarank'); ?>"
                                            >
                                                <span class="dashicons dashicons-admin-page" style="font-size: 14px; width: 14px; height: 14px; vertical-align: middle;"></span>
                                                <?php esc_html_e('Copy', 'instarank'); ?>
                                            </button>
                                        </div>
                                        <div style="margin-top: 6px; font-size: 11px; color: #3b82f6; word-break: break-all; font-family: monospace;">
                                            <?php echo esc_html($field_value); ?>
                                        </div>
                                    </div>
                                <?php endif; ?>

                                <!-- Image Preview -->
                                <div class="pseo-image-preview-container" style="margin-bottom: 10px;">
                                    <?php if ($has_image) : ?>
                                        <div class="pseo-image-preview" style="position: relative; display: inline-block; max-width: 100%;">
                                            <img
                                                src="<?php echo esc_url($field_value); ?>"
                                                alt="<?php echo esc_attr($display_key); ?>"
                                                style="max-width: 200px; max-height: 150px; border-radius: 4px; border: 1px solid #e2e4e7; display: block;"
                                                onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';"
                                            />
                                            <div style="display: none; width: 200px; height: 100px; background: #f3f4f6; border-radius: 4px; align-items: center; justify-content: center; color: #9ca3af; font-size: 12px;">
                                                <span class="dashicons dashicons-warning" style="margin-right: 4px;"></span>
                                                <?php esc_html_e('Image failed to load', 'instarank'); ?>
                                            </div>
                                        </div>
                                    <?php else : ?>
                                        <div style="width: 200px; height: 100px; background: #f9fafb; border: 2px dashed #d1d5db; border-radius: 4px; display: flex; align-items: center; justify-content: center; color: #9ca3af; font-size: 12px;">
                                            <span class="dashicons dashicons-format-image" style="font-size: 24px; margin-right: 8px;"></span>
                                            <?php esc_html_e('No image set', 'instarank'); ?>
                                        </div>
                                    <?php endif; ?>
                                </div>

                                <!-- External image warning and import option -->
                                <?php if ($is_external) : ?>
                                    <div class="pseo-external-warning" style="margin-bottom: 10px; padding: 8px 10px; background: #fef3c7; border: 1px solid #fcd34d; border-radius: 4px; font-size: 11px;">
                                        <span class="dashicons dashicons-warning" style="color: #d97706; font-size: 14px; vertical-align: middle; margin-right: 4px;"></span>
                                        <span style="color: #92400e;"><?php esc_html_e('This image is hosted externally. Import it to your Media Library for better reliability.', 'instarank'); ?></span>
                                        <button
                                            type="button"
                                            class="button pseo-import-image-btn"
                                            data-field-key="<?php echo esc_attr($field_key); ?>"
                                            data-image-url="<?php echo esc_attr($field_value); ?>"
                                            style="margin-left: 8px; height: 24px; padding: 0 10px; font-size: 11px; background: #f97316; border-color: #ea580c; color: white;"
                                        >
                                            <span class="dashicons dashicons-download" style="font-size: 14px; width: 14px; height: 14px; vertical-align: middle; margin-right: 2px;"></span>
                                            <?php esc_html_e('Import to Library', 'instarank'); ?>
                                        </button>
                                    </div>
                                <?php endif; ?>

                                <!-- URL Input with buttons -->
                                <div style="display: flex; gap: 8px; align-items: flex-start;">
                                    <div style="flex: 1;">
                                        <input
                                            type="url"
                                            id="pseo_field_<?php echo esc_attr($field_key); ?>"
                                            name="instarank_pseo_fields[<?php echo esc_attr($field_key); ?>]"
                                            class="widefat pseo-field-input pseo-image-url-input"
                                            value="<?php echo esc_attr($field_value); ?>"
                                            placeholder="<?php esc_attr_e('Enter image URL or upload...', 'instarank'); ?>"
                                            style="font-size: 12px; border-color: #d1d5db;"
                                            data-field-key="<?php echo esc_attr($field_key); ?>"
                                            data-site-url="<?php echo esc_attr($site_url); ?>"
                                        />
                                    </div>
                                    <button
                                        type="button"
                                        class="button pseo-upload-image-btn"
                                        data-field-key="<?php echo esc_attr($field_key); ?>"
                                        style="height: 30px; padding: 0 12px;"
                                    >
                                        <span class="dashicons dashicons-upload" style="font-size: 16px; width: 16px; height: 16px; vertical-align: middle; margin-right: 2px;"></span>
                                        <?php esc_html_e('Upload', 'instarank'); ?>
                                    </button>
                                    <?php if ($has_image) : ?>
                                        <button
                                            type="button"
                                            class="button pseo-clear-image-btn"
                                            data-field-key="<?php echo esc_attr($field_key); ?>"
                                            style="height: 30px; padding: 0 10px; color: #b91c1c;"
                                            title="<?php esc_attr_e('Clear image', 'instarank'); ?>"
                                        >
                                            <span class="dashicons dashicons-no-alt" style="font-size: 16px; width: 16px; height: 16px; vertical-align: middle;"></span>
                                        </button>
                                    <?php endif; ?>
                                </div>

                                <?php if ($has_image) : ?>
                                    <div style="margin-top: 6px;">
                                        <a href="<?php echo esc_url($field_value); ?>" target="_blank" rel="noopener" style="font-size: 11px; color: #6b7280; text-decoration: none;">
                                            <span class="dashicons dashicons-external" style="font-size: 12px; width: 12px; height: 12px; vertical-align: middle;"></span>
                                            <?php esc_html_e('View full size', 'instarank'); ?>
                                        </a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>

                <?php if (!empty($editable_fields)) : ?>
                    <!-- Text Fields (Editable) -->
                    <div style="margin-bottom: 16px;">
                        <h4 style="margin: 0 0 10px 0; font-size: 12px; text-transform: uppercase; color: #6b7280; border-bottom: 1px solid #e5e7eb; padding-bottom: 6px;">
                            <?php esc_html_e('Text Fields', 'instarank'); ?>
                        </h4>
                        <?php
                        foreach ($editable_fields as $field_key => $field_value) :
                            $display_key = ucfirst(str_replace(['_', '-'], ' ', $field_key));
                            $is_long = strlen($field_value) > 100;
                            ?>
                            <div class="pseo-field-row" style="margin-bottom: 12px; background: white; padding: 10px 12px; border-radius: 6px; border: 1px solid #e2e4e7;">
                                <label for="pseo_field_<?php echo esc_attr($field_key); ?>" style="display: block; font-weight: 600; color: #2271b1; font-size: 12px; margin-bottom: 4px;">
                                    <?php echo esc_html($display_key); ?>
                                </label>
                                <?php if ($is_long) : ?>
                                    <textarea
                                        id="pseo_field_<?php echo esc_attr($field_key); ?>"
                                        name="instarank_pseo_fields[<?php echo esc_attr($field_key); ?>]"
                                        class="widefat pseo-field-input"
                                        rows="3"
                                        style="font-size: 13px; border-color: #d1d5db;"
                                    ><?php echo esc_textarea($field_value); ?></textarea>
                                <?php else : ?>
                                    <input
                                        type="text"
                                        id="pseo_field_<?php echo esc_attr($field_key); ?>"
                                        name="instarank_pseo_fields[<?php echo esc_attr($field_key); ?>]"
                                        class="widefat pseo-field-input"
                                        value="<?php echo esc_attr($field_value); ?>"
                                        style="font-size: 13px; border-color: #d1d5db;"
                                    />
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>

                <?php if (!empty($url_fields)) : ?>
                    <!-- URL Fields -->
                    <div style="margin-bottom: 16px;">
                        <h4 style="margin: 0 0 10px 0; font-size: 12px; text-transform: uppercase; color: #6b7280; border-bottom: 1px solid #e5e7eb; padding-bottom: 6px;">
                            <?php esc_html_e('URL Fields', 'instarank'); ?>
                        </h4>
                        <?php
                        foreach ($url_fields as $field_key => $field_value) :
                            $display_key = ucfirst(str_replace(['_', '-'], ' ', $field_key));
                            ?>
                            <div class="pseo-field-row" style="margin-bottom: 12px; background: white; padding: 10px 12px; border-radius: 6px; border: 1px solid #e2e4e7;">
                                <label for="pseo_field_<?php echo esc_attr($field_key); ?>" style="display: block; font-weight: 600; color: #2271b1; font-size: 12px; margin-bottom: 4px;">
                                    <?php echo esc_html($display_key); ?>
                                    <a href="<?php echo esc_url($field_value); ?>" target="_blank" rel="noopener" style="margin-left: 8px; font-weight: normal;">
                                        <span class="dashicons dashicons-external" style="font-size: 12px; width: 12px; height: 12px;"></span>
                                    </a>
                                </label>
                                <input
                                    type="url"
                                    id="pseo_field_<?php echo esc_attr($field_key); ?>"
                                    name="instarank_pseo_fields[<?php echo esc_attr($field_key); ?>]"
                                    class="widefat pseo-field-input"
                                    value="<?php echo esc_attr($field_value); ?>"
                                    style="font-size: 13px; border-color: #d1d5db;"
                                />
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>

                <?php if (!empty($html_fields)) : ?>
                    <!-- HTML Fields (Read-only display) -->
                    <div style="margin-bottom: 16px;">
                        <h4 style="margin: 0 0 10px 0; font-size: 12px; text-transform: uppercase; color: #6b7280; border-bottom: 1px solid #e5e7eb; padding-bottom: 6px;">
                            <?php esc_html_e('HTML Content Fields', 'instarank'); ?>
                        </h4>
                        <?php
                        foreach ($html_fields as $field_key => $field_value) :
                            $display_key = ucfirst(str_replace(['_', '-'], ' ', $field_key));
                            ?>
                            <div class="pseo-field-row" style="margin-bottom: 12px; background: white; padding: 10px 12px; border-radius: 6px; border: 1px solid #e2e4e7;">
                                <label for="pseo_field_<?php echo esc_attr($field_key); ?>" style="display: block; font-weight: 600; color: #2271b1; font-size: 12px; margin-bottom: 4px;">
                                    <?php echo esc_html($display_key); ?>
                                    <span style="font-weight: normal; color: #9ca3af; font-size: 11px; margin-left: 6px;"><?php esc_html_e('(HTML)', 'instarank'); ?></span>
                                </label>
                                <textarea
                                    id="pseo_field_<?php echo esc_attr($field_key); ?>"
                                    name="instarank_pseo_fields[<?php echo esc_attr($field_key); ?>]"
                                    class="widefat pseo-field-input pseo-field-html"
                                    rows="4"
                                    style="font-size: 12px; font-family: monospace; border-color: #d1d5db; background: #f9fafb;"
                                ><?php echo esc_textarea($field_value); ?></textarea>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>

            <div style="margin-top: 12px; padding-top: 12px; border-top: 1px solid #fed7aa; display: flex; justify-content: space-between; align-items: center;">
                <span style="font-size: 11px; color: #757575;">
                    <span class="dashicons dashicons-info" style="font-size: 14px; width: 14px; height: 14px; vertical-align: middle; margin-right: 4px;"></span>
                    <?php
                    /* translators: %d: number of fields */
                    echo esc_html(sprintf(__('%d fields from dataset', 'instarank'), count($pseo_fields)));
                    ?>
                </span>
                <span style="font-size: 11px; color: #059669; font-weight: 500;">
                    <span class="dashicons dashicons-saved" style="font-size: 14px; width: 14px; height: 14px; vertical-align: middle; margin-right: 2px;"></span>
                    <?php esc_html_e('Changes saved with post', 'instarank'); ?>
                </span>
            </div>
        </div>

        <style>
        .pseo-field-input:focus {
            border-color: #f97316 !important;
            box-shadow: 0 0 0 1px #f97316;
            outline: none;
        }
        .pseo-field-row:hover {
            border-color: #fed7aa !important;
        }
        .pseo-image-field .pseo-image-preview img {
            transition: transform 0.2s ease;
            cursor: pointer;
        }
        .pseo-image-field .pseo-image-preview img:hover {
            transform: scale(1.02);
        }
        .pseo-upload-image-btn:hover {
            background: #f97316 !important;
            border-color: #f97316 !important;
            color: white !important;
        }
        .pseo-clear-image-btn:hover {
            background: #fef2f2 !important;
            border-color: #fca5a5 !important;
        }
        </style>

        <script>
        jQuery(document).ready(function($) {
            // Handle image upload button click
            $('.pseo-upload-image-btn').on('click', function(e) {
                e.preventDefault();

                var button = $(this);
                var fieldKey = button.data('field-key');
                var inputField = $('#pseo_field_' + fieldKey);
                var previewContainer = button.closest('.pseo-image-field').find('.pseo-image-preview-container');

                // Create media frame
                var mediaFrame = wp.media({
                    title: '<?php echo esc_js(__('Select or Upload Image', 'instarank')); ?>',
                    button: {
                        text: '<?php echo esc_js(__('Use this image', 'instarank')); ?>'
                    },
                    multiple: false,
                    library: {
                        type: 'image'
                    }
                });

                // When image is selected
                mediaFrame.on('select', function() {
                    var attachment = mediaFrame.state().get('selection').first().toJSON();
                    var imageUrl = attachment.url;

                    // Update input field
                    inputField.val(imageUrl).trigger('change');

                    // Update preview
                    previewContainer.html(
                        '<div class="pseo-image-preview" style="position: relative; display: inline-block; max-width: 100%;">' +
                            '<img src="' + imageUrl + '" alt="" style="max-width: 200px; max-height: 150px; border-radius: 4px; border: 1px solid #e2e4e7; display: block;" />' +
                        '</div>'
                    );

                    // Add clear button if not present
                    var btnGroup = button.parent().parent();
                    if (btnGroup.find('.pseo-clear-image-btn').length === 0) {
                        button.after(
                            '<button type="button" class="button pseo-clear-image-btn" data-field-key="' + fieldKey + '" style="height: 30px; padding: 0 10px; color: #b91c1c;" title="<?php echo esc_js(__('Clear image', 'instarank')); ?>">' +
                                '<span class="dashicons dashicons-no-alt" style="font-size: 16px; width: 16px; height: 16px; vertical-align: middle;"></span>' +
                            '</button>'
                        );
                        // Rebind clear button event
                        bindClearButton();
                    }

                    // Add view full size link
                    var linkContainer = button.closest('.pseo-image-field').find('a[target="_blank"]').parent();
                    if (linkContainer.length === 0) {
                        previewContainer.after(
                            '<div style="margin-top: 6px;">' +
                                '<a href="' + imageUrl + '" target="_blank" rel="noopener" style="font-size: 11px; color: #6b7280; text-decoration: none;">' +
                                    '<span class="dashicons dashicons-external" style="font-size: 12px; width: 12px; height: 12px; vertical-align: middle;"></span> ' +
                                    '<?php echo esc_js(__('View full size', 'instarank')); ?>' +
                                '</a>' +
                            '</div>'
                        );
                    } else {
                        linkContainer.find('a').attr('href', imageUrl);
                    }
                });

                mediaFrame.open();
            });

            // Handle clear image button
            function bindClearButton() {
                $('.pseo-clear-image-btn').off('click').on('click', function(e) {
                    e.preventDefault();

                    var button = $(this);
                    var fieldKey = button.data('field-key');
                    var inputField = $('#pseo_field_' + fieldKey);
                    var previewContainer = button.closest('.pseo-image-field').find('.pseo-image-preview-container');

                    // Clear input
                    inputField.val('').trigger('change');

                    // Show empty placeholder
                    previewContainer.html(
                        '<div style="width: 200px; height: 100px; background: #f9fafb; border: 2px dashed #d1d5db; border-radius: 4px; display: flex; align-items: center; justify-content: center; color: #9ca3af; font-size: 12px;">' +
                            '<span class="dashicons dashicons-format-image" style="font-size: 24px; margin-right: 8px;"></span>' +
                            '<?php echo esc_js(__('No image set', 'instarank')); ?>' +
                        '</div>'
                    );

                    // Remove view full size link
                    button.closest('.pseo-image-field').find('a[target="_blank"]').parent().remove();

                    // Remove the clear button
                    button.remove();
                });
            }

            bindClearButton();

            // Update preview when URL is manually changed
            $('.pseo-image-url-input').on('change', function() {
                var input = $(this);
                var imageUrl = input.val();
                var previewContainer = input.closest('.pseo-image-field').find('.pseo-image-preview-container');
                var fieldKey = input.data('field-key');

                if (imageUrl && imageUrl.match(/^https?:\/\/.+/)) {
                    // Show loading state
                    previewContainer.html(
                        '<div style="width: 200px; height: 100px; background: #f9fafb; border: 1px solid #e2e4e7; border-radius: 4px; display: flex; align-items: center; justify-content: center; color: #9ca3af; font-size: 12px;">' +
                            '<span class="dashicons dashicons-update spin" style="animation: spin 1s linear infinite;"></span> ' +
                            '<?php echo esc_js(__('Loading...', 'instarank')); ?>' +
                        '</div>'
                    );

                    // Test if image loads
                    var testImg = new Image();
                    testImg.onload = function() {
                        previewContainer.html(
                            '<div class="pseo-image-preview" style="position: relative; display: inline-block; max-width: 100%;">' +
                                '<img src="' + imageUrl + '" alt="" style="max-width: 200px; max-height: 150px; border-radius: 4px; border: 1px solid #e2e4e7; display: block;" />' +
                            '</div>'
                        );
                    };
                    testImg.onerror = function() {
                        previewContainer.html(
                            '<div style="width: 200px; height: 100px; background: #fef2f2; border: 1px solid #fecaca; border-radius: 4px; display: flex; align-items: center; justify-content: center; color: #dc2626; font-size: 12px;">' +
                                '<span class="dashicons dashicons-warning" style="margin-right: 4px;"></span>' +
                                '<?php echo esc_js(__('Invalid image URL', 'instarank')); ?>' +
                            '</div>'
                        );
                    };
                    testImg.src = imageUrl;
                } else if (!imageUrl) {
                    previewContainer.html(
                        '<div style="width: 200px; height: 100px; background: #f9fafb; border: 2px dashed #d1d5db; border-radius: 4px; display: flex; align-items: center; justify-content: center; color: #9ca3af; font-size: 12px;">' +
                            '<span class="dashicons dashicons-format-image" style="font-size: 24px; margin-right: 8px;"></span>' +
                            '<?php echo esc_js(__('No image set', 'instarank')); ?>' +
                        '</div>'
                    );
                }
            });

            // Handle Import External Image button
            $('.pseo-import-image-btn').on('click', function(e) {
                e.preventDefault();

                var button = $(this);
                var fieldKey = button.data('field-key');
                var imageUrl = button.data('image-url');
                var inputField = $('#pseo_field_' + fieldKey);
                var warningContainer = button.closest('.pseo-external-warning');
                var fieldContainer = button.closest('.pseo-image-field');
                var previewContainer = fieldContainer.find('.pseo-image-preview-container');

                // Disable button and show loading state
                var originalText = button.html();
                button.prop('disabled', true).html(
                    '<span class="dashicons dashicons-update spin" style="font-size: 14px; width: 14px; height: 14px; vertical-align: middle; animation: spin 1s linear infinite;"></span> ' +
                    '<?php echo esc_js(__('Importing...', 'instarank')); ?>'
                );

                // Make AJAX request to import the image
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'instarank_import_external_image',
                        nonce: '<?php echo esc_js(wp_create_nonce('instarank_metabox_nonce')); ?>',
                        image_url: imageUrl,
                        field_key: fieldKey,
                        post_id: $('#post_ID').val()
                    },
                    success: function(response) {
                        if (response.success) {
                            // Update the input field with new local URL
                            // Use native setter for React compatibility in Gutenberg
                            var nativeInputValueSetter = Object.getOwnPropertyDescriptor(window.HTMLInputElement.prototype, 'value').set;
                            nativeInputValueSetter.call(inputField[0], response.data.url);
                            inputField[0].dispatchEvent(new Event('input', { bubbles: true }));
                            inputField[0].dispatchEvent(new Event('change', { bubbles: true }));

                            // Also update the data attribute for the import button
                            button.data('image-url', response.data.url);

                            // Update the preview image
                            previewContainer.find('img').attr('src', response.data.url);

                            // Mark the post as dirty so Gutenberg knows to save
                            if (typeof wp !== 'undefined' && wp.data && wp.data.dispatch) {
                                wp.data.dispatch('core/editor').editPost({ meta: { _instarank_pseo_modified: Date.now().toString() } });
                            }

                            // Remove the external warning
                            warningContainer.fadeOut(300, function() {
                                $(this).remove();
                            });

                            // Remove the "External" badge from the label
                            fieldContainer.find('.pseo-external-badge').fadeOut(300, function() {
                                $(this).remove();
                            });

                            // Show success message with saved indicator
                            button.html(
                                '<span class="dashicons dashicons-yes-alt" style="font-size: 14px; width: 14px; height: 14px; vertical-align: middle; color: #059669;"></span> ' +
                                '<?php echo esc_js(__('Imported & Saved!', 'instarank')); ?>'
                            );

                            // Show a toast notification
                            var toast = $('<div class="pseo-import-toast" style="position: fixed; bottom: 30px; right: 30px; background: #059669; color: white; padding: 12px 20px; border-radius: 8px; font-size: 13px; z-index: 999999; box-shadow: 0 4px 12px rgba(0,0,0,0.15);">' +
                                '<span class="dashicons dashicons-saved" style="margin-right: 8px; vertical-align: middle;"></span>' +
                                '<?php echo esc_js(__('Image imported and saved to database!', 'instarank')); ?>' +
                            '</div>');
                            $('body').append(toast);
                            setTimeout(function() {
                                toast.fadeOut(300, function() { $(this).remove(); });
                            }, 3000);

                            // Fade out the button after 1 second
                            setTimeout(function() {
                                button.fadeOut();
                            }, 1000);

                            // Log for debugging
                            console.log('[InstaRank] Image imported:', response.data.original_url, '→', response.data.url);
                            console.log('[InstaRank] Content updated:', response.data.content_updated);
                        } else {
                            // Show error
                            button.prop('disabled', false).html(originalText);
                            alert('<?php echo esc_js(__('Failed to import image: ', 'instarank')); ?>' + (response.data.message || 'Unknown error'));
                        }
                    },
                    error: function(xhr, status, error) {
                        button.prop('disabled', false).html(originalText);
                        alert('<?php echo esc_js(__('Failed to import image: ', 'instarank')); ?>' + error);
                    }
                });
            });
        });
        </script>

        <style>
        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        </style>
        <?php
    }

    /**
     * Save pSEO field values and update content
     *
     * @param int $post_id The post ID.
     */
    public function save_pseo_fields($post_id) {
        // Skip if no pSEO fields submitted
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified below
        if (!isset($_POST['instarank_pseo_fields']) || !is_array($_POST['instarank_pseo_fields'])) {
            return;
        }

        // Verify nonce - use the existing metabox nonce
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification is done here
        if (!isset($_POST['instarank_metabox_nonce']) || !wp_verify_nonce(sanitize_key($_POST['instarank_metabox_nonce']), 'instarank_metabox_nonce')) {
            return;
        }

        // Skip if not a pSEO generated page
        $is_pseo_page = get_post_meta($post_id, '_instarank_pseo_generated', true);
        if (!$is_pseo_page) {
            return;
        }

        // Get current field values
        $current_fields = get_post_meta($post_id, '_instarank_pseo_fields', true);
        if (!is_array($current_fields)) {
            $current_fields = [];
        }

        // Process submitted fields - each field is sanitized individually in the loop
        $new_fields = [];
        // phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.NonceVerification.Missing -- wp_unslash applied via array_map, fields sanitized individually below, nonce verified above
        $submitted_fields = isset($_POST['instarank_pseo_fields']) && is_array($_POST['instarank_pseo_fields'])
            ? array_map('wp_unslash', $_POST['instarank_pseo_fields'])
            : [];
        // phpcs:enable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.NonceVerification.Missing

        $site_url = get_site_url();

        foreach ($submitted_fields as $key => $value) {
            $key = sanitize_key($key);
            // Ensure value is a string before sanitizing
            $value = is_string($value) ? $value : '';

            // Check if this was previously an image/URL field (to handle clearing)
            $was_url_field = isset($current_fields[$key]) && filter_var($current_fields[$key], FILTER_VALIDATE_URL);
            $was_image_field = isset($current_fields[$key]) && ($this->is_image_field_name($key) || $this->is_image_url($current_fields[$key]));

            // IMPORTANT: Handle race condition when image was imported via AJAX
            // If the database has a local WordPress URL but form has an external URL,
            // keep the database value (the imported one)
            if ($was_image_field && isset($current_fields[$key])) {
                $db_is_local = strpos($current_fields[$key], $site_url) === 0;
                $form_is_external = filter_var($value, FILTER_VALIDATE_URL) && strpos($value, $site_url) !== 0;

                if ($db_is_local && $form_is_external) {
                    // Database has local URL, form has external - keep database value
                    // This happens when AJAX import ran but form wasn't updated
                    $new_fields[$key] = $current_fields[$key];
                    continue;
                }
            }

            // Check if field contains HTML
            if (isset($current_fields[$key]) && preg_match('/<[^>]+>/', $current_fields[$key])) {
                // Allow HTML for fields that had HTML before
                $new_fields[$key] = wp_kses_post($value);
            } elseif (filter_var($value, FILTER_VALIDATE_URL)) {
                // URL fields - new value is a URL
                $new_fields[$key] = esc_url_raw($value);
            } elseif (empty($value) && ($was_url_field || $was_image_field)) {
                // Field was previously a URL/image but is now being cleared
                // Store empty string to indicate deletion
                $new_fields[$key] = '';
            } else {
                // Plain text fields
                $new_fields[$key] = sanitize_textarea_field($value);
            }
        }

        // Merge with current fields (preserves any fields not in the form)
        $updated_fields = array_merge($current_fields, $new_fields);

        // Check if any fields actually changed
        $fields_changed = false;
        foreach ($new_fields as $key => $value) {
            if (!isset($current_fields[$key]) || $current_fields[$key] !== $value) {
                $fields_changed = true;
                break;
            }
        }

        if (!$fields_changed) {
            return;
        }

        // Update the post content with new field values BEFORE updating meta
        // This way we can compare old vs new values
        $this->update_content_with_fields($post_id, $new_fields, $current_fields);

        // Now update the stored fields
        update_post_meta($post_id, '_instarank_pseo_fields', $updated_fields);
        update_post_meta($post_id, '_instarank_pseo_fields_modified', gmdate('Y-m-d H:i:s'));
    }

    /**
     * Update post content by replacing field placeholders with new values
     *
     * @param int   $post_id        The post ID.
     * @param array $new_fields     The new field values.
     * @param array $current_fields The current/old field values for comparison.
     */
    private function update_content_with_fields($post_id, $new_fields, $current_fields = null) {
        $post = get_post($post_id);
        if (!$post) {
            return;
        }

        $content = $post->post_content;
        $title = $post->post_title;
        $content_changed = false;
        $title_changed = false;

        // If current_fields not provided, fetch from database (fallback for backward compatibility)
        if ($current_fields === null) {
            $current_fields = get_post_meta($post_id, '_instarank_pseo_fields', true);
            if (!is_array($current_fields)) {
                $current_fields = [];
            }
        }

        foreach ($new_fields as $field_key => $new_value) {
            // Get the original value if it was different
            if (isset($current_fields[$field_key]) && $current_fields[$field_key] !== $new_value) {
                $old_value = $current_fields[$field_key];

                // Skip if old value is empty (nothing to replace)
                if (empty($old_value)) {
                    continue;
                }

                // Check if this is an image field being cleared
                $is_image_field = $this->is_image_field_name($field_key) || $this->is_image_url($old_value);
                $is_clearing_image = $is_image_field && empty($new_value);

                // For non-image fields, skip if new value is empty
                if (!$is_image_field && empty($new_value)) {
                    continue;
                }

                // Replace in content (handle HTML content carefully)
                if (strpos($content, $old_value) !== false) {
                    if ($is_clearing_image) {
                        // For images being cleared, we need to handle different contexts:
                        // 1. img src="" - replace the URL, leaving an empty src
                        // 2. background-image: url() - similar handling
                        $content = str_replace($old_value, '', $content);
                    } else {
                        $content = str_replace($old_value, $new_value, $content);
                    }
                    $content_changed = true;
                }

                // Also check for HTML-encoded versions
                $old_value_encoded = esc_html($old_value);
                if ($old_value_encoded !== $old_value && strpos($content, $old_value_encoded) !== false) {
                    if ($is_clearing_image) {
                        $content = str_replace($old_value_encoded, '', $content);
                    } else {
                        $content = str_replace($old_value_encoded, esc_html($new_value), $content);
                    }
                    $content_changed = true;
                }

                // Also check for URL-encoded versions (common in JSON/schema)
                $old_value_url_encoded = rawurlencode($old_value);
                if (strpos($content, $old_value_url_encoded) !== false) {
                    $content = str_replace($old_value_url_encoded, $is_clearing_image ? '' : rawurlencode($new_value), $content);
                    $content_changed = true;
                }

                // Replace in title (only for non-image fields or when updating, not clearing)
                if (!$is_clearing_image && strpos($title, $old_value) !== false) {
                    $title = str_replace($old_value, $new_value, $title);
                    $title_changed = true;
                }

                // Update schema/structured data if it exists
                $this->update_schema_image($post_id, $old_value, $is_clearing_image ? '' : $new_value);
            }
        }

        // Update post if content or title changed
        if ($content_changed || $title_changed) {
            // Temporarily remove this filter to avoid infinite loop
            remove_action('save_post', [$this, 'save_meta_box']);
            remove_action('save_post', [$this, 'save_pseo_fields']);

            $update_args = ['ID' => $post_id];

            if ($content_changed) {
                $update_args['post_content'] = $content;
            }

            if ($title_changed) {
                $update_args['post_title'] = $title;
            }

            wp_update_post($update_args);

            // Re-add the filters
            add_action('save_post', [$this, 'save_meta_box']);
            add_action('save_post', [$this, 'save_pseo_fields']);
        }
    }

    /**
     * AJAX handler for saving pSEO fields from Gutenberg sidebar
     */
    public function ajax_save_pseo_fields() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'instarank_gutenberg')) {
            wp_send_json_error(['message' => 'Invalid nonce']);
            return;
        }

        // Get post ID
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        if (!$post_id) {
            wp_send_json_error(['message' => 'Invalid post ID']);
            return;
        }

        // Check permissions
        if (!current_user_can('edit_post', $post_id)) {
            wp_send_json_error(['message' => 'Permission denied']);
            return;
        }

        // Check if this is a pSEO page
        $is_pseo_page = get_post_meta($post_id, '_instarank_pseo_generated', true);
        if (!$is_pseo_page) {
            wp_send_json_error(['message' => 'Not a pSEO generated page']);
            return;
        }

        // Get submitted fields
        $fields_json = isset($_POST['fields']) ? sanitize_text_field(wp_unslash($_POST['fields'])) : '{}';
        $submitted_fields = json_decode($fields_json, true);

        if (!is_array($submitted_fields) || empty($submitted_fields)) {
            wp_send_json_error(['message' => 'No fields to save']);
            return;
        }

        // Get current fields
        $current_fields = get_post_meta($post_id, '_instarank_pseo_fields', true);
        if (!is_array($current_fields)) {
            $current_fields = [];
        }

        // Process and sanitize submitted fields
        $new_fields = [];
        foreach ($submitted_fields as $key => $value) {
            $key = sanitize_key($key);

            // Check if field contains HTML
            if (isset($current_fields[$key]) && preg_match('/<[^>]+>/', $current_fields[$key])) {
                // Allow HTML for fields that had HTML before
                $new_fields[$key] = wp_kses_post($value);
            } elseif (filter_var($value, FILTER_VALIDATE_URL)) {
                // URL fields
                $new_fields[$key] = esc_url_raw($value);
            } else {
                // Plain text fields
                $new_fields[$key] = sanitize_textarea_field($value);
            }
        }

        // Merge with current fields
        $updated_fields = array_merge($current_fields, $new_fields);

        // Update the stored fields
        update_post_meta($post_id, '_instarank_pseo_fields', $updated_fields);
        update_post_meta($post_id, '_instarank_pseo_fields_modified', gmdate('Y-m-d H:i:s'));

        // Update the post content with new field values
        $this->update_content_with_fields_ajax($post_id, $new_fields, $current_fields);

        wp_send_json_success([
            'message' => 'Fields saved successfully',
            'updated_count' => count($new_fields),
        ]);
    }

    /**
     * Update post content with field changes (for AJAX calls)
     *
     * @param int   $post_id        The post ID.
     * @param array $new_fields     The new field values.
     * @param array $current_fields The current field values (for comparison).
     */
    private function update_content_with_fields_ajax($post_id, $new_fields, $current_fields) {
        $post = get_post($post_id);
        if (!$post) {
            return;
        }

        $content = $post->post_content;
        $title = $post->post_title;
        $content_changed = false;
        $title_changed = false;

        foreach ($new_fields as $field_key => $new_value) {
            // Get the original value
            if (isset($current_fields[$field_key]) && $current_fields[$field_key] !== $new_value) {
                $old_value = $current_fields[$field_key];

                // Skip if old value is empty (nothing to replace)
                if (empty($old_value)) {
                    continue;
                }

                // Check if this is an image field being cleared
                $is_image_field = $this->is_image_field_name($field_key) || $this->is_image_url($old_value);
                $is_clearing_image = $is_image_field && empty($new_value);

                // For non-image fields, skip if new value is empty
                if (!$is_image_field && empty($new_value)) {
                    continue;
                }

                // Replace in content
                if (strpos($content, $old_value) !== false) {
                    if ($is_clearing_image) {
                        $content = str_replace($old_value, '', $content);
                    } else {
                        $content = str_replace($old_value, $new_value, $content);
                    }
                    $content_changed = true;
                }

                // Also check for HTML-encoded versions
                $old_value_encoded = esc_html($old_value);
                if ($old_value_encoded !== $old_value && strpos($content, $old_value_encoded) !== false) {
                    if ($is_clearing_image) {
                        $content = str_replace($old_value_encoded, '', $content);
                    } else {
                        $content = str_replace($old_value_encoded, esc_html($new_value), $content);
                    }
                    $content_changed = true;
                }

                // Also check for URL-encoded versions (common in JSON/schema)
                $old_value_url_encoded = rawurlencode($old_value);
                if (strpos($content, $old_value_url_encoded) !== false) {
                    $content = str_replace($old_value_url_encoded, $is_clearing_image ? '' : rawurlencode($new_value), $content);
                    $content_changed = true;
                }

                // Replace in title (only for non-image fields or when updating, not clearing)
                if (!$is_clearing_image && strpos($title, $old_value) !== false) {
                    $title = str_replace($old_value, $new_value, $title);
                    $title_changed = true;
                }

                // Update schema/structured data if it exists
                $this->update_schema_image($post_id, $old_value, $is_clearing_image ? '' : $new_value);
            }
        }

        // Update post if content or title changed
        if ($content_changed || $title_changed) {
            $update_args = ['ID' => $post_id];

            if ($content_changed) {
                $update_args['post_content'] = $content;
            }

            if ($title_changed) {
                $update_args['post_title'] = $title;
            }

            // Use wp_update_post directly without triggering save_post hooks
            remove_action('save_post', [$this, 'save_meta_box']);
            remove_action('save_post', [$this, 'save_pseo_fields']);

            wp_update_post($update_args);

            add_action('save_post', [$this, 'save_meta_box']);
            add_action('save_post', [$this, 'save_pseo_fields']);
        }
    }

    /**
     * AJAX handler for importing external images to WordPress Media Library
     * This downloads an external image and uploads it to the media library
     */
    public function ajax_import_external_image() {
        // Verify nonce
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if (!wp_verify_nonce($nonce, 'instarank_gutenberg') && !wp_verify_nonce($nonce, 'instarank_metabox_nonce')) {
            wp_send_json_error(['message' => 'Invalid nonce']);
            return;
        }

        // Check user capabilities
        if (!current_user_can('upload_files')) {
            wp_send_json_error(['message' => 'You do not have permission to upload files']);
            return;
        }

        // Get and validate the image URL
        $image_url = isset($_POST['image_url']) ? esc_url_raw(wp_unslash($_POST['image_url'])) : '';
        if (empty($image_url)) {
            wp_send_json_error(['message' => 'Image URL is required']);
            return;
        }

        // Validate it's a valid URL
        if (!filter_var($image_url, FILTER_VALIDATE_URL)) {
            wp_send_json_error(['message' => 'Invalid image URL']);
            return;
        }

        // Check if it's already a local WordPress URL
        $site_url = get_site_url();
        if (strpos($image_url, $site_url) === 0) {
            wp_send_json_success([
                'message' => 'Image is already in WordPress',
                'url' => $image_url,
                'already_local' => true,
            ]);
            return;
        }

        // Get optional parameters
        $field_key = isset($_POST['field_key']) ? sanitize_key(wp_unslash($_POST['field_key'])) : '';
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;

        // Load required WordPress functions
        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';

        // Download the image
        $tmp_file = download_url($image_url, 30); // 30 second timeout

        if (is_wp_error($tmp_file)) {
            wp_send_json_error([
                'message' => 'Failed to download image: ' . $tmp_file->get_error_message(),
            ]);
            return;
        }

        // Get file info
        $url_path = wp_parse_url($image_url, PHP_URL_PATH);
        $filename = $url_path ? basename($url_path) : 'imported-image.jpg';

        // Remove query string from filename
        $filename = preg_replace('/\?.*$/', '', $filename);

        // Ensure filename has an extension
        $file_type = wp_check_filetype($filename);
        if (empty($file_type['ext'])) {
            // Try to get extension from mime type
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mime_type = finfo_file($finfo, $tmp_file);
            finfo_close($finfo);

            $ext_map = [
                'image/jpeg' => 'jpg',
                'image/png' => 'png',
                'image/gif' => 'gif',
                'image/webp' => 'webp',
                'image/svg+xml' => 'svg',
            ];

            if (isset($ext_map[$mime_type])) {
                $filename .= '.' . $ext_map[$mime_type];
            } else {
                $filename .= '.jpg'; // Default to jpg
            }
        }

        // Make filename unique to avoid conflicts
        $filename = wp_unique_filename(wp_upload_dir()['path'], $filename);

        // Prepare file array for sideloading
        $file_array = [
            'name' => $filename,
            'tmp_name' => $tmp_file,
        ];

        // Sideload the file into the media library
        $attachment_id = media_handle_sideload($file_array, $post_id);

        // Clean up temp file if it still exists
        if (file_exists($tmp_file)) {
            wp_delete_file($tmp_file);
        }

        if (is_wp_error($attachment_id)) {
            wp_send_json_error([
                'message' => 'Failed to import image: ' . $attachment_id->get_error_message(),
            ]);
            return;
        }

        // Get the new WordPress URL
        $new_url = wp_get_attachment_url($attachment_id);

        // Add alt text if field_key is available
        if ($field_key) {
            $alt_text = ucwords(str_replace(['_', '-'], ' ', $field_key));
            update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt_text);
        }

        // Store source URL for reference
        update_post_meta($attachment_id, '_instarank_source_url', $image_url);

        // IMPORTANT: Update pSEO fields and post content with the new local URL
        if ($post_id && $field_key) {
            // Update pSEO fields metadata
            $pseo_fields = get_post_meta($post_id, '_instarank_pseo_fields', true);
            if (is_array($pseo_fields) && isset($pseo_fields[$field_key])) {
                $pseo_fields[$field_key] = $new_url;
                update_post_meta($post_id, '_instarank_pseo_fields', $pseo_fields);
                update_post_meta($post_id, '_instarank_pseo_fields_modified', gmdate('Y-m-d H:i:s'));
            }

            // Update post content - replace old external URL with new local URL
            $post = get_post($post_id);
            if ($post) {
                $content = $post->post_content;
                $content_changed = false;

                // Replace direct URL
                if (strpos($content, $image_url) !== false) {
                    $content = str_replace($image_url, $new_url, $content);
                    $content_changed = true;
                }

                // Also check for HTML-encoded versions
                $old_encoded = esc_html($image_url);
                if ($old_encoded !== $image_url && strpos($content, $old_encoded) !== false) {
                    $content = str_replace($old_encoded, esc_html($new_url), $content);
                    $content_changed = true;
                }

                // Also check for URL-encoded versions (common in JSON/schema)
                $old_url_encoded = rawurlencode($image_url);
                if (strpos($content, $old_url_encoded) !== false) {
                    $content = str_replace($old_url_encoded, rawurlencode($new_url), $content);
                    $content_changed = true;
                }

                // Also check for escaped slashes (common in JSON)
                $old_escaped = str_replace('/', '\\/', $image_url);
                $new_escaped = str_replace('/', '\\/', $new_url);
                if (strpos($content, $old_escaped) !== false) {
                    $content = str_replace($old_escaped, $new_escaped, $content);
                    $content_changed = true;
                }

                // Update post if content changed
                if ($content_changed) {
                    wp_update_post([
                        'ID' => $post_id,
                        'post_content' => $content,
                    ]);
                }

                // Update schema/structured data
                $this->update_schema_image($post_id, $image_url, $new_url);
            }
        }

        wp_send_json_success([
            'message' => 'Image imported successfully',
            'attachment_id' => $attachment_id,
            'url' => $new_url,
            'original_url' => $image_url,
            'content_updated' => isset($content_changed) ? $content_changed : false,
        ]);
    }

    /**
     * Update schema/structured data when an image URL changes
     *
     * @param int    $post_id   The post ID.
     * @param string $old_url   The old image URL.
     * @param string $new_url   The new image URL (can be empty to remove).
     */
    private function update_schema_image($post_id, $old_url, $new_url) {
        if (empty($old_url)) {
            return;
        }

        // Check common schema storage meta keys
        $schema_meta_keys = [
            '_instarank_schema',
            '_yoast_wpseo_schema_article_type',
            '_schema_json',
            '_rank_math_schema_Article',
            '_aioseo_schema',
        ];

        foreach ($schema_meta_keys as $meta_key) {
            $schema_data = get_post_meta($post_id, $meta_key, true);

            if (empty($schema_data)) {
                continue;
            }

            // Handle both string (JSON) and array formats
            $is_string = is_string($schema_data);
            $schema_string = $is_string ? $schema_data : wp_json_encode($schema_data);

            // Replace old URL with new URL in the schema
            if (strpos($schema_string, $old_url) !== false) {
                $schema_string = str_replace($old_url, $new_url, $schema_string);

                // Also check for escaped versions (common in JSON)
                $old_escaped = str_replace('/', '\\/', $old_url);
                $new_escaped = empty($new_url) ? '' : str_replace('/', '\\/', $new_url);
                $schema_string = str_replace($old_escaped, $new_escaped, $schema_string);

                // Save back in the original format
                if ($is_string) {
                    update_post_meta($post_id, $meta_key, $schema_string);
                } else {
                    $schema_array = json_decode($schema_string, true);
                    if (is_array($schema_array)) {
                        update_post_meta($post_id, $meta_key, $schema_array);
                    }
                }
            }
        }

        // Also check for images in script tags in post content (inline schema)
        // This is already handled by the main content replacement
    }

    /**
     * Check if a URL is an external image (not already in WordPress)
     *
     * @param string $url The URL to check.
     * @return bool True if external, false if local.
     */
    private function is_external_image_url($url) {
        if (empty($url)) {
            return false;
        }

        $site_url = get_site_url();
        return strpos($url, $site_url) !== 0;
    }
}

// Initialize
InstaRank_Classic_Editor::instance();
