<?php
/**
 * Change Manager
 * Handles SEO change storage, approval, application, and rollback
 */

defined('ABSPATH') || exit;

class InstaRank_Change_Manager {

    /**
     * Store a pending change
     */
    public function store_change($post_id, $change_type, $old_value, $new_value, $reason = '') {
        global $wpdb;

        $table = $wpdb->prefix . 'instarank_changes';

        // Validate change type is allowed
        $allowed_types = get_option('instarank_allowed_change_types', [
            // Meta tags
            'meta_title',
            'meta_description',
            'image_alt',
            'canonical',
            'og_title',
            'og_description',
            'og_image',
            'og_url',
            'robots_noindex',
            'focus_keyword',
            'post_title',
            // Content issues
            'low_word_count',
            'duplicate_title',
            'duplicate_description',
            'missing_h1',
            'multiple_h1',
            'broken_links',
            'slow_page',
            'missing_image_alt',
            'large_image',
            'missing_structured_data',
            // Additional SEO fields
            'robots_meta',
            'meta_keywords',
            'schema_markup'
        ]);
        if (!in_array($change_type, $allowed_types)) {
            // error_log("InstaRank: Change type '$change_type' is not allowed");
            return false;
        }

        // Get post info
        $post = get_post($post_id);
        if (!$post) {
            // error_log("InstaRank: Post $post_id not found");
            return false;
        }

        // Insert change record
        $metadata = [
            'post_title' => $post->post_title,
            'post_url' => get_permalink($post_id),
            'reason' => $reason
        ];

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
        $result = $wpdb->insert($table, [
            'post_id' => $post_id,
            'change_type' => $change_type,
            'field_name' => null, // Will be set when applied
            'old_value' => $old_value,
            'new_value' => $new_value,
            'status' => 'pending',
            'metadata' => json_encode($metadata)
        ]);

        if ($result) {
            $change_id = $wpdb->insert_id;

            // Send webhook notification
            $webhook = new InstaRank_Webhook_Sender();
            $webhook->send_change_received([
                'change_id' => $change_id,
                'post_id' => $post_id,
                'change_type' => $change_type
            ]);

            return $change_id;
        }

        return false;
    }

    /**
     * Get changes by status
     */
    public function get_changes($status = 'all', $limit = 100) {
        global $wpdb;

        $table = $wpdb->prefix . 'instarank_changes';

        if ($status === 'all') {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            return $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT * FROM {$wpdb->prefix}instarank_changes ORDER BY created_at DESC LIMIT %d",
                    $limit
                ),
                ARRAY_A
            );
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        return $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}instarank_changes WHERE status = %s ORDER BY created_at DESC LIMIT %d",
                $status,
                $limit
            ),
            ARRAY_A
        );
    }

    /**
     * Approve and apply a change
     */
    public function approve_change($change_id) {
        global $wpdb;

        $table = $wpdb->prefix . 'instarank_changes';

        // Get the change
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $change = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d AND status = 'pending'", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $change_id
        ), ARRAY_A);

        if (!$change) {
            return false;
        }

        // Apply the change
        $applied = $this->apply_change($change);

        if ($applied) {
            // Update status
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table update
            $wpdb->update($table, [
                'status' => 'applied',
                'applied_at' => current_time('mysql')
            ], ['id' => $change_id]);

            // Send webhook notification
            $webhook = new InstaRank_Webhook_Sender();
            $webhook->send_change_applied([
                'change_id' => $change_id,
                'post_id' => $change['post_id'],
                'change_type' => $change['change_type']
            ]);

            return true;
        }

        return false;
    }

    /**
     * Apply a change to the post
     */
    private function apply_change($change) {
        $detector = new InstaRank_SEO_Detector();
        $seo_plugin = $detector->get_active_seo_plugin();
        $post_id = $change['post_id'];
        $change_type = $change['change_type'];
        $new_value = $change['new_value'];

        switch ($change_type) {
            case 'meta_title':
                return $this->update_meta_title($post_id, $new_value, $seo_plugin);

            case 'meta_description':
                return $this->update_meta_description($post_id, $new_value, $seo_plugin);

            case 'focus_keyword':
                return $this->update_focus_keyword($post_id, $new_value, $seo_plugin);

            case 'post_title':
                return $this->update_post_title($post_id, $new_value);

            case 'image_alt':
                return $this->update_image_alt($change['attachment_id'] ?? 0, $new_value);

            case 'canonical':
                return $this->update_canonical($post_id, $new_value, $seo_plugin);

            case 'og_title':
                return $this->update_og_title($post_id, $new_value, $seo_plugin);

            case 'og_description':
                return $this->update_og_description($post_id, $new_value, $seo_plugin);

            case 'og_image':
                return $this->update_og_image($post_id, $new_value, $seo_plugin);

            case 'og_url':
                return $this->update_og_url($post_id, $new_value, $seo_plugin);

            case 'robots_noindex':
                return $this->update_robots_noindex($post_id, $new_value, $seo_plugin);

            case 'content':
            case 'post_content':
                return $this->update_post_content($post_id, $new_value);

            case 'robots_meta':
                return $this->update_robots_meta($post_id, $new_value, $seo_plugin);

            case 'schema_markup':
                return $this->update_schema_markup($post_id, $new_value, $seo_plugin);

            case 'twitter_title':
                return $this->update_twitter_title($post_id, $new_value, $seo_plugin);

            case 'twitter_description':
                return $this->update_twitter_description($post_id, $new_value, $seo_plugin);

            case 'twitter_image':
                return $this->update_twitter_image($post_id, $new_value, $seo_plugin);

            default:
                // error_log("InstaRank: Unknown change type '$change_type'");
                return false;
        }
    }

    /**
     * Update robots meta directives
     * Supports: robots_meta issues
     */
    private function update_robots_meta($post_id, $value, $seo_plugin) {
        switch ($seo_plugin) {
            case 'yoast':
                return update_post_meta($post_id, '_yoast_wpseo_meta-robots-adv', $value);

            case 'rankmath':
                // Rank Math uses array format for robots
                $robots = get_post_meta($post_id, 'rank_math_robots', true);
                if (!is_array($robots)) {
                    $robots = [];
                }
                // Parse value like "nofollow,noarchive"
                $directives = array_map('trim', explode(',', $value));
                foreach ($directives as $directive) {
                    $robots[$directive] = $directive;
                }
                return update_post_meta($post_id, 'rank_math_robots', $robots);

            case 'aioseo':
                return update_post_meta($post_id, '_aioseo_robots_default', $value);

            default:
                return update_post_meta($post_id, 'instarank_robots_meta', $value);
        }
    }

    /**
     * Update schema markup
     * Supports: missing_structured_data, missing_schema_* fixes
     */
    private function update_schema_markup($post_id, $value, $seo_plugin) {
        // Parse JSON-LD if value is JSON string
        $schema_data = json_decode($value, true);

        if (is_array($schema_data) && isset($schema_data['@type'])) {
            // Valid JSON-LD schema
            return $this->apply_json_ld_schema($post_id, $schema_data, $value, $seo_plugin);
        }

        // Legacy support: simple string value
        switch ($seo_plugin) {
            case 'yoast':
                return update_post_meta($post_id, '_yoast_wpseo_schema_page_type', $value);

            case 'rankmath':
                return update_post_meta($post_id, 'rank_math_rich_snippet', $value);

            case 'aioseo':
                return update_post_meta($post_id, '_aioseo_local_business_type', $value);

            default:
                return update_post_meta($post_id, 'instarank_schema_markup', $value);
        }
    }

    /**
     * Apply JSON-LD schema markup to a post
     * Handles different SEO plugins and provides fallback
     */
    private function apply_json_ld_schema($post_id, $schema_data, $schema_json, $seo_plugin) {
        switch ($seo_plugin) {
            case 'yoast':
                return $this->apply_yoast_schema($post_id, $schema_data, $schema_json);

            case 'rankmath':
                return $this->apply_rankmath_schema($post_id, $schema_data, $schema_json);

            case 'aioseo':
                return $this->apply_aioseo_schema($post_id, $schema_data, $schema_json);

            default:
                // Fallback: store raw JSON-LD for wp_head injection
                return update_post_meta($post_id, 'instarank_schema_json_ld', $schema_json);
        }
    }

    /**
     * Apply schema for Yoast SEO
     */
    private function apply_yoast_schema($post_id, $schema_data, $schema_json) {
        // Yoast auto-generates schema, so we store custom JSON-LD
        // and also set the page type for Yoast's benefit
        update_post_meta($post_id, '_yoast_wpseo_schema_page_type', $schema_data['@type']);
        update_post_meta($post_id, 'instarank_schema_json_ld', $schema_json);
        return true;
    }

    /**
     * Apply schema for Rank Math
     */
    private function apply_rankmath_schema($post_id, $schema_data, $schema_json) {
        // Map schema types to Rank Math rich snippet types
        $snippet_map = [
            'Article' => 'article',
            'BlogPosting' => 'article',
            'LocalBusiness' => 'local',
            'Person' => 'person',
            'Product' => 'product',
            'FAQPage' => 'faqpage',
            'Organization' => 'organization',
        ];

        $schema_type = $schema_data['@type'];
        $snippet_type = isset($snippet_map[$schema_type]) ? $snippet_map[$schema_type] : 'off';

        // Set Rank Math rich snippet type
        update_post_meta($post_id, 'rank_math_rich_snippet', $snippet_type);

        // Store full JSON-LD for custom schemas or additional data
        update_post_meta($post_id, 'instarank_schema_json_ld', $schema_json);

        return true;
    }

    /**
     * Apply schema for All in One SEO
     */
    private function apply_aioseo_schema($post_id, $schema_data, $schema_json) {
        // AIOSEO has limited built-in schema support
        // Store JSON-LD for custom injection
        update_post_meta($post_id, 'instarank_schema_json_ld', $schema_json);

        // If it's a LocalBusiness schema, set AIOSEO's local business type
        if ($schema_data['@type'] === 'LocalBusiness') {
            update_post_meta($post_id, '_aioseo_local_business_type', 'LocalBusiness');
        }

        return true;
    }

    /**
     * Update Twitter Card title
     */
    private function update_twitter_title($post_id, $value, $seo_plugin) {
        switch ($seo_plugin) {
            case 'yoast':
                return update_post_meta($post_id, '_yoast_wpseo_twitter-title', $value);

            case 'rankmath':
                return update_post_meta($post_id, 'rank_math_twitter_title', $value);

            case 'aioseo':
                return update_post_meta($post_id, '_aioseo_twitter_title', $value);

            default:
                return update_post_meta($post_id, 'instarank_twitter_title', $value);
        }
    }

    /**
     * Update Twitter Card description
     */
    private function update_twitter_description($post_id, $value, $seo_plugin) {
        switch ($seo_plugin) {
            case 'yoast':
                return update_post_meta($post_id, '_yoast_wpseo_twitter-description', $value);

            case 'rankmath':
                return update_post_meta($post_id, 'rank_math_twitter_description', $value);

            case 'aioseo':
                return update_post_meta($post_id, '_aioseo_twitter_description', $value);

            default:
                return update_post_meta($post_id, 'instarank_twitter_description', $value);
        }
    }

    /**
     * Update Twitter Card image
     */
    private function update_twitter_image($post_id, $value, $seo_plugin) {
        switch ($seo_plugin) {
            case 'yoast':
                return update_post_meta($post_id, '_yoast_wpseo_twitter-image', $value);

            case 'rankmath':
                return update_post_meta($post_id, 'rank_math_twitter_image', $value);

            case 'aioseo':
                return update_post_meta($post_id, '_aioseo_twitter_image_url', $value);

            default:
                return update_post_meta($post_id, 'instarank_twitter_image', $value);
        }
    }

    /**
     * Update meta title based on active SEO plugin
     */
    private function update_meta_title($post_id, $value, $seo_plugin) {
        switch ($seo_plugin) {
            case 'yoast':
                return update_post_meta($post_id, '_yoast_wpseo_title', $value);

            case 'rankmath':
                return update_post_meta($post_id, 'rank_math_title', $value);

            case 'aioseo':
                return update_post_meta($post_id, '_aioseo_title', $value);

            default:
                return update_post_meta($post_id, 'instarank_meta_title', $value);
        }
    }

    /**
     * Update meta description based on active SEO plugin
     */
    private function update_meta_description($post_id, $value, $seo_plugin) {
        switch ($seo_plugin) {
            case 'yoast':
                return update_post_meta($post_id, '_yoast_wpseo_metadesc', $value);

            case 'rankmath':
                return update_post_meta($post_id, 'rank_math_description', $value);

            case 'aioseo':
                return update_post_meta($post_id, '_aioseo_description', $value);

            default:
                return update_post_meta($post_id, 'instarank_meta_description', $value);
        }
    }

    /**
     * Update focus keyword based on active SEO plugin
     */
    private function update_focus_keyword($post_id, $value, $seo_plugin) {
        switch ($seo_plugin) {
            case 'yoast':
                return update_post_meta($post_id, '_yoast_wpseo_focuskw', $value);

            case 'rankmath':
                return update_post_meta($post_id, 'rank_math_focus_keyword', $value);

            case 'aioseo':
                return update_post_meta($post_id, '_aioseo_keyphrases', json_encode([
                    ['keyphrase' => $value]
                ]));

            default:
                return update_post_meta($post_id, 'instarank_focus_keyword', $value);
        }
    }

    /**
     * Update post title
     */
    private function update_post_title($post_id, $value) {
        return wp_update_post([
            'ID' => $post_id,
            'post_title' => $value
        ]) !== 0;
    }

    /**
     * Update image alt text
     */
    private function update_image_alt($attachment_id, $value) {
        if (empty($attachment_id)) {
            return false;
        }

        return update_post_meta($attachment_id, '_wp_attachment_image_alt', $value);
    }

    /**
     * Update canonical URL based on active SEO plugin
     * Supports: canonical_points_to_redirect, canonical_https_to_http fixes
     */
    private function update_canonical($post_id, $value, $seo_plugin) {
        switch ($seo_plugin) {
            case 'yoast':
                return update_post_meta($post_id, '_yoast_wpseo_canonical', $value);

            case 'rankmath':
                return update_post_meta($post_id, 'rank_math_canonical_url', $value);

            case 'aioseo':
                return update_post_meta($post_id, '_aioseo_canonical_url', $value);

            default:
                return update_post_meta($post_id, 'instarank_canonical', $value);
        }
    }

    /**
     * Update Open Graph title based on active SEO plugin
     * Supports: open_graph_incomplete fix
     */
    private function update_og_title($post_id, $value, $seo_plugin) {
        switch ($seo_plugin) {
            case 'yoast':
                return update_post_meta($post_id, '_yoast_wpseo_opengraph-title', $value);

            case 'rankmath':
                return update_post_meta($post_id, 'rank_math_facebook_title', $value);

            case 'aioseo':
                return update_post_meta($post_id, '_aioseo_og_title', $value);

            default:
                return update_post_meta($post_id, 'instarank_og_title', $value);
        }
    }

    /**
     * Update Open Graph description based on active SEO plugin
     * Supports: open_graph_incomplete fix
     */
    private function update_og_description($post_id, $value, $seo_plugin) {
        switch ($seo_plugin) {
            case 'yoast':
                return update_post_meta($post_id, '_yoast_wpseo_opengraph-description', $value);

            case 'rankmath':
                return update_post_meta($post_id, 'rank_math_facebook_description', $value);

            case 'aioseo':
                return update_post_meta($post_id, '_aioseo_og_description', $value);

            default:
                return update_post_meta($post_id, 'instarank_og_description', $value);
        }
    }

    /**
     * Update Open Graph image based on active SEO plugin
     * Supports: open_graph_incomplete fix
     */
    private function update_og_image($post_id, $value, $seo_plugin) {
        switch ($seo_plugin) {
            case 'yoast':
                return update_post_meta($post_id, '_yoast_wpseo_opengraph-image', $value);

            case 'rankmath':
                return update_post_meta($post_id, 'rank_math_facebook_image', $value);

            case 'aioseo':
                return update_post_meta($post_id, '_aioseo_og_image_url', $value);

            default:
                return update_post_meta($post_id, 'instarank_og_image', $value);
        }
    }

    /**
     * Update Open Graph URL based on active SEO plugin
     * Supports: open_graph_url_mismatch fix
     */
    private function update_og_url($post_id, $value, $seo_plugin) {
        switch ($seo_plugin) {
            case 'yoast':
                return update_post_meta($post_id, '_yoast_wpseo_opengraph-url', $value);

            case 'rankmath':
                return update_post_meta($post_id, 'rank_math_facebook_url', $value);

            case 'aioseo':
                return update_post_meta($post_id, '_aioseo_og_url', $value);

            default:
                return update_post_meta($post_id, 'instarank_og_url', $value);
        }
    }

    /**
     * Update robots noindex setting based on active SEO plugin
     * Supports: not_indexable fix
     */
    private function update_robots_noindex($post_id, $value, $seo_plugin) {
        // Value should be 0 (indexable) or 1 (noindex)
        $noindex_value = ($value === '1' || $value === 1 || $value === true) ? '1' : '0';

        switch ($seo_plugin) {
            case 'yoast':
                return update_post_meta($post_id, '_yoast_wpseo_meta-robots-noindex', $noindex_value);

            case 'rankmath':
                // Rank Math uses array format
                $robots = get_post_meta($post_id, 'rank_math_robots', true);
                if (!is_array($robots)) {
                    $robots = [];
                }
                if ($noindex_value === '1') {
                    $robots['index'] = 'noindex';
                } else {
                    $robots['index'] = 'index';
                }
                return update_post_meta($post_id, 'rank_math_robots', $robots);

            case 'aioseo':
                return update_post_meta($post_id, '_aioseo_robots_noindex', $noindex_value);

            default:
                return update_post_meta($post_id, 'instarank_robots_noindex', $noindex_value);
        }
    }

    /**
     * Update post content
     * Supports: low_word_count fix
     */
    private function update_post_content($post_id, $value) {
        return wp_update_post([
            'ID' => $post_id,
            'post_content' => $value
        ]) !== 0;
    }

    /**
     * Reject a change
     */
    public function reject_change($change_id) {
        global $wpdb;

        $table = $wpdb->prefix . 'instarank_changes';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table update
        $result = $wpdb->update($table, [
            'status' => 'rejected',
            'applied_at' => current_time('mysql')
        ], [
            'id' => $change_id,
            'status' => 'pending'
        ]);

        if ($result) {
            // Send webhook notification
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter
            $change = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM $table WHERE id = %d", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                $change_id
            ), ARRAY_A);

            $webhook = new InstaRank_Webhook_Sender();
            $webhook->send_change_rejected([
                'change_id' => $change_id,
                'post_id' => $change['post_id'],
                'change_type' => $change['change_type']
            ]);

            return true;
        }

        return false;
    }

    /**
     * Rollback a change
     */
    public function rollback_change($change_id) {
        global $wpdb;

        $table = $wpdb->prefix . 'instarank_changes';

        // Get the change
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $change = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d AND status = 'applied'", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $change_id
        ), ARRAY_A);

        if (!$change) {
            return false;
        }

        // Create a new change record with old and new values swapped
        $rollback_change = [
            'post_id' => $change['post_id'],
            'change_type' => $change['change_type'],
            'old_value' => $change['new_value'],
            'new_value' => $change['old_value']
        ];

        // Apply the rollback
        $applied = $this->apply_change($rollback_change);

        if ($applied) {
            // Update original change status
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table update
            $wpdb->update($table, [
                'status' => 'rolled_back',
                'rolled_back_at' => current_time('mysql')
            ], ['id' => $change_id]);

            // Send webhook notification
            $webhook = new InstaRank_Webhook_Sender();
            $webhook->send_change_rolled_back([
                'change_id' => $change_id,
                'post_id' => $change['post_id'],
                'change_type' => $change['change_type']
            ]);

            return true;
        }

        return false;
    }

    /**
     * Clean old change records
     */
    public function clean_old_changes() {
        global $wpdb;

        $table = $wpdb->prefix . 'instarank_changes';
        $retention_days = intval(get_option('instarank_rollback_days', 90));
        $cutoff_date = gmdate('Y-m-d H:i:s', strtotime("-$retention_days days"));

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $wpdb->query($wpdb->prepare(
            "DELETE FROM $table WHERE created_at < %s AND status IN ('applied', 'rejected', 'rolled_back')", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $cutoff_date
        ));
    }

    /**
     * Clear all change history
     */
    public function clear_all_changes() {
        global $wpdb;

        $table = $wpdb->prefix . 'instarank_changes';

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name is safe, uses $wpdb->prefix
        return $wpdb->query("TRUNCATE TABLE $table");
    }
}
