<?php
/**
 * Breadcrumbs System
 *
 * Generates breadcrumb navigation with BreadcrumbList JSON-LD schema
 *
 * @package InstaRank
 * @since 1.4.0
 */

defined('ABSPATH') || exit;

class InstaRank_Breadcrumbs {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Breadcrumb items
     */
    private $items = [];

    /**
     * Settings
     */
    private $settings = [];

    /**
     * Get singleton instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        // Load settings
        $this->settings = [
            'separator' => get_option('instarank_breadcrumb_separator', '›'),
            'home_text' => get_option('instarank_breadcrumb_home_text', 'Home'),
            'show_home' => get_option('instarank_breadcrumb_show_home', true),
            'show_current' => get_option('instarank_breadcrumb_show_current', true),
            'schema_enabled' => get_option('instarank_breadcrumb_schema_enabled', true),
        ];

        // Register shortcode
        add_shortcode('instarank_breadcrumbs', [$this, 'shortcode']);

        // Auto-inject schema if enabled
        if ($this->settings['schema_enabled']) {
            add_action('wp_head', [$this, 'output_schema'], 1);
        }
    }

    /**
     * Generate breadcrumbs
     */
    public function generate() {
        $this->items = [];

        // Always add home
        if ($this->settings['show_home']) {
            $this->add_item([
                'url' => home_url('/'),
                'text' => $this->settings['home_text']
            ]);
        }

        // Single post
        if (is_singular()) {
            $this->generate_single_breadcrumbs();
        }
        // Archive pages
        elseif (is_archive()) {
            $this->generate_archive_breadcrumbs();
        }
        // Search
        elseif (is_search()) {
            $this->add_item([
                'text' => 'Search Results'
            ]);
        }
        // 404
        elseif (is_404()) {
            $this->add_item([
                'text' => '404 Not Found'
            ]);
        }

        return $this->items;
    }

    /**
     * Generate breadcrumbs for single posts/pages
     */
    private function generate_single_breadcrumbs() {
        global $post;

        if (!$post) {
            return;
        }

        $post_type = get_post_type($post);
        $post_type_object = get_post_type_object($post_type);

        // Handle pages with hierarchy
        if ($post_type === 'page') {
            $this->add_parent_pages($post);
        }
        // Handle posts with category
        elseif ($post_type === 'post') {
            // Get primary category or first category
            $categories = get_the_category($post->ID);
            if (!empty($categories)) {
                $category = $categories[0];
                $this->add_term_hierarchy($category);
            }
        }
        // Handle custom post types
        else {
            // Add post type archive if it has one
            if ($post_type_object && $post_type_object->has_archive) {
                $this->add_item([
                    'url' => get_post_type_archive_link($post_type),
                    'text' => $post_type_object->labels->name
                ]);
            }

            // Add taxonomy terms if applicable
            $taxonomies = get_object_taxonomies($post_type, 'objects');
            foreach ($taxonomies as $taxonomy) {
                if ($taxonomy->public) {
                    $terms = get_the_terms($post->ID, $taxonomy->name);
                    if (!empty($terms) && !is_wp_error($terms)) {
                        $this->add_term_hierarchy($terms[0]);
                        break; // Only add first taxonomy
                    }
                }
            }
        }

        // Add current post/page
        if ($this->settings['show_current']) {
            $this->add_item([
                'text' => get_the_title($post)
            ]);
        }
    }

    /**
     * Add parent pages to breadcrumbs
     */
    private function add_parent_pages($post) {
        $parents = [];
        $parent_id = $post->post_parent;

        while ($parent_id) {
            $parent = get_post($parent_id);
            $parents[] = [
                'url' => get_permalink($parent),
                'text' => get_the_title($parent)
            ];
            $parent_id = $parent->post_parent;
        }

        // Reverse to get correct order
        $parents = array_reverse($parents);
        foreach ($parents as $parent) {
            $this->add_item($parent);
        }
    }

    /**
     * Add term hierarchy to breadcrumbs
     */
    private function add_term_hierarchy($term) {
        if (!$term || is_wp_error($term)) {
            return;
        }

        $parents = [];
        $parent_id = $term->parent;

        while ($parent_id) {
            $parent = get_term($parent_id, $term->taxonomy);
            if (!is_wp_error($parent)) {
                $parents[] = [
                    'url' => get_term_link($parent),
                    'text' => $parent->name
                ];
                $parent_id = $parent->parent;
            } else {
                break;
            }
        }

        // Reverse to get correct order
        $parents = array_reverse($parents);
        foreach ($parents as $parent) {
            $this->add_item($parent);
        }

        // Add current term
        $this->add_item([
            'url' => get_term_link($term),
            'text' => $term->name
        ]);
    }

    /**
     * Generate archive breadcrumbs
     */
    private function generate_archive_breadcrumbs() {
        // Category archive
        if (is_category()) {
            $category = get_queried_object();
            $this->add_term_hierarchy($category);
        }
        // Tag archive
        elseif (is_tag()) {
            $tag = get_queried_object();
            $this->add_item([
                'text' => $tag->name
            ]);
        }
        // Taxonomy archive
        elseif (is_tax()) {
            $term = get_queried_object();
            $this->add_term_hierarchy($term);
        }
        // Post type archive
        elseif (is_post_type_archive()) {
            $post_type = get_queried_object();
            $this->add_item([
                'text' => $post_type->labels->name
            ]);
        }
        // Author archive
        elseif (is_author()) {
            $author = get_queried_object();
            $this->add_item([
                'text' => 'Author: ' . $author->display_name
            ]);
        }
        // Date archive
        elseif (is_date()) {
            if (is_year()) {
                $this->add_item([
                    'text' => get_the_date('Y')
                ]);
            } elseif (is_month()) {
                $this->add_item([
                    'url' => get_year_link(get_the_date('Y')),
                    'text' => get_the_date('Y')
                ]);
                $this->add_item([
                    'text' => get_the_date('F')
                ]);
            } elseif (is_day()) {
                $this->add_item([
                    'url' => get_year_link(get_the_date('Y')),
                    'text' => get_the_date('Y')
                ]);
                $this->add_item([
                    'url' => get_month_link(get_the_date('Y'), get_the_date('m')),
                    'text' => get_the_date('F')
                ]);
                $this->add_item([
                    'text' => get_the_date('j')
                ]);
            }
        }
    }

    /**
     * Add item to breadcrumbs
     */
    private function add_item($item) {
        $this->items[] = array_merge([
            'url' => '',
            'text' => ''
        ], $item);
    }

    /**
     * Render breadcrumbs HTML
     */
    public function render($args = []) {
        // Merge with defaults
        $args = array_merge([
            'wrapper_class' => 'instarank-breadcrumbs',
            'separator' => $this->settings['separator'],
            'before' => '',
            'after' => ''
        ], $args);

        // Generate breadcrumbs
        $items = $this->generate();

        if (empty($items)) {
            return '';
        }

        $output = $args['before'];
        $output .= '<nav class="' . esc_attr($args['wrapper_class']) . '" aria-label="Breadcrumb">';
        $output .= '<ol class="breadcrumb-list">';

        $count = count($items);
        foreach ($items as $index => $item) {
            $is_last = ($index === $count - 1);

            $output .= '<li class="breadcrumb-item' . ($is_last ? ' active' : '') . '">';

            if (!empty($item['url']) && !$is_last) {
                $output .= '<a href="' . esc_url($item['url']) . '">';
                $output .= esc_html($item['text']);
                $output .= '</a>';
            } else {
                $output .= '<span>' . esc_html($item['text']) . '</span>';
            }

            $output .= '</li>';

            // Add separator
            if (!$is_last) {
                $output .= '<li class="breadcrumb-separator" aria-hidden="true">';
                $output .= esc_html($args['separator']);
                $output .= '</li>';
            }
        }

        $output .= '</ol>';
        $output .= '</nav>';
        $output .= $args['after'];

        return $output;
    }

    /**
     * Generate BreadcrumbList JSON-LD schema
     */
    public function get_schema() {
        $items = $this->generate();

        if (empty($items)) {
            return null;
        }

        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'BreadcrumbList',
            'itemListElement' => []
        ];

        foreach ($items as $index => $item) {
            $list_item = [
                '@type' => 'ListItem',
                'position' => $index + 1,
                'name' => $item['text']
            ];

            // Add URL if available
            if (!empty($item['url'])) {
                $list_item['item'] = $item['url'];
            }

            $schema['itemListElement'][] = $list_item;
        }

        return $schema;
    }

    /**
     * Output schema in wp_head
     */
    public function output_schema() {
        // Don't output on admin, feeds, or if not a valid page type
        if (is_admin() || is_feed() || is_front_page()) {
            return;
        }

        $schema = $this->get_schema();

        if ($schema) {
            echo '<script type="application/ld+json">' . "\n";
            echo wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
            echo "\n" . '</script>' . "\n";
        }
    }

    /**
     * Shortcode handler
     */
    public function shortcode($atts) {
        $atts = shortcode_atts([
            'separator' => $this->settings['separator'],
            'class' => 'instarank-breadcrumbs'
        ], $atts);

        return $this->render([
            'wrapper_class' => $atts['class'],
            'separator' => $atts['separator']
        ]);
    }

    /**
     * Template function wrapper
     */
    public function display($args = []) {
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- render() method returns escaped HTML
        echo $this->render($args);
    }
}

/**
 * Template function for themes
 */
function instarank_breadcrumbs($args = []) {
    $breadcrumbs = InstaRank_Breadcrumbs::instance();
    $breadcrumbs->display($args);
}

// Initialize
InstaRank_Breadcrumbs::instance();
