<?php
/**
 * Authentication Manager
 * Handles API key generation, verification, and security
 */

defined('ABSPATH') || exit;

class InstaRank_Auth_Manager {

    private $api_key_option = 'instarank_api_key';
    private $failed_attempts_option = 'instarank_failed_attempts';
    private $lockout_time_option = 'instarank_lockout_time';
    private $max_failed_attempts = 5;
    private $lockout_duration = 900; // 15 minutes in seconds

    /**
     * Generate a new API key
     */
    public function generate_api_key() {
        $api_key = 'wp_' . bin2hex(random_bytes(32));
        update_option($this->api_key_option, $api_key);
        
        // Reset failed attempts when generating new key
        delete_option($this->failed_attempts_option);
        delete_option($this->lockout_time_option);

        return $api_key;
    }

    /**
     * Get the current API key (generate if doesn't exist)
     */
    public function get_current_api_key() {
        $api_key = get_option($this->api_key_option);

        if (empty($api_key)) {
            $api_key = $this->generate_api_key();
        }

        return $api_key;
    }

    /**
     * Verify API key
     */
    public function verify_api_key($provided_key) {
        // Check if account is locked out
        if ($this->is_locked_out()) {
            $this->log_auth_event('locked_out', $provided_key);
            return false;
        }

        $stored_key = $this->get_current_api_key();

        if (hash_equals($stored_key, $provided_key)) {
            // Successful authentication - reset failed attempts
            delete_option($this->failed_attempts_option);
            delete_option($this->lockout_time_option);
            $this->log_auth_event('success', $provided_key);
            return true;
        }

        // Failed authentication - increment counter
        $this->increment_failed_attempts();
        $this->log_auth_event('failed', $provided_key);
        return false;
    }

    /**
     * Check if account is locked out due to too many failed attempts
     */
    public function is_locked_out() {
        $lockout_time = get_option($this->lockout_time_option);

        if (!$lockout_time) {
            return false;
        }

        // Check if lockout period has expired
        if (current_time('timestamp') > $lockout_time) {
            delete_option($this->failed_attempts_option);
            delete_option($this->lockout_time_option);
            return false;
        }

        return true;
    }

    /**
     * Increment failed authentication attempts
     */
    private function increment_failed_attempts() {
        $failed_attempts = intval(get_option($this->failed_attempts_option, 0));
        $failed_attempts++;

        update_option($this->failed_attempts_option, $failed_attempts);

        // Lock out if too many failed attempts
        if ($failed_attempts >= $this->max_failed_attempts) {
            $lockout_until = current_time('timestamp') + $this->lockout_duration;
            update_option($this->lockout_time_option, $lockout_until);

            // Send webhook notification
            $webhook = new InstaRank_Webhook_Sender();
            $webhook->send_security_alert([
                'type' => 'lockout',
                'failed_attempts' => $failed_attempts,
                'lockout_until' => gmdate('Y-m-d H:i:s', $lockout_until)
            ]);
        }
    }

    /**
     * Get remaining lockout time in seconds
     */
    public function get_lockout_remaining() {
        $lockout_time = get_option($this->lockout_time_option);

        if (!$lockout_time) {
            return 0;
        }

        $remaining = $lockout_time - current_time('timestamp');
        return max(0, $remaining);
    }

    /**
     * Reset failed attempts counter (admin action)
     */
    public function reset_failed_attempts() {
        delete_option($this->failed_attempts_option);
        delete_option($this->lockout_time_option);
        return true;
    }

    /**
     * Get failed attempts count
     */
    public function get_failed_attempts() {
        return intval(get_option($this->failed_attempts_option, 0));
    }

    /**
     * Log authentication event
     */
    private function log_auth_event($event_type, $provided_key = '') {
        global $wpdb;

        $table = $wpdb->prefix . 'instarank_auth_log';

        // Create table if it doesn't exist
        $charset_collate = $wpdb->get_charset_collate();
        $sql = "CREATE TABLE IF NOT EXISTS $table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            event_type varchar(50) NOT NULL,
            ip_address varchar(100) DEFAULT NULL,
            user_agent text DEFAULT NULL,
            api_key_preview varchar(20) DEFAULT NULL,
            timestamp datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY event_type (event_type),
            KEY timestamp (timestamp)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);

        // Log the event
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom auth log table
        $wpdb->insert($table, [
            'event_type' => $event_type,
            'ip_address' => $this->get_client_ip(),
            'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '',
            'api_key_preview' => substr($provided_key, 0, 10) . '...',
            'timestamp' => current_time('mysql')
        ]);
    }

    /**
     * Get client IP address
     */
    private function get_client_ip() {
        $ip_keys = [
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'
        ];

        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER)) {
                $ip = explode(',', sanitize_text_field(wp_unslash($_SERVER[$key])))[0];
                $ip = trim($ip);

                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }

        return 'unknown';
    }

    /**
     * Get recent authentication logs
     */
    public function get_auth_logs($limit = 50) {
        global $wpdb;

        $table = $wpdb->prefix . 'instarank_auth_log';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table ORDER BY timestamp DESC LIMIT %d", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $limit
        ), ARRAY_A);
    }

    /**
     * Clean old auth logs (keep last 90 days)
     */
    public function clean_old_logs() {
        global $wpdb;

        $table = $wpdb->prefix . 'instarank_auth_log';
        $days_ago = gmdate('Y-m-d H:i:s', strtotime('-90 days'));

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $wpdb->query($wpdb->prepare(
            "DELETE FROM $table WHERE timestamp < %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $days_ago
        ));
    }

    /**
     * Regenerate API key
     */
    public function regenerate_api_key() {
        // Disconnect current integration
        update_option('instarank_connected', false);
        update_option('instarank_project_id', '');

        // Generate new key
        $new_key = $this->generate_api_key();

        // Send webhook notification
        $webhook = new InstaRank_Webhook_Sender();
        $webhook->send_security_alert([
            'type' => 'api_key_regenerated',
            'timestamp' => current_time('mysql')
        ]);

        return $new_key;
    }
}
