<?php
/**
 * API Handler
 * Handles communication with InstaRank platform
 */

defined('ABSPATH') || exit;

class InstaRank_API_Handler {

    private $api_url;
    private $api_key;

    public function __construct() {
        $this->api_url = INSTARANK_API_URL;
        $this->api_key = get_option('instarank_api_key', '');
    }

    /**
     * Make API request to InstaRank platform
     */
    private function make_request($endpoint, $method = 'GET', $data = []) {
        $url = trailingslashit($this->api_url) . ltrim($endpoint, '/');

        $args = [
            'method' => $method,
            'headers' => [
                'Content-Type' => 'application/json',
                'X-WordPress-API-Key' => $this->api_key,
                'X-WordPress-Site' => get_site_url()
            ],
            'timeout' => 30
        ];

        if ($method !== 'GET' && !empty($data)) {
            $args['body'] = json_encode($data);
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            return [
                'success' => false,
                'error' => $response->get_error_message()
            ];
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $decoded = json_decode($body, true);

        if ($status_code >= 200 && $status_code < 300) {
            return [
                'success' => true,
                'data' => $decoded
            ];
        }

        return [
            'success' => false,
            'error' => $decoded['error'] ?? 'Unknown error',
            'status_code' => $status_code
        ];
    }

    /**
     * Register this WordPress site with InstaRank
     */
    public function register_site($project_id) {
        $detector = new InstaRank_SEO_Detector();

        $data = [
            'project_id' => $project_id,
            'site_url' => get_site_url(),
            'site_name' => get_bloginfo('name'),
            'wp_version' => get_bloginfo('version'),
            'plugin_version' => INSTARANK_VERSION,
            'seo_plugin' => $detector->get_active_seo_plugin(),
            'api_key' => $this->api_key
        ];

        $result = $this->make_request('integrations/wordpress/register', 'POST', $data);

        if ($result['success']) {
            update_option('instarank_connected', true);
            update_option('instarank_project_id', $project_id);
            update_option('instarank_connected_at', current_time('mysql'));

            // Send webhook
            $webhook = new InstaRank_Webhook_Sender();
            $webhook->send_connection_established($project_id);
        }

        return $result;
    }

    /**
     * Test connection to InstaRank platform
     */
    public function test_connection() {
        return $this->make_request('integrations/wordpress/test', 'GET');
    }

    /**
     * Sync posts to InstaRank platform
     */
    public function sync_posts() {
        $args = [
            'post_type' => ['post', 'page'],
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'orderby' => 'modified',
            'order' => 'DESC'
        ];

        $query = new WP_Query($args);
        $detector = new InstaRank_SEO_Detector();
        $posts = [];

        foreach ($query->posts as $post) {
            $meta = $detector->get_post_meta($post->ID);

            $posts[] = [
                'id' => $post->ID,
                'title' => $post->post_title,
                'url' => get_permalink($post->ID),
                'type' => $post->post_type,
                'modified' => $post->post_modified_gmt,
                'seo' => [
                    'meta_title' => $meta['title'] ?? '',
                    'meta_description' => $meta['description'] ?? '',
                    'focus_keyword' => $meta['focus_keyword'] ?? ''
                ]
            ];
        }

        $data = [
            'posts' => $posts,
            'total' => count($posts)
        ];

        $result = $this->make_request('integrations/wordpress/sync', 'POST', $data);

        if ($result['success']) {
            update_option('instarank_last_sync', current_time('mysql'));

            // Send webhook
            $webhook = new InstaRank_Webhook_Sender();
            $webhook->send_sync_completed([
                'posts' => count($posts),
                'duration' => 0
            ]);
        }

        return $result;
    }

    /**
     * Report change status back to InstaRank
     */
    public function report_change_status($change_id, $status, $error = null) {
        $data = [
            'change_id' => $change_id,
            'status' => $status,
            'timestamp' => current_time('mysql')
        ];

        if ($error) {
            $data['error'] = $error;
        }

        return $this->make_request('integrations/wordpress/change-status', 'POST', $data);
    }

    /**
     * Get pending tasks from InstaRank
     */
    public function get_pending_tasks() {
        return $this->make_request('integrations/wordpress/tasks', 'GET');
    }

    /**
     * Send health check to InstaRank
     */
    public function send_health_check() {
        $detector = new InstaRank_SEO_Detector();

        $data = [
            'status' => 'healthy',
            'plugin_version' => INSTARANK_VERSION,
            'wp_version' => get_bloginfo('version'),
            'seo_plugin' => $detector->get_active_seo_plugin(),
            'post_count' => wp_count_posts()->publish ?? 0,
            'last_sync' => get_option('instarank_last_sync', null)
        ];

        return $this->make_request('integrations/wordpress/health', 'POST', $data);
    }

    /**
     * Disconnect from InstaRank
     */
    public function disconnect() {
        $result = $this->make_request('integrations/wordpress/disconnect', 'POST', [
            'site_url' => get_site_url()
        ]);

        if ($result['success']) {
            update_option('instarank_connected', false);
            update_option('instarank_project_id', '');
        }

        return $result;
    }

    /**
     * Get integration status
     */
    public function get_integration_status() {
        return $this->make_request('integrations/wordpress/status', 'GET');
    }
}
