<?php
/**
 * InstaRank ACF Detector
 *
 * Detects Advanced Custom Fields (ACF) field groups and fields
 * for programmatic SEO mapping
 *
 * @package InstaRank
 * @since 2.1.0
 */

defined('ABSPATH') || exit;

class InstaRank_ACF_Detector {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * ACF field type mappings to standard types
     */
    private $type_mappings = [
        // Text types
        'text' => 'text',
        'textarea' => 'textarea',
        'wysiwyg' => 'wysiwyg',
        'oembed' => 'text',
        'password' => 'text',

        // Number types
        'number' => 'number',
        'range' => 'number',

        // Email/URL types
        'email' => 'email',
        'url' => 'url',
        'link' => 'url',
        'page_link' => 'url',

        // Media types
        'image' => 'image',
        'file' => 'file',
        'gallery' => 'image',

        // Choice types
        'select' => 'select',
        'radio' => 'select',
        'button_group' => 'select',
        'checkbox' => 'checkbox',
        'true_false' => 'checkbox',

        // Date types
        'date_picker' => 'date',
        'date_time_picker' => 'datetime',
        'time_picker' => 'time',

        // Other types
        'color_picker' => 'color',

        // Relational types (store IDs as text)
        'post_object' => 'text',
        'relationship' => 'text',
        'taxonomy' => 'text',
        'user' => 'text',

        // Complex types (unsupported but shown)
        'repeater' => 'repeater',
        'flexible_content' => 'flexible_content',
        'group' => 'group',
        'clone' => 'group',
    ];

    /**
     * Unsupported complex field types
     */
    private $unsupported_types = ['repeater', 'flexible_content', 'group', 'clone'];

    /**
     * Get singleton instance
     */
    public static function instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Check if ACF plugin is active
     *
     * @return bool True if ACF is active
     */
    public function is_acf_active() {
        // Check for ACF Pro or Free
        return function_exists('acf_get_field_groups') &&
               (class_exists('ACF') || defined('ACF_VERSION'));
    }

    /**
     * Get ACF version info
     *
     * @return array ACF version information
     */
    public function get_acf_info() {
        if (!$this->is_acf_active()) {
            return [
                'active' => false,
                'version' => null,
                'pro' => false
            ];
        }

        return [
            'active' => true,
            'version' => defined('ACF_VERSION') ? ACF_VERSION : 'unknown',
            'pro' => defined('ACF_PRO') && ACF_PRO
        ];
    }

    /**
     * Get all ACF field groups
     *
     * @param string|null $post_type Optional: filter by post type location rule
     * @return array Field groups with their fields
     */
    public function get_all_field_groups($post_type = null) {
        if (!$this->is_acf_active()) {
            return [];
        }

        // Get all field groups
        $field_groups = acf_get_field_groups();
        $result = [];

        foreach ($field_groups as $group) {
            // If post_type filter is specified, check location rules
            if ($post_type !== null && !$this->group_applies_to_post_type($group, $post_type)) {
                continue;
            }

            $group_data = [
                'key' => $group['key'],
                'title' => $group['title'],
                'active' => $group['active'],
                'menu_order' => $group['menu_order'],
                'location_rules' => $group['location'],
                'fields' => $this->get_fields_for_group($group['key'])
            ];

            $result[] = $group_data;
        }

        return $result;
    }

    /**
     * Check if a field group applies to a specific post type
     *
     * @param array $group Field group data
     * @param string $post_type Post type to check
     * @return bool True if group applies to post type
     */
    private function group_applies_to_post_type($group, $post_type) {
        if (empty($group['location']) || !is_array($group['location'])) {
            return false;
        }

        // ACF location rules are structured as AND groups containing OR conditions
        // [[{param: 'post_type', operator: '==', value: 'page'}]]
        foreach ($group['location'] as $and_group) {
            foreach ($and_group as $rule) {
                if ($rule['param'] === 'post_type') {
                    if ($rule['operator'] === '==' && $rule['value'] === $post_type) {
                        return true;
                    }
                    if ($rule['operator'] === '!=' && $rule['value'] !== $post_type) {
                        return true;
                    }
                }
            }
        }

        return false;
    }

    /**
     * Get fields for a specific field group
     *
     * @param string $group_key Field group key
     * @return array Fields in the group
     */
    public function get_fields_for_group($group_key) {
        if (!$this->is_acf_active()) {
            return [];
        }

        $fields = acf_get_fields($group_key);

        if (!$fields || !is_array($fields)) {
            return [];
        }

        $result = [];
        foreach ($fields as $field) {
            $result[] = $this->normalize_acf_field($field, $group_key);
        }

        return $result;
    }

    /**
     * Normalize ACF field to standard format
     *
     * @param array $field ACF field data
     * @param string $group_key Parent group key
     * @return array Normalized field
     */
    private function normalize_acf_field($field, $group_key) {
        $acf_type = $field['type'];
        $mapped_type = $this->type_mappings[$acf_type] ?? 'text';
        $is_unsupported = in_array($acf_type, $this->unsupported_types);

        $normalized = [
            'field_name' => $field['name'],
            'field_key' => $field['key'],
            'field_label' => $field['label'],
            'field_type' => $mapped_type,
            'acf_field_type' => $acf_type,
            'required' => !empty($field['required']),
            'default_value' => $field['default_value'] ?? null,
            'instructions' => $field['instructions'] ?? null,
            'placeholder' => $field['placeholder'] ?? null,
            'group_key' => $group_key,
            'is_supported' => !$is_unsupported,
            'locations' => ["acf/{$group_key}/{$field['name']}"],
            'page_builder' => 'acf',
            'block_type' => 'acf_field',
            'property' => $field['name']
        ];

        // Add choices for select/radio/checkbox fields
        if (isset($field['choices']) && is_array($field['choices'])) {
            $normalized['choices'] = [];
            foreach ($field['choices'] as $value => $label) {
                $normalized['choices'][] = [
                    'value' => $value,
                    'label' => $label
                ];
            }
        }

        // Add min/max for number/range fields
        if (in_array($acf_type, ['number', 'range'])) {
            if (isset($field['min'])) $normalized['min'] = $field['min'];
            if (isset($field['max'])) $normalized['max'] = $field['max'];
            if (isset($field['step'])) $normalized['step'] = $field['step'];
        }

        // Add return format info for image/file
        if (in_array($acf_type, ['image', 'file', 'gallery'])) {
            $normalized['return_format'] = $field['return_format'] ?? 'array';
        }

        // Add sub_fields info for complex types (for reference)
        if ($is_unsupported && isset($field['sub_fields'])) {
            $normalized['sub_field_count'] = count($field['sub_fields']);
        }

        return $normalized;
    }

    /**
     * Get all ACF fields across all groups (flattened)
     *
     * @param string|null $post_type Optional: filter by post type
     * @return array All fields
     */
    public function get_all_fields($post_type = null) {
        $groups = $this->get_all_field_groups($post_type);
        $all_fields = [];

        foreach ($groups as $group) {
            foreach ($group['fields'] as $field) {
                // Add group info to each field
                $field['group_title'] = $group['title'];
                $all_fields[] = $field;
            }
        }

        return $all_fields;
    }

    /**
     * Check if a field name is an ACF field
     *
     * @param string $field_name Field name to check
     * @param string|null $post_type Optional: limit search to post type
     * @return bool|array False if not ACF field, field data if it is
     */
    public function is_acf_field($field_name, $post_type = null) {
        if (!$this->is_acf_active()) {
            return false;
        }

        $all_fields = $this->get_all_fields($post_type);

        foreach ($all_fields as $field) {
            if ($field['field_name'] === $field_name) {
                return $field;
            }
        }

        return false;
    }

    /**
     * Get ACF field value for a post
     *
     * @param string $field_name Field name
     * @param int $post_id Post ID
     * @return mixed Field value or null
     */
    public function get_field_value($field_name, $post_id) {
        if (!$this->is_acf_active() || !function_exists('get_field')) {
            return null;
        }

        return get_field($field_name, $post_id);
    }

    /**
     * Update ACF field value for a post
     *
     * @param string $field_name Field name
     * @param mixed $value Value to set
     * @param int $post_id Post ID
     * @return bool Success status
     */
    public function update_field_value($field_name, $value, $post_id) {
        if (!$this->is_acf_active() || !function_exists('update_field')) {
            // Fallback to update_post_meta
            return update_post_meta($post_id, $field_name, $value);
        }

        return update_field($field_name, $value, $post_id);
    }

    /**
     * Get complete ACF detection result
     *
     * @param string|null $post_type Optional: filter by post type
     * @return array Complete ACF status and fields
     */
    public function detect($post_type = null) {
        $acf_info = $this->get_acf_info();

        if (!$acf_info['active']) {
            return [
                'success' => true,
                'acf_active' => false,
                'acf_version' => null,
                'acf_pro' => false,
                'post_type_filter' => $post_type,
                'field_groups' => [],
                'all_fields' => [],
                'stats' => [
                    'total_groups' => 0,
                    'total_fields' => 0,
                    'supported_fields' => 0,
                    'unsupported_fields' => 0
                ]
            ];
        }

        $field_groups = $this->get_all_field_groups($post_type);
        $all_fields = [];
        $supported_count = 0;
        $unsupported_count = 0;

        foreach ($field_groups as $group) {
            foreach ($group['fields'] as $field) {
                $field['group_title'] = $group['title'];
                $all_fields[] = $field;

                if ($field['is_supported']) {
                    $supported_count++;
                } else {
                    $unsupported_count++;
                }
            }
        }

        return [
            'success' => true,
            'acf_active' => true,
            'acf_version' => $acf_info['version'],
            'acf_pro' => $acf_info['pro'],
            'post_type_filter' => $post_type,
            'field_groups' => $field_groups,
            'all_fields' => $all_fields,
            'stats' => [
                'total_groups' => count($field_groups),
                'total_fields' => count($all_fields),
                'supported_fields' => $supported_count,
                'unsupported_fields' => $unsupported_count
            ]
        ];
    }
}
