/**
 * InstaRank - Kadence Dynamic Fields Integration
 *
 * Enhances the Kadence Blocks Pro dynamic content interface by adding
 * InstaRank dataset fields as quick-select options in the custom field dropdown.
 */
(function() {
    'use strict';

    // Configuration from WordPress localized script
    const config = window.instarankKadence || {};
    const templateId = config.templateId || 0;
    const fieldMappings = config.fieldMappings || {};
    const datasetColumns = config.datasetColumns || [];
    const datasetName = config.datasetName || '';

    // Early exit if no field mappings configured
    if (Object.keys(fieldMappings).length === 0 && datasetColumns.length === 0) {
        console.log('[InstaRank] No field mappings or dataset columns found for this template');
        return;
    }

    console.log('[InstaRank] Kadence Dynamic Fields Integration loaded');
    console.log('[InstaRank] Template ID:', templateId);
    console.log('[InstaRank] Field mappings:', fieldMappings);
    console.log('[InstaRank] Dataset columns:', datasetColumns);

    /**
     * Create the InstaRank fields panel that appears in Kadence dynamic content settings
     */
    function createInstaRankFieldsPanel() {
        const panel = document.createElement('div');
        panel.className = 'instarank-fields-panel';
        panel.innerHTML = `
            <div class="instarank-fields-header" style="
                background: linear-gradient(135deg, #f97316 0%, #ea580c 100%);
                color: white;
                padding: 8px 12px;
                border-radius: 4px 4px 0 0;
                font-weight: 600;
                font-size: 12px;
                display: flex;
                align-items: center;
                gap: 6px;
            ">
                <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                    <path d="M12 2L2 7l10 5 10-5-10-5zM2 17l10 5 10-5M2 12l10 5 10-5"></path>
                </svg>
                InstaRank Dataset Fields
            </div>
            <div class="instarank-fields-body" style="
                background: #fff7ed;
                border: 1px solid #fed7aa;
                border-top: none;
                border-radius: 0 0 4px 4px;
                padding: 8px;
            ">
                <div class="instarank-dataset-info" style="
                    font-size: 11px;
                    color: #9a3412;
                    margin-bottom: 8px;
                    padding-bottom: 8px;
                    border-bottom: 1px dashed #fed7aa;
                ">
                    ${datasetName ? `Dataset: <strong>${datasetName}</strong>` : 'Configure mappings in InstaRank → Templates'}
                </div>
                <div class="instarank-fields-list" style="
                    display: flex;
                    flex-wrap: wrap;
                    gap: 4px;
                ">
                    <!-- Fields will be injected here -->
                </div>
            </div>
        `;
        return panel;
    }

    /**
     * Create a clickable field button
     */
    function createFieldButton(fieldName, datasetColumn, type = 'text') {
        const button = document.createElement('button');
        button.type = 'button';
        button.className = 'instarank-field-btn';
        button.dataset.field = fieldName;
        button.dataset.column = datasetColumn || fieldName;
        button.dataset.type = type;

        // Style the button
        button.style.cssText = `
            background: white;
            border: 1px solid #fdba74;
            border-radius: 3px;
            padding: 4px 8px;
            font-size: 11px;
            cursor: pointer;
            transition: all 0.15s ease;
            display: inline-flex;
            align-items: center;
            gap: 4px;
        `;

        // Icon based on type
        const icons = {
            text: '📝',
            image: '🖼️',
            url: '🔗',
            number: '#',
            date: '📅',
        };
        const icon = icons[type] || icons.text;

        button.innerHTML = `<span style="font-size: 10px;">${icon}</span> ${fieldName}`;

        // Hover effect
        button.addEventListener('mouseenter', () => {
            button.style.background = '#fff7ed';
            button.style.borderColor = '#f97316';
        });
        button.addEventListener('mouseleave', () => {
            button.style.background = 'white';
            button.style.borderColor = '#fdba74';
        });

        return button;
    }

    /**
     * Inject field into Kadence's custom field input
     * This handles the full flow: enable dynamic content -> select Post Custom Field -> enter field name
     */
    function injectFieldValue(fieldName) {
        console.log('[InstaRank] Attempting to inject field:', fieldName);

        // Step 1: Check if we need to click "Add Dynamic Content" button first
        const addDynamicBtn = findAddDynamicContentButton();
        if (addDynamicBtn) {
            console.log('[InstaRank] Found "Add Dynamic Content" button, clicking it first');
            addDynamicBtn.click();

            // Wait for UI to update, then continue with selecting Post Custom Field
            setTimeout(() => {
                selectPostCustomFieldAndInject(fieldName);
            }, 300);
            return;
        }

        // Step 2: If dynamic content is already enabled, select Post Custom Field
        selectPostCustomFieldAndInject(fieldName);
    }

    /**
     * Find the "Add Dynamic Content" button
     */
    function findAddDynamicContentButton() {
        const buttons = document.querySelectorAll('button');
        for (const btn of buttons) {
            if (btn.textContent.includes('Add Dynamic Content') ||
                btn.textContent.includes('Enable Dynamic')) {
                return btn;
            }
        }
        return null;
    }

    /**
     * Select "Post Custom Field" from dropdown and inject the field name
     */
    function selectPostCustomFieldAndInject(fieldName) {
        // Find the Content dropdown and click it to open
        const contentDropdown = findContentDropdown();

        if (contentDropdown) {
            console.log('[InstaRank] Found Content dropdown, clicking to open');
            contentDropdown.click();

            // Wait for dropdown to open, then select Post Custom Field
            setTimeout(() => {
                const postCustomFieldOption = findPostCustomFieldOption();
                if (postCustomFieldOption) {
                    console.log('[InstaRank] Selecting Post Custom Field option');
                    postCustomFieldOption.click();

                    // Wait for the custom field input to appear, then fill it
                    setTimeout(() => {
                        fillCustomFieldInput(fieldName);
                    }, 300);
                } else {
                    console.log('[InstaRank] Post Custom Field option not found');
                    copyToClipboardFallback(fieldName);
                }
            }, 200);
        } else {
            // Maybe Post Custom Field is already selected, try to find the input directly
            console.log('[InstaRank] No dropdown found, looking for input directly');
            fillCustomFieldInput(fieldName);
        }
    }

    /**
     * Find the Content dropdown button
     */
    function findContentDropdown() {
        // Look for CustomSelectControl buttons in the sidebar
        const sidebar = document.querySelector('.interface-interface-skeleton__sidebar');
        if (!sidebar) return null;

        // Find buttons that are part of a custom select control
        const selectButtons = sidebar.querySelectorAll('button[aria-haspopup="listbox"], .components-custom-select-control button, [class*="custom-select"] button');

        for (const btn of selectButtons) {
            // Check if this button is in a "Content" labeled control
            const control = btn.closest('.components-base-control');
            if (control) {
                const label = control.querySelector('label, .components-base-control__label');
                if (label && label.textContent.toLowerCase().includes('content')) {
                    return btn;
                }
            }

            // Also check by the button's current value
            if (btn.textContent.includes('None') ||
                btn.textContent.includes('Post Title') ||
                btn.textContent.includes('Post Custom Field') ||
                btn.textContent.includes('Post URL')) {
                return btn;
            }
        }

        return null;
    }

    /**
     * Find the "Post Custom Field" option in an open dropdown
     */
    function findPostCustomFieldOption() {
        // Look in popovers for the option
        const popovers = document.querySelectorAll('.components-popover__content');
        for (const popover of popovers) {
            const options = popover.querySelectorAll('[role="option"], button');
            for (const opt of options) {
                if (opt.textContent.trim() === 'Post Custom Field') {
                    return opt;
                }
            }
        }

        // Also check for options directly in the DOM
        const allOptions = document.querySelectorAll('[role="option"]');
        for (const opt of allOptions) {
            if (opt.textContent.trim() === 'Post Custom Field') {
                return opt;
            }
        }

        return null;
    }

    /**
     * Fill in the custom field name input
     */
    function fillCustomFieldInput(fieldName) {
        const sidebar = document.querySelector('.interface-interface-skeleton__sidebar');
        if (!sidebar) {
            copyToClipboardFallback(fieldName);
            return;
        }

        // Look for text inputs in the sidebar that could be the custom field input
        const inputs = sidebar.querySelectorAll('input[type="text"]:not([readonly]):not([disabled])');

        let targetInput = null;

        for (const input of inputs) {
            const rect = input.getBoundingClientRect();
            if (rect.width === 0 || rect.height === 0) continue;

            // Check if this input is for custom field key
            const control = input.closest('.components-base-control');
            if (control) {
                const labelText = control.textContent.toLowerCase();
                // Kadence labels this as "Custom Field Key" or similar
                if (labelText.includes('custom') ||
                    labelText.includes('field') ||
                    labelText.includes('key') ||
                    labelText.includes('meta')) {
                    targetInput = input;
                    break;
                }
            }

            // Also accept any visible input in the dynamic content area
            const panelBody = input.closest('.components-panel__body');
            if (panelBody && (panelBody.textContent.includes('Dynamic') ||
                             panelBody.textContent.includes('Content'))) {
                targetInput = input;
                break;
            }
        }

        if (targetInput) {
            // Set the value using native setter to trigger React
            const nativeInputValueSetter = Object.getOwnPropertyDescriptor(
                window.HTMLInputElement.prototype,
                'value'
            ).set;

            nativeInputValueSetter.call(targetInput, fieldName);

            // Dispatch events to notify React
            targetInput.dispatchEvent(new Event('input', { bubbles: true }));
            targetInput.dispatchEvent(new Event('change', { bubbles: true }));

            // Focus and blur to ensure the change is registered
            targetInput.focus();
            setTimeout(() => {
                targetInput.blur();
                console.log('[InstaRank] Field injected successfully:', fieldName);
                showNotification(`Field "${fieldName}" set!`);
            }, 100);
        } else {
            console.log('[InstaRank] Could not find custom field input');
            copyToClipboardFallback(fieldName);
        }
    }

    /**
     * Fallback: copy field name to clipboard
     */
    function copyToClipboardFallback(fieldName) {
        navigator.clipboard.writeText(fieldName).then(() => {
            console.log('[InstaRank] Copied to clipboard:', fieldName);
            showNotification(`"${fieldName}" copied! Select "Post Custom Field" and paste.`);
        }).catch(() => {
            alert(`Field name: ${fieldName}\n\nSelect "Post Custom Field" from Content dropdown and paste this value.`);
        });
    }

    /**
     * Show a temporary notification
     */
    function showNotification(message) {
        const notification = document.createElement('div');
        notification.style.cssText = `
            position: fixed;
            bottom: 20px;
            right: 20px;
            background: #16a34a;
            color: white;
            padding: 12px 20px;
            border-radius: 6px;
            font-size: 13px;
            z-index: 999999;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            animation: slideIn 0.3s ease;
        `;
        notification.textContent = message;
        document.body.appendChild(notification);

        setTimeout(() => {
            notification.style.animation = 'slideOut 0.3s ease';
            setTimeout(() => notification.remove(), 300);
        }, 3000);
    }

    /**
     * Inject InstaRank fields into Kadence's Content dropdown menu
     * This adds our dataset fields as selectable options in the dropdown
     */
    function injectIntoKadenceDropdown() {
        // Kadence uses WordPress's CustomSelectControl which renders options in a popover
        // The structure is: .components-popover__content > div with role="listbox" > buttons with role="option"

        // Find all popovers that might contain Kadence's dropdown
        const popovers = document.querySelectorAll('.components-popover__content');

        for (const popover of popovers) {
            // Check if this is a Kadence dynamic content dropdown by looking for known options
            const hasKadenceOptions = popover.textContent.includes('Post Custom Field') ||
                                      popover.textContent.includes('Post Title') ||
                                      popover.textContent.includes('Featured Image URL');

            if (!hasKadenceOptions) continue;

            // Check if we already injected
            if (popover.querySelector('.instarank-dropdown-separator')) continue;

            console.log('[InstaRank] Found Kadence dropdown popover');

            // Find the container with the options (usually has role="listbox" or is a direct child)
            let optionsContainer = popover.querySelector('[role="listbox"]');
            if (!optionsContainer) {
                // Try finding the first scrollable/list container
                optionsContainer = popover.querySelector('div > div') || popover.firstElementChild;
            }

            if (!optionsContainer) {
                console.log('[InstaRank] Could not find options container in popover');
                continue;
            }

            console.log('[InstaRank] Found options container, injecting fields');

            // Create separator header
            const separator = document.createElement('div');
            separator.className = 'instarank-dropdown-separator';
            separator.style.cssText = `
                padding: 10px 16px 6px;
                font-size: 11px;
                font-weight: 600;
                color: #f97316;
                text-transform: uppercase;
                letter-spacing: 0.5px;
                border-top: 2px solid #f97316;
                margin-top: 8px;
                background: linear-gradient(135deg, #fff7ed 0%, #ffedd5 100%);
            `;
            separator.innerHTML = `
                <span style="display: flex; align-items: center; gap: 6px;">
                    <svg width="14" height="14" viewBox="0 0 24 24" fill="currentColor">
                        <path d="M12 2L2 7l10 5 10-5-10-5zM2 17l10 5 10-5M2 12l10 5 10-5"></path>
                    </svg>
                    InstaRank Dataset Fields
                </span>
            `;
            optionsContainer.appendChild(separator);

            // Add dataset fields
            const addedFields = new Set();

            // First add mapped fields
            for (const [fieldName, mapping] of Object.entries(fieldMappings)) {
                if (!addedFields.has(fieldName)) {
                    addDropdownOption(optionsContainer, fieldName, mapping.type || 'text');
                    addedFields.add(fieldName);
                }
            }

            // Then add dataset columns
            for (const column of datasetColumns) {
                if (!addedFields.has(column)) {
                    addDropdownOption(optionsContainer, column, 'text');
                    addedFields.add(column);
                }
            }

            console.log('[InstaRank] Injected', addedFields.size, 'dataset fields into dropdown');
        }
    }

    /**
     * Add a single option to the Kadence dropdown
     */
    function addDropdownOption(optionsList, fieldName, type) {
        // Create option that matches Kadence's styling
        const option = document.createElement('button');
        option.type = 'button';
        option.className = 'instarank-dropdown-option components-button';
        option.setAttribute('role', 'option');
        option.setAttribute('data-field', fieldName);
        option.style.cssText = `
            padding: 8px 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 13px;
            width: 100%;
            text-align: left;
            background: transparent;
            border: none;
            border-radius: 0;
            transition: background 0.1s;
            min-height: 36px;
        `;

        // Icon based on type
        const icons = {
            text: '📝',
            image: '🖼️',
            url: '🔗',
            number: '#️⃣',
            date: '📅',
        };
        const icon = icons[type] || icons.text;

        option.innerHTML = `
            <span style="font-size: 14px; flex-shrink: 0;">${icon}</span>
            <span style="flex: 1; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;">${fieldName}</span>
            <span style="flex-shrink: 0; font-size: 9px; color: #ea580c; background: #ffedd5; padding: 2px 6px; border-radius: 3px; font-weight: 600;">DATASET</span>
        `;

        // Hover effect - match WordPress admin style
        option.addEventListener('mouseenter', () => {
            option.style.background = '#f97316';
            option.style.color = 'white';
            option.querySelector('span:last-child').style.background = 'rgba(255,255,255,0.2)';
            option.querySelector('span:last-child').style.color = 'white';
        });
        option.addEventListener('mouseleave', () => {
            option.style.background = 'transparent';
            option.style.color = 'inherit';
            option.querySelector('span:last-child').style.background = '#ffedd5';
            option.querySelector('span:last-child').style.color = '#ea580c';
        });

        // Click handler - select this field
        option.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();

            console.log('[InstaRank] Dataset field selected from dropdown:', fieldName);

            // We need to:
            // 1. First select "Post Custom Field" as the source type
            // 2. Then fill in the custom field name

            // Find and click "Post Custom Field" option first
            selectPostCustomFieldOption();

            // After a short delay, inject the field name
            setTimeout(() => {
                injectFieldValue(fieldName);
            }, 300);
        });

        optionsList.appendChild(option);
    }

    /**
     * Select "Post Custom Field" option in Kadence's dropdown
     */
    function selectPostCustomFieldOption() {
        // Find the "Post Custom Field" option and click it
        const allOptions = document.querySelectorAll('[role="option"], [class*="select__option"]');

        for (const opt of allOptions) {
            if (opt.textContent.includes('Post Custom Field')) {
                console.log('[InstaRank] Clicking Post Custom Field option');
                opt.click();
                return true;
            }
        }

        return false;
    }

    /**
     * Find and inject our panel into Kadence's dynamic content UI
     */
    function injectIntoKadenceUI() {
        // Also try to inject into dropdown menus
        injectIntoKadenceDropdown();

        // Look for Kadence's dynamic content dropdown or panel
        // Kadence's "Post Custom Field" option appears when dynamic content is enabled

        // Target: The input for custom field name (appears after selecting "Post Custom Field")
        const possibleContainers = [
            // Kadence dynamic source selector area
            '.kadence-dynamic-source-control',
            '.kadence-dynamic-field-control',
            // Kadence advanced heading dynamic settings
            '.kt-adv-heading-dynamic-content',
            // Generic Kadence dynamic panel
            '[class*="kadence"][class*="dynamic"]',
            // Block editor sidebar controls
            '.components-panel__body',
            '.interface-interface-skeleton__sidebar .components-panel',
        ];

        for (const selector of possibleContainers) {
            const containers = document.querySelectorAll(selector);

            for (const container of containers) {
                // Check if this container has Kadence dynamic content controls
                const hasCustomFieldOption = container.textContent.includes('Post Custom Field') ||
                                              container.textContent.includes('Custom Field') ||
                                              container.textContent.includes('post_custom_field');

                // Also check for the "kb_custom_input" parameter which is what Kadence uses
                const hasKbCustomInput = container.innerHTML.includes('kb_custom_input') ||
                                          container.innerHTML.includes('data-custom');

                if ((hasCustomFieldOption || hasKbCustomInput) && !container.querySelector('.instarank-fields-panel')) {
                    // Find the best place to inject our panel
                    // Usually after the source/field selection dropdown

                    const panel = createInstaRankFieldsPanel();
                    const fieldsList = panel.querySelector('.instarank-fields-list');

                    // Add field buttons
                    // First, add mapped fields from field_mappings
                    const addedFields = new Set();

                    for (const [fieldName, mapping] of Object.entries(fieldMappings)) {
                        if (!addedFields.has(fieldName)) {
                            const type = mapping.type || 'text';
                            const btn = createFieldButton(fieldName, mapping.dataset_column, type);
                            btn.addEventListener('click', (e) => {
                                e.preventDefault();
                                e.stopPropagation();
                                injectFieldValue(fieldName);
                            });
                            fieldsList.appendChild(btn);
                            addedFields.add(fieldName);
                        }
                    }

                    // Then add any dataset columns not already added
                    for (const column of datasetColumns) {
                        if (!addedFields.has(column)) {
                            const btn = createFieldButton(column, column, 'text');
                            btn.addEventListener('click', (e) => {
                                e.preventDefault();
                                e.stopPropagation();
                                injectFieldValue(column);
                            });
                            fieldsList.appendChild(btn);
                            addedFields.add(column);
                        }
                    }

                    // Only inject if we have fields
                    if (fieldsList.children.length > 0) {
                        // Find a good insertion point
                        const insertionPoint = container.querySelector('.components-base-control') ||
                                               container.querySelector('.components-panel__row') ||
                                               container.firstElementChild;

                        if (insertionPoint) {
                            insertionPoint.parentNode.insertBefore(panel, insertionPoint.nextSibling);
                            console.log('[InstaRank] Injected fields panel into Kadence UI');
                        } else {
                            container.appendChild(panel);
                            console.log('[InstaRank] Appended fields panel to container');
                        }
                    }
                }
            }
        }
    }

    /**
     * Watch for changes in the editor sidebar
     * Kadence's dynamic content panel appears dynamically
     */
    function watchForKadenceUI() {
        // Use MutationObserver to detect when Kadence panels are opened
        const observer = new MutationObserver((mutations) => {
            for (const mutation of mutations) {
                if (mutation.addedNodes.length > 0) {
                    // Check if any of the added nodes contain Kadence dynamic content UI
                    for (const node of mutation.addedNodes) {
                        if (node.nodeType === Node.ELEMENT_NODE) {
                            const hasKadenceDynamic = node.querySelector &&
                                (node.querySelector('[class*="kadence"][class*="dynamic"]') ||
                                 node.textContent?.includes('Post Custom Field') ||
                                 node.textContent?.includes('kadenceDynamic'));

                            if (hasKadenceDynamic) {
                                // Small delay to let Kadence fully render
                                setTimeout(injectIntoKadenceUI, 100);
                            }
                        }
                    }
                }
            }
        });

        // Observe the editor sidebar and main content area
        const targets = [
            document.querySelector('.interface-interface-skeleton__sidebar'),
            document.querySelector('.interface-interface-skeleton__content'),
            document.querySelector('#editor'),
            document.body,
        ].filter(Boolean);

        for (const target of targets) {
            observer.observe(target, {
                childList: true,
                subtree: true,
            });
        }

        console.log('[InstaRank] Watching for Kadence dynamic content UI');
    }

    /**
     * Add CSS animations
     */
    function addStyles() {
        const style = document.createElement('style');
        style.textContent = `
            @keyframes slideIn {
                from {
                    transform: translateX(100%);
                    opacity: 0;
                }
                to {
                    transform: translateX(0);
                    opacity: 1;
                }
            }
            @keyframes slideOut {
                from {
                    transform: translateX(0);
                    opacity: 1;
                }
                to {
                    transform: translateX(100%);
                    opacity: 0;
                }
            }
            .instarank-field-btn:active {
                transform: scale(0.95);
            }
        `;
        document.head.appendChild(style);
    }

    /**
     * Initialize when DOM is ready
     */
    function init() {
        addStyles();

        // Wait for the editor to be ready
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => {
                setTimeout(() => {
                    injectIntoKadenceUI();
                    watchForKadenceUI();
                }, 1000);
            });
        } else {
            setTimeout(() => {
                injectIntoKadenceUI();
                watchForKadenceUI();
            }, 1000);
        }
    }

    // Start
    init();
})();
