/**
 * InstaRank - Universal Custom Fields Integration
 *
 * Provides a unified custom fields panel that works across all page builders:
 * - Kadence Blocks Pro
 * - Elementor Pro
 * - Gutenberg (WordPress core)
 * - Divi Builder
 * - Beaver Builder
 * - Bricks Builder
 * - And more...
 *
 * @package InstaRank
 * @version 1.5.1
 */
(function() {
    'use strict';

    // Configuration from WordPress localized script
    const config = window.instarankCustomFields || {};
    const datasetColumns = config.datasetColumns || [];
    const datasetName = config.datasetName || '';
    const fieldMappings = config.fieldMappings || {};
    const activeBuilder = config.activeBuilder || 'gutenberg';

    // Early exit if no fields configured
    if (datasetColumns.length === 0 && Object.keys(fieldMappings).length === 0) {
        console.log('[InstaRank] No dataset columns or field mappings found');
        return;
    }

    console.log('[InstaRank] Universal Custom Fields Integration loaded');
    console.log('[InstaRank] Active builder:', activeBuilder);
    console.log('[InstaRank] Dataset columns:', datasetColumns);

    // Field count for display
    const totalFields = new Set([...Object.keys(fieldMappings), ...datasetColumns]).size;

    /**
     * Create the InstaRank Custom Fields floating panel
     * This panel appears in the editor and lets users select dataset fields
     */
    function createCustomFieldsPanel() {
        // Don't create if already exists
        if (document.getElementById('instarank-custom-fields-panel')) {
            return;
        }

        const panel = document.createElement('div');
        panel.id = 'instarank-custom-fields-panel';
        panel.className = 'instarank-custom-fields-panel';
        panel.innerHTML = `
            <div class="instarank-panel-header" id="instarank-panel-drag-handle">
                <div class="instarank-panel-logo">
                    <div class="instarank-logo-icon">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round">
                            <polygon points="12 2 2 7 12 12 22 7 12 2"></polygon>
                            <polyline points="2 17 12 22 22 17"></polyline>
                            <polyline points="2 12 12 17 22 12"></polyline>
                        </svg>
                    </div>
                    <div class="instarank-panel-title-group">
                        <span class="instarank-panel-title">InstaRank Fields</span>
                        <span class="instarank-panel-subtitle">${totalFields} fields available</span>
                    </div>
                </div>
                <div class="instarank-panel-controls">
                    <button type="button" class="instarank-panel-btn instarank-panel-minimize" title="Minimize">
                        <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round">
                            <line x1="5" y1="12" x2="19" y2="12"></line>
                        </svg>
                    </button>
                    <button type="button" class="instarank-panel-btn instarank-panel-close" title="Close">
                        <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round">
                            <line x1="18" y1="6" x2="6" y2="18"></line>
                            <line x1="6" y1="6" x2="18" y2="18"></line>
                        </svg>
                    </button>
                </div>
            </div>
            <div class="instarank-panel-body">
                ${datasetName ? `
                    <div class="instarank-dataset-badge">
                        <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"></path>
                        </svg>
                        <span>${datasetName}</span>
                    </div>
                ` : `
                    <div class="instarank-dataset-warning">
                        <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="12" cy="12" r="10"></circle>
                            <line x1="12" y1="8" x2="12" y2="12"></line>
                            <line x1="12" y1="16" x2="12.01" y2="16"></line>
                        </svg>
                        <span>No dataset linked. Go to <strong>InstaRank → Templates</strong> to link one.</span>
                    </div>
                `}
                <div class="instarank-search-wrapper">
                    <svg class="instarank-search-icon" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="11" cy="11" r="8"></circle>
                        <path d="m21 21-4.35-4.35"></path>
                    </svg>
                    <input type="text" placeholder="Search fields..." class="instarank-search-input" />
                </div>
                <div class="instarank-fields-container">
                    <div class="instarank-fields-list">
                        <!-- Fields will be injected here -->
                    </div>
                </div>
            </div>
            <div class="instarank-panel-footer">
                <div class="instarank-footer-hint">
                    <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M15 15l-2 5L9 9l11 4-5 2zm0 0l5 5M7.188 2.239l.777 2.897M5.136 7.965l-2.898-.777M13.95 4.05l-2.122 2.122m-5.657 5.656l-2.12 2.122"></path>
                    </svg>
                    <span>Click to insert • Drag to move</span>
                </div>
            </div>
        `;

        // Add styles
        addPanelStyles();

        // Populate fields
        const fieldsList = panel.querySelector('.instarank-fields-list');
        populateFields(fieldsList);

        // Add search functionality
        const searchInput = panel.querySelector('.instarank-search-input');
        searchInput.addEventListener('input', (e) => {
            filterFields(fieldsList, e.target.value);
        });

        // Add minimize/close functionality
        panel.querySelector('.instarank-panel-minimize').addEventListener('click', () => {
            panel.classList.toggle('minimized');
            const minBtn = panel.querySelector('.instarank-panel-minimize svg');
            if (panel.classList.contains('minimized')) {
                minBtn.innerHTML = '<line x1="12" y1="5" x2="12" y2="19"></line><line x1="5" y1="12" x2="19" y2="12"></line>';
            } else {
                minBtn.innerHTML = '<line x1="5" y1="12" x2="19" y2="12"></line>';
            }
        });

        panel.querySelector('.instarank-panel-close').addEventListener('click', () => {
            panel.style.display = 'none';
            showToggleButton();
        });

        // Make panel draggable
        makeDraggable(panel);

        document.body.appendChild(panel);

        return panel;
    }

    /**
     * Populate fields in the panel
     */
    function populateFields(container) {
        container.innerHTML = '';
        const addedFields = new Set();

        // First add mapped fields
        for (const [fieldName, mapping] of Object.entries(fieldMappings)) {
            if (!addedFields.has(fieldName)) {
                const fieldButton = createFieldButton(fieldName, mapping.type || 'text', mapping.dataset_column);
                container.appendChild(fieldButton);
                addedFields.add(fieldName);
            }
        }

        // Then add dataset columns
        for (const column of datasetColumns) {
            if (!addedFields.has(column)) {
                const fieldButton = createFieldButton(column, 'text', column);
                container.appendChild(fieldButton);
                addedFields.add(column);
            }
        }

        if (container.children.length === 0) {
            container.innerHTML = `
                <div class="instarank-no-fields">
                    <svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                        <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"></path>
                        <polyline points="3.27 6.96 12 12.01 20.73 6.96"></polyline>
                        <line x1="12" y1="22.08" x2="12" y2="12"></line>
                    </svg>
                    <p>No fields available</p>
                    <small>Link a dataset first</small>
                </div>
            `;
        }
    }

    /**
     * Create a clickable field button
     */
    function createFieldButton(fieldName, type, datasetColumn) {
        const button = document.createElement('button');
        button.type = 'button';
        button.className = 'instarank-field-button';
        button.dataset.field = fieldName;
        button.dataset.column = datasetColumn || fieldName;
        button.dataset.type = type;

        // Icons based on type
        const icons = {
            text: '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M4 7V4h16v3"/><path d="M9 20h6"/><path d="M12 4v16"/></svg>',
            image: '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="3" width="18" height="18" rx="2" ry="2"/><circle cx="8.5" cy="8.5" r="1.5"/><polyline points="21 15 16 10 5 21"/></svg>',
            url: '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M10 13a5 5 0 0 0 7.54.54l3-3a5 5 0 0 0-7.07-7.07l-1.72 1.71"/><path d="M14 11a5 5 0 0 0-7.54-.54l-3 3a5 5 0 0 0 7.07 7.07l1.71-1.71"/></svg>',
            number: '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M4 9h16"/><path d="M4 15h16"/><path d="M10 3L8 21"/><path d="M16 3l-2 18"/></svg>',
            date: '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="4" width="18" height="18" rx="2" ry="2"/><line x1="16" y1="2" x2="16" y2="6"/><line x1="8" y1="2" x2="8" y2="6"/><line x1="3" y1="10" x2="21" y2="10"/></svg>',
        };
        const icon = icons[type] || icons.text;

        // Determine type color
        const typeColors = {
            text: '#6366f1',
            image: '#10b981',
            url: '#3b82f6',
            number: '#f59e0b',
            date: '#ec4899',
        };
        const typeColor = typeColors[type] || typeColors.text;

        button.innerHTML = `
            <span class="instarank-field-icon" style="color: ${typeColor}">${icon}</span>
            <span class="instarank-field-name">${fieldName}</span>
            <span class="instarank-field-type" style="background: ${typeColor}15; color: ${typeColor}">${type}</span>
        `;

        // Click handler
        button.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();

            // Visual feedback
            button.classList.add('clicked');
            setTimeout(() => button.classList.remove('clicked'), 200);

            handleFieldClick(fieldName, type);
        });

        return button;
    }

    /**
     * Filter fields based on search input
     */
    function filterFields(container, searchTerm) {
        const buttons = container.querySelectorAll('.instarank-field-button');
        const term = searchTerm.toLowerCase();
        let visibleCount = 0;

        buttons.forEach(button => {
            const fieldName = button.dataset.field.toLowerCase();
            const matches = fieldName.includes(term);
            button.style.display = matches ? '' : 'none';
            if (matches) visibleCount++;
        });

        // Update subtitle with count
        const subtitle = document.querySelector('.instarank-panel-subtitle');
        if (subtitle) {
            subtitle.textContent = term ? `${visibleCount} of ${totalFields} fields` : `${totalFields} fields available`;
        }
    }

    /**
     * Handle field button click - inject into active builder
     */
    function handleFieldClick(fieldName, type) {
        console.log('[InstaRank] Field clicked:', fieldName, 'Type:', type, 'Builder:', activeBuilder);

        let success = false;

        switch (activeBuilder) {
            case 'kadence_blocks':
            case 'kadence':
                success = injectIntoKadence(fieldName);
                break;
            case 'elementor':
                success = injectIntoElementor(fieldName);
                break;
            case 'divi':
                success = injectIntoDivi(fieldName);
                break;
            case 'beaver_builder':
                success = injectIntoBeaverBuilder(fieldName);
                break;
            case 'bricks':
                success = injectIntoBricks(fieldName);
                break;
            case 'gutenberg':
            default:
                success = injectIntoGutenberg(fieldName);
                break;
        }

        if (!success) {
            success = injectIntoFocusedInput(fieldName);
        }

        if (!success) {
            copyToClipboard(fieldName);
        }
    }

    /**
     * Inject field into Kadence Blocks Pro dynamic content
     */
    function injectIntoKadence(fieldName) {
        console.log('[InstaRank] Attempting Kadence injection for:', fieldName);

        const addDynamicBtn = findButtonByText(['Add Dynamic Content', 'Enable Dynamic']);
        if (addDynamicBtn) {
            console.log('[InstaRank] Clicking "Add Dynamic Content" button');
            addDynamicBtn.click();

            setTimeout(() => {
                selectKadencePostCustomField(fieldName);
            }, 400);
            return true;
        }

        return selectKadencePostCustomField(fieldName);
    }

    /**
     * Select "Post Custom Field" in Kadence and fill in the field name
     */
    function selectKadencePostCustomField(fieldName) {
        const sidebar = document.querySelector('.interface-interface-skeleton__sidebar');
        if (!sidebar) {
            console.log('[InstaRank] Sidebar not found');
            return false;
        }

        const contentDropdown = findKadenceContentDropdown(sidebar);

        if (contentDropdown) {
            console.log('[InstaRank] Found content dropdown, clicking to open');
            contentDropdown.click();

            setTimeout(() => {
                const postCustomFieldOption = findOptionByText('Post Custom Field');
                if (postCustomFieldOption) {
                    console.log('[InstaRank] Selecting "Post Custom Field"');
                    postCustomFieldOption.click();

                    setTimeout(() => {
                        fillKadenceCustomFieldInput(fieldName);
                    }, 300);
                } else {
                    fillKadenceCustomFieldInput(fieldName);
                }
            }, 200);

            return true;
        }

        return fillKadenceCustomFieldInput(fieldName);
    }

    /**
     * Find Kadence's Content dropdown
     */
    function findKadenceContentDropdown(container) {
        const ariaDropdowns = container.querySelectorAll('button[aria-haspopup="listbox"]');
        for (const btn of ariaDropdowns) {
            const control = btn.closest('.components-base-control');
            if (control) {
                const label = control.querySelector('label, .components-base-control__label');
                if (label && label.textContent.toLowerCase().includes('content')) {
                    return btn;
                }
            }
            const text = btn.textContent;
            if (text.includes('None') || text.includes('Post Title') ||
                text.includes('Post Custom Field') || text.includes('Post URL') ||
                text.includes('Featured Image')) {
                return btn;
            }
        }

        const customSelects = container.querySelectorAll('.components-custom-select-control button, [class*="custom-select"] button');
        for (const btn of customSelects) {
            const control = btn.closest('.components-base-control');
            if (control && control.textContent.toLowerCase().includes('content')) {
                return btn;
            }
        }

        return null;
    }

    /**
     * Fill in the custom field input with the field name
     */
    function fillKadenceCustomFieldInput(fieldName) {
        const sidebar = document.querySelector('.interface-interface-skeleton__sidebar');
        if (!sidebar) {
            console.log('[InstaRank] Sidebar not found for input fill');
            return false;
        }

        const inputs = sidebar.querySelectorAll('input[type="text"]:not([readonly]):not([disabled])');

        for (const input of inputs) {
            const rect = input.getBoundingClientRect();
            if (rect.width === 0 || rect.height === 0) continue;

            const control = input.closest('.components-base-control');
            if (control) {
                const labelText = control.textContent.toLowerCase();
                if (labelText.includes('custom') || labelText.includes('field') ||
                    labelText.includes('key') || labelText.includes('meta')) {
                    return setInputValue(input, fieldName);
                }
            }

            const placeholder = input.placeholder?.toLowerCase() || '';
            if (placeholder.includes('custom') || placeholder.includes('field') ||
                placeholder.includes('key') || placeholder.includes('meta')) {
                return setInputValue(input, fieldName);
            }
        }

        const dynamicPanel = sidebar.querySelector('.components-panel__body.is-opened');
        if (dynamicPanel && dynamicPanel.textContent.includes('Dynamic')) {
            const panelInputs = dynamicPanel.querySelectorAll('input[type="text"]');
            for (const input of panelInputs) {
                const rect = input.getBoundingClientRect();
                if (rect.width > 0 && rect.height > 0) {
                    return setInputValue(input, fieldName);
                }
            }
        }

        console.log('[InstaRank] Could not find custom field input in Kadence');
        return false;
    }

    /**
     * Inject field into Elementor Pro dynamic tags
     */
    function injectIntoElementor(fieldName) {
        console.log('[InstaRank] Attempting Elementor injection for:', fieldName);

        const panel = document.querySelector('#elementor-panel, .elementor-panel');
        if (!panel) {
            console.log('[InstaRank] Elementor panel not found');
            return false;
        }

        const dynamicBtn = panel.querySelector('.e-control-dynamic-switcher, [data-e-control="dynamic"]');
        if (dynamicBtn) {
            dynamicBtn.click();

            setTimeout(() => {
                const customFieldOption = findElementorDynamicOption(['ACF Field', 'Custom Field', 'Post Meta']);
                if (customFieldOption) {
                    customFieldOption.click();

                    setTimeout(() => {
                        fillElementorFieldInput(fieldName);
                    }, 300);
                }
            }, 300);

            return true;
        }

        const inputs = panel.querySelectorAll('input[type="text"]');
        for (const input of inputs) {
            if (input.placeholder?.toLowerCase().includes('field') ||
                input.placeholder?.toLowerCase().includes('key')) {
                return setInputValue(input, fieldName);
            }
        }

        return false;
    }

    /**
     * Find Elementor dynamic option
     */
    function findElementorDynamicOption(keywords) {
        const options = document.querySelectorAll('.elementor-tag-list__item, .e-dynamic-content-item');
        for (const option of options) {
            const text = option.textContent;
            for (const keyword of keywords) {
                if (text.includes(keyword)) {
                    return option;
                }
            }
        }
        return null;
    }

    /**
     * Fill Elementor field input
     */
    function fillElementorFieldInput(fieldName) {
        const modal = document.querySelector('.elementor-dynamic-cover, .dialog-widget-content');
        if (!modal) return false;

        const inputs = modal.querySelectorAll('input[type="text"]');
        for (const input of inputs) {
            if (input.placeholder?.toLowerCase().includes('key') ||
                input.placeholder?.toLowerCase().includes('field')) {
                return setInputValue(input, fieldName);
            }
        }

        if (inputs.length > 0) {
            return setInputValue(inputs[0], fieldName);
        }

        return false;
    }

    /**
     * Inject field into Divi Builder
     */
    function injectIntoDivi(fieldName) {
        const modal = document.querySelector('.et-fb-settings-option-modal, .et_pb_modal_settings_container');
        if (!modal) return false;

        const inputs = modal.querySelectorAll('input[type="text"]');
        for (const input of inputs) {
            const label = input.closest('.et-fb-option-container, .et_pb_option')
                ?.querySelector('label, .et-fb-option__label');
            if (label) {
                const labelText = label.textContent.toLowerCase();
                if (labelText.includes('custom field') || labelText.includes('meta')) {
                    return setInputValue(input, fieldName);
                }
            }
        }

        return false;
    }

    /**
     * Inject field into Beaver Builder
     */
    function injectIntoBeaverBuilder(fieldName) {
        const panel = document.querySelector('.fl-builder-settings, .fl-lightbox-content');
        if (!panel) return false;

        const connectBtn = panel.querySelector('.fl-field-connections-toggle, [data-connection]');
        if (connectBtn) {
            connectBtn.click();

            setTimeout(() => {
                const metaOption = panel.querySelector('[data-connection="post_meta"], [data-type="custom_field"]');
                if (metaOption) {
                    metaOption.click();

                    setTimeout(() => {
                        const inputs = panel.querySelectorAll('input[type="text"]');
                        for (const input of inputs) {
                            const name = input.name?.toLowerCase() || '';
                            if (name.includes('key') || name.includes('field') || name.includes('meta')) {
                                return setInputValue(input, fieldName);
                            }
                        }
                    }, 300);
                }
            }, 300);

            return true;
        }

        return false;
    }

    /**
     * Inject field into Bricks Builder
     */
    function injectIntoBricks(fieldName) {
        const panel = document.querySelector('#bricks-panel, .bricks-panel');
        if (!panel) return false;

        const dynamicBtn = panel.querySelector('[data-dynamic], .bricks-dynamic-data-toggle');
        if (dynamicBtn) {
            dynamicBtn.click();

            setTimeout(() => {
                const customFieldOption = panel.querySelector('[data-source="cf"], [data-source="meta"]');
                if (customFieldOption) {
                    customFieldOption.click();

                    setTimeout(() => {
                        const inputs = panel.querySelectorAll('input[type="text"]');
                        for (const input of inputs) {
                            const placeholder = input.placeholder?.toLowerCase() || '';
                            if (placeholder.includes('key') || placeholder.includes('field') || placeholder.includes('meta')) {
                                return setInputValue(input, fieldName);
                            }
                        }
                    }, 300);
                }
            }, 300);

            return true;
        }

        return false;
    }

    /**
     * Inject field into Gutenberg (WordPress core)
     */
    function injectIntoGutenberg(fieldName) {
        const sidebar = document.querySelector('.interface-interface-skeleton__sidebar, .edit-post-sidebar');
        if (!sidebar) return false;

        const customFieldsPanel = sidebar.querySelector('[class*="custom-fields"], [aria-label*="Custom Fields"]');
        if (customFieldsPanel) {
            const inputs = customFieldsPanel.querySelectorAll('input[type="text"]');
            if (inputs.length > 0) {
                return setInputValue(inputs[0], fieldName);
            }
        }

        return false;
    }

    /**
     * Inject into any focused input as fallback
     */
    function injectIntoFocusedInput(fieldName) {
        const activeElement = document.activeElement;
        if (activeElement && (activeElement.tagName === 'INPUT' || activeElement.tagName === 'TEXTAREA')) {
            return setInputValue(activeElement, fieldName);
        }
        return false;
    }

    /**
     * Set value in an input field using native setter for React compatibility
     */
    function setInputValue(input, value) {
        try {
            const nativeInputValueSetter = Object.getOwnPropertyDescriptor(
                window.HTMLInputElement.prototype,
                'value'
            ).set;

            nativeInputValueSetter.call(input, value);

            input.dispatchEvent(new Event('input', { bubbles: true }));
            input.dispatchEvent(new Event('change', { bubbles: true }));

            input.focus();
            setTimeout(() => {
                input.blur();
                showNotification(`Field "${value}" inserted!`, 'success');
                console.log('[InstaRank] Successfully set field value:', value);
            }, 100);

            return true;
        } catch (e) {
            console.error('[InstaRank] Error setting input value:', e);
            return false;
        }
    }

    /**
     * Copy field name to clipboard as last resort
     */
    function copyToClipboard(fieldName) {
        navigator.clipboard.writeText(fieldName).then(() => {
            showNotification(`"${fieldName}" copied to clipboard!`, 'info');
        }).catch(() => {
            showNotification(`Field: ${fieldName}`, 'warning');
        });
    }

    /**
     * Find a button by text content
     */
    function findButtonByText(texts) {
        const buttons = document.querySelectorAll('button');
        for (const btn of buttons) {
            for (const text of texts) {
                if (btn.textContent.includes(text)) {
                    return btn;
                }
            }
        }
        return null;
    }

    /**
     * Find an option element by text
     */
    function findOptionByText(text) {
        const popovers = document.querySelectorAll('.components-popover__content');
        for (const popover of popovers) {
            const options = popover.querySelectorAll('[role="option"], button, li');
            for (const opt of options) {
                if (opt.textContent.trim() === text) {
                    return opt;
                }
            }
        }

        const allOptions = document.querySelectorAll('[role="option"]');
        for (const opt of allOptions) {
            if (opt.textContent.trim() === text) {
                return opt;
            }
        }

        return null;
    }

    /**
     * Show notification toast
     */
    function showNotification(message, type = 'success') {
        // Remove any existing notifications
        document.querySelectorAll('.instarank-notification').forEach(n => n.remove());

        const icons = {
            success: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"></path><polyline points="22 4 12 14.01 9 11.01"></polyline></svg>',
            info: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><circle cx="12" cy="12" r="10"></circle><path d="M12 16v-4"></path><path d="M12 8h.01"></path></svg>',
            warning: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="m21.73 18-8-14a2 2 0 0 0-3.48 0l-8 14A2 2 0 0 0 4 21h16a2 2 0 0 0 1.73-3Z"></path><path d="M12 9v4"></path><path d="M12 17h.01"></path></svg>',
            error: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><circle cx="12" cy="12" r="10"></circle><path d="m15 9-6 6"></path><path d="m9 9 6 6"></path></svg>',
        };

        const colors = {
            success: { bg: '#10b981', text: 'white' },
            info: { bg: '#3b82f6', text: 'white' },
            warning: { bg: '#f59e0b', text: 'white' },
            error: { bg: '#ef4444', text: 'white' },
        };

        const color = colors[type] || colors.success;
        const icon = icons[type] || icons.success;

        const notification = document.createElement('div');
        notification.className = 'instarank-notification';
        notification.innerHTML = `
            <span class="instarank-notification-icon">${icon}</span>
            <span class="instarank-notification-text">${message}</span>
        `;
        notification.style.cssText = `
            position: fixed;
            bottom: 24px;
            right: 24px;
            background: ${color.bg};
            color: ${color.text};
            padding: 12px 18px;
            border-radius: 10px;
            font-size: 13px;
            font-weight: 500;
            z-index: 999999;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            display: flex;
            align-items: center;
            gap: 10px;
            animation: instarank-toast-in 0.4s cubic-bezier(0.16, 1, 0.3, 1);
            max-width: 320px;
        `;

        document.body.appendChild(notification);

        setTimeout(() => {
            notification.style.animation = 'instarank-toast-out 0.3s cubic-bezier(0.16, 1, 0.3, 1) forwards';
            setTimeout(() => notification.remove(), 300);
        }, 3000);
    }

    /**
     * Show toggle button when panel is closed
     */
    function showToggleButton() {
        let toggleBtn = document.getElementById('instarank-toggle-button');
        if (!toggleBtn) {
            toggleBtn = document.createElement('button');
            toggleBtn.id = 'instarank-toggle-button';
            toggleBtn.type = 'button';
            toggleBtn.innerHTML = `
                <svg width="22" height="22" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round">
                    <polygon points="12 2 2 7 12 12 22 7 12 2"></polygon>
                    <polyline points="2 17 12 22 22 17"></polyline>
                    <polyline points="2 12 12 17 22 12"></polyline>
                </svg>
            `;
            toggleBtn.title = 'Open InstaRank Fields';
            toggleBtn.style.cssText = `
                position: fixed;
                bottom: 24px;
                right: 24px;
                width: 52px;
                height: 52px;
                border-radius: 16px;
                background: linear-gradient(135deg, #f97316 0%, #ea580c 100%);
                color: white;
                border: none;
                cursor: pointer;
                z-index: 999998;
                box-shadow: 0 8px 32px rgba(249, 115, 22, 0.4);
                display: flex;
                align-items: center;
                justify-content: center;
                transition: all 0.3s cubic-bezier(0.16, 1, 0.3, 1);
            `;
            toggleBtn.addEventListener('mouseenter', () => {
                toggleBtn.style.transform = 'scale(1.08) translateY(-2px)';
                toggleBtn.style.boxShadow = '0 12px 40px rgba(249, 115, 22, 0.5)';
            });
            toggleBtn.addEventListener('mouseleave', () => {
                toggleBtn.style.transform = 'scale(1) translateY(0)';
                toggleBtn.style.boxShadow = '0 8px 32px rgba(249, 115, 22, 0.4)';
            });
            toggleBtn.addEventListener('click', () => {
                toggleBtn.style.display = 'none';
                const panel = document.getElementById('instarank-custom-fields-panel');
                if (panel) {
                    panel.style.display = '';
                } else {
                    createCustomFieldsPanel();
                }
            });
            document.body.appendChild(toggleBtn);
        }
        toggleBtn.style.display = '';
    }

    /**
     * Make panel draggable
     */
    function makeDraggable(panel) {
        const handle = panel.querySelector('#instarank-panel-drag-handle');
        let isDragging = false;
        let offsetX, offsetY;

        handle.addEventListener('mousedown', (e) => {
            if (e.target.closest('button')) return;
            isDragging = true;
            offsetX = e.clientX - panel.offsetLeft;
            offsetY = e.clientY - panel.offsetTop;
            panel.style.transition = 'none';
            handle.style.cursor = 'grabbing';
        });

        document.addEventListener('mousemove', (e) => {
            if (!isDragging) return;
            panel.style.left = (e.clientX - offsetX) + 'px';
            panel.style.top = (e.clientY - offsetY) + 'px';
            panel.style.right = 'auto';
            panel.style.bottom = 'auto';
        });

        document.addEventListener('mouseup', () => {
            isDragging = false;
            panel.style.transition = '';
            handle.style.cursor = 'grab';
        });
    }

    /**
     * Add panel styles
     */
    function addPanelStyles() {
        if (document.getElementById('instarank-custom-fields-styles')) return;

        const style = document.createElement('style');
        style.id = 'instarank-custom-fields-styles';
        style.textContent = `
            @keyframes instarank-toast-in {
                from { transform: translateX(100%); opacity: 0; }
                to { transform: translateX(0); opacity: 1; }
            }
            @keyframes instarank-toast-out {
                from { transform: translateX(0); opacity: 1; }
                to { transform: translateX(100%); opacity: 0; }
            }
            @keyframes instarank-panel-in {
                from { transform: translateY(20px); opacity: 0; }
                to { transform: translateY(0); opacity: 1; }
            }

            .instarank-custom-fields-panel {
                position: fixed;
                right: 24px;
                top: 80px;
                width: 300px;
                background: #ffffff;
                border-radius: 16px;
                box-shadow: 0 20px 60px rgba(0, 0, 0, 0.15), 0 0 0 1px rgba(0, 0, 0, 0.05);
                z-index: 999998;
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', sans-serif;
                overflow: hidden;
                animation: instarank-panel-in 0.4s cubic-bezier(0.16, 1, 0.3, 1);
            }

            .instarank-custom-fields-panel.minimized .instarank-panel-body,
            .instarank-custom-fields-panel.minimized .instarank-panel-footer {
                display: none;
            }

            .instarank-custom-fields-panel.minimized {
                width: auto;
            }

            .instarank-panel-header {
                background: linear-gradient(135deg, #f97316 0%, #ea580c 100%);
                color: white;
                padding: 14px 16px;
                display: flex;
                justify-content: space-between;
                align-items: center;
                cursor: grab;
                user-select: none;
            }

            .instarank-panel-header:active {
                cursor: grabbing;
            }

            .instarank-panel-logo {
                display: flex;
                align-items: center;
                gap: 12px;
            }

            .instarank-logo-icon {
                width: 36px;
                height: 36px;
                background: rgba(255, 255, 255, 0.2);
                border-radius: 10px;
                display: flex;
                align-items: center;
                justify-content: center;
            }

            .instarank-panel-title-group {
                display: flex;
                flex-direction: column;
                gap: 2px;
            }

            .instarank-panel-title {
                font-weight: 600;
                font-size: 14px;
                letter-spacing: -0.01em;
            }

            .instarank-panel-subtitle {
                font-size: 11px;
                opacity: 0.85;
                font-weight: 400;
            }

            .instarank-panel-controls {
                display: flex;
                gap: 6px;
            }

            .instarank-panel-btn {
                background: rgba(255, 255, 255, 0.15);
                border: none;
                border-radius: 8px;
                width: 30px;
                height: 30px;
                cursor: pointer;
                display: flex;
                align-items: center;
                justify-content: center;
                transition: all 0.2s ease;
                color: white;
            }

            .instarank-panel-btn:hover {
                background: rgba(255, 255, 255, 0.25);
                transform: scale(1.05);
            }

            .instarank-panel-body {
                padding: 16px;
                background: #fafafa;
            }

            .instarank-dataset-badge {
                display: inline-flex;
                align-items: center;
                gap: 6px;
                background: linear-gradient(135deg, #fff7ed 0%, #ffedd5 100%);
                color: #c2410c;
                padding: 8px 12px;
                border-radius: 8px;
                font-size: 12px;
                font-weight: 500;
                margin-bottom: 12px;
                border: 1px solid #fed7aa;
            }

            .instarank-dataset-warning {
                display: flex;
                align-items: flex-start;
                gap: 8px;
                background: #fef3c7;
                color: #92400e;
                padding: 10px 12px;
                border-radius: 8px;
                font-size: 11px;
                line-height: 1.4;
                margin-bottom: 12px;
                border: 1px solid #fcd34d;
            }

            .instarank-dataset-warning svg {
                flex-shrink: 0;
                margin-top: 1px;
            }

            .instarank-search-wrapper {
                position: relative;
                margin-bottom: 12px;
            }

            .instarank-search-icon {
                position: absolute;
                left: 12px;
                top: 50%;
                transform: translateY(-50%);
                color: #9ca3af;
                pointer-events: none;
            }

            .instarank-search-input {
                width: 100%;
                padding: 10px 12px 10px 38px;
                border: 1px solid #e5e7eb;
                border-radius: 10px;
                font-size: 13px;
                outline: none;
                transition: all 0.2s ease;
                background: white;
                box-sizing: border-box;
            }

            .instarank-search-input:focus {
                border-color: #f97316;
                box-shadow: 0 0 0 3px rgba(249, 115, 22, 0.1);
            }

            .instarank-search-input::placeholder {
                color: #9ca3af;
            }

            .instarank-fields-container {
                background: white;
                border-radius: 12px;
                border: 1px solid #e5e7eb;
                overflow: hidden;
            }

            .instarank-fields-list {
                max-height: 320px;
                overflow-y: auto;
                padding: 6px;
            }

            .instarank-fields-list::-webkit-scrollbar {
                width: 6px;
            }

            .instarank-fields-list::-webkit-scrollbar-track {
                background: transparent;
            }

            .instarank-fields-list::-webkit-scrollbar-thumb {
                background: #d1d5db;
                border-radius: 3px;
            }

            .instarank-fields-list::-webkit-scrollbar-thumb:hover {
                background: #9ca3af;
            }

            .instarank-field-button {
                display: flex;
                align-items: center;
                gap: 10px;
                padding: 10px 12px;
                background: transparent;
                border: none;
                border-radius: 8px;
                cursor: pointer;
                transition: all 0.15s ease;
                text-align: left;
                width: 100%;
                margin-bottom: 2px;
            }

            .instarank-field-button:hover {
                background: #f3f4f6;
            }

            .instarank-field-button:active,
            .instarank-field-button.clicked {
                background: #e5e7eb;
                transform: scale(0.98);
            }

            .instarank-field-icon {
                flex-shrink: 0;
                width: 28px;
                height: 28px;
                display: flex;
                align-items: center;
                justify-content: center;
                background: #f3f4f6;
                border-radius: 6px;
            }

            .instarank-field-name {
                flex: 1;
                font-size: 13px;
                font-weight: 500;
                color: #1f2937;
                overflow: hidden;
                text-overflow: ellipsis;
                white-space: nowrap;
            }

            .instarank-field-type {
                flex-shrink: 0;
                font-size: 9px;
                font-weight: 600;
                padding: 3px 6px;
                border-radius: 4px;
                text-transform: uppercase;
                letter-spacing: 0.03em;
            }

            .instarank-no-fields {
                text-align: center;
                color: #6b7280;
                padding: 32px 20px;
            }

            .instarank-no-fields svg {
                color: #d1d5db;
                margin-bottom: 12px;
            }

            .instarank-no-fields p {
                margin: 0 0 4px;
                font-weight: 500;
                color: #4b5563;
            }

            .instarank-no-fields small {
                color: #9ca3af;
                font-size: 12px;
            }

            .instarank-panel-footer {
                padding: 12px 16px;
                background: white;
                border-top: 1px solid #f3f4f6;
            }

            .instarank-footer-hint {
                display: flex;
                align-items: center;
                justify-content: center;
                gap: 6px;
                color: #9ca3af;
                font-size: 11px;
            }

            .instarank-notification-icon {
                display: flex;
            }
        `;
        document.head.appendChild(style);
    }

    /**
     * Initialize the panel
     */
    function init() {
        const initPanel = () => {
            createCustomFieldsPanel();
        };

        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => {
                setTimeout(initPanel, 1500);
            });
        } else {
            setTimeout(initPanel, 1500);
        }
    }

    // Start
    init();
})();
