/**
 * InstaRank Classic Editor JavaScript
 */

(function($) {
    'use strict';

    const InstaRankClassicEditor = {
        init: function() {
            this.bindEvents();
            this.updateAllPreviews();
            this.initializeCharCounters();
            this.updateKeywordAnalysis();
        },

        bindEvents: function() {
            // Tab switching
            $('.instarank-tab').on('click', this.switchTab.bind(this));

            // Real-time preview updates
            $('#instarank_meta_title, #post-title-0, #title').on('input', this.updateSERPPreview.bind(this));
            $('#instarank_meta_description').on('input', this.updateSERPPreview.bind(this));

            // Character counters
            $('#instarank_meta_title').on('input', function() {
                InstaRankClassicEditor.updateCharCounter($(this));
            });
            $('#instarank_meta_description').on('input', function() {
                InstaRankClassicEditor.updateCharCounter($(this));
            });

            // Focus keyword analysis
            $('#instarank_focus_keyword, #instarank_meta_title, #instarank_meta_description, #content').on('input',
                this.debounce(this.updateKeywordAnalysis.bind(this), 500)
            );

            // Social preview updates
            $('#instarank_og_title, #instarank_meta_title, #post-title-0, #title').on('input', this.updateFacebookPreview.bind(this));
            $('#instarank_og_description, #instarank_meta_description').on('input', this.updateFacebookPreview.bind(this));
            $('#instarank_og_image').on('change', this.updateFacebookPreview.bind(this));

            $('#instarank_twitter_title, #instarank_og_title, #instarank_meta_title, #post-title-0, #title').on('input', this.updateTwitterPreview.bind(this));
            $('#instarank_twitter_description, #instarank_og_description, #instarank_meta_description').on('input', this.updateTwitterPreview.bind(this));
            $('#instarank_twitter_image').on('change', this.updateTwitterPreview.bind(this));

            // Image selection
            $('.instarank-select-image').on('click', this.selectImage.bind(this));
            $('.instarank-remove-image').on('click', this.removeImage.bind(this));
        },

        switchTab: function(e) {
            e.preventDefault();
            const $button = $(e.currentTarget);
            const tab = $button.data('tab');

            // Update buttons
            $('.instarank-tab').removeClass('active');
            $button.addClass('active');

            // Update content
            $('.instarank-tab-content').removeClass('active');
            $(`.instarank-tab-content[data-content="${tab}"]`).addClass('active');
        },

        updateAllPreviews: function() {
            this.updateSERPPreview();
            this.updateFacebookPreview();
            this.updateTwitterPreview();
        },

        updateSERPPreview: function() {
            const title = $('#instarank_meta_title').val() || $('#title').val() || $('#post-title-0').val() || instarankClassicEditor.postTitle || 'Your Post Title';
            const description = $('#instarank_meta_description').val() || instarankClassicEditor.postExcerpt || 'Your meta description will appear here...';
            const url = instarankClassicEditor.postUrl || window.location.href;

            // Parse URL to get domain
            let displayUrl = url;
            try {
                const urlObj = new URL(url);
                displayUrl = urlObj.hostname + urlObj.pathname;
            } catch (e) {
                // Invalid URL, use as is
            }

            $('.serp-preview-url').text(displayUrl);
            $('.serp-preview-title').text(this.truncate(title, 70));
            $('.serp-preview-description').text(this.truncate(description, 160));
        },

        updateFacebookPreview: function() {
            const title = $('#instarank_og_title').val() || $('#instarank_meta_title').val() || $('#title').val() || $('#post-title-0').val() || instarankClassicEditor.postTitle || 'Your Post Title';
            const description = $('#instarank_og_description').val() || $('#instarank_meta_description').val() || instarankClassicEditor.postExcerpt || 'Your post description will appear here...';
            const imageUrl = $('#instarank_og_image').val();

            const $preview = $('.facebook-preview');
            $preview.find('.social-preview-title').text(this.truncate(title, 100));
            $preview.find('.social-preview-description').text(this.truncate(description, 200));
            $preview.find('.social-preview-url').text(instarankClassicEditor.siteName || 'Your Site');

            const $image = $preview.find('.social-preview-image');
            if (imageUrl) {
                $image.css('background-image', `url(${imageUrl})`).addClass('has-image');
            } else {
                $image.css('background-image', '').removeClass('has-image');
            }
        },

        updateTwitterPreview: function() {
            const title = $('#instarank_twitter_title').val() || $('#instarank_og_title').val() || $('#instarank_meta_title').val() || $('#title').val() || $('#post-title-0').val() || instarankClassicEditor.postTitle || 'Your Post Title';
            const description = $('#instarank_twitter_description').val() || $('#instarank_og_description').val() || $('#instarank_meta_description').val() || instarankClassicEditor.postExcerpt || 'Your post description will appear here...';
            const imageUrl = $('#instarank_twitter_image').val() || $('#instarank_og_image').val();

            const $preview = $('.twitter-preview');
            $preview.find('.social-preview-title').text(this.truncate(title, 70));
            $preview.find('.social-preview-description').text(this.truncate(description, 200));
            $preview.find('.social-preview-url').text(instarankClassicEditor.siteName || 'Your Site');

            const $image = $preview.find('.social-preview-image');
            if (imageUrl) {
                $image.css('background-image', `url(${imageUrl})`).addClass('has-image');
            } else {
                $image.css('background-image', '').removeClass('has-image');
            }
        },

        initializeCharCounters: function() {
            $('#instarank_meta_title').each(function() {
                InstaRankClassicEditor.updateCharCounter($(this));
            });
            $('#instarank_meta_description').each(function() {
                InstaRankClassicEditor.updateCharCounter($(this));
            });
        },

        updateCharCounter: function($input) {
            const $counter = $input.siblings('.instarank-char-counter');
            const currentLength = $input.val().length;
            const maxLength = parseInt($counter.data('max'));
            const optimalLength = parseInt($counter.data('optimal'));

            const percentage = (currentLength / maxLength) * 100;

            let status = 'short';
            if (currentLength > maxLength) {
                status = 'too-long';
            } else if (currentLength >= optimalLength) {
                status = 'good';
            }

            $counter.removeClass('status-short status-good status-too-long').addClass(`status-${status}`);
            $counter.find('.count').text(`${currentLength} / ${maxLength}`);
            $counter.find('.fill').css('width', `${Math.min(percentage, 100)}%`);
        },

        updateKeywordAnalysis: function() {
            const keyword = $('#instarank_focus_keyword').val().toLowerCase().trim();

            if (!keyword) {
                $('.keyword-check').removeClass('found not-found');
                return;
            }

            // Check title
            const title = ($('#instarank_meta_title').val() || $('#title').val() || $('#post-title-0').val() || '').toLowerCase();
            const inTitle = title.includes(keyword);
            $('.keyword-check[data-check="title"]')
                .removeClass('found not-found')
                .addClass(inTitle ? 'found' : 'not-found');

            // Check description
            const description = ($('#instarank_meta_description').val() || '').toLowerCase();
            const inDescription = description.includes(keyword);
            $('.keyword-check[data-check="description"]')
                .removeClass('found not-found')
                .addClass(inDescription ? 'found' : 'not-found');

            // Check content
            let content = '';
            if ($('#content').length) {
                // TinyMCE Classic Editor
                if (typeof tinymce !== 'undefined' && tinymce.get('content')) {
                    content = tinymce.get('content').getContent({format: 'text'}).toLowerCase();
                } else {
                    content = $('#content').val().toLowerCase();
                }
            }
            const inContent = content.includes(keyword);
            $('.keyword-check[data-check="content"]')
                .removeClass('found not-found')
                .addClass(inContent ? 'found' : 'not-found');
        },

        selectImage: function(e) {
            e.preventDefault();
            const $button = $(e.currentTarget);
            const target = $button.data('target');

            // If media frame already exists, reopen it
            if (this.mediaFrame) {
                this.mediaFrame.open();
                return;
            }

            // Create new media frame
            this.mediaFrame = wp.media({
                title: 'Select Image',
                button: {
                    text: 'Use this image'
                },
                multiple: false
            });

            // When image is selected
            this.mediaFrame.on('select', function() {
                const attachment = InstaRankClassicEditor.mediaFrame.state().get('selection').first().toJSON();
                const imageUrl = attachment.url;

                $(`#instarank_${target}`).val(imageUrl).trigger('change');
                $button.siblings('.instarank-remove-image').show();
                $button.siblings('.image-preview').show().find('img').attr('src', imageUrl);
            });

            this.mediaFrame.open();
        },

        removeImage: function(e) {
            e.preventDefault();
            const $button = $(e.currentTarget);
            const target = $button.data('target');

            $(`#instarank_${target}`).val('').trigger('change');
            $button.hide();
            $button.siblings('.image-preview').hide().find('img').attr('src', '');
        },

        truncate: function(str, maxLength) {
            if (str.length <= maxLength) {
                return str;
            }
            return str.substr(0, maxLength) + '...';
        },

        debounce: function(func, wait) {
            let timeout;
            return function executedFunction(...args) {
                const later = () => {
                    clearTimeout(timeout);
                    func(...args);
                };
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
            };
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        if ($('.instarank-classic-metabox').length) {
            InstaRankClassicEditor.init();
        }
    });

})(jQuery);
