/**
 * InstaRank SEO Admin JavaScript
 */

(function($) {
    'use strict';

    // Initialize when document is ready
    $(document).ready(function() {
        InstaRankAdmin.init();
    });

    const InstaRankAdmin = {
        init: function() {
            this.createConfirmModal();
            this.bindEvents();
            this.initTooltips();
            this.initTabs();
            this.initCheckboxSelection();
            this.loadProgrammaticSEOStats();
        },

        /**
         * Create the confirmation modal HTML
         */
        createConfirmModal: function() {
            if ($('#instarank-confirm-modal').length) {
                return;
            }

            const modalHtml = `
                <div id="instarank-confirm-modal" class="ir-modal-overlay" style="display: none;">
                    <div class="ir-modal-container">
                        <div class="ir-modal-header">
                            <span class="ir-modal-icon"></span>
                            <h3 class="ir-modal-title"></h3>
                            <button type="button" class="ir-modal-close-btn" aria-label="Close">
                                <span class="dashicons dashicons-no-alt"></span>
                            </button>
                        </div>
                        <div class="ir-modal-body">
                            <p class="ir-modal-message"></p>
                            <p class="ir-modal-description"></p>
                        </div>
                        <div class="ir-modal-footer">
                            <button type="button" class="ir-btn ir-modal-cancel">Cancel</button>
                            <button type="button" class="ir-btn ir-modal-confirm">Confirm</button>
                        </div>
                    </div>
                </div>
            `;

            $('body').append(modalHtml);

            // Add modal styles
            const styles = `
                <style id="instarank-modal-styles">
                    .ir-modal-overlay {
                        position: fixed;
                        top: 0;
                        left: 0;
                        right: 0;
                        bottom: 0;
                        background: rgba(0, 0, 0, 0.6);
                        z-index: 100000;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                        animation: ir-fadeIn 0.2s ease;
                    }
                    @keyframes ir-fadeIn {
                        from { opacity: 0; }
                        to { opacity: 1; }
                    }
                    @keyframes ir-slideIn {
                        from { opacity: 0; transform: scale(0.95) translateY(-10px); }
                        to { opacity: 1; transform: scale(1) translateY(0); }
                    }
                    .ir-modal-container {
                        background: #fff;
                        border-radius: 12px;
                        box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
                        max-width: 480px;
                        width: 90%;
                        animation: ir-slideIn 0.25s ease;
                        overflow: hidden;
                    }
                    .ir-modal-header {
                        display: flex;
                        align-items: center;
                        padding: 20px 24px;
                        border-bottom: 1px solid #e5e7eb;
                        gap: 12px;
                    }
                    .ir-modal-icon {
                        width: 40px;
                        height: 40px;
                        border-radius: 50%;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                        flex-shrink: 0;
                    }
                    .ir-modal-icon .dashicons {
                        font-size: 24px;
                        width: 24px;
                        height: 24px;
                    }
                    .ir-modal-icon--warning {
                        background: #fef3c7;
                        color: #d97706;
                    }
                    .ir-modal-icon--danger {
                        background: #fee2e2;
                        color: #dc2626;
                    }
                    .ir-modal-icon--info {
                        background: #dbeafe;
                        color: #2563eb;
                    }
                    .ir-modal-icon--success {
                        background: #d1fae5;
                        color: #059669;
                    }
                    .ir-modal-title {
                        margin: 0;
                        font-size: 18px;
                        font-weight: 600;
                        color: #111827;
                        flex: 1;
                    }
                    .ir-modal-close-btn {
                        background: none;
                        border: none;
                        padding: 4px;
                        cursor: pointer;
                        color: #6b7280;
                        border-radius: 6px;
                        transition: all 0.15s;
                    }
                    .ir-modal-close-btn:hover {
                        background: #f3f4f6;
                        color: #374151;
                    }
                    .ir-modal-close-btn .dashicons {
                        font-size: 20px;
                        width: 20px;
                        height: 20px;
                    }
                    .ir-modal-body {
                        padding: 20px 24px;
                    }
                    .ir-modal-message {
                        margin: 0 0 8px 0;
                        font-size: 15px;
                        color: #374151;
                        line-height: 1.5;
                    }
                    .ir-modal-description {
                        margin: 0;
                        font-size: 14px;
                        color: #6b7280;
                        line-height: 1.5;
                    }
                    .ir-modal-footer {
                        display: flex;
                        justify-content: flex-end;
                        gap: 12px;
                        padding: 16px 24px;
                        background: #f9fafb;
                        border-top: 1px solid #e5e7eb;
                    }
                    .ir-modal-cancel {
                        background: #fff !important;
                        border: 1px solid #d1d5db !important;
                        color: #374151 !important;
                    }
                    .ir-modal-cancel:hover {
                        background: #f9fafb !important;
                        border-color: #9ca3af !important;
                    }
                    .ir-modal-confirm--danger {
                        background: #dc2626 !important;
                        border-color: #dc2626 !important;
                        color: #fff !important;
                    }
                    .ir-modal-confirm--danger:hover {
                        background: #b91c1c !important;
                        border-color: #b91c1c !important;
                    }
                    .ir-modal-confirm--warning {
                        background: #d97706 !important;
                        border-color: #d97706 !important;
                        color: #fff !important;
                    }
                    .ir-modal-confirm--warning:hover {
                        background: #b45309 !important;
                        border-color: #b45309 !important;
                    }
                    .ir-modal-confirm--primary {
                        background: #2563eb !important;
                        border-color: #2563eb !important;
                        color: #fff !important;
                    }
                    .ir-modal-confirm--primary:hover {
                        background: #1d4ed8 !important;
                        border-color: #1d4ed8 !important;
                    }
                </style>
            `;

            if (!$('#instarank-modal-styles').length) {
                $('head').append(styles);
            }
        },

        /**
         * Show confirmation modal
         * @param {Object} options - Modal options
         * @returns {Promise} - Resolves true if confirmed, false if cancelled
         */
        showConfirm: function(options) {
            return new Promise(function(resolve) {
                const defaults = {
                    title: 'Confirm Action',
                    message: 'Are you sure you want to proceed?',
                    description: '',
                    type: 'warning', // warning, danger, info, success
                    confirmText: 'Confirm',
                    cancelText: 'Cancel'
                };

                const settings = $.extend({}, defaults, options);
                const modal = $('#instarank-confirm-modal');

                // Set icon based on type
                const iconClass = {
                    warning: 'dashicons-warning',
                    danger: 'dashicons-dismiss',
                    info: 'dashicons-info',
                    success: 'dashicons-yes-alt'
                };

                modal.find('.ir-modal-icon')
                    .removeClass('ir-modal-icon--warning ir-modal-icon--danger ir-modal-icon--info ir-modal-icon--success')
                    .addClass('ir-modal-icon--' + settings.type)
                    .html('<span class="dashicons ' + iconClass[settings.type] + '"></span>');

                modal.find('.ir-modal-title').text(settings.title);
                modal.find('.ir-modal-message').text(settings.message);
                modal.find('.ir-modal-description').html(settings.description || '').toggle(!!settings.description);
                modal.find('.ir-modal-confirm')
                    .text(settings.confirmText)
                    .removeClass('ir-modal-confirm--danger ir-modal-confirm--warning ir-modal-confirm--primary')
                    .addClass('ir-modal-confirm--' + settings.type);
                modal.find('.ir-modal-cancel').text(settings.cancelText);

                // Show modal
                modal.fadeIn(200);

                // Bind events
                const cleanup = function() {
                    modal.find('.ir-modal-confirm').off('click.confirm');
                    modal.find('.ir-modal-cancel').off('click.confirm');
                    modal.find('.ir-modal-close-btn').off('click.confirm');
                    modal.off('click.confirm');
                    $(document).off('keydown.confirm');
                };

                modal.find('.ir-modal-confirm').on('click.confirm', function() {
                    cleanup();
                    modal.fadeOut(150);
                    resolve(true);
                });

                modal.find('.ir-modal-cancel, .ir-modal-close-btn').on('click.confirm', function() {
                    cleanup();
                    modal.fadeOut(150);
                    resolve(false);
                });

                // Close on overlay click
                modal.on('click.confirm', function(e) {
                    if ($(e.target).is('.ir-modal-overlay')) {
                        cleanup();
                        modal.fadeOut(150);
                        resolve(false);
                    }
                });

                // Close on Escape key
                $(document).on('keydown.confirm', function(e) {
                    if (e.key === 'Escape') {
                        cleanup();
                        modal.fadeOut(150);
                        resolve(false);
                    }
                });
            });
        },

        bindEvents: function() {
            // Copy API Key
            $('#instarank-copy-key').on('click', this.copyApiKey);

            // Copy Sitemap URL
            $('#instarank-copy-sitemap-url').on('click', this.copySitemapUrl);

            // Regenerate API Key
            $('#instarank-regenerate-key').on('click', this.regenerateApiKey);

            // Connect with Popup (OAuth-style)
            $('#instarank-connect-popup').on('click', this.connectWithPopup);

            // Test Connection
            $('#instarank-test-connection').on('click', this.testConnection);

            // Check Connection Status
            $('#instarank-check-connection').on('click', this.checkConnectionStatus);

            // Sync Now
            $('#instarank-sync-now').on('click', this.syncNow);

            // Approve Change
            $('.instarank-approve-change').on('click', this.approveChange);

            // Reject Change
            $('.instarank-reject-change').on('click', this.rejectChange);

            // Rollback Change
            $('.instarank-rollback-change').on('click', this.rollbackChange);

            // View Details
            $('.instarank-view-details').on('click', this.viewChangeDetails);

            // Select All Checkboxes
            $('#instarank-select-all').on('change', this.toggleSelectAll);

            // Disconnect
            $('#instarank-disconnect').on('click', this.disconnect);

            // Reset Auth Attempts
            $('#instarank-reset-auth-attempts').on('click', this.resetAuthAttempts);

            // Clear History
            $('#instarank-clear-history').on('click', this.clearHistory);

            // Reset Robots.txt
            $('#instarank-reset-robots').on('click', this.resetRobotsTxt);

            // Close Modal
            $('.instarank-modal-close').on('click', this.closeModal);

            // Close modal when clicking outside
            $(window).on('click', function(e) {
                if ($(e.target).hasClass('instarank-modal')) {
                    InstaRankAdmin.closeModal();
                }
            });
        },

        initTooltips: function() {
            // Add tooltips to elements with data-tooltip attribute
            $('[data-tooltip]').each(function() {
                $(this).attr('title', $(this).data('tooltip'));
            });
        },

        copyApiKey: function() {
            InstaRankAdmin.copyToClipboard('#instarank-api-key', '#instarank-copy-key', 'API key');
        },

        copySitemapUrl: function() {
            InstaRankAdmin.copyToClipboard('#sitemap_url', '#instarank-copy-sitemap-url', 'Sitemap URL');
        },

        copyToClipboard: function(inputSelector, buttonSelector, label) {
            const value = $(inputSelector).val();
            const button = $(buttonSelector);

            navigator.clipboard.writeText(value).then(function() {
                const originalText = button.html();
                button.html('<span class="dashicons dashicons-yes"></span> Copied!');
                button.css({
                    'background': '#059669',
                    'border-color': '#059669',
                    'color': '#fff'
                });

                // Add subtle animation
                button.css('transform', 'scale(1.05)');

                setTimeout(function() {
                    button.css('transform', 'scale(1)');
                }, 100);

                setTimeout(function() {
                    button.html(originalText);
                    button.css({
                        'background': '',
                        'border-color': '',
                        'color': ''
                    });
                }, 2000);
            }).catch(function() {
                InstaRankAdmin.showNotice('Failed to copy ' + label + '. Please copy it manually.', 'error');
            });
        },

        regenerateApiKey: async function() {
            const confirmed = await InstaRankAdmin.showConfirm({
                title: 'Regenerate API Key?',
                message: 'This will create a new API key and disconnect your current integration.',
                description: 'You will need to reconnect using the new API key from your InstaRank dashboard.',
                type: 'warning',
                confirmText: 'Regenerate Key',
                cancelText: 'Keep Current'
            });

            if (!confirmed) return;

            const button = $(this);
            button.prop('disabled', true).addClass('updating-message');

            $.post(ajaxurl, {
                action: 'instarank_regenerate_key',
                nonce: instarankAdmin.nonce
            }, function(response) {
                if (response.success) {
                    $('#instarank-api-key').val(response.data.api_key);
                    InstaRankAdmin.showNotice('API key regenerated successfully. Please reconnect from your InstaRank dashboard.', 'success');
                } else {
                    InstaRankAdmin.showNotice('Failed to regenerate API key: ' + response.data.message, 'error');
                }
            }).always(function() {
                button.prop('disabled', false).removeClass('updating-message');
            });
        },

        connectWithPopup: function() {
            const button = $(this);
            const apiKey = instarankAdmin.apiKey || $('#instarank-api-key').val();
            const siteUrl = instarankAdmin.siteUrl;

            button.prop('disabled', true).addClass('updating-message');

            // Call InstaRank API to initiate OAuth flow
            const instarankApiUrl = (instarankAdmin.apiUrl || 'https://app.instarank.com').replace('/v1', '');

            $.ajax({
                url: instarankApiUrl + '/api/wordpress-oauth/init',
                method: 'POST',
                contentType: 'application/json',
                data: JSON.stringify({
                    siteUrl: siteUrl,
                    apiKey: apiKey,
                    returnUrl: window.location.href
                }),
                success: function(response) {
                    if (response.success && response.oauthUrl) {
                        // Open popup window
                        const width = 600;
                        const height = 700;
                        const left = (screen.width / 2) - (width / 2);
                        const top = (screen.height / 2) - (height / 2);

                        const popup = window.open(
                            response.oauthUrl,
                            'InstaRank Connection',
                            'width=' + width + ',height=' + height + ',top=' + top + ',left=' + left + ',resizable=yes,scrollbars=yes'
                        );

                        // Listen for messages from popup
                        window.addEventListener('message', function(event) {
                            // In production, verify event.origin matches your InstaRank domain
                            if (event.data && event.data.type === 'wordpress-oauth-success') {
                                // Close popup if still open
                                if (popup && !popup.closed) {
                                    popup.close();
                                }

                                // Update local WordPress options
                                const integration = event.data.integration;

                                // Call WordPress to update connection status
                                $.post(ajaxurl, {
                                    action: 'instarank_confirm_oauth_connection',
                                    nonce: instarankAdmin.nonce,
                                    integration: integration
                                }, function(updateResponse) {
                                    if (updateResponse.success) {
                                        InstaRankAdmin.showNotice('Successfully connected to InstaRank! Refreshing...', 'success');
                                        setTimeout(function() {
                                            location.reload();
                                        }, 1500);
                                    } else {
                                        InstaRankAdmin.showNotice('Connection established but failed to update local settings. Please refresh the page.', 'warning');
                                    }
                                });
                            }
                        });

                        // Check if popup was blocked
                        setTimeout(function() {
                            if (!popup || popup.closed || typeof popup.closed === 'undefined') {
                                InstaRankAdmin.showNotice('Popup was blocked. Please allow popups for this site and try again.', 'error');
                                button.prop('disabled', false).removeClass('updating-message');
                            }
                        }, 1000);
                    } else {
                        InstaRankAdmin.showNotice('Failed to initiate connection: ' + (response.error || 'Unknown error'), 'error');
                        button.prop('disabled', false).removeClass('updating-message');
                    }
                },
                error: function(xhr) {
                    let errorMsg = 'Failed to connect to InstaRank server.';
                    if (xhr.responseJSON && xhr.responseJSON.error) {
                        errorMsg = xhr.responseJSON.error;
                    }
                    InstaRankAdmin.showNotice(errorMsg, 'error');
                    button.prop('disabled', false).removeClass('updating-message');
                }
            });
        },

        testConnection: function() {
            const button = $(this);
            button.prop('disabled', true).addClass('updating-message');

            $.post(ajaxurl, {
                action: 'instarank_test_connection',
                nonce: instarankAdmin.nonce
            }, function(response) {
                if (response.success) {
                    InstaRankAdmin.showNotice('Connection test successful!', 'success');
                } else {
                    InstaRankAdmin.showNotice('Connection test failed: ' + response.data.message, 'error');
                }
            }).always(function() {
                button.prop('disabled', false).removeClass('updating-message');
            });
        },

        checkConnectionStatus: function() {
            const button = $(this);
            const originalHtml = button.html();
            button.prop('disabled', true).html('<span class="dashicons dashicons-update dashicons-spin"></span> Checking...');

            $.post(ajaxurl, {
                action: 'instarank_check_connection_status',
                nonce: instarankAdmin.nonce
            }, function(response) {
                if (response.success) {
                    if (response.data.connected) {
                        InstaRankAdmin.showNotice('Connected! Your WordPress site is linked to InstaRank. Refreshing...', 'success');
                        setTimeout(function() {
                            location.reload();
                        }, 1500);
                    } else {
                        InstaRankAdmin.showNotice('Not connected yet. Please connect from your InstaRank dashboard using the API key below.', 'info');
                    }
                } else {
                    InstaRankAdmin.showNotice('Failed to check connection: ' + response.data.message, 'error');
                }
            }).always(function() {
                button.prop('disabled', false).html(originalHtml);
            });
        },

        syncNow: function() {
            const button = $(this);
            button.prop('disabled', true).addClass('updating-message');

            $.post(ajaxurl, {
                action: 'instarank_sync_now',
                nonce: instarankAdmin.nonce
            }, function(response) {
                if (response.success) {
                    InstaRankAdmin.showNotice('Sync completed! Synced ' + response.data.count + ' posts.', 'success');
                    setTimeout(function() {
                        location.reload();
                    }, 1500);
                } else {
                    InstaRankAdmin.showNotice('Sync failed: ' + response.data.message, 'error');
                }
            }).always(function() {
                button.prop('disabled', false).removeClass('updating-message');
            });
        },

        approveChange: async function() {
            const changeId = $(this).data('change-id');
            const row = $(this).closest('tr');
            const button = $(this);

            const confirmed = await InstaRankAdmin.showConfirm({
                title: 'Apply SEO Change?',
                message: 'This will update your post with the recommended SEO improvements.',
                description: 'The change will be applied immediately and can be rolled back later if needed.',
                type: 'info',
                confirmText: 'Apply Change',
                cancelText: 'Not Now'
            });

            if (!confirmed) return;

            button.prop('disabled', true).addClass('updating-message');

            $.post(ajaxurl, {
                action: 'instarank_approve_change',
                change_id: changeId,
                nonce: instarankAdmin.nonce
            }, function(response) {
                if (response.success) {
                    InstaRankAdmin.showNotice('Change applied successfully!', 'success');
                    row.fadeOut(300, function() {
                        $(this).remove();
                    });
                } else {
                    InstaRankAdmin.showNotice('Failed to apply change: ' + response.data.message, 'error');
                    button.prop('disabled', false).removeClass('updating-message');
                }
            });
        },

        rejectChange: async function() {
            const changeId = $(this).data('change-id');
            const row = $(this).closest('tr');
            const button = $(this);

            const confirmed = await InstaRankAdmin.showConfirm({
                title: 'Reject This Change?',
                message: 'This SEO recommendation will be discarded.',
                description: 'You can always request new recommendations later from your InstaRank dashboard.',
                type: 'warning',
                confirmText: 'Reject Change',
                cancelText: 'Keep It'
            });

            if (!confirmed) return;

            button.prop('disabled', true).addClass('updating-message');

            $.post(ajaxurl, {
                action: 'instarank_reject_change',
                change_id: changeId,
                nonce: instarankAdmin.nonce
            }, function(response) {
                if (response.success) {
                    InstaRankAdmin.showNotice('Change rejected.', 'success');
                    row.fadeOut(300, function() {
                        $(this).remove();
                    });
                } else {
                    InstaRankAdmin.showNotice('Failed to reject change: ' + response.data.message, 'error');
                    button.prop('disabled', false).removeClass('updating-message');
                }
            });
        },

        rollbackChange: async function() {
            const changeId = $(this).data('change-id');
            const button = $(this);

            const confirmed = await InstaRankAdmin.showConfirm({
                title: 'Rollback This Change?',
                message: 'This will restore the previous value before this SEO change was applied.',
                description: 'Your content will be reverted to its original state.',
                type: 'warning',
                confirmText: 'Rollback',
                cancelText: 'Keep Current'
            });

            if (!confirmed) return;

            button.prop('disabled', true).addClass('updating-message');

            $.post(ajaxurl, {
                action: 'instarank_rollback_change',
                change_id: changeId,
                nonce: instarankAdmin.nonce
            }, function(response) {
                if (response.success) {
                    InstaRankAdmin.showNotice('Change rolled back successfully!', 'success');
                    setTimeout(function() {
                        location.reload();
                    }, 1500);
                } else {
                    InstaRankAdmin.showNotice('Failed to rollback change: ' + response.data.message, 'error');
                    button.prop('disabled', false).removeClass('updating-message');
                }
            });
        },

        viewChangeDetails: function() {
            const changeId = $(this).data('change-id');

            $.post(ajaxurl, {
                action: 'instarank_get_change_details',
                change_id: changeId,
                nonce: instarankAdmin.nonce
            }, function(response) {
                if (response.success) {
                    InstaRankAdmin.showChangeDetails(response.data.change);
                } else {
                    InstaRankAdmin.showNotice('Failed to load change details.', 'error');
                }
            });
        },

        showChangeDetails: function(change) {
            const html = `
                <h2>Change Details</h2>
                <table class="widefat">
                    <tr>
                        <th>Post Title:</th>
                        <td><strong>${change.post_title}</strong></td>
                    </tr>
                    <tr>
                        <th>Post URL:</th>
                        <td><a href="${change.post_url}" target="_blank">${change.post_url}</a></td>
                    </tr>
                    <tr>
                        <th>Change Type:</th>
                        <td>${change.change_type.replace('_', ' ')}</td>
                    </tr>
                    <tr>
                        <th>Old Value:</th>
                        <td><div style="max-height: 200px; overflow-y: auto;">${change.old_value || '(empty)'}</div></td>
                    </tr>
                    <tr>
                        <th>New Value:</th>
                        <td><div style="max-height: 200px; overflow-y: auto; color: #0e5c2d; font-weight: 500;">${change.new_value || '(empty)'}</div></td>
                    </tr>
                    <tr>
                        <th>Reason:</th>
                        <td>${change.reason || '(none provided)'}</td>
                    </tr>
                    <tr>
                        <th>Status:</th>
                        <td><span class="instarank-status-badge instarank-status-${change.status}">${change.status}</span></td>
                    </tr>
                    <tr>
                        <th>Created:</th>
                        <td>${change.created_at}</td>
                    </tr>
                    ${change.applied_at ? `<tr><th>Applied:</th><td>${change.applied_at}</td></tr>` : ''}
                </table>
            `;

            $('#instarank-modal-body').html(html);
            $('#instarank-change-modal').fadeIn(200);
        },

        toggleSelectAll: function() {
            $('.instarank-change-checkbox').prop('checked', $(this).prop('checked'));
        },

        disconnect: async function() {
            const button = $(this);

            const confirmed = await InstaRankAdmin.showConfirm({
                title: 'Disconnect from InstaRank?',
                message: 'This will disconnect your WordPress site from InstaRank and clear all programmatic SEO data.',
                description: 'Your API key will remain the same, so you can reconnect anytime. All field mappings, datasets, and template configurations will be cleared.',
                type: 'warning',
                confirmText: 'Disconnect',
                cancelText: 'Stay Connected'
            });

            if (!confirmed) return;

            button.prop('disabled', true).addClass('updating-message');

            $.post(ajaxurl, {
                action: 'instarank_disconnect',
                nonce: instarankAdmin.nonce
            }, function(response) {
                if (response.success) {
                    InstaRankAdmin.showNotice('Disconnected successfully. All data has been cleared.', 'success');
                    setTimeout(function() {
                        location.reload();
                    }, 1500);
                } else {
                    InstaRankAdmin.showNotice('Failed to disconnect: ' + response.data.message, 'error');
                    button.prop('disabled', false).removeClass('updating-message');
                }
            });
        },

        resetAuthAttempts: async function() {
            const button = $(this);

            const confirmed = await InstaRankAdmin.showConfirm({
                title: 'Reset Authentication Counter?',
                message: 'This will clear all failed authentication attempts.',
                description: 'Use this if you have been locked out due to too many failed API requests.',
                type: 'info',
                confirmText: 'Reset Counter',
                cancelText: 'Cancel'
            });

            if (!confirmed) return;

            button.prop('disabled', true).addClass('updating-message');

            $.post(ajaxurl, {
                action: 'instarank_reset_auth_attempts',
                nonce: instarankAdmin.nonce
            }, function(response) {
                if (response.success) {
                    InstaRankAdmin.showNotice('Authentication counter reset successfully.', 'success');
                } else {
                    InstaRankAdmin.showNotice('Failed to reset counter.', 'error');
                }
            }).always(function() {
                button.prop('disabled', false).removeClass('updating-message');
            });
        },

        clearHistory: async function() {
            const button = $(this);

            // First confirmation
            const firstConfirm = await InstaRankAdmin.showConfirm({
                title: 'Clear All Change History?',
                message: 'This will permanently delete ALL SEO change records from your database.',
                description: '<strong style="color: #dc2626;">Warning:</strong> This action cannot be undone. You will lose all history of applied, pending, and rejected changes.',
                type: 'danger',
                confirmText: 'Yes, Delete All',
                cancelText: 'Cancel'
            });

            if (!firstConfirm) return;

            // Second confirmation for extra safety
            const secondConfirm = await InstaRankAdmin.showConfirm({
                title: 'Final Confirmation',
                message: 'Are you absolutely sure? This is your last chance to cancel.',
                description: 'All change records will be permanently deleted. This includes rollback history.',
                type: 'danger',
                confirmText: 'Delete Everything',
                cancelText: 'Go Back'
            });

            if (!secondConfirm) return;

            button.prop('disabled', true).addClass('updating-message');

            $.post(ajaxurl, {
                action: 'instarank_clear_history',
                nonce: instarankAdmin.nonce
            }, function(response) {
                if (response.success) {
                    InstaRankAdmin.showNotice('All change history has been cleared.', 'success');
                    setTimeout(function() {
                        location.reload();
                    }, 1500);
                } else {
                    InstaRankAdmin.showNotice('Failed to clear history: ' + response.data.message, 'error');
                    button.prop('disabled', false).removeClass('updating-message');
                }
            });
        },

        resetRobotsTxt: async function() {
            const confirmed = await InstaRankAdmin.showConfirm({
                title: 'Reset robots.txt?',
                message: 'This will restore your robots.txt to the default WordPress content.',
                description: 'Your current custom robots.txt configuration will be replaced.',
                type: 'warning',
                confirmText: 'Reset to Default',
                cancelText: 'Keep Current'
            });

            if (!confirmed) return;

            $.post(ajaxurl, {
                action: 'instarank_reset_robots_txt',
                nonce: instarankAdmin.nonce
            }, function(response) {
                if (response.success) {
                    $('#robots_txt_content').val(response.data.content);
                    InstaRankAdmin.showNotice('Robots.txt reset to default.', 'success');
                } else {
                    InstaRankAdmin.showNotice('Failed to reset robots.txt.', 'error');
                }
            });
        },

        closeModal: function() {
            $('#instarank-change-modal').fadeOut(200);
        },

        showNotice: function(message, type) {
            type = type || 'success';
            const noticeClass = type === 'error' ? 'notice-error' : (type === 'warning' ? 'notice-warning' : (type === 'info' ? 'notice-info' : 'notice-success'));

            // Create notice with icon
            const icons = {
                error: 'warning',
                warning: 'warning',
                info: 'info',
                success: 'yes-alt'
            };
            const icon = icons[type] || 'yes-alt';
            const notice = $('<div class="notice ' + noticeClass + ' is-dismissible"><p><span class="dashicons dashicons-' + icon + '" style="margin-right: 8px;"></span>' + message + '</p><button type="button" class="notice-dismiss"><span class="screen-reader-text">Dismiss</span></button></div>');

            // Remove any existing notices
            $('.wrap > .notice.is-dismissible').fadeOut(200, function() {
                $(this).remove();
            });

            // Add new notice
            $('.wrap h1').first().after(notice);

            // Make dismissible
            notice.find('.notice-dismiss').on('click', function() {
                notice.fadeOut(200, function() {
                    $(this).remove();
                });
            });

            // Auto-dismiss after 5 seconds
            setTimeout(function() {
                if (notice.is(':visible')) {
                    notice.fadeOut(300, function() {
                        $(this).remove();
                    });
                }
            }, 5000);

            // Smooth scroll to notice
            $('html, body').animate({
                scrollTop: notice.offset().top - 60
            }, 400, 'swing');
        },

        /**
         * Initialize tab switching for Settings page
         */
        initTabs: function() {
            $('.ir-tab').on('click', function() {
                // Don't interfere with tabs that use onclick navigation
                if ($(this).attr('onclick')) {
                    return;
                }

                const tabId = $(this).data('tab');

                // Remove active class from all tabs
                $('.ir-tab').removeClass('ir-tab--active');
                $(this).addClass('ir-tab--active');

                // Hide all panels
                $('.ir-tab-panel').removeClass('active').hide();

                // Show selected panel
                $('.ir-tab-panel[data-panel="' + tabId + '"]').addClass('active').show();
            });
        },

        /**
         * Initialize checkbox selection for Changes table
         */
        initCheckboxSelection: function() {
            const self = this;

            // Select all checkbox
            $('#instarank-select-all').on('change', function() {
                $('.instarank-change-checkbox').prop('checked', $(this).prop('checked'));
                self.updateSelectedCount();
            });

            // Individual checkboxes
            $(document).on('change', '.instarank-change-checkbox', function() {
                self.updateSelectedCount();

                // Update select-all checkbox state
                const totalCheckboxes = $('.instarank-change-checkbox').length;
                const checkedCheckboxes = $('.instarank-change-checkbox:checked').length;

                $('#instarank-select-all').prop('checked', totalCheckboxes === checkedCheckboxes);
            });
        },

        /**
         * Update selected count display
         */
        updateSelectedCount: function() {
            const count = $('.instarank-change-checkbox:checked').length;
            $('.ir-selected-count').text('Selected: ' + count);
        },

        /**
         * Load Programmatic SEO stats from SaaS API
         */
        loadProgrammaticSEOStats: function() {
            // Only load on dashboard page
            if (!$('#pseo-templates-count').length) {
                return;
            }

            const apiKey = instarankAdmin.apiKey;
            if (!apiKey) {
                console.log('[InstaRank] No API key found, skipping stats load');
                this.showStatsError();
                return;
            }

            const apiUrl = instarankAdmin.apiUrl + '/api/programmatic-seo/stats';

            $.ajax({
                url: apiUrl,
                method: 'GET',
                headers: {
                    'X-WordPress-API-Key': apiKey
                },
                success: function(response) {
                    if (response.success) {
                        InstaRankAdmin.updateProgrammaticSEOStats(response);
                    } else {
                        InstaRankAdmin.showStatsError();
                    }
                },
                error: function(xhr, status, error) {
                    console.error('[InstaRank] Failed to load P-SEO stats:', error);
                    InstaRankAdmin.showStatsError();
                }
            });
        },

        /**
         * Update Programmatic SEO stats in the dashboard
         */
        updateProgrammaticSEOStats: function(data) {
            // Update stat numbers
            $('#pseo-templates-count').text(data.templates.configured || 0);
            $('#pseo-fields-count').text(data.fields.totalMapped || 0);
            $('#pseo-columns-count').text(data.dataset.columnCount || 0);

            // Update status badge
            const $badge = $('#pseo-status-badge');
            $badge.removeClass('ir-badge--success ir-badge--warning');

            if (data.isReady) {
                $badge.addClass('ir-badge--success');
                $badge.html('<span class="ir-badge__dot"></span> Ready');
            } else if (data.dataset.hasDataset || data.templates.configured > 0) {
                $badge.addClass('ir-badge--warning');
                $badge.html('<span class="ir-badge__dot"></span> Setup Incomplete');
            } else {
                $badge.css({'background': '#f5f5f5', 'color': '#666'});
                $badge.text('Not Configured');
            }

            // Update info section
            const $info = $('#pseo-info');
            let infoHtml = '';

            if (data.dataset.hasDataset) {
                infoHtml += '<p style="margin: 0 0 8px 0;">';
                infoHtml += '<strong>Dataset:</strong> ';
                infoHtml += '<span style="color: #10B981;">' + this.escapeHtml(data.dataset.name) + '</span>';
                infoHtml += '</p>';
            } else {
                infoHtml += '<p style="margin: 0 0 8px 0; color: #dba617;">';
                infoHtml += '<span class="dashicons dashicons-warning" style="font-size: 16px; vertical-align: middle;"></span> ';
                infoHtml += 'No dataset linked';
                infoHtml += '</p>';
            }

            if (data.templates.configured > 0) {
                infoHtml += '<p style="margin: 0; color: #10B981;">';
                infoHtml += '<span class="dashicons dashicons-yes" style="font-size: 16px; vertical-align: middle;"></span> ';
                infoHtml += data.templates.configured + ' template' + (data.templates.configured > 1 ? 's' : '') + ' configured';
                infoHtml += '</p>';
            } else {
                infoHtml += '<p style="margin: 0; color: #888;">';
                infoHtml += 'No templates configured yet';
                infoHtml += '</p>';
            }

            $info.html(infoHtml);
        },

        /**
         * Show error state for stats
         */
        showStatsError: function() {
            $('#pseo-templates-count').text('—');
            $('#pseo-fields-count').text('—');
            $('#pseo-columns-count').text('—');

            const $badge = $('#pseo-status-badge');
            $badge.css({'background': '#f5f5f5', 'color': '#666'});
            $badge.text('Not Configured');

            $('#pseo-info').html('<p style="margin: 0; color: #888;">Unable to load stats</p>');
        },

        /**
         * Escape HTML to prevent XSS
         */
        escapeHtml: function(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    };

    // Make InstaRankAdmin available globally
    window.InstaRankAdmin = InstaRankAdmin;

})(jQuery);
