<?php
/**
 * REST API Endpoints
 * Handles all REST API routes for InstaRank integration
 */

defined('ABSPATH') || exit;

class InstaRank_API_Endpoints {

    public function __construct() {
        add_action('rest_api_init', [$this, 'register_routes']);
    }

    /**
     * Register REST API routes
     */
    public function register_routes() {
        // Apply changes endpoint
        register_rest_route('instarank/v1', '/changes/apply', [
            'methods' => 'POST',
            'callback' => [$this, 'apply_changes'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get changes endpoint
        register_rest_route('instarank/v1', '/changes', [
            'methods' => 'GET',
            'callback' => [$this, 'get_changes'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get change status endpoint
        register_rest_route('instarank/v1', '/changes/(?P<id>\d+)', [
            'methods' => 'GET',
            'callback' => [$this, 'get_change_status'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Bulk approve/reject changes endpoint
        register_rest_route('instarank/v1', '/changes/bulk', [
            'methods' => 'POST',
            'callback' => [$this, 'bulk_changes'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Test connection endpoint
        register_rest_route('instarank/v1', '/test', [
            'methods' => 'GET',
            'callback' => [$this, 'test_connection'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get site info endpoint
        register_rest_route('instarank/v1', '/info', [
            'methods' => 'GET',
            'callback' => [$this, 'get_site_info'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get posts endpoint
        register_rest_route('instarank/v1', '/posts', [
            'methods' => 'GET',
            'callback' => [$this, 'get_posts'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Resolve a URL to a WordPress post ID (handles front page)
        register_rest_route('instarank/v1', '/posts/resolve', [
            'methods' => 'GET',
            'callback' => [$this, 'resolve_post_by_url'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Update homepage SEO meta (for when homepage shows latest posts)
        register_rest_route('instarank/v1', '/home/meta', [
            'methods' => 'POST',
            'callback' => [$this, 'update_home_meta'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Health check endpoint (intentionally public)
        // This endpoint is publicly accessible as it only returns non-sensitive information
        // (plugin version and timestamp) for monitoring and debugging purposes
        register_rest_route('instarank/v1', '/health', [
            'methods' => 'GET',
            'callback' => [$this, 'health_check'],
            'permission_callback' => '__return_true' // Public endpoint: returns only plugin version and timestamp
        ]);

        // Connection confirmation endpoint (called by InstaRank when integration is established)
        register_rest_route('instarank/v1', '/connection/confirm', [
            'methods' => 'POST',
            'callback' => [$this, 'confirm_connection'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Dynamic Elements endpoint - render a dynamic element
        register_rest_route('instarank/v1', '/dynamic-elements/render', [
            'methods' => 'POST',
            'callback' => [$this, 'render_dynamic_element'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // NEW: Page Types Support Endpoints

        // Get homepage configuration
        register_rest_route('instarank/v1', '/homepage', [
            'methods' => 'GET',
            'callback' => [$this, 'get_homepage_info'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get all categories
        register_rest_route('instarank/v1', '/categories', [
            'methods' => 'GET',
            'callback' => [$this, 'get_categories'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get all tags
        register_rest_route('instarank/v1', '/tags', [
            'methods' => 'GET',
            'callback' => [$this, 'get_tags'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get all authors
        register_rest_route('instarank/v1', '/authors', [
            'methods' => 'GET',
            'callback' => [$this, 'get_authors'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get all custom post types
        register_rest_route('instarank/v1', '/post-types', [
            'methods' => 'GET',
            'callback' => [$this, 'get_post_types'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Register a custom post type from InstaRank
        register_rest_route('instarank/v1', '/programmatic/post-types', [
            'methods' => 'POST',
            'callback' => [$this, 'register_custom_post_type'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Update a custom post type from InstaRank
        register_rest_route('instarank/v1', '/programmatic/post-types/(?P<slug>[a-z0-9_-]+)', [
            'methods' => 'PUT',
            'callback' => [$this, 'update_custom_post_type'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Delete a custom post type from InstaRank
        register_rest_route('instarank/v1', '/programmatic/post-types/(?P<slug>[a-z0-9_-]+)', [
            'methods' => 'DELETE',
            'callback' => [$this, 'delete_custom_post_type'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Create or update programmatic pages
        register_rest_route('instarank/v1', '/programmatic/pages', [
            'methods' => 'POST',
            'callback' => [$this, 'sync_programmatic_page'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Check if a page exists (for matching before sync)
        register_rest_route('instarank/v1', '/programmatic/check-existing', [
            'methods' => 'POST',
            'callback' => [$this, 'check_existing_page'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Delete a programmatic page
        register_rest_route('instarank/v1', '/programmatic/pages/(?P<id>\d+)', [
            'methods' => 'DELETE',
            'callback' => [$this, 'delete_programmatic_page'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Detect custom fields in a template
        register_rest_route('instarank/v1', '/templates/(?P<id>\d+)/fields', [
            'methods' => 'GET',
            'callback' => [$this, 'detect_template_fields'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get field mappings for a template
        register_rest_route('instarank/v1', '/templates/(?P<id>\d+)/mappings', [
            'methods' => 'GET',
            'callback' => [$this, 'get_template_mappings'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // ACF (Advanced Custom Fields) ENDPOINTS

        // Get ACF fields for a post type (or all if no post_type specified)
        register_rest_route('instarank/v1', '/acf/fields', [
            'methods' => 'GET',
            'callback' => [$this, 'get_acf_fields'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get ACF status and info
        register_rest_route('instarank/v1', '/acf/status', [
            'methods' => 'GET',
            'callback' => [$this, 'get_acf_status'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get all taxonomies
        register_rest_route('instarank/v1', '/taxonomies', [
            'methods' => 'GET',
            'callback' => [$this, 'get_taxonomies'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Update category SEO meta
        register_rest_route('instarank/v1', '/categories/(?P<id>\d+)/meta', [
            'methods' => 'POST',
            'callback' => [$this, 'update_category_meta'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Update tag SEO meta
        register_rest_route('instarank/v1', '/tags/(?P<id>\d+)/meta', [
            'methods' => 'POST',
            'callback' => [$this, 'update_tag_meta'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Update author SEO meta
        register_rest_route('instarank/v1', '/authors/(?P<id>\d+)/meta', [
            'methods' => 'POST',
            'callback' => [$this, 'update_author_meta'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Search page endpoint
        register_rest_route('instarank/v1', '/search', [
            'methods' => ['GET', 'POST'],
            'callback' => [$this, 'handle_search_page'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // 404 page endpoint
        register_rest_route('instarank/v1', '/404-page', [
            'methods' => ['GET', 'POST'],
            'callback' => [$this, 'handle_404_page'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Attachments list endpoint
        register_rest_route('instarank/v1', '/attachments', [
            'methods' => 'GET',
            'callback' => [$this, 'list_attachments'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Attachment info endpoint
        register_rest_route('instarank/v1', '/attachments/(?P<id>\d+)', [
            'methods' => 'GET',
            'callback' => [$this, 'get_attachment_info'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Update attachment meta endpoint
        register_rest_route('instarank/v1', '/attachments/(?P<id>\d+)/meta', [
            'methods' => 'POST',
            'callback' => [$this, 'update_attachment_meta'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // IMAGE OPTIMIZATION ENDPOINTS

        // Upload optimized image to Media Library
        register_rest_route('instarank/v1', '/media/upload-optimized', [
            'methods' => 'POST',
            'callback' => [$this, 'upload_optimized_image'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Replace image URLs in content
        register_rest_route('instarank/v1', '/media/replace-urls', [
            'methods' => 'POST',
            'callback' => [$this, 'replace_image_urls'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Bulk upload optimized images
        register_rest_route('instarank/v1', '/media/bulk-upload-optimized', [
            'methods' => 'POST',
            'callback' => [$this, 'bulk_upload_optimized_images'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // PAGE BUILDER CSS REGENERATION ENDPOINTS

        // Regenerate CSS for a single post
        register_rest_route('instarank/v1', '/regenerate-css', [
            'methods' => 'POST',
            'callback' => [$this, 'regenerate_css'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Clear all builder caches
        register_rest_route('instarank/v1', '/clear-builder-caches', [
            'methods' => 'POST',
            'callback' => [$this, 'clear_builder_caches'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get CSS regeneration status for a post
        register_rest_route('instarank/v1', '/css-status/(?P<id>\d+)', [
            'methods' => 'GET',
            'callback' => [$this, 'get_css_status'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Schedule CSS regeneration for later
        register_rest_route('instarank/v1', '/schedule-css-regeneration', [
            'methods' => 'POST',
            'callback' => [$this, 'schedule_css_regeneration'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // SPINTAX ENDPOINTS

        // Validate spintax syntax
        register_rest_route('instarank/v1', '/spintax/validate', [
            'methods' => 'POST',
            'callback' => [$this, 'validate_spintax'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Preview spintax variations
        register_rest_route('instarank/v1', '/spintax/preview', [
            'methods' => 'POST',
            'callback' => [$this, 'preview_spintax'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Process spintax (single spin)
        register_rest_route('instarank/v1', '/spintax/spin', [
            'methods' => 'POST',
            'callback' => [$this, 'spin_spintax'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // WooCommerce check endpoint
        register_rest_route('instarank/v1', '/woocommerce/is-active', [
            'methods' => 'GET',
            'callback' => [$this, 'check_woocommerce_active'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // WooCommerce products list endpoint
        register_rest_route('instarank/v1', '/woocommerce/products', [
            'methods' => 'GET',
            'callback' => [$this, 'list_woocommerce_products'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // WooCommerce product info endpoint
        register_rest_route('instarank/v1', '/woocommerce/products/(?P<id>\d+)', [
            'methods' => 'GET',
            'callback' => [$this, 'get_woocommerce_product'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // WooCommerce product meta update endpoint
        register_rest_route('instarank/v1', '/woocommerce/products/(?P<id>\d+)/meta', [
            'methods' => 'POST',
            'callback' => [$this, 'update_woocommerce_product_meta'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // WooCommerce shop endpoint
        register_rest_route('instarank/v1', '/woocommerce/shop', [
            'methods' => ['GET', 'POST'],
            'callback' => [$this, 'handle_woocommerce_shop'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // WooCommerce categories endpoint
        register_rest_route('instarank/v1', '/woocommerce/categories', [
            'methods' => 'GET',
            'callback' => [$this, 'list_woocommerce_categories'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // WooCommerce category meta endpoint
        register_rest_route('instarank/v1', '/woocommerce/categories/(?P<id>\d+)/meta', [
            'methods' => 'POST',
            'callback' => [$this, 'update_woocommerce_category_meta'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // WooCommerce cart endpoint
        register_rest_route('instarank/v1', '/woocommerce/cart', [
            'methods' => 'GET',
            'callback' => [$this, 'get_woocommerce_cart_page'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // WooCommerce checkout endpoint
        register_rest_route('instarank/v1', '/woocommerce/checkout', [
            'methods' => 'GET',
            'callback' => [$this, 'get_woocommerce_checkout_page'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Comprehensive page types endpoint
        register_rest_route('instarank/v1', '/page-types', [
            'methods' => 'GET',
            'callback' => [$this, 'get_page_types_info'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // === NEW: BULK OPERATIONS FOR SAAS PLATFORM ===

        // Bulk update post meta (for bulk SEO operations from platform)
        register_rest_route('instarank/v1', '/bulk-meta', [
            'methods' => 'POST',
            'callback' => [$this, 'bulk_update_meta'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Bulk update robots meta (NoIndex/NoFollow)
        register_rest_route('instarank/v1', '/bulk-robots', [
            'methods' => 'POST',
            'callback' => [$this, 'bulk_update_robots'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get media library (for image SEO from platform)
        register_rest_route('instarank/v1', '/media', [
            'methods' => 'GET',
            'callback' => [$this, 'get_media_library'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Update image alt text
        register_rest_route('instarank/v1', '/media/(?P<id>\d+)/alt-text', [
            'methods' => 'POST',
            'callback' => [$this, 'update_image_alt_text'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Find media attachment by URL
        register_rest_route('instarank/v1', '/media/find-by-url', [
            'methods' => 'POST',
            'callback' => [$this, 'find_attachment_by_url'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Update alt text by image URL
        register_rest_route('instarank/v1', '/media/update-alt-by-url', [
            'methods' => 'POST',
            'callback' => [$this, 'update_alt_text_by_url'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Analyze content for focus keyword (real-time from editor)
        register_rest_route('instarank/v1', '/analyze-content', [
            'methods' => 'POST',
            'callback' => [$this, 'analyze_content'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // === CRAWL DATA ENDPOINTS FOR WORDPRESS-BASED CRAWLING ===

        // Get crawl data for all published pages (paginated)
        register_rest_route('instarank/v1', '/crawl-data', [
            'methods' => 'GET',
            'callback' => [$this, 'get_crawl_data'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get crawl data for a single page by ID or URL
        register_rest_route('instarank/v1', '/crawl-data/page', [
            'methods' => 'GET',
            'callback' => [$this, 'get_page_crawl_data'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // === MULTI-LANGUAGE ENDPOINTS ===

        // Get multi-language plugin info
        register_rest_route('instarank/v1', '/multilang/info', [
            'methods' => 'GET',
            'callback' => [$this, 'get_multilang_info'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Set post language
        register_rest_route('instarank/v1', '/multilang/set-language', [
            'methods' => 'POST',
            'callback' => [$this, 'set_post_language'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Link translations
        register_rest_route('instarank/v1', '/multilang/link-translations', [
            'methods' => 'POST',
            'callback' => [$this, 'link_translations'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // === UNUSED IMAGES DETECTION ENDPOINTS ===

        // Analyze media library for unused images with usage tracking
        register_rest_route('instarank/v1', '/media/analyze-usage', [
            'methods' => 'GET',
            'callback' => [$this, 'analyze_media_usage'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get detailed usage info for a specific attachment
        register_rest_route('instarank/v1', '/media/(?P<id>\d+)/usage', [
            'methods' => 'GET',
            'callback' => [$this, 'get_attachment_usage'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Delete attachment and return backup data
        register_rest_route('instarank/v1', '/media/(?P<id>\d+)/delete', [
            'methods' => 'POST',
            'callback' => [$this, 'delete_attachment_with_backup'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Bulk delete attachments with backup data
        register_rest_route('instarank/v1', '/media/bulk-delete', [
            'methods' => 'POST',
            'callback' => [$this, 'bulk_delete_attachments'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Restore a deleted attachment from backup data
        register_rest_route('instarank/v1', '/media/restore', [
            'methods' => 'POST',
            'callback' => [$this, 'restore_attachment_from_backup'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Rename attachment file (SEO-friendly filename)
        register_rest_route('instarank/v1', '/media/(?P<id>\d+)/rename', [
            'methods' => 'POST',
            'callback' => [$this, 'rename_attachment_file'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Bulk rename attachments
        register_rest_route('instarank/v1', '/media/bulk-rename', [
            'methods' => 'POST',
            'callback' => [$this, 'bulk_rename_attachments'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Update all image metadata (title, caption, description, alt text)
        register_rest_route('instarank/v1', '/media/(?P<id>\d+)/metadata', [
            'methods' => 'POST',
            'callback' => [$this, 'update_image_metadata'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Bulk update image metadata
        register_rest_route('instarank/v1', '/media/bulk-update-metadata', [
            'methods' => 'POST',
            'callback' => [$this, 'bulk_update_image_metadata'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Update image attributes in post content (lazy loading, dimensions)
        register_rest_route('instarank/v1', '/content/update-image-attributes', [
            'methods' => 'POST',
            'callback' => [$this, 'update_content_image_attributes'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Bulk update image attributes in multiple posts
        register_rest_route('instarank/v1', '/content/bulk-update-image-attributes', [
            'methods' => 'POST',
            'callback' => [$this, 'bulk_update_content_image_attributes'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // === SITEMAP, ROBOTS.TXT & LLMS.TXT MANAGEMENT ===

        // Get sitemap settings
        register_rest_route('instarank/v1', '/sitemap/settings', [
            'methods' => 'GET',
            'callback' => [$this, 'get_sitemap_settings'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Update sitemap settings
        register_rest_route('instarank/v1', '/sitemap/settings', [
            'methods' => 'POST',
            'callback' => [$this, 'update_sitemap_settings'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get available post types and taxonomies for sitemap
        register_rest_route('instarank/v1', '/sitemap/available', [
            'methods' => 'GET',
            'callback' => [$this, 'get_available_sitemap_options'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Regenerate sitemap cache
        register_rest_route('instarank/v1', '/sitemap/regenerate', [
            'methods' => 'POST',
            'callback' => [$this, 'regenerate_sitemap'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get sitemap preview (rendered XML)
        register_rest_route('instarank/v1', '/sitemap/preview', [
            'methods' => 'GET',
            'callback' => [$this, 'get_sitemap_preview'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get robots.txt content
        register_rest_route('instarank/v1', '/robots-txt', [
            'methods' => 'GET',
            'callback' => [$this, 'get_robots_txt'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Update robots.txt content
        register_rest_route('instarank/v1', '/robots-txt', [
            'methods' => 'POST',
            'callback' => [$this, 'update_robots_txt'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Reset robots.txt to default
        register_rest_route('instarank/v1', '/robots-txt/reset', [
            'methods' => 'POST',
            'callback' => [$this, 'reset_robots_txt'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Get llms.txt content and settings
        register_rest_route('instarank/v1', '/llms-txt', [
            'methods' => 'GET',
            'callback' => [$this, 'get_llms_txt'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Update llms.txt content
        register_rest_route('instarank/v1', '/llms-txt', [
            'methods' => 'POST',
            'callback' => [$this, 'update_llms_txt'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Reset llms.txt to default
        register_rest_route('instarank/v1', '/llms-txt/reset', [
            'methods' => 'POST',
            'callback' => [$this, 'reset_llms_txt'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);
    }

    /**
     * Verify API key from request headers
     */
    public function verify_api_key($request) {
        $auth_manager = new InstaRank_Auth_Manager();
        $api_key = $request->get_header('X-WordPress-API-Key');

        if (empty($api_key)) {
            return new WP_Error('missing_api_key', __('API key is required', 'instarank'), ['status' => 401]);
        }

        if (!$auth_manager->verify_api_key($api_key)) {
            return new WP_Error('invalid_api_key', __('Invalid API key', 'instarank'), ['status' => 403]);
        }

        return true;
    }

    /**
     * Apply changes endpoint
     */
    public function apply_changes($request) {
        $params = $request->get_json_params();

        if (empty($params['changes']) || !is_array($params['changes'])) {
            return new WP_Error('invalid_request', __('Changes array is required', 'instarank'), ['status' => 400]);
        }

        $change_manager = new InstaRank_Change_Manager();
        $results = [];

        foreach ($params['changes'] as $change_data) {
            // Validate required fields
            if (empty($change_data['post_id']) || empty($change_data['change_type'])) {
                $results[] = [
                    'success' => false,
                    'error' => 'Missing required fields'
                ];
                continue;
            }

            // Store the change
            $change_id = $change_manager->store_change(
                $change_data['post_id'],
                $change_data['change_type'],
                $change_data['old_value'] ?? '',
                $change_data['new_value'] ?? '',
                $change_data['reason'] ?? ''
            );

            if ($change_id) {
                // Auto-approve if enabled
                if (get_option('instarank_auto_approve', false)) {
                    $applied = $change_manager->approve_change($change_id);
                    $results[] = [
                        'success' => true,
                        'change_id' => $change_id,
                        'auto_applied' => $applied
                    ];
                } else {
                    $results[] = [
                        'success' => true,
                        'change_id' => $change_id,
                        'status' => 'pending'
                    ];
                }
            } else {
                $results[] = [
                    'success' => false,
                    'error' => 'Failed to store change'
                ];
            }
        }

        return rest_ensure_response([
            'success' => true,
            'results' => $results
        ]);
    }

    /**
     * Get changes endpoint
     */
    public function get_changes($request) {
        $change_manager = new InstaRank_Change_Manager();
        $status = $request->get_param('status') ?? 'all';
        $limit = intval($request->get_param('limit') ?? 100);

        $changes = $change_manager->get_changes($status, $limit);

        return rest_ensure_response([
            'success' => true,
            'changes' => $changes,
            'total' => count($changes)
        ]);
    }

    /**
     * Get change status endpoint
     */
    public function get_change_status($request) {
        global $wpdb;

        $change_id = intval($request['id']);
        $table = $wpdb->prefix . 'instarank_changes';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $change = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $change_id
        ), ARRAY_A);

        if (!$change) {
            return new WP_Error('not_found', __('Change not found', 'instarank'), ['status' => 404]);
        }

        return rest_ensure_response([
            'success' => true,
            'change' => $change
        ]);
    }

    /**
     * Bulk approve/reject changes endpoint
     */
    public function bulk_changes($request) {
        $params = $request->get_json_params();

        if (empty($params['action']) || !in_array($params['action'], ['approve', 'reject'], true)) {
            return new WP_Error('invalid_action', __('Action must be "approve" or "reject"', 'instarank'), ['status' => 400]);
        }

        if (empty($params['change_ids']) || !is_array($params['change_ids'])) {
            return new WP_Error('invalid_request', __('change_ids array is required', 'instarank'), ['status' => 400]);
        }

        $change_manager = new InstaRank_Change_Manager();
        $results = [];
        $success_count = 0;
        $error_count = 0;

        foreach ($params['change_ids'] as $change_id) {
            $change_id = intval($change_id);

            if ($params['action'] === 'approve') {
                $result = $change_manager->approve_change($change_id);
            } else {
                $result = $change_manager->reject_change($change_id);
            }

            if ($result) {
                $success_count++;
                $results[] = [
                    'change_id' => $change_id,
                    'success' => true
                ];
            } else {
                $error_count++;
                $results[] = [
                    'change_id' => $change_id,
                    'success' => false,
                    'error' => 'Failed to process change'
                ];
            }
        }

        return rest_ensure_response([
            'success' => true,
            'processed' => count($params['change_ids']),
            'succeeded' => $success_count,
            'failed' => $error_count,
            'results' => $results
        ]);
    }

    /**
     * Test connection endpoint
     */
    public function test_connection($request) {
        $detector = new InstaRank_SEO_Detector();
        $site_info = $this->get_full_site_info();

        return rest_ensure_response([
            'success' => true,
            'message' => 'Connection successful',
            'plugin_version' => INSTARANK_VERSION,
            'wp_version' => get_bloginfo('version'),
            'seo_plugin' => $detector->get_active_seo_plugin(),
            'seo_plugin_version' => $site_info['seo_plugin_version'],
            'site_url' => get_site_url(),
            'site_name' => get_bloginfo('name'),
            'site_tagline' => get_bloginfo('description'),
            'page_builder' => $site_info['page_builder'],
            'page_builder_version' => $site_info['page_builder_version'],
            'active_theme' => $site_info['active_theme'],
            'active_theme_version' => $site_info['active_theme_version'],
            'has_woocommerce' => $site_info['has_woocommerce'],
            'has_multisite' => is_multisite()
        ]);
    }

    /**
     * Get full site info for agent context
     */
    private function get_full_site_info() {
        $detector = new InstaRank_SEO_Detector();
        $theme = wp_get_theme();

        // Detect primary page builder
        $page_builder = 'gutenberg';
        $page_builder_version = get_bloginfo('version');

        if (defined('ELEMENTOR_VERSION')) {
            $page_builder = 'elementor';
            $page_builder_version = ELEMENTOR_VERSION;
        } elseif (defined('ET_BUILDER_VERSION')) {
            $page_builder = 'divi';
            $page_builder_version = ET_BUILDER_VERSION;
        } elseif (defined('WPB_VC_VERSION')) {
            $page_builder = 'wpbakery';
            $page_builder_version = WPB_VC_VERSION;
        } elseif (defined('FL_BUILDER_VERSION')) {
            $page_builder = 'beaver_builder';
            $page_builder_version = FL_BUILDER_VERSION;
        } elseif (defined('BRICKS_VERSION')) {
            $page_builder = 'bricks';
            $page_builder_version = BRICKS_VERSION;
        } elseif (defined('CT_VERSION')) {
            $page_builder = 'oxygen';
            $page_builder_version = CT_VERSION;
        } elseif (defined('KADENCE_BLOCKS_VERSION')) {
            $page_builder = 'kadence_blocks';
            $page_builder_version = KADENCE_BLOCKS_VERSION;
        } elseif (defined('SPECTRA_VERSION')) {
            $page_builder = 'spectra';
            $page_builder_version = SPECTRA_VERSION;
        } elseif (defined('JETELEMENTS_VERSION')) {
            $page_builder = 'jetelements';
            $page_builder_version = JETELEMENTS_VERSION;
        }

        // Detect SEO plugin version
        $seo_plugin_version = null;
        if (defined('WPSEO_VERSION')) {
            $seo_plugin_version = WPSEO_VERSION;
        } elseif (defined('RANK_MATH_VERSION')) {
            $seo_plugin_version = RANK_MATH_VERSION;
        } elseif (defined('AIOSEO_VERSION')) {
            $seo_plugin_version = AIOSEO_VERSION;
        } elseif (defined('SEOPRESS_VERSION')) {
            $seo_plugin_version = SEOPRESS_VERSION;
        }

        return [
            'seo_plugin' => $detector->get_active_seo_plugin(),
            'seo_plugin_version' => $seo_plugin_version,
            'page_builder' => $page_builder,
            'page_builder_version' => $page_builder_version,
            'active_theme' => $theme->get('Name'),
            'active_theme_version' => $theme->get('Version'),
            'has_woocommerce' => class_exists('WooCommerce'),
            'has_multisite' => is_multisite()
        ];
    }

    /**
     * Get site info endpoint
     */
    public function get_site_info($request) {
        $detector = new InstaRank_SEO_Detector();
        $site_info = $this->get_full_site_info();

        return rest_ensure_response([
            'success' => true,
            'site' => [
                'url' => get_site_url(),
                'name' => get_bloginfo('name'),
                'description' => get_bloginfo('description'),
                'tagline' => get_bloginfo('description'),
                'language' => get_bloginfo('language'),
                'wp_version' => get_bloginfo('version'),
                'plugin_version' => INSTARANK_VERSION,
                'seo_plugin' => $detector->get_active_seo_plugin(),
                'seo_plugin_version' => $site_info['seo_plugin_version'],
                'page_builder' => $site_info['page_builder'],
                'page_builder_version' => $site_info['page_builder_version'],
                'active_theme' => $site_info['active_theme'],
                'active_theme_version' => $site_info['active_theme_version'],
                'has_woocommerce' => $site_info['has_woocommerce'],
                'has_multisite' => $site_info['has_multisite'],
                'post_count' => wp_count_posts()->publish ?? 0,
                'page_count' => wp_count_posts('page')->publish ?? 0,
                'connected' => get_option('instarank_connected', false),
                'last_sync' => get_option('instarank_last_sync', null)
            ]
        ]);
    }

    /**
     * Get posts endpoint
     */
    public function get_posts($request) {
        $detector = new InstaRank_SEO_Detector();

        // Get post_type parameter, default to all public post types except attachments
        $post_type_param = $request->get_param('post_type');
        if ($post_type_param) {
            $post_type = $post_type_param;
        } else {
            // Get all public post types (posts, pages, products, custom post types)
            $public_post_types = get_post_types(['public' => true], 'names');
            // Remove attachment type as it's not content
            $post_type = array_values(array_filter($public_post_types, function($type) {
                return $type !== 'attachment';
            }));
        }

        $limit = intval($request->get_param('limit') ?? 100);
        $offset = intval($request->get_param('offset') ?? 0);
        $search = $request->get_param('search');

        $args = [
            'post_type' => $post_type,
            'post_status' => 'publish',
            'posts_per_page' => $limit,
            'offset' => $offset,
            'orderby' => 'modified',
            'order' => 'DESC'
        ];

        if (!empty($search)) {
            $args['s'] = sanitize_text_field($search);
        }

        $query = new WP_Query($args);
        $posts = [];

        foreach ($query->posts as $post) {
            $meta = $detector->get_post_meta($post->ID);
            $word_count = str_word_count(wp_strip_all_tags($post->post_content));

            // Detect noindex/nofollow from SEO plugin meta
            $noindex = false;
            $nofollow = false;
            if (!empty($meta['robots'])) {
                $robots = is_array($meta['robots']) ? implode(',', $meta['robots']) : $meta['robots'];
                $noindex = stripos($robots, 'noindex') !== false;
                $nofollow = stripos($robots, 'nofollow') !== false;
            }
            // Also check individual meta flags if set by SEO detector
            if (isset($meta['noindex'])) $noindex = (bool)$meta['noindex'];
            if (isset($meta['nofollow'])) $nofollow = (bool)$meta['nofollow'];

            $posts[] = [
                'id' => $post->ID,
                'title' => $post->post_title,
                'url' => get_permalink($post->ID),
                'type' => $post->post_type,
                'status' => $post->post_status,
                'published' => $post->post_date_gmt,
                'modified' => $post->post_modified_gmt,
                'author' => get_the_author_meta('display_name', $post->post_author),
                'word_count' => $word_count,
                'seo' => [
                    'meta_title' => $meta['title'] ?? '',
                    'meta_description' => $meta['description'] ?? '',
                    'focus_keyword' => $meta['focus_keyword'] ?? '',
                    'noindex' => $noindex,
                    'nofollow' => $nofollow
                ]
            ];
        }

        return rest_ensure_response([
            'success' => true,
            'posts' => $posts,
            'total' => $query->found_posts,
            'limit' => $limit,
            'offset' => $offset
        ]);
    }

    /**
     * Resolve a URL to a WordPress post ID
     * - Handles normal post/page permalinks
     * - Handles static front page mapping to site root
     * - Handles posts page (blog) if configured
     */
    public function resolve_post_by_url($request) {
        $raw_url = $request->get_param('url');
        if (empty($raw_url)) {
            return new WP_Error('invalid_request', __('URL is required', 'instarank'), ['status' => 400]);
        }

        // Normalize input URL (lowercase host, strip query/hash, ensure trailing slash)
        $parsed = wp_parse_url($raw_url);
        if (!$parsed || empty($parsed['scheme']) || empty($parsed['host'])) {
            return new WP_Error('invalid_request', __('Invalid URL format', 'instarank'), ['status' => 400]);
        }

        $path = isset($parsed['path']) ? $parsed['path'] : '/';
        if ($path === '') { $path = '/'; }
        if (substr($path, -1) !== '/') { $path .= '/'; }
        $normalized_input = strtolower($parsed['scheme'] . '://' . $parsed['host']) . $path;

        // Compute normalized site root (with trailing slash)
        $site_root = trailingslashit(home_url('/'));

        // If input equals site root, try static front page
        if ($normalized_input === strtolower($site_root)) {
            $front_id = get_option('page_on_front');
            if ($front_id) {
                return rest_ensure_response([
                    'success' => true,
                    'post' => [
                        'id' => intval($front_id),
                        'url' => get_permalink($front_id),
                        'type' => get_post_type($front_id) ?: 'page',
                    ],
                ]);
            }

            // If posts page is used as homepage
            $show_on_front = get_option('show_on_front');
            if ($show_on_front === 'posts') {
                $posts_page_id = get_option('page_for_posts');
                if ($posts_page_id) {
                    return rest_ensure_response([
                        'success' => true,
                        'post' => [
                            'id' => intval($posts_page_id),
                            'url' => get_permalink($posts_page_id),
                            'type' => 'page',
                        ],
                    ]);
                }
            }
        }

        // Attempt to resolve via core
        $post_id = url_to_postid($normalized_input);
        if ($post_id) {
            return rest_ensure_response([
                'success' => true,
                'post' => [
                    'id' => intval($post_id),
                    'url' => get_permalink($post_id),
                    'type' => get_post_type($post_id) ?: 'post',
                ],
            ]);
        }

        return rest_ensure_response([
            'success' => false,
            'error' => 'not_found',
            'message' => __('No post found for the given URL', 'instarank'),
        ]);
    }

    /**
     * Update homepage SEO meta values for sites where homepage is the posts index
     * Supports AIOSEO, Yoast SEO, Rank Math, and native WordPress (fallback)
     */
    public function update_home_meta($request) {
        $params = $request->get_json_params();
        $change_type = isset($params['change_type']) ? sanitize_text_field($params['change_type']) : '';
        $new_value = isset($params['new_value']) ? wp_kses_post($params['new_value']) : '';

        if (empty($change_type) || empty($new_value)) {
            return new WP_Error('invalid_request', __('change_type and new_value are required', 'instarank'), ['status' => 400]);
        }

        $detector = new InstaRank_SEO_Detector();
        $seo_plugin = $detector->get_active_seo_plugin();
        $applied = false;
        $plugin_used = 'none';

        // Try AIOSEO
        if ($seo_plugin === 'aioseo' || (!$applied && (defined('AIOSEO_VERSION') || class_exists('AIOSEO\\Plugin\\AIOSEO')))) {
            $result = $this->update_aioseo_home_meta($change_type, $new_value);
            if (!is_wp_error($result)) {
                $applied = true;
                $plugin_used = 'aioseo';
            }
        }

        // Try Yoast SEO
        if (!$applied && ($seo_plugin === 'yoast' || defined('WPSEO_VERSION'))) {
            $result = $this->update_yoast_home_meta($change_type, $new_value);
            if (!is_wp_error($result)) {
                $applied = true;
                $plugin_used = 'yoast';
            }
        }

        // Try Rank Math
        if (!$applied && ($seo_plugin === 'rankmath' || class_exists('RankMath'))) {
            $result = $this->update_rankmath_home_meta($change_type, $new_value);
            if (!is_wp_error($result)) {
                $applied = true;
                $plugin_used = 'rankmath';
            }
        }

        // Fallback: Use InstaRank custom options (works without any SEO plugin)
        if (!$applied) {
            $result = $this->update_instarank_home_meta($change_type, $new_value);
            if (!is_wp_error($result)) {
                $applied = true;
                $plugin_used = 'instarank';
            }
        }

        if ($applied) {
            return rest_ensure_response([
                'success' => true,
                'plugin' => $plugin_used,
                'applied' => true,
            ]);
        }

        return new WP_Error('not_supported', __('Unable to update homepage meta', 'instarank'), ['status' => 400]);
    }

    /**
     * Update AIOSEO homepage meta
     */
    private function update_aioseo_home_meta($change_type, $new_value) {
        $rawExisting = get_option('aioseo_options', null);
        if (is_null($rawExisting)) {
            return new WP_Error('not_found', 'AIOSEO options not found');
        }

        $raw = is_null($rawExisting) ? '' : $rawExisting;
        $options = [];
        if (is_string($raw) && $raw !== '') {
            $decoded = json_decode($raw, true);
            if (is_array($decoded)) { $options = $decoded; }
        } elseif (is_array($raw)) {
            $options = $raw;
        }

        if (!isset($options['searchAppearance'])) { $options['searchAppearance'] = []; }
        if (!isset($options['searchAppearance']['global'])) { $options['searchAppearance']['global'] = []; }
        if (!isset($options['searchAppearance']['global']['homePage'])) { $options['searchAppearance']['global']['homePage'] = []; }

        switch ($change_type) {
            case 'meta_description':
            case 'og_description':
                $options['searchAppearance']['global']['homePage']['metaDescription'] = $new_value;
                if ($change_type === 'og_description') {
                    $options['searchAppearance']['global']['homePage']['ogDescription'] = $new_value;
                }
                break;
            case 'meta_title':
            case 'og_title':
                $options['searchAppearance']['global']['homePage']['title'] = $new_value;
                $options['searchAppearance']['global']['homePage']['metaTitle'] = $new_value;
                if ($change_type === 'og_title') {
                    $options['searchAppearance']['global']['homePage']['ogTitle'] = $new_value;
                }
                break;
            case 'og_image':
                $options['searchAppearance']['global']['homePage']['ogImage'] = $new_value;
                break;
            default:
                return new WP_Error('not_supported', 'Change type not supported');
        }

        update_option('aioseo_options', wp_json_encode($options, JSON_UNESCAPED_UNICODE));
        return true;
    }

    /**
     * Update Yoast SEO homepage meta
     */
    private function update_yoast_home_meta($change_type, $new_value) {
        switch ($change_type) {
            case 'meta_description':
            case 'og_description':
                update_option('wpseo_metadesc-home-wpseo', $new_value);
                if ($change_type === 'og_description') {
                    update_option('wpseo_opengraph-description', $new_value);
                }
                break;
            case 'meta_title':
            case 'og_title':
                update_option('wpseo_title-home-wpseo', $new_value);
                if ($change_type === 'og_title') {
                    update_option('wpseo_opengraph-title', $new_value);
                }
                break;
            case 'og_image':
                update_option('wpseo_opengraph-image', $new_value);
                break;
            default:
                return new WP_Error('not_supported', 'Change type not supported');
        }
        return true;
    }

    /**
     * Update Rank Math homepage meta
     */
    private function update_rankmath_home_meta($change_type, $new_value) {
        $titles = get_option('rank-math-options-titles', []);
        if (!is_array($titles)) { $titles = []; }

        switch ($change_type) {
            case 'meta_description':
            case 'og_description':
                $titles['homepage_description'] = $new_value;
                if ($change_type === 'og_description') {
                    $titles['homepage_facebook_description'] = $new_value;
                    $titles['homepage_twitter_description'] = $new_value;
                }
                break;
            case 'meta_title':
            case 'og_title':
                $titles['homepage_title'] = $new_value;
                if ($change_type === 'og_title') {
                    $titles['homepage_facebook_title'] = $new_value;
                    $titles['homepage_twitter_title'] = $new_value;
                }
                break;
            case 'og_image':
                $titles['homepage_facebook_image'] = $new_value;
                $titles['homepage_twitter_image'] = $new_value;
                break;
            default:
                return new WP_Error('not_supported', 'Change type not supported');
        }

        update_option('rank-math-options-titles', $titles);
        return true;
    }

    /**
     * Update InstaRank custom homepage meta (fallback when no SEO plugin is active)
     * This stores the values and outputs them via wp_head hook
     */
    private function update_instarank_home_meta($change_type, $new_value) {
        $home_meta = get_option('instarank_home_meta', []);
        if (!is_array($home_meta)) { $home_meta = []; }

        switch ($change_type) {
            case 'meta_description':
                $home_meta['meta_description'] = $new_value;
                break;
            case 'meta_title':
                $home_meta['meta_title'] = $new_value;
                break;
            case 'og_title':
                $home_meta['og_title'] = $new_value;
                break;
            case 'og_description':
                $home_meta['og_description'] = $new_value;
                break;
            case 'og_image':
                $home_meta['og_image'] = $new_value;
                break;
            default:
                return new WP_Error('not_supported', 'Change type not supported');
        }

        update_option('instarank_home_meta', $home_meta);
        return true;
    }

    /**
     * Health check endpoint
     */
    public function health_check($request) {
        return rest_ensure_response([
            'status' => 'healthy',
            'timestamp' => current_time('mysql'),
            'plugin_version' => INSTARANK_VERSION
        ]);
    }

    /**
     * Confirm connection endpoint
     * Called by InstaRank platform when integration is established
     */
    public function confirm_connection($request) {
        $params = $request->get_json_params();
        
        // Validate required fields
        if (empty($params['project_id'])) {
            return new WP_Error('invalid_request', __('Project ID is required', 'instarank'), ['status' => 400]);
        }

        $project_id = sanitize_text_field($params['project_id']);
        $integration_id = isset($params['integration_id']) ? sanitize_text_field($params['integration_id']) : '';
        
        // Update WordPress options to reflect connected status
        update_option('instarank_connected', true);
        update_option('instarank_project_id', $project_id);
        update_option('instarank_integration_id', $integration_id);
        update_option('instarank_connected_at', current_time('mysql'));
        update_option('instarank_last_sync', current_time('mysql'));

        // Log the connection
        // error_log('InstaRank: Connection confirmed for project ' . $project_id);

        return rest_ensure_response([
            'success' => true,
            'message' => 'Connection confirmed successfully',
            'project_id' => $project_id,
            'integration_id' => $integration_id,
            'connected_at' => current_time('mysql')
        ]);
    }

    /**
     * Get homepage configuration
     * Returns information about the homepage setup
     */
    public function get_homepage_info($request) {
        $show_on_front = get_option('show_on_front', 'posts');
        $page_on_front = get_option('page_on_front', 0);
        $page_for_posts = get_option('page_for_posts', 0);

        $homepage_data = [
            'type' => $show_on_front, // 'posts' or 'page'
            'url' => home_url('/'),
        ];

        if ($show_on_front === 'page' && $page_on_front) {
            // Static homepage
            $homepage_data['page_id'] = intval($page_on_front);
            $homepage_data['page_title'] = get_the_title($page_on_front);
            $homepage_data['page_url'] = get_permalink($page_on_front);

            // Get SEO meta for the homepage
            $detector = new InstaRank_SEO_Detector();
            $homepage_data['seo'] = $detector->get_post_meta($page_on_front);
        } else {
            // Blog posts as homepage
            $homepage_data['is_blog'] = true;
            if ($page_for_posts) {
                $homepage_data['blog_page_id'] = intval($page_for_posts);
            }
        }

        return rest_ensure_response([
            'success' => true,
            'homepage' => $homepage_data
        ]);
    }

    /**
     * Get all categories with their URLs and SEO data
     */
    public function get_categories($request) {
        $detector = new InstaRank_SEO_Detector();
        $limit = intval($request->get_param('limit') ?? 100);
        $offset = intval($request->get_param('offset') ?? 0);

        $args = [
            'taxonomy' => 'category',
            'hide_empty' => false,
            'number' => $limit,
            'offset' => $offset,
            'orderby' => 'count',
            'order' => 'DESC'
        ];

        $categories = get_terms($args);
        $category_data = [];

        foreach ($categories as $category) {
            $category_url = get_term_link($category);
            if (is_wp_error($category_url)) {
                continue;
            }

            $seo_meta = $detector->get_term_meta($category->term_id, 'category');

            $category_data[] = [
                'id' => $category->term_id,
                'name' => $category->name,
                'slug' => $category->slug,
                'description' => $category->description,
                'url' => $category_url,
                'count' => $category->count,
                'parent' => $category->parent,
                'seo' => $seo_meta
            ];
        }

        $total = wp_count_terms(['taxonomy' => 'category']);

        return rest_ensure_response([
            'success' => true,
            'categories' => $category_data,
            'total' => $total,
            'limit' => $limit,
            'offset' => $offset
        ]);
    }

    /**
     * Get all tags with their URLs and SEO data
     */
    public function get_tags($request) {
        $detector = new InstaRank_SEO_Detector();
        $limit = intval($request->get_param('limit') ?? 100);
        $offset = intval($request->get_param('offset') ?? 0);

        $args = [
            'taxonomy' => 'post_tag',
            'hide_empty' => false,
            'number' => $limit,
            'offset' => $offset,
            'orderby' => 'count',
            'order' => 'DESC'
        ];

        $tags = get_terms($args);
        $tag_data = [];

        foreach ($tags as $tag) {
            $tag_url = get_term_link($tag);
            if (is_wp_error($tag_url)) {
                continue;
            }

            $seo_meta = $detector->get_term_meta($tag->term_id, 'post_tag');

            $tag_data[] = [
                'id' => $tag->term_id,
                'name' => $tag->name,
                'slug' => $tag->slug,
                'description' => $tag->description,
                'url' => $tag_url,
                'count' => $tag->count,
                'seo' => $seo_meta
            ];
        }

        $total = wp_count_terms(['taxonomy' => 'post_tag']);

        return rest_ensure_response([
            'success' => true,
            'tags' => $tag_data,
            'total' => $total,
            'limit' => $limit,
            'offset' => $offset
        ]);
    }

    /**
     * Get all authors with their URLs
     */
    public function get_authors($request) {
        $args = [
            'who' => 'authors',
            'has_published_posts' => true,
            'orderby' => 'post_count',
            'order' => 'DESC',
            'number' => intval($request->get_param('limit') ?? 100),
            'offset' => intval($request->get_param('offset') ?? 0)
        ];

        $authors = get_users($args);
        $author_data = [];

        foreach ($authors as $author) {
            $author_url = get_author_posts_url($author->ID);

            $author_data[] = [
                'id' => $author->ID,
                'name' => $author->display_name,
                'slug' => $author->user_nicename,
                'url' => $author_url,
                'post_count' => count_user_posts($author->ID),
                'description' => get_user_meta($author->ID, 'description', true),
                'seo' => $this->get_author_seo_meta($author->ID)
            ];
        }

        $total = count_users();

        return rest_ensure_response([
            'success' => true,
            'authors' => $author_data,
            'total' => $total['total_users'] ?? 0
        ]);
    }

    /**
     * Get all custom post types
     */
    public function get_post_types($request) {
        // Get all public post types (both built-in and custom)
        $post_types = get_post_types([
            'public' => true
        ], 'objects');

        $post_type_data = [];

        // Exclude certain post types that shouldn't be used for programmatic SEO
        $excluded_types = ['attachment', 'revision', 'nav_menu_item', 'custom_css', 'customize_changeset', 'oembed_cache', 'user_request', 'wp_block'];

        foreach ($post_types as $post_type) {
            // Skip excluded types
            if (in_array($post_type->name, $excluded_types)) {
                continue;
            }

            $post_type_data[] = [
                'name' => $post_type->name,
                'label' => $post_type->label,
                'singular_label' => $post_type->labels->singular_name,
                'description' => $post_type->description,
                'has_archive' => $post_type->has_archive,
                'hierarchical' => $post_type->hierarchical,
                'public' => $post_type->public,
                'is_builtin' => $post_type->_builtin,
                'count' => wp_count_posts($post_type->name)->publish ?? 0
            ];
        }

        return rest_ensure_response([
            'success' => true,
            'post_types' => $post_type_data
        ]);
    }

    /**
     * Register a custom post type from InstaRank
     */
    public function register_custom_post_type($request) {
        $params = $request->get_json_params();

        if (!$params || !isset($params['slug']) || !isset($params['config'])) {
            return new WP_Error(
                'missing_params',
                'Missing required parameters: slug and config',
                ['status' => 400]
            );
        }

        $slug = sanitize_key($params['slug']);
        $config = $params['config'];

        // Check if post type already exists
        if (post_type_exists($slug)) {
            return new WP_Error(
                'post_type_exists',
                'A post type with this slug already exists',
                ['status' => 409]
            );
        }

        // Validate slug
        if (strlen($slug) > 20) {
            return new WP_Error(
                'invalid_slug',
                'Post type slug cannot exceed 20 characters',
                ['status' => 400]
            );
        }

        // Store the custom post type configuration in options
        $stored_post_types = get_option('instarank_custom_post_types', []);
        $stored_post_types[$slug] = [
            'slug' => $slug,
            'config' => $config,
            'created_at' => current_time('mysql'),
            'source' => 'instarank'
        ];
        update_option('instarank_custom_post_types', $stored_post_types);

        // Register the post type immediately
        $result = register_post_type($slug, $config);

        if (is_wp_error($result)) {
            // Remove from stored post types if registration failed
            unset($stored_post_types[$slug]);
            update_option('instarank_custom_post_types', $stored_post_types);

            return new WP_Error(
                'registration_failed',
                $result->get_error_message(),
                ['status' => 500]
            );
        }

        // Flush rewrite rules to ensure the post type URLs work
        flush_rewrite_rules();

        return rest_ensure_response([
            'success' => true,
            'message' => 'Custom post type registered successfully',
            'post_type' => [
                'slug' => $slug,
                'label' => $config['label'] ?? $slug,
                'registered_at' => current_time('mysql')
            ]
        ]);
    }

    /**
     * Update a custom post type from InstaRank
     */
    public function update_custom_post_type($request) {
        $slug = $request->get_param('slug');
        $params = $request->get_json_params();

        if (!$params || !isset($params['config'])) {
            return new WP_Error(
                'missing_params',
                'Missing required parameter: config',
                ['status' => 400]
            );
        }

        $config = $params['config'];
        $slug = sanitize_key($slug);

        // Check if post type exists
        if (!post_type_exists($slug)) {
            return new WP_Error(
                'post_type_not_found',
                'Post type does not exist. Use POST to create a new post type.',
                ['status' => 404]
            );
        }

        // Get stored post types
        $stored_post_types = get_option('instarank_custom_post_types', []);

        // Check if this post type was created by InstaRank
        if (!isset($stored_post_types[$slug])) {
            return new WP_Error(
                'not_instarank_post_type',
                'This post type was not created by InstaRank and cannot be updated via this endpoint',
                ['status' => 403]
            );
        }

        // Update the stored configuration
        $stored_post_types[$slug] = [
            'slug' => $slug,
            'config' => $config,
            'created_at' => $stored_post_types[$slug]['created_at'] ?? current_time('mysql'),
            'updated_at' => current_time('mysql'),
            'source' => 'instarank'
        ];
        update_option('instarank_custom_post_types', $stored_post_types);

        // Re-register the post type with updated config
        // Note: WordPress doesn't have a native "update" - we just re-register
        $result = register_post_type($slug, $config);

        if (is_wp_error($result)) {
            return new WP_Error(
                'update_failed',
                $result->get_error_message(),
                ['status' => 500]
            );
        }

        // Flush rewrite rules
        flush_rewrite_rules();

        return rest_ensure_response([
            'success' => true,
            'message' => 'Custom post type updated successfully',
            'post_type' => [
                'slug' => $slug,
                'label' => $config['label'] ?? $slug,
                'updated_at' => current_time('mysql')
            ]
        ]);
    }

    /**
     * Delete a custom post type from InstaRank
     */
    public function delete_custom_post_type($request) {
        $slug = sanitize_key($request->get_param('slug'));

        if (empty($slug)) {
            return new WP_Error(
                'missing_slug',
                'Post type slug is required',
                ['status' => 400]
            );
        }

        // Get stored post types
        $stored_post_types = get_option('instarank_custom_post_types', []);

        // Check if this post type was created by InstaRank
        if (!isset($stored_post_types[$slug])) {
            // Post type not found in InstaRank storage - might not exist or wasn't created by InstaRank
            // Return success anyway to allow cleanup on the SaaS side
            return rest_ensure_response([
                'success' => true,
                'message' => 'Post type not found in InstaRank registry (may have been already deleted or was not created by InstaRank)',
                'slug' => $slug
            ]);
        }

        // Check if there are any posts of this type
        $post_count = wp_count_posts($slug);
        $total_posts = 0;
        if ($post_count) {
            foreach ($post_count as $status => $count) {
                $total_posts += $count;
            }
        }

        // Option: Force delete even with existing posts, or warn
        $force = $request->get_param('force') === 'true' || $request->get_param('force') === true;

        if ($total_posts > 0 && !$force) {
            return new WP_Error(
                'posts_exist',
                "Cannot delete post type '{$slug}' because it has {$total_posts} existing posts. Use force=true to delete anyway (posts will become orphaned).",
                ['status' => 409, 'post_count' => $total_posts]
            );
        }

        // Remove from stored post types
        unset($stored_post_types[$slug]);
        update_option('instarank_custom_post_types', $stored_post_types);

        // Unregister the post type (only works for custom post types registered during this request)
        // For persistent removal, the post type won't be re-registered on next load
        if (post_type_exists($slug)) {
            unregister_post_type($slug);
        }

        // Flush rewrite rules to remove URL patterns
        flush_rewrite_rules();

        return rest_ensure_response([
            'success' => true,
            'message' => 'Custom post type deleted successfully',
            'slug' => $slug,
            'posts_affected' => $total_posts
        ]);
    }

    /**
     * Sync a programmatic page to WordPress
     */
    public function sync_programmatic_page($request) {
        $params = $request->get_json_params();

        if (!$params) {
            return new WP_Error(
                'missing_params',
                'Missing request data',
                ['status' => 400]
            );
        }

        // Extract parameters
        $title = sanitize_text_field($params['title'] ?? '');
        $slug = sanitize_title($params['slug'] ?? '');

        // Handle content - might be string or array (for Elementor)
    $content = $params['content'] ?? '';

    // Normalize content if it's a string (fix smart quotes and en-dashes in block comments)
    if (is_string($content)) {
        // CRITICAL FIX: Decode JSON Unicode escape sequences that come from JSON.stringify
        // JSON.stringify converts < to \u003c, > to \u003e, etc.
        // These need to be decoded back to actual HTML characters
        // Pattern: \u or \\u followed by 4 hex digits (handles both single and double escaping)
        // Single-escaped: \u003c → <
        // Double-escaped: \\u003c → < (for backward compatibility)
        $content = preg_replace_callback(
            '/\\\\\\\\?u([0-9a-fA-F]{4})/',
            function($matches) {
                // Convert hex to decimal, then to character
                return html_entity_decode('&#' . hexdec($matches[1]) . ';', ENT_QUOTES, 'UTF-8');
            },
            $content
        );

        // IMPORTANT: Do NOT use stripcslashes() here!
        // stripcslashes() removes backslashes from \u003c (Unicode escapes used in Kadence block JSON)
        // which corrupts the block structure. Kadence buttons use "text":"\u003cstrong\u003e..." format.
        // Convert literal \\n to newline (double-escaped from JSON)
        // BUT protect JSON inside block comments where \n is a valid JSON escape sequence.
        // Blindly replacing \n in JSON strings creates invalid JSON (unescaped newlines).
        $content = $this->replace_escapes_outside_block_json($content);
        // Robust replacement for corrupted block comments using regex
        // Matches <! followed by any dash-like char (Unicode property Pd), and / followed by any dash-like char >
        $content = preg_replace('/<!\s*\p{Pd}\s*wp:/u', '<!-- wp:', $content);

        $content = preg_replace('/\/\s*\p{Pd}\s*>/u', '/-->', $content);

        // Fix HTML encoded block comments (e.g. &lt;!-- wp: or &lt;!– wp:)
        // IMPORTANT: Only decode block comment delimiters, NOT the JSON inside them
        // Decoding &quot; inside JSON breaks the attribute parsing
        if (strpos($content, '&lt;!--') !== false) {
            // Only decode the comment delimiters
            $content = str_replace(['&lt;!--', '--&gt;'], ['<!--', '-->'], $content);
        }

        // Fallback regex for mixed encoding/dashes
        $content = preg_replace('/&lt;!\s*\p{Pd}*\s*wp:/u', '<!-- wp:', $content);

        $content = str_replace(['--&gt;', '/--&gt;', '/–&gt;'], '-->', $content);

        // Fix smart quotes within block attributes
        // We use a regex to only target the JSON part of the block comment
        // Using (.*?) to capture attributes non-greedily until the closing tag
        $content = preg_replace_callback(
            '/<!--\s*wp:([a-z0-9\/-]+)\s+(.*?)\s*\/-->/s',
            function($matches) {
                $block_name = $matches[1];
                $json_attrs = $matches[2];
                
                // Replace smart quotes with straight quotes in the JSON
                $fixed_json = str_replace(
                    ['“', '”', '″', '″', '«', '»', '„', '‟'], 
                    '"', 
                    $json_attrs
                );
                
                return "<!-- wp:$block_name $fixed_json /-->";
            },
            $content
        );

        // CRITICAL FIX: Wrap standalone Kadence buttons in advancedbtn containers
        // Self-closing kadence/singlebtn blocks don't render properly in WordPress
        // They must be wrapped in kadence/advancedbtn containers
        $content = $this->wrap_standalone_kadence_buttons($content);
    }

    if (!is_array($content)) {
            // IMPORTANT: Don't use wp_kses_post for block editor content
            // Block editor comments like <!-- wp:kadence/singlebtn --> must be preserved
            // wp_kses_post would HTML-encode them, breaking block rendering
            // Only sanitize if it's not block editor content
            if (strpos($content, '<!-- wp:') === false) {
                // Not block editor content, safe to sanitize
                $content = wp_kses_post($content);
            }
            // For block editor content, trust the source (already validated in SaaS)
        }

        $excerpt = sanitize_textarea_field($params['excerpt'] ?? '');
        $status = in_array($params['status'] ?? 'draft', ['publish', 'draft', 'pending', 'private'])
            ? $params['status'] : 'draft';
        $post_type = sanitize_key($params['post_type'] ?? 'post');
        $page_builder = sanitize_key($params['page_builder'] ?? 'gutenberg');

        // Auto-detect SEO plugin if not provided
        if (!empty($params['seo_plugin'])) {
            $seo_plugin = sanitize_key($params['seo_plugin']);
        } else {
            $detector = new InstaRank_SEO_Detector();
            $seo_plugin = $detector->get_active_seo_plugin();
        }

        $meta_title = sanitize_text_field($params['meta_title'] ?? '');
        $meta_description = sanitize_textarea_field($params['meta_description'] ?? '');
        $custom_fields = $params['custom_fields'] ?? [];

        // Check if auto-import images is enabled
        // Can be passed in request OR stored in custom_fields from generation
        $auto_import_images = false;
        if (!empty($params['auto_import_images'])) {
            $auto_import_images = true;
        } elseif (isset($custom_fields['_instarank_auto_import_images']) && $custom_fields['_instarank_auto_import_images']) {
            $auto_import_images = true;
        }

        // Debug log for auto-import
        file_put_contents(
            __DIR__ . '/../instarank_debug.log',
            gmdate('Y-m-d H:i:s') . " - Auto-import check: params=" . json_encode($params['auto_import_images'] ?? 'not set') .
            ", custom_fields=" . json_encode($custom_fields['_instarank_auto_import_images'] ?? 'not set') .
            ", result=" . ($auto_import_images ? 'true' : 'false') . "\n",
            FILE_APPEND
        );

        // Process spintax if enabled (Hybrid approach: WordPress can process spintax locally)
        $spintax_config = $params['spintax'] ?? null;
        if ($spintax_config && !empty($spintax_config['enabled'])) {
            $spintax_engine = new InstaRank_Spintax_Engine([
                'seed' => isset($spintax_config['seed']) ? (int) $spintax_config['seed'] : null,
                'preserve_case' => $spintax_config['preserve_case'] ?? true,
            ]);

            // Get variables for replacement (custom fields from dataset)
            $spintax_variables = $spintax_config['variables'] ?? $custom_fields;

            // Fields to process spintax on
            $spintax_fields = $spintax_config['fields'] ?? ['title', 'content', 'excerpt', 'meta_title', 'meta_description'];

            // Process each field if it contains spintax
            if (in_array('title', $spintax_fields) && $spintax_engine->has_spintax($title)) {
                $title = $spintax_engine->spin($title, $spintax_variables);
            }
            if (in_array('content', $spintax_fields) && is_string($content) && $spintax_engine->has_spintax($content)) {
                $content = $spintax_engine->spin($content, $spintax_variables);
            }
            if (in_array('excerpt', $spintax_fields) && $spintax_engine->has_spintax($excerpt)) {
                $excerpt = $spintax_engine->spin($excerpt, $spintax_variables);
            }
            if (in_array('meta_title', $spintax_fields) && $spintax_engine->has_spintax($meta_title)) {
                $meta_title = $spintax_engine->spin($meta_title, $spintax_variables);
            }
            if (in_array('meta_description', $spintax_fields) && $spintax_engine->has_spintax($meta_description)) {
                $meta_description = $spintax_engine->spin($meta_description, $spintax_variables);
            }
        }
        $wordpress_post_id = isset($params['wordpress_post_id']) ? intval($params['wordpress_post_id']) : null;

        // Validate required fields
        if (empty($title)) {
            return new WP_Error(
                'missing_title',
                'Title is required',
                ['status' => 400]
            );
        }

        // Check if post type exists
        if (!post_type_exists($post_type)) {
            return new WP_Error(
                'invalid_post_type',
                "Post type '{$post_type}' does not exist",
                ['status' => 400]
            );
        }

        // Prepare post data
        // For Elementor, if content is an array, set post_content to empty
        // The actual content will be stored in _elementor_data meta field
        $post_content = is_array($content) ? '' : $content;

        // Check if this is block editor content that needs to bypass WordPress filters
        $is_block_content = !is_array($content) && strpos($content, '<!-- wp:') !== false;

        // Validate Kadence blocks before insertion to prevent corruption
        if ($is_block_content && strpos($post_content, 'wp:kadence/') !== false) {
            $validation = $this->validate_kadence_blocks($post_content);

            // Log warnings but don't block insertion
            if (!empty($validation['warnings'])) {
                file_put_contents(
                    __DIR__ . '/../instarank_debug.log',
                    gmdate('Y-m-d H:i:s') . " - Kadence validation warnings for '{$title}': " .
                    implode(', ', $validation['warnings']) . "\n",
                    FILE_APPEND
                );
            }

            // Block insertion if there are critical errors
            if (!$validation['valid']) {
                return new WP_Error(
                    'invalid_kadence_blocks',
                    'Kadence block validation failed: ' . implode(', ', $validation['errors']),
                    ['status' => 400, 'validation' => $validation]
                );
            }
        }

        // Handle parent page (for hierarchical post types like pages)
        $parent_id = isset($params['parent']) ? intval($params['parent']) : 0;
        // Parent slug for hierarchical post type URL resolution
        $parent_slug_value = isset($params['parent_slug']) ? sanitize_title($params['parent_slug']) : '';

        $post_data = [
            'post_title' => $title,
            'post_name' => $slug,
            'post_content' => $is_block_content ? '' : $post_content, // Empty for block content, we'll set it directly
            'post_excerpt' => $excerpt,
            'post_status' => $status,
            'post_type' => $post_type,
            'post_parent' => $parent_id,
        ];

        // Create or update post
        if ($wordpress_post_id && get_post($wordpress_post_id)) {
            // Update existing post
            $post_data['ID'] = $wordpress_post_id;
            $post_id = wp_update_post($post_data, true);
        } else {
            // Create new post
            $post_id = wp_insert_post($post_data, true);
        }

        // For block editor content, update post_content directly in database to bypass all WordPress filters
        if ($is_block_content && !is_wp_error($post_id)) {
            global $wpdb;

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Intentional direct query to bypass WordPress content filters for block editor content
            $result = $wpdb->update(
                $wpdb->posts,
                ['post_content' => $post_content],
                ['ID' => $post_id],
                ['%s'],
                ['%d']
            );

            // Clear post cache so WordPress reloads the content
            clean_post_cache($post_id);
        }

        if (is_wp_error($post_id)) {
            return new WP_Error(
                'post_creation_failed',
                $post_id->get_error_message(),
                ['status' => 500]
            );
        }

        // Save parent slug meta for hierarchical post type URL resolution
        if (!empty($parent_slug_value)) {
            update_post_meta($post_id, '_instarank_parent_slug', $parent_slug_value);
        }

        // Store page builder metadata and handle content accordingly
        if ($page_builder) {
            update_post_meta($post_id, '_instarank_page_builder', $page_builder);

            // Handle builder-specific content and meta fields
            switch ($page_builder) {
                case 'elementor':
                    if (!empty($content)) {
                        $elementor_data = null;

                        // Content might already be an array or a JSON string
                        if (is_array($content)) {
                            $elementor_data = $content;
                        } else {
                            // Try to decode as JSON
                            $decoded_content = json_decode($content, true);
                            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded_content)) {
                                $elementor_data = $decoded_content;
                            }
                        }

                        if ($elementor_data !== null) {
                            // Store Elementor data in the correct meta field as array
                            update_post_meta($post_id, '_elementor_data', $elementor_data);
                            update_post_meta($post_id, '_elementor_edit_mode', 'builder');
                            update_post_meta($post_id, '_elementor_template_type', 'wp-post');
                            update_post_meta($post_id, '_elementor_version', '3.16.0');
                            update_post_meta($post_id, '_elementor_css', ''); // Clear CSS to force regeneration

                            // Update post content to empty or placeholder for Elementor
                            wp_update_post([
                                'ID' => $post_id,
                                'post_content' => '' // Elementor manages its own content
                            ]);

                            // Trigger Elementor to regenerate CSS and process the content
                            if (class_exists('\Elementor\Plugin')) {
                                // Clear cache for this post
                                \Elementor\Plugin::$instance->files_manager->clear_cache();

                                // Regenerate CSS for this specific post
                                $css_file = \Elementor\Core\Files\CSS\Post::create($post_id);
                                $css_file->update();
                            }
                        }
                    }
                    break;

                case 'divi':
                    // Divi stores content as shortcodes in post_content (already set above)
                    // Set Divi-specific meta fields
                    update_post_meta($post_id, '_et_pb_use_builder', 'on');
                    update_post_meta($post_id, '_et_pb_old_content', '');
                    update_post_meta($post_id, '_et_pb_page_layout', 'et_no_sidebar');
                    update_post_meta($post_id, '_et_pb_show_title', 'on');
                    update_post_meta($post_id, '_et_pb_post_hide_nav', 'default');
                    update_post_meta($post_id, '_et_pb_side_nav', 'off');

                    // Regenerate Divi CSS if function exists
                    if (function_exists('et_core_page_resource_fallback')) {
                        et_core_page_resource_fallback($post_id, true, true);
                    }
                    break;

                case 'beaver_builder':
                case 'beaver-builder':
                    if (!empty($content)) {
                        $beaver_data = null;

                        // Beaver Builder content might be array or JSON
                        if (is_array($content)) {
                            $beaver_data = $content;
                        } else {
                            $decoded_content = json_decode($content, true);
                            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded_content)) {
                                $beaver_data = $decoded_content;
                            }
                        }

                        if ($beaver_data !== null) {
                            // Store Beaver Builder data
                            update_post_meta($post_id, '_fl_builder_enabled', 1);
                            update_post_meta($post_id, '_fl_builder_data', $beaver_data);
                            update_post_meta($post_id, '_fl_builder_draft', $beaver_data);

                            // Clear post content (Beaver Builder manages its own)
                            wp_update_post([
                                'ID' => $post_id,
                                'post_content' => ''
                            ]);

                            // Regenerate CSS/JS if Beaver Builder is active
                            if (class_exists('FLBuilder')) {
                                FLBuilder::render_css($post_id);
                                FLBuilder::render_js($post_id);
                            }
                        }
                    }
                    break;

                case 'bricks':
                    if (!empty($content)) {
                        $bricks_data = null;

                        // Bricks content is similar to Elementor (JSON array)
                        if (is_array($content)) {
                            $bricks_data = $content;
                        } else {
                            $decoded_content = json_decode($content, true);
                            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded_content)) {
                                $bricks_data = $decoded_content;
                            }
                        }

                        if ($bricks_data !== null) {
                            // Store Bricks data
                            update_post_meta($post_id, '_bricks_page_content_2', $bricks_data);
                            update_post_meta($post_id, '_bricks_editor_mode', 'bricks');

                            // Clear post content (Bricks manages its own)
                            wp_update_post([
                                'ID' => $post_id,
                                'post_content' => ''
                            ]);

                            // Regenerate CSS if Bricks is active
                            if (class_exists('\Bricks\Assets')) {
                                \Bricks\Assets::generate_css_file($post_id);
                            }
                        }
                    }
                    break;

                case 'oxygen':
                    if (!empty($content)) {
                        // Oxygen stores content as shortcodes (similar to Divi)
                        update_post_meta($post_id, 'ct_builder_shortcodes', $content);
                        update_post_meta($post_id, 'ct_builder_json', $content);

                        // Clear post content (Oxygen manages its own)
                        wp_update_post([
                            'ID' => $post_id,
                            'post_content' => ''
                        ]);

                        // Regenerate CSS if Oxygen is active
                        if (function_exists('oxygen_vsb_cache_universal_css')) {
                            oxygen_vsb_cache_universal_css();
                        }
                    }
                    break;

                case 'wpbakery':
                    // WPBakery stores content as shortcodes in post_content (already set above)
                    // Set WPBakery-specific meta fields
                    update_post_meta($post_id, '_wpb_vc_js_status', 'true');
                    update_post_meta($post_id, '_wpb_shortcodes_custom_css', '');

                    // Regenerate CSS if WPBakery is active
                    if (class_exists('Vc_Manager')) {
                        // Trigger WPBakery to process shortcodes and generate CSS
                        if (method_exists('WPBMap', 'addAllMappedShortcodes')) {
                            WPBMap::addAllMappedShortcodes();
                        }
                    }
                    break;

                case 'brizy':
                    if (!empty($content)) {
                        $brizy_data = null;

                        // Brizy content is JSON
                        if (is_array($content)) {
                            $brizy_data = $content;
                        } else {
                            $decoded_content = json_decode($content, true);
                            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded_content)) {
                                $brizy_data = $decoded_content;
                            }
                        }

                        if ($brizy_data !== null) {
                            // Store Brizy data
                            update_post_meta($post_id, 'brizy_post_uid', wp_generate_uuid4());
                            update_post_meta($post_id, 'brizy', json_encode($brizy_data));
                            update_post_meta($post_id, 'brizy-post-editor-version', '1');

                            // Clear post content (Brizy manages its own)
                            wp_update_post([
                                'ID' => $post_id,
                                'post_content' => ''
                            ]);
                        }
                    }
                    break;

                case 'kadence':
                case 'kadence_blocks':
                case 'gutenberg':
                case 'generateblocks':
                    // Block editor content is already handled by the block content bypass above (lines 1466-1481)
                    // No additional meta fields needed - content is in post_content
                    break;

                default:
                    // Unknown builder - store content as-is
                    break;
            }
        }

        // Handle SEO metadata based on plugin
        if ($seo_plugin === 'yoast') {
            if ($meta_title) {
                update_post_meta($post_id, '_yoast_wpseo_title', $meta_title);
            }
            if ($meta_description) {
                update_post_meta($post_id, '_yoast_wpseo_metadesc', $meta_description);
            }
        } elseif ($seo_plugin === 'rank_math') {
            if ($meta_title) {
                update_post_meta($post_id, 'rank_math_title', $meta_title);
            }
            if ($meta_description) {
                update_post_meta($post_id, 'rank_math_description', $meta_description);
            }
        } else {
            // Fallback to generic meta fields
            if ($meta_title) {
                update_post_meta($post_id, '_instarank_meta_title', $meta_title);
            }
            if ($meta_description) {
                update_post_meta($post_id, '_instarank_meta_description', $meta_description);
            }
        }

        // Store custom fields and handle WordPress-specific fields
        if (!empty($custom_fields) && is_array($custom_fields)) {
            // Log custom fields processing for debugging
            file_put_contents(
                __DIR__ . '/../instarank_debug.log',
                gmdate('Y-m-d H:i:s') . " - Processing " . count($custom_fields) . " custom fields for post '{$title}' (ID: {$post_id})\n",
                FILE_APPEND
            );

            // Initialize ACF detector for handling ACF fields
            $acf_detector = InstaRank_ACF_Detector::instance();
            $acf_active = $acf_detector->is_acf_active();

            // WordPress core fields that need special handling
            // Keys match the field names defined in class-field-detector.php get_standard_fields()
            $wp_core_fields = [
                'parent' => 'post_parent',
                'page_template' => '_wp_page_template',
                'menu_order' => 'menu_order',
                'comment_status' => 'comment_status',
                'ping_status' => 'ping_status',
                'post_password' => 'post_password',
                'author' => 'post_author',
                'publish_date' => 'post_date',
                'featured_image' => '_thumbnail_id',
                'Page_Title' => 'post_title',
                'Slug' => 'post_name',
                'status' => 'post_status',
                'excerpt' => 'post_excerpt',
            ];

            // WordPress layout/display fields (theme-specific)
            $wp_layout_fields = [
                'page_layout', 'sidebar_layout', 'header_style', 'footer_style',
                'hide_title', 'breadcrumbs', 'custom_body_class'
            ];

            // SEO plugin fields (AIOSEO/Yoast/RankMath)
            $seo_fields = [
                'SEO_Title' => ['aioseo' => '_aioseo_title', 'yoast' => '_yoast_wpseo_title', 'rankmath' => 'rank_math_title'],
                'SEO_Meta_Description' => ['aioseo' => '_aioseo_description', 'yoast' => '_yoast_wpseo_metadesc', 'rankmath' => 'rank_math_description'],
                'SEO_Focus_Keyword' => ['aioseo' => '_aioseo_keyphrases', 'yoast' => '_yoast_wpseo_focuskw', 'rankmath' => 'rank_math_focus_keyword'],
                'canonical_url' => ['aioseo' => '_aioseo_canonical_url', 'yoast' => '_yoast_wpseo_canonical', 'rankmath' => 'rank_math_canonical_url'],
                'robots_index' => ['aioseo' => '_aioseo_robots_noindex', 'yoast' => '_yoast_wpseo_meta-robots-noindex', 'rankmath' => 'rank_math_robots'],
                'robots_follow' => ['aioseo' => '_aioseo_robots_nofollow', 'yoast' => '_yoast_wpseo_meta-robots-nofollow', 'rankmath' => 'rank_math_robots'],
                'og_image' => ['aioseo' => '_aioseo_og_image_url', 'yoast' => '_yoast_wpseo_opengraph-image', 'rankmath' => 'rank_math_facebook_image'],
                'og_title' => ['aioseo' => '_aioseo_og_title', 'yoast' => '_yoast_wpseo_opengraph-title', 'rankmath' => 'rank_math_facebook_title'],
                'og_description' => ['aioseo' => '_aioseo_og_description', 'yoast' => '_yoast_wpseo_opengraph-description', 'rankmath' => 'rank_math_facebook_description'],
                'twitter_card_type' => ['aioseo' => '_aioseo_twitter_card_type', 'yoast' => '_yoast_wpseo_twitter-card-type', 'rankmath' => 'rank_math_twitter_card_type'],
                'twitter_title' => ['aioseo' => '_aioseo_twitter_title', 'yoast' => '_yoast_wpseo_twitter-title', 'rankmath' => 'rank_math_twitter_title'],
                'twitter_description' => ['aioseo' => '_aioseo_twitter_description', 'yoast' => '_yoast_wpseo_twitter-description', 'rankmath' => 'rank_math_twitter_description'],
                'schema_type' => ['rankmath' => 'rank_math_rich_snippet'],
            ];

            foreach ($custom_fields as $field_key => $field_value) {
                // Skip empty values for optional fields
                if ($field_value === '' || $field_value === null) {
                    continue;
                }

                // Handle WordPress core fields
                if (isset($wp_core_fields[$field_key])) {
                    $wp_field = $wp_core_fields[$field_key];

                    // Special handling for author - can be username, email, or ID
                    if ($wp_field === 'post_author') {
                        $author_id = null;
                        if (is_numeric($field_value)) {
                            $author_id = intval($field_value);
                        } else {
                            // Try to find user by username or email
                            $user = get_user_by('login', $field_value);
                            if (!$user) {
                                $user = get_user_by('email', $field_value);
                            }
                            if ($user) {
                                $author_id = $user->ID;
                            }
                        }
                        if ($author_id && get_user_by('id', $author_id)) {
                            wp_update_post([
                                'ID' => $post_id,
                                'post_author' => $author_id
                            ]);
                        }
                        continue;
                    }

                    // Special handling for featured image - can be URL or attachment ID
                    if ($wp_field === '_thumbnail_id') {
                        if (is_numeric($field_value)) {
                            // It's an attachment ID
                            set_post_thumbnail($post_id, intval($field_value));
                        } elseif (filter_var($field_value, FILTER_VALIDATE_URL)) {
                            // It's a URL - try to find existing attachment or upload
                            $attachment_id = $this->get_or_create_attachment_from_url($field_value, $post_id);
                            if ($attachment_id) {
                                set_post_thumbnail($post_id, $attachment_id);
                            }
                        }
                        continue;
                    }

                    // Special handling for publish_date - validate date format
                    if ($wp_field === 'post_date') {
                        $date = strtotime($field_value);
                        if ($date !== false) {
                            wp_update_post([
                                'ID' => $post_id,
                                'post_date' => gmdate('Y-m-d H:i:s', $date),
                                'post_date_gmt' => get_gmt_from_date(gmdate('Y-m-d H:i:s', $date))
                            ]);
                        }
                        continue;
                    }

                    // Update post field directly if it's a post property
                    if (in_array($wp_field, ['post_parent', 'menu_order', 'comment_status', 'ping_status', 'post_password', 'post_title', 'post_name', 'post_status', 'post_excerpt'])) {
                        wp_update_post([
                            'ID' => $post_id,
                            $wp_field => $field_value
                        ]);
                    } else {
                        // Update as post meta
                        update_post_meta($post_id, $wp_field, $field_value);
                    }
                    continue;
                }

                // Handle SEO plugin fields
                if (isset($seo_fields[$field_key])) {
                    $seo_mapping = $seo_fields[$field_key];

                    // Detect active SEO plugin and apply fields
                    if ($seo_plugin === 'aioseo' && isset($seo_mapping['aioseo'])) {
                        // Special handling for AIOSEO focus keyword (expects JSON array)
                        if ($field_key === 'SEO_Focus_Keyword') {
                            $keyphrases = json_encode([
                                'focus' => [
                                    'keyphrase' => $field_value,
                                    'score' => 0
                                ]
                            ]);
                            update_post_meta($post_id, $seo_mapping['aioseo'], $keyphrases);
                        } else {
                            update_post_meta($post_id, $seo_mapping['aioseo'], $field_value);
                        }
                    } elseif ($seo_plugin === 'yoast' && isset($seo_mapping['yoast'])) {
                        update_post_meta($post_id, $seo_mapping['yoast'], $field_value);
                    } elseif ($seo_plugin === 'rank_math' && isset($seo_mapping['rankmath'])) {
                        update_post_meta($post_id, $seo_mapping['rankmath'], $field_value);
                    } else {
                        // Fallback to custom meta field
                        update_post_meta($post_id, '_instarank_' . $field_key, $field_value);
                    }
                    continue;
                }

                // Handle layout/theme fields
                if (in_array($field_key, $wp_layout_fields)) {
                    // Special handling for hide_title with Kadence
                    if ($field_key === 'hide_title') {
                        // Kadence uses array format for title settings
                        $hide_title_value = (strtolower($field_value) === 'true' || $field_value === true || $field_value === '1');

                        // Set Kadence title settings
                        update_post_meta($post_id, '_kad_post_transparent', $hide_title_value ? 'hide' : 'default');
                        update_post_meta($post_id, '_kad_post_title', $hide_title_value ? 'hide' : 'default');
                    }

                    // Store with theme-agnostic prefix for compatibility
                    update_post_meta($post_id, '_' . $field_key, $field_value);
                    // Also store without prefix for themes that don't use underscore
                    update_post_meta($post_id, $field_key, $field_value);
                    continue;
                }

                // Handle custom code fields (sanitize carefully)
                if (in_array($field_key, ['custom_header_code', 'custom_footer_code', 'custom_css'])) {
                    // Don't sanitize code fields - store as-is
                    update_post_meta($post_id, '_instarank_' . $field_key, $field_value);
                    continue;
                }

                // Handle redirect URL
                if ($field_key === 'redirect_url' && !empty($field_value)) {
                    update_post_meta($post_id, '_instarank_redirect_url', esc_url_raw($field_value));
                    // Also set RankMath redirect if available
                    if (class_exists('RankMath')) {
                        update_post_meta($post_id, 'rank_math_redirection_url_to', esc_url_raw($field_value));
                        update_post_meta($post_id, 'rank_math_redirection_type', '301');
                    }
                    continue;
                }

                // Check if this is an ACF field and handle it properly
                if ($acf_active) {
                    $acf_field_info = $acf_detector->is_acf_field($field_key, $post_type);
                    if ($acf_field_info !== false) {
                        // This is an ACF field - use update_field() for proper handling
                        // This ensures ACF reference keys are set correctly
                        $transformed_value = $this->transform_acf_value($field_value, $acf_field_info['acf_field_type']);
                        $acf_detector->update_field_value($field_key, $transformed_value, $post_id);

                        // Log ACF field storage
                        file_put_contents(
                            __DIR__ . '/../instarank_debug.log',
                            gmdate('Y-m-d H:i:s') . " - Stored ACF field: {$field_key} (type: {$acf_field_info['acf_field_type']}) = " .
                            substr(is_array($transformed_value) ? json_encode($transformed_value) : $transformed_value, 0, 50) . "\n",
                            FILE_APPEND
                        );
                        continue;
                    }
                }

                // Regular custom fields - preserve field name structure
                // Use sanitize_text_field instead of sanitize_key to preserve hyphens/underscores
                $sanitized_key = sanitize_text_field($field_key);
                update_post_meta($post_id, $sanitized_key, $field_value);

                // Log each custom field stored
                file_put_contents(
                    __DIR__ . '/../instarank_debug.log',
                    gmdate('Y-m-d H:i:s') . " - Stored custom field: {$sanitized_key} = " . substr($field_value, 0, 50) . (strlen($field_value) > 50 ? '...' : '') . "\n",
                    FILE_APPEND
                );
            }

            // Log completion of custom fields processing
            file_put_contents(
                __DIR__ . '/../instarank_debug.log',
                gmdate('Y-m-d H:i:s') . " - Completed storing custom fields for post ID: {$post_id}\n",
                FILE_APPEND
            );

            // Store all pSEO dataset field values together for metabox display
            // This allows the InstaRank SEO metabox to show all field values at a glance
            update_post_meta($post_id, '_instarank_pseo_fields', $custom_fields);
            update_post_meta($post_id, '_instarank_pseo_generated', true);
            update_post_meta($post_id, '_instarank_pseo_generated_at', gmdate('Y-m-d H:i:s'));
        }

        // Sync AIOSEO data to its database table if plugin is active
        if ($seo_plugin === 'aioseo' && class_exists('AIOSEO\Plugin\Common\Models\Post')) {
            try {
                // Get or create AIOSEO post model
                $aioseo_post = \AIOSEO\Plugin\Common\Models\Post::getPost($post_id);

                if ($aioseo_post) {
                    // Get values from post meta
                    $title = get_post_meta($post_id, '_aioseo_title', true);
                    $description = get_post_meta($post_id, '_aioseo_description', true);
                    $keyphrases = get_post_meta($post_id, '_aioseo_keyphrases', true);
                    $canonical_url = get_post_meta($post_id, '_aioseo_canonical_url', true);

                    // Update AIOSEO model
                    if ($title) $aioseo_post->title = $title;
                    if ($description) $aioseo_post->description = $description;
                    if ($keyphrases) $aioseo_post->keyphrases = $keyphrases;
                    if ($canonical_url) $aioseo_post->canonical_url = $canonical_url;

                    // Save to AIOSEO database
                    $aioseo_post->save();
                }
            } catch (\Exception $e) {
                // Silently continue if AIOSEO sync fails
                // Error is ignored to prevent blocking the main operation
            }
        }

        // Auto-import external images if enabled
        $image_import_result = null;
        if ($auto_import_images) {
            // Get the current post content
            $current_post = get_post($post_id);
            $current_content = $current_post->post_content;

            // Only process if there's content and it's not a page builder that stores content elsewhere
            $skip_content_builders = ['elementor', 'beaver_builder', 'beaver-builder', 'brizy'];
            if (!empty($current_content) && !in_array($page_builder, $skip_content_builders, true)) {
                // Import external images and update content
                $image_import_result = $this->auto_import_external_images($current_content, $post_id, $title);

                // If images were imported, update the post content with local URLs
                if ($image_import_result['imported_count'] > 0) {
                    $updated_content = $image_import_result['content'];

                    // For block editor content, use direct database update to bypass WordPress filters
                    // wp_update_post() applies content filters that HTML-encode block comments inside divs
                    if ($is_block_content) {
                        global $wpdb;
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->update(
                            $wpdb->posts,
                            ['post_content' => $updated_content],
                            ['ID' => $post_id],
                            ['%s'],
                            ['%d']
                        );
                        clean_post_cache($post_id);
                    } else {
                        wp_update_post([
                            'ID' => $post_id,
                            'post_content' => $updated_content,
                        ]);
                    }

                    // Log the import result
                    file_put_contents(
                        __DIR__ . '/../instarank_debug.log',
                        gmdate('Y-m-d H:i:s') . " - Auto-imported {$image_import_result['imported_count']} images for post '{$title}' (ID: {$post_id})" . ($is_block_content ? ' [direct DB update]' : '') . "\n",
                        FILE_APPEND
                    );
                }
            }
        }

        // Get the post object
        $post = get_post($post_id);

        // FINAL FIX: Check if content has HTML-encoded block comments and fix them
        // This handles edge cases where block comments inside divs got HTML-encoded
        if ($is_block_content && $post->post_content) {
            $content = $post->post_content;
            $needs_fix = false;

            // Check for HTML-encoded block comments (e.g., &lt;!-- wp: or &quot; in block JSON)
            if (strpos($content, '&lt;!--') !== false ||
                strpos($content, '&quot;') !== false ||
                strpos($content, '&gt;') !== false) {
                $needs_fix = true;
                // Decode HTML entities
                $content = html_entity_decode($content, ENT_QUOTES | ENT_HTML5, 'UTF-8');
            }

            // Check for en-dash block comments (<!– instead of <!--)
            if (strpos($content, "<!–") !== false || strpos($content, "/–>") !== false) {
                $needs_fix = true;
                // Fix using regex to handle all dash-like characters
                $content = preg_replace('/<!\s*\p{Pd}\s*wp:/u', '<!-- wp:', $content);
                $content = preg_replace('/\/\s*\p{Pd}\s*>/u', '/-->', $content);
            }

            if ($needs_fix) {
                global $wpdb;
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->update(
                    $wpdb->posts,
                    ['post_content' => $content],
                    ['ID' => $post_id],
                    ['%s'],
                    ['%d']
                );
                clean_post_cache($post_id);

                // Re-fetch post after fix
                $post = get_post($post_id);

                file_put_contents(
                    __DIR__ . '/../instarank_debug.log',
                    gmdate('Y-m-d H:i:s') . " - Fixed HTML-encoded block comments for post ID: {$post_id}\n",
                    FILE_APPEND
                );
            }
        }

        // Build response with optional image import stats
        $response_data = [
            'success' => true,
            'id' => $post_id,
            'link' => get_permalink($post_id),
            'edit_link' => get_edit_post_link($post_id, 'raw'),
            'status' => $post->post_status,
            'message' => $wordpress_post_id ? 'Page updated successfully' : 'Page created successfully',
        ];

        // Add image import stats if auto-import was enabled
        if ($image_import_result !== null) {
            $response_data['image_import'] = [
                'imported_count' => $image_import_result['imported_count'],
                'skipped_count' => $image_import_result['skipped_count'],
                'images' => $image_import_result['images'],
            ];
        }

        return rest_ensure_response($response_data);
    }

    /**
     * Check if a page exists in WordPress for matching purposes
     * Supports matching by: slug, title, custom_field, wordpress_post_id
     */
    public function check_existing_page($request) {
        $params = $request->get_json_params();

        if (!$params) {
            return new WP_Error(
                'missing_params',
                'Missing request data',
                ['status' => 400]
            );
        }

        $match_by = sanitize_key($params['match_by'] ?? 'slug');
        $match_value = $params['match_value'] ?? '';
        $post_type = sanitize_key($params['post_type'] ?? 'post');

        if (empty($match_value)) {
            return new WP_Error(
                'missing_match_value',
                'match_value is required',
                ['status' => 400]
            );
        }

        $found_post = null;

        switch ($match_by) {
            case 'wordpress_post_id':
                $post_id = intval($match_value);
                $post = get_post($post_id);
                if ($post && $post->post_type === $post_type) {
                    $found_post = $post;
                }
                break;

            case 'slug':
                $args = [
                    'name' => sanitize_title($match_value),
                    'post_type' => $post_type,
                    'post_status' => ['publish', 'pending', 'draft', 'private'],
                    'numberposts' => 1,
                ];
                $posts = get_posts($args);
                if (!empty($posts)) {
                    $found_post = $posts[0];
                }
                break;

            case 'title':
                $args = [
                    'title' => sanitize_text_field($match_value),
                    'post_type' => $post_type,
                    'post_status' => ['publish', 'pending', 'draft', 'private'],
                    'numberposts' => 1,
                ];
                $posts = get_posts($args);
                if (!empty($posts)) {
                    $found_post = $posts[0];
                }
                break;

            case 'custom_field':
                // match_value should be "field_name=field_value" format
                $parts = explode('=', $match_value, 2);
                if (count($parts) === 2) {
                    $field_name = sanitize_text_field($parts[0]);
                    $field_value = $parts[1]; // Don't sanitize value to allow exact match

                    // Try to get from cache first to avoid slow meta_query
                    $cache_key = 'instarank_cf_' . md5($post_type . $field_name . $field_value);
                    $cached_post_id = wp_cache_get($cache_key, 'instarank');

                    if ($cached_post_id !== false) {
                        $found_post = get_post($cached_post_id);
                    } else {
                        $args = [
                            'post_type' => $post_type,
                            'post_status' => ['publish', 'pending', 'draft', 'private'],
                            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Required for custom field matching feature
                            'meta_query' => [
                                [
                                    'key' => $field_name,
                                    'value' => $field_value,
                                    'compare' => '=',
                                ],
                            ],
                            'numberposts' => 1,
                        ];
                        $posts = get_posts($args);
                        if (!empty($posts)) {
                            $found_post = $posts[0];
                            // Cache for 5 minutes
                            wp_cache_set($cache_key, $found_post->ID, 'instarank', 300);
                        }
                    }
                }
                break;

            default:
                return new WP_Error(
                    'invalid_match_by',
                    "Invalid match_by value: {$match_by}. Use: wordpress_post_id, slug, title, or custom_field",
                    ['status' => 400]
                );
        }

        if ($found_post) {
            return rest_ensure_response([
                'exists' => true,
                'post_id' => $found_post->ID,
                'title' => $found_post->post_title,
                'slug' => $found_post->post_name,
                'status' => $found_post->post_status,
                'edit_url' => get_edit_post_link($found_post->ID, 'raw'),
                'permalink' => get_permalink($found_post->ID),
            ]);
        }

        return rest_ensure_response([
            'exists' => false,
            'post_id' => null,
        ]);
    }

    /**
     * Delete a programmatic page from WordPress
     */
    public function delete_programmatic_page($request) {
        $post_id = intval($request->get_param('id'));

        if (!$post_id) {
            return new WP_Error(
                'missing_id',
                'Post ID is required',
                ['status' => 400]
            );
        }

        $post = get_post($post_id);

        if (!$post) {
            return new WP_Error(
                'post_not_found',
                "Post with ID {$post_id} not found",
                ['status' => 404]
            );
        }

        // Check if this is an InstaRank-generated post
        $is_pseo_post = get_post_meta($post_id, '_instarank_pseo_generated', true);

        // Force delete (bypass trash) if requested
        $force_delete = $request->get_param('force') === 'true' || $request->get_param('force') === true;

        // Delete the post (move to trash or permanently delete)
        $result = wp_delete_post($post_id, $force_delete);

        if (!$result) {
            return new WP_Error(
                'delete_failed',
                "Failed to delete post with ID {$post_id}",
                ['status' => 500]
            );
        }

        return rest_ensure_response([
            'success' => true,
            'post_id' => $post_id,
            'deleted' => true,
            'trashed' => !$force_delete,
            'message' => $force_delete ? 'Post permanently deleted' : 'Post moved to trash',
        ]);
    }

    /**
     * Get all taxonomies and their terms
     */
    public function get_taxonomies($request) {
        $taxonomies = get_taxonomies([
            'public' => true,
            '_builtin' => false
        ], 'objects');

        $taxonomy_data = [];

        foreach ($taxonomies as $taxonomy) {
            $terms = get_terms([
                'taxonomy' => $taxonomy->name,
                'hide_empty' => false,
                'number' => 50
            ]);

            $term_data = [];
            foreach ($terms as $term) {
                $term_url = get_term_link($term);
                if (!is_wp_error($term_url)) {
                    $term_data[] = [
                        'id' => $term->term_id,
                        'name' => $term->name,
                        'slug' => $term->slug,
                        'url' => $term_url,
                        'count' => $term->count
                    ];
                }
            }

            $taxonomy_data[] = [
                'name' => $taxonomy->name,
                'label' => $taxonomy->label,
                'singular_label' => $taxonomy->labels->singular_name,
                'hierarchical' => $taxonomy->hierarchical,
                'public' => $taxonomy->public,
                'terms' => $term_data,
                'term_count' => wp_count_terms(['taxonomy' => $taxonomy->name])
            ];
        }

        return rest_ensure_response([
            'success' => true,
            'taxonomies' => $taxonomy_data
        ]);
    }

    /**
     * Update category SEO meta
     */
    public function update_category_meta($request) {
        $category_id = intval($request['id']);
        $params = $request->get_json_params();

        return $this->update_term_meta_generic($category_id, 'category', $params);
    }

    /**
     * Update tag SEO meta
     */
    public function update_tag_meta($request) {
        $tag_id = intval($request['id']);
        $params = $request->get_json_params();

        return $this->update_term_meta_generic($tag_id, 'post_tag', $params);
    }

    /**
     * Generic method to update term (category/tag) SEO meta
     */
    private function update_term_meta_generic($term_id, $taxonomy, $params) {
        $change_type = isset($params['change_type']) ? sanitize_text_field($params['change_type']) : '';
        $new_value = isset($params['new_value']) ? wp_kses_post($params['new_value']) : '';

        if (empty($change_type) || empty($new_value)) {
            return new WP_Error('invalid_request', __('change_type and new_value are required', 'instarank'), ['status' => 400]);
        }

        $detector = new InstaRank_SEO_Detector();
        $seo_plugin = $detector->get_active_seo_plugin();

        // Update based on active SEO plugin
        switch ($seo_plugin) {
            case 'yoast':
                return $this->update_yoast_term_meta($term_id, $taxonomy, $change_type, $new_value);

            case 'rankmath':
                return $this->update_rankmath_term_meta($term_id, $taxonomy, $change_type, $new_value);

            case 'aioseo':
                return $this->update_aioseo_term_meta($term_id, $taxonomy, $change_type, $new_value);

            default:
                // Fallback: use WordPress core term meta
                return $this->update_core_term_meta($term_id, $change_type, $new_value);
        }
    }

    /**
     * Update Yoast SEO term meta
     */
    private function update_yoast_term_meta($term_id, $taxonomy, $change_type, $new_value) {
        $meta_key_map = [
            'meta_title' => 'wpseo_title',
            'meta_description' => 'wpseo_desc',
            'canonical_url' => 'wpseo_canonical'
        ];

        $meta_key = $meta_key_map[$change_type] ?? null;
        if (!$meta_key) {
            return new WP_Error('not_supported', 'Change type not supported');
        }

        update_term_meta($term_id, '_' . $meta_key, $new_value);

        return rest_ensure_response([
            'success' => true,
            'plugin' => 'yoast',
            'applied' => true
        ]);
    }

    /**
     * Update Rank Math term meta
     */
    private function update_rankmath_term_meta($term_id, $taxonomy, $change_type, $new_value) {
        $meta_key_map = [
            'meta_title' => 'rank_math_title',
            'meta_description' => 'rank_math_description',
            'canonical_url' => 'rank_math_canonical_url'
        ];

        $meta_key = $meta_key_map[$change_type] ?? null;
        if (!$meta_key) {
            return new WP_Error('not_supported', 'Change type not supported');
        }

        update_term_meta($term_id, $meta_key, $new_value);

        return rest_ensure_response([
            'success' => true,
            'plugin' => 'rankmath',
            'applied' => true
        ]);
    }

    /**
     * Update AIOSEO term meta
     */
    private function update_aioseo_term_meta($term_id, $taxonomy, $change_type, $new_value) {
        $meta_key_map = [
            'meta_title' => '_aioseo_title',
            'meta_description' => '_aioseo_description',
            'canonical_url' => '_aioseo_canonical_url'
        ];

        $meta_key = $meta_key_map[$change_type] ?? null;
        if (!$meta_key) {
            return new WP_Error('not_supported', 'Change type not supported');
        }

        update_term_meta($term_id, $meta_key, $new_value);

        return rest_ensure_response([
            'success' => true,
            'plugin' => 'aioseo',
            'applied' => true
        ]);
    }

    /**
     * Update core WordPress term meta (fallback)
     */
    private function update_core_term_meta($term_id, $change_type, $new_value) {
        $meta_key = 'instarank_' . $change_type;
        update_term_meta($term_id, $meta_key, $new_value);

        return rest_ensure_response([
            'success' => true,
            'plugin' => 'core',
            'applied' => true
        ]);
    }

    /**
     * Get or create attachment from URL
     * Used for setting featured images from remote URLs
     *
     * @param string $url The image URL
     * @param int $post_id The post ID to attach the image to
     * @return int|false Attachment ID or false on failure
     */
    private function get_or_create_attachment_from_url($url, $post_id) {
        global $wpdb;

        // First check if we already have this URL as an attachment
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $existing_id = $wpdb->get_var($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = '_instarank_source_url' AND meta_value = %s LIMIT 1",
            $url
        ));

        if ($existing_id) {
            return intval($existing_id);
        }

        // Also check by filename in media library
        $filename = basename(wp_parse_url($url, PHP_URL_PATH));
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $existing_by_name = $wpdb->get_var($wpdb->prepare(
            "SELECT ID FROM {$wpdb->posts} WHERE post_type = 'attachment' AND post_title = %s LIMIT 1",
            pathinfo($filename, PATHINFO_FILENAME)
        ));

        if ($existing_by_name) {
            return intval($existing_by_name);
        }

        // Download and sideload the image
        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';

        // Download image to temp file
        $tmp = download_url($url);
        if (is_wp_error($tmp)) {
            return false;
        }

        // Prepare file array for media_handle_sideload
        $file_array = [
            'name' => $filename,
            'tmp_name' => $tmp
        ];

        // Handle the sideload
        $attachment_id = media_handle_sideload($file_array, $post_id);

        // Clean up temp file
        if (file_exists($tmp)) {
            // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink
            unlink($tmp);
        }

        if (is_wp_error($attachment_id)) {
            return false;
        }

        // Store the source URL for future lookups
        update_post_meta($attachment_id, '_instarank_source_url', $url);

        return $attachment_id;
    }

    /**
     * Auto-import all external images in content to WordPress Media Library
     * Replaces external URLs with local WordPress URLs
     *
     * @param string $content    The post content (HTML/block content).
     * @param int    $post_id    The post ID to attach images to.
     * @param string $post_title Optional post title for generating alt text.
     * @return array Contains 'content' (modified), 'imported_count', 'skipped_count', 'images' (details).
     */
    private function auto_import_external_images($content, $post_id, $post_title = '') {
        if (empty($content) || !is_string($content)) {
            return [
                'content' => $content,
                'imported_count' => 0,
                'skipped_count' => 0,
                'images' => [],
            ];
        }

        $site_url = get_site_url();
        $imported_count = 0;
        $skipped_count = 0;
        $images = [];

        // Regex patterns to find image URLs in content
        // Pattern 1: img src attributes
        // Pattern 2: background-image URLs
        // Pattern 3: Kadence/Gutenberg image URLs in JSON attributes
        $patterns = [
            // Match img src="URL" (handles both single and double quotes)
            '/<img[^>]+src=["\']([^"\']+)["\'][^>]*>/i',
            // Match background-image: url(URL) or url('URL') or url("URL")
            '/background-image:\s*url\(["\']?([^"\')]+)["\']?\)/i',
            // Match Kadence/Gutenberg image URLs in block JSON attributes
            '/"url"\s*:\s*"([^"]+\.(jpg|jpeg|png|gif|webp|svg))"/i',
            '/"imgURL"\s*:\s*"([^"]+\.(jpg|jpeg|png|gif|webp|svg))"/i',
            '/"backgroundImg"\s*:\s*\[\s*\{\s*"[^"]*"\s*:\s*"([^"]+\.(jpg|jpeg|png|gif|webp|svg))"/i',
        ];

        // Collect all unique external image URLs
        $external_urls = [];

        foreach ($patterns as $pattern) {
            if (preg_match_all($pattern, $content, $matches)) {
                foreach ($matches[1] as $url) {
                    // Clean URL (remove escaped slashes from JSON)
                    $clean_url = str_replace('\\/', '/', $url);
                    $clean_url = html_entity_decode($clean_url, ENT_QUOTES, 'UTF-8');

                    // Skip if not a valid URL
                    if (!filter_var($clean_url, FILTER_VALIDATE_URL)) {
                        continue;
                    }

                    // Skip if already a local URL
                    if (strpos($clean_url, $site_url) === 0) {
                        continue;
                    }

                    // Skip data: URLs
                    if (strpos($clean_url, 'data:') === 0) {
                        continue;
                    }

                    // Skip non-image URLs (check extension or common image CDNs)
                    if (!$this->is_image_url($clean_url)) {
                        continue;
                    }

                    // Store original URL pattern for replacement
                    $external_urls[$clean_url] = $url;
                }
            }
        }

        // Import each unique external image
        foreach ($external_urls as $clean_url => $original_url_pattern) {
            // Generate smart alt text
            $alt_text = $this->generate_smart_alt_text($clean_url, $content, $post_title);

            // Import the image (with duplicate detection)
            $attachment_id = $this->get_or_create_attachment_from_url($clean_url, $post_id);

            if ($attachment_id) {
                // Get the new local URL
                $local_url = wp_get_attachment_url($attachment_id);

                if ($local_url) {
                    // Update alt text if we generated one
                    if (!empty($alt_text)) {
                        update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt_text);
                    }

                    // Replace all occurrences of this URL in content
                    // Handle both clean URL and escaped version
                    $content = str_replace($clean_url, $local_url, $content);
                    $content = str_replace(str_replace('/', '\\/', $clean_url), str_replace('/', '\\/', $local_url), $content);

                    // If original pattern was different (HTML entities, etc.), replace that too
                    if ($original_url_pattern !== $clean_url) {
                        $content = str_replace($original_url_pattern, $local_url, $content);
                    }

                    $imported_count++;
                    $images[] = [
                        'original_url' => $clean_url,
                        'local_url' => $local_url,
                        'attachment_id' => $attachment_id,
                        'alt_text' => $alt_text,
                    ];
                } else {
                    $skipped_count++;
                }
            } else {
                $skipped_count++;
            }
        }

        return [
            'content' => $content,
            'imported_count' => $imported_count,
            'skipped_count' => $skipped_count,
            'images' => $images,
        ];
    }

    /**
     * Check if a URL points to an image
     *
     * @param string $url The URL to check.
     * @return bool True if this appears to be an image URL.
     */
    private function is_image_url($url) {
        // Check file extension
        $image_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg', 'bmp', 'ico', 'avif'];
        $path = wp_parse_url($url, PHP_URL_PATH);
        if ($path) {
            $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION));
            if (in_array($ext, $image_extensions, true)) {
                return true;
            }
        }

        // Check for common image CDN patterns
        $image_cdn_patterns = [
            'unsplash.com',
            'pexels.com',
            'pixabay.com',
            'cloudinary.com',
            'imgix.net',
            'images.unsplash.com',
            'images.pexels.com',
            'cdn.pixabay.com',
            'wp-content/uploads',
            'amazonaws.com',
            'googleusercontent.com',
            'cloudfront.net',
        ];

        foreach ($image_cdn_patterns as $pattern) {
            if (strpos($url, $pattern) !== false) {
                return true;
            }
        }

        return false;
    }

    /**
     * Generate smart alt text from filename, nearby content, or post title
     *
     * @param string $url        The image URL.
     * @param string $content    The surrounding content.
     * @param string $post_title The post title as fallback.
     * @return string The generated alt text.
     */
    private function generate_smart_alt_text($url, $content, $post_title = '') {
        // Try to extract alt text from nearby heading in content
        // Look for headings (h1-h6) that appear near the image URL
        $alt_from_heading = $this->find_nearby_heading($url, $content);
        if (!empty($alt_from_heading)) {
            return $alt_from_heading;
        }

        // Try to extract meaningful text from filename
        $path = wp_parse_url($url, PHP_URL_PATH);
        if ($path) {
            $filename = pathinfo($path, PATHINFO_FILENAME);

            // Clean up filename: remove numbers, underscores, dashes
            $cleaned = preg_replace('/[-_]+/', ' ', $filename);
            $cleaned = preg_replace('/\d+/', '', $cleaned);
            $cleaned = trim($cleaned);

            // Check if we have a meaningful filename (at least 3 chars)
            if (strlen($cleaned) >= 3) {
                // Capitalize words
                return ucwords(strtolower($cleaned));
            }
        }

        // Fall back to post title if available
        if (!empty($post_title)) {
            return $post_title;
        }

        return '';
    }

    /**
     * Find a nearby heading in content relative to an image URL
     *
     * @param string $url     The image URL to find context for.
     * @param string $content The HTML content.
     * @return string The heading text if found, empty string otherwise.
     */
    private function find_nearby_heading($url, $content) {
        // Find position of the image URL
        $url_pos = strpos($content, $url);
        if ($url_pos === false) {
            // Try escaped version
            $url_pos = strpos($content, str_replace('/', '\\/', $url));
        }

        if ($url_pos === false) {
            return '';
        }

        // Look for headings before the image (within ~500 chars)
        $search_start = max(0, $url_pos - 500);
        $before_content = substr($content, $search_start, $url_pos - $search_start);

        // Match h1-h6 headings
        if (preg_match_all('/<h[1-6][^>]*>([^<]+)<\/h[1-6]>/i', $before_content, $matches)) {
            // Return the last heading before the image (most relevant)
            $headings = $matches[1];
            if (!empty($headings)) {
                $heading = end($headings);
                // Clean up the heading text
                $heading = wp_strip_all_tags($heading);
                $heading = trim($heading);
                if (strlen($heading) >= 3 && strlen($heading) <= 125) {
                    return $heading;
                }
            }
        }

        // Also check for Gutenberg heading blocks
        if (preg_match_all('/<!-- wp:heading[^>]*-->[^<]*<h[1-6][^>]*>([^<]+)<\/h[1-6]>/i', $before_content, $matches)) {
            $headings = $matches[1];
            if (!empty($headings)) {
                $heading = end($headings);
                $heading = wp_strip_all_tags($heading);
                $heading = trim($heading);
                if (strlen($heading) >= 3 && strlen($heading) <= 125) {
                    return $heading;
                }
            }
        }

        return '';
    }

    /**
     * Update author SEO meta
     */
    public function update_author_meta($request) {
        $author_id = intval($request['id']);
        $params = $request->get_json_params();

        $change_type = isset($params['change_type']) ? sanitize_text_field($params['change_type']) : '';
        $new_value = isset($params['new_value']) ? wp_kses_post($params['new_value']) : '';

        if (empty($change_type) || empty($new_value)) {
            return new WP_Error('invalid_request', __('change_type and new_value are required', 'instarank'), ['status' => 400]);
        }

        // Update author meta based on change type
        switch ($change_type) {
            case 'description':
                update_user_meta($author_id, 'description', $new_value);
                break;

            case 'meta_title':
                update_user_meta($author_id, 'instarank_meta_title', $new_value);
                break;

            case 'meta_description':
                update_user_meta($author_id, 'instarank_meta_description', $new_value);
                break;

            default:
                return new WP_Error('not_supported', 'Change type not supported');
        }

        return rest_ensure_response([
            'success' => true,
            'applied' => true
        ]);
    }

    /**
     * Get author SEO meta
     */
    private function get_author_seo_meta($author_id) {
        return [
            'meta_title' => get_user_meta($author_id, 'instarank_meta_title', true),
            'meta_description' => get_user_meta($author_id, 'instarank_meta_description', true),
            'description' => get_user_meta($author_id, 'description', true)
        ];
    }

    /**
     * Handle search page requests
     */
    public function handle_search_page($request) {
        if ($request->get_method() === 'GET') {
            return rest_ensure_response([
                'type' => 'search',
                'title' => get_option('blogname') . ' - Search Results',
                'noindex' => get_option('instarank_search_noindex', false),
                'robots' => get_option('instarank_search_robots', 'noindex, follow')
            ]);
        }

        // POST - Update search page settings
        $data = $request->get_json_params();
        if (isset($data['title'])) {
            update_option('instarank_search_page_title', sanitize_text_field($data['title']));
        }
        if (isset($data['noindex'])) {
            update_option('instarank_search_noindex', (bool)$data['noindex']);
        }
        if (isset($data['robots'])) {
            update_option('instarank_search_robots', sanitize_text_field($data['robots']));
        }

        return rest_ensure_response([
            'success' => true,
            'message' => 'Search page settings updated'
        ]);
    }

    /**
     * Handle 404 page requests
     */
    public function handle_404_page($request) {
        if ($request->get_method() === 'GET') {
            $page_id = get_option('page_for_posts') ?: 0;
            $page = $page_id ? get_post($page_id) : null;

            return rest_ensure_response([
                'type' => '404',
                'page_id' => $page_id,
                'url' => $page ? get_permalink($page) : home_url('/404/'),
                'title' => $page ? $page->post_title : '404 Page Not Found',
                'content' => $page ? $page->post_content : 'The page you are looking for could not be found.'
            ]);
        }

        // POST - Update 404 page content
        $data = $request->get_json_params();
        if (isset($data['title']) || isset($data['content'])) {
            update_option('instarank_404_page_title', sanitize_text_field($data['title'] ?? ''));
            update_option('instarank_404_page_content', wp_kses_post($data['content'] ?? ''));
        }

        return rest_ensure_response([
            'success' => true,
            'message' => '404 page updated'
        ]);
    }

    /**
     * List attachments
     */
    public function list_attachments($request) {
        $limit = (int)$request->get_param('limit') ?: 1000;
        $paged = (int)$request->get_param('paged') ?: 1;

        $args = [
            'post_type' => 'attachment',
            'posts_per_page' => $limit,
            'paged' => $paged,
            'orderby' => 'date',
            'order' => 'DESC'
        ];

        $query = new \WP_Query($args);
        $attachments = [];

        foreach ($query->posts as $post) {
            $attachments[] = [
                'id' => $post->ID,
                'filename' => basename($post->guid),
                'mime_type' => $post->post_mime_type,
                'url' => wp_get_attachment_url($post->ID),
                'alt' => get_post_meta($post->ID, '_wp_attachment_image_alt', true),
                'title' => $post->post_title,
                'description' => $post->post_excerpt,
                'parent_id' => $post->post_parent
            ];
        }

        return rest_ensure_response([
            'attachments' => $attachments,
            'total' => $query->found_posts,
            'pages' => $query->max_num_pages
        ]);
    }

    /**
     * Get attachment info
     */
    public function get_attachment_info($request) {
        $attachment_id = (int)$request->get_param('id');
        $post = get_post($attachment_id);

        if (!$post || $post->post_type !== 'attachment') {
            return new \WP_Error('not_found', 'Attachment not found', ['status' => 404]);
        }

        return rest_ensure_response([
            'id' => $post->ID,
            'filename' => basename($post->guid),
            'mime_type' => $post->post_mime_type,
            'url' => wp_get_attachment_url($post->ID),
            'alt' => get_post_meta($post->ID, '_wp_attachment_image_alt', true),
            'title' => $post->post_title,
            'description' => $post->post_excerpt,
            'parent_id' => $post->post_parent
        ]);
    }

    /**
     * Update attachment metadata
     */
    public function update_attachment_meta($request) {
        $attachment_id = (int)$request->get_param('id');
        $post = get_post($attachment_id);

        if (!$post || $post->post_type !== 'attachment') {
            return new \WP_Error('not_found', 'Attachment not found', ['status' => 404]);
        }

        $data = $request->get_json_params();

        // Update alt text
        if (isset($data['alt'])) {
            update_post_meta($attachment_id, '_wp_attachment_image_alt', sanitize_text_field($data['alt']));
        }

        // Update title
        if (isset($data['title'])) {
            wp_update_post([
                'ID' => $attachment_id,
                'post_title' => sanitize_text_field($data['title'])
            ]);
        }

        // Update description
        if (isset($data['description'])) {
            wp_update_post([
                'ID' => $attachment_id,
                'post_excerpt' => sanitize_text_field($data['description'])
            ]);
        }

        return rest_ensure_response([
            'success' => true,
            'message' => 'Attachment updated'
        ]);
    }

    /**
     * Check if WooCommerce is active
     */
    public function check_woocommerce_active($request) {
        $active = function_exists('WC') || class_exists('WooCommerce');
        $version = '';

        if ($active && function_exists('WC')) {
            $version = WC()->version;
        }

        return rest_ensure_response([
            'active' => $active,
            'version' => $version
        ]);
    }

    /**
     * List WooCommerce products
     */
    public function list_woocommerce_products($request) {
        if (!class_exists('WooCommerce')) {
            return new \WP_Error('woocommerce_inactive', 'WooCommerce is not active', ['status' => 400]);
        }

        $limit = (int)$request->get_param('limit') ?: 1000;
        $paged = (int)$request->get_param('paged') ?: 1;

        $args = [
            'post_type' => 'product',
            'posts_per_page' => $limit,
            'paged' => $paged,
            'orderby' => 'date',
            'order' => 'DESC'
        ];

        $query = new \WP_Query($args);
        $products = [];

        foreach ($query->posts as $post) {
            $product = wc_get_product($post->ID);
            if ($product) {
                $products[] = $this->format_product_data($product);
            }
        }

        return rest_ensure_response([
            'products' => $products,
            'total' => $query->found_posts,
            'pages' => $query->max_num_pages
        ]);
    }

    /**
     * Get WooCommerce product info
     */
    public function get_woocommerce_product($request) {
        if (!class_exists('WooCommerce')) {
            return new \WP_Error('woocommerce_inactive', 'WooCommerce is not active', ['status' => 400]);
        }

        $product_id = (int)$request->get_param('id');
        $product = wc_get_product($product_id);

        if (!$product) {
            return new \WP_Error('not_found', 'Product not found', ['status' => 404]);
        }

        return rest_ensure_response($this->format_product_data($product));
    }

    /**
     * Update WooCommerce product metadata
     */
    public function update_woocommerce_product_meta($request) {
        if (!class_exists('WooCommerce')) {
            return new \WP_Error('woocommerce_inactive', 'WooCommerce is not active', ['status' => 400]);
        }

        $product_id = (int)$request->get_param('id');
        $product = wc_get_product($product_id);

        if (!$product) {
            return new \WP_Error('not_found', 'Product not found', ['status' => 404]);
        }

        $data = $request->get_json_params();

        // Update post title
        if (isset($data['title'])) {
            wp_update_post(['ID' => $product_id, 'post_title' => sanitize_text_field($data['title'])]);
        }

        // Update post content (description)
        if (isset($data['description'])) {
            wp_update_post(['ID' => $product_id, 'post_content' => wp_kses_post($data['description'])]);
        }

        // Update product excerpt (short description)
        if (isset($data['shortDescription'])) {
            wp_update_post(['ID' => $product_id, 'post_excerpt' => wp_kses_post($data['shortDescription'])]);
        }

        return rest_ensure_response(['success' => true, 'message' => 'Product updated']);
    }

    /**
     * Handle WooCommerce shop page
     */
    public function handle_woocommerce_shop($request) {
        if (!class_exists('WooCommerce')) {
            return new \WP_Error('woocommerce_inactive', 'WooCommerce is not active', ['status' => 400]);
        }

        if ($request->get_method() === 'GET') {
            $shop_page_id = wc_get_page_id('shop');
            $shop_page = $shop_page_id ? get_post($shop_page_id) : null;

            return rest_ensure_response([
                'url' => $shop_page ? get_permalink($shop_page) : wc_get_page_permalink('shop'),
                'page_id' => $shop_page_id,
                'title' => $shop_page ? $shop_page->post_title : 'Shop',
                'description' => $shop_page ? $shop_page->post_excerpt : '',
                'product_count' => wp_count_posts('product')->publish ?? 0
            ]);
        }

        // POST - Update shop settings
        $data = $request->get_json_params();
        if (isset($data['title']) || isset($data['description'])) {
            $shop_page_id = wc_get_page_id('shop');
            if ($shop_page_id) {
                wp_update_post([
                    'ID' => $shop_page_id,
                    'post_title' => sanitize_text_field($data['title'] ?? ''),
                    'post_excerpt' => sanitize_text_field($data['description'] ?? '')
                ]);
            }
        }

        return rest_ensure_response(['success' => true, 'message' => 'Shop page updated']);
    }

    /**
     * List WooCommerce product categories
     */
    public function list_woocommerce_categories($request) {
        if (!class_exists('WooCommerce')) {
            return new \WP_Error('woocommerce_inactive', 'WooCommerce is not active', ['status' => 400]);
        }

        $categories = get_terms(['taxonomy' => 'product_cat', 'hide_empty' => false]);
        $formatted_categories = [];

        foreach ($categories as $category) {
            $formatted_categories[] = [
                'id' => $category->term_id,
                'name' => $category->name,
                'slug' => $category->slug,
                'url' => get_term_link($category),
                'description' => $category->description,
                'product_count' => $category->count
            ];
        }

        return rest_ensure_response(['categories' => $formatted_categories]);
    }

    /**
     * Update WooCommerce product category metadata
     */
    public function update_woocommerce_category_meta($request) {
        if (!class_exists('WooCommerce')) {
            return new \WP_Error('woocommerce_inactive', 'WooCommerce is not active', ['status' => 400]);
        }

        $category_id = (int)$request->get_param('id');
        $category = get_term($category_id, 'product_cat');

        if (!$category || is_wp_error($category)) {
            return new \WP_Error('not_found', 'Category not found', ['status' => 404]);
        }

        $data = $request->get_json_params();

        if (isset($data['name']) || isset($data['description'])) {
            wp_update_term($category_id, 'product_cat', [
                'name' => sanitize_text_field($data['name'] ?? $category->name),
                'description' => wp_kses_post($data['description'] ?? $category->description)
            ]);
        }

        return rest_ensure_response(['success' => true, 'message' => 'Category updated']);
    }

    /**
     * Get WooCommerce cart page
     */
    public function get_woocommerce_cart_page($request) {
        if (!class_exists('WooCommerce')) {
            return new \WP_Error('woocommerce_inactive', 'WooCommerce is not active', ['status' => 400]);
        }

        $cart_page_id = wc_get_page_id('cart');
        $cart_page = $cart_page_id ? get_post($cart_page_id) : null;

        return rest_ensure_response([
            'type' => 'cart',
            'url' => wc_get_page_permalink('cart'),
            'page_id' => $cart_page_id,
            'title' => $cart_page ? $cart_page->post_title : 'Cart',
            'has_custom_page' => $cart_page_id > 0
        ]);
    }

    /**
     * Get WooCommerce checkout page
     */
    public function get_woocommerce_checkout_page($request) {
        if (!class_exists('WooCommerce')) {
            return new \WP_Error('woocommerce_inactive', 'WooCommerce is not active', ['status' => 400]);
        }

        $checkout_page_id = wc_get_page_id('checkout');
        $checkout_page = $checkout_page_id ? get_post($checkout_page_id) : null;

        return rest_ensure_response([
            'type' => 'checkout',
            'url' => wc_get_page_permalink('checkout'),
            'page_id' => $checkout_page_id,
            'title' => $checkout_page ? $checkout_page->post_title : 'Checkout',
            'has_custom_page' => $checkout_page_id > 0
        ]);
    }

    /**
     * Format product data
     */
    private function format_product_data($product) {
        return [
            'id' => $product->get_id(),
            'title' => $product->get_title(),
            'slug' => $product->get_slug(),
            'url' => $product->get_permalink(),
            'description' => $product->get_description(),
            'short_description' => $product->get_short_description(),
            'price' => $product->get_price(),
            'sale_price' => $product->get_sale_price(),
            'sku' => $product->get_sku(),
            'stock_status' => $product->get_stock_status(),
            'featured_image' => wp_get_attachment_url($product->get_image_id()),
            'categories' => wp_get_post_terms($product->get_id(), 'product_cat', ['fields' => 'ids']),
            'tags' => wp_get_post_terms($product->get_id(), 'product_tag', ['fields' => 'ids'])
        ];
    }

    /**
     * Get comprehensive page types information
     * Returns all supported page types and their SEO capabilities
     */
    public function get_page_types_info() {
        $page_types = [
            'post' => [
                'name' => 'Blog Post',
                'description' => 'Standard blog post/article',
                'seo_fields' => ['title', 'meta_description', 'h1', 'content', 'canonical', 'open_graph'],
                'supports_meta' => true,
                'pattern' => '/\d{4}/\d{2}/.*/',
                'examples' => ['example.com/2025/01/my-post']
            ],
            'page' => [
                'name' => 'Page',
                'description' => 'Static page (About, Contact, etc)',
                'seo_fields' => ['title', 'meta_description', 'h1', 'content', 'canonical'],
                'supports_meta' => true,
                'pattern' => '/[^/]+/?$',
                'examples' => ['example.com/about', 'example.com/contact']
            ],
            'homepage' => [
                'name' => 'Homepage',
                'description' => 'Site homepage',
                'seo_fields' => ['title', 'meta_description', 'content', 'open_graph'],
                'supports_meta' => true,
                'pattern' => '/$',
                'examples' => ['example.com/']
            ],
            'category' => [
                'name' => 'Category Archive',
                'description' => 'Category archive page',
                'seo_fields' => ['title', 'meta_description', 'canonical'],
                'supports_meta' => true,
                'pattern' => '/category/[^/]+/?$',
                'examples' => ['example.com/category/news', 'example.com/category/business']
            ],
            'tag' => [
                'name' => 'Tag Archive',
                'description' => 'Tag archive page',
                'seo_fields' => ['title', 'meta_description', 'canonical'],
                'supports_meta' => true,
                'pattern' => '/tag/[^/]+/?$',
                'examples' => ['example.com/tag/seo', 'example.com/tag/marketing']
            ],
            'author' => [
                'name' => 'Author Archive',
                'description' => 'Author profile/archive page',
                'seo_fields' => ['title', 'meta_description', 'bio'],
                'supports_meta' => true,
                'pattern' => '/author/[^/]+/?$',
                'examples' => ['example.com/author/john-doe']
            ],
            'search' => [
                'name' => 'Search Results',
                'description' => 'Search results page',
                'seo_fields' => ['title', 'noindex'],
                'supports_meta' => true,
                'pattern' => '/\?s=.*/',
                'examples' => ['example.com/?s=keyword']
            ],
            '404' => [
                'name' => '404 Page',
                'description' => 'Page not found error page',
                'seo_fields' => ['title', 'content', 'canonical'],
                'supports_meta' => true,
                'pattern' => '/(404|not-found)/?$',
                'examples' => ['example.com/404', 'example.com/not-found']
            ],
            'attachment' => [
                'name' => 'Attachment/Media',
                'description' => 'Image or media attachment page',
                'seo_fields' => ['alt_text', 'title', 'description'],
                'supports_meta' => true,
                'pattern' => '/[0-9]{4}/[0-9]{2}/[^/]+/',
                'examples' => ['example.com/2025/01/image.jpg']
            ],
            'product' => [
                'name' => 'WooCommerce Product',
                'description' => 'Product page (WooCommerce)',
                'seo_fields' => ['title', 'meta_description', 'short_description', 'schema_markup', 'images'],
                'supports_meta' => true,
                'requires_plugin' => 'WooCommerce',
                'pattern' => '/product/[^/]+/?$',
                'examples' => ['example.com/product/my-product']
            ],
            'shop' => [
                'name' => 'Shop Page',
                'description' => 'Shop/products listing page (WooCommerce)',
                'seo_fields' => ['title', 'meta_description'],
                'supports_meta' => true,
                'requires_plugin' => 'WooCommerce',
                'pattern' => '/shop/?$',
                'examples' => ['example.com/shop']
            ],
            'product_category' => [
                'name' => 'Product Category',
                'description' => 'Product category archive (WooCommerce)',
                'seo_fields' => ['title', 'meta_description'],
                'supports_meta' => true,
                'requires_plugin' => 'WooCommerce',
                'pattern' => '/product-category/[^/]+/?$',
                'examples' => ['example.com/product-category/electronics']
            ],
            'product_tag' => [
                'name' => 'Product Tag',
                'description' => 'Product tag archive (WooCommerce)',
                'seo_fields' => ['title', 'meta_description'],
                'supports_meta' => true,
                'requires_plugin' => 'WooCommerce',
                'pattern' => '/product-tag/[^/]+/?$',
                'examples' => ['example.com/product-tag/sale']
            ],
            'cart' => [
                'name' => 'Cart Page',
                'description' => 'Shopping cart page (WooCommerce)',
                'seo_fields' => ['noindex'],
                'supports_meta' => true,
                'requires_plugin' => 'WooCommerce',
                'pattern' => '/cart/?$',
                'examples' => ['example.com/cart']
            ],
            'checkout' => [
                'name' => 'Checkout Page',
                'description' => 'Checkout page (WooCommerce)',
                'seo_fields' => ['noindex'],
                'supports_meta' => true,
                'requires_plugin' => 'WooCommerce',
                'pattern' => '/checkout/?$',
                'examples' => ['example.com/checkout']
            ]
        ];

        // Get active plugins
        $active_plugins = get_option('active_plugins', []);
        $has_woocommerce = false;
        foreach ($active_plugins as $plugin) {
            if (strpos($plugin, 'woocommerce') !== false) {
                $has_woocommerce = true;
                break;
            }
        }

        // Filter out WooCommerce types if not active
        if (!$has_woocommerce) {
            $page_types = array_filter($page_types, function($type) {
                return !isset($type['requires_plugin']);
            });
        }

        return [
            'success' => true,
            'page_types' => $page_types,
            'total_types' => count($page_types),
            'plugins' => [
                'woocommerce' => [
                    'active' => $has_woocommerce,
                    'description' => 'E-commerce plugin'
                ]
            ]
        ];
    }

    /**
     * Upload optimized image to WordPress Media Library
     * POST /wp-json/instarank/v1/media/upload-optimized
     *
     * Expected JSON body:
     * {
     *   "image_url": "https://...",
     *   "filename": "optimized-image.webp",
     *   "alt_text": "Description",
     *   "title": "Image Title",
     *   "caption": "Image Caption",
     *   "description": "Image Description"
     * }
     */
    public function upload_optimized_image($request) {
        $params = $request->get_json_params();

        // Validate required params
        if (empty($params['image_url'])) {
            return new WP_Error('missing_image_url', 'image_url is required', ['status' => 400]);
        }

        $image_url = $params['image_url'];
        $filename = !empty($params['filename']) ? $params['filename'] : basename(wp_parse_url($image_url, PHP_URL_PATH));
        $alt_text = !empty($params['alt_text']) ? $params['alt_text'] : '';
        $title = !empty($params['title']) ? $params['title'] : '';
        $caption = !empty($params['caption']) ? $params['caption'] : '';
        $description = !empty($params['description']) ? $params['description'] : '';

        // Require WordPress media functions
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');

        // Download image to temp file
        $temp_file = download_url($image_url);

        if (is_wp_error($temp_file)) {
            return new WP_Error(
                'download_failed',
                'Failed to download image: ' . $temp_file->get_error_message(),
                ['status' => 500]
            );
        }

        // Prepare file array for media_handle_sideload
        $file_array = [
            'name' => $filename,
            'tmp_name' => $temp_file
        ];

        // Upload to Media Library
        $attachment_id = media_handle_sideload($file_array, 0);

        // Clean up temp file
        if (file_exists($temp_file)) {
            wp_delete_file($temp_file);
        }

        if (is_wp_error($attachment_id)) {
            return new WP_Error(
                'upload_failed',
                'Failed to upload to Media Library: ' . $attachment_id->get_error_message(),
                ['status' => 500]
            );
        }

        // Update image metadata
        if (!empty($alt_text)) {
            update_post_meta($attachment_id, '_wp_attachment_image_alt', sanitize_text_field($alt_text));
        }

        // Update post data
        $update_data = ['ID' => $attachment_id];
        if (!empty($title)) {
            $update_data['post_title'] = sanitize_text_field($title);
        }
        if (!empty($caption)) {
            $update_data['post_excerpt'] = sanitize_text_field($caption);
        }
        if (!empty($description)) {
            $update_data['post_content'] = sanitize_text_field($description);
        }

        if (count($update_data) > 1) {
            wp_update_post($update_data);
        }

        // Get attachment URL
        $attachment_url = wp_get_attachment_url($attachment_id);

        return [
            'success' => true,
            'attachment_id' => $attachment_id,
            'attachment_url' => $attachment_url,
            'filename' => $filename,
            'message' => 'Image uploaded successfully to WordPress Media Library'
        ];
    }

    /**
     * Replace image URLs in WordPress content
     * POST /wp-json/instarank/v1/media/replace-urls
     *
     * Expected JSON body:
     * {
     *   "replacements": [
     *     {"old_url": "https://old.com/image.png", "new_url": "https://new.com/image.webp", "attachment_id": 123},
     *     ...
     *   ]
     * }
     */
    public function replace_image_urls($request) {
        $params = $request->get_json_params();

        if (empty($params['replacements']) || !is_array($params['replacements'])) {
            return new WP_Error('invalid_replacements', 'replacements array is required', ['status' => 400]);
        }

        $replacements = $params['replacements'];
        $results = [
            'posts_updated' => 0,
            'replacements_made' => 0,
            'details' => []
        ];

        foreach ($replacements as $replacement) {
            if (empty($replacement['old_url']) || empty($replacement['new_url'])) {
                continue;
            }

            $old_url = $replacement['old_url'];
            $new_url = $replacement['new_url'];
            $attachment_id = !empty($replacement['attachment_id']) ? intval($replacement['attachment_id']) : null;

            // Find all posts containing the old URL
            global $wpdb;
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $posts = $wpdb->get_results($wpdb->prepare(
                "SELECT ID, post_content FROM {$wpdb->posts} WHERE post_content LIKE %s AND post_status = 'publish'",
                '%' . $wpdb->esc_like($old_url) . '%'
            ));

            $posts_updated_for_this_url = 0;

            foreach ($posts as $post) {
                // Replace URL in post content
                $updated_content = str_replace($old_url, $new_url, $post->post_content);

                if ($updated_content !== $post->post_content) {
                    wp_update_post([
                        'ID' => $post->ID,
                        'post_content' => $updated_content
                    ]);

                    $posts_updated_for_this_url++;
                    $results['replacements_made']++;
                }
            }

            // Also check for featured images
            if ($attachment_id) {
                $meta_posts = get_posts([
                    'post_type' => 'any',
                    'meta_query' => [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Required for finding posts with specific featured image
                        [
                            'key' => '_thumbnail_id',
                            'value' => $attachment_id,
                            'compare' => '='
                        ]
                    ],
                    'posts_per_page' => -1
                ]);

                // Update featured image references if needed
                foreach ($meta_posts as $meta_post) {
                    // Featured image already uses attachment_id, no URL replacement needed
                    // But count it for reporting
                    $posts_updated_for_this_url++;
                }
            }

            $results['details'][] = [
                'old_url' => $old_url,
                'new_url' => $new_url,
                'posts_affected' => $posts_updated_for_this_url
            ];

            $results['posts_updated'] += $posts_updated_for_this_url;
        }

        return [
            'success' => true,
            'posts_updated' => $results['posts_updated'],
            'replacements_made' => $results['replacements_made'],
            'total_urls_processed' => count($replacements),
            'details' => $results['details']
        ];
    }

    /**
     * Bulk upload optimized images
     * POST /wp-json/instarank/v1/media/bulk-upload-optimized
     *
     * Expected JSON body:
     * {
     *   "images": [
     *     {"image_url": "https://...", "filename": "image.webp", "alt_text": "..."},
     *     ...
     *   ]
     * }
     */
    public function bulk_upload_optimized_images($request) {
        $params = $request->get_json_params();

        if (empty($params['images']) || !is_array($params['images'])) {
            return new WP_Error('invalid_images', 'images array is required', ['status' => 400]);
        }

        $images = $params['images'];
        $results = [
            'success_count' => 0,
            'fail_count' => 0,
            'details' => []
        ];

        foreach ($images as $image_data) {
            if (empty($image_data['image_url'])) {
                $results['fail_count']++;
                $results['details'][] = [
                    'success' => false,
                    'error' => 'Missing image_url'
                ];
                continue;
            }

            // Create a fake request for upload_optimized_image
            $fake_request = new WP_REST_Request('POST', '/wp-json/instarank/v1/media/upload-optimized');
            $fake_request->set_body(json_encode($image_data));
            $fake_request->add_header('content-type', 'application/json');

            $result = $this->upload_optimized_image($fake_request);

            if (is_wp_error($result)) {
                $results['fail_count']++;
                $results['details'][] = [
                    'success' => false,
                    'image_url' => $image_data['image_url'],
                    'error' => $result->get_error_message()
                ];
            } else {
                $results['success_count']++;
                $results['details'][] = array_merge(
                    ['success' => true],
                    $result
                );
            }
        }

        return [
            'success' => true,
            'total_processed' => count($images),
            'success_count' => $results['success_count'],
            'fail_count' => $results['fail_count'],
            'details' => $results['details']
        ];
    }

    /**
     * Bulk update post meta (for SaaS platform bulk operations)
     *
     * POST /wp-json/instarank/v1/bulk-meta
     * Body: {
     *   "updates": [
     *     {
     *       "post_id": 123,
     *       "meta": {
     *         "_instarank_meta_title": "New Title",
     *         "_instarank_meta_description": "New Description"
     *       }
     *     }
     *   ]
     * }
     */
    public function bulk_update_meta($request) {
        $updates = $request->get_param('updates');

        if (empty($updates) || !is_array($updates)) {
            return new WP_Error('invalid_data', 'Updates array is required', ['status' => 400]);
        }

        $results = [
            'success' => true,
            'total' => count($updates),
            'updated' => 0,
            'failed' => 0,
            'errors' => []
        ];

        foreach ($updates as $update) {
            $post_id = isset($update['post_id']) ? intval($update['post_id']) : 0;
            $meta = isset($update['meta']) ? $update['meta'] : [];

            if (empty($post_id) || empty($meta)) {
                $results['failed']++;
                $results['errors'][] = [
                    'post_id' => $post_id,
                    'error' => 'Missing post_id or meta'
                ];
                continue;
            }

            // Verify post exists
            if (!get_post($post_id)) {
                $results['failed']++;
                $results['errors'][] = [
                    'post_id' => $post_id,
                    'error' => 'Post not found'
                ];
                continue;
            }

            // Update each meta field
            $updated_fields = 0;
            foreach ($meta as $meta_key => $meta_value) {
                // Only allow InstaRank meta fields
                if (strpos($meta_key, '_instarank_') !== 0) {
                    continue;
                }

                update_post_meta($post_id, $meta_key, $meta_value);
                $updated_fields++;
            }

            if ($updated_fields > 0) {
                $results['updated']++;
            } else {
                $results['failed']++;
                $results['errors'][] = [
                    'post_id' => $post_id,
                    'error' => 'No valid meta fields to update'
                ];
            }
        }

        return $results;
    }

    /**
     * Bulk update robots meta (NoIndex/NoFollow)
     *
     * POST /wp-json/instarank/v1/bulk-robots
     * Body: {
     *   "post_ids": [123, 456, 789],
     *   "noindex": true,
     *   "nofollow": false
     * }
     */
    public function bulk_update_robots($request) {
        $post_ids = $request->get_param('post_ids');
        $noindex = $request->get_param('noindex');
        $nofollow = $request->get_param('nofollow');

        if (empty($post_ids) || !is_array($post_ids)) {
            return new WP_Error('invalid_data', 'post_ids array is required', ['status' => 400]);
        }

        $results = [
            'success' => true,
            'total' => count($post_ids),
            'updated' => 0,
            'failed' => 0
        ];

        foreach ($post_ids as $post_id) {
            $post_id = intval($post_id);

            if (!get_post($post_id)) {
                $results['failed']++;
                continue;
            }

            if ($noindex !== null) {
                update_post_meta($post_id, '_instarank_noindex', $noindex ? '1' : '0');
            }

            if ($nofollow !== null) {
                update_post_meta($post_id, '_instarank_nofollow', $nofollow ? '1' : '0');
            }

            $results['updated']++;
        }

        return $results;
    }

    /**
     * Get media library (for AI image alt text generation)
     *
     * GET /wp-json/instarank/v1/media?per_page=100&page=1&missing_alt=true
     */
    public function get_media_library($request) {
        $per_page = $request->get_param('per_page') ?: 100;
        $page = $request->get_param('page') ?: 1;
        $missing_alt = $request->get_param('missing_alt') === 'true';

        $args = [
            'post_type' => 'attachment',
            'post_status' => 'inherit',
            'posts_per_page' => min($per_page, 100),
            'paged' => $page,
            'post_mime_type' => 'image'
        ];

        // Filter for images missing alt text
        if ($missing_alt) {
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Necessary for filtering images without alt text
            $args['meta_query'] = [
                'relation' => 'OR',
                [
                    'key' => '_wp_attachment_image_alt',
                    'compare' => 'NOT EXISTS'
                ],
                [
                    'key' => '_wp_attachment_image_alt',
                    'value' => '',
                    'compare' => '='
                ]
            ];
        }

        $query = new WP_Query($args);
        $images = [];

        foreach ($query->posts as $attachment) {
            $images[] = [
                'id' => $attachment->ID,
                'url' => wp_get_attachment_url($attachment->ID),
                'title' => $attachment->post_title,
                'alt' => get_post_meta($attachment->ID, '_wp_attachment_image_alt', true),
                'caption' => $attachment->post_excerpt,
                'description' => $attachment->post_content,
                'filename' => basename(get_attached_file($attachment->ID)),
                'mime_type' => $attachment->post_mime_type,
                'file_size' => filesize(get_attached_file($attachment->ID))
            ];
        }

        return [
            'success' => true,
            'images' => $images,
            'total' => $query->found_posts,
            'page' => $page,
            'per_page' => $per_page,
            'total_pages' => $query->max_num_pages
        ];
    }

    /**
     * Update image alt text
     *
     * POST /wp-json/instarank/v1/media/123/alt-text
     * Body: { "alt_text": "New alt text" }
     */
    public function update_image_alt_text($request) {
        $attachment_id = $request->get_param('id');
        $alt_text = $request->get_param('alt_text');

        if (empty($attachment_id)) {
            return new WP_Error('invalid_id', 'Attachment ID is required', ['status' => 400]);
        }

        // Verify attachment exists
        if (!wp_attachment_is_image($attachment_id)) {
            return new WP_Error('invalid_attachment', 'Invalid image attachment', ['status' => 404]);
        }

        // Update alt text
        update_post_meta($attachment_id, '_wp_attachment_image_alt', sanitize_text_field($alt_text));

        return [
            'success' => true,
            'attachment_id' => $attachment_id,
            'alt_text' => $alt_text
        ];
    }

    /**
     * Update all image metadata (title, caption, description, alt text)
     *
     * POST /wp-json/instarank/v1/media/123/metadata
     * Body: { "title": "Image Title", "caption": "Short caption", "description": "Full description", "alt_text": "Alt text" }
     */
    public function update_image_metadata($request) {
        $attachment_id = $request->get_param('id');
        $title = $request->get_param('title');
        $caption = $request->get_param('caption');
        $description = $request->get_param('description');
        $alt_text = $request->get_param('alt_text');

        if (empty($attachment_id)) {
            return new WP_Error('invalid_id', 'Attachment ID is required', ['status' => 400]);
        }

        // Verify attachment exists
        $attachment = get_post($attachment_id);
        if (!$attachment || $attachment->post_type !== 'attachment') {
            return new WP_Error('invalid_attachment', 'Invalid attachment', ['status' => 404]);
        }

        $updated_fields = [];

        // Update post fields (title, caption/excerpt, description/content)
        $post_update = ['ID' => $attachment_id];

        if ($title !== null) {
            $post_update['post_title'] = sanitize_text_field($title);
            $updated_fields['title'] = $title;
        }

        if ($caption !== null) {
            $post_update['post_excerpt'] = sanitize_textarea_field($caption);
            $updated_fields['caption'] = $caption;
        }

        if ($description !== null) {
            $post_update['post_content'] = wp_kses_post($description);
            $updated_fields['description'] = $description;
        }

        // Update post if any post fields were provided
        if (count($post_update) > 1) {
            $result = wp_update_post($post_update, true);
            if (is_wp_error($result)) {
                return new WP_Error('update_failed', 'Failed to update attachment: ' . $result->get_error_message(), ['status' => 500]);
            }
        }

        // Update alt text (stored in post meta)
        if ($alt_text !== null) {
            update_post_meta($attachment_id, '_wp_attachment_image_alt', sanitize_text_field($alt_text));
            $updated_fields['alt_text'] = $alt_text;
        }

        // Get the updated attachment data
        $updated_attachment = get_post($attachment_id);
        $attachment_url = wp_get_attachment_url($attachment_id);

        return [
            'success' => true,
            'attachment_id' => intval($attachment_id),
            'updated_fields' => $updated_fields,
            'current_values' => [
                'title' => $updated_attachment->post_title,
                'caption' => $updated_attachment->post_excerpt,
                'description' => $updated_attachment->post_content,
                'alt_text' => get_post_meta($attachment_id, '_wp_attachment_image_alt', true),
                'url' => $attachment_url,
                'admin_url' => admin_url('post.php?post=' . $attachment_id . '&action=edit')
            ]
        ];
    }

    /**
     * Bulk update image metadata for multiple attachments
     *
     * POST /wp-json/instarank/v1/media/bulk-update-metadata
     * Body: {
     *   "items": [
     *     { "id": 123, "title": "...", "caption": "...", "description": "...", "alt_text": "..." },
     *     { "id": 456, "title": "...", "caption": "...", "description": "...", "alt_text": "..." }
     *   ]
     * }
     */
    public function bulk_update_image_metadata($request) {
        $items = $request->get_param('items');

        if (empty($items) || !is_array($items)) {
            return new WP_Error('invalid_items', 'Items array is required', ['status' => 400]);
        }

        $results = [];
        $success_count = 0;
        $failed_count = 0;

        foreach ($items as $item) {
            $attachment_id = isset($item['id']) ? intval($item['id']) : 0;

            if (empty($attachment_id)) {
                $results[] = [
                    'id' => $attachment_id,
                    'success' => false,
                    'error' => 'Invalid attachment ID'
                ];
                $failed_count++;
                continue;
            }

            // Verify attachment exists
            $attachment = get_post($attachment_id);
            if (!$attachment || $attachment->post_type !== 'attachment') {
                $results[] = [
                    'id' => $attachment_id,
                    'success' => false,
                    'error' => 'Attachment not found'
                ];
                $failed_count++;
                continue;
            }

            $updated_fields = [];

            // Update post fields
            $post_update = ['ID' => $attachment_id];

            if (isset($item['title'])) {
                $post_update['post_title'] = sanitize_text_field($item['title']);
                $updated_fields['title'] = $item['title'];
            }

            if (isset($item['caption'])) {
                $post_update['post_excerpt'] = sanitize_textarea_field($item['caption']);
                $updated_fields['caption'] = $item['caption'];
            }

            if (isset($item['description'])) {
                $post_update['post_content'] = wp_kses_post($item['description']);
                $updated_fields['description'] = $item['description'];
            }

            // Update post if any fields were provided
            if (count($post_update) > 1) {
                $result = wp_update_post($post_update, true);
                if (is_wp_error($result)) {
                    $results[] = [
                        'id' => $attachment_id,
                        'success' => false,
                        'error' => $result->get_error_message()
                    ];
                    $failed_count++;
                    continue;
                }
            }

            // Update alt text
            if (isset($item['alt_text'])) {
                update_post_meta($attachment_id, '_wp_attachment_image_alt', sanitize_text_field($item['alt_text']));
                $updated_fields['alt_text'] = $item['alt_text'];
            }

            $results[] = [
                'id' => $attachment_id,
                'success' => true,
                'updated_fields' => $updated_fields,
                'admin_url' => admin_url('post.php?post=' . $attachment_id . '&action=edit')
            ];
            $success_count++;
        }

        return [
            'success' => $success_count > 0,
            'total' => count($items),
            'succeeded' => $success_count,
            'failed' => $failed_count,
            'results' => $results
        ];
    }

    /**
     * Find attachment by URL
     *
     * POST /wp-json/instarank/v1/media/find-by-url
     * Body: { "url": "https://example.com/wp-content/uploads/2023/01/image.jpg" }
     */
    public function find_attachment_by_url($request) {
        $url = $request->get_param('url');

        if (empty($url)) {
            return new WP_Error('invalid_url', 'URL is required', ['status' => 400]);
        }

        // Try to find attachment by URL
        $attachment_id = attachment_url_to_postid($url);

        if ($attachment_id) {
            return [
                'success' => true,
                'attachment_id' => $attachment_id,
                'url' => $url
            ];
        }

        global $wpdb;
        $filename = basename(wp_parse_url($url, PHP_URL_PATH));

        // Try alternative: search by exact filename in guid
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $attachment_id = $wpdb->get_var($wpdb->prepare(
            "SELECT ID FROM {$wpdb->posts} WHERE post_type = 'attachment' AND guid LIKE %s LIMIT 1",
            '%' . $wpdb->esc_like($filename)
        ));

        if ($attachment_id) {
            return [
                'success' => true,
                'attachment_id' => intval($attachment_id),
                'url' => $url,
                'matched_by' => 'filename'
            ];
        }

        // Try to strip size suffix from filename (e.g., image-150x150.jpg -> image.jpg)
        $filename_base = preg_replace('/-\d+x\d+(\.[a-zA-Z]+)$/', '$1', $filename);
        if ($filename_base !== $filename) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $attachment_id = $wpdb->get_var($wpdb->prepare(
                "SELECT ID FROM {$wpdb->posts} WHERE post_type = 'attachment' AND guid LIKE %s LIMIT 1",
                '%' . $wpdb->esc_like($filename_base)
            ));

            if ($attachment_id) {
                return [
                    'success' => true,
                    'attachment_id' => intval($attachment_id),
                    'url' => $url,
                    'matched_by' => 'filename_base'
                ];
            }
        }

        // Try searching by post_title (filename without extension)
        $title_search = pathinfo($filename_base, PATHINFO_FILENAME);
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $attachment_id = $wpdb->get_var($wpdb->prepare(
            "SELECT ID FROM {$wpdb->posts} WHERE post_type = 'attachment' AND post_title = %s LIMIT 1",
            $title_search
        ));

        if ($attachment_id) {
            return [
                'success' => true,
                'attachment_id' => intval($attachment_id),
                'url' => $url,
                'matched_by' => 'post_title'
            ];
        }

        // Try searching in _wp_attached_file meta
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $attachment_id = $wpdb->get_var($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta}
             WHERE meta_key = '_wp_attached_file' AND meta_value LIKE %s LIMIT 1",
            '%' . $wpdb->esc_like($filename_base)
        ));

        if ($attachment_id) {
            return [
                'success' => true,
                'attachment_id' => intval($attachment_id),
                'url' => $url,
                'matched_by' => 'attached_file_meta'
            ];
        }

        return new WP_Error('not_found', 'Attachment not found for URL', ['status' => 404]);
    }

    /**
     * Update alt text by image URL
     *
     * POST /wp-json/instarank/v1/media/update-alt-by-url
     * Body: { "url": "https://example.com/wp-content/uploads/image.jpg", "alt_text": "New alt text" }
     */
    public function update_alt_text_by_url($request) {
        $url = $request->get_param('url');
        $alt_text = $request->get_param('alt_text');

        if (empty($url)) {
            return new WP_Error('invalid_url', 'URL is required', ['status' => 400]);
        }

        // Find attachment ID by URL using comprehensive search
        $attachment_id = $this->find_attachment_id_by_url($url);

        if (!$attachment_id) {
            return new WP_Error('not_found', 'Attachment not found for URL', ['status' => 404]);
        }

        // Verify it's an image
        if (!wp_attachment_is_image($attachment_id)) {
            return new WP_Error('invalid_attachment', 'URL does not point to an image attachment', ['status' => 400]);
        }

        // Update alt text
        update_post_meta($attachment_id, '_wp_attachment_image_alt', sanitize_text_field($alt_text));

        return [
            'success' => true,
            'attachment_id' => intval($attachment_id),
            'url' => $url,
            'alt_text' => $alt_text
        ];
    }

    /**
     * Helper function to find attachment ID by URL with multiple fallback strategies
     *
     * @param string $url The image URL
     * @return int|false Attachment ID or false if not found
     */
    private function find_attachment_id_by_url($url) {
        // Try standard WordPress function first
        $attachment_id = attachment_url_to_postid($url);
        if ($attachment_id) {
            return $attachment_id;
        }

        global $wpdb;
        $filename = basename(wp_parse_url($url, PHP_URL_PATH));

        // Try exact filename in guid
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $attachment_id = $wpdb->get_var($wpdb->prepare(
            "SELECT ID FROM {$wpdb->posts} WHERE post_type = 'attachment' AND guid LIKE %s LIMIT 1",
            '%' . $wpdb->esc_like($filename)
        ));
        if ($attachment_id) {
            return intval($attachment_id);
        }

        // Try to strip size suffix (e.g., image-150x150.jpg -> image.jpg)
        $filename_base = preg_replace('/-\d+x\d+(\.[a-zA-Z]+)$/', '$1', $filename);
        if ($filename_base !== $filename) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $attachment_id = $wpdb->get_var($wpdb->prepare(
                "SELECT ID FROM {$wpdb->posts} WHERE post_type = 'attachment' AND guid LIKE %s LIMIT 1",
                '%' . $wpdb->esc_like($filename_base)
            ));
            if ($attachment_id) {
                return intval($attachment_id);
            }
        }

        // Try searching by post_title
        $title_search = pathinfo($filename_base, PATHINFO_FILENAME);
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $attachment_id = $wpdb->get_var($wpdb->prepare(
            "SELECT ID FROM {$wpdb->posts} WHERE post_type = 'attachment' AND post_title = %s LIMIT 1",
            $title_search
        ));
        if ($attachment_id) {
            return intval($attachment_id);
        }

        // Try searching in _wp_attached_file meta
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $attachment_id = $wpdb->get_var($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta}
             WHERE meta_key = '_wp_attached_file' AND meta_value LIKE %s LIMIT 1",
            '%' . $wpdb->esc_like($filename_base)
        ));
        if ($attachment_id) {
            return intval($attachment_id);
        }

        return false;
    }

    /**
     * Analyze content for focus keyword (called from editor)
     *
     * POST /wp-json/instarank/v1/analyze-content
     * Body: {
     *   "content": "Post content...",
     *   "focus_keyword": "SEO tips",
     *   "title": "Post title",
     *   "description": "Meta description"
     * }
     *
     * NOTE: This is a lightweight local analysis. Platform can provide more advanced NLP analysis.
     */
    public function analyze_content($request) {
        $content = $request->get_param('content');
        $focus_keyword = $request->get_param('focus_keyword');
        $title = $request->get_param('title') ?: '';
        $description = $request->get_param('description') ?: '';

        if (empty($focus_keyword)) {
            return new WP_Error('missing_keyword', 'Focus keyword is required', ['status' => 400]);
        }

        $keyword_lower = strtolower($focus_keyword);
        $content_lower = strtolower(wp_strip_all_tags($content));
        $title_lower = strtolower($title);
        $desc_lower = strtolower($description);

        // Count keyword occurrences
        $content_count = substr_count($content_lower, $keyword_lower);
        $word_count = str_word_count($content_lower);
        $density = $word_count > 0 ? ($content_count / $word_count) * 100 : 0;

        // Check keyword placement
        $in_title = strpos($title_lower, $keyword_lower) !== false;
        $in_description = strpos($desc_lower, $keyword_lower) !== false;
        $in_first_paragraph = false;

        // Extract first paragraph
        $paragraphs = explode("\n\n", $content_lower);
        if (!empty($paragraphs[0])) {
            $in_first_paragraph = strpos($paragraphs[0], $keyword_lower) !== false;
        }

        // Calculate score (0-100)
        $score = 0;
        if ($in_title) $score += 30;
        if ($in_description) $score += 20;
        if ($in_first_paragraph) $score += 20;
        if ($density >= 0.5 && $density <= 2.5) $score += 20; // Optimal density
        if ($content_count >= 3) $score += 10; // At least 3 occurrences

        return [
            'success' => true,
            'focus_keyword' => $focus_keyword,
            'found_in_title' => $in_title,
            'found_in_description' => $in_description,
            'found_in_first_paragraph' => $in_first_paragraph,
            'occurrences' => $content_count,
            'word_count' => $word_count,
            'density' => round($density, 2),
            'score' => min($score, 100),
            'recommendation' => $this->get_keyword_recommendation($density, $content_count, $in_title)
        ];
    }

    /**
     * Get keyword recommendation based on analysis
     */
    private function get_keyword_recommendation($density, $count, $in_title) {
        if (!$in_title) {
            return 'Add your focus keyword to the title for better SEO.';
        }

        if ($density < 0.5) {
            return 'Use your focus keyword more frequently in the content (currently too low).';
        }

        if ($density > 2.5) {
            return 'Reduce keyword usage to avoid over-optimization (currently too high).';
        }

        if ($count < 3) {
            return 'Use your focus keyword at least 3 times in the content.';
        }

        return 'Great! Your keyword usage is optimal.';
    }

    /**
     * Wrap standalone Kadence buttons in advancedbtn containers
     *
     * WordPress Kadence buttons must be wrapped in an advancedbtn container.
     * This function finds standalone singlebtn blocks and wraps them properly.
     *
     * @param string $content The content with potential standalone buttons
     * @return string The content with buttons properly wrapped
     */
    private function wrap_standalone_kadence_buttons($content) {
        file_put_contents(__DIR__ . '/../instarank_debug.log', gmdate('Y-m-d H:i:s') . " - wrap_standalone_kadence_buttons called\n", FILE_APPEND);

        // First, fix character encoding issues
        // Fix curly quotes and en-dashes that break WordPress blocks
        // Using hex codes to avoid PHP parsing issues
        $content = str_replace(
            [
                "\xe2\x80\x9c", // left double quote
                "\xe2\x80\x9d", // right double quote
                "\xe2\x80\x98", // left single quote
                "\xe2\x80\x99", // right single quote
                "\xe2\x80\x93", // en-dash
                "\xe2\x80\x94", // em-dash
                "<!\xe2\x80\x93", // <!-- with en-dash
                "/\xe2\x80\x93>"  // /--> with en-dash
            ],
            ['"', '"', "'", "'", '--', '--', '<!--', '/-->'],
            $content
        );

        // Decode HTML entities in content BUT protect JSON inside block comments.
        // Using the safe decoder that preserves JSON attribute integrity.
        // Blindly calling html_entity_decode on all content would convert &quot; → "
        // inside JSON strings, corrupting the JSON structure.
        $content = $this->decode_entities_outside_block_json($content);

        if (!$content || strpos($content, 'wp:kadence/singlebtn') === false) {
            return $content;
        }

        // Pattern to match self-closing kadence/singlebtn blocks
        // These are buttons that end with /--> instead of having HTML content
        // Use [\s\S]*? to match any character including newlines (non-greedy)
        $pattern = '/(<!-- wp:kadence\/singlebtn\s+(\{[\s\S]*?\})\s*\/-->)/';

        // Find all self-closing button blocks
        preg_match_all($pattern, $content, $matches, PREG_SET_ORDER | PREG_OFFSET_CAPTURE);

        // Process matches in reverse order to maintain string positions
        for ($i = count($matches) - 1; $i >= 0; $i--) {
            $match = $matches[$i];
            $fullMatch = $match[0][0];
            $position = $match[0][1];
            $jsonAttrs = $match[2][0];

            // Check if this button is already wrapped by looking backwards
            $beforeButton = substr($content, max(0, $position - 200), min(200, $position));

            // Skip if already wrapped in advancedbtn
            if (strpos($beforeButton, 'wp:kadence/advancedbtn') !== false ||
                strpos($beforeButton, 'kb-buttons-wrap') !== false) {
                continue;
            }

            try {
                // Parse the JSON to get button attributes
                $attrs = json_decode($jsonAttrs, true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    continue;
                }

                $uniqueID = isset($attrs['uniqueID']) ? $attrs['uniqueID'] : 'btn-' . uniqid();

                // Generate wrapper ID
                $wrapperID = str_replace('_', '_wrap_', $uniqueID);

                // Add className to button attributes if not present
                // This is REQUIRED for Kadence to recognize the button as valid
                if (!isset($attrs['className'])) {
                    $attrs['className'] = 'wp-block-kadence-singlebtn';
                }

                // Re-encode the modified attributes
                $modifiedJsonAttrs = json_encode($attrs, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);

                // Keep button as self-closing but with updated attributes
                $buttonBlock = sprintf(
                    '<!-- wp:kadence/singlebtn %s /-->',
                    $modifiedJsonAttrs
                );

                // Wrap in advancedbtn container
                // IMPORTANT: Button block must be on separate line to prevent HTML encoding
                // The div is just a wrapper for rendering, the block comment structure is what WordPress reads
                $wrappedButton = sprintf(
                    '<!-- wp:kadence/advancedbtn {"uniqueID":"%s"} -->' . "\n" .
                    '<div class="wp-block-kadence-advancedbtn kb-buttons-wrap kb-btns%s">' . "\n" .
                    '%s' . "\n" .
                    '</div>' . "\n" .
                    '<!-- /wp:kadence/advancedbtn -->',
                    $wrapperID,
                    $wrapperID,  // Class name = kb-btns + uniqueID
                    $buttonBlock
                );

                // Replace the standalone button with the wrapped version
                $content = substr($content, 0, $position) . $wrappedButton . substr($content, $position + strlen($fullMatch));

            } catch (Exception $e) {
                // If JSON parsing fails, skip this button
                continue;
            }
        }

        return $content;
    }

    /**
     * Decode HTML entities in content BUT skip JSON attributes inside block comments.
     * Block comments look like: <!-- wp:block/type {"key":"value"} -->
     * Decoding entities inside the {...} JSON portion can corrupt JSON structure
     * (e.g., &quot; → " breaks JSON string boundaries).
     *
     * @param string $content The content to decode
     * @return string Content with entities decoded outside block JSON
     */
    private function decode_entities_outside_block_json($content) {
        $placeholders = [];
        $index = 0;

        // Protect JSON inside block comments by replacing with placeholders.
        // Uses a non-backtracking approach: match the block comment start,
        // then capture everything up to the closing --> delimiter.
        $protected = preg_replace_callback(
            '/(<!-- wp:[a-z0-9\/-]+ )(\{[^}]*(?:\}(?!\s*(?:\/-->|-->))[^}]*)*\})(\s*\/-->|\s*-->)/i',
            function($matches) use (&$placeholders, &$index) {
                $placeholder = "___BLOCK_JSON_{$index}___";
                $placeholders[$placeholder] = $matches[2];
                $index++;
                return $matches[1] . $placeholder . $matches[3];
            },
            $content
        );

        // If preg_replace_callback failed (PCRE backtracking limit), return original
        if ($protected === null) {
            return $content;
        }

        // Safely decode entities on everything (JSON is protected by placeholders)
        $decoded = html_entity_decode($protected, ENT_QUOTES | ENT_HTML5, 'UTF-8');

        // Restore protected JSON sections
        foreach ($placeholders as $placeholder => $json) {
            $decoded = str_replace($placeholder, $json, $decoded);
        }

        return $decoded;
    }

    /**
     * Replace literal \n, \r, \t with actual whitespace characters,
     * but NOT inside JSON attributes of block comments where \n is a valid JSON escape.
     *
     * @param string $content The content to process
     * @return string Content with escape sequences replaced outside block JSON
     */
    private function replace_escapes_outside_block_json($content) {
        $placeholders = [];
        $index = 0;

        // Protect JSON inside block comments using non-backtracking pattern
        $protected = preg_replace_callback(
            '/(<!-- wp:[a-z0-9\/-]+ )(\{[^}]*(?:\}(?!\s*(?:\/-->|-->))[^}]*)*\})(\s*\/-->|\s*-->)/i',
            function($matches) use (&$placeholders, &$index) {
                $placeholder = "___BLOCK_JSON_ESC_{$index}___";
                $placeholders[$placeholder] = $matches[2];
                $index++;
                return $matches[1] . $placeholder . $matches[3];
            },
            $content
        );

        // If preg_replace_callback failed (PCRE backtracking limit), return original
        if ($protected === null) {
            return $content;
        }

        // Safely replace escape sequences in non-JSON content
        $protected = str_replace('\\n', "\n", $protected);
        $protected = str_replace('\\r', "\r", $protected);
        $protected = str_replace('\\t', "\t", $protected);

        // Restore protected JSON sections
        foreach ($placeholders as $placeholder => $json) {
            $protected = str_replace($placeholder, $json, $protected);
        }

        return $protected;
    }

    /**
     * Validate Kadence block content for common corruption issues
     * This prevents corrupted blocks from being inserted into WordPress
     *
     * @param string $content The block content to validate
     * @return array Validation result with 'valid' boolean and 'errors' array
     */
    private function validate_kadence_blocks($content) {
        $errors = [];
        $warnings = [];

        // Only validate if this is Kadence block content
        if (strpos($content, 'wp:kadence/') === false) {
            return ['valid' => true, 'errors' => [], 'warnings' => []];
        }

        // Check for smart quotes in block comments (Unicode: " " ' ')
        // Smart quotes break JSON parsing in WordPress block editor
        if (preg_match('/<!--[^>]*[\x{201C}\x{201D}\x{2018}\x{2019}]/u', $content)) {
            $errors[] = 'Smart quotes detected in block comments - these will break block rendering';
        }

        // Check for malformed block comment delimiters
        // HTML-encoded comments or em-dashes instead of hyphens will break parsing
        if (preg_match('/&lt;!--|<!–|<!\s*–/', $content)) {
            $errors[] = 'Malformed block comment delimiters detected - blocks will not render';
        }

        // Check for HTML entities in block comments
        // Entities like &quot; will break JSON parsing
        if (preg_match('/<!--[^>]*&[a-z]+;[^>]*-->/i', $content)) {
            $warnings[] = 'HTML entities found in block comments - may cause rendering issues';
        }

        // Check for invalid JSON in block comments
        // Extract entire block comments first, then manually parse JSON using brace counting
        // This properly handles nested JSON objects which regex can't reliably match
        preg_match_all('/<!-- wp:kadence\/([a-z0-9-]+)\s+(.+?)\s*(\/)?-->/is', $content, $blocks, PREG_SET_ORDER);

        foreach ($blocks as $block) {
            $blockType = $block[1];
            $jsonCandidate = $block[2];

            // Find the opening brace
            $startPos = strpos($jsonCandidate, '{');
            if ($startPos === false) continue;

            // Count braces to find matching closing brace
            $depth = 0;
            $endPos = $startPos;
            $length = strlen($jsonCandidate);

            for ($i = $startPos; $i < $length; $i++) {
                if ($jsonCandidate[$i] === '{') $depth++;
                if ($jsonCandidate[$i] === '}') {
                    $depth--;
                    if ($depth === 0) {
                        $endPos = $i;
                        break;
                    }
                }
            }

            // Extract the balanced JSON string
            $jsonString = substr($jsonCandidate, $startPos, $endPos - $startPos + 1);

            // Attempt to parse JSON
            json_decode($jsonString);
            if (json_last_error() !== JSON_ERROR_NONE) {
                $errors[] = sprintf(
                    'Invalid JSON in kadence/%s block: %s',
                    $blockType,
                    substr($jsonString, 0, 50) . '...'
                );
            }
        }

        // Check for unclosed block tags using a single-pass approach.
        // Previous regexes used [^\/] and [^>] which fail when JSON attributes
        // contain / (URLs) or > (decoded from \u003e), causing undercounting.
        // Instead, match all block comments and classify each one.
        preg_match_all('/<!--\s*(\/?)wp:kadence\/[a-z0-9-]+([\s\S]*?)\s*-->/i', $content, $tagMatches, PREG_SET_ORDER);

        $openTags = 0;
        $closeTags = 0;
        $selfClosingTags = 0;

        foreach ($tagMatches as $tag) {
            $isClosing = ($tag[1] === '/');
            $isSelfClosing = (substr(rtrim($tag[0]), -4) === '/-->');

            if ($isClosing) {
                $closeTags++;
            } elseif ($isSelfClosing) {
                $selfClosingTags++;
            } else {
                $openTags++;
            }
        }

        // Each non-self-closing opening tag should have a matching closing tag
        if ($openTags !== $closeTags) {
            $warnings[] = sprintf(
                'Potentially unbalanced block tags: %d open, %d close, %d self-closing',
                $openTags,
                $closeTags,
                $selfClosingTags
            );
        }

        // Check for unreplaced placeholders
        // If placeholders like {{field}} remain, the template processing may have failed
        if (preg_match('/\{\{[a-zA-Z0-9_-]+\}\}/', $content)) {
            $warnings[] = 'Unreplaced placeholders detected in content - template processing may be incomplete';
        }

        return [
            'valid' => empty($errors),
            'errors' => $errors,
            'warnings' => $warnings
        ];
    }

    /**
     * Detect custom fields in a template
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function detect_template_fields($request) {
        $template_id = $request['id'];

        // Validate template exists
        $post = get_post($template_id);

        if (!$post) {
            return new WP_Error(
                'template_not_found',
                'Template not found',
                ['status' => 404]
            );
        }

        // Use field detector to find custom fields
        $detector = InstaRank_Field_Detector::instance();
        $result = $detector->detect_fields($template_id);

        if (!$result['success']) {
            return new WP_Error(
                'detection_failed',
                $result['error'],
                ['status' => 500]
            );
        }

        // Log the detection for debugging
        file_put_contents(
            __DIR__ . '/../instarank_debug.log',
            gmdate('Y-m-d H:i:s') . " - Detected fields for template '{$result['title']}' (ID: {$template_id}): " .
            $result['total_fields'] . " fields found\n",
            FILE_APPEND
        );

        return rest_ensure_response([
            'success' => true,
            'data' => $result
        ]);
    }

    /**
     * Get field mappings for a template
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function get_template_mappings($request) {
        $template_id = $request['id'];

        // Validate template exists
        $post = get_post($template_id);

        if (!$post) {
            return new WP_Error(
                'template_not_found',
                'Template not found',
                ['status' => 404]
            );
        }

        // Get mappings from wp_options
        $option_name = 'instarank_field_mappings_' . $template_id;
        $mappings = get_option($option_name);

        if (empty($mappings)) {
            return rest_ensure_response([
                'success' => true,
                'data' => [
                    'template_id' => $template_id,
                    'has_mappings' => false,
                    'mappings' => [],
                    'message' => 'No field mappings configured for this template'
                ]
            ]);
        }

        // Log the retrieval for debugging
        file_put_contents(
            __DIR__ . '/../instarank_debug.log',
            gmdate('Y-m-d H:i:s') . " - Retrieved field mappings for template ID {$template_id}: " .
            count($mappings['mappings']) . " fields mapped\n",
            FILE_APPEND
        );

        return rest_ensure_response([
            'success' => true,
            'data' => [
                'template_id' => $template_id,
                'has_mappings' => true,
                'mappings' => $mappings['mappings'],
                'last_updated' => $mappings['last_updated'] ?? null,
                'updated_by' => $mappings['updated_by'] ?? null
            ]
        ]);
    }

    /**
     * Validate spintax syntax
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function validate_spintax($request) {
        $params = $request->get_json_params();
        $text = $params['text'] ?? '';

        if (empty($text)) {
            return new WP_Error(
                'missing_text',
                'Text parameter is required',
                ['status' => 400]
            );
        }

        $engine = new InstaRank_Spintax_Engine();
        $result = $engine->validate($text);

        return rest_ensure_response([
            'success' => true,
            'data' => $result
        ]);
    }

    /**
     * Preview spintax variations
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function preview_spintax($request) {
        $params = $request->get_json_params();
        $text = $params['text'] ?? '';
        $count = isset($params['count']) ? min(intval($params['count']), 100) : 5;
        $variables = $params['variables'] ?? [];
        $seed = isset($params['seed']) ? intval($params['seed']) : null;

        if (empty($text)) {
            return new WP_Error(
                'missing_text',
                'Text parameter is required',
                ['status' => 400]
            );
        }

        $engine = new InstaRank_Spintax_Engine([
            'seed' => $seed
        ]);

        // Check if text contains spintax
        if (!$engine->has_spintax($text)) {
            return rest_ensure_response([
                'success' => true,
                'data' => [
                    'has_spintax' => false,
                    'variations' => [$text]
                ]
            ]);
        }

        // Generate variations
        $variations = $engine->spin_many($text, $count, $variables);

        // Also get validation stats
        $validation = $engine->validate($text);

        return rest_ensure_response([
            'success' => true,
            'data' => [
                'has_spintax' => true,
                'variations' => $variations,
                'stats' => $validation['stats'] ?? []
            ]
        ]);
    }

    /**
     * Process spintax and return single result
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function spin_spintax($request) {
        $params = $request->get_json_params();
        $text = $params['text'] ?? '';
        $variables = $params['variables'] ?? [];
        $seed = isset($params['seed']) ? intval($params['seed']) : null;

        if (empty($text)) {
            return new WP_Error(
                'missing_text',
                'Text parameter is required',
                ['status' => 400]
            );
        }

        $result = InstaRank_Spintax_Engine::process($text, $variables, $seed);

        return rest_ensure_response([
            'success' => true,
            'data' => [
                'original' => $text,
                'result' => $result
            ]
        ]);
    }

    /**
     * Get crawl data for all published pages
     * Returns full HTML content and SEO data for each page
     * This endpoint is used by the SaaS platform for WordPress-based crawling
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function get_crawl_data($request) {
        $detector = new InstaRank_SEO_Detector();

        // Pagination parameters
        $limit = intval($request->get_param('limit') ?? 50);
        $offset = intval($request->get_param('offset') ?? 0);
        $post_type_param = $request->get_param('post_type');
        $include_content = $request->get_param('include_content') !== 'false';

        // Adaptive memory-conscious batch size based on available PHP memory
        // Detect memory limit and adjust batch size accordingly
        $memory_limit = $this->get_memory_limit_bytes();

        if ($memory_limit <= 16 * 1024 * 1024) {
            // 16MB or less - very restrictive hosting (e.g., shared hosting)
            $max_batch = 3;
            $include_content = false; // Disable HTML content to save memory
        } elseif ($memory_limit <= 32 * 1024 * 1024) {
            // 32MB - basic shared hosting
            $max_batch = 5;
        } elseif ($memory_limit <= 64 * 1024 * 1024) {
            // 64MB - standard shared hosting
            $max_batch = 10;
        } elseif ($memory_limit <= 128 * 1024 * 1024) {
            // 128MB - typical WordPress hosting
            $max_batch = 15;
        } else {
            // 256MB+ - VPS/dedicated hosting
            $max_batch = 25;
        }

        $limit = min($limit, $max_batch);

        // Get post types
        if ($post_type_param) {
            $post_types = is_array($post_type_param) ? $post_type_param : explode(',', $post_type_param);
        } else {
            // Get all public post types except attachments
            $public_post_types = get_post_types(['public' => true], 'names');
            $post_types = array_values(array_filter($public_post_types, function($type) {
                return $type !== 'attachment';
            }));
        }

        // Query posts
        $args = [
            'post_type' => $post_types,
            'post_status' => 'publish',
            'posts_per_page' => $limit,
            'offset' => $offset,
            'orderby' => 'ID',
            'order' => 'ASC'
        ];

        $query = new WP_Query($args);
        $pages = [];

        foreach ($query->posts as $post) {
            $url = get_permalink($post->ID);
            $meta = $detector->get_post_meta($post->ID);

            // Get rendered HTML content
            $html = '';
            if ($include_content) {
                // Switch to the post context for proper rendering
                setup_postdata($post);

                // Get the rendered content (using WordPress core filter)
                // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Using WP core filter
                $content = apply_filters('the_content', $post->post_content);

                // Build full HTML structure
                $html = $this->build_page_html($post, $content, $meta);

                wp_reset_postdata();
            }

            // Extract internal links from rendered content (not raw post_content)
            $internal_links = $this->extract_links_from_content($content, home_url());

            // Get featured image
            $featured_image = null;
            $thumbnail_id = get_post_thumbnail_id($post->ID);
            if ($thumbnail_id) {
                $featured_image = [
                    'url' => wp_get_attachment_url($thumbnail_id),
                    'alt' => get_post_meta($thumbnail_id, '_wp_attachment_image_alt', true)
                ];
            }

            // Get images from rendered content (not raw post_content)
            $content_images = $this->extract_images_from_content($content);

            // Get schema/structured data
            $schema_data = $this->get_post_schema_data($post->ID);

            // Get categories and tags
            $categories = wp_get_post_categories($post->ID, ['fields' => 'names']);
            $tags = wp_get_post_tags($post->ID, ['fields' => 'names']);

            $pages[] = [
                'id' => $post->ID,
                'url' => $url,
                'title' => $post->post_title,
                'post_type' => $post->post_type,
                'status' => 'published',
                'status_code' => 200,
                'published_at' => $post->post_date_gmt,
                'modified_at' => $post->post_modified_gmt,
                'author' => get_the_author_meta('display_name', $post->post_author),
                'html' => $html,
                'html_size_bytes' => strlen($html),
                'seo' => [
                    'meta_title' => $meta['title'] ?? $post->post_title,
                    'meta_description' => $meta['description'] ?? '',
                    'focus_keyword' => $meta['focus_keyword'] ?? '',
                    'canonical_url' => $meta['canonical'] ?? $url,
                    'robots_meta' => $meta['robots'] ?? 'index, follow',
                    'og_title' => $meta['og_title'] ?? $meta['title'] ?? $post->post_title,
                    'og_description' => $meta['og_description'] ?? $meta['description'] ?? ''
                ],
                'headings' => $this->extract_headings($content),
                'word_count' => str_word_count(wp_strip_all_tags($content)),
                'links' => $internal_links,
                'images' => [
                    'featured' => $featured_image,
                    'content_images' => $content_images
                ],
                'categories' => $categories,
                'tags' => $tags,
                'schema' => $schema_data
            ];

            // Free memory after each post to prevent exhaustion on large sites
            unset($html, $content, $meta, $internal_links, $content_images, $schema_data);
        }

        // Get total count for pagination
        $count_args = [
            'post_type' => $post_types,
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'fields' => 'ids'
        ];
        $total_query = new WP_Query($count_args);
        $total_pages = $total_query->found_posts;

        // Free total query memory
        unset($total_query);

        return rest_ensure_response([
            'success' => true,
            'pages' => $pages,
            'pagination' => [
                'total' => $total_pages,
                'limit' => $limit,
                'offset' => $offset,
                'has_more' => ($offset + count($pages)) < $total_pages
            ],
            'site' => [
                'url' => home_url(),
                'name' => get_bloginfo('name'),
                'wordpress_version' => get_bloginfo('version'),
                'plugin_version' => INSTARANK_VERSION,
                'seo_plugin' => $detector->get_active_seo_plugin(),
                'memory_limit_mb' => round($memory_limit / (1024 * 1024)),
                'content_included' => $include_content
            ]
        ]);
    }

    /**
     * Get crawl data for a single page
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function get_page_crawl_data($request) {
        $detector = new InstaRank_SEO_Detector();
        $post_id = $request->get_param('id');
        $url = $request->get_param('url');

        if (empty($post_id) && empty($url)) {
            return new WP_Error('invalid_request', 'Either id or url parameter is required', ['status' => 400]);
        }

        // Resolve post by URL if needed
        if (!$post_id && $url) {
            $post_id = url_to_postid($url);
            if (!$post_id) {
                // Try to find by exact URL match
                global $wpdb;
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $post_id = $wpdb->get_var($wpdb->prepare(
                    "SELECT ID FROM $wpdb->posts WHERE post_status = 'publish' AND guid = %s",
                    $url
                ));
            }
        }

        if (!$post_id) {
            return new WP_Error('not_found', 'Page not found', ['status' => 404]);
        }

        $post = get_post($post_id);
        if (!$post || $post->post_status !== 'publish') {
            return new WP_Error('not_found', 'Page not found or not published', ['status' => 404]);
        }

        $meta = $detector->get_post_meta($post->ID);
        $page_url = get_permalink($post->ID);

        // Build full HTML (using WordPress core filter)
        setup_postdata($post);
        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Using WP core filter
        $content = apply_filters('the_content', $post->post_content);
        $html = $this->build_page_html($post, $content, $meta);
        wp_reset_postdata();

        // Extract all data from rendered content (not raw post_content)
        $internal_links = $this->extract_links_from_content($content, home_url());
        $content_images = $this->extract_images_from_content($content);
        $schema_data = $this->get_post_schema_data($post->ID);

        $featured_image = null;
        $thumbnail_id = get_post_thumbnail_id($post->ID);
        if ($thumbnail_id) {
            $featured_image = [
                'url' => wp_get_attachment_url($thumbnail_id),
                'alt' => get_post_meta($thumbnail_id, '_wp_attachment_image_alt', true)
            ];
        }

        return rest_ensure_response([
            'success' => true,
            'page' => [
                'id' => $post->ID,
                'url' => $page_url,
                'title' => $post->post_title,
                'post_type' => $post->post_type,
                'status' => 'published',
                'status_code' => 200,
                'published_at' => $post->post_date_gmt,
                'modified_at' => $post->post_modified_gmt,
                'author' => get_the_author_meta('display_name', $post->post_author),
                'html' => $html,
                'html_size_bytes' => strlen($html),
                'seo' => [
                    'meta_title' => $meta['title'] ?? $post->post_title,
                    'meta_description' => $meta['description'] ?? '',
                    'focus_keyword' => $meta['focus_keyword'] ?? '',
                    'canonical_url' => $meta['canonical'] ?? $page_url,
                    'robots_meta' => $meta['robots'] ?? 'index, follow',
                    'og_title' => $meta['og_title'] ?? $meta['title'] ?? $post->post_title,
                    'og_description' => $meta['og_description'] ?? $meta['description'] ?? ''
                ],
                'headings' => $this->extract_headings($content),
                'word_count' => str_word_count(wp_strip_all_tags($content)),
                'links' => $internal_links,
                'images' => [
                    'featured' => $featured_image,
                    'content_images' => $content_images
                ],
                'categories' => wp_get_post_categories($post->ID, ['fields' => 'names']),
                'tags' => wp_get_post_tags($post->ID, ['fields' => 'names']),
                'schema' => $schema_data
            ]
        ]);
    }

    /**
     * Build full HTML page from post data
     */
    private function build_page_html($post, $rendered_content, $meta) {
        $title = $meta['title'] ?? $post->post_title;
        $description = $meta['description'] ?? '';
        $canonical = $meta['canonical'] ?? get_permalink($post->ID);
        $robots = $meta['robots'] ?? 'index, follow';
        $og_title = $meta['og_title'] ?? $title;
        $og_description = $meta['og_description'] ?? $description;
        $og_image = '';

        $thumbnail_id = get_post_thumbnail_id($post->ID);
        if ($thumbnail_id) {
            $og_image = wp_get_attachment_url($thumbnail_id);
        }

        $site_name = get_bloginfo('name');
        $language = get_bloginfo('language');

        // Build HTML document
        $html = '<!DOCTYPE html>';
        $html .= '<html lang="' . esc_attr($language) . '">';
        $html .= '<head>';
        $html .= '<meta charset="UTF-8">';
        $html .= '<meta name="viewport" content="width=device-width, initial-scale=1.0">';
        $html .= '<title>' . esc_html($title) . '</title>';

        if ($description) {
            $html .= '<meta name="description" content="' . esc_attr($description) . '">';
        }

        $html .= '<meta name="robots" content="' . esc_attr($robots) . '">';
        $html .= '<link rel="canonical" href="' . esc_url($canonical) . '">';

        // Open Graph
        $html .= '<meta property="og:type" content="article">';
        $html .= '<meta property="og:title" content="' . esc_attr($og_title) . '">';
        if ($og_description) {
            $html .= '<meta property="og:description" content="' . esc_attr($og_description) . '">';
        }
        $html .= '<meta property="og:url" content="' . esc_url(get_permalink($post->ID)) . '">';
        $html .= '<meta property="og:site_name" content="' . esc_attr($site_name) . '">';
        if ($og_image) {
            $html .= '<meta property="og:image" content="' . esc_url($og_image) . '">';
        }

        // Twitter Card
        $html .= '<meta name="twitter:card" content="summary_large_image">';
        $html .= '<meta name="twitter:title" content="' . esc_attr($og_title) . '">';
        if ($og_description) {
            $html .= '<meta name="twitter:description" content="' . esc_attr($og_description) . '">';
        }
        if ($og_image) {
            $html .= '<meta name="twitter:image" content="' . esc_url($og_image) . '">';
        }

        // Add JSON-LD structured data scripts
        $html .= $this->get_json_ld_scripts($post->ID);

        $html .= '</head>';
        $html .= '<body>';
        $html .= '<article>';
        $html .= '<h1>' . esc_html($post->post_title) . '</h1>';
        $html .= '<div class="entry-content">' . $rendered_content . '</div>';
        $html .= '</article>';
        $html .= '</body>';
        $html .= '</html>';

        return $html;
    }

    /**
     * Get PHP memory limit in bytes
     * Parses the ini value and converts shorthand notation (e.g., 128M) to bytes
     *
     * @return int Memory limit in bytes (-1 if unlimited)
     */
    private function get_memory_limit_bytes() {
        $memory_limit = ini_get('memory_limit');

        if ($memory_limit === '-1') {
            return PHP_INT_MAX; // Unlimited
        }

        $memory_limit = trim($memory_limit);
        $last = strtolower($memory_limit[strlen($memory_limit) - 1]);
        $memory_limit = (int) $memory_limit;

        switch ($last) {
            case 'g':
                $memory_limit *= 1024;
                // fall through
            case 'm':
                $memory_limit *= 1024;
                // fall through
            case 'k':
                $memory_limit *= 1024;
        }

        return $memory_limit;
    }

    /**
     * Extract headings from content
     */
    private function extract_headings($content) {
        $headings = [
            'h1' => [],
            'h2' => [],
            'h3' => [],
            'h4' => [],
            'h5' => [],
            'h6' => []
        ];

        // Match all headings
        for ($level = 1; $level <= 6; $level++) {
            if (preg_match_all('/<h' . $level . '[^>]*>(.*?)<\/h' . $level . '>/si', $content, $matches)) {
                $headings['h' . $level] = array_map('wp_strip_all_tags', $matches[1]);
            }
        }

        return $headings;
    }

    /**
     * Extract internal links from content
     */
    private function extract_links_from_content($content, $site_url) {
        $links = [];
        $site_host = wp_parse_url($site_url, PHP_URL_HOST);

        // Match all anchor tags
        if (preg_match_all('/<a[^>]+href=["\']([^"\']+)["\'][^>]*>(.*?)<\/a>/si', $content, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $match) {
                $href = $match[1];
                $anchor_text = wp_strip_all_tags($match[2]);

                // Skip empty hrefs, anchors, and javascript
                if (empty($href) || strpos($href, '#') === 0 || strpos($href, 'javascript:') === 0 || strpos($href, 'mailto:') === 0) {
                    continue;
                }

                // Make relative URLs absolute
                if (strpos($href, '/') === 0) {
                    $href = $site_url . $href;
                }

                // Check if internal link
                $link_host = wp_parse_url($href, PHP_URL_HOST);
                $is_internal = empty($link_host) || $link_host === $site_host;

                $links[] = [
                    'url' => $href,
                    'anchor_text' => $anchor_text,
                    'is_internal' => $is_internal
                ];
            }
        }

        return $links;
    }

    /**
     * Extract images from content
     */
    private function extract_images_from_content($content) {
        $images = [];

        // Match all img tags
        if (preg_match_all('/<img[^>]+>/si', $content, $matches)) {
            foreach ($matches[0] as $img_tag) {
                $src = '';
                $alt = '';
                $width = null;
                $height = null;

                // Extract src
                if (preg_match('/src=["\']([^"\']+)["\']/i', $img_tag, $src_match)) {
                    $src = $src_match[1];
                }

                // Extract alt
                if (preg_match('/alt=["\']([^"\']*)["\']?/i', $img_tag, $alt_match)) {
                    $alt = $alt_match[1];
                }

                // Extract dimensions
                if (preg_match('/width=["\']?(\d+)/i', $img_tag, $width_match)) {
                    $width = intval($width_match[1]);
                }
                if (preg_match('/height=["\']?(\d+)/i', $img_tag, $height_match)) {
                    $height = intval($height_match[1]);
                }

                if ($src) {
                    $images[] = [
                        'src' => $src,
                        'alt' => $alt,
                        'has_alt' => !empty($alt),
                        'width' => $width,
                        'height' => $height
                    ];
                }
            }
        }

        return $images;
    }

    /**
     * Get schema/structured data for a post
     * Captures actual JSON-LD output from SEO plugins (Yoast, RankMath, AIOSEO)
     */
    private function get_post_schema_data($post_id) {
        $schema = [
            'json_ld' => [],
            'source' => null,
            'raw_scripts' => []
        ];

        // Method 1: Capture actual wp_head output to get JSON-LD scripts
        // This captures what Yoast/RankMath/AIOSEO actually output
        $post = get_post($post_id);
        if ($post) {
            // Set up the post for wp_head to work correctly
            global $wp_query;
            $original_post = isset($GLOBALS['post']) ? $GLOBALS['post'] : null;
            $original_query = $wp_query;

            // Create a mock query for single post
            $GLOBALS['post'] = $post;
            setup_postdata($post);

            // Capture wp_head output
            ob_start();
            wp_head();
            $head_output = ob_get_clean();

            // Restore original state
            if ($original_post) {
                $GLOBALS['post'] = $original_post;
                setup_postdata($original_post);
            }
            wp_reset_postdata();

            // Extract JSON-LD scripts from head output
            if (preg_match_all('/<script[^>]+type=["\']?application\/ld\+json["\']?[^>]*>([\s\S]*?)<\/script>/i', $head_output, $matches)) {
                foreach ($matches[1] as $index => $json_content) {
                    $json_content = trim($json_content);
                    if (!empty($json_content)) {
                        $decoded = json_decode($json_content, true);
                        if ($decoded !== null) {
                            $schema['json_ld'][] = $decoded;
                            $schema['raw_scripts'][] = $json_content;

                            // Detect source from JSON-LD content
                            if (!$schema['source']) {
                                if (strpos($json_content, 'yoast') !== false || strpos($json_content, 'Yoast') !== false) {
                                    $schema['source'] = 'yoast';
                                } elseif (strpos($json_content, 'rankmath') !== false || strpos($json_content, 'RankMath') !== false) {
                                    $schema['source'] = 'rankmath';
                                } elseif (strpos($json_content, 'aioseo') !== false || strpos($json_content, 'AIOSEO') !== false) {
                                    $schema['source'] = 'aioseo';
                                }
                            }
                        }
                    }
                }
            }
        }

        // Method 2: Also check post meta as fallback
        // Yoast SEO schema meta
        $yoast_schema = get_post_meta($post_id, '_yoast_wpseo_schema_graph', true);
        if ($yoast_schema && empty($schema['json_ld'])) {
            $schema['source'] = 'yoast';
            $schema['meta_data'] = $yoast_schema;
        }

        // RankMath schema meta
        $rankmath_schema = get_post_meta($post_id, 'rank_math_schema_Article', true);
        if ($rankmath_schema && empty($schema['json_ld'])) {
            $schema['source'] = 'rankmath';
            $schema['meta_data'] = $rankmath_schema;
        }

        // AIOSEO schema meta
        $aioseo_schema = get_post_meta($post_id, '_aioseo_schema_graph', true);
        if ($aioseo_schema && empty($schema['json_ld'])) {
            $schema['source'] = 'aioseo';
            $schema['meta_data'] = $aioseo_schema;
        }

        // Custom InstaRank schema
        $instarank_schema = get_post_meta($post_id, '_instarank_schema', true);
        if ($instarank_schema) {
            $schema['instarank'] = $instarank_schema;
        }

        // Add count for easy checking
        $schema['count'] = count($schema['json_ld']);

        return $schema;
    }

    /**
     * Get JSON-LD scripts as HTML string for inclusion in page HTML
     */
    private function get_json_ld_scripts($post_id) {
        $schema_data = $this->get_post_schema_data($post_id);
        $scripts = '';

        if (!empty($schema_data['raw_scripts'])) {
            foreach ($schema_data['raw_scripts'] as $json_content) {
                $scripts .= '<script type="application/ld+json">' . $json_content . '</script>';
            }
        }

        return $scripts;
    }

    /**
     * Transform value for ACF field based on field type
     *
     * @param mixed $value The value to transform
     * @param string $acf_type The ACF field type
     * @return mixed Transformed value
     */
    private function transform_acf_value($value, $acf_type) {
        switch ($acf_type) {
            case 'true_false':
                // ACF true_false expects 1 or 0
                return ($value === 'true' || $value === '1' || $value === true || $value === 1) ? 1 : 0;

            case 'checkbox':
                // ACF checkbox expects array
                if (!is_array($value)) {
                    return array_filter(array_map('trim', explode(',', (string) $value)));
                }
                return $value;

            case 'date_picker':
                // ACF date_picker expects YYYYMMDD format
                if (!empty($value)) {
                    $timestamp = strtotime($value);
                    if ($timestamp !== false) {
                        return gmdate('Ymd', $timestamp);
                    }
                }
                return $value;

            case 'date_time_picker':
                // ACF date_time_picker expects Y-m-d H:i:s format
                if (!empty($value)) {
                    $timestamp = strtotime($value);
                    if ($timestamp !== false) {
                        return gmdate('Y-m-d H:i:s', $timestamp);
                    }
                }
                return $value;

            case 'time_picker':
                // ACF time_picker expects H:i:s format
                if (!empty($value)) {
                    $timestamp = strtotime($value);
                    if ($timestamp !== false) {
                        return gmdate('H:i:s', $timestamp);
                    }
                }
                return $value;

            case 'number':
            case 'range':
                // Ensure numeric value
                return is_numeric($value) ? (float) $value : $value;

            case 'post_object':
            case 'relationship':
            case 'user':
                // These expect ID(s)
                if (is_numeric($value)) {
                    return (int) $value;
                }
                // If comma-separated IDs, convert to array of integers
                if (is_string($value) && strpos($value, ',') !== false) {
                    return array_map('intval', array_filter(explode(',', $value)));
                }
                return $value;

            case 'taxonomy':
                // Taxonomy expects term ID(s) or term slug(s)
                if (is_string($value) && strpos($value, ',') !== false) {
                    return array_map('trim', explode(',', $value));
                }
                return $value;

            case 'select':
            case 'radio':
            case 'button_group':
                // Ensure string value for select fields
                return (string) $value;

            case 'gallery':
                // Gallery expects array of attachment IDs
                if (!is_array($value)) {
                    return array_filter(array_map('intval', explode(',', (string) $value)));
                }
                return array_map('intval', $value);

            default:
                // Return as-is for text, textarea, wysiwyg, url, email, image, file, color_picker
                return $value;
        }
    }

    /**
     * ACF ENDPOINTS
     */

    /**
     * Get ACF fields for a post type or all field groups
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response
     */
    public function get_acf_fields($request) {
        $acf_detector = InstaRank_ACF_Detector::instance();

        // Get optional post_type filter
        $post_type = $request->get_param('post_type');

        // Get ACF detection result
        $result = $acf_detector->detect($post_type);

        return rest_ensure_response($result);
    }

    /**
     * Get ACF status and info
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response
     */
    public function get_acf_status($request) {
        $acf_detector = InstaRank_ACF_Detector::instance();

        $info = $acf_detector->get_acf_info();

        // Add additional context
        $response = [
            'success' => true,
            'acf_active' => $info['active'],
            'acf_version' => $info['version'],
            'acf_pro' => $info['pro'],
            'site_url' => get_site_url(),
            'timestamp' => current_time('c')
        ];

        // If ACF is active, add summary stats
        if ($info['active']) {
            $all_fields = $acf_detector->get_all_fields();
            $field_groups = $acf_detector->get_all_field_groups();

            $supported_count = 0;
            $unsupported_count = 0;

            foreach ($all_fields as $field) {
                if ($field['is_supported']) {
                    $supported_count++;
                } else {
                    $unsupported_count++;
                }
            }

            $response['stats'] = [
                'total_groups' => count($field_groups),
                'total_fields' => count($all_fields),
                'supported_fields' => $supported_count,
                'unsupported_fields' => $unsupported_count
            ];
        }

        return rest_ensure_response($response);
    }

    // ========================================
    // MULTI-LANGUAGE METHODS
    // ========================================

    /**
     * Get multi-language plugin info
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response|WP_Error Response object.
     */
    public function get_multilang_info($request) {
        $multilang = InstaRank_MultiLang::instance();

        return rest_ensure_response([
            'success' => true,
            'data' => $multilang->get_info(),
        ]);
    }

    /**
     * Set language for a post
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response|WP_Error Response object.
     */
    public function set_post_language($request) {
        $params = $request->get_json_params();

        if (empty($params['post_id']) || empty($params['lang'])) {
            return new WP_Error(
                'missing_params',
                __('post_id and lang are required', 'instarank'),
                ['status' => 400]
            );
        }

        $post_id = intval($params['post_id']);
        $lang = sanitize_text_field($params['lang']);

        // Verify post exists
        if (!get_post($post_id)) {
            return new WP_Error(
                'post_not_found',
                __('Post not found', 'instarank'),
                ['status' => 404]
            );
        }

        $multilang = InstaRank_MultiLang::instance();

        if (!$multilang->is_active()) {
            return new WP_Error(
                'no_multilang_plugin',
                __('No multi-language plugin is active (WPML or Polylang required)', 'instarank'),
                ['status' => 400]
            );
        }

        $success = $multilang->set_post_language($post_id, $lang);

        if (!$success) {
            return new WP_Error(
                'set_language_failed',
                __('Failed to set post language', 'instarank'),
                ['status' => 500]
            );
        }

        return rest_ensure_response([
            'success' => true,
            /* translators: 1: language code, 2: post ID */
            'message' => sprintf(__('Language set to %1$s for post %2$d', 'instarank'), $lang, $post_id),
            'data' => [
                'post_id' => $post_id,
                'lang' => $lang,
            ],
        ]);
    }

    /**
     * Link posts as translations
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response|WP_Error Response object.
     */
    public function link_translations($request) {
        $params = $request->get_json_params();

        if (empty($params['translations']) || !is_array($params['translations'])) {
            return new WP_Error(
                'missing_params',
                __('translations array is required (format: {lang: post_id})', 'instarank'),
                ['status' => 400]
            );
        }

        $translations = [];
        foreach ($params['translations'] as $lang => $post_id) {
            $lang = sanitize_text_field($lang);
            $post_id = intval($post_id);

            if (!get_post($post_id)) {
                return new WP_Error(
                    'post_not_found',
                    /* translators: %d: post ID */
                    sprintf(__('Post not found: %d', 'instarank'), $post_id),
                    ['status' => 404]
                );
            }

            $translations[$lang] = $post_id;
        }

        if (count($translations) < 2) {
            return new WP_Error(
                'insufficient_translations',
                __('At least 2 translations are required to link', 'instarank'),
                ['status' => 400]
            );
        }

        $multilang = InstaRank_MultiLang::instance();

        if (!$multilang->is_active()) {
            return new WP_Error(
                'no_multilang_plugin',
                __('No multi-language plugin is active (WPML or Polylang required)', 'instarank'),
                ['status' => 400]
            );
        }

        $success = $multilang->link_translations($translations);

        if (!$success) {
            return new WP_Error(
                'link_translations_failed',
                __('Failed to link translations', 'instarank'),
                ['status' => 500]
            );
        }

        return rest_ensure_response([
            'success' => true,
            /* translators: %d: number of translations */
            'message' => sprintf(__('Linked %d translations', 'instarank'), count($translations)),
            'data' => [
                'translations' => $translations,
            ],
        ]);
    }

    /**
     * Render a dynamic element
     *
     * This endpoint is called from InstaRank SaaS to fetch dynamic content
     * that may require WordPress-specific data (e.g., Media Library images,
     * custom field values, etc.)
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function render_dynamic_element($request) {
        $params = $request->get_json_params();

        $shortcode = $params['shortcode'] ?? null;
        $field_data = $params['fieldData'] ?? [];

        if (empty($shortcode)) {
            return new WP_Error(
                'missing_shortcode',
                'Shortcode is required',
                ['status' => 400]
            );
        }

        // Log the request for debugging
        file_put_contents(
            __DIR__ . '/../instarank_debug.log',
            gmdate('Y-m-d H:i:s') . " - Dynamic Element Render Request: shortcode={$shortcode}\n",
            FILE_APPEND
        );

        // For now, return a simple response
        // The actual rendering happens in the SaaS application
        // This endpoint is here for future WordPress-specific rendering needs
        // (e.g., fetching Media Library images, custom field values, etc.)

        return rest_ensure_response([
            'success' => true,
            'rendered_html' => '<!-- Dynamic element placeholder: ' . esc_html($shortcode) . ' -->',
            'message' => 'Dynamic element rendering is handled by InstaRank SaaS. This endpoint is reserved for WordPress-specific data fetching in future updates.'
        ]);
    }

    /**
     * PAGE BUILDER CSS REGENERATION METHODS
     */

    /**
     * Regenerate CSS for a single post
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function regenerate_css($request) {
        $params = $request->get_json_params();
        $post_id = isset($params['post_id']) ? intval($params['post_id']) : 0;
        $builder = isset($params['builder']) ? sanitize_text_field($params['builder']) : '';
        $force = isset($params['force']) ? (bool) $params['force'] : true;

        if (!$post_id) {
            return new WP_Error('missing_post_id', 'post_id is required', ['status' => 400]);
        }

        $post = get_post($post_id);
        if (!$post) {
            return new WP_Error('post_not_found', 'Post not found', ['status' => 404]);
        }

        // Auto-detect builder if not provided
        if (empty($builder)) {
            $builder = $this->detect_page_builder($post_id);
        }

        $result = $this->regenerate_builder_css($post_id, $builder, $force);

        return rest_ensure_response([
            'success' => $result['success'],
            'message' => $result['message'],
            'builder' => $builder,
            'post_id' => $post_id,
            'cache_cleared' => $result['cache_cleared'] ?? false,
            'css_regenerated' => $result['css_regenerated'] ?? false,
        ]);
    }

    /**
     * Detect page builder for a post
     *
     * @param int $post_id
     * @return string
     */
    private function detect_page_builder($post_id) {
        // Check Elementor
        if (get_post_meta($post_id, '_elementor_edit_mode', true) === 'builder') {
            return 'elementor';
        }

        // Check Bricks
        if (get_post_meta($post_id, '_bricks_page_content_2', true)) {
            return 'bricks';
        }

        // Check Kadence
        if (get_post_meta($post_id, '_kad_blocks_custom_css', true) || get_post_meta($post_id, '_kad_blocks_head_css', true)) {
            return 'kadence';
        }

        // Check Oxygen
        if (get_post_meta($post_id, 'ct_builder_json', true)) {
            return 'oxygen';
        }

        // Check Beaver Builder
        if (get_post_meta($post_id, '_fl_builder_enabled', true)) {
            return 'beaver-builder';
        }

        // Check Divi
        if (get_post_meta($post_id, '_et_pb_use_builder', true) === 'on') {
            return 'divi';
        }

        // Check for block editor (Gutenberg)
        $post = get_post($post_id);
        if ($post && has_blocks($post->post_content)) {
            return 'gutenberg';
        }

        return 'unknown';
    }

    /**
     * Regenerate CSS for a specific builder
     *
     * @param int $post_id
     * @param string $builder
     * @param bool $force
     * @return array
     */
    private function regenerate_builder_css($post_id, $builder, $force = true) {
        $result = ['success' => false, 'message' => '', 'cache_cleared' => false, 'css_regenerated' => false];

        switch ($builder) {
            case 'elementor':
                if (class_exists('Elementor\Plugin')) {
                    try {
                        // Clear Elementor cache for this post
                        \Elementor\Plugin::$instance->files_manager->clear_cache();

                        // Regenerate CSS files
                        $css_file = new \Elementor\Core\Files\CSS\Post($post_id);
                        $css_file->update();

                        $result['success'] = true;
                        $result['message'] = 'Elementor CSS regenerated successfully';
                        $result['cache_cleared'] = true;
                        $result['css_regenerated'] = true;
                    } catch (Exception $e) {
                        $result['message'] = 'Elementor CSS regeneration failed: ' . $e->getMessage();
                    }
                } else {
                    $result['message'] = 'Elementor plugin not active';
                }
                break;

            case 'bricks':
                if (class_exists('Bricks\Assets')) {
                    try {
                        // Clear Bricks cache
                        \Bricks\Assets::generate_css_file($post_id);

                        $result['success'] = true;
                        $result['message'] = 'Bricks CSS regenerated successfully';
                        $result['css_regenerated'] = true;
                    } catch (Exception $e) {
                        $result['message'] = 'Bricks CSS regeneration failed: ' . $e->getMessage();
                    }
                } else {
                    $result['message'] = 'Bricks plugin not active';
                }
                break;

            case 'kadence':
                if (class_exists('Kadence_Blocks_Frontend')) {
                    try {
                        // Clear Kadence cache
                        delete_post_meta($post_id, '_kad_blocks_custom_css');
                        delete_post_meta($post_id, '_kad_blocks_head_css');

                        // Trigger regeneration on next page load - using Kadence Blocks hook
                        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
                        do_action('kadence_blocks_force_render_inline_css_in_content', true);

                        $result['success'] = true;
                        $result['message'] = 'Kadence CSS cache cleared - will regenerate on next page load';
                        $result['cache_cleared'] = true;
                    } catch (Exception $e) {
                        $result['message'] = 'Kadence CSS cache clear failed: ' . $e->getMessage();
                    }
                } else {
                    $result['message'] = 'Kadence Blocks plugin not active';
                }
                break;

            case 'oxygen':
                if (defined('CT_VERSION')) {
                    try {
                        // Clear Oxygen cache
                        delete_option('oxygen_vsb_universal_css_cache');
                        delete_post_meta($post_id, 'ct_builder_shortcodes');

                        $result['success'] = true;
                        $result['message'] = 'Oxygen CSS cache cleared';
                        $result['cache_cleared'] = true;
                    } catch (Exception $e) {
                        $result['message'] = 'Oxygen CSS cache clear failed: ' . $e->getMessage();
                    }
                } else {
                    $result['message'] = 'Oxygen plugin not active';
                }
                break;

            case 'beaver-builder':
                if (class_exists('FLBuilder')) {
                    try {
                        // Clear Beaver Builder cache
                        \FLBuilderModel::delete_asset_cache_for_post($post_id);

                        $result['success'] = true;
                        $result['message'] = 'Beaver Builder CSS cache cleared';
                        $result['cache_cleared'] = true;
                    } catch (Exception $e) {
                        $result['message'] = 'Beaver Builder CSS cache clear failed: ' . $e->getMessage();
                    }
                } else {
                    $result['message'] = 'Beaver Builder plugin not active';
                }
                break;

            case 'divi':
                if (function_exists('et_core_clear_wp_cache')) {
                    try {
                        // Clear Divi cache
                        et_core_clear_wp_cache($post_id);
                        delete_post_meta($post_id, '_et_pb_static_css_file');

                        $result['success'] = true;
                        $result['message'] = 'Divi CSS cache cleared';
                        $result['cache_cleared'] = true;
                    } catch (Exception $e) {
                        $result['message'] = 'Divi CSS cache clear failed: ' . $e->getMessage();
                    }
                } else {
                    $result['message'] = 'Divi theme/plugin not active';
                }
                break;

            case 'gutenberg':
                // Gutenberg doesn't need CSS regeneration
                $result['success'] = true;
                $result['message'] = 'Gutenberg (Block Editor) - no CSS regeneration needed';
                break;

            default:
                $result['message'] = 'Unknown or unsupported page builder: ' . $builder;
        }

        return $result;
    }

    /**
     * Clear all builder caches
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function clear_builder_caches($request) {
        $params = $request->get_json_params();
        $builders = isset($params['builders']) ? $params['builders'] : ['elementor', 'bricks', 'kadence', 'oxygen', 'beaver-builder', 'divi'];

        $cleared = [];

        foreach ($builders as $builder) {
            $builder = sanitize_text_field($builder);

            switch ($builder) {
                case 'elementor':
                    if (class_exists('Elementor\Plugin')) {
                        \Elementor\Plugin::$instance->files_manager->clear_cache();
                        $cleared[] = 'elementor';
                    }
                    break;

                case 'bricks':
                    if (class_exists('Bricks\Assets')) {
                        \Bricks\Assets::clear_cache();
                        $cleared[] = 'bricks';
                    }
                    break;

                case 'kadence':
                    if (class_exists('Kadence_Blocks_Frontend')) {
                        global $wpdb;
                        // Delete Kadence Blocks CSS cache - direct query needed as no API exists
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->postmeta} WHERE meta_key IN (%s, %s)", '_kad_blocks_custom_css', '_kad_blocks_head_css'));
                        $cleared[] = 'kadence';
                    }
                    break;

                case 'oxygen':
                    if (defined('CT_VERSION')) {
                        delete_option('oxygen_vsb_universal_css_cache');
                        $cleared[] = 'oxygen';
                    }
                    break;

                case 'beaver-builder':
                    if (class_exists('FLBuilder')) {
                        \FLBuilderModel::delete_all_asset_cache();
                        $cleared[] = 'beaver-builder';
                    }
                    break;

                case 'divi':
                    if (function_exists('et_core_clear_wp_cache')) {
                        et_core_clear_wp_cache();
                        $cleared[] = 'divi';
                    }
                    break;
            }
        }

        return rest_ensure_response([
            'success' => true,
            'message' => 'Builder caches cleared',
            'builders_cleared' => $cleared,
        ]);
    }

    /**
     * Get CSS regeneration status for a post
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function get_css_status($request) {
        $post_id = intval($request['id']);

        if (!$post_id) {
            return new WP_Error('missing_post_id', 'post_id is required', ['status' => 400]);
        }

        $post = get_post($post_id);
        if (!$post) {
            return new WP_Error('post_not_found', 'Post not found', ['status' => 404]);
        }

        $builder = $this->detect_page_builder($post_id);
        $css_exists = false;
        $css_file_path = '';
        $last_regenerated = '';

        // Check for CSS file existence based on builder
        switch ($builder) {
            case 'elementor':
                $upload_dir = wp_upload_dir();
                $css_file_path = $upload_dir['basedir'] . '/elementor/css/post-' . $post_id . '.css';
                $css_exists = file_exists($css_file_path);
                if ($css_exists) {
                    $last_regenerated = gmdate('Y-m-d H:i:s', filemtime($css_file_path));
                }
                break;

            case 'bricks':
                $upload_dir = wp_upload_dir();
                $css_file_path = $upload_dir['basedir'] . '/bricks/css/post-' . $post_id . '.css';
                $css_exists = file_exists($css_file_path);
                if ($css_exists) {
                    $last_regenerated = gmdate('Y-m-d H:i:s', filemtime($css_file_path));
                }
                break;

            default:
                $css_exists = true; // Assume CSS exists for other builders
        }

        return rest_ensure_response([
            'success' => true,
            'builder' => $builder,
            'css_exists' => $css_exists,
            'css_file_path' => $css_file_path,
            'last_regenerated' => $last_regenerated,
        ]);
    }

    /**
     * Schedule CSS regeneration for later
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function schedule_css_regeneration($request) {
        $params = $request->get_json_params();
        $post_ids = isset($params['post_ids']) ? $params['post_ids'] : [];
        $schedule_time = isset($params['schedule_time']) ? $params['schedule_time'] : null;

        if (empty($post_ids) || !is_array($post_ids)) {
            return new WP_Error('missing_post_ids', 'post_ids array is required', ['status' => 400]);
        }

        // For now, we'll use WordPress cron to schedule the regeneration
        // In a production environment, you'd want a more robust job queue

        $timestamp = $schedule_time ? strtotime($schedule_time) : time() + 60; // Default: 1 minute from now
        $job_id = 'css_regen_' . md5(json_encode($post_ids) . $timestamp);

        // Schedule the event
        wp_schedule_single_event($timestamp, 'instarank_regenerate_css_batch', [$post_ids]);

        return rest_ensure_response([
            'success' => true,
            'message' => 'CSS regeneration scheduled',
            'job_id' => $job_id,
            'scheduled_count' => count($post_ids),
            'scheduled_time' => gmdate('Y-m-d H:i:s', $timestamp),
        ]);
    }

    // ========================================
    // UNUSED IMAGES DETECTION METHODS
    // ========================================

    /**
     * Analyze media library for unused images
     * Returns all images with their usage count and locations
     *
     * GET /wp-json/instarank/v1/media/analyze-usage
     * Parameters: per_page, page, unused_only (true/false)
     */
    public function analyze_media_usage($request) {
        $per_page = intval($request->get_param('per_page')) ?: 100;
        $page = intval($request->get_param('page')) ?: 1;
        $unused_only = $request->get_param('unused_only') === 'true';

        // Limit per_page to prevent timeouts
        $per_page = min($per_page, 100);

        // Get all image attachments
        $args = [
            'post_type' => 'attachment',
            'post_status' => 'inherit',
            'posts_per_page' => $per_page,
            'paged' => $page,
            'post_mime_type' => 'image',
            'orderby' => 'date',
            'order' => 'DESC'
        ];

        $query = new WP_Query($args);
        $results = [];

        foreach ($query->posts as $attachment) {
            $usage = $this->analyze_single_attachment_usage($attachment->ID);

            // Skip if filtering for unused only and this image is used
            if ($unused_only && $usage['usage_count'] > 0) {
                continue;
            }

            $file_path = get_attached_file($attachment->ID);
            $file_size = ($file_path && file_exists($file_path)) ? filesize($file_path) : 0;
            $metadata = wp_get_attachment_metadata($attachment->ID);

            $results[] = [
                'id' => $attachment->ID,
                'url' => wp_get_attachment_url($attachment->ID),
                'filename' => basename($file_path ?: $attachment->guid),
                'mime_type' => $attachment->post_mime_type,
                'file_size' => $file_size,
                'width' => isset($metadata['width']) ? $metadata['width'] : null,
                'height' => isset($metadata['height']) ? $metadata['height'] : null,
                'alt' => get_post_meta($attachment->ID, '_wp_attachment_image_alt', true),
                'title' => $attachment->post_title,
                'caption' => $attachment->post_excerpt,
                'description' => $attachment->post_content,
                'uploaded_date' => $attachment->post_date,
                'usage_count' => $usage['usage_count'],
                'usage_locations' => $usage['locations'],
                'is_featured_image' => $usage['is_featured_image'],
                'detection_type' => $this->determine_detection_type($usage)
            ];
        }

        return rest_ensure_response([
            'success' => true,
            'images' => $results,
            'total' => $query->found_posts,
            'page' => $page,
            'per_page' => $per_page,
            'total_pages' => $query->max_num_pages
        ]);
    }

    /**
     * Analyze usage for a single attachment
     * Checks all possible locations where an image might be used
     *
     * @param int $attachment_id The attachment ID
     * @return array Usage data with count and locations
     */
    private function analyze_single_attachment_usage($attachment_id) {
        global $wpdb;

        $locations = [];
        $usage_count = 0;
        $is_featured_image = false;

        $attachment_url = wp_get_attachment_url($attachment_id);

        // 1. Check if used as featured image (thumbnail)
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $featured_posts = $wpdb->get_results($wpdb->prepare(
            "SELECT pm.post_id FROM {$wpdb->postmeta} pm
             INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
             WHERE pm.meta_key = '_thumbnail_id' AND pm.meta_value = %d
             AND p.post_status IN ('publish', 'draft', 'pending', 'private')",
            $attachment_id
        ));

        foreach ($featured_posts as $row) {
            $post = get_post($row->post_id);
            if ($post) {
                $is_featured_image = true;
                $usage_count++;
                $locations[] = [
                    'type' => 'featured_image',
                    'post_id' => $post->ID,
                    'post_title' => $post->post_title,
                    'post_url' => get_permalink($post->ID),
                    'post_type' => $post->post_type
                ];
            }
        }

        // 2. Check post content (posts, pages, custom post types)
        // Look for both full URL and wp-image-{id} class
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $content_posts = $wpdb->get_results($wpdb->prepare(
            "SELECT ID, post_title, post_type, post_status FROM {$wpdb->posts}
             WHERE post_status IN ('publish', 'draft', 'pending', 'private')
             AND post_type NOT IN ('attachment', 'revision', 'nav_menu_item')
             AND (post_content LIKE %s OR post_content LIKE %s)",
            '%' . $wpdb->esc_like($attachment_url) . '%',
            '%wp-image-' . $attachment_id . '%'
        ));

        foreach ($content_posts as $post) {
            // Check if we already counted this post (as featured image)
            $already_counted = false;
            foreach ($locations as $loc) {
                if (isset($loc['post_id']) && $loc['post_id'] == $post->ID && $loc['type'] === 'featured_image') {
                    $already_counted = true;
                    break;
                }
            }

            if (!$already_counted) {
                $usage_count++;
                $locations[] = [
                    'type' => 'post_content',
                    'post_id' => $post->ID,
                    'post_title' => $post->post_title,
                    'post_url' => get_permalink($post->ID),
                    'post_type' => $post->post_type
                ];
            }
        }

        // 3. Check page builder meta (Elementor, Kadence, Divi, Beaver Builder, Bricks)
        $builder_meta_keys = [
            '_elementor_data' => 'Elementor',
            '_kadence_blocks_meta' => 'Kadence',
            '_et_builder_version' => 'Divi',
            '_fl_builder_data' => 'Beaver Builder',
            '_bricks_page_content_2' => 'Bricks'
        ];

        foreach ($builder_meta_keys as $meta_key => $builder_name) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $builder_posts = $wpdb->get_results($wpdb->prepare(
                "SELECT pm.post_id FROM {$wpdb->postmeta} pm
                 INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
                 WHERE pm.meta_key = %s
                 AND (pm.meta_value LIKE %s OR pm.meta_value LIKE %s)
                 AND p.post_status IN ('publish', 'draft', 'pending', 'private')",
                $meta_key,
                '%' . $wpdb->esc_like($attachment_url) . '%',
                '%' . $wpdb->esc_like('"id":' . $attachment_id) . '%'
            ));

            foreach ($builder_posts as $row) {
                // Check if already counted
                $already_counted = false;
                foreach ($locations as $loc) {
                    if (isset($loc['post_id']) && $loc['post_id'] == $row->post_id) {
                        $already_counted = true;
                        break;
                    }
                }

                if (!$already_counted) {
                    $post = get_post($row->post_id);
                    if ($post) {
                        $usage_count++;
                        $locations[] = [
                            'type' => 'page_builder',
                            'builder' => $builder_name,
                            'post_id' => $post->ID,
                            'post_title' => $post->post_title,
                            'post_url' => get_permalink($post->ID),
                            'post_type' => $post->post_type
                        ];
                    }
                }
            }
        }

        // 4. Check theme customizer/options
        $theme_mods = get_theme_mods();
        if ($theme_mods && is_array($theme_mods)) {
            $options_str = wp_json_encode($theme_mods);
            if (strpos($options_str, $attachment_url) !== false ||
                strpos($options_str, '"' . $attachment_id . '"') !== false) {
                $usage_count++;
                $locations[] = [
                    'type' => 'theme_customizer',
                    'location' => 'Theme Settings'
                ];
            }
        }

        // 5. Check widgets
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $widget_options = $wpdb->get_results(
            "SELECT option_name, option_value FROM {$wpdb->options}
             WHERE option_name LIKE 'widget_%'"
        );

        foreach ($widget_options as $widget) {
            if (strpos($widget->option_value, $attachment_url) !== false ||
                strpos($widget->option_value, '"' . $attachment_id . '"') !== false) {
                $usage_count++;
                $locations[] = [
                    'type' => 'widget',
                    'widget_name' => str_replace('widget_', '', $widget->option_name)
                ];
                break; // Count widgets as one usage
            }
        }

        // 6. Check WooCommerce product galleries
        if (class_exists('WooCommerce')) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $gallery_posts = $wpdb->get_results($wpdb->prepare(
                "SELECT pm.post_id FROM {$wpdb->postmeta} pm
                 INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
                 WHERE pm.meta_key = '_product_image_gallery'
                 AND pm.meta_value LIKE %s
                 AND p.post_status IN ('publish', 'draft', 'pending', 'private')",
                '%' . $attachment_id . '%'
            ));

            foreach ($gallery_posts as $row) {
                // Verify it's actually in the gallery (not just substring match)
                $gallery = get_post_meta($row->post_id, '_product_image_gallery', true);
                $gallery_ids = explode(',', $gallery);
                if (in_array($attachment_id, array_map('intval', $gallery_ids))) {
                    $post = get_post($row->post_id);
                    if ($post) {
                        $usage_count++;
                        $locations[] = [
                            'type' => 'woocommerce_gallery',
                            'post_id' => $post->ID,
                            'post_title' => $post->post_title,
                            'post_url' => get_permalink($post->ID)
                        ];
                    }
                }
            }
        }

        // 7. Check site icon and logo
        $site_icon_id = get_option('site_icon');
        if ($site_icon_id && intval($site_icon_id) === intval($attachment_id)) {
            $usage_count++;
            $locations[] = [
                'type' => 'site_icon',
                'location' => 'Site Icon (Favicon)'
            ];
        }

        $custom_logo_id = get_theme_mod('custom_logo');
        if ($custom_logo_id && intval($custom_logo_id) === intval($attachment_id)) {
            $usage_count++;
            $locations[] = [
                'type' => 'site_logo',
                'location' => 'Site Logo'
            ];
        }

        return [
            'usage_count' => $usage_count,
            'locations' => $locations,
            'is_featured_image' => $is_featured_image
        ];
    }

    /**
     * Determine detection type based on usage analysis
     *
     * @param array $usage Usage analysis result
     * @return string|null Detection type or null if not unused
     */
    private function determine_detection_type($usage) {
        if ($usage['usage_count'] === 0) {
            return 'media_library_orphan';
        } elseif ($usage['usage_count'] === 1) {
            return 'low_usage';
        }
        return null;
    }

    /**
     * Get detailed usage info for a specific attachment
     *
     * GET /wp-json/instarank/v1/media/{id}/usage
     */
    public function get_attachment_usage($request) {
        $attachment_id = intval($request->get_param('id'));

        if (!$attachment_id) {
            return new WP_Error('invalid_id', 'Attachment ID is required', ['status' => 400]);
        }

        $attachment = get_post($attachment_id);
        if (!$attachment || $attachment->post_type !== 'attachment') {
            return new WP_Error('not_found', 'Attachment not found', ['status' => 404]);
        }

        $usage = $this->analyze_single_attachment_usage($attachment_id);

        return rest_ensure_response([
            'success' => true,
            'attachment_id' => $attachment_id,
            'filename' => basename(get_attached_file($attachment_id) ?: $attachment->guid),
            'url' => wp_get_attachment_url($attachment_id),
            'usage' => $usage
        ]);
    }

    /**
     * Delete attachment and return backup data for restore
     *
     * POST /wp-json/instarank/v1/media/{id}/delete
     */
    public function delete_attachment_with_backup($request) {
        $attachment_id = intval($request->get_param('id'));

        if (!$attachment_id) {
            return new WP_Error('invalid_id', 'Attachment ID is required', ['status' => 400]);
        }

        $attachment = get_post($attachment_id);
        if (!$attachment || $attachment->post_type !== 'attachment') {
            return new WP_Error('not_found', 'Attachment not found', ['status' => 404]);
        }

        // Note: Permission is already verified via API key authentication in permission_callback
        // The API key grants full access to the site's media operations

        // Get all metadata before deletion
        $file_path = get_attached_file($attachment_id);
        $url = wp_get_attachment_url($attachment_id);
        $metadata = wp_get_attachment_metadata($attachment_id);

        // Read file content for backup (base64 encoded)
        $file_content = null;
        $file_size = 0;
        if ($file_path && file_exists($file_path)) {
            $file_size = filesize($file_path);
            // Only encode files under 10MB to prevent memory issues
            if ($file_size < 10 * 1024 * 1024) {
                // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
                $file_content = base64_encode(file_get_contents($file_path));
            }
        }

        $backup_data = [
            'original_id' => $attachment_id,
            'url' => $url,
            'filename' => basename($file_path ?: $attachment->guid),
            'mime_type' => $attachment->post_mime_type,
            'file_size' => $file_size,
            'alt_text' => get_post_meta($attachment_id, '_wp_attachment_image_alt', true),
            'title' => $attachment->post_title,
            'caption' => $attachment->post_excerpt,
            'description' => $attachment->post_content,
            'metadata' => $metadata,
            'file_content' => $file_content
        ];

        // Delete the attachment (this also deletes the files)
        $deleted = wp_delete_attachment($attachment_id, true);

        if (!$deleted) {
            return new WP_Error('delete_failed', 'Failed to delete attachment', ['status' => 500]);
        }

        return rest_ensure_response([
            'success' => true,
            'message' => 'Attachment deleted successfully',
            'backup_data' => $backup_data
        ]);
    }

    /**
     * Bulk delete attachments with backup data
     *
     * POST /wp-json/instarank/v1/media/bulk-delete
     * Body: { "attachment_ids": [1, 2, 3] }
     */
    public function bulk_delete_attachments($request) {
        $params = $request->get_json_params();
        $attachment_ids = isset($params['attachment_ids']) ? $params['attachment_ids'] : [];

        if (empty($attachment_ids) || !is_array($attachment_ids)) {
            return new WP_Error('invalid_request', 'attachment_ids array is required', ['status' => 400]);
        }

        // Limit batch size
        if (count($attachment_ids) > 50) {
            return new WP_Error('too_many', 'Maximum 50 attachments per request', ['status' => 400]);
        }

        $results = [];
        $success_count = 0;
        $failed_count = 0;

        foreach ($attachment_ids as $attachment_id) {
            $attachment_id = intval($attachment_id);
            $attachment = get_post($attachment_id);

            if (!$attachment || $attachment->post_type !== 'attachment') {
                $failed_count++;
                $results[] = [
                    'id' => $attachment_id,
                    'success' => false,
                    'error' => 'Not found'
                ];
                continue;
            }

            // Note: Permission is already verified via API key authentication in permission_callback
            // The API key grants full access to the site's media operations

            // Get backup data
            $file_path = get_attached_file($attachment_id);
            $file_content = null;
            $file_size = 0;

            if ($file_path && file_exists($file_path)) {
                $file_size = filesize($file_path);
                if ($file_size < 10 * 1024 * 1024) {
                    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
                    $file_content = base64_encode(file_get_contents($file_path));
                }
            }

            $backup_data = [
                'original_id' => $attachment_id,
                'url' => wp_get_attachment_url($attachment_id),
                'filename' => basename($file_path ?: $attachment->guid),
                'mime_type' => $attachment->post_mime_type,
                'file_size' => $file_size,
                'alt_text' => get_post_meta($attachment_id, '_wp_attachment_image_alt', true),
                'title' => $attachment->post_title,
                'caption' => $attachment->post_excerpt,
                'description' => $attachment->post_content,
                'file_content' => $file_content
            ];

            $deleted = wp_delete_attachment($attachment_id, true);

            if ($deleted) {
                $success_count++;
                $results[] = [
                    'id' => $attachment_id,
                    'success' => true,
                    'backup_data' => $backup_data
                ];
            } else {
                $failed_count++;
                $results[] = [
                    'id' => $attachment_id,
                    'success' => false,
                    'error' => 'Delete failed'
                ];
            }
        }

        return rest_ensure_response([
            'success' => $failed_count === 0,
            'total' => count($attachment_ids),
            'deleted' => $success_count,
            'failed' => $failed_count,
            'results' => $results
        ]);
    }

    /**
     * Restore a deleted attachment from backup data
     *
     * POST /wp-json/instarank/v1/media/restore
     * Body: { "filename": "image.jpg", "file_content": "base64...", "mime_type": "image/jpeg", ... }
     */
    public function restore_attachment_from_backup($request) {
        $params = $request->get_json_params();

        $required_fields = ['filename', 'file_content', 'mime_type'];
        foreach ($required_fields as $field) {
            if (empty($params[$field])) {
                return new WP_Error('invalid_request', "$field is required", ['status' => 400]);
            }
        }

        // Note: Permission is already verified via API key authentication in permission_callback
        // The API key grants full access to the site's media operations

        // Decode file content
        // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode
        $file_content = base64_decode($params['file_content']);
        if ($file_content === false) {
            return new WP_Error('invalid_content', 'Invalid file content (base64 decode failed)', ['status' => 400]);
        }

        // Create temp file
        $upload_dir = wp_upload_dir();
        $unique_filename = wp_unique_filename($upload_dir['path'], sanitize_file_name($params['filename']));
        $temp_file = $upload_dir['path'] . '/' . $unique_filename;

        // Write file
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
        $written = file_put_contents($temp_file, $file_content);
        if ($written === false) {
            return new WP_Error('write_failed', 'Failed to write file', ['status' => 500]);
        }

        // Prepare attachment data
        $attachment = [
            'post_mime_type' => sanitize_mime_type($params['mime_type']),
            'post_title' => isset($params['title']) ? sanitize_text_field($params['title']) : sanitize_file_name(pathinfo($params['filename'], PATHINFO_FILENAME)),
            'post_content' => isset($params['description']) ? wp_kses_post($params['description']) : '',
            'post_excerpt' => isset($params['caption']) ? sanitize_text_field($params['caption']) : '',
            'post_status' => 'inherit'
        ];

        // Insert attachment
        $attachment_id = wp_insert_attachment($attachment, $temp_file);

        if (is_wp_error($attachment_id)) {
            // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink
            @unlink($temp_file);
            return new WP_Error('insert_failed', 'Failed to create attachment: ' . $attachment_id->get_error_message(), ['status' => 500]);
        }

        // Generate metadata
        require_once ABSPATH . 'wp-admin/includes/image.php';
        $attach_data = wp_generate_attachment_metadata($attachment_id, $temp_file);
        wp_update_attachment_metadata($attachment_id, $attach_data);

        // Set alt text if provided
        if (!empty($params['alt_text'])) {
            update_post_meta($attachment_id, '_wp_attachment_image_alt', sanitize_text_field($params['alt_text']));
        }

        return rest_ensure_response([
            'success' => true,
            'attachment_id' => $attachment_id,
            'url' => wp_get_attachment_url($attachment_id),
            'message' => 'Attachment restored successfully'
        ]);
    }

    /**
     * Rename an attachment file to an SEO-friendly filename
     *
     * POST /wp-json/instarank/v1/media/{id}/rename
     * Body: { "new_filename": "seo-friendly-name" } (without extension)
     */
    public function rename_attachment_file($request) {
        $attachment_id = intval($request->get_param('id'));
        $params = $request->get_json_params();

        if (!$attachment_id) {
            return new WP_Error('invalid_id', 'Attachment ID is required', ['status' => 400]);
        }

        if (empty($params['new_filename'])) {
            return new WP_Error('invalid_request', 'new_filename is required', ['status' => 400]);
        }

        $attachment = get_post($attachment_id);
        if (!$attachment || $attachment->post_type !== 'attachment') {
            return new WP_Error('not_found', 'Attachment not found', ['status' => 404]);
        }

        // Get the current file path
        $old_file_path = get_attached_file($attachment_id);
        if (!$old_file_path || !file_exists($old_file_path)) {
            return new WP_Error('file_not_found', 'Attachment file not found on server', ['status' => 404]);
        }

        // Get file info
        $path_info = pathinfo($old_file_path);
        $extension = strtolower($path_info['extension']);
        $upload_dir = wp_upload_dir();

        // Sanitize the new filename
        $new_basename = sanitize_file_name($params['new_filename']);
        // Remove any extension that might have been included
        $new_basename = preg_replace('/\.[^.]+$/', '', $new_basename);
        // Ensure lowercase and no special chars
        $new_basename = strtolower($new_basename);
        $new_basename = preg_replace('/[^a-z0-9-]/', '-', $new_basename);
        $new_basename = preg_replace('/-+/', '-', $new_basename);
        $new_basename = trim($new_basename, '-');

        if (empty($new_basename)) {
            return new WP_Error('invalid_filename', 'Invalid filename after sanitization', ['status' => 400]);
        }

        // Initialize WP_Filesystem
        global $wp_filesystem;
        if (empty($wp_filesystem)) {
            require_once ABSPATH . '/wp-admin/includes/file.php';
            WP_Filesystem();
        }

        // Generate unique filename if it already exists
        $new_filename = $new_basename . '.' . $extension;
        $new_file_path = $path_info['dirname'] . '/' . wp_unique_filename($path_info['dirname'], $new_filename);
        $actual_new_filename = basename($new_file_path);

        // Rename the main file using WP_Filesystem
        if (!$wp_filesystem->move($old_file_path, $new_file_path)) {
            return new WP_Error('rename_failed', 'Failed to rename file', ['status' => 500]);
        }

        // Update attachment metadata
        update_attached_file($attachment_id, $new_file_path);

        // Get and update attachment metadata for image sizes
        $metadata = wp_get_attachment_metadata($attachment_id);
        $old_url = wp_get_attachment_url($attachment_id);

        if ($metadata && isset($metadata['file'])) {
            // Update main file in metadata
            $metadata['file'] = str_replace(basename($old_file_path), $actual_new_filename, $metadata['file']);

            // Rename and update all image sizes
            if (isset($metadata['sizes']) && is_array($metadata['sizes'])) {
                foreach ($metadata['sizes'] as $size_name => $size_data) {
                    if (isset($size_data['file'])) {
                        $old_size_file = $path_info['dirname'] . '/' . $size_data['file'];
                        if (file_exists($old_size_file)) {
                            // Generate new size filename
                            $size_extension = pathinfo($size_data['file'], PATHINFO_EXTENSION);
                            $size_suffix = preg_replace('/^' . preg_quote($path_info['filename'], '/') . '/', '', pathinfo($size_data['file'], PATHINFO_FILENAME));
                            $new_size_filename = $new_basename . $size_suffix . '.' . $size_extension;
                            $new_size_path = $path_info['dirname'] . '/' . $new_size_filename;

                            if ($wp_filesystem->move($old_size_file, $new_size_path)) {
                                $metadata['sizes'][$size_name]['file'] = $new_size_filename;
                            }
                        }
                    }
                }
            }

            wp_update_attachment_metadata($attachment_id, $metadata);
        }

        // Update post title and slug if not set
        $update_data = [
            'ID' => $attachment_id,
            'post_name' => $new_basename,
        ];
        // Only update title if it matches old filename
        if ($attachment->post_title === $path_info['filename']) {
            $update_data['post_title'] = ucwords(str_replace('-', ' ', $new_basename));
        }
        wp_update_post($update_data);

        // Update references in post content (optional - can be slow on large sites)
        $new_url = wp_get_attachment_url($attachment_id);
        $posts_updated = 0;

        if (!empty($params['update_references']) && $old_url !== $new_url) {
            global $wpdb;

            // Find posts containing the old URL
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $posts_with_image = $wpdb->get_results($wpdb->prepare(
                "SELECT ID, post_content FROM {$wpdb->posts} WHERE post_content LIKE %s AND post_status != 'trash'",
                '%' . $wpdb->esc_like(basename($old_file_path)) . '%'
            ));

            foreach ($posts_with_image as $post) {
                $new_content = str_replace(
                    [basename($old_file_path), $old_url],
                    [$actual_new_filename, $new_url],
                    $post->post_content
                );

                if ($new_content !== $post->post_content) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $wpdb->update(
                        $wpdb->posts,
                        ['post_content' => $new_content],
                        ['ID' => $post->ID]
                    );
                    $posts_updated++;
                }
            }

            // Clear caches
            if ($posts_updated > 0) {
                wp_cache_flush();
            }
        }

        return rest_ensure_response([
            'success' => true,
            'attachment_id' => $attachment_id,
            'old_filename' => basename($old_file_path),
            'new_filename' => $actual_new_filename,
            'old_url' => $old_url,
            'new_url' => $new_url,
            'posts_updated' => $posts_updated,
            'message' => 'Attachment renamed successfully'
        ]);
    }

    /**
     * Bulk rename attachments
     *
     * POST /wp-json/instarank/v1/media/bulk-rename
     * Body: { "attachments": [{ "id": 123, "new_filename": "seo-name" }, ...], "update_references": true }
     */
    public function bulk_rename_attachments($request) {
        $params = $request->get_json_params();
        $attachments = isset($params['attachments']) ? $params['attachments'] : [];
        $update_references = !empty($params['update_references']);

        if (empty($attachments) || !is_array($attachments)) {
            return new WP_Error('invalid_request', 'attachments array is required', ['status' => 400]);
        }

        if (count($attachments) > 50) {
            return new WP_Error('too_many', 'Maximum 50 attachments per request', ['status' => 400]);
        }

        $results = [];
        $success_count = 0;
        $failed_count = 0;

        foreach ($attachments as $attachment_data) {
            if (empty($attachment_data['id']) || empty($attachment_data['new_filename'])) {
                $failed_count++;
                $results[] = [
                    'id' => $attachment_data['id'] ?? null,
                    'success' => false,
                    'error' => 'Missing id or new_filename'
                ];
                continue;
            }

            // Create a mock request for the single rename function
            $mock_request = new WP_REST_Request('POST');
            $mock_request->set_param('id', $attachment_data['id']);
            $mock_request->set_body_params([
                'new_filename' => $attachment_data['new_filename'],
                'update_references' => $update_references
            ]);

            $result = $this->rename_attachment_file($mock_request);

            if (is_wp_error($result)) {
                $failed_count++;
                $results[] = [
                    'id' => $attachment_data['id'],
                    'success' => false,
                    'error' => $result->get_error_message()
                ];
            } else {
                $response_data = $result->get_data();
                if ($response_data['success']) {
                    $success_count++;
                    $results[] = [
                        'id' => $attachment_data['id'],
                        'success' => true,
                        'old_filename' => $response_data['old_filename'],
                        'new_filename' => $response_data['new_filename'],
                        'new_url' => $response_data['new_url']
                    ];
                } else {
                    $failed_count++;
                    $results[] = [
                        'id' => $attachment_data['id'],
                        'success' => false,
                        'error' => $response_data['error'] ?? 'Unknown error'
                    ];
                }
            }
        }

        return rest_ensure_response([
            'success' => $failed_count === 0,
            'renamed' => $success_count,
            'failed' => $failed_count,
            'results' => $results
        ]);
    }

    /**
     * Update image attributes in post content (lazy loading, dimensions)
     *
     * POST /wp-json/instarank/v1/content/update-image-attributes
     * Body: {
     *   "post_id": 123,
     *   "image_url": "https://example.com/image.jpg",
     *   "add_lazy_loading": true,
     *   "add_dimensions": true,
     *   "width": 800,
     *   "height": 600
     * }
     */
    public function update_content_image_attributes($request) {
        $params = $request->get_json_params();
        $post_id = isset($params['post_id']) ? intval($params['post_id']) : 0;
        $image_url = isset($params['image_url']) ? esc_url($params['image_url']) : '';
        $add_lazy_loading = !empty($params['add_lazy_loading']);
        $add_dimensions = !empty($params['add_dimensions']);
        $width = isset($params['width']) ? intval($params['width']) : 0;
        $height = isset($params['height']) ? intval($params['height']) : 0;

        if (!$post_id) {
            return new WP_Error('invalid_request', 'post_id is required', ['status' => 400]);
        }

        if (!$image_url) {
            return new WP_Error('invalid_request', 'image_url is required', ['status' => 400]);
        }

        // Get the post
        $post = get_post($post_id);
        if (!$post) {
            return new WP_Error('not_found', 'Post not found', ['status' => 404]);
        }

        $content = $post->post_content;
        $original_content = $content;
        $changes_made = [];

        // Find and update img tags with this URL
        // Escape the URL for use in regex
        $escaped_url = preg_quote($image_url, '/');
        // Also match URL variants (with/without domain, scaled versions)
        $url_without_domain = preg_replace('/^https?:\/\/[^\/]+/', '', $image_url);
        $escaped_url_partial = preg_quote($url_without_domain, '/');

        // Pattern to match img tags containing this URL
        $pattern = '/<img[^>]*src=["\']([^"\']*' . $escaped_url_partial . '[^"\']*)["\'][^>]*>/i';

        $content = preg_replace_callback($pattern, function($matches) use ($add_lazy_loading, $add_dimensions, $width, $height, &$changes_made) {
            $img_tag = $matches[0];
            $original_tag = $img_tag;

            // Add lazy loading if needed and not present
            if ($add_lazy_loading) {
                if (stripos($img_tag, 'loading=') === false) {
                    $img_tag = preg_replace('/<img\s/i', '<img loading="lazy" ', $img_tag);
                    $changes_made[] = 'added_lazy_loading';
                }
            }

            // Add dimensions if needed
            if ($add_dimensions && $width > 0 && $height > 0) {
                // Check if width is missing
                if (stripos($img_tag, 'width=') === false) {
                    $img_tag = preg_replace('/<img\s/i', '<img width="' . $width . '" ', $img_tag);
                    $changes_made[] = 'added_width';
                }
                // Check if height is missing
                if (stripos($img_tag, 'height=') === false) {
                    $img_tag = preg_replace('/<img\s/i', '<img height="' . $height . '" ', $img_tag);
                    $changes_made[] = 'added_height';
                }
            }

            return $img_tag;
        }, $content);

        // Also check page builder meta for this image
        $page_builder_updated = false;
        $page_builder_meta_keys = [
            '_elementor_data',
            '_kadence_blocks_meta',
            '_et_builder_data',
            '_fl_builder_data',
            '_bricks_page_content_2'
        ];

        foreach ($page_builder_meta_keys as $meta_key) {
            $meta_value = get_post_meta($post_id, $meta_key, true);
            if (!empty($meta_value)) {
                $original_meta = $meta_value;
                $is_json = is_string($meta_value) && (substr($meta_value, 0, 1) === '[' || substr($meta_value, 0, 1) === '{');

                if ($is_json) {
                    $decoded = json_decode($meta_value, true);
                    if ($decoded) {
                        $decoded = $this->update_image_attributes_in_array($decoded, $image_url, $add_lazy_loading, $add_dimensions, $width, $height);
                        $new_meta = wp_json_encode($decoded);
                        if ($new_meta !== $meta_value) {
                            update_post_meta($post_id, $meta_key, wp_slash($new_meta));
                            $page_builder_updated = true;
                        }
                    }
                }
            }
        }

        // Update the post content if changed
        $content_updated = false;
        if ($content !== $original_content) {
            $result = wp_update_post([
                'ID' => $post_id,
                'post_content' => $content
            ], true);

            if (is_wp_error($result)) {
                return $result;
            }
            $content_updated = true;
        }

        if (!$content_updated && !$page_builder_updated && empty($changes_made)) {
            return rest_ensure_response([
                'success' => true,
                'message' => 'No changes needed - attributes already present or image not found in content',
                'post_id' => $post_id,
                'changes_made' => []
            ]);
        }

        // Clear caches
        clean_post_cache($post_id);

        return rest_ensure_response([
            'success' => true,
            'post_id' => $post_id,
            'image_url' => $image_url,
            'content_updated' => $content_updated,
            'page_builder_updated' => $page_builder_updated,
            'changes_made' => array_unique($changes_made),
            'message' => 'Image attributes updated successfully'
        ]);
    }

    /**
     * Recursively update image attributes in array structures (for page builder data)
     */
    private function update_image_attributes_in_array($data, $image_url, $add_lazy_loading, $add_dimensions, $width, $height) {
        if (!is_array($data)) {
            if (is_string($data) && strpos($data, '<img') !== false && strpos($data, $image_url) !== false) {
                // This is HTML content containing our image
                $pattern = '/<img[^>]*src=["\'][^"\']*' . preg_quote($image_url, '/') . '[^"\']*["\'][^>]*>/i';
                $data = preg_replace_callback($pattern, function($matches) use ($add_lazy_loading, $add_dimensions, $width, $height) {
                    $img_tag = $matches[0];

                    if ($add_lazy_loading && stripos($img_tag, 'loading=') === false) {
                        $img_tag = preg_replace('/<img\s/i', '<img loading="lazy" ', $img_tag);
                    }

                    if ($add_dimensions && $width > 0 && $height > 0) {
                        if (stripos($img_tag, 'width=') === false) {
                            $img_tag = preg_replace('/<img\s/i', '<img width="' . $width . '" ', $img_tag);
                        }
                        if (stripos($img_tag, 'height=') === false) {
                            $img_tag = preg_replace('/<img\s/i', '<img height="' . $height . '" ', $img_tag);
                        }
                    }

                    return $img_tag;
                }, $data);
            }
            return $data;
        }

        foreach ($data as $key => $value) {
            $data[$key] = $this->update_image_attributes_in_array($value, $image_url, $add_lazy_loading, $add_dimensions, $width, $height);
        }

        return $data;
    }

    /**
     * Bulk update image attributes in multiple posts
     *
     * POST /wp-json/instarank/v1/content/bulk-update-image-attributes
     * Body: {
     *   "images": [
     *     { "post_id": 123, "image_url": "...", "add_lazy_loading": true, "add_dimensions": true, "width": 800, "height": 600 },
     *     ...
     *   ]
     * }
     */
    public function bulk_update_content_image_attributes($request) {
        $params = $request->get_json_params();
        $images = isset($params['images']) ? $params['images'] : [];

        if (empty($images) || !is_array($images)) {
            return new WP_Error('invalid_request', 'images array is required', ['status' => 400]);
        }

        if (count($images) > 100) {
            return new WP_Error('too_many', 'Maximum 100 images per request', ['status' => 400]);
        }

        $results = [];
        $success_count = 0;
        $failed_count = 0;

        foreach ($images as $image_data) {
            if (empty($image_data['post_id']) || empty($image_data['image_url'])) {
                $failed_count++;
                $results[] = [
                    'post_id' => $image_data['post_id'] ?? null,
                    'image_url' => $image_data['image_url'] ?? null,
                    'success' => false,
                    'error' => 'Missing post_id or image_url'
                ];
                continue;
            }

            // Create a mock request for the single update function
            $mock_request = new WP_REST_Request('POST');
            $mock_request->set_body_params([
                'post_id' => $image_data['post_id'],
                'image_url' => $image_data['image_url'],
                'add_lazy_loading' => !empty($image_data['add_lazy_loading']),
                'add_dimensions' => !empty($image_data['add_dimensions']),
                'width' => isset($image_data['width']) ? intval($image_data['width']) : 0,
                'height' => isset($image_data['height']) ? intval($image_data['height']) : 0
            ]);

            $result = $this->update_content_image_attributes($mock_request);

            if (is_wp_error($result)) {
                $failed_count++;
                $results[] = [
                    'post_id' => $image_data['post_id'],
                    'image_url' => $image_data['image_url'],
                    'success' => false,
                    'error' => $result->get_error_message()
                ];
            } else {
                $response_data = $result->get_data();
                if ($response_data['success']) {
                    $success_count++;
                    $results[] = [
                        'post_id' => $image_data['post_id'],
                        'image_url' => $image_data['image_url'],
                        'success' => true,
                        'changes_made' => $response_data['changes_made'] ?? []
                    ];
                } else {
                    $failed_count++;
                    $results[] = [
                        'post_id' => $image_data['post_id'],
                        'image_url' => $image_data['image_url'],
                        'success' => false,
                        'error' => $response_data['error'] ?? 'Unknown error'
                    ];
                }
            }
        }

        return rest_ensure_response([
            'success' => $failed_count === 0,
            'updated' => $success_count,
            'failed' => $failed_count,
            'results' => $results
        ]);
    }

    // ========== SITEMAP, ROBOTS.TXT & LLMS.TXT MANAGEMENT ==========

    /**
     * Get sitemap settings
     */
    public function get_sitemap_settings($request) {
        return rest_ensure_response([
            'success' => true,
            'enabled' => (bool) get_option('instarank_sitemap_enabled', true),
            'post_types' => get_option('instarank_sitemap_post_types', ['post', 'page']),
            'taxonomies' => get_option('instarank_sitemap_taxonomies', ['category', 'post_tag']),
            'sitemap_url' => home_url('/sitemap.xml'),
            'last_regenerated' => get_option('instarank_sitemap_last_regenerated', null),
        ]);
    }

    /**
     * Update sitemap settings
     */
    public function update_sitemap_settings($request) {
        $params = $request->get_json_params();

        $enabled = isset($params['enabled']) ? (bool) $params['enabled'] : null;
        $post_types = isset($params['post_types']) ? $params['post_types'] : null;
        $taxonomies = isset($params['taxonomies']) ? $params['taxonomies'] : null;

        if ($enabled !== null) {
            update_option('instarank_sitemap_enabled', $enabled);
        }

        if ($post_types !== null && is_array($post_types)) {
            // Validate post types exist
            $public_post_types = array_keys(get_post_types(['public' => true], 'names'));
            $valid_post_types = array_intersect($post_types, $public_post_types);
            update_option('instarank_sitemap_post_types', array_values($valid_post_types));
        }

        if ($taxonomies !== null && is_array($taxonomies)) {
            // Validate taxonomies exist
            $public_taxonomies = array_keys(get_taxonomies(['public' => true], 'names'));
            $valid_taxonomies = array_intersect($taxonomies, $public_taxonomies);
            update_option('instarank_sitemap_taxonomies', array_values($valid_taxonomies));
        }

        // Invalidate sitemap cache if generator exists
        if (class_exists('InstaRank_Sitemap_Generator')) {
            InstaRank_Sitemap_Generator::instance()->invalidate_cache();
        }

        update_option('instarank_sitemap_last_regenerated', current_time('mysql'));

        return rest_ensure_response([
            'success' => true,
            'message' => 'Sitemap settings updated',
            'enabled' => (bool) get_option('instarank_sitemap_enabled', true),
            'post_types' => get_option('instarank_sitemap_post_types', ['post', 'page']),
            'taxonomies' => get_option('instarank_sitemap_taxonomies', ['category', 'post_tag']),
        ]);
    }

    /**
     * Get available post types and taxonomies for sitemap configuration
     */
    public function get_available_sitemap_options($request) {
        $post_types = get_post_types(['public' => true], 'objects');
        $taxonomies = get_taxonomies(['public' => true], 'objects');

        $post_type_data = [];
        foreach ($post_types as $pt) {
            if ($pt->name === 'attachment') {
                continue; // Skip attachments
            }
            $count = wp_count_posts($pt->name);
            $post_type_data[] = [
                'name' => $pt->name,
                'label' => $pt->label,
                'count' => isset($count->publish) ? (int) $count->publish : 0,
            ];
        }

        $taxonomy_data = [];
        foreach ($taxonomies as $tax) {
            if ($tax->name === 'post_format') {
                continue; // Skip post formats
            }
            $count = wp_count_terms(['taxonomy' => $tax->name, 'hide_empty' => false]);
            $taxonomy_data[] = [
                'name' => $tax->name,
                'label' => $tax->label,
                'count' => is_wp_error($count) ? 0 : (int) $count,
            ];
        }

        return rest_ensure_response([
            'success' => true,
            'post_types' => $post_type_data,
            'taxonomies' => $taxonomy_data,
        ]);
    }

    /**
     * Regenerate sitemap cache
     */
    public function regenerate_sitemap($request) {
        // Invalidate all sitemap caches
        if (class_exists('InstaRank_Sitemap_Generator')) {
            InstaRank_Sitemap_Generator::instance()->invalidate_cache();
        }

        update_option('instarank_sitemap_last_regenerated', current_time('mysql'));

        return rest_ensure_response([
            'success' => true,
            'message' => 'Sitemap cache regenerated',
            'regenerated_at' => get_option('instarank_sitemap_last_regenerated'),
        ]);
    }

    /**
     * Get sitemap preview (rendered XML index)
     */
    public function get_sitemap_preview($request) {
        $enabled = (bool) get_option('instarank_sitemap_enabled', true);

        if (!$enabled) {
            return rest_ensure_response([
                'success' => true,
                'enabled' => false,
                'preview' => '<!-- Sitemap is disabled -->',
            ]);
        }

        $post_types = get_option('instarank_sitemap_post_types', ['post', 'page']);
        $taxonomies = get_option('instarank_sitemap_taxonomies', ['category', 'post_tag']);

        $preview = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $preview .= '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

        foreach ($post_types as $pt) {
            $preview .= '  <sitemap>' . "\n";
            $preview .= '    <loc>' . esc_url(home_url("/sitemap-{$pt}.xml")) . '</loc>' . "\n";
            $preview .= '    <lastmod>' . gmdate('c') . '</lastmod>' . "\n";
            $preview .= '  </sitemap>' . "\n";
        }

        foreach ($taxonomies as $tax) {
            $preview .= '  <sitemap>' . "\n";
            $preview .= '    <loc>' . esc_url(home_url("/sitemap-tax-{$tax}.xml")) . '</loc>' . "\n";
            $preview .= '    <lastmod>' . gmdate('c') . '</lastmod>' . "\n";
            $preview .= '  </sitemap>' . "\n";
        }

        $preview .= '</sitemapindex>';

        return rest_ensure_response([
            'success' => true,
            'enabled' => true,
            'preview' => $preview,
            'url' => home_url('/sitemap.xml'),
        ]);
    }

    /**
     * Get robots.txt content
     */
    public function get_robots_txt($request) {
        $robots = InstaRank_Robots_Txt::instance();

        return rest_ensure_response([
            'success' => true,
            'content' => $robots->get_content(),
            'default_content' => $robots->get_default_content(),
            'is_custom' => !empty(get_option('instarank_robots_txt', '')),
            'robots_url' => home_url('/robots.txt'),
        ]);
    }

    /**
     * Update robots.txt content
     */
    public function update_robots_txt($request) {
        $params = $request->get_json_params();
        $content = isset($params['content']) ? $params['content'] : '';

        $robots = InstaRank_Robots_Txt::instance();

        if ($robots->save_content($content)) {
            return rest_ensure_response([
                'success' => true,
                'message' => 'Robots.txt updated',
                'content' => $robots->get_content(),
            ]);
        }

        return new WP_Error('save_failed', 'Failed to save robots.txt content', ['status' => 500]);
    }

    /**
     * Reset robots.txt to default
     */
    public function reset_robots_txt($request) {
        $robots = InstaRank_Robots_Txt::instance();
        $robots->reset_to_default();

        return rest_ensure_response([
            'success' => true,
            'message' => 'Robots.txt reset to default',
            'content' => $robots->get_default_content(),
        ]);
    }

    /**
     * Get llms.txt content and settings
     */
    public function get_llms_txt($request) {
        $llms = InstaRank_LLMs_Txt::instance();

        return rest_ensure_response([
            'success' => true,
            'enabled' => $llms->is_enabled(),
            'content' => $llms->get_content(),
            'default_content' => $llms->get_default_content(),
            'is_custom' => !empty(get_option('instarank_llms_txt', '')),
            'llms_url' => $llms->get_url(),
        ]);
    }

    /**
     * Update llms.txt content
     */
    public function update_llms_txt($request) {
        $params = $request->get_json_params();
        $content = isset($params['content']) ? $params['content'] : '';
        $enabled = isset($params['enabled']) ? (bool) $params['enabled'] : null;

        $llms = InstaRank_LLMs_Txt::instance();

        // Update enabled state if provided
        if ($enabled !== null) {
            if ($enabled) {
                $llms->enable();
            } else {
                $llms->disable();
            }
        }

        // Save content
        if ($llms->save_content($content)) {
            return rest_ensure_response([
                'success' => true,
                'message' => 'LLMs.txt updated',
                'enabled' => $llms->is_enabled(),
                'content' => $llms->get_content(),
            ]);
        }

        return new WP_Error('save_failed', 'Failed to save llms.txt content', ['status' => 500]);
    }

    /**
     * Reset llms.txt to default
     */
    public function reset_llms_txt($request) {
        $llms = InstaRank_LLMs_Txt::instance();
        $llms->reset_to_default();

        return rest_ensure_response([
            'success' => true,
            'message' => 'LLMs.txt reset to default',
            'content' => $llms->get_default_content(),
        ]);
    }
}

// Initialize the REST API endpoints
new InstaRank_API_Endpoints();
