<?php
/**
 * Agent API Endpoints
 * File system access, code execution, and codebase indexing for AI agent
 *
 * @package InstaRank
 * @since 1.3.0
 */

defined('ABSPATH') || exit;

class InstaRank_Agent_Endpoints {

    /**
     * Security validator for dangerous operations
     */
    private $blocked_paths = [
        'wp-admin',
        'wp-includes',
        '../',
        '..',
    ];

    private $sensitive_files = [
        'wp-config.php',
        '.htaccess',
        '.env',
    ];

    private $allowed_shell_commands = [
        'ls', 'cat', 'head', 'tail', 'grep', 'find', 'wc', 'du', 'df',
        'pwd', 'whoami', 'date', 'echo', 'php -v', 'php -i',
        'mysql --version', 'wp --version'
    ];

    public function __construct() {
        add_action('rest_api_init', [$this, 'register_routes']);
    }

    /**
     * Register REST API routes for agent capabilities
     */
    public function register_routes() {
        $namespace = 'instarank/v1';

        // ==========================================
        // FILE SYSTEM ENDPOINTS
        // ==========================================

        // Read file contents
        register_rest_route($namespace, '/filesystem/read', [
            'methods' => 'POST',
            'callback' => [$this, 'read_file'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Write file contents
        register_rest_route($namespace, '/filesystem/write', [
            'methods' => 'POST',
            'callback' => [$this, 'write_file'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // List directory contents
        register_rest_route($namespace, '/filesystem/list', [
            'methods' => 'POST',
            'callback' => [$this, 'list_directory'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Preview diff before writing
        register_rest_route($namespace, '/filesystem/diff', [
            'methods' => 'POST',
            'callback' => [$this, 'preview_diff'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Search files
        register_rest_route($namespace, '/filesystem/search', [
            'methods' => 'POST',
            'callback' => [$this, 'search_files'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // ==========================================
        // CODE EXECUTION ENDPOINTS
        // ==========================================

        // Execute WP-CLI command
        register_rest_route($namespace, '/wpcli/execute', [
            'methods' => 'POST',
            'callback' => [$this, 'execute_wpcli'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Execute PHP code
        register_rest_route($namespace, '/php/execute', [
            'methods' => 'POST',
            'callback' => [$this, 'execute_php'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Execute database query
        register_rest_route($namespace, '/db/query', [
            'methods' => 'POST',
            'callback' => [$this, 'execute_db_query'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Execute shell command (restricted)
        register_rest_route($namespace, '/shell/execute', [
            'methods' => 'POST',
            'callback' => [$this, 'execute_shell'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // ==========================================
        // CODEBASE INDEXING ENDPOINTS
        // ==========================================

        // Index codebase
        register_rest_route($namespace, '/codebase/index', [
            'methods' => 'POST',
            'callback' => [$this, 'index_codebase'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Search codebase
        register_rest_route($namespace, '/codebase/search', [
            'methods' => 'POST',
            'callback' => [$this, 'search_codebase'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);
    }

    /**
     * Verify API key from request (uses InstaRank_Auth_Manager)
     * Supports multiple header formats:
     * - X-WordPress-API-Key: <key>
     * - X-API-Key: <key>
     * - Authorization: Bearer <key>
     */
    public function verify_api_key($request) {
        $api_key = null;

        // Try X-WordPress-API-Key header first
        $api_key = $request->get_header('X-WordPress-API-Key');

        // Try X-API-Key header
        if (empty($api_key)) {
            $api_key = $request->get_header('X-API-Key');
        }

        // Try Authorization: Bearer header
        if (empty($api_key)) {
            $auth_header = $request->get_header('Authorization');
            if (!empty($auth_header) && preg_match('/^Bearer\s+(.+)$/i', $auth_header, $matches)) {
                $api_key = $matches[1];
            }
        }

        // Try from $_SERVER (for Apache mod_rewrite passthrough)
        if (empty($api_key) && !empty($_SERVER['HTTP_X_WORDPRESS_API_KEY'])) {
            $api_key = sanitize_text_field(wp_unslash($_SERVER['HTTP_X_WORDPRESS_API_KEY']));
        }
        if (empty($api_key) && !empty($_SERVER['REDIRECT_HTTP_X_WORDPRESS_API_KEY'])) {
            $api_key = sanitize_text_field(wp_unslash($_SERVER['REDIRECT_HTTP_X_WORDPRESS_API_KEY']));
        }

        if (empty($api_key)) {
            return false;
        }

        $auth_manager = new InstaRank_Auth_Manager();
        if (!$auth_manager->verify_api_key($api_key)) {
            return false;
        }

        return true;
    }

    /**
     * Validate file path for security
     */
    private function validate_path($path) {
        // Normalize path
        $path = str_replace('\\', '/', $path);

        // Check for path traversal
        if (strpos($path, '..') !== false) {
            return [
                'valid' => false,
                'error' => 'Path traversal not allowed'
            ];
        }

        // Check blocked paths
        foreach ($this->blocked_paths as $blocked) {
            if (strpos($path, $blocked) === 0) {
                return [
                    'valid' => false,
                    'error' => 'Access to this path is restricted: ' . $blocked
                ];
            }
        }

        // Check if path is within WordPress installation
        $full_path = $this->get_full_path($path);
        $wp_root = realpath(ABSPATH);

        if (strpos(realpath(dirname($full_path)), $wp_root) !== 0 && file_exists($full_path)) {
            return [
                'valid' => false,
                'error' => 'Path must be within WordPress installation'
            ];
        }

        // Check for sensitive files
        $is_sensitive = false;
        foreach ($this->sensitive_files as $sensitive) {
            if (basename($path) === $sensitive) {
                $is_sensitive = true;
                break;
            }
        }

        return [
            'valid' => true,
            'is_sensitive' => $is_sensitive,
            'full_path' => $full_path
        ];
    }

    /**
     * Get full path from relative path
     */
    private function get_full_path($relative_path) {
        // Remove leading slash
        $relative_path = ltrim($relative_path, '/');
        return ABSPATH . $relative_path;
    }

    // ==========================================
    // FILE SYSTEM HANDLERS
    // ==========================================

    /**
     * Read file contents
     */
    public function read_file($request) {
        $path = sanitize_text_field($request->get_param('path'));
        $line_start = intval($request->get_param('line_start'));
        $line_end = intval($request->get_param('line_end'));

        // Validate path
        $validation = $this->validate_path($path);
        if (!$validation['valid']) {
            return new WP_REST_Response([
                'success' => false,
                'error' => $validation['error']
            ], 403);
        }

        $full_path = $validation['full_path'];

        // Check if file exists
        if (!file_exists($full_path)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'File not found: ' . $path
            ], 404);
        }

        // Check if it's a file (not directory)
        if (is_dir($full_path)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Path is a directory, not a file'
            ], 400);
        }

        // Read file content
        $content = file_get_contents($full_path);
        if ($content === false) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Failed to read file'
            ], 500);
        }

        // Handle line range
        if ($line_start > 0 || $line_end > 0) {
            $lines = explode("\n", $content);
            $start = max(0, $line_start - 1);
            $end = $line_end > 0 ? min(count($lines), $line_end) : count($lines);
            $lines = array_slice($lines, $start, $end - $start);
            $content = implode("\n", $lines);
        }

        // Get file info
        $stat = stat($full_path);
        $file_info = [
            'path' => $path,
            'name' => basename($path),
            'type' => 'file',
            'size' => filesize($full_path),
            'modified' => gmdate('c', filemtime($full_path)),
            'permissions' => substr(sprintf('%o', fileperms($full_path)), -4),
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_is_writable
            'is_writable' => is_writable($full_path)
        ];

        return new WP_REST_Response([
            'success' => true,
            'content' => $content,
            'file_info' => $file_info
        ], 200);
    }

    /**
     * Write file contents
     */
    public function write_file($request) {
        $path = sanitize_text_field($request->get_param('path'));
        $content = $request->get_param('content');
        $create_backup = $request->get_param('create_backup') !== false;

        // Validate path
        $validation = $this->validate_path($path);
        if (!$validation['valid']) {
            return new WP_REST_Response([
                'success' => false,
                'error' => $validation['error']
            ], 403);
        }

        $full_path = $validation['full_path'];
        $backup_path = null;

        // Create backup if file exists and backup is requested
        if (file_exists($full_path) && $create_backup) {
            $backup_path = $full_path . '.bak.' . gmdate('YmdHis');
            if (!copy($full_path, $backup_path)) {
                return new WP_REST_Response([
                    'success' => false,
                    'error' => 'Failed to create backup'
                ], 500);
            }
        }

        // Generate diff if file exists
        $diff = null;
        if (file_exists($full_path)) {
            $old_content = file_get_contents($full_path);
            $diff = $this->generate_diff($old_content, $content);
        }

        // Create directory if it doesn't exist
        $dir = dirname($full_path);
        if (!is_dir($dir)) {
            if (!wp_mkdir_p($dir)) {
                return new WP_REST_Response([
                    'success' => false,
                    'error' => 'Failed to create directory'
                ], 500);
            }
        }

        // Write file
        if (file_put_contents($full_path, $content) === false) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Failed to write file'
            ], 500);
        }

        return new WP_REST_Response([
            'success' => true,
            'backup_path' => $backup_path ? str_replace(ABSPATH, '', $backup_path) : null,
            'diff' => $diff
        ], 200);
    }

    /**
     * List directory contents
     */
    public function list_directory($request) {
        $path = sanitize_text_field($request->get_param('path') ?: '/');
        $recursive = (bool) $request->get_param('recursive');
        $pattern = sanitize_text_field($request->get_param('pattern'));
        $max_depth = intval($request->get_param('max_depth') ?: 3);

        // Validate path
        $validation = $this->validate_path($path);
        if (!$validation['valid']) {
            return new WP_REST_Response([
                'success' => false,
                'error' => $validation['error']
            ], 403);
        }

        $full_path = $validation['full_path'];

        // Check if path exists and is a directory
        if (!is_dir($full_path)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Path is not a directory'
            ], 400);
        }

        $entries = $this->scan_directory($full_path, $path, $pattern, $recursive, $max_depth, 0);

        return new WP_REST_Response([
            'success' => true,
            'entries' => $entries
        ], 200);
    }

    /**
     * Recursively scan directory
     */
    private function scan_directory($full_path, $relative_path, $pattern, $recursive, $max_depth, $current_depth) {
        $entries = [];
        $items = scandir($full_path);

        foreach ($items as $item) {
            if ($item === '.' || $item === '..') {
                continue;
            }

            $item_full_path = $full_path . '/' . $item;
            $item_relative_path = ltrim($relative_path . '/' . $item, '/');

            // Apply pattern filter
            if ($pattern && !fnmatch($pattern, $item)) {
                if (!is_dir($item_full_path)) {
                    continue;
                }
            }

            $is_dir = is_dir($item_full_path);

            $entry = [
                'name' => $item,
                'path' => $item_relative_path,
                'type' => $is_dir ? 'directory' : 'file',
                'size' => $is_dir ? 0 : filesize($item_full_path),
                'modified' => gmdate('c', filemtime($item_full_path)),
                'permissions' => substr(sprintf('%o', fileperms($item_full_path)), -4)
            ];

            $entries[] = $entry;

            // Recurse into subdirectories
            if ($is_dir && $recursive && $current_depth < $max_depth) {
                $sub_entries = $this->scan_directory(
                    $item_full_path,
                    $item_relative_path,
                    $pattern,
                    $recursive,
                    $max_depth,
                    $current_depth + 1
                );
                $entries = array_merge($entries, $sub_entries);
            }
        }

        return $entries;
    }

    /**
     * Preview diff before writing
     */
    public function preview_diff($request) {
        $path = sanitize_text_field($request->get_param('path'));
        $new_content = $request->get_param('new_content');

        // Validate path
        $validation = $this->validate_path($path);
        if (!$validation['valid']) {
            return new WP_REST_Response([
                'success' => false,
                'error' => $validation['error']
            ], 403);
        }

        $full_path = $validation['full_path'];

        // Get current content
        $old_content = '';
        if (file_exists($full_path)) {
            $old_content = file_get_contents($full_path);
        }

        $diff = $this->generate_diff($old_content, $new_content);

        return new WP_REST_Response([
            'success' => true,
            'diff' => $diff['diff'],
            'additions' => $diff['additions'],
            'deletions' => $diff['deletions']
        ], 200);
    }

    /**
     * Generate unified diff between two strings
     */
    private function generate_diff($old, $new) {
        $old_lines = explode("\n", $old);
        $new_lines = explode("\n", $new);

        $diff_lines = [];
        $additions = 0;
        $deletions = 0;

        // Simple diff algorithm
        $max_lines = max(count($old_lines), count($new_lines));

        for ($i = 0; $i < $max_lines; $i++) {
            $old_line = isset($old_lines[$i]) ? $old_lines[$i] : null;
            $new_line = isset($new_lines[$i]) ? $new_lines[$i] : null;

            if ($old_line === $new_line) {
                $diff_lines[] = '  ' . ($old_line ?? '');
            } else {
                if ($old_line !== null) {
                    $diff_lines[] = '- ' . $old_line;
                    $deletions++;
                }
                if ($new_line !== null) {
                    $diff_lines[] = '+ ' . $new_line;
                    $additions++;
                }
            }
        }

        return [
            'diff' => implode("\n", $diff_lines),
            'additions' => $additions,
            'deletions' => $deletions
        ];
    }

    /**
     * Search files for content
     */
    public function search_files($request) {
        $query = $request->get_param('query');
        $path = sanitize_text_field($request->get_param('path') ?: '/');
        $pattern = sanitize_text_field($request->get_param('pattern') ?: '*.php');
        $is_regex = (bool) $request->get_param('regex');
        $max_results = intval($request->get_param('max_results') ?: 50);

        // Validate path
        $validation = $this->validate_path($path);
        if (!$validation['valid']) {
            return new WP_REST_Response([
                'success' => false,
                'error' => $validation['error']
            ], 403);
        }

        $full_path = $validation['full_path'];
        $results = [];

        // Get all matching files
        $files = $this->get_files_matching_pattern($full_path, $pattern);

        foreach ($files as $file) {
            if (count($results) >= $max_results) {
                break;
            }

            $content = file_get_contents($file);
            $lines = explode("\n", $content);
            $relative_path = str_replace(ABSPATH, '', $file);

            foreach ($lines as $line_num => $line) {
                if (count($results) >= $max_results) {
                    break;
                }

                $matches = false;
                if ($is_regex) {
                    $matches = @preg_match('/' . $query . '/i', $line);
                } else {
                    $matches = stripos($line, $query) !== false;
                }

                if ($matches) {
                    // Get context (3 lines before and after)
                    $context_start = max(0, $line_num - 3);
                    $context_end = min(count($lines), $line_num + 4);
                    $context_lines = array_slice($lines, $context_start, $context_end - $context_start);

                    $results[] = [
                        'file' => $relative_path,
                        'line' => $line_num + 1,
                        'content' => trim($line),
                        'context' => implode("\n", $context_lines)
                    ];
                }
            }
        }

        return new WP_REST_Response([
            'success' => true,
            'results' => $results,
            'total_matches' => count($results)
        ], 200);
    }

    /**
     * Get files matching a glob pattern
     */
    private function get_files_matching_pattern($dir, $pattern) {
        $files = [];
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $file) {
            if ($file->isFile() && fnmatch($pattern, $file->getFilename())) {
                $files[] = $file->getPathname();
            }
        }

        return $files;
    }

    // ==========================================
    // CODE EXECUTION HANDLERS
    // ==========================================

    /**
     * Execute WP-CLI command
     */
    public function execute_wpcli($request) {
        $command = sanitize_text_field($request->get_param('command'));
        $timeout = intval($request->get_param('timeout') ?: 30);

        // Check if WP-CLI is available
        $wp_cli_path = $this->find_wp_cli();
        if (!$wp_cli_path) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'WP-CLI not found on this server'
            ], 500);
        }

        // Build command (ensure proper escaping)
        $full_command = sprintf(
            'cd %s && %s %s --path=%s 2>&1',
            escapeshellarg(ABSPATH),
            $wp_cli_path,
            $command, // Already sanitized
            escapeshellarg(ABSPATH)
        );

        $start_time = microtime(true);

        // Execute with timeout
        $output = [];
        $return_var = 0;
        exec($full_command, $output, $return_var);

        $execution_time = round((microtime(true) - $start_time) * 1000);

        return new WP_REST_Response([
            'success' => $return_var === 0,
            'output' => implode("\n", $output),
            'exit_code' => $return_var,
            'execution_time' => $execution_time
        ], 200);
    }

    /**
     * Find WP-CLI executable
     */
    private function find_wp_cli() {
        $paths = ['wp', '/usr/local/bin/wp', '/usr/bin/wp', ABSPATH . 'wp-cli.phar'];

        foreach ($paths as $path) {
            if (file_exists($path) || shell_exec("which $path 2>/dev/null")) {
                return $path;
            }
        }

        return null;
    }

    /**
     * Execute PHP code in WordPress context
     */
    public function execute_php($request) {
        $code = $request->get_param('code');
        $sandbox = $request->get_param('sandbox') !== false;
        $timeout = intval($request->get_param('timeout') ?: 30);

        // Security: Check for dangerous functions
        if ($sandbox) {
            $dangerous_functions = [
                'exec', 'shell_exec', 'system', 'passthru', 'popen',
                'proc_open', 'pcntl_exec', 'eval', 'assert', 'create_function',
                'file_put_contents', 'fwrite', 'unlink', 'rmdir', 'chmod'
            ];

            foreach ($dangerous_functions as $func) {
                if (preg_match('/\b' . preg_quote($func, '/') . '\s*\(/i', $code)) {
                    return new WP_REST_Response([
                        'success' => false,
                        'error' => "Function '$func' is not allowed in sandbox mode"
                    ], 403);
                }
            }
        }

        $start_time = microtime(true);
        $memory_before = memory_get_usage();

        // Capture output
        ob_start();
        $return_value = null;
        $error = null;
        $error_type = null;

        try {
            // Set timeout
            // phpcs:ignore Squiz.PHP.DiscouragedFunctions.Discouraged
            set_time_limit($timeout);

            // Execute the code (intentional for agent code execution capability)
            // phpcs:ignore Generic.PHP.ForbiddenFunctions.Found
            $return_value = eval($code);
        } catch (ParseError $e) {
            $error = $e->getMessage();
            $error_type = 'syntax';
        } catch (Error $e) {
            $error = $e->getMessage();
            $error_type = 'runtime';
        } catch (Exception $e) {
            $error = $e->getMessage();
            $error_type = 'runtime';
        }

        $output = ob_get_clean();
        $execution_time = round((microtime(true) - $start_time) * 1000);
        $memory_usage = memory_get_usage() - $memory_before;

        $response = [
            'success' => $error === null,
            'output' => $output,
            'execution_time' => $execution_time,
            'memory_usage' => $memory_usage
        ];

        if ($return_value !== null) {
            $response['return_value'] = $return_value;
        }

        if ($error) {
            $response['error'] = $error;
            $response['error_type'] = $error_type;
        }

        return new WP_REST_Response($response, 200);
    }

    /**
     * Execute database query
     */
    public function execute_db_query($request) {
        global $wpdb;

        $query = $request->get_param('query');
        $type = sanitize_text_field($request->get_param('type'));
        $params = $request->get_param('params') ?: [];
        $dry_run = $request->get_param('dry_run') !== false;

        // Validate query type
        $allowed_types = ['select', 'insert', 'update', 'delete', 'describe'];
        if (!in_array($type, $allowed_types)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Invalid query type. Allowed: ' . implode(', ', $allowed_types)
            ], 400);
        }

        // Security: Block dangerous operations
        $blocked_patterns = [
            '/DROP\s+DATABASE/i',
            '/DROP\s+TABLE/i',
            '/TRUNCATE/i',
            '/ALTER\s+DATABASE/i',
            '/CREATE\s+DATABASE/i'
        ];

        foreach ($blocked_patterns as $pattern) {
            if (preg_match($pattern, $query)) {
                return new WP_REST_Response([
                    'success' => false,
                    'error' => 'This operation is not allowed for security reasons'
                ], 403);
            }
        }

        $start_time = microtime(true);

        // Prepare query with parameters if provided
        if (!empty($params)) {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query is dynamically provided by agent with user-supplied params
            $query = $wpdb->prepare($query, $params);
        }

        // Handle dry run for UPDATE/DELETE
        if ($dry_run && in_array($type, ['update', 'delete'])) {
            // Convert to SELECT to show affected rows
            // Note: where_clause is extracted from the already-prepared query, so it's safe
            // phpcs:disable WordPress.DB.PreparedSQL.NotPrepared
            // phpcs:disable PluginCheck.Security.DirectDB.UnescapedDBParameter
            if ($type === 'update') {
                // Extract WHERE clause and do a count
                if (preg_match('/WHERE\s+(.+)$/i', $query, $matches)) {
                    $where_clause = $matches[1];
                    preg_match('/UPDATE\s+([^\s]+)/i', $query, $table_match);
                    $table = $table_match[1] ?? '';
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $affected = $wpdb->get_var("SELECT COUNT(*) FROM " . esc_sql($table) . " WHERE " . $where_clause);
                }
            } else if ($type === 'delete') {
                preg_match('/FROM\s+([^\s]+)/i', $query, $table_match);
                $table = $table_match[1] ?? '';
                if (preg_match('/WHERE\s+(.+)$/i', $query, $matches)) {
                    $where_clause = $matches[1];
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $affected = $wpdb->get_var("SELECT COUNT(*) FROM " . esc_sql($table) . " WHERE " . $where_clause);
                }
            }
            // phpcs:enable WordPress.DB.PreparedSQL.NotPrepared
            // phpcs:enable PluginCheck.Security.DirectDB.UnescapedDBParameter

            return new WP_REST_Response([
                'success' => true,
                'data' => null,
                'affected_rows' => $affected ?? 0,
                'execution_time' => round((microtime(true) - $start_time) * 1000),
                'dry_run' => true,
                'message' => 'Dry run - no changes made. Would affect ' . ($affected ?? 0) . ' rows.'
            ], 200);
        }

        // Execute query
        // Note: $query is either the original prepared query or has been run through $wpdb->prepare with params
        // Security is enforced by: 1) validating query type, 2) blocking dangerous patterns, 3) API key auth
        $result = null;
        $affected_rows = 0;
        $error = null;

        // phpcs:disable WordPress.DB.PreparedSQL.NotPrepared
        // phpcs:disable PluginCheck.Security.DirectDB.UnescapedDBParameter
        try {
            switch ($type) {
                case 'select':
                case 'describe':
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $result = $wpdb->get_results($query, ARRAY_A);
                    break;
                case 'insert':
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $wpdb->query($query);
                    $result = ['insert_id' => $wpdb->insert_id];
                    $affected_rows = $wpdb->rows_affected;
                    break;
                case 'update':
                case 'delete':
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $wpdb->query($query);
                    $affected_rows = $wpdb->rows_affected;
                    break;
            }

            if ($wpdb->last_error) {
                $error = $wpdb->last_error;
            }
        } catch (Exception $e) {
            $error = $e->getMessage();
        }
        // phpcs:enable WordPress.DB.PreparedSQL.NotPrepared
        // phpcs:enable PluginCheck.Security.DirectDB.UnescapedDBParameter

        $execution_time = round((microtime(true) - $start_time) * 1000);

        return new WP_REST_Response([
            'success' => $error === null,
            'data' => $result,
            'affected_rows' => $affected_rows,
            'execution_time' => $execution_time,
            'error' => $error
        ], $error ? 500 : 200);
    }

    /**
     * Execute shell command (restricted)
     */
    public function execute_shell($request) {
        $command = $request->get_param('command');
        $working_directory = sanitize_text_field($request->get_param('working_directory'));
        $timeout = intval($request->get_param('timeout') ?: 30);

        // Security: Check if command is in allowed list
        $command_base = explode(' ', $command)[0];
        $is_allowed = false;

        foreach ($this->allowed_shell_commands as $allowed) {
            if (strpos($command, $allowed) === 0) {
                $is_allowed = true;
                break;
            }
        }

        if (!$is_allowed) {
            return new WP_REST_Response([
                'success' => false,
                'error' => "Command '$command_base' is not in the allowed list. Allowed: " .
                          implode(', ', array_slice($this->allowed_shell_commands, 0, 5)) . '...'
            ], 403);
        }

        // Security: Block dangerous operators
        $dangerous_operators = ['&&', '||', ';', '|', '>', '<', '`', '$(', '${'];
        foreach ($dangerous_operators as $op) {
            if (strpos($command, $op) !== false) {
                return new WP_REST_Response([
                    'success' => false,
                    'error' => "Shell operator '$op' is not allowed"
                ], 403);
            }
        }

        $start_time = microtime(true);

        // Set working directory
        $cwd = ABSPATH;
        if ($working_directory) {
            $validation = $this->validate_path($working_directory);
            if ($validation['valid']) {
                $cwd = $validation['full_path'];
            }
        }

        // Execute command (intentional for agent shell execution capability)
        $descriptorspec = [
            0 => ['pipe', 'r'],
            1 => ['pipe', 'w'],
            2 => ['pipe', 'w']
        ];

        // phpcs:ignore Generic.PHP.ForbiddenFunctions.Found
        $process = proc_open($command, $descriptorspec, $pipes, $cwd);

        if (!is_resource($process)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Failed to execute command'
            ], 500);
        }

        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
        fclose($pipes[0]);

        $stdout = stream_get_contents($pipes[1]);
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
        fclose($pipes[1]);

        $stderr = stream_get_contents($pipes[2]);
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
        fclose($pipes[2]);

        $exit_code = proc_close($process);
        $execution_time = round((microtime(true) - $start_time) * 1000);

        return new WP_REST_Response([
            'success' => $exit_code === 0,
            'stdout' => $stdout,
            'stderr' => $stderr,
            'exit_code' => $exit_code,
            'execution_time' => $execution_time
        ], 200);
    }

    // ==========================================
    // CODEBASE INDEXING HANDLERS
    // ==========================================

    /**
     * Index codebase for search
     */
    public function index_codebase($request) {
        $paths = $request->get_param('paths') ?: [];
        $types = $request->get_param('types') ?: ['theme', 'plugin'];
        $force_refresh = (bool) $request->get_param('force_refresh');

        $start_time = microtime(true);
        $files = [];

        // Determine which directories to index
        $dirs_to_index = [];

        if (in_array('theme', $types)) {
            $dirs_to_index[] = get_stylesheet_directory();
            if (get_template_directory() !== get_stylesheet_directory()) {
                $dirs_to_index[] = get_template_directory();
            }
        }

        if (in_array('plugin', $types)) {
            $dirs_to_index[] = WP_PLUGIN_DIR;
        }

        if (in_array('mu-plugin', $types) && defined('WPMU_PLUGIN_DIR')) {
            $dirs_to_index[] = WPMU_PLUGIN_DIR;
        }

        // Add custom paths
        foreach ($paths as $path) {
            $validation = $this->validate_path($path);
            if ($validation['valid']) {
                $dirs_to_index[] = $validation['full_path'];
            }
        }

        // Index PHP files
        foreach ($dirs_to_index as $dir) {
            if (!is_dir($dir)) continue;

            $php_files = $this->get_files_matching_pattern($dir, '*.php');

            foreach ($php_files as $file) {
                $relative_path = str_replace(ABSPATH, '', $file);
                $content = file_get_contents($file);

                // Extract symbols
                $symbols = $this->extract_php_symbols($content);

                $files[] = [
                    'path' => $relative_path,
                    'type' => pathinfo($file, PATHINFO_EXTENSION),
                    'hash' => md5($content),
                    'functions' => $symbols['functions'],
                    'classes' => $symbols['classes'],
                    'hooks' => $symbols['hooks'],
                    'variables' => $symbols['variables']
                ];
            }
        }

        $processing_time = round((microtime(true) - $start_time) * 1000);

        return new WP_REST_Response([
            'success' => true,
            'indexed_files' => count($files),
            'processing_time' => $processing_time,
            'files' => $files
        ], 200);
    }

    /**
     * Extract PHP symbols from content
     */
    private function extract_php_symbols($content) {
        $symbols = [
            'functions' => [],
            'classes' => [],
            'hooks' => [],
            'variables' => []
        ];

        // Extract functions
        preg_match_all('/function\s+([a-zA-Z_][a-zA-Z0-9_]*)\s*\(([^)]*)\)/m', $content, $func_matches, PREG_OFFSET_CAPTURE);
        foreach ($func_matches[1] as $i => $match) {
            $line = substr_count(substr($content, 0, $match[1]), "\n") + 1;
            $params = array_map('trim', explode(',', $func_matches[2][$i][0]));
            $symbols['functions'][] = [
                'name' => $match[0],
                'line' => $line,
                'params' => array_filter($params)
            ];
        }

        // Extract classes
        preg_match_all('/class\s+([a-zA-Z_][a-zA-Z0-9_]*)/m', $content, $class_matches, PREG_OFFSET_CAPTURE);
        foreach ($class_matches[1] as $match) {
            $line = substr_count(substr($content, 0, $match[1]), "\n") + 1;
            $symbols['classes'][] = [
                'name' => $match[0],
                'line' => $line,
                'methods' => []
            ];
        }

        // Extract WordPress hooks (actions and filters)
        preg_match_all('/(?:add_action|add_filter)\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $hook_matches, PREG_OFFSET_CAPTURE);
        foreach ($hook_matches[1] as $i => $match) {
            $line = substr_count(substr($content, 0, $match[1]), "\n") + 1;
            $type = strpos($hook_matches[0][$i][0], 'add_action') !== false ? 'action' : 'filter';
            $symbols['hooks'][] = [
                'name' => $match[0],
                'type' => $type,
                'line' => $line
            ];
        }

        return $symbols;
    }

    /**
     * Search codebase for symbols
     */
    public function search_codebase($request) {
        $query = $request->get_param('query');
        $type = sanitize_text_field($request->get_param('type') ?: 'pattern');
        $scope = $request->get_param('scope') ?: [];
        $limit = intval($request->get_param('limit') ?: 10);

        // For now, this performs a basic file search
        // In a full implementation, this would search the indexed symbols
        $search_request = new WP_REST_Request('POST');
        $search_request->set_param('query', $query);
        $search_request->set_param('pattern', '*.php');
        $search_request->set_param('max_results', $limit);

        return $this->search_files($search_request);
    }
}

// Initialize the agent endpoints
new InstaRank_Agent_Endpoints();
