<?php
namespace owthub\inspqu;

use owthub\inspqu\INSPQU_App_Helper;
use owthub\inspqu\INSPQU_Query_Helper;
use owthub\inspqu\public\INSPQU_Public;

/**
 * WordPress Widget for InspirePulse Quotes
 * 
 * @link       https://onlinewebtutorblog.com
 * @since      1.6
 * @package    InspirePulse_Quotes
 * @subpackage InspirePulse_Quotes/includes
 * @copyright  Copyright (c) 2025, Sanjay Kumar
 * @license    GPL-2.0+ https://www.gnu.org/licenses/gpl-2.0.html
 * @author     Online Web Tutor
 */
if ( ! defined( 'ABSPATH' ) ) exit;

class INSPQU_Widget extends \WP_Widget {

    private $public_instance;

    public function __construct() {
        parent::__construct(
            'inspqu_widget',
            esc_html__( 'InspirePulse Quotes', 'inspirepulse-quotes' ),
            array(
                'description' => esc_html__( 'Display inspirational quotes with customizable design and settings.', 'inspirepulse-quotes' ),
            )
        );

        // Get public instance for rendering
        $this->public_instance = new INSPQU_Public( INSPQU_TEXT_DOMAIN, INSPQU_VERSION );
    }

    /**
     * Front-end display of widget
     */
    public function widget( $args, $instance ) {
        echo $args['before_widget'];

        if ( ! empty( $instance['title'] ) ) {
            echo $args['before_title'] . apply_filters( 'widget_title', $instance['title'] ) . $args['after_title'];
        }

        // Build shortcode attributes from widget settings
        $shortcode_atts = array();

        if ( ! empty( $instance['design'] ) ) {
            $shortcode_atts[] = 'design="' . esc_attr( $instance['design'] ) . '"';
        }

        if ( ! empty( $instance['timer'] ) ) {
            $shortcode_atts[] = 'timer="' . absint( $instance['timer'] ) . '"';
        }

        if ( ! empty( $instance['category'] ) ) {
            $shortcode_atts[] = 'category="' . esc_attr( $instance['category'] ) . '"';
        }

        if ( ! empty( $instance['author'] ) ) {
            $shortcode_atts[] = 'author="' . esc_attr( $instance['author'] ) . '"';
        }

        if ( ! empty( $instance['limit'] ) ) {
            $shortcode_atts[] = 'limit="' . absint( $instance['limit'] ) . '"';
        }

        if ( ! empty( $instance['orderby'] ) ) {
            $shortcode_atts[] = 'orderby="' . esc_attr( $instance['orderby'] ) . '"';
        }

        if ( ! empty( $instance['order'] ) ) {
            $shortcode_atts[] = 'order="' . esc_attr( $instance['order'] ) . '"';
        }

        // Build shortcode
        $shortcode = '[inspqu';
        if ( ! empty( $shortcode_atts ) ) {
            $shortcode .= ' ' . implode( ' ', $shortcode_atts );
        }
        $shortcode .= ']';

        // Render shortcode
        echo do_shortcode( $shortcode );

        echo $args['after_widget'];
    }

    /**
     * Back-end widget form
     */
    public function form( $instance ) {
        $title = ! empty( $instance['title'] ) ? $instance['title'] : '';
        $design = ! empty( $instance['design'] ) ? $instance['design'] : 'default';
        $timer = ! empty( $instance['timer'] ) ? $instance['timer'] : '';
        $category = ! empty( $instance['category'] ) ? $instance['category'] : '';
        $author = ! empty( $instance['author'] ) ? $instance['author'] : '';
        $limit = ! empty( $instance['limit'] ) ? $instance['limit'] : '';
        $orderby = ! empty( $instance['orderby'] ) ? $instance['orderby'] : 'random';
        $order = ! empty( $instance['order'] ) ? $instance['order'] : 'ASC';

        $designs = array(
            'default' => esc_html__( 'Elegant Dark (Default)', 'inspirepulse-quotes' ),
            'elegant dark' => esc_html__( 'Elegant Dark', 'inspirepulse-quotes' ),
            'gradient glow box' => esc_html__( 'Gradient Glow Box', 'inspirepulse-quotes' ),
            'paper vintage look' => esc_html__( 'Paper Vintage Look', 'inspirepulse-quotes' ),
            'floating glass overlay' => esc_html__( 'Floating Glass Overlay', 'inspirepulse-quotes' ),
            'big center text + corner' => esc_html__( 'Big Center Text + Corner', 'inspirepulse-quotes' ),
        );

        // Get categories
        $categories = INSPQU_Query_Helper::inspqu_get_data( 'categories', array( 'status' => 1 ), 'name ASC' );
        ?>
        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>">
                <?php esc_html_e( 'Title:', 'inspirepulse-quotes' ); ?>
            </label>
            <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" 
                   name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" 
                   type="text" value="<?php echo esc_attr( $title ); ?>">
        </p>

        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'design' ) ); ?>">
                <?php esc_html_e( 'Design:', 'inspirepulse-quotes' ); ?>
            </label>
            <select class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'design' ) ); ?>" 
                    name="<?php echo esc_attr( $this->get_field_name( 'design' ) ); ?>">
                <?php foreach ( $designs as $key => $label ) : ?>
                    <option value="<?php echo esc_attr( $key ); ?>" <?php selected( $design, $key ); ?>>
                        <?php echo esc_html( $label ); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </p>

        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'timer' ) ); ?>">
                <?php esc_html_e( 'Timer (seconds):', 'inspirepulse-quotes' ); ?>
            </label>
            <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'timer' ) ); ?>" 
                   name="<?php echo esc_attr( $this->get_field_name( 'timer' ) ); ?>" 
                   type="number" min="1" value="<?php echo esc_attr( $timer ); ?>" 
                   placeholder="<?php esc_attr_e( 'Leave empty for global setting', 'inspirepulse-quotes' ); ?>">
        </p>

        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'category' ) ); ?>">
                <?php esc_html_e( 'Category (name or ID, comma-separated):', 'inspirepulse-quotes' ); ?>
            </label>
            <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'category' ) ); ?>" 
                   name="<?php echo esc_attr( $this->get_field_name( 'category' ) ); ?>" 
                   type="text" value="<?php echo esc_attr( $category ); ?>" 
                   placeholder="<?php esc_attr_e( 'e.g., Motivation, Success', 'inspirepulse-quotes' ); ?>">
            <?php if ( ! empty( $categories ) ) : ?>
                <small>
                    <?php esc_html_e( 'Available:', 'inspirepulse-quotes' ); ?>
                    <?php echo esc_html( implode( ', ', array_column( $categories, 'name' ) ) ); ?>
                </small>
            <?php endif; ?>
        </p>

        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'author' ) ); ?>">
                <?php esc_html_e( 'Author (comma-separated):', 'inspirepulse-quotes' ); ?>
            </label>
            <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'author' ) ); ?>" 
                   name="<?php echo esc_attr( $this->get_field_name( 'author' ) ); ?>" 
                   type="text" value="<?php echo esc_attr( $author ); ?>" 
                   placeholder="<?php esc_attr_e( 'e.g., Einstein, Steve Jobs', 'inspirepulse-quotes' ); ?>">
        </p>

        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'limit' ) ); ?>">
                <?php esc_html_e( 'Limit:', 'inspirepulse-quotes' ); ?>
            </label>
            <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'limit' ) ); ?>" 
                   name="<?php echo esc_attr( $this->get_field_name( 'limit' ) ); ?>" 
                   type="number" min="1" value="<?php echo esc_attr( $limit ); ?>" 
                   placeholder="<?php esc_attr_e( 'Leave empty for all', 'inspirepulse-quotes' ); ?>">
        </p>

        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'orderby' ) ); ?>">
                <?php esc_html_e( 'Order By:', 'inspirepulse-quotes' ); ?>
            </label>
            <select class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'orderby' ) ); ?>" 
                    name="<?php echo esc_attr( $this->get_field_name( 'orderby' ) ); ?>">
                <option value="random" <?php selected( $orderby, 'random' ); ?>>
                    <?php esc_html_e( 'Random', 'inspirepulse-quotes' ); ?>
                </option>
                <option value="date" <?php selected( $orderby, 'date' ); ?>>
                    <?php esc_html_e( 'Date', 'inspirepulse-quotes' ); ?>
                </option>
                <option value="author" <?php selected( $orderby, 'author' ); ?>>
                    <?php esc_html_e( 'Author', 'inspirepulse-quotes' ); ?>
                </option>
                <option value="id" <?php selected( $orderby, 'id' ); ?>>
                    <?php esc_html_e( 'ID', 'inspirepulse-quotes' ); ?>
                </option>
            </select>
        </p>

        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'order' ) ); ?>">
                <?php esc_html_e( 'Order:', 'inspirepulse-quotes' ); ?>
            </label>
            <select class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'order' ) ); ?>" 
                    name="<?php echo esc_attr( $this->get_field_name( 'order' ) ); ?>">
                <option value="ASC" <?php selected( $order, 'ASC' ); ?>>
                    <?php esc_html_e( 'Ascending', 'inspirepulse-quotes' ); ?>
                </option>
                <option value="DESC" <?php selected( $order, 'DESC' ); ?>>
                    <?php esc_html_e( 'Descending', 'inspirepulse-quotes' ); ?>
                </option>
            </select>
        </p>
        <?php
    }

    /**
     * Sanitize widget form values as they are saved
     */
    public function update( $new_instance, $old_instance ) {
        $instance = array();
        $instance['title'] = ! empty( $new_instance['title'] ) ? sanitize_text_field( $new_instance['title'] ) : '';
        $instance['design'] = ! empty( $new_instance['design'] ) ? sanitize_text_field( $new_instance['design'] ) : 'default';
        $instance['timer'] = ! empty( $new_instance['timer'] ) ? absint( $new_instance['timer'] ) : '';
        $instance['category'] = ! empty( $new_instance['category'] ) ? sanitize_text_field( $new_instance['category'] ) : '';
        $instance['author'] = ! empty( $new_instance['author'] ) ? sanitize_text_field( $new_instance['author'] ) : '';
        $instance['limit'] = ! empty( $new_instance['limit'] ) ? absint( $new_instance['limit'] ) : '';
        $instance['orderby'] = ! empty( $new_instance['orderby'] ) ? sanitize_text_field( $new_instance['orderby'] ) : 'random';
        $instance['order'] = ! empty( $new_instance['order'] ) ? strtoupper( sanitize_text_field( $new_instance['order'] ) ) : 'ASC';

        return $instance;
    }
}
