<?php 
namespace owthub\inspqu;
/**
 * @link       https://onlinewebtutorblog.com
 * @since      1.0
 * @package    InspirePulse_Quotes
 * @subpackage InspirePulse_Quotes/includes
 * @copyright  Copyright (c) 2025, Sanjay Kumar
 * @license    GPL-2.0+ https://www.gnu.org/licenses/gpl-2.0.html
 * @author     Online Web Tutor
 */
if ( ! defined( 'ABSPATH' ) ) exit;

class INSPQU_Query_Helper {

    public static function inspqu_get_data( $table_key, $where = [], $order_by = '', $limit = 0, $exclude_ids = [] ) {

        global $wpdb;

        $table_name = esc_sql(
            INSPQU_App_Helper::inspqu_get_table_name_by_key( $table_key )
        );

        $sql = "SELECT * FROM {$table_name}";

        $allowed_columns = [
            'id',
            'quote_text',
            'quote_author',
            'category_id',
            'status',
            'created_at',
            'updated_at',
            'name',
        ];

        if ( ! empty( $where ) ) {

            $conditions = [];

            foreach ( $where as $column => $value ) {
                if ( ! in_array( $column, $allowed_columns, true ) ) {
                    continue;
                }

                $conditions[] = $wpdb->prepare(
                    "{$column} = %s",
                    $value
                );
            }

            if ( $conditions ) {
                $sql .= ' WHERE ' . implode( ' AND ', $conditions );
            }
        }

        // Exclude specific IDs
        if ( ! empty( $exclude_ids ) && is_array( $exclude_ids ) ) {
            $exclude_ids = array_map( 'absint', $exclude_ids );
            $exclude_ids = array_filter( $exclude_ids );
            if ( ! empty( $exclude_ids ) ) {
                $exclude_placeholders = implode( ',', array_fill( 0, count( $exclude_ids ), '%d' ) );
                $exclude_condition = $wpdb->prepare( "id NOT IN ({$exclude_placeholders})", $exclude_ids );
                if ( strpos( $sql, 'WHERE' ) !== false ) {
                    $sql .= ' AND ' . $exclude_condition;
                } else {
                    $sql .= ' WHERE ' . $exclude_condition;
                }
            }
        }

        if ( 'RAND()' === $order_by ) {

            $sql .= ' ORDER BY RAND()';

        } elseif ( ! empty( $order_by ) ) {

            $order_by_map = [
                'id ASC'          => 'id ASC',
                'id DESC'         => 'id DESC',
                'date ASC'        => 'created_at ASC',
                'date DESC'       => 'created_at DESC',
                'author ASC'      => 'quote_author ASC',
                'author DESC'     => 'quote_author DESC',
            ];

            if ( isset( $order_by_map[ $order_by ] ) ) {
                $sql .= ' ORDER BY ' . $order_by_map[ $order_by ];
            }
        }

        // Add limit
        if ( $limit > 0 ) {
            $limit = absint( $limit );
            $sql .= $wpdb->prepare( ' LIMIT %d', $limit );
        }

        return $wpdb->get_results( $sql, ARRAY_A );
    }

    /**
     * Get quotes with category and author filtering
     * 
     * @param array $args Query arguments
     * @return array Quotes with category information
     */
    public static function inspqu_get_quotes_with_filters( $args = [] ) {
        global $wpdb;

        $defaults = [
            'category'    => '',
            'author'       => '',
            'limit'        => 0,
            'orderby'      => 'random',
            'order'        => 'ASC',
            'id'           => 0,
            'exclude'      => [],
            'status'       => 1,
        ];

        $args = wp_parse_args( $args, $defaults );

        $quotes_table = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key( 'quotes' ) );
        $categories_table = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key( 'categories' ) );
        $rel_table = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key( 'quote_category_relation' ) );

        // Build SQL query
        $sql = "SELECT q.*, c.name as category_name, c.id as category_id 
                FROM {$quotes_table} q
                LEFT JOIN {$rel_table} r ON q.id = r.quote_id
                LEFT JOIN {$categories_table} c ON r.category_id = c.id
                WHERE 1=1";

        $where_conditions = [];
        $where_values = [];

        // Filter by specific quote ID
        if ( ! empty( $args['id'] ) ) {
            $where_conditions[] = "q.id = %d";
            $where_values[] = absint( $args['id'] );
        }

        // Filter by category
        if ( ! empty( $args['category'] ) ) {
            $category_names = is_array( $args['category'] ) ? $args['category'] : explode( ',', $args['category'] );
            $category_names = array_map( 'trim', $category_names );
            $category_names = array_filter( $category_names );
            
            if ( ! empty( $category_names ) ) {
                // Check if any are numeric (category IDs)
                $category_ids = [];
                $category_name_list = [];
                
                foreach ( $category_names as $cat ) {
                    if ( is_numeric( $cat ) ) {
                        $category_ids[] = absint( $cat );
                    } else {
                        $category_name_list[] = $cat;
                    }
                }
                
                $cat_conditions = [];
                if ( ! empty( $category_ids ) ) {
                    $placeholders = implode( ',', array_fill( 0, count( $category_ids ), '%d' ) );
                    $cat_conditions[] = "c.id IN ({$placeholders})";
                    $where_values = array_merge( $where_values, $category_ids );
                }
                
                if ( ! empty( $category_name_list ) ) {
                    $placeholders = implode( ',', array_fill( 0, count( $category_name_list ), '%s' ) );
                    $cat_conditions[] = "c.name IN ({$placeholders})";
                    $where_values = array_merge( $where_values, $category_name_list );
                }
                
                if ( ! empty( $cat_conditions ) ) {
                    $where_conditions[] = '(' . implode( ' OR ', $cat_conditions ) . ')';
                }
            }
        }

        // Filter by author
        if ( ! empty( $args['author'] ) ) {
            $authors = is_array( $args['author'] ) ? $args['author'] : explode( ',', $args['author'] );
            $authors = array_map( 'trim', $authors );
            $authors = array_filter( $authors );
            
            if ( ! empty( $authors ) ) {
                $placeholders = implode( ',', array_fill( 0, count( $authors ), '%s' ) );
                $where_conditions[] = "q.quote_author IN ({$placeholders})";
                $where_values = array_merge( $where_values, $authors );
            }
        }

        // Filter by status
        if ( isset( $args['status'] ) && $args['status'] !== '' ) {
            if ( $args['status'] === 'all' ) {
                // Don't filter by status
            } else {
                $where_conditions[] = "q.status = %d";
                $where_values[] = absint( $args['status'] );
            }
        } else {
            // Default to active only
            $where_conditions[] = "q.status = %d";
            $where_values[] = 1;
        }

        // Exclude specific quote IDs
        if ( ! empty( $args['exclude'] ) ) {
            $exclude_ids = is_array( $args['exclude'] ) ? $args['exclude'] : explode( ',', $args['exclude'] );
            $exclude_ids = array_map( 'absint', $exclude_ids );
            $exclude_ids = array_filter( $exclude_ids );
            
            if ( ! empty( $exclude_ids ) ) {
                $placeholders = implode( ',', array_fill( 0, count( $exclude_ids ), '%d' ) );
                $where_conditions[] = "q.id NOT IN ({$placeholders})";
                $where_values = array_merge( $where_values, $exclude_ids );
            }
        }

        // Add WHERE conditions
        if ( ! empty( $where_conditions ) ) {
            $prepared_sql = $wpdb->prepare( implode( ' AND ', $where_conditions ), $where_values );
            $sql .= ' AND ' . str_replace( '%', '%%', $prepared_sql );
        }

        // Order by
        if ( $args['orderby'] === 'random' ) {
            $sql .= ' ORDER BY RAND()';
        } else {
            $order_by_map = [
                'id'     => 'q.id',
                'date'   => 'q.created_at',
                'author' => 'q.quote_author',
            ];
            
            $order_column = isset( $order_by_map[ $args['orderby'] ] ) ? $order_by_map[ $args['orderby'] ] : 'q.id';
            $order_direction = strtoupper( $args['order'] ) === 'DESC' ? 'DESC' : 'ASC';
            $sql .= " ORDER BY {$order_column} {$order_direction}";
        }

        // Limit
        if ( $args['limit'] > 0 ) {
            $limit = absint( $args['limit'] );
            $sql .= $wpdb->prepare( ' LIMIT %d', $limit );
        }

        $results = $wpdb->get_results( $sql, ARRAY_A );

        // Remove duplicates (quotes can have multiple categories)
        if ( ! empty( $results ) ) {
            $unique_quotes = [];
            $seen_ids = [];
            
            foreach ( $results as $quote ) {
                if ( ! in_array( $quote['id'], $seen_ids ) ) {
                    $unique_quotes[] = $quote;
                    $seen_ids[] = $quote['id'];
                }
            }
            
            return $unique_quotes;
        }

        return $results;
    }

    public static function inspqu_get_row( $table_key, $where = [], $order_by = '' ) {

        global $wpdb;

        $table_name = esc_sql(
            INSPQU_App_Helper::inspqu_get_table_name_by_key( $table_key )
        );

        $sql = "SELECT * FROM {$table_name}";

        $allowed_columns = [
            'id',
            'quote_text',
            'quote_author',
            'quote_id',
            'category_id',
            'status',
            'created_at',
            'updated_at',
            'name',
        ];

        if ( ! empty( $where ) ) {

            $conditions = [];

            foreach ( $where as $column => $value ) {
                if ( ! in_array( $column, $allowed_columns, true ) ) {
                    continue;
                }

                $conditions[] = $wpdb->prepare(
                    "{$column} = %s",
                    $value
                );
            }

            if ( $conditions ) {
                $sql .= ' WHERE ' . implode( ' AND ', $conditions );
            }
        }

        if ( ! empty( $order_by ) ) {

            $order_by_map = [
                'id ASC'          => 'id ASC',
                'id DESC'         => 'id DESC',
            ];

            if ( isset( $order_by_map[ $order_by ] ) ) {
                $sql .= ' ORDER BY ' . $order_by_map[ $order_by ];
            }
        }

        $sql .= ' LIMIT 1';

        return $wpdb->get_row( $sql, ARRAY_A );
    }

    public static function inspqu_quotes() {
        global $wpdb;

        $quotes_table      = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key('quotes') );
        $categories_table  = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key('categories') );
        $rel_table         = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key('quote_category_relation') );

        try{

            $quotes_data = INSPQU_App_Helper::inspqu_get_default_quotes();

            if (!isset($quotes_data['quotes']) || !is_array($quotes_data['quotes'])) {
                error_log("IPQ: No quotes found in JSON");
                return;
            }

            $quotes = $quotes_data['quotes'];
            $category_ids = [];

            // Range trackers
            $first_cat_id = null;
            $last_cat_id  = null;
            $first_quote_id = null;
            $last_quote_id  = null;
            $first_rel_quote_id = null;
            $last_rel_quote_id  = null;

            foreach ($quotes as $quote) {

                $quote_text     = !empty($quote['quote_text']) ? sanitize_text_field($quote['quote_text']) : '';
                $quote_author   = !empty($quote['quote_author']) ? sanitize_text_field($quote['quote_author']) : '';
                $category_name  = !empty($quote['quote_category']) ? sanitize_text_field($quote['quote_category']) : '';

                if (empty($quote_text) || empty($category_name)) {
                    continue;
                }

                if (!isset($category_ids[$category_name])) {

                    $existing_cat_id = $wpdb->get_var(
                        $wpdb->prepare("SELECT id FROM $categories_table WHERE name = %s", $category_name)
                    );

                    if ($existing_cat_id) {
                        $category_ids[$category_name] = (int) $existing_cat_id;
                    } else {
                        $wpdb->insert(
                            $categories_table,
                            [
                                'name'   => $category_name,
                                'slug'   => sanitize_title($category_name),
                                'status' => 1,
                                'created_at' => current_time('mysql'),
                                'updated_at' => current_time('mysql'),
                            ],
                            ['%s', '%s', '%d','%s','%s']
                        );

                        $inserted_cat_id = $wpdb->insert_id;
                        $category_ids[$category_name] = (int) $inserted_cat_id;

                        if (!$first_cat_id) $first_cat_id = $inserted_cat_id;
                        $last_cat_id = $inserted_cat_id;
                    }
                }

                $cat_id = $category_ids[$category_name];

                $wpdb->insert(
                    $quotes_table,
                    [
                        'quote_text'   => $quote_text,
                        'quote_author' => $quote_author,
                        'status'       => 1,
                        'created_at'   => current_time('mysql'),
                        'updated_at'   => current_time('mysql'),
                    ],
                    ['%s', '%s', '%d','%s','%s']
                );

                $quote_id = $wpdb->insert_id;

                if (!$quote_id) continue;

                if (!$first_quote_id) $first_quote_id = $quote_id;
                $last_quote_id = $quote_id;

                $wpdb->insert(
                    $rel_table,
                    [
                        'quote_id'    => $quote_id,
                        'category_id' => $cat_id
                    ],
                    ['%d', '%d']
                );

                if (!$first_rel_quote_id) $first_rel_quote_id = $quote_id;
                $last_rel_quote_id = $quote_id;
            }

            $import_ranges = [
                'categories_range'   => [ 'start_id' => $first_cat_id, 'end_id' => $last_cat_id ],
                'cat_quotes_range'   => [ 'start_id' => $first_rel_quote_id, 'end_id' => $last_rel_quote_id ],
                'quotes_range'       => [ 'start_id' => $first_quote_id, 'end_id' => $last_quote_id ]
            ];

            update_option('inspqu_import_ranges', $import_ranges);

            return true;
            
        } catch(Exception $ex) {

            return false;
        }
    }
}