<?php 
namespace owthub\inspqu;
/**
 * @link       https://onlinewebtutorblog.com
 * @since      1.0
 * @package    InspirePulse_Quotes
 * @subpackage InspirePulse_Quotes/includes
 * @copyright  Copyright (c) 2025, Sanjay Kumar
 * @license    GPL-2.0+ https://www.gnu.org/licenses/gpl-2.0.html
 * @author     Online Web Tutor
 */
if ( ! defined( 'ABSPATH' ) ) exit;
 
class INSPQU_App_Helper {

    public static $activator;

    public static function inspqu_load_activator() {
        if (!self::$activator) {
            self::$activator = new INSPQU_Activator();
        }
    }

    public static function inspqu_db_tables() {

        self::inspqu_load_activator();

        return [
            'quotes'                    => self::$activator->inspqu_quotes_table(),
            'categories'                => self::$activator->inspqu_categories_table(),
            'backups'                   => self::$activator->inspqu_backups_table(),
            'quote_category_relation'   => self::$activator->inspqu_quotes_categories_relation_table(),
        ];
    }

    public static function inspqu_get_table_name_by_key($key) {
        $tables = self::inspqu_db_tables();
        return $tables[$key] ?? null;
    }

    public static function inspqu_load_admin_page( $file, $args = [] ) {

        $path = INSPQU_PATH . 'admin/partials/' . $file . '.php';

        if ( file_exists( $path ) ) {
            if ( ! empty( $args ) && is_array( $args ) ) {
                extract( $args, EXTR_SKIP );
            }
            include $path;
        } else {
            echo wp_kses_post(
                sprintf(
                    '<div class="notice notice-error"><p><strong>%s</strong> %s</p></div>',
                    esc_html__( 'Admin Page Not Found:', 'inspirepulse-quotes' ),
                    esc_html( $file . '.php' )
                )
            );
        }
    }

    public static function inspqu_get_default_quotes() {

        $quotes_path = INSPQU_PATH . 'data/inspqu-quotes.json';

        if ( file_exists( $quotes_path ) ) {
            $json = file_get_contents( $quotes_path );
            $quotes = json_decode( $json, true );
            if ( is_array( $quotes ) ) {
                return $quotes;
            }
        }
        return [];
    }

    // Returns plural value
    public static function inspqu_pluralize($word) {
        $lower = strtolower($word);
    
        $irregular = [
            'person' => 'people',
            'man'    => 'men',
            'woman'  => 'women',
            'child'  => 'children',
            'tooth'  => 'teeth',
            'foot'   => 'feet',
            'mouse'  => 'mice',
            'goose'  => 'geese'
        ];

        $irregular_plurals = array_flip($irregular);
    
        if (isset($irregular_plurals[$lower])) {
            return $word; // already plural
        }

        if (isset($irregular[$lower])) {
            return $irregular[$lower];
        }

        if (preg_match('/[^aeiou]ies$/i', $word)) {
            return $word;
        }
    
        if (preg_match('/(ches|shes|xes|zes)$/i', $word)) {
            return $word; // already plural
        }
    
        if (preg_match('/[^s]s$/i', $word)) {
            return $word; // already plural
        }
  
        if (preg_match('/[^aeiou]y$/i', $word)) {
            return preg_replace('/y$/i', 'ies', $word);
        }
        if (preg_match('/(s|sh|ch|x|z)$/i', $word)) {
            return $word . 'es';
        }
    
        return $word . 's';
    }      
    
    // Returns singular value
    public static function inspqu_singularize($word) {
        $lower = strtolower($word);

        $irregular = [
            'people'    => 'person',
            'men'       => 'man',
            'women'     => 'woman',
            'children'  => 'child',
            'teeth'     => 'tooth',
            'feet'      => 'foot',
            'mice'      => 'mouse',
            'geese'     => 'goose'
        ];
        if (isset($irregular[$lower])) {
            return $irregular[$lower];
        }
    
        if (preg_match('/[^aeiou]ies$/i', $word)) {
            return preg_replace('/ies$/i', 'y', $word);
        }
    
        if (preg_match('/(s|sh|ch|x|z)es$/i', $word)) {
            return preg_replace('/es$/i', '', $word);
        }

        if (preg_match('/[^s]s$/i', $word)) {
            return substr($word, 0, -1);
        }

        return $word;
    } 

    /**
     * Get base upload dir and url
    */
    private static function inspqu_get_base() {
        return wp_upload_dir();
    }

    /**
     * Database Backup Directory Path
    */
    public static function inspqu_get_db_backup_base_dir() {
        $upload_dir = self::inspqu_get_base();
        return $upload_dir['basedir'] . '/' . INSPQU_UPLOADS_FOLDER . INSPQU_DB_BKP_FOLDER;
    }

    /**
     * Database Backup Directory URL
    */
    public static function inspqu_get_db_backup_base_url() {
        $upload_dir = self::inspqu_get_base();
        return $upload_dir['baseurl'] . '/' . INSPQU_UPLOADS_FOLDER . INSPQU_DB_BKP_FOLDER;
    }

    public static function inspqu_get_db_backup_file_prefix(){
        return INSPQU_BKP_FILE_NAME_PREFIX;
    }

    public static function inspqu_export_tables_schama_with_data( $tables, $filePath ) {

        global $wpdb;

        $sql = '';

        foreach ( $tables as $table ) {

            if ( strpos( $table, $wpdb->prefix ) !== 0 ) {
                continue;
            }

            $table_name = esc_sql( $table );

            $create = $wpdb->get_row(
                "SHOW CREATE TABLE `{$table_name}`",
                ARRAY_N
            );

            if ( empty( $create[1] ) ) {
                continue;
            }

            $sql .= "DROP TABLE IF EXISTS `{$table_name}`;\n";
            $sql .= $create[1] . ";\n\n";

            $rows = $wpdb->get_results(
                "SELECT * FROM `{$table_name}`",
                ARRAY_A
            );

            if ( empty( $rows ) ) {
                continue;
            }

            foreach ( $rows as $row ) {

                $columns = array_map(
                    static function ( $col ) {
                        return '`' . esc_sql( $col ) . '`';
                    },
                    array_keys( $row )
                );

                $values = array_map(
                    static function ( $val ) {
                        if ( is_null( $val ) ) {
                            return 'NULL';
                        }
                        return "'" . esc_sql( $val ) . "'";
                    },
                    array_values( $row )
                );

                $sql .= "INSERT INTO `{$table_name}` (" .
                    implode( ',', $columns ) .
                    ') VALUES (' .
                    implode( ',', $values ) .
                    ");\n";
            }

            $sql .= "\n\n";
        }

        file_put_contents( $filePath, $sql );
    }

    /**
     * Import SQL backup file
     * 
     * @param string $filePath Path to SQL file
     * @return array Result array with status and message
     */
    public static function inspqu_import_sql_backup( $filePath ) {

        global $wpdb;

        if ( ! file_exists( $filePath ) ) {
            return [
                'sts' => 0,
                'msg' => esc_html__( 'Backup file not found.', 'inspirepulse-quotes' )
            ];
        }

        // Check file size (max 10MB)
        $file_size = filesize( $filePath );
        if ( $file_size > 10 * 1024 * 1024 ) {
            return [
                'sts' => 0,
                'msg' => esc_html__( 'File size exceeds 10MB limit.', 'inspirepulse-quotes' )
            ];
        }

        // Read SQL file
        $sql_content = file_get_contents( $filePath );
        if ( false === $sql_content ) {
            return [
                'sts' => 0,
                'msg' => esc_html__( 'Failed to read backup file.', 'inspirepulse-quotes' )
            ];
        }

        // Get allowed tables
        $allowed_tables = self::inspqu_db_tables();
        $allowed_table_names = array_values( $allowed_tables );
        
        // Exclude backups table from import
        $backup_table = self::inspqu_get_table_name_by_key( 'backups' );
        $allowed_table_names = array_filter( $allowed_table_names, function( $table ) use ( $backup_table ) {
            return $table !== $backup_table;
        } );

        // Start transaction
        $wpdb->query( 'START TRANSACTION' );

        try {
            // Split SQL into individual statements
            $statements = self::inspqu_split_sql_statements( $sql_content );

            $imported_tables = [];
            $errors = [];

            foreach ( $statements as $statement ) {
                $statement = trim( $statement );
                
                if ( empty( $statement ) || preg_match( '/^--/', $statement ) ) {
                    continue;
                }

                // Check if statement affects allowed tables
                $is_allowed = false;
                foreach ( $allowed_table_names as $allowed_table ) {
                    if ( stripos( $statement, $allowed_table ) !== false ) {
                        $is_allowed = true;
                        break;
                    }
                }

                if ( ! $is_allowed ) {
                    continue;
                }

                // Execute statement
                $result = $wpdb->query( $statement );
                
                if ( false === $result ) {
                    $errors[] = $wpdb->last_error;
                } else {
                    // Extract table name from statement
                    if ( preg_match( '/CREATE TABLE.*?`?([a-z0-9_]+)`?/i', $statement, $matches ) ) {
                        $imported_tables[] = $matches[1];
                    } elseif ( preg_match( '/INSERT INTO.*?`?([a-z0-9_]+)`?/i', $statement, $matches ) ) {
                        $imported_tables[] = $matches[1];
                    }
                }
            }

            if ( ! empty( $errors ) ) {
                $wpdb->query( 'ROLLBACK' );
                return [
                    'sts' => 0,
                    'msg' => sprintf(
                        /* translators: %s: error messages */
                        esc_html__( 'Import failed: %s', 'inspirepulse-quotes' ),
                        implode( ', ', array_unique( $errors ) )
                    )
                ];
            }

            // Clear cache
            wp_cache_flush();

            // Commit transaction
            $wpdb->query( 'COMMIT' );

            return [
                'sts' => 1,
                'msg' => sprintf(
                    /* translators: %d: number of tables imported */
                    esc_html__( 'Backup imported successfully! %d table(s) restored.', 'inspirepulse-quotes' ),
                    count( array_unique( $imported_tables ) )
                )
            ];

        } catch ( Exception $e ) {
            $wpdb->query( 'ROLLBACK' );
            return [
                'sts' => 0,
                'msg' => sprintf(
                    /* translators: %s: error message */
                    esc_html__( 'Import error: %s', 'inspirepulse-quotes' ),
                    $e->getMessage()
                )
            ];
        }
    }

    /**
     * Split SQL content into individual statements
     * 
     * @param string $sql SQL content
     * @return array Array of SQL statements
     */
    private static function inspqu_split_sql_statements( $sql ) {
        // Remove comments
        $sql = preg_replace( '/--.*$/m', '', $sql );
        $sql = preg_replace( '/\/\*.*?\*\//s', '', $sql );

        // Split by semicolon, but preserve semicolons inside quotes
        $statements = [];
        $current = '';
        $in_quotes = false;
        $quote_char = '';

        for ( $i = 0; $i < strlen( $sql ); $i++ ) {
            $char = $sql[ $i ];
            $prev_char = $i > 0 ? $sql[ $i - 1 ] : '';

            if ( ! $in_quotes && ( $char === '"' || $char === "'" || $char === '`' ) ) {
                $in_quotes = true;
                $quote_char = $char;
            } elseif ( $in_quotes && $char === $quote_char && $prev_char !== '\\' ) {
                $in_quotes = false;
                $quote_char = '';
            }

            $current .= $char;

            if ( ! $in_quotes && $char === ';' ) {
                $statements[] = trim( $current );
                $current = '';
            }
        }

        if ( ! empty( trim( $current ) ) ) {
            $statements[] = trim( $current );
        }

        return array_filter( $statements );
    }

    public static function inspqu_has_backup_data(){

        $data = INSPQU_Query_Helper::inspqu_get_data('backups');

        if(count($data) > 0){
            return true;
        }
        return false;
    }

    public static function inspqu_remove_submenu_pages_link(){
        return [
            'inspqu-submenu-edit-quote',
            'inspqu-submenu-show-backups',
            'inspqu-submenu-shortcodes'
        ];
    }

    public static function inspqu_get_quote_templates() {
        return [
            [
                'template'  => 'data/templates/quote-1-template.png',
                'shortcode' => '[inspqu design="Elegant Dark"]'
            ],
            [
                'template'  => 'data/templates/quote-2-template.png',
                'shortcode' => '[inspqu design="Gradient Glow Box"]'
            ],
            [
                'template'  => 'data/templates/quote-3-template.png',
                'shortcode' => '[inspqu design="Paper Vintage Look"]'
            ],
            [
                'template'  => 'data/templates/quote-4-template.png',
                'shortcode' => '[inspqu design="Floating Glass Overlay"]'
            ],
            [
                'template'  => 'data/templates/quote-5-template.png',
                'shortcode' => '[inspqu design="Big Center Text + Corner"]'
            ]
        ];
    }

    public static function inspqu_get_allowed_quote_templates( $design_key = '' ) {

        if( $design_key == 'default' ){
            return 'quote-1';
        }

        $templates = [
            'elegant dark' => 'quote-1',
            'gradient glow box' => 'quote-2',
            'paper vintage look' => 'quote-3',
            'floating glass overlay' => 'quote-4',
            'big center text + corner' => 'quote-5'
        ];

        if (!empty($design_key)){
            return $templates[$design_key];
        }

        return $templates;
    }

    public static function inspqu_get_allowed_quote_templates_keys() {
        
        return array_keys( self::inspqu_get_allowed_quote_templates() );
    }

    public static function get_inspqu_url() {
        return INSPQU_URL;
    }

    public static function get_inspqu_path(){
        return INSPQU_PATH;
    }

    public static function inspqu_get_default_timer_value() {
        return INSPQU_DELAY_TIMER;
    }

    public static function inspqu_get_text_domain() {
        return INSPQU_TEXT_DOMAIN;
    }

    public static function get_inspqu_version() {
        return INSPQU_VERSION;
    }

    public static function inspqu_options() {
        return [
            'inspqu_timer_miliseconds',
            'inspqu_import_ranges'
        ];
    }

    public static function inspqu_allowed_ajax_actions() {

        return [ 
            'inspqu_add_quote', 
            'inspqu_add_category', 
            'inspqu_delete_data_action', 
            'inspqu_update_category',
            'inspqu_update_quote',
            'inspqu_install_default_quotes',
            'inspqu_remove_default_quotes',
            'inspqu_generate_db_backup',
            'inspqu_save_timer_seconds',
            'inspqu_import_backup'
        ];
    }

    public static function inspqu_allowed_admin_pages() {

        return [
            'toplevel_page_inspqu-dashboard',
			'inspirepulse-quotes_page_inspqu-all-quotes',
			'inspirepulse-quotes_page_inspqu-add-quote',
			'inspirepulse-quotes_page_inspqu-categories',
			'inspirepulse-quotes_page_inspqu-settings',
			'inspirepulse-quotes_page_inspqu-about',
			'inspirepulse-quotes_page_inspqu-edit-quote',
            'inspirepulse-quotes_page_inspqu-shortcodes',
            'inspirepulse-quotes_page_inspqu-show-backups',
        ];
    }
}
