<?php
namespace owthub\inspqu;

use owthub\inspqu\INSPQU_App_Helper;
use owthub\inspqu\INSPQU_Query_Helper;

/**
 * Gutenberg Block for InspirePulse Quotes
 *
 * @link       https://onlinewebtutorblog.com
 * @since      1.7
 * @package    InspirePulse_Quotes
 * @subpackage InspirePulse_Quotes/includes
 * @copyright  Copyright (c) 2025, Sanjay Kumar
 * @license    GPL-2.0+ https://www.gnu.org/licenses/gpl-2.0.html
 * @author     Online Web Tutor
 */
if ( ! defined( 'ABSPATH' ) ) exit;

class INSPQU_Gutenberg_Block {

    /**
     * Register block
     */
    public static function register() {
        if ( ! function_exists( 'register_block_type' ) ) {
            return;
        }

        // Register block script
        wp_register_script(
            'inspqu-gutenberg-block',
            INSPQU_URL . 'admin/js/inspqu-gutenberg-block.js',
            array( 'wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-i18n' ),
            INSPQU_VERSION,
            true
        );

        // Register block style
        wp_register_style(
            'inspqu-gutenberg-block-editor',
            INSPQU_URL . 'admin/css/inspqu-gutenberg-block.css',
            array( 'wp-edit-blocks' ),
            INSPQU_VERSION
        );

        // Localize script with data
        $categories = INSPQU_Query_Helper::inspqu_get_data( 'categories', array( 'status' => 1 ), 'name ASC' );
        $designs = array(
            array( 'value' => 'default', 'label' => esc_html__( 'Elegant Dark (Default)', 'inspirepulse-quotes' ) ),
            array( 'value' => 'elegant dark', 'label' => esc_html__( 'Elegant Dark', 'inspirepulse-quotes' ) ),
            array( 'value' => 'gradient glow box', 'label' => esc_html__( 'Gradient Glow Box', 'inspirepulse-quotes' ) ),
            array( 'value' => 'paper vintage look', 'label' => esc_html__( 'Paper Vintage Look', 'inspirepulse-quotes' ) ),
            array( 'value' => 'floating glass overlay', 'label' => esc_html__( 'Floating Glass Overlay', 'inspirepulse-quotes' ) ),
            array( 'value' => 'big center text + corner', 'label' => esc_html__( 'Big Center Text + Corner', 'inspirepulse-quotes' ) ),
        );

        $default_timer = intval( get_option( 'inspqu_timer_miliseconds' ) );
        if ( $default_timer <= 0 ) {
            $default_timer = INSPQU_App_Helper::inspqu_get_default_timer_value();
        }
        $default_timer = $default_timer / 1000; // Convert to seconds

        wp_localize_script( 'inspqu-gutenberg-block', 'inspquBlockData', array(
            'categories' => $categories,
            'designs' => $designs,
            'defaultTimer' => $default_timer,
        ) );

        // Register block type
        register_block_type( 'inspqu/quotes', array(
            'editor_script' => 'inspqu-gutenberg-block',
            'editor_style' => 'inspqu-gutenberg-block-editor',
            'render_callback' => array( __CLASS__, 'render_block' ),
            'attributes' => array(
                'design' => array(
                    'type' => 'string',
                    'default' => 'default',
                ),
                'timer' => array(
                    'type' => 'number',
                    'default' => 0,
                ),
                'category' => array(
                    'type' => 'string',
                    'default' => '',
                ),
                'author' => array(
                    'type' => 'string',
                    'default' => '',
                ),
                'limit' => array(
                    'type' => 'number',
                    'default' => 0,
                ),
                'orderby' => array(
                    'type' => 'string',
                    'default' => 'random',
                ),
                'order' => array(
                    'type' => 'string',
                    'default' => 'ASC',
                ),
                'showSharing' => array(
                    'type' => 'boolean',
                    'default' => false,
                ),
            ),
        ) );
    }

    /**
     * Render block on frontend
     */
    public static function render_block( $attributes ) {
        $design = ! empty( $attributes['design'] ) ? $attributes['design'] : 'default';
        $timer = ! empty( $attributes['timer'] ) ? absint( $attributes['timer'] ) : '';
        $category = ! empty( $attributes['category'] ) ? $attributes['category'] : '';
        $author = ! empty( $attributes['author'] ) ? $attributes['author'] : '';
        $limit = ! empty( $attributes['limit'] ) ? absint( $attributes['limit'] ) : '';
        $orderby = ! empty( $attributes['orderby'] ) ? $attributes['orderby'] : 'random';
        $order = ! empty( $attributes['order'] ) ? $attributes['order'] : 'ASC';
        $show_sharing = ! empty( $attributes['showSharing'] ) ? 'true' : 'false';

        // Build shortcode
        $shortcode = '[inspqu';
        if ( $design !== 'default' ) {
            $shortcode .= ' design="' . esc_attr( $design ) . '"';
        }
        if ( ! empty( $timer ) ) {
            $shortcode .= ' timer="' . absint( $timer ) . '"';
        }
        if ( ! empty( $category ) ) {
            $shortcode .= ' category="' . esc_attr( $category ) . '"';
        }
        if ( ! empty( $author ) ) {
            $shortcode .= ' author="' . esc_attr( $author ) . '"';
        }
        if ( ! empty( $limit ) ) {
            $shortcode .= ' limit="' . absint( $limit ) . '"';
        }
        if ( $orderby !== 'random' ) {
            $shortcode .= ' orderby="' . esc_attr( $orderby ) . '"';
            $shortcode .= ' order="' . esc_attr( $order ) . '"';
        }
        if ( $show_sharing === 'true' ) {
            $shortcode .= ' show_sharing="true"';
        }
        $shortcode .= ']';

        return do_shortcode( $shortcode );
    }
}

