<?php 
namespace owthub\inspqu;
/**
 * @link       https://onlinewebtutorblog.com
 * @since      1.0
 * @package    InspirePulse_Quotes
 * @subpackage InspirePulse_Quotes/includes
 * @copyright  Copyright (c) 2025, Sanjay Kumar
 * @license    GPL-2.0+ https://www.gnu.org/licenses/gpl-2.0.html
 * @author     Online Web Tutor
 */
if ( ! defined( 'ABSPATH' ) ) exit;

class INSPQU_Ajax {
    
    public function inspqu_handler() {

        check_ajax_referer( 'inspqu_actions', 'inspqu_nonce', true );

        if (!current_user_can('manage_options')) {
            wp_send_json( array('sts' => 0, 'msg' => 'Insufficient permissions.') );
        }

        $param = isset($_REQUEST['param']) ? sanitize_text_field( wp_unslash( $_REQUEST['param'] ) ) : '';
        
        $allowed_params = INSPQU_App_Helper::inspqu_allowed_ajax_actions();

        if ( ! in_array( $param, $allowed_params ) ) {

            wp_send_json([
                'sts' => 0,
                'msg' => 'Invalid Action',
            ]);
        }

        global $wpdb;

        if ( $param === 'inspqu_add_quote' ) {

            $quotes_table = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key( 'quotes' ) );
            $rel_table    = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key( 'quote_category_relation' ) );

            // Sanitize Inputs
            $quote_text     = isset( $_POST['quote_text'] ) ? wp_kses_post( wp_unslash( $_POST['quote_text'] ) ) : '';
            $quote_author   = isset( $_POST['quote_author'] ) ? sanitize_text_field( wp_unslash( $_POST['quote_author'] ) ) : '';
            $quote_category = isset( $_POST['quote_category'] ) ? absint( wp_unslash( $_POST['quote_category'] ) ) : 0;

            if ( empty( $quote_text ) ) {
                wp_send_json( [ 'sts' => 0, 'msg' => 'Quote text is required' ] );
            }

            if ( $quote_category <= 0 ) {
                wp_send_json( [ 'sts' => 0, 'msg' => 'Please select a category' ] );
            }

            $insert_quote = $wpdb->insert(
                $quotes_table,
                [
                    'quote_text'   => $quote_text,
                    'quote_author' => $quote_author,
                    'status'       => 1,
                    'created_at'   => current_time( 'mysql' ),
                    'updated_at'   => current_time( 'mysql' ),
                ],
                [ '%s', '%s', '%d', '%s', '%s' ]
            );

            if ( ! $insert_quote ) {
                wp_send_json( [ 'sts' => 0, 'msg' => 'Failed to save quote. Please try again.' ] );
            }

            $quote_id = $wpdb->insert_id;

            $insert_rel = $wpdb->insert(
                $rel_table,
                [
                    'quote_id'    => $quote_id,
                    'category_id' => $quote_category,
                ],
                [ '%d', '%d' ]
            );

            if ( ! $insert_rel ) {
                wp_send_json( [ 'sts' => 0, 'msg' => 'Quote saved, but category relation failed.' ] );
            }

            // Clear Cache by Category
            wp_cache_delete(
                'inspqu_categories_with_counts',
                'inspqu_categories'
            );

            wp_cache_delete(
                'inspqu_quotes_' . absint( $quote_category ),
                'inspqu_quotes'
            );

            wp_cache_delete(
                'inspqu_rel_quote_' . absint( $quote_id ),
                'inspqu_relations'
            );

            wp_send_json(
                [
                    'sts' => 1,
                    'msg' => 'Quote Added Successfully!',
                ]
            );

        } elseif ( $param === 'inspqu_add_category' ) {

            $categories_table = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key( 'categories' ) );

            // Sanitize Inputs
            $category_name   = isset( $_POST['category_name'] ) ? sanitize_text_field( wp_unslash ( $_POST['category_name'] ) ) : '';
            $category_status = isset( $_POST['status'] ) ? absint( $_POST['status'] ) : 1;

            if ( empty( $category_name ) ) {
                wp_send_json( [ 'sts' => 0, 'msg' => 'Category name is required' ] );
            }

            $slug = sanitize_title( $category_name );

            $table = esc_sql( $categories_table );

            $has_slug = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM {$table} WHERE slug = %s",
                    $slug
                )
            );

            if ( 0 === $has_slug ) {

                $insert_category = $wpdb->insert(
                    $categories_table,
                    [
                        'name'       => $category_name,
                        'slug'       => $slug,
                        'status'     => $category_status,
                        'created_at' => current_time( 'mysql' ),
                        'updated_at' => current_time( 'mysql' ),
                    ],
                    [ '%s', '%s', '%d', '%s', '%s' ]
                );

                if ( ! $insert_category ) {
                    wp_send_json( [ 'sts' => 0, 'msg' => 'Failed to Add Category.' ] );
                }

                wp_send_json(
                    [
                        'sts' => 1,
                        'msg' => 'Category Added Successfully!',
                    ]
                );

            } else {

                wp_send_json(
                    [
                        'sts' => 0,
                        'msg' => 'Category Already Exists',
                    ]
                );
            }
        } else if ( $param === "inspqu_delete_data_action" ) {

            $rowId  = isset($_POST['rowId']) ? absint($_POST['rowId']) : 0;
            $module = isset($_POST['module']) ? sanitize_text_field( wp_unslash ($_POST['module'] ) ) : '';

            if ( ! $rowId || empty( $module ) ) {
                wp_send_json([ 'sts' => 0, 'msg' => 'Invalid Request!' ]);
            }

            $table_name = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key( $module ) );

            if ( ! $table_name ) {
                wp_send_json([ 'sts' => 0, 'msg' => 'Invalid module name!' ]);
            }

            if ( $module === 'backups' ) {
                $backup_folderDir = INSPQU_App_Helper::inspqu_get_db_backup_base_dir();
                $file_data        = INSPQU_Query_Helper::inspqu_get_row( $module, ['id' => $rowId] );
                $file_name        = $file_data['filename'] ?? '';

                if ( ! empty( $file_name ) ) {
                    $backup_filepath = $backup_folderDir . '/' . $file_name;

                    if ( file_exists( $backup_filepath ) ) {
                        $backup_filepath = wp_normalize_path( $backup_filepath );
                        wp_delete_file( $backup_filepath );
                    }
                }
            }

            $deleted = $wpdb->delete(
                $table_name,
                [ 'id' => $rowId ],
                [ '%d' ]
            );

            if ( $deleted === false ) {
                wp_send_json([ 'sts' => 0, 'msg' => 'Database error while deleting data!' ]);
            }

            $module_singular = INSPQU_App_Helper::inspqu_singularize( $module );
            $module_translated = '';
            
            // Translate common module names
            switch ( $module_singular ) {
                case 'quote':
                    $module_translated = esc_html__( 'Quote', 'inspirepulse-quotes' );
                    break;
                case 'category':
                    $module_translated = esc_html__( 'Category', 'inspirepulse-quotes' );
                    break;
                case 'backup':
                    $module_translated = esc_html__( 'Backup', 'inspirepulse-quotes' );
                    break;
                default:
                    $module_translated = ucfirst( $module_singular );
            }
            
            wp_send_json([
                'sts'        => 1,
                'msg'        => sprintf(
                    /* translators: %s: module name (quote, category, backup, etc.) */
                    esc_html__( '%s deleted successfully!', 'inspirepulse-quotes' ),
                    $module_translated
                ),
                'deleted_id' => $rowId,
            ]);

        } else if ( $param === "inspqu_update_category" ) {

            $categories_table = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key('categories') );

            $category_id     = isset($_POST['id']) ? absint($_POST['id']) : 0;
            $category_name   = isset($_POST['category_name']) ? sanitize_text_field( wp_unslash( $_POST['category_name'] ) ) : '';
            $category_status = isset($_POST['status']) ? absint($_POST['status']) : 1;

            if ( $category_id <= 0 ) {
                wp_send_json([ 'sts' => 0, 'msg' => 'Invalid Category ID.' ]);
            }

            if ( empty( $category_name ) ) {
                wp_send_json([ 'sts' => 0, 'msg' => 'Category name is required' ]);
            }

            $slug = sanitize_title( $category_name );

            $table = esc_sql( $categories_table );

            $has_slug = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM {$table} WHERE slug = %s AND id != %d",
                    $slug,
                    $category_id
                )
            );

            if ( $has_slug > 0 ) {
                wp_send_json([ 'sts' => 0, 'msg' => 'Category slug already exists, choose another name' ]);
            }

            $update = $wpdb->update(
                $categories_table,
                [
                    'name'       => $category_name,
                    'slug'       => $slug,
                    'status'     => $category_status,
                    'updated_at' => current_time('mysql'),
                ],
                [ 'id' => $category_id ],
                [ '%s', '%s', '%d', '%s' ],
                [ '%d' ]
            );

            if ( $update === false ) {
                wp_send_json([ 'sts' => 0, 'msg' => 'Failed to Update Category.' ]);
            }

            wp_send_json([ 'sts' => 1, 'msg' => 'Category Updated Successfully!' ]);

        } else if ( $param === "inspqu_update_quote" ) {

            $quotes_table = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key('quotes') );
            $rel_table    = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key('quote_category_relation') );

            $quote_id      = isset($_POST['id']) ? absint($_POST['id']) : 0;
            $quote_text    = isset($_POST['quote_text']) ? wp_kses_post( wp_unslash( $_POST['quote_text'] ) ) : '';
            $quote_author  = isset($_POST['quote_author']) ? sanitize_text_field( wp_unslash( $_POST['quote_author'] ) ) : '';
            $quote_category = isset($_POST['quote_category']) ? absint($_POST['quote_category']) : 0;
            $quote_status  = isset($_POST['status']) ? absint($_POST['status']) : 1;

            if ( $quote_id <= 0 ) {
                wp_send_json([ 'sts' => 0, 'msg' => 'Invalid Quote ID.' ]);
            }

            if ( empty( $quote_text ) ) {
                wp_send_json([ 'sts' => 0, 'msg' => 'Quote text is required.' ]);
            }

            if ( $quote_category <= 0 ) {
                wp_send_json([ 'sts' => 0, 'msg' => 'Please select a category.' ]);
            }

            $update_quote = $wpdb->update(
                $quotes_table,
                [
                    'quote_text'   => $quote_text,
                    'quote_author' => $quote_author,
                    'status'       => $quote_status,
                    'updated_at'   => current_time('mysql'),
                ],
                [ 'id' => $quote_id ],
                [ '%s', '%s', '%d', '%s' ],
                [ '%d' ]
            );

            if ( $update_quote === false ) {
                wp_send_json([ 'sts' => 0, 'msg' => 'Failed to update quote.' ]);
            }

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $wpdb->delete( $rel_table, [ 'quote_id' => $quote_id ], [ '%d' ] );

            $insert_rel = $wpdb->insert(
                $rel_table,
                [
                    'quote_id'    => $quote_id,
                    'category_id' => $quote_category,
                ],
                [ '%d', '%d' ]
            );

            if ( ! $insert_rel ) {
                wp_send_json([ 'sts' => 0, 'msg' => 'Quote updated, but category relation failed.' ]);
            }

            wp_send_json([ 'sts' => 1, 'msg' => 'Quote Updated Successfully!' ]);

        } else if ( $param === "inspqu_install_default_quotes" ) {

            $data_exists = isset($_POST['data_exists']) ? absint($_POST['data_exists']) : 0;

            if ( $data_exists === 1 ) {

                $rel_table   = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key('quote_category_relation') );
                $cat_table   = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key('categories') );
                $quote_table = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key('quotes') );

                $ranges = get_option('inspqu_import_ranges');

                if (
                    ! empty( $ranges['quotes_range']['start_id'] ) &&
                    ! empty( $ranges['quotes_range']['end_id'] )
                ) {
                    $q_start = absint( $ranges['quotes_range']['start_id'] );
                    $q_end   = absint( $ranges['quotes_range']['end_id'] );

                    $table = esc_sql( $quote_table );

                    $wpdb->query(
                        $wpdb->prepare(
                            "DELETE FROM {$table} WHERE id BETWEEN %d AND %d",
                            $q_start,
                            $q_end
                        )
                    );
                }

                if (
                    ! empty( $ranges['categories_range']['start_id'] ) &&
                    ! empty( $ranges['categories_range']['end_id'] )
                ) {
                    $c_start = absint( $ranges['categories_range']['start_id'] );
                    $c_end   = absint( $ranges['categories_range']['end_id'] );

                    $table = esc_sql( $cat_table );

                    $wpdb->query(
                        $wpdb->prepare(
                            "DELETE FROM {$table} WHERE id BETWEEN %d AND %d",
                            $c_start,
                            $c_end
                        )
                    );
                }

                if (
                    ! empty( $ranges['cat_quotes_range']['start_id'] ) &&
                    ! empty( $ranges['cat_quotes_range']['end_id'] )
                ) {
                    $r_start = absint( $ranges['cat_quotes_range']['start_id'] );
                    $r_end   = absint( $ranges['cat_quotes_range']['end_id'] );

                    $table = esc_sql( $rel_table );

                    $wpdb->query(
                        $wpdb->prepare(
                            "DELETE FROM {$table} WHERE quote_id BETWEEN %d AND %d",
                            $r_start,
                            $r_end
                        )
                    );
                }

                delete_option('inspqu_import_ranges');
            }

            $output = INSPQU_Query_Helper::inspqu_quotes();

            if ( $output ) {
                wp_send_json([ 'sts' => 1, 'msg' => 'Quote Imported Successfully!' ]);
            } else {
                wp_send_json([ 'sts' => 0, 'msg' => 'Failed to Import Quote' ]);
            }
        } else if ( $param === "inspqu_remove_default_quotes" ) {

            $ranges = get_option( 'inspqu_import_ranges' );

            if ( empty( $ranges ) || empty( $ranges['quotes_range']['start_id'] ) ) {
                wp_send_json([
                    'sts' => 0,
                    'msg' => 'No default quote range found in database.'
                ]);
            }

            $rel_table   = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key( 'quote_category_relation' ) );
            $cat_table   = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key( 'categories' ) );
            $quote_table = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key( 'quotes' ) );

            $q_start   = absint( $ranges['quotes_range']['start_id'] );
            $q_end     = absint( $ranges['quotes_range']['end_id'] );
            $c_start   = absint( $ranges['categories_range']['start_id'] );
            $c_end     = absint( $ranges['categories_range']['end_id'] );
            $rel_start = absint( $ranges['cat_quotes_range']['start_id'] );
            $rel_end   = absint( $ranges['cat_quotes_range']['end_id'] );

            // Start transaction
            $wpdb->query( "START TRANSACTION" );

            $wpdb->query(
                $wpdb->prepare(
                    "DELETE FROM {$rel_table} WHERE quote_id BETWEEN %d AND %d",
                    $rel_start,
                    $rel_end
                )
            );

            $del_quotes = $wpdb->query(
                $wpdb->prepare(
                    "DELETE FROM {$quote_table} WHERE id BETWEEN %d AND %d",
                    $q_start,
                    $q_end
                )
            );

            $wpdb->query(
                $wpdb->prepare(
                    "DELETE FROM {$cat_table} WHERE id BETWEEN %d AND %d",
                    $c_start,
                    $c_end
                )
            );

            if ( $del_quotes !== false ) {
                $wpdb->query( "COMMIT" );
                delete_option( 'inspqu_import_ranges' );

                wp_send_json([
                    'sts' => 1,
                    'msg' => 'Default quotes removed successfully from database!'
                ]);
            } else {
                $wpdb->query( "ROLLBACK" );

                wp_send_json([
                    'sts' => 0,
                    'msg' => 'Failed to remove default quotes.'
                ]);
            }

        } else if ( $param === "inspqu_generate_db_backup" ) {

            try {

                $version_number       = time();
                $backup_file_prefix   = INSPQU_App_Helper::inspqu_get_db_backup_file_prefix();
                $backup_folderDir     = INSPQU_App_Helper::inspqu_get_db_backup_base_dir();
                $backup_folderURL     = INSPQU_App_Helper::inspqu_get_db_backup_base_url();

                if ( ! is_dir( $backup_folderDir ) ) {
                    wp_mkdir_p( $backup_folderDir, 0777, true );
                }

                $fileVersionName = $backup_file_prefix . '-' . $version_number . ".sql";
                $bkpFilePath     = $backup_folderDir . $fileVersionName;
                $bkpFileURL      = $backup_folderURL . $fileVersionName;

                $dbTablesArray = INSPQU_App_Helper::inspqu_db_tables();

                // Exclude backups table
                $excludeTables = [
                    esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key('backups') )
                ];

                // Collect all tables should be exported
                $dbTablesArray = array_filter( $dbTablesArray, fn( $table ) => ! in_array( $table, $excludeTables, true ) );

                // Generate Backup
                INSPQU_App_Helper::inspqu_export_tables_schama_with_data( $dbTablesArray, $bkpFilePath );

                $fileSizeBytes = filesize( $bkpFilePath );
                $fileSizeKB    = round( $fileSizeBytes / 1024, 2 );

                $backupTable = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key('backups') );

                // Clear previous latest
                $wpdb->update(
                    $backupTable,
                    [ 'flag' => null ],
                    [ 'flag' => 'latest' ],
                    [ '%s' ]
                );

                // Insert new backup
                $wpdb->insert(
                    $backupTable,
                    [
                        'filename'   => $fileVersionName,
                        'filesize'   => $fileSizeKB . ' KB',
                        'flag'       => 'latest',
                        'filepath'   => $bkpFileURL,
                        'type'       => 'export',
                        'modules'    => maybe_serialize( $dbTablesArray ),
                        'created_by' => get_current_user_id(),
                        'created_at' => current_time( 'mysql' ),
                    ]
                );

                wp_send_json([
                    'sts' => 1,
                    'msg' => 'Backup generated successfully!'
                ]);

            } catch ( Exception $e ) {
                wp_send_json([
                    'sts' => 0,
                    'msg' => 'Failed to generate backup'
                ]);
            }
        } else if ( $param === "inspqu_save_timer_seconds" ) {

            $timer_seconds = isset( $_POST['timer_seconds'] ) ? absint( $_POST['timer_seconds'] ) : 0;

            if ( $timer_seconds > 0 ) {

                update_option( 'inspqu_timer_miliseconds', $timer_seconds * 1000 );

                wp_send_json([
                    'sts' => 1,
                    'msg' => 'Auto-Refresh Timer Successfully Updated',
                ]);

            } else {

                wp_send_json([
                    'sts' => 0,
                    'msg' => 'Seconds value can\'t be less than 1',
                ]);

            }

        } else if ( $param === "inspqu_import_backup" ) {

            // Check if file was uploaded
            if ( ! isset( $_FILES['backup_file'] ) || $_FILES['backup_file']['error'] !== UPLOAD_ERR_OK ) {
                wp_send_json([
                    'sts' => 0,
                    'msg' => esc_html__( 'No file uploaded or upload error occurred.', 'inspirepulse-quotes' )
                ]);
            }

            $file = $_FILES['backup_file'];

            // Validate file type
            $file_ext = strtolower( pathinfo( $file['name'], PATHINFO_EXTENSION ) );
            if ( $file_ext !== 'sql' ) {
                wp_send_json([
                    'sts' => 0,
                    'msg' => esc_html__( 'Invalid file type. Only .sql files are allowed.', 'inspirepulse-quotes' )
                ]);
            }

            // Check file size (10MB max)
            if ( $file['size'] > 10 * 1024 * 1024 ) {
                wp_send_json([
                    'sts' => 0,
                    'msg' => esc_html__( 'File size exceeds 10MB limit.', 'inspirepulse-quotes' )
                ]);
            }

            // Check confirmation checkbox
            $confirm_import = isset( $_POST['confirm_import'] ) && $_POST['confirm_import'] === 'on';
            if ( ! $confirm_import ) {
                wp_send_json([
                    'sts' => 0,
                    'msg' => esc_html__( 'Please confirm that you understand importing will replace your current data.', 'inspirepulse-quotes' )
                ]);
            }

            // Check if user wants to create backup before import
            $create_backup = isset( $_POST['create_backup_before_import'] ) && $_POST['create_backup_before_import'] === 'on';

            // Create backup before import if requested
            if ( $create_backup ) {
                try {
                    $version_number       = time();
                    $backup_file_prefix   = INSPQU_App_Helper::inspqu_get_db_backup_file_prefix();
                    $backup_folderDir     = INSPQU_App_Helper::inspqu_get_db_backup_base_dir();

                    if ( ! is_dir( $backup_folderDir ) ) {
                        wp_mkdir_p( $backup_folderDir, 0777, true );
                    }

                    $fileVersionName = $backup_file_prefix . '-pre-import-' . $version_number . ".sql";
                    $bkpFilePath     = $backup_folderDir . $fileVersionName;

                    $dbTablesArray = INSPQU_App_Helper::inspqu_db_tables();
                    $excludeTables = [
                        esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key('backups') )
                    ];
                    $dbTablesArray = array_filter( $dbTablesArray, fn( $table ) => ! in_array( $table, $excludeTables, true ) );

                    INSPQU_App_Helper::inspqu_export_tables_schama_with_data( $dbTablesArray, $bkpFilePath );

                    $fileSizeBytes = filesize( $bkpFilePath );
                    $fileSizeKB    = round( $fileSizeBytes / 1024, 2 );

                    $backupTable = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key('backups') );
                    $backup_folderURL = INSPQU_App_Helper::inspqu_get_db_backup_base_url();

                    $wpdb->insert(
                        $backupTable,
                        [
                            'filename'   => $fileVersionName,
                            'filesize'   => $fileSizeKB . ' KB',
                            'flag'       => null,
                            'filepath'   => $backup_folderURL . $fileVersionName,
                            'type'       => 'pre-import',
                            'modules'    => maybe_serialize( $dbTablesArray ),
                            'created_by' => get_current_user_id(),
                            'created_at' => current_time( 'mysql' ),
                        ]
                    );
                } catch ( Exception $e ) {
                    wp_send_json([
                        'sts' => 0,
                        'msg' => esc_html__( 'Failed to create backup before import.', 'inspirepulse-quotes' )
                    ]);
                }
            }

            // Move uploaded file to temp location
            $upload_dir = wp_upload_dir();
            $temp_dir = $upload_dir['basedir'] . '/inspqu-temp/';
            if ( ! is_dir( $temp_dir ) ) {
                wp_mkdir_p( $temp_dir, 0777, true );
            }

            $temp_file = $temp_dir . 'import-' . time() . '.sql';
            
            if ( ! move_uploaded_file( $file['tmp_name'], $temp_file ) ) {
                wp_send_json([
                    'sts' => 0,
                    'msg' => esc_html__( 'Failed to process uploaded file.', 'inspirepulse-quotes' )
                ]);
            }

            // Import SQL file
            $import_result = INSPQU_App_Helper::inspqu_import_sql_backup( $temp_file );

            // Delete temp file
            if ( file_exists( $temp_file ) ) {
                unlink( $temp_file );
            }

            wp_send_json( $import_result );

        } else {

            wp_send_json([
                'sts' => 0,
                'msg' => 'Invalid Operation',
            ]);
        }
    }
}