<?php
namespace owthub\inspqu\admin;
use owthub\inspqu\INSPQU_App_Helper;
use owthub\inspqu\INSPQU_Query_Helper;
/**
 * @link       https://onlinewebtutorblog.com
 * @since      1.0
 * @package    InspirePulse_Quotes
 * @subpackage InspirePulse_Quotes/admin
 * @copyright  Copyright (c) 2025, Sanjay Kumar
 * @license    GPL-2.0+ https://www.gnu.org/licenses/gpl-2.0.html
 * @author     Online Web Tutor
 */
if ( ! defined( 'ABSPATH' ) ) exit;

class INSPQU_Admin {

	private $InspirePulseQuotes;
	private $version;

	public function __construct( $InspirePulseQuotes, $version ) {

		$this->InspirePulseQuotes = $InspirePulseQuotes;
		$this->version = $version;
	}

	public function inspqu_enqueue_styles($hook_suffix) {

		$allowed_pages = INSPQU_App_Helper::inspqu_allowed_admin_pages();

		if ( ! in_array( $hook_suffix, $allowed_pages, true ) ) {
			return;
		}

		wp_enqueue_style( 'inspqu-toastr', INSPQU_URL . 'admin/css/toastr.min.css', array(), $this->version, 'all' );
		
		global $wp_styles;
		$swal_already_loaded = false;

		if ( isset( $wp_styles ) && ! empty( $wp_styles->registered ) ) {
			foreach ( $wp_styles->registered as $style ) {
				if ( isset( $style->src ) && strpos( $style->src, 'sweetalert2.min.css' ) !== false ) {
					$swal_already_loaded = true;
					break;
				}
			}
		}

		if ( ! $swal_already_loaded ) {
			wp_enqueue_style( 'inspqu-swal2', INSPQU_URL . 'admin/css/sweetalert2.min.css', array(), $this->version, 'all' );
		}

		wp_enqueue_style( 'inspqu-custom', INSPQU_URL . 'admin/css/inspqu-admin.css', array(), $this->version, 'all' );
	}

	public function inspqu_enqueue_scripts($hook_suffix) {

		$allowed_pages = INSPQU_App_Helper::inspqu_allowed_admin_pages();

		if ( ! in_array( $hook_suffix, $allowed_pages, true ) ) {
			return;
		}

		wp_enqueue_script( 'inspqu-swal2', INSPQU_URL . 'admin/js/sweetalert2.min.js', array( 'jquery' ), $this->version, false );
		wp_enqueue_script( 'inspqu-validate', INSPQU_URL . 'admin/js/jquery.validate.min.js', array( 'jquery' ), $this->version, false );
		wp_enqueue_script( 'inspqu-toastr', INSPQU_URL . 'admin/js/toastr.min.js', array( 'jquery' ), $this->version, false );
		wp_enqueue_script( 'inspqu-helper', INSPQU_URL . 'admin/js/inspqu-helper.js', array( 'jquery' ), $this->version, false );
		wp_enqueue_script( 'inspqu-custom', INSPQU_URL . 'admin/js/inspqu-admin.js', array( 'jquery', 'inspqu-helper' ), $this->version, false );
		wp_localize_script( 'inspqu-custom', 'inspqu_object', array(
			'ajaxurl' => admin_url('admin-ajax.php'),
			'nonce' => wp_create_nonce('inspqu_actions'),
			'remove_submenus' => INSPQU_App_Helper::inspqu_remove_submenu_pages_link(),
			'i18n' => array(
				'confirmTitle' => esc_html__( 'Are you sure?', 'inspirepulse-quotes' ),
				'confirmYes' => esc_html__( 'Yes', 'inspirepulse-quotes' ),
				'confirmNo' => esc_html__( 'No', 'inspirepulse-quotes' ),
				'quoteContentEmpty' => esc_html__( 'Quote content cannot be empty!', 'inspirepulse-quotes' ),
				'categoryNameEmpty' => esc_html__( 'Category name cannot be empty!', 'inspirepulse-quotes' ),
				'delayTimerRequired' => esc_html__( 'Delay timer value required!', 'inspirepulse-quotes' ),
				'confirmDelete' => esc_html__( 'Are you sure want to delete?', 'inspirepulse-quotes' ),
				'confirmInstall' => esc_html__( 'Are you sure want to install quotes?', 'inspirepulse-quotes' ),
				'confirmReinstall' => esc_html__( 'Default Quotes already installed. Are you looking to Re-Install it?', 'inspirepulse-quotes' ),
				'confirmRemove' => esc_html__( 'Are you sure want to Remove Test Quotes?', 'inspirepulse-quotes' ),
				'confirmBackup' => esc_html__( 'This will generate a backup file of all database datas of this plugin.', 'inspirepulse-quotes' ),
				'addNewCategory' => esc_html__( 'Add New Category', 'inspirepulse-quotes' ),
				'editCategory' => esc_html__( 'Edit Category', 'inspirepulse-quotes' ),
				'shortcodeCopied' => esc_html__( 'Shortcode Copied Successfully', 'inspirepulse-quotes' ),
				'shortcodeCopyFailed' => esc_html__( 'Failed to Copy Shortcode', 'inspirepulse-quotes' ),
				'deletedSuccessfully' => esc_html__( 'Deleted successfully', 'inspirepulse-quotes' ),
				'deleteFailed' => esc_html__( 'Delete failed', 'inspirepulse-quotes' ),
				'quotesImportedSuccessfully' => esc_html__( 'Quotes Imported Successfully!', 'inspirepulse-quotes' ),
				'failedToImport' => esc_html__( 'Failed to Import', 'inspirepulse-quotes' ),
				'quotesRemovedSuccessfully' => esc_html__( 'Quotes Removed Successfully!', 'inspirepulse-quotes' ),
				'failedToRemove' => esc_html__( 'Failed to Remove', 'inspirepulse-quotes' ),
				'backupGeneratedSuccessfully' => esc_html__( 'Backup Generated Successfully!', 'inspirepulse-quotes' ),
				'failedToGenerateBackup' => esc_html__( 'Failed to Generate Backup', 'inspirepulse-quotes' ),
				'noFileSelected' => esc_html__( 'Please select a backup file.', 'inspirepulse-quotes' ),
				'invalidFileType' => esc_html__( 'Invalid file type. Only .sql files are allowed.', 'inspirepulse-quotes' ),
				'confirmImportRequired' => esc_html__( 'Please confirm that you understand importing will replace your current data.', 'inspirepulse-quotes' ),
				'importFailed' => esc_html__( 'Failed to import backup. Please try again.', 'inspirepulse-quotes' ),
			),
		));
	}

	public function inspqu_register_admin_menu() {

		// Menu
		add_menu_page(
			esc_html__( 'InspirePulse Quotes', 'inspirepulse-quotes' ),
			esc_html__( 'InspirePulse Quotes', 'inspirepulse-quotes' ),
			'manage_options',
			'inspqu-dashboard',
			[$this, 'inspqu_dashboard_page'],
			'dashicons-format-quote',
			30
		);

		// Submenus
		add_submenu_page(
			'inspqu-dashboard',
			esc_html__( 'Dashboard', 'inspirepulse-quotes' ),
			esc_html__( 'Dashboard', 'inspirepulse-quotes' ),
			'manage_options',
			'inspqu-dashboard',
			[$this, 'inspqu_dashboard_page']
		);
		add_submenu_page(
			'inspqu-dashboard',
			esc_html__( 'All Quotes', 'inspirepulse-quotes' ),
			esc_html__( 'All Quotes', 'inspirepulse-quotes' ),
			'manage_options',
			'inspqu-all-quotes',
			[$this, 'inspqu_all_quotes_page']
		);
		add_submenu_page(
			'inspqu-dashboard',
			esc_html__( 'Add New Quote', 'inspirepulse-quotes' ),
			esc_html__( 'Add New Quote', 'inspirepulse-quotes' ),
			'manage_options',
			'inspqu-add-quote',
			[$this, 'inspqu_add_quote_page']
		);
		add_submenu_page(
			'inspqu-dashboard',
			esc_html__( 'Categories', 'inspirepulse-quotes' ),
			esc_html__( 'Categories', 'inspirepulse-quotes' ),
			'manage_options',
			'inspqu-categories',
			[$this, 'inspqu_categories_page']
		);
		add_submenu_page(
			'inspqu-dashboard',
			esc_html__( 'Settings', 'inspirepulse-quotes' ),
			esc_html__( 'Settings', 'inspirepulse-quotes' ),
			'manage_options',
			'inspqu-settings',
			[$this, 'inspqu_settings_page']
		);
		add_submenu_page(
			'inspqu-dashboard',
			esc_html__( 'About InspirePulse Quotes', 'inspirepulse-quotes' ),
			esc_html__( 'About InspirePulse', 'inspirepulse-quotes' ),
			'manage_options',
			'inspqu-about',
			[$this, 'inspqu_about_page']
		);
		add_submenu_page(
			'inspqu-dashboard',
			esc_html__( 'Edit Quote', 'inspirepulse-quotes' ),
			'',
			'manage_options',
			'inspqu-edit-quote',
			[$this, 'inspqu_add_quote_page']
		);
		add_submenu_page(
			'inspqu-dashboard',
			esc_html__( 'Show Backup Files', 'inspirepulse-quotes' ),
			'',
			'manage_options',
			'inspqu-show-backups',
			[$this, 'inspqu_show_backups_page']
		);
		add_submenu_page(
			'inspqu-dashboard',
			esc_html__( 'Show Shortcode', 'inspirepulse-quotes' ),
			'',
			'manage_options',
			'inspqu-shortcodes',
			[$this, 'inspqu_shortcodes_page']
		);
	}

	public function inspqu_plugin_action_links($links){
		$settings_link = '<a href="admin.php?page=inspqu-settings">' . esc_html__( 'Settings', 'inspirepulse-quotes' ) . '</a>';
        $links[] = $settings_link;
        return $links;
	}

	public function inspqu_dashboard_page() {
		
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Unauthorized access', 'inspirepulse-quotes' ), 403 );
		}

		INSPQU_App_Helper::inspqu_load_admin_page('inspqu-dashboard');
	}

	public function inspqu_all_quotes_page() {
		
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Unauthorized access', 'inspirepulse-quotes' ), 403 );
		}

		if ( isset( $_GET['cat_id'] ) || isset( $_GET['author'] ) ) {
			if (
				! isset( $_GET['inspqu_nonce'] ) ||
				! wp_verify_nonce(
					sanitize_text_field( wp_unslash( $_GET['inspqu_nonce'] ) ),
					'inspqu_filter_quotes'
				)
			) {
				wp_die( esc_html__( 'Invalid request', 'inspirepulse-quotes' ), 403 );
			}
		}

		$category_id = isset( $_GET['cat_id'] )
			? absint( wp_unslash( $_GET['cat_id'] ) )
			: null;

		$author_name = isset( $_GET['author'] )
			? sanitize_text_field( wp_unslash( $_GET['author'] ) )
			: null;

		global $wpdb;

		$category_name = null;
		$quotes = [];

		if ( !empty($category_id) ) {

			$category = INSPQU_Query_Helper::inspqu_get_row( 'categories', [ 'id' => $category_id ] );

			if ( !empty($category) ) {

				$category_name = sanitize_text_field( $category['name'] ?? '' );
				$quote_ids = INSPQU_Query_Helper::inspqu_get_data( 'quote_category_relation', [ 'category_id' => $category_id ] );

				if ( ! empty( $quote_ids ) ) {

					sort( $quote_ids );

					$cache_key   = 'inspqu_quotes_' . absint( $category_id );
					$cache_group = 'inspqu_quotes';

					$quotes = wp_cache_get( $cache_key, $cache_group );

					if ( false === $quotes ) {

						$quote_tbl = esc_sql(
							INSPQU_App_Helper::inspqu_get_table_name_by_key( 'quotes' )
						);

						$ids = array_map(
							'absint',
							array_column( $quote_ids, 'quote_id' )
						);

						if ( empty( $ids ) ) {
							return [];
						}

						$placeholders = implode( ',', array_fill( 0, count( $ids ), '%d' ) );

						$sql = "
							SELECT *
							FROM {$quote_tbl}
							WHERE id IN ($placeholders)
							ORDER BY id DESC
						";
						$sql = $wpdb->prepare( $sql, ...$ids );

						$quotes = $wpdb->get_results( $sql, ARRAY_A );

						wp_cache_set( $cache_key, $quotes, $cache_group, HOUR_IN_SECONDS );
					}

					$relation_map = [];
					foreach ( $quote_ids as $rel ) {
						$relation_map[ absint( $rel['quote_id'] ) ] = absint( $rel['category_id'] );
					}
				}
			}

		} else if ( !empty($author_name) ) {
			$quotes = INSPQU_Query_Helper::inspqu_get_data( 'quotes', [ 'quote_author' => $author_name ], 'id DESC' );
		} else {
			$quotes = INSPQU_Query_Helper::inspqu_get_data( 'quotes', [], 'id DESC' );
		}

		foreach ($quotes as &$quote) {
			$quote_id = absint( $quote['id'] );

			$rel_data = INSPQU_Query_Helper::inspqu_get_row( 'quote_category_relation', [ 'quote_id' => $quote_id ] );
			if ( !empty($rel_data) ) {

				if ( !empty($category_id) ) {

					$cat_id = absint( $category_id );
					$category_data = INSPQU_Query_Helper::inspqu_get_row( 'categories', [ 'id' => $cat_id ] );

					$quote['category_name'] = sanitize_text_field( $category_data['name'] ?? '' );
					$quote['category_id']   = $cat_id;
				} else {
					
					$cat_id = absint( $rel_data['category_id'] );
					$category_data = INSPQU_Query_Helper::inspqu_get_row( 'categories', [ 'id' => $cat_id ] );

					$quote['category_name'] = sanitize_text_field( $category_data['name'] ?? '' );
					$quote['category_id']   = $cat_id;
				}
			} else {
				$quote['category_name'] = '';
				$quote['category_id'] = null;
			}
		}
		unset($quote);

		INSPQU_App_Helper::inspqu_load_admin_page( 'inspqu-quotes', [
			'inspqu_quotes' => $quotes,
			'inspqu_query_params' => [
				'category_name' => $category_name,
				'author_name'   => $author_name
			]
		]);
	}
	
	public function inspqu_add_quote_page() {

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Unauthorized access', 'inspirepulse-quotes' ), 403 );
		}

		if ( isset( $_GET['id'] ) ) {
			if (
				! isset( $_GET['inspqu_nonce'] ) ||
				! wp_verify_nonce(
					sanitize_text_field( wp_unslash( $_GET['inspqu_nonce'] ) ),
					'inspqu_edit_quote'
				)
			) {
				wp_die( esc_html__( 'Invalid request', 'inspirepulse-quotes' ), 403 );
			}
		}

		$quote_id = isset( $_GET['id'] )
			? absint( wp_unslash( $_GET['id'] ) )
			: null;

		global $wpdb;

		$rel_tbl = esc_sql(INSPQU_App_Helper::inspqu_get_table_name_by_key( 'quote_category_relation' ));

		$quote_id = null;
		if ( isset( $_GET['id'] ) ) {
			$quote_id = absint( wp_unslash( $_GET['id'] ) );
		}

		$quote_data = [];
		$categories = INSPQU_Query_Helper::inspqu_get_data( 'categories', [ 'status' => 1 ], 'id DESC' );

		if ( !empty($quote_id) ) {

			$cache_key   = 'inspqu_rel_quote_' . absint( $quote_id );
			$cache_group = 'inspqu_relations';
			
			$rel_data = wp_cache_get( $cache_key, $cache_group );

			if ( false === $rel_data ) {

				$quote_data = INSPQU_Query_Helper::inspqu_get_row( 'quotes', [ 'id' => $quote_id ] );

				$rel_data = $wpdb->get_row( $wpdb->prepare(
						"SELECT category_id FROM {$rel_tbl} WHERE quote_id = %d LIMIT 1",
						$quote_id
				), ARRAY_A );

				wp_cache_set( $cache_key, $rel_data, $cache_group, HOUR_IN_SECONDS );
			}

			if ( !empty($rel_data) ) {
				$quote_data['category_id'] = absint($rel_data['category_id']);
			}
		}

		INSPQU_App_Helper::inspqu_load_admin_page( 'inspqu-add-quote', [
			'inspqu_categories' => $categories,
			'inspqu_quote'      => $quote_data
		] );
	}

	public function inspqu_categories_page() {

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Unauthorized access', 'inspirepulse-quotes' ), 403 );
		}

		global $wpdb;

		$allowed_tables = [
			'categories',
			'quotes',
			'quote_category_relation',
		];

		$category_tbl_key = 'categories';
		$quote_tbl_key    = 'quotes';
		$rel_tbl_key      = 'quote_category_relation';

		if (
			! in_array( $category_tbl_key, $allowed_tables, true ) ||
			! in_array( $quote_tbl_key, $allowed_tables, true ) ||
			! in_array( $rel_tbl_key, $allowed_tables, true )
		) {
			wp_die( esc_html__( 'Invalid table reference', 'inspirepulse-quotes' ), 403 );
		}

		$cache_key   = 'inspqu_categories_with_counts';
		$cache_group = 'inspqu_categories';

		$categories = wp_cache_get( $cache_key, $cache_group );

		if ( false === $categories ) {

			$category_tbl = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key( $category_tbl_key ) );
			$quote_tbl    = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key( $quote_tbl_key ) );
			$rel_tbl      = esc_sql( INSPQU_App_Helper::inspqu_get_table_name_by_key( $rel_tbl_key ) );

			$sql = "
				SELECT 
					c.id,
					c.name AS category_name,
					c.status,
					COUNT(r.quote_id) AS total_quotes
				FROM {$category_tbl} AS c
				LEFT JOIN {$rel_tbl} AS r ON c.id = r.category_id
				LEFT JOIN {$quote_tbl} AS q ON r.quote_id = q.id
				GROUP BY c.id
				ORDER BY c.id DESC
			";

			$categories = $wpdb->get_results( $sql, ARRAY_A );

			wp_cache_set( $cache_key, $categories, $cache_group, HOUR_IN_SECONDS);
		}

		INSPQU_App_Helper::inspqu_load_admin_page( 'inspqu-categories', [
			'inspqu_categories' => $categories,
		] );
	}

	public function inspqu_show_backups_page() {
		
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Unauthorized access', 'inspirepulse-quotes' ), 403 );
		}

		$backups = INSPQU_Query_Helper::inspqu_get_data('backups');
		INSPQU_App_Helper::inspqu_load_admin_page('inspqu-backups', [
			'inspqu_backups' => $backups
		]);
	}

	public function inspqu_shortcodes_page() {

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Unauthorized access', 'inspirepulse-quotes' ), 403 );
		}

		$layouts = INSPQU_App_Helper::inspqu_get_quote_templates();
		INSPQU_App_Helper::inspqu_load_admin_page('inspqu-shortcode', [
			'inspqu_layouts' => $layouts
		]);
	}

	public function inspqu_settings_page() {

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Unauthorized access', 'inspirepulse-quotes' ), 403 );
		}

		INSPQU_App_Helper::inspqu_load_admin_page('inspqu-settings');
	}

	public function inspqu_about_page(){

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Unauthorized access', 'inspirepulse-quotes' ), 403 );
		}

		INSPQU_App_Helper::inspqu_load_admin_page('inspqu-about');
	}
}
