<?php
namespace ISFWC\Initial_Stock_For_WooCommerce;

defined( 'ABSPATH' ) || exit;

use WC_Product;
use WC_Product_Variation;

/**
 * Declare compatibility with WooCommerce HPOS (custom order tables).
 */
add_action(
	'before_woocommerce_init',
	static function () {
		if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
				'custom_order_tables',
				ISFWC_PLUGIN_BASENAME,
				true
			);
		}
	}
);

final class ISFWC_Plugin {

	private static ?ISFWC_Plugin $instance = null;

	public const META_INITIAL_QTY    = '_isfwc_initial_stock_qty';
	public const META_INITIAL_DATE   = '_isfwc_initial_stock_date';
	public const META_INITIAL_STATUS = '_isfwc_initial_stock_status';

	public static function instance(): ISFWC_Plugin {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action( 'plugins_loaded', [ $this, 'maybe_bootstrap' ] );
	}

	public function maybe_bootstrap(): void {
		if ( ! function_exists( 'WC' ) ) {
			add_action( 'admin_notices', [ $this, 'woocommerce_required_notice' ] );
			return;
		}

		if ( is_admin() ) {
			$this->register_admin_hooks();
		}
	}

	public function woocommerce_required_notice(): void {
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}

		echo '<div class="notice notice-error"><p>' .
			esc_html__( 'Initial Stock for WooCommerce requires WooCommerce to be installed and active.', 'initial-stock-for-woocommerce' ) .
			'</p></div>';
	}

	private function register_admin_hooks(): void {

		// Simple products.
		add_action(
			'woocommerce_product_options_inventory_product_data',
			[ $this, 'add_simple_product_fields' ]
		);

		add_action(
			'woocommerce_admin_process_product_object',
			[ $this, 'save_simple_product_fields' ]
		);

		// Variations.
		add_action(
			'woocommerce_variation_options_inventory',
			[ $this, 'add_variation_fields' ],
			10,
			3
		);

		add_action(
			'woocommerce_save_product_variation',
			[ $this, 'save_variation_fields' ],
			10,
			2
		);
	}

	/* ---------------------------------------------------------------------
	 * SIMPLE PRODUCTS
	 * ------------------------------------------------------------------- */

	public function add_simple_product_fields(): void {
		global $post;

		if ( ! $post instanceof \WP_Post ) {
			return;
		}

		$product_id   = (int) $post->ID;
		$initial_qty  = get_post_meta( $product_id, self::META_INITIAL_QTY, true );
		$initial_date = get_post_meta( $product_id, self::META_INITIAL_DATE, true );

		echo '<div class="options_group">';

		wp_nonce_field( 'isfwc_save_simple_fields', 'isfwc_simple_nonce' );
		?>

		<p class="form-field">
			<label><?php esc_html_e( 'Initial stock quantity', 'initial-stock-for-woocommerce' ); ?></label>
			<input type="number" class="short" name="_isfwc_initial_stock_qty" min="0" step="1"
				value="<?php echo '' !== $initial_qty ? esc_attr( $initial_qty ) : ''; ?>" />
		</p>

		<p class="form-field">
			<label><?php esc_html_e( 'Initial stock date', 'initial-stock-for-woocommerce' ); ?></label>
			<input type="date" class="short" name="_isfwc_initial_stock_date"
				value="<?php echo $initial_date ? esc_attr( substr( $initial_date, 0, 10 ) ) : ''; ?>" />
		</p>

		<?php
		echo '</div>';
	}

	public function save_simple_product_fields( WC_Product $product ): void {

		if (
			! isset( $_POST['isfwc_simple_nonce'] ) ||
			! wp_verify_nonce(
				sanitize_text_field( wp_unslash( $_POST['isfwc_simple_nonce'] ) ),
				'isfwc_save_simple_fields'
			)
		) {
			return;
		}

		$product_id = $product->get_id();

		$qty  = isset( $_POST['_isfwc_initial_stock_qty'] )
			? sanitize_text_field( wp_unslash( $_POST['_isfwc_initial_stock_qty'] ) )
			: '';

		$date = isset( $_POST['_isfwc_initial_stock_date'] )
			? sanitize_text_field( wp_unslash( $_POST['_isfwc_initial_stock_date'] ) )
			: '';

		$has_qty  = ( '' !== $qty );
		$has_date = ( '' !== $date && preg_match( '/^\d{4}-\d{2}-\d{2}$/', $date ) );

		$has_qty
			? update_post_meta( $product_id, self::META_INITIAL_QTY, max( 0, (int) $qty ) )
			: delete_post_meta( $product_id, self::META_INITIAL_QTY );

		$has_date
			? update_post_meta( $product_id, self::META_INITIAL_DATE, $date )
			: delete_post_meta( $product_id, self::META_INITIAL_DATE );

		update_post_meta(
			$product_id,
			self::META_INITIAL_STATUS,
			( $has_qty || $has_date ) ? 'manual' : 'disabled'
		);
	}

	/* ---------------------------------------------------------------------
	 * VARIATIONS
	 * ------------------------------------------------------------------- */

	public function add_variation_fields( int $loop, array $variation_data, $variation ): void {

		$variation_id = $variation instanceof WC_Product_Variation
			? $variation->get_id()
			: (int) $variation->ID;

		$initial_qty  = get_post_meta( $variation_id, self::META_INITIAL_QTY, true );
		$initial_date = get_post_meta( $variation_id, self::META_INITIAL_DATE, true );

		wp_nonce_field( 'isfwc_save_variation_fields', 'isfwc_variation_nonce' );
		?>

		<div class="form-row form-row-full">
			<p class="form-field">
				<label><?php esc_html_e( 'Initial stock quantity', 'initial-stock-for-woocommerce' ); ?></label>
				<input type="number" class="short"
					name="isfwc_initial_stock_qty[<?php echo esc_attr( $loop ); ?>]"
					min="0" step="1"
					value="<?php echo '' !== $initial_qty ? esc_attr( $initial_qty ) : ''; ?>" />
			</p>

			<p class="form-field">
				<label><?php esc_html_e( 'Initial stock date', 'initial-stock-for-woocommerce' ); ?></label>
				<input type="date" class="short"
					name="isfwc_initial_stock_date[<?php echo esc_attr( $loop ); ?>]"
					value="<?php echo $initial_date ? esc_attr( substr( $initial_date, 0, 10 ) ) : ''; ?>" />
			</p>
		</div>

		<?php
	}

	public function save_variation_fields( int $variation_id, int $i ): void {

		if (
			! isset( $_POST['isfwc_variation_nonce'] ) ||
			! wp_verify_nonce(
				sanitize_text_field( wp_unslash( $_POST['isfwc_variation_nonce'] ) ),
				'isfwc_save_variation_fields'
			)
		) {
			return;
		}

		$qty_array  = filter_input( INPUT_POST, 'isfwc_initial_stock_qty', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY ) ?: [];
		$date_array = filter_input( INPUT_POST, 'isfwc_initial_stock_date', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY ) ?: [];

		$qty  = isset( $qty_array[ $i ] ) ? sanitize_text_field( wp_unslash( $qty_array[ $i ] ) ) : '';
		$date = isset( $date_array[ $i ] ) ? sanitize_text_field( wp_unslash( $date_array[ $i ] ) ) : '';

		$has_qty  = ( '' !== $qty );
		$has_date = ( '' !== $date && preg_match( '/^\d{4}-\d{2}-\d{2}$/', $date ) );

		$has_qty
			? update_post_meta( $variation_id, self::META_INITIAL_QTY, max( 0, (int) $qty ) )
			: delete_post_meta( $variation_id, self::META_INITIAL_QTY );

		$has_date
			? update_post_meta( $variation_id, self::META_INITIAL_DATE, $date )
			: delete_post_meta( $variation_id, self::META_INITIAL_DATE );

		update_post_meta(
			$variation_id,
			self::META_INITIAL_STATUS,
			( $has_qty || $has_date ) ? 'manual' : 'disabled'
		);
	}
}

ISFWC_Plugin::instance();
