<?php

/**
 * Plugin Name: ingenidev Buy One Get One Free (BOGO) for WooCommerce
 * Plugin URI: https://ingenidev.com
 * Description: The ultimate WooCommerce plugin for creating powerful Buy One Get One Free (BOGO) deals, dynamic pricing rules, and quantity discounts to boost sales and increase average order value.

 * Version: 1.0.0
 * Author: ingenidev
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires at least: 5.0
 * Tested up to: 6.9
 * Requires PHP: 7.4
 * WC requires at least: 8.2
 * WC tested up to: 9.3
 * Text Domain: ingenidev-bogo-offers-for-woocommerce
 * Requires Plugins: woocommerce
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('ING_BOGO_VERSION', '1.0.0');
define('ING_BOGO_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('ING_BOGO_PLUGIN_URL', plugin_dir_url(__FILE__));

// Include the main BOGO functionality
require_once ING_BOGO_PLUGIN_DIR . 'ing-bogo-functions.php';

// HPOS Compatibility Declaration
add_action( 'before_woocommerce_init', function() {
    if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
    }
} );

// Activation hook
register_activation_hook(__FILE__, 'ing_bogo_activate');

function ing_bogo_activate()
{
    // Reserved for future use
}

// Deactivation hook
register_deactivation_hook(__FILE__, 'ing_bogo_deactivate');

function ing_bogo_deactivate()
{
    // Cleanup if needed
}

// Uninstall hook
register_uninstall_hook(__FILE__, 'ing_bogo_uninstall');

function ing_bogo_uninstall()
{
    // Reserved for future cleanup
}

// Add admin menu
add_action('admin_menu', 'ing_bogo_admin_menu');

// Enqueue admin styles
add_action('admin_enqueue_scripts', 'ing_bogo_admin_styles');

function ing_bogo_admin_menu()
{
    // Check if WooCommerce is active
    if (!class_exists('WooCommerce')) {
        return;
    }

    // Check if ingenidev menu already exists using WordPress's built-in menu handling
    global $admin_page_hooks;
    $ingenidev_menu_exists = isset($admin_page_hooks['ingenidev']);

    // Only add main ingenidev menu if it doesn't exist
    if (!$ingenidev_menu_exists) {
        add_menu_page(
            'ingenidev',
            'ingenidev',
            'manage_woocommerce',
            'ingenidev',
            'ing_bogo_admin_page',
            'dashicons-admin-generic',
            30
        );
    }

    // Add Buy One Get One submenu
    add_submenu_page(
        'ingenidev',
        __('Buy One Get One', 'ingenidev-bogo-offers-for-woocommerce'),
        __('Buy One Get One', 'ingenidev-bogo-offers-for-woocommerce'),
        'manage_woocommerce',
        'ing_bogo',
        'ing_bogo_admin_page'
    );
}

function ing_bogo_admin_styles($hook)
{
    // Only load on our admin page
    if (strpos($hook, 'ing_bogo') === false && $hook !== 'toplevel_page_ingenidev') {
        return;
    }

    wp_enqueue_style(
        'ing-bogo-admin-styles',
        ING_BOGO_PLUGIN_URL . 'assets/css/ing-bogo-admin.css',
        array(),
        ING_BOGO_VERSION
    );
}

function ing_bogo_admin_page()
{
    $bogo_products = ing_get_all_bogo_products();
?>
    <div class="wrap">
        <h1><?php echo esc_html__('ingenidev BOGO for woocommerce', 'ingenidev-bogo-offers-for-woocommerce'); ?></h1>
        <p><?php echo esc_html__('This plugin allows you to set up Buy One Get One Free offers for WooCommerce products.', 'ingenidev-bogo-offers-for-woocommerce'); ?></p>

        <h2><?php echo esc_html__('How to Use:', 'ingenidev-bogo-offers-for-woocommerce'); ?></h2>
        <ul>
            <li><strong><?php echo esc_html__('For Simple Products:', 'ingenidev-bogo-offers-for-woocommerce'); ?></strong> <?php echo esc_html__('Edit the product, scroll to the "Buy One Get One, by ingenidev" meta box, check "Enable BOGO Offer", and set the Buy Quantity (e.g., 2) and Free Quantity (e.g., 1).', 'ingenidev-bogo-offers-for-woocommerce'); ?></li>
            <li><strong><?php echo esc_html__('For Variable Products:', 'ingenidev-bogo-offers-for-woocommerce'); ?></strong> <?php echo esc_html__('Edit the product, go to Variations, and for each variation you want to enable BOGO, check "Enable BOGO Offer" and set the quantities.', 'ingenidev-bogo-offers-for-woocommerce'); ?></li>
            <li><?php echo esc_html__('Once enabled, customers will see the offer on the product page and in the cart/checkout.', 'ingenidev-bogo-offers-for-woocommerce'); ?></li>
            <li><?php echo esc_html__('The plugin automatically calculates free items based on the paid quantity and applies discounts.', 'ingenidev-bogo-offers-for-woocommerce'); ?></li>
        </ul>

        <h2><?php echo esc_html__('Notes:', 'ingenidev-bogo-offers-for-woocommerce'); ?></h2>
        <ul>
            <li><?php echo esc_html__('Free items are added to the cart automatically.', 'ingenidev-bogo-offers-for-woocommerce'); ?></li>
            <li><?php echo esc_html__('Stock levels are respected; if stock is low, fewer free items may be given.', 'ingenidev-bogo-offers-for-woocommerce'); ?></li>
            <li><?php echo esc_html__('Customers can adjust quantities, but reducing below the paid amount will disable free items for that item.', 'ingenidev-bogo-offers-for-woocommerce'); ?></li>
        </ul>

        <h2><?php echo esc_html__('BOGO Products Overview', 'ingenidev-bogo-offers-for-woocommerce'); ?></h2>
        <?php if (empty($bogo_products)): ?>
            <div class="notice notice-info">
                <p><?php echo esc_html__('No products have BOGO offers enabled yet. Configure BOGO settings on individual product edit pages.', 'ingenidev-bogo-offers-for-woocommerce'); ?></p>
            </div>
        <?php else: ?>
            <p><?php echo esc_html__('Overview of all products with BOGO offers enabled:', 'ingenidev-bogo-offers-for-woocommerce'); ?></p>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php echo esc_html__('Product', 'ingenidev-bogo-offers-for-woocommerce'); ?></th>
                        <th><?php echo esc_html__('Type', 'ingenidev-bogo-offers-for-woocommerce'); ?></th>
                        <th><?php echo esc_html__('SKU', 'ingenidev-bogo-offers-for-woocommerce'); ?></th>
                        <th><?php echo esc_html__('Price', 'ingenidev-bogo-offers-for-woocommerce'); ?></th>
                        <th><?php echo esc_html__('BOGO Rule', 'ingenidev-bogo-offers-for-woocommerce'); ?></th>
                        <th><?php echo esc_html__('Calculation', 'ingenidev-bogo-offers-for-woocommerce'); ?></th>
                        <th><?php echo esc_html__('Stock', 'ingenidev-bogo-offers-for-woocommerce'); ?></th>
                        <th><?php echo esc_html__('Actions', 'ingenidev-bogo-offers-for-woocommerce'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($bogo_products as $bogo_product): ?>
                        <tr class="<?php echo ($bogo_product['stock'] === 0 || $bogo_product['stock_status'] === 'outofstock') ? 'stock-out' : ''; ?>">
                            <td>
                                <strong><?php echo esc_html($bogo_product['name']); ?></strong>
                                <?php if ($bogo_product['type'] === 'variation'): ?>
                                    <?php /* translators: %s: parent product ID */ ?>
                                    <br><small style="color: #666;"><?php echo esc_html(sprintf(__('Variation of Product #%s', 'ingenidev-bogo-offers-for-woocommerce'), $bogo_product['parent_id'])); ?></small>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if ($bogo_product['type'] === 'simple'): ?>
                                    <span class="dashicons dashicons-admin-post" title="<?php echo esc_attr__('Simple Product', 'ingenidev-bogo-offers-for-woocommerce'); ?>"></span> <?php echo esc_html__('Simple', 'ingenidev-bogo-offers-for-woocommerce'); ?>
                                <?php else: ?>
                                    <span class="dashicons dashicons-networking" title="<?php echo esc_attr__('Product Variation', 'ingenidev-bogo-offers-for-woocommerce'); ?>"></span> <?php echo esc_html__('Variation', 'ingenidev-bogo-offers-for-woocommerce'); ?>
                                <?php endif; ?>
                            </td>
                            <td><?php echo esc_html($bogo_product['sku'] ?: '—'); ?></td>
                            <td>
                                <?php if ($bogo_product['sale_price'] && $bogo_product['sale_price'] < $bogo_product['regular_price']): ?>
                                    <span style="text-decoration: line-through; color: #999;"><?php echo wp_kses( wc_price($bogo_product['regular_price']), array(
   'span' => array(
       'class' => array(),
   ),
   'bdi' => array(),
) ); ?></span>
                                    <strong style="color: #28a745;"><?php echo wp_kses( wc_price($bogo_product['sale_price']), array(
   'span' => array(
       'class' => array(),
   ),
   'bdi' => array(),
) ); ?></strong>
                                <?php else: ?>
                                    <strong><?php echo wp_kses( wc_price($bogo_product['price']), array(
   'span' => array(
       'class' => array(),
   ),
   'bdi' => array(),
) ); ?></strong>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php /* translators: 1: buy quantity, 2: free quantity */ ?>
                                <strong><?php echo esc_html(sprintf(__('Buy %1$s Get %2$s Free', 'ingenidev-bogo-offers-for-woocommerce'), $bogo_product['buy_qty'], $bogo_product['free_qty'])); ?></strong>
                            </td>
                            <td>
                                <?php /* translators: %s: calculation type (Cycle or Total) */ ?>
                                <?php echo esc_html(sprintf(__('%s Based', 'ingenidev-bogo-offers-for-woocommerce'), ucfirst($bogo_product['calculation_type']))); ?>
                                <?php if ($bogo_product['calculation_type'] === 'total'): ?>
                                    <br><small style="color: #666;"><?php echo esc_html__('(Treats all as paid)', 'ingenidev-bogo-offers-for-woocommerce'); ?></small>
                                <?php else: ?>
                                    <br><small style="color: #666;"><?php echo esc_html__('(Tracks paid items)', 'ingenidev-bogo-offers-for-woocommerce'); ?></small>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if ($bogo_product['stock'] === null): ?>
                                    <span class="dashicons dashicons-minus" title="<?php echo esc_attr__('Not managed', 'ingenidev-bogo-offers-for-woocommerce'); ?>"></span>
                                <?php elseif ($bogo_product['stock'] === 0 || $bogo_product['stock_status'] === 'outofstock'): ?>
                                    <span class="dashicons dashicons-dismiss" style="color: #dc3232;" title="<?php echo esc_attr__('Out of stock', 'ingenidev-bogo-offers-for-woocommerce'); ?>"></span> 0
                                <?php else: ?>
                                    <span class="dashicons dashicons-yes" style="color: #46b450;" title="<?php echo esc_attr__('In stock', 'ingenidev-bogo-offers-for-woocommerce'); ?>"></span> <?php echo esc_html($bogo_product['stock']); ?>
                                <?php endif; ?>
                            </td>
                            <td>
                                <a href="<?php echo esc_url($bogo_product['edit_link']); ?>" class="button button-small" title="<?php echo esc_attr__('Edit Product', 'ingenidev-bogo-offers-for-woocommerce'); ?>">
                                    <span class="dashicons dashicons-edit"></span> <?php echo esc_html__('Edit', 'ingenidev-bogo-offers-for-woocommerce'); ?>
                                </a>
                                <a href="<?php echo esc_url($bogo_product['permalink']); ?>" target="_blank" class="button button-small" title="<?php echo esc_attr__('View Product', 'ingenidev-bogo-offers-for-woocommerce'); ?>">
                                    <span class="dashicons dashicons-visibility"></span> <?php echo esc_html__('View', 'ingenidev-bogo-offers-for-woocommerce'); ?>
                                </a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php /* translators: %s: number of BOGO products */ ?>
            <p><em><?php echo esc_html(sprintf(__('Total BOGO products: %s', 'ingenidev-bogo-offers-for-woocommerce'), count($bogo_products))); ?></em></p>
        <?php endif; ?>
    </div>
<?php
}
