<?php

/**
 * BOGO functionality for ingenidev Buy One Get One Free plugin
 */

// BOGO settings via product meta

if (!defined('ABSPATH')) {
    exit;
}

// Enqueue frontend scripts for BOGO functionality
add_action('wp_enqueue_scripts', 'ing_bogo_frontend_scripts');

function ing_bogo_frontend_scripts()
{
    // Only load on product pages where BOGO functionality might be used
    if (!is_singular('product') && !is_shop() && !is_product_category() && !is_product_tag()) {
        return;
    }

    wp_enqueue_script(
        'ing-bogo-frontend',
        ING_BOGO_PLUGIN_URL . 'assets/js/ing-bogo-frontend.js',
        array('jquery'),
        ING_BOGO_VERSION,
        true
    );
}

/**
 * Store a stable BOGO marker in cart item data when adding to cart.
 * We do NOT set paid_qty here because changing cart item data would create
 * unique cart rows and prevent normal cart merging. paid_qty is tracked
 * on the cart_contents entries and is updated in the add_to_cart handler.
 */
function ing_add_bogo_cart_item_data($cart_item_data, $product_id, $variation_id)
{
    $meta_id = $variation_id ?: $product_id;
    // BOGO enabled check will be done in later hooks, no group needed
    return $cart_item_data;
}
add_action('woocommerce_add_cart_item_data', 'ing_add_bogo_cart_item_data', 10, 3);

/**
 * Reverse-engineer paid_qty from a displayed total if needed.
 * Looks for a paid value p such that p + floor(p/2) == $total.
 * If none found, return $total (treat as paid-only).
 */
function ing_bogo_reverse_total_to_paid($total, $product_id = null)
{
    $total = intval($total);

    // If no product_id provided, fall back to the old hardcoded logic for backward compatibility
    if (!$product_id) {
        // Use default Buy 2 Get 1 logic for backward compatibility
        for ($p = $total; $p >= 0; $p--) {
            if (($p + intval(floor($p / 2))) === $total) {
                return $p;
            }
        }
        return $total;
    }

    // Use the actual BOGO settings from the product
    $meta_id = $product_id;
    $buy_qty = intval(get_post_meta($meta_id, '_bogo_buy_qty', true)) ?: 2;
    $free_per_cycle = intval(get_post_meta($meta_id, '_bogo_free_qty', true)) ?: 1;

    for ($p = $total; $p >= 0; $p--) {
        $expected_free = floor($p / $buy_qty) * $free_per_cycle;
        if (($p + $expected_free) === $total) {
            return $p;
        }
    }
    return $total;
}

/**
 * Merge cart items for BOGO product to update paid_qty instead of adding a duplicate.
 *
 * We use the incoming $quantity as the number of paid units to add. The function handles three cases:
 *  - No existing item for the product in cart: set paid_qty on the newly added item.
 *  - Existing different cart row for same product: increment its paid_qty and remove the newly added duplicate.
 *  - The added row is the existing row (WooCommerce merged): derive previous paid_qty and add the incoming amount.
 */
function ing_merge_bogo_cart_items($cart_item_key, $product_id, $quantity, $variation_id, $variation, $cart_item_data)
{
    $meta_id = $variation_id ?: $product_id;
    if (get_post_meta($meta_id, '_bogo_enabled', true) !== 'yes') {
        return;
    }

    if (empty(WC()->cart)) {
        return;
    }

    $quantity_to_add = max(0, intval($quantity));
    $cart_contents = WC()->cart->get_cart();

    $found_key = null;
    $found_item = null;
    foreach ($cart_contents as $key => $item) {
        if ($item['product_id'] === $product_id && $item['variation_id'] === $variation_id) {
            $found_key = $key;
            $found_item = $item;
            break;
        }
    }

    // No existing item -> set paid_qty on the newly added item
    if ($found_key === null) {
        if (isset(WC()->cart->cart_contents[$cart_item_key])) {
            WC()->cart->cart_contents[$cart_item_key]['paid_qty'] = $quantity_to_add;
        }
        return;
    }

    // If an existing different row is found, add to its paid_qty and remove the new row
    if ($found_key !== $cart_item_key) {
        $existing_paid = isset($found_item['paid_qty']) ? intval($found_item['paid_qty']) : ing_bogo_reverse_total_to_paid(intval($found_item['quantity']), $meta_id);
        $new_paid = $existing_paid + $quantity_to_add;
        WC()->cart->cart_contents[$found_key]['paid_qty'] = $new_paid;

        if (isset(WC()->cart->cart_contents[$found_key]['bumped_down'])) {
            unset(WC()->cart->cart_contents[$found_key]['bumped_down']);
        }

        WC()->cart->remove_cart_item($cart_item_key);
        return;
    }

    // If the added item is the same row (WooCommerce merged automatically),
    // infer the previous displayed total and derive previous paid, then add.
    $current_displayed_total = isset($found_item['quantity']) ? intval($found_item['quantity']) : 0;
    $prev_displayed_total = max(0, $current_displayed_total - $quantity_to_add);

    if (isset($found_item['paid_qty'])) {
        $prev_paid = intval($found_item['paid_qty']);
    } else {
        $prev_paid = ing_bogo_reverse_total_to_paid($prev_displayed_total, $meta_id);
    }

    $new_paid = $prev_paid + $quantity_to_add;
    WC()->cart->cart_contents[$cart_item_key]['paid_qty'] = $new_paid;

    if (isset(WC()->cart->cart_contents[$cart_item_key]['bumped_down'])) {
        unset(WC()->cart->cart_contents[$cart_item_key]['bumped_down']);
    }
    return;
}
add_action('woocommerce_add_to_cart', 'ing_merge_bogo_cart_items', 10, 6);

/**
 * Update paid_qty when quantities change on cart page
 *
 * Interpret cart quantity updates (which show the displayed total) as:
 * - Reduction: user set a lower total => treat as paid-only and mark bumped_down.
 * - Increase: add the diff to paid_qty and clear bumped_down so BOGO can apply.
 */
function ing_update_bogo_cart_item_quantity($cart_item_key, $quantity, $old_quantity, $cart)
{
    $cart_item = $cart->get_cart()[$cart_item_key];
    $meta_id = $cart_item['variation_id'] ?: $cart_item['product_id'];
    if (get_post_meta($meta_id, '_bogo_enabled', true) === 'yes') {
        $quantity = intval($quantity);
        $old_quantity = intval($old_quantity);
        $current_paid = isset($cart_item['paid_qty']) ? intval($cart_item['paid_qty']) : $old_quantity;
        $diff = $quantity - $old_quantity;

        if ($diff > 0) {
            // Increase: user is manually adding more items
            $bumped_down = isset($cart_item['bumped_down']) && $cart_item['bumped_down'];

            if ($bumped_down) {
                // If previously bumped down, treat the increase as paid items
                WC()->cart->cart_contents[$cart_item_key]['paid_qty'] = $current_paid + $diff;
                if (isset(WC()->cart->cart_contents[$cart_item_key]['bumped_down'])) {
                    unset(WC()->cart->cart_contents[$cart_item_key]['bumped_down']);
                }
            } else {
                // Calculate paid quantity using cycle method
                $buy_qty = intval(get_post_meta($meta_id, '_bogo_buy_qty', true)) ?: 2;
                $free_per_cycle = intval(get_post_meta($meta_id, '_bogo_free_qty', true)) ?: 1;
                
                // Maintain the BOGO ratio using cycle method
                $total_items_per_cycle = $buy_qty + $free_per_cycle;
                $cycles = floor($quantity / $total_items_per_cycle);
                $remainder = $quantity % $total_items_per_cycle;
                $new_paid_qty = ($cycles * $buy_qty) + min($remainder, $buy_qty);

                WC()->cart->cart_contents[$cart_item_key]['paid_qty'] = $new_paid_qty;
            }
        } elseif ($diff < 0) {
            // Reduction: user wants fewer total units -> treat as paid-only change and mark bumped_down
            WC()->cart->cart_contents[$cart_item_key]['paid_qty'] = $quantity;
            WC()->cart->cart_contents[$cart_item_key]['bumped_down'] = true;
        } else {
            // No change: do nothing
        }
    }
}
add_action('woocommerce_after_cart_item_quantity_update', 'ing_update_bogo_cart_item_quantity', 10, 4);

/**
 * Calculate free_qty and total_qty with stock cap
 */
function ing_calculate_bogo($paid_qty, $product_id, $cart_item = null)
{
    $meta_id = $product_id;
    $buy_qty = intval(get_post_meta($meta_id, '_bogo_buy_qty', true)) ?: 2;
    $free_per_cycle = intval(get_post_meta($meta_id, '_bogo_free_qty', true)) ?: 1;

    // Calculate free items based ONLY on tracked paid quantity (cycle method)
    // This prevents feedback loops where modified quantities are recalculated
    $free_qty = floor($paid_qty / $buy_qty) * $free_per_cycle;

    $total_qty = $paid_qty + $free_qty;

    $stock = get_post_meta($product_id, '_stock', true);
    if ($stock && is_numeric($stock) && $total_qty > $stock) {
        $free_qty = max(0, $stock - $paid_qty);
        $total_qty = $paid_qty + $free_qty;
    }
    return array('free_qty' => $free_qty, 'total_qty' => $total_qty);
}

/**
 * Adjust cart item quantity to total_qty and pricing before calculate totals
 */
function ing_adjust_bogo_totals($cart)
{
    if (is_admin() && !defined('DOING_AJAX')) return;
    foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
        $meta_id = isset($cart_item['variation_id']) && $cart_item['variation_id'] ? $cart_item['variation_id'] : $cart_item['product_id'];
        if (get_post_meta($meta_id, '_bogo_enabled', true) === 'yes') {
            // If paid_qty is not set, initialize it using reverse calculation from current quantity
            // This prevents treating already-adjusted quantities as new paid quantities
            if (!isset($cart_item['paid_qty'])) {
                $initial_paid = ing_bogo_reverse_total_to_paid(intval($cart_item['quantity']), $meta_id);
                $cart->cart_contents[$cart_item_key]['paid_qty'] = $initial_paid;
                $paid_qty = $initial_paid;
            } else {
                $paid_qty = max(0, intval($cart_item['paid_qty']));
            }
            $bumped_down = isset($cart_item['bumped_down']) && $cart_item['bumped_down'];
            // Variation meta must be set to enable BOGO for a variation — do NOT fallback to parent
            $buy_qty = intval(get_post_meta($meta_id, '_bogo_buy_qty', true)) ?: 2;
            $free_per_cycle = intval(get_post_meta($meta_id, '_bogo_free_qty', true)) ?: 1;
            $expected_free = floor($paid_qty / $buy_qty) * $free_per_cycle;
            $cart->cart_contents[$cart_item_key]['expected_free'] = $expected_free;

            if ($bumped_down) {
                // No free items when user explicitly reduced quantity on cart
                $free_qty = 0;
                $total_qty = $paid_qty;
                // Cap to stock if needed
                $stock = get_post_meta($cart_item['product_id'], '_stock', true);
                if ($stock && is_numeric($stock) && $total_qty > $stock) {
                    $total_qty = intval($stock);
                    $paid_qty = min($paid_qty, $total_qty);
                }
            } else {
                $calc = ing_calculate_bogo($paid_qty, $meta_id, $cart_item);
                $free_qty = $calc['free_qty'];
                $total_qty = $calc['total_qty'];
                if ($free_qty < $expected_free) {
                    $cart->cart_contents[$cart_item_key]['stock_capped'] = true;
                    $cart->cart_contents[$cart_item_key]['actual_free'] = $free_qty;
                }
            }

            if ($total_qty <= 0) {
                $total_qty = $paid_qty;
            }

            // Set displayed quantity to total (paid + free)
            // Only update if the new total is different from current quantity to prevent unnecessary updates
            if ($cart->cart_contents[$cart_item_key]['quantity'] != $total_qty) {
                $cart->cart_contents[$cart_item_key]['quantity'] = $total_qty;
            }

            // Use sale price if available, otherwise regular price
            $original_price = $cart_item['data']->get_sale_price();
            if (!$original_price) {
                $original_price = $cart_item['data']->get_regular_price();
            }
            if (!$original_price) {
                $original_price = $cart_item['data']->get_price();
            }
            $original_price = floatval($original_price);
            $effective_price = $total_qty > 0 ? ($paid_qty * $original_price) / $total_qty : $original_price;
            $cart_item['data']->set_price($effective_price);
        }
    }
}
add_action('woocommerce_before_calculate_totals', 'ing_adjust_bogo_totals', 10, 1);

/**
 * Add BOGO note to cart and checkout item data
 */
function ing_add_bogo_item_data($item_data, $cart_item)
{
    // Use variation meta if present, otherwise product meta (simple products)
    $meta_id = isset($cart_item['variation_id']) && $cart_item['variation_id'] ? $cart_item['variation_id'] : $cart_item['product_id'];
    if (get_post_meta($meta_id, '_bogo_enabled', true) === 'yes') {
        $paid_qty = isset($cart_item['paid_qty']) ? intval($cart_item['paid_qty']) : $cart_item['quantity'];
        // Calculate using the meta_id so variation settings apply
        $calc = ing_calculate_bogo($paid_qty, $meta_id, $cart_item);
        $free_qty = $calc['free_qty'];
        $total_qty = $calc['total_qty'];
        // If quantity is less than total, no free
        if ($cart_item['quantity'] < $total_qty) {
            $free_qty = 0;
        }
        $buy_qty = intval(get_post_meta($meta_id, '_bogo_buy_qty', true)) ?: 2;
        $free_per_cycle = intval(get_post_meta($meta_id, '_bogo_free_qty', true)) ?: 1;
        $expected_free = floor($paid_qty / $buy_qty) * $free_per_cycle;
        $bumped_down = isset($cart_item['bumped_down']) && $cart_item['bumped_down'];
        if ($bumped_down) {
            if ($free_qty == 0) {
                $item_data[] = array(
                    'name' => __('Your Offer', 'ingenidev-bogo-offers-for-woocommerce'),
                    'value' => __('Quantity reduced by user, no free items.', 'ingenidev-bogo-offers-for-woocommerce'),
                );
            } else {
                $item_data[] = array(
                    'name' => __('BOGO Offer', 'ingenidev-bogo-offers-for-woocommerce'),
                    /* translators: %s: number of free items */
                    'value' => sprintf(__('Includes %s free item(s). Quantity reduced by user.', 'ingenidev-bogo-offers-for-woocommerce'), $free_qty),
                );
            }
        } else {
            if ($free_qty > 0) {
                $item_data[] = array(
                    'name' => __('BOGO Offer', 'ingenidev-bogo-offers-for-woocommerce'),
                    /* translators: 1: buy quantity, 2: free quantity per cycle, 3: total free items */
                    'value' => sprintf(__('Buy %1$s Get %2$s Free - Includes %3$s free item(s)', 'ingenidev-bogo-offers-for-woocommerce'), $buy_qty, $free_per_cycle, $free_qty),
                );
            }
            if ($expected_free > $free_qty) {
                $item_data[] = array(
                    'name' => __('Stock Limitation', 'ingenidev-bogo-offers-for-woocommerce'),
                    /* translators: 1: actual free items given, 2: expected free items */
                    'value' => sprintf(__('Due to stock limits, only %1$s free item(s) added instead of %2$s.', 'ingenidev-bogo-offers-for-woocommerce'), $free_qty, $expected_free),
                );
            }
        }
        if ($free_qty > 0) {
            $sale_price = $cart_item['data']->get_sale_price();
            if (!$sale_price) {
                $sale_price = $cart_item['data']->get_regular_price();
            }
            if (!$sale_price) {
                $sale_price = $cart_item['data']->get_price();
            }
            $regular_price = $cart_item['data']->get_regular_price();
            if (!$regular_price) {
                $regular_price = $cart_item['data']->get_price();
            }
            $discounted_total = $paid_qty * $sale_price;
            $full_total = $total_qty * $regular_price;
            $item_data[] = array(
                'name' => __('Original Price', 'ingenidev-bogo-offers-for-woocommerce'),
                'value' => "<s>" . wp_kses( wc_price($full_total), array(
   'span' => array(
       'class' => array(),
   ),
   'bdi' => array(),
) ) . "</s>",
            );
            $item_data[] = array(
                'name' => __('Your Price', 'ingenidev-bogo-offers-for-woocommerce'),
                'value' => '<strong style="color:green">' . wp_kses( wc_price($discounted_total), array(
   'span' => array(
       'class' => array(),
   ),
   'bdi' => array(),
) ) . '</strong>',
            );
        }
    }
    return $item_data;
}
add_filter('woocommerce_get_item_data', 'ing_add_bogo_item_data', 10, 2);

/**
 * Add BOGO notice to product page
 */
function ing_add_bogo_notice_single_product()
{
    global $product;
    $product_id = $product->get_id();
    if ($product->is_type('variable')) {
        $children = $product->get_children();
        foreach ($children as $variation_id) {
            if (get_post_meta($variation_id, '_bogo_enabled', true) === 'yes') {
                $buy_qty = intval(get_post_meta($variation_id, '_bogo_buy_qty', true)) ?: 2;
                $free_qty = intval(get_post_meta($variation_id, '_bogo_free_qty', true)) ?: 1;
                echo "<div id='bogo-variation-" . esc_attr($variation_id) . "' style='display: none;' data-buy-qty='" . esc_attr($buy_qty) . "' data-free-qty='" . esc_attr($free_qty) . "'></div>";
            }
        }
        echo "<div id='bogo_placeholder_" . esc_attr($product_id) . "'></div>";
    } else {
        if (get_post_meta($product_id, '_bogo_enabled', true) === 'yes') {
            $buy_qty = intval(get_post_meta($product_id, '_bogo_buy_qty', true)) ?: 2;
            $free_qty = intval(get_post_meta($product_id, '_bogo_free_qty', true)) ?: 1;
            /* translators: 1: buy quantity, 2: free quantity */
            echo '<div class="bogo-notice" style="color: red; font-weight: bold; margin-bottom: 10px;">' . esc_html(sprintf(__('Buy %1$s Get %2$s Free Offer!', 'ingenidev-bogo-offers-for-woocommerce'), $buy_qty, $free_qty)) . '</div>';
        }
    }
}
add_action('woocommerce_before_add_to_cart_button', 'ing_add_bogo_notice_single_product');

/**
 * Add BOGO notice to product list
 */
function ing_add_bogo_notice_loop()
{
    global $product;
    $product_id = $product->get_id();
    $has_bogo = false;
    $buy_qty = 2;
    $free_qty = 1;
    if ($product->is_type('variable')) {
        $children = $product->get_children();
        foreach ($children as $variation_id) {
            if (get_post_meta($variation_id, '_bogo_enabled', true) === 'yes') {
                $has_bogo = true;
                $buy_qty = intval(get_post_meta($variation_id, '_bogo_buy_qty', true)) ?: 2;
                $free_qty = intval(get_post_meta($variation_id, '_bogo_free_qty', true)) ?: 1;
                echo "<div id='bogo-variation-" . esc_attr($variation_id) . "' style='display: none;' data-product-id='" . esc_attr($product_id) . "' data-buy-qty='" . esc_attr($buy_qty) . "' data-free-qty='" . esc_attr($free_qty) . "'></div>";
            }
        }
        if ($has_bogo) {
            echo "<div id='bogo_placeholder_" . esc_attr($product_id) . "' class='bogo-notice' style='color: red; font-weight: bold; margin-top: 5px;'></div>";
        }
    } else {
        if (get_post_meta($product_id, '_bogo_enabled', true) === 'yes') {
            $has_bogo = true;
            $buy_qty = intval(get_post_meta($product_id, '_bogo_buy_qty', true)) ?: 2;
            $free_qty = intval(get_post_meta($product_id, '_bogo_free_qty', true)) ?: 1;
            /* translators: 1: buy quantity, 2: free quantity */
            echo '<div class="bogo-notice" style="color: red; font-weight: bold; margin-top: 5px;">' . esc_html(sprintf(__('Buy %1$s Get %2$s Free!', 'ingenidev-bogo-offers-for-woocommerce'), $buy_qty, $free_qty)) . '</div>';
        }
    }
}
add_action('woocommerce_after_shop_loop_item_title', 'ing_add_bogo_notice_loop');

// BOGO loop JS functionality is now handled by the enqueued script
function ing_add_bogo_loop_js()
{
    // This function is deprecated - functionality moved to ing-bogo-frontend.js
    // Keeping for backward compatibility but no longer outputs inline scripts
}
add_action('wp_footer', 'ing_add_bogo_loop_js');

// Add custom meta box for BOGO settings (only for simple products)
function ing_add_bogo_meta_box()
{
    global $post;
    $product = wc_get_product($post->ID);
    if ($product && $product->is_type('simple')) {
        add_meta_box(
            'bogo_meta_box',
            __('Buy One Get One, by ingenidev', 'ingenidev-bogo-offers-for-woocommerce'),
            'ing_bogo_meta_box_callback',
            'product',
            'normal',
            'high'
        );
    }
    // Note: Variation BOGO settings are handled separately in ing_add_bogo_variation_fields
}
add_action('add_meta_boxes', 'ing_add_bogo_meta_box');

// Meta box callback
function ing_bogo_meta_box_callback($post)
{
    wp_nonce_field('bogo_save_meta', 'bogo_meta_nonce');

    $enabled = get_post_meta($post->ID, '_bogo_enabled', true);
    $buy_qty = get_post_meta($post->ID, '_bogo_buy_qty', true) ?: 2;
    $free_qty = get_post_meta($post->ID, '_bogo_free_qty', true) ?: 1;
    $calculation_type = get_post_meta($post->ID, '_bogo_calculation_type', true) ?: 'cycle';

    echo '<table class="form-table">';
    echo '<tr><th><label for="bogo_enabled">' . esc_html__('Enable BOGO Offer', 'ingenidev-bogo-offers-for-woocommerce') . '</label></th><td><input type="checkbox" id="bogo_enabled" name="bogo_enabled" value="yes" ' . checked($enabled, 'yes', false) . ' /><br><small>' . esc_html__('Check to enable Buy X Get Y Free offer for this product.', 'ingenidev-bogo-offers-for-woocommerce') . '</small></td></tr>';
    echo '<tr><th><label for="bogo_buy_qty">' . esc_html__('Buy Quantity', 'ingenidev-bogo-offers-for-woocommerce') . '</label></th><td><input type="number" id="bogo_buy_qty" name="bogo_buy_qty" value="' . esc_attr($buy_qty) . '" min="1" /><br><small>' . esc_html__('Number of items to buy (e.g., 2 for Buy 2 Get 1).', 'ingenidev-bogo-offers-for-woocommerce') . '</small></td></tr>';
    echo '<tr><th><label for="bogo_free_qty">' . esc_html__('Free Quantity', 'ingenidev-bogo-offers-for-woocommerce') . '</label></th><td><input type="number" id="bogo_free_qty" name="bogo_free_qty" value="' . esc_attr($free_qty) . '" min="0" /><br><small>' . esc_html__('Number of free items per buy cycle (e.g., 1 for Buy 2 Get 1).', 'ingenidev-bogo-offers-for-woocommerce') . '</small></td></tr>';
    
    // Calculation Type field - COMMENTED OUT (only cycle method is used now)
    /*
    echo '<tr><th><label for="bogo_calculation_type">' . esc_html__('Calculation Type', 'ingenidev-bogo-offers-for-woocommerce') . '</label></th><td>';
    echo '<select id="bogo_calculation_type" name="bogo_calculation_type">';
    echo '<option value="cycle" ' . selected($calculation_type, 'cycle', false) . '>' . esc_html__('Cycle Based (Track Paid Items)', 'ingenidev-bogo-offers-for-woocommerce') . '</option>';
    echo '<option value="total" ' . selected($calculation_type, 'total', false) . '>' . esc_html__('Total Based (Treat All as Paid)', 'ingenidev-bogo-offers-for-woocommerce') . '</option>';
    echo '</select>';
    echo '<br><small>' . esc_html__('Cycle: Track actual paid items (standard BOGO). Total: Treat all items as paid for maximum benefit.', 'ingenidev-bogo-offers-for-woocommerce') . '</small></td></tr>';
    */
    
    echo '</table>';
}

// Save meta data
function ing_save_bogo_meta($post_id)
{
    if (!isset($_POST['bogo_meta_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['bogo_meta_nonce'])), 'bogo_save_meta')) return;
    if (!current_user_can('edit_post', $post_id)) return;

    $enabled = isset($_POST['bogo_enabled']) ? 'yes' : '';
    update_post_meta($post_id, '_bogo_enabled', $enabled);

    $buy_qty = intval(wp_unslash($_POST['bogo_buy_qty'] ?? 2));
    update_post_meta($post_id, '_bogo_buy_qty', max(1, $buy_qty));

    $free_qty = intval(wp_unslash($_POST['bogo_free_qty'] ?? 1));
    update_post_meta($post_id, '_bogo_free_qty', max(0, $free_qty));

    $calculation_type = sanitize_text_field(wp_unslash($_POST['bogo_calculation_type'] ?? 'cycle'));
    update_post_meta($post_id, '_bogo_calculation_type', $calculation_type);
}
add_action('save_post', 'ing_save_bogo_meta');

// Add BOGO fields to variations
function ing_add_bogo_variation_fields($loop, $variation_data, $variation)
{
    $enabled = get_post_meta($variation->ID, '_bogo_enabled', true);
    $buy_qty = get_post_meta($variation->ID, '_bogo_buy_qty', true) ?: 2;
    $free_qty = get_post_meta($variation->ID, '_bogo_free_qty', true) ?: 1;
    $calculation_type = get_post_meta($variation->ID, '_bogo_calculation_type', true) ?: 'cycle';

    echo '<div class="variation-bogo-fields" style="border: 1px solid #ddd; padding: 10px; margin: 10px 0; background: #f9f9f9;" data-variation-id="' . esc_attr($variation->ID) . '">';
    echo '<h4>' . esc_html__('BOGO Settings for this Variation', 'ingenidev-bogo-offers-for-woocommerce') . '</h4>';
    echo '<p><label><input type="checkbox" class="variation_bogo_enabled" name="variation_bogo_enabled[' . esc_html($variation->ID) . ']" value="yes" ' . checked($enabled, 'yes', false) . ' /> ' . esc_html__('Enable BOGO Offer', 'ingenidev-bogo-offers-for-woocommerce') . '</label></p>';
    echo '<p><label>' . esc_html__('Buy Quantity:', 'ingenidev-bogo-offers-for-woocommerce') . ' <input type="number" class="variation_bogo_buy_qty" name="variation_bogo_buy_qty[' . esc_html($variation->ID) . ']" value="' . esc_attr($buy_qty) . '" min="1" style="width: 80px;" /></label></p>';
    echo '<p><label>' . esc_html__('Free Quantity:', 'ingenidev-bogo-offers-for-woocommerce') . ' <input type="number" class="variation_bogo_free_qty" name="variation_bogo_free_qty[' . esc_html($variation->ID) . ']" value="' . esc_attr($free_qty) . '" min="0" style="width: 80px;" /></label></p>';
    
    // Calculation Type field - COMMENTED OUT (only cycle method is used now)
    /*
    echo '<p><label>' . esc_html__('Calculation Type:', 'ingenidev-bogo-offers-for-woocommerce') . ' <select class="variation_bogo_calculation_type" name="variation_bogo_calculation_type[' . esc_html($variation->ID) . ']" style="width: 200px;">';
    echo '<option value="cycle" ' . selected($calculation_type, 'cycle', false) . '>' . esc_html__('Cycle Based (Track Paid Items)', 'ingenidev-bogo-offers-for-woocommerce') . '</option>';
    echo '<option value="total" ' . selected($calculation_type, 'total', false) . '>' . esc_html__('Total Based (Treat All as Paid)', 'ingenidev-bogo-offers-for-woocommerce') . '</option>';
    echo '</select></label></p>';
    */
    
    echo '</div>';
}
add_action('woocommerce_product_after_variable_attributes', 'ing_add_bogo_variation_fields', 50, 3);

// Save variation BOGO meta
function ing_save_bogo_variation_meta($variation_id, $i)
{
    // Verify nonce for security - check if main product form nonce exists and is valid
    if (!isset($_POST['bogo_meta_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['bogo_meta_nonce'])), 'bogo_save_meta')) {
        return;
    }

    // Check if variation BOGO data exists in POST
    if (isset($_POST['variation_bogo_enabled'][$variation_id])) {
        $enabled_value = sanitize_text_field(wp_unslash($_POST['variation_bogo_enabled'][$variation_id]));
        if ($enabled_value === 'yes') {
            update_post_meta($variation_id, '_bogo_enabled', 'yes');
        } else {
            delete_post_meta($variation_id, '_bogo_enabled');
        }
    } else {
        delete_post_meta($variation_id, '_bogo_enabled');
    }

    // Save buy quantity
    if (isset($_POST['variation_bogo_buy_qty'][$variation_id])) {
        $buy_qty = intval(wp_unslash($_POST['variation_bogo_buy_qty'][$variation_id]));
        update_post_meta($variation_id, '_bogo_buy_qty', max(1, $buy_qty));
    }

    // Save free quantity
    if (isset($_POST['variation_bogo_free_qty'][$variation_id])) {
        $free_qty = intval(wp_unslash($_POST['variation_bogo_free_qty'][$variation_id]));
        update_post_meta($variation_id, '_bogo_free_qty', max(0, $free_qty));
    }

    // Save calculation type
    if (isset($_POST['variation_bogo_calculation_type'][$variation_id])) {
        $calculation_type_value = sanitize_text_field(wp_unslash($_POST['variation_bogo_calculation_type'][$variation_id]));
        // Ensure it's either 'cycle' or 'total'
        if (!in_array($calculation_type_value, ['cycle', 'total'])) {
            $calculation_type_value = 'cycle';
        }
        update_post_meta($variation_id, '_bogo_calculation_type', $calculation_type_value);
    }
}
add_action('woocommerce_save_product_variation', 'ing_save_bogo_variation_meta', 50, 2);

/**
 * Get all products with BOGO enabled for admin overview
 */
function ing_get_all_bogo_products()
{
    $bogo_products = array();

    // Get all published products
    $args = array(
        'post_type' => 'product',
        'posts_per_page' => -1,
        'post_status' => 'publish'
    );

    $products = get_posts($args);

    foreach ($products as $product_post) {
        $product = wc_get_product($product_post->ID);

        if ($product->is_type('simple')) {
            // Check simple product for BOGO
            if (get_post_meta($product->get_id(), '_bogo_enabled', true) === 'yes') {
                $bogo_products[] = array(
                    'id' => $product->get_id(),
                    'name' => $product->get_name(),
                    'type' => 'simple',
                    'sku' => $product->get_sku(),
                    'price' => $product->get_price(),
                    'regular_price' => $product->get_regular_price(),
                    'sale_price' => $product->get_sale_price(),
                    'buy_qty' => intval(get_post_meta($product->get_id(), '_bogo_buy_qty', true)) ?: 2,
                    'free_qty' => intval(get_post_meta($product->get_id(), '_bogo_free_qty', true)) ?: 1,
                    'calculation_type' => get_post_meta($product->get_id(), '_bogo_calculation_type', true) ?: 'cycle',
                    'stock' => $product->get_stock_quantity(),
                    'stock_status' => $product->get_stock_status(),
                    'permalink' => get_permalink($product->get_id()),
                    'edit_link' => get_edit_post_link($product->get_id())
                );
            }
        } elseif ($product->is_type('variable')) {
            // Check variations for BOGO
            $children = $product->get_children();
            foreach ($children as $variation_id) {
                if (get_post_meta($variation_id, '_bogo_enabled', true) === 'yes') {
                    $variation = wc_get_product($variation_id);
                    $bogo_products[] = array(
                        'id' => $variation_id,
                        'parent_id' => $product->get_id(),
                        'name' => $product->get_name() . ' - ' . $variation->get_attribute_summary(),
                        'type' => 'variation',
                        'sku' => $variation->get_sku(),
                        'price' => $variation->get_price(),
                        'regular_price' => $variation->get_regular_price(),
                        'sale_price' => $variation->get_sale_price(),
                        'buy_qty' => intval(get_post_meta($variation_id, '_bogo_buy_qty', true)) ?: 2,
                        'free_qty' => intval(get_post_meta($variation_id, '_bogo_free_qty', true)) ?: 1,
                        'calculation_type' => get_post_meta($variation_id, '_bogo_calculation_type', true) ?: 'cycle',
                        'stock' => $variation->get_stock_quantity(),
                        'stock_status' => $variation->get_stock_status(),
                        'permalink' => get_permalink($product->get_id()),
                        'edit_link' => get_edit_post_link($product->get_id())
                    );
                }
            }
        }
    }

    return $bogo_products;
}
