<?php 

/**
 * Plugin Name:       Infinity Testimonial for Elementor & Gutenberg
 * Plugin URI:        https://devnahian.com/ifs-testimonial
 * Description:       Infinity testimonial for Elementor & Gutenberg. A high-performance, modular testimonial engine featuring dynamic group filtering and pro slider capabilities.
 * Version:           1.2.0
 * Author:            Abdullah Nahian
 * Author URI:        https://devnahian.com
 * Text Domain:       infinity-testimonials
 * Domain Path:       /languages
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Requires at least: 5.8
 * Requires PHP:      7.4
 */

if (!defined('ABSPATH')) exit;

// Define Path Constants
define('IFS_PATH', plugin_dir_path(__FILE__));
define('IFS_URL', plugin_dir_url(__FILE__));

/**
 * 1. CORE INITIALIZATION
 * Registers the Custom Post Type and Taxonomy.
 */
add_action('init', function() {
    register_post_type('ifs_testimonial', [
        'label' => 'Testimonials',
        'public' => true,
        'show_ui' => true, // Set to true to allow standard editing
        'show_in_menu' => false, // Hidden from main sidebar to use custom UI
        'supports' => ['title', 'editor', 'thumbnail', 'excerpt'],
        'has_archive' => false,
    ]);

    register_taxonomy('testimonial_cat', 'ifs_testimonial', [
        'hierarchical' => true,
        'label' => 'Categories',
        'show_ui' => true,
        'show_in_menu' => false,
    ]);
});

/**
 * 2. ADMIN MENU
 * Creates the custom "IFS Pro" top-level menu.
 */
add_action('admin_menu', function() {
    add_menu_page(
        'IFS Pro', 
        'IFS Testimonial', 
        'manage_options', 
        'ifs-testimonial', 
        'ifs_main_router', 
        'dashicons-testimonial', 
        2
    );
});

/**
 * 3. PREMIUM UI STYLES
 * Injected into the admin head for the custom dashboard look.
 */
add_action('admin_head', function() {
    ?>
    <style>
        :root { 
            --side-bg: #0f172a; 
            --side-hover: #1e293b;
            --accent: #3b82f6; 
            --accent-glow: rgba(59, 130, 246, 0.4);
            --bg: #f8fafc; 
            --border: #e2e8f0; 
            --text-dim: #94a3b8;
        }
        
        /* Layout Fixes for WordPress Admin */
        #wpcontent { padding-left: 0 !important; background: var(--bg); }
        #wpfooter { display: none; }
        .update-nag, .notice { margin-left: 20px !important; }

        .ifs-wrapper { display: flex; min-height: 100vh; font-family: 'Inter', -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif; }
        
        /* Sidebar Styling */
        .ifs-sidebar { width: 260px; background: var(--side-bg); color: #fff; flex-shrink: 0; display: flex; flex-direction: column; box-shadow: 5px 0 15px rgba(0,0,0,0.1); }
        .ifs-logo { padding: 40px 25px; font-weight: 800; font-size: 20px; color: #fff; letter-spacing: -1px; border-bottom: 1px solid #1e293b; }
        .ifs-nav-group { padding: 20px 15px; }
        
        .ifs-nav-link { 
            display: flex; align-items: center; padding: 14px 18px; color: var(--text-dim); 
            text-decoration: none; border-radius: 12px; margin-bottom: 8px; transition: 0.3s; font-weight: 500;
        }
        .ifs-nav-link svg { width: 20px; height: 20px; margin-right: 12px; stroke: currentColor; stroke-width: 2; fill: none; transition: 0.3s; }
        .ifs-nav-link:hover { background: var(--side-hover); color: #fff; }
        .ifs-nav-link.active { background: var(--accent); color: #fff; box-shadow: 0 4px 12px var(--accent-glow); }
        .ifs-nav-link.active svg { stroke: #fff; }
        
        /* Content Area */
        .ifs-main { flex-grow: 1; padding: 45px 60px; }
        .ifs-card { background: #fff; border-radius: 16px; padding: 35px; border: 1px solid var(--border); box-shadow: 0 4px 6px -1px rgba(0,0,0,0.05); }
        .ifs-sub-nav { display: flex; gap: 30px; border-bottom: 2px solid var(--border); margin-bottom: 40px; }
        .ifs-sub-tab { padding-bottom: 15px; text-decoration: none; color: #64748b; font-weight: 600; border-bottom: 3px solid transparent; transition: 0.2s; }
        .ifs-sub-tab.active { color: var(--accent); border-bottom-color: var(--accent); }
        
        /* Custom Toggle Switch */
        .switch { position: relative; display: inline-block; width: 44px; height: 24px; }
        .switch input { opacity: 0; width: 0; height: 0; }
        .slider { position: absolute; cursor: pointer; top: 0; left: 0; right: 0; bottom: 0; background: #cbd5e1; transition: .4s; border-radius: 24px; }
        .slider:before { position: absolute; content: ""; height: 18px; width: 18px; left: 3px; bottom: 3px; background: white; transition: .4s; border-radius: 50%; }
        input:checked + .slider { background: var(--accent); }
        input:checked + .slider:before { transform: translateX(20px); }
    </style>
    <?php
});

/**
 * 4. ENQUEUE ASSETS
 */
add_action('admin_enqueue_scripts', function($hook) {
    if ($hook !== 'toplevel_page_ifs-testimonial') return;
    wp_enqueue_media();
});

/**
 * 5. MAIN ROUTER & NAVIGATION
 * Handles the logic for switching between Dashboard, Testimonials, Categories, and Settings.
 */
function ifs_main_router() {
    $tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'overview';
    ?>
    <div class="ifs-wrapper">
        <div class="ifs-sidebar">
            <div class="ifs-logo">IFS PRO</div>
            <nav class="ifs-nav-group">
                <a href="?page=ifs-testimonial&tab=overview" class="ifs-nav-link <?php echo $tab === 'overview' ? 'active' : ''; ?>">
                    <svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path d="M3.75 6A2.25 2.25 0 016 3.75h2.25A2.25 2.25 0 0110.5 6v2.25a2.25 2.25 0 01-2.25 2.25H6a2.25 2.25 0 01-2.25-2.25V6zM3.75 15.75A2.25 2.25 0 016 13.5h2.25a2.25 2.25 0 012.25 2.25V18a2.25 2.25 0 01-2.25 2.25H6A2.25 2.25 0 013.75 18v-2.25zM13.5 6a2.25 2.25 0 012.25-2.25H18A2.25 2.25 0 0120.25 6v2.25A2.25 2.25 0 0118 10.5h-2.25a2.25 2.25 0 01-2.25-2.25V6zM13.5 15.75a2.25 2.25 0 012.25-2.25H18a2.25 2.25 0 012.25 2.25V18A2.25 2.25 0 0118 20.25h-2.25A2.25 2.25 0 0113.5 18v-2.25z" stroke-linecap="round" stroke-linejoin="round"/></svg>
                    Dashboard
                </a>
                <a href="?page=ifs-testimonial&tab=testimonials" class="ifs-nav-link <?php echo $tab === 'testimonials' ? 'active' : ''; ?>">
                    <svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path d="M7.5 8.25h9m-9 3H12m-9.75 1.51c0 1.6 1.123 2.994 2.707 3.227 1.129.166 2.27.293 3.423.379.35.026.67.21.865.501L12 21l2.755-4.133a1.14 1.14 0 01.865-.501 48.172 48.172 0 003.423-.379c1.584-.233 2.707-1.626 2.707-3.228V6.741c0-1.602-1.123-2.995-2.707-3.228A48.394 48.394 0 0012 3c-2.392 0-4.744.175-7.043.513C3.373 3.746 2.25 5.14 2.25 6.741v6.018z" stroke-linecap="round" stroke-linejoin="round"/></svg>
                    Testimonials
                </a>
                <a href="?page=ifs-testimonial&tab=categories" class="ifs-nav-link <?php echo $tab === 'categories' ? 'active' : ''; ?>">
                    <svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path d="M9.568 3H5.25A2.25 2.25 0 003 5.25v4.318c0 .597.237 1.17.659 1.591l9.581 9.581a2.25 2.25 0 003.182 0l4.318-4.318a2.25 2.25 0 000-3.182L11.159 3.659A2.25 2.25 0 009.568 3z" stroke-linecap="round" stroke-linejoin="round"/><path d="M6 6h.008v.008H6V6z" stroke-linecap="round" stroke-linejoin="round"/></svg>
                    Categories
                </a>
                <a href="?page=ifs-testimonial&tab=settings" class="ifs-nav-link <?php echo $tab === 'settings' ? 'active' : ''; ?>">
                    <svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path d="M4.5 12a7.5 7.5 0 1115 0 7.5 7.5 0 01-15 0z" stroke-linecap="round" stroke-linejoin="round"/><path d="M12 9v3m0 0v3m0-3h3m-3 0H9" stroke-linecap="round" stroke-linejoin="round"/></svg>
                    Settings
                </a>
            </nav>
        </div>
        <div class="ifs-main">
            <?php
            // Modular Routing
            switch($tab) {
                case 'testimonials': 
                    if (file_exists(IFS_PATH . 'admin/testimonials/tabs.php')) {
                        include IFS_PATH . 'admin/testimonials/tabs.php';
                    } else {
                        echo '<div class="ifs-card">Testimonials Module Not Found.</div>';
                    }
                    break;
                case 'categories':   
                    if (file_exists(IFS_PATH . 'admin/categories/tabs.php')) {
                        include IFS_PATH . 'admin/categories/tabs.php';
                    } else {
                        echo '<div class="ifs-card">Categories Module Not Found.</div>';
                    }
                    break;
                case 'settings':     
                    if (file_exists(IFS_PATH . 'admin/settings.php')) {
                        include IFS_PATH . 'admin/settings.php';
                    } else {
                        echo '<div class="ifs-card">Settings Module Not Found.</div>';
                    }
                    break;
                default: 
                    if (file_exists(IFS_PATH . 'admin/dashboard.php')) {
                        include IFS_PATH . 'admin/dashboard.php';
                    } else {
                        echo '<div class="ifs-card"><h1>Welcome to IFS Pro</h1><p>Start by creating your first testimonial!</p></div>';
                    }
                    break;
            }
            ?>
        </div>
    </div>
    <?php
}

/**
 * 6. ELEMENTOR WIDGETS REGISTRATION
 */
add_action('elementor/widgets/register', function($widgets_manager) {
    $widget_files = [
        'ifs-testimonial.php',
    ];

    foreach ($widget_files as $file) {
        $path = IFS_PATH . 'elementor/' . $file;
        if (file_exists($path)) {
            require_once($path);
        }
    }

    // Register instances
    if (class_exists('\IFS_Testimonial')) $widgets_manager->register(new \IFS_Testimonial());
});

/**
 * 7. CUSTOM ELEMENTOR CATEGORY
 */
add_action('elementor/elements/categories_registered', function($elements_manager) {
    $elements_manager->add_category(
        'ifs-category',
        [
            'title' => esc_html__('IFS Widgets', 'infinity-testimonials'),
            'icon'  => 'fa fa-plug',
        ]
    );
});


