<?php
/**
 * Script per compilare i file di traduzione .po in .mo
 *
 * Uso: php compile-languages.php
 *
 * @package DarkModeSwitcherUniversal
 * @since 1.0.0
 */
if (!defined('ABSPATH')) exit;
// Prevenire esecuzione via web
if (isset($_SERVER['HTTP_HOST'])) {
    die('Questo script deve essere eseguito da command line.');
}

echo "🌍 Compilazione file traduzioni Improve Accessibility - Dark Mode Switcher\n";
echo "========================================================\n\n";

/**
 * Funzione per compilare file .po in .mo
 */
function compile_po_to_mo($po_file, $mo_file) {
    if (!file_exists($po_file)) {
        printf("❌ File non trovato: %s\n", esc_html($po_file));
        return false;
    }

    // Leggi il file .po
    $po_content = file_get_contents($po_file);
    if (!$po_content) {
        printf("❌ Impossibile leggere: %s\n", esc_html($po_file));
        return false;
    }

    // Parse delle stringhe
    $translations = array();
    $current_msgid = '';
    $current_msgstr = '';
    $in_msgid = false;
    $in_msgstr = false;

    $lines = explode("\n", $po_content);

    foreach ($lines as $line) {
        $line = trim($line);

        // Skip commenti e linee vuote
        if (empty($line) || $line[0] === '#') {
            continue;
        }

        // Msgid
        if (strpos($line, 'msgid ') === 0) {
            // Salva la traduzione precedente se esiste
            if (!empty($current_msgid) && !empty($current_msgstr)) {
                $translations[$current_msgid] = $current_msgstr;
            }

            $current_msgid = substr($line, 7, -1); // Rimuovi 'msgid "' e '"'
            $current_msgstr = '';
            $in_msgid = true;
            $in_msgstr = false;
            continue;
        }

        // Msgstr
        if (strpos($line, 'msgstr ') === 0) {
            $current_msgstr = substr($line, 8, -1); // Rimuovi 'msgstr "' e '"'
            $in_msgid = false;
            $in_msgstr = true;
            continue;
        }

        // Linee di continuazione
        if ($line[0] === '"' && $line[-1] === '"') {
            $content = substr($line, 1, -1);

            if ($in_msgid) {
                $current_msgid .= $content;
            } elseif ($in_msgstr) {
                $current_msgstr .= $content;
            }
        }
    }

    // Aggiungi l'ultima traduzione
    if (!empty($current_msgid) && !empty($current_msgstr)) {
        $translations[$current_msgid] = $current_msgstr;
    }

    // Crea file .mo
    if (create_mo_file($translations, $mo_file)) {
        printf("✅ Compilato: %s\n", esc_html($po_file));
        echo "   📊 " . count($translations) . " traduzioni\n";
        return true;
    } else {
        printf("❌ Errore nella compilazione: %s\n", esc_html($mo_file));
        return false;
    }
}

/**
 * Crea file .mo binario
 */
function create_mo_file($translations, $mo_file) {
    // Header MO file
    $mo_data = pack('V', 0x950412de); // Magic number
    $mo_data .= pack('V', 0);          // Version
    $mo_data .= pack('V', count($translations)); // Numero stringhe
    $mo_data .= pack('V', 28);         // Offset tabella originali
    $mo_data .= pack('V', 28 + count($translations) * 8); // Offset tabella traduzioni
    $mo_data .= pack('V', 0);          // Hash table size
    $mo_data .= pack('V', 0);          // Hash table offset

    // Prepara le stringhe
    $originals = array_keys($translations);
    $translations_array = array_values($translations);

    // Calcola offset per le stringhe
    $strings_offset = 28 + count($translations) * 16;

    // Tabella originali
    $current_offset = $strings_offset;
    foreach ($originals as $original) {
        $mo_data .= pack('V', strlen($original)); // Lunghezza
        $mo_data .= pack('V', $current_offset);   // Offset
        $current_offset += strlen($original) + 1;
    }

    // Tabella traduzioni
    foreach ($translations_array as $translation) {
        $mo_data .= pack('V', strlen($translation)); // Lunghezza
        $mo_data .= pack('V', $current_offset);      // Offset
        $current_offset += strlen($translation) + 1;
    }

    // Stringhe originali
    foreach ($originals as $original) {
        $mo_data .= $original . "\0";
    }

    // Stringhe tradotte
    foreach ($translations_array as $translation) {
        $mo_data .= $translation . "\0";
    }

    // Salva file
    return file_put_contents($mo_file, $mo_data) !== false;
}

/**
 * Funzione principale
 */
function main() {
    $languages_dir = __DIR__;

    // File da compilare
    $files_to_compile = array(
        'dark-mode-switcher-it_IT.po' => 'dark-mode-switcher-it_IT.mo',
        'dark-mode-switcher-en_US.po' => 'dark-mode-switcher-en_US.mo'
    );

    $compiled = 0;
    $total = count($files_to_compile);

    foreach ($files_to_compile as $po_file => $mo_file) {
        $po_path = $languages_dir . '/' . $po_file;
        $mo_path = $languages_dir . '/' . $mo_file;

        printf( "🔄 Compilazione: %s\n", esc_html($po_file));;

        if (compile_po_to_mo($po_path, $mo_path)) {
            $compiled++;
        }

        echo "\n";
    }

    echo "========================================================\n";
    printf("📊 Riepilogo: %s\n", esc_html($compiled/$total));

    if ($compiled === $total) {
        echo "🎉 Tutte le traduzioni sono state compilate!\n";
        echo "\n📁 File generati nella cartella languages/:\n";

        foreach ($files_to_compile as $po_file => $mo_file) {
            $mo_path = $languages_dir . '/' . $mo_file;
            if (file_exists($mo_path)) {
                $size = filesize($mo_path);
                printf("   • %s (%s bytes)\n", esc_html($mo_file), esc_html(number_format($size)));
            }
        }

        echo "\n✨ Il plugin è ora pronto per il multilingual!\n";
        return true;
    } else {
        echo "⚠️ Alcune compilazioni sono fallite.\n";
        return false;
    }
}

// Esegui lo script
if (!main()) {
    exit(1);
}

echo "\n🌍 Script completato con successo!\n";
echo "💡 Suggerimento: Puoi aggiungere altre lingue creando nuovi file .po\n";
echo "   e aggiungendoli all'array \$files_to_compile in questo script.\n";
?>