<?php
/**
 * Classe per la gestione delle API REST
 *
 * @package DarkModeSwitcherUniversal
 * @since 1.0.0
 */

// Prevengo accesso diretto
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Classe IADS_API
 * Gestisce le API REST del plugin
 */
class IADS_API {

    /**
     * Istanza singleton
     */
    private static $instance = null;

    /**
     * Namespace API
     */
    private $namespace = 'iads/v1';

    /**
     * Ottieni istanza singleton
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Costruttore privato per singleton
     */
    private function __construct() {
        $this->init();
    }

    /**
     * Inizializzazione
     */
    public function init() {
        add_action('rest_api_init', array($this, 'register_routes'));
        add_filter('rest_pre_serve_request', array($this, 'add_cors_headers'), 10, 4);
    }

    /**
     * Registra tutte le route API
     */
    public function register_routes() {
        // GET /wp-json/iads/v1/theme - tema corrente
        register_rest_route($this->namespace, '/theme', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_theme'),
            'permission_callback' => '__return_true',
            'args' => array(
                'user_id' => array(
                    'type' => 'integer',
                    'required' => false,
                    'sanitize_callback' => 'absint'
                ),
                'session_id' => array(
                    'type' => 'string',
                    'required' => false,
                    'sanitize_callback' => 'sanitize_text_field'
                )
            )
        ));

        // POST /wp-json/iads/v1/theme - Salva tema
        register_rest_route($this->namespace, '/theme', array(
            'methods' => 'POST',
            'callback' => array($this, 'save_theme'),
            'permission_callback' => array($this, 'check_theme_permission'),
            'args' => array(
                'theme' => array(
                    'type' => 'string',
                    'required' => true,
                    'enum' => array('light', 'dark', 'auto'),
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'user_id' => array(
                    'type' => 'integer',
                    'required' => false,
                    'sanitize_callback' => 'absint'
                ),
                'session_id' => array(
                    'type' => 'string',
                    'required' => false,
                    'sanitize_callback' => 'sanitize_text_field'
                )
            )
        ));

        // GET /wp-json/iads/v1/settings - impostazioni (solo admin)
        register_rest_route($this->namespace, '/settings', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_settings'),
            'permission_callback' => array($this, 'check_admin_permission'),
        ));

        // POST /wp-json/iads/v1/settings - Salva impostazioni (solo admin)
        register_rest_route($this->namespace, '/settings', array(
            'methods' => 'POST',
            'callback' => array($this, 'save_settings'),
            'permission_callback' => array($this, 'check_admin_permission'),
            'args' => array(
                'settings' => array(
                    'type' => 'object',
                    'required' => true,
                    'validate_callback' => array($this, 'validate_settings')
                )
            )
        ));

        // GET /wp-json/iads/v1/analytics - Statistiche (solo admin)
        register_rest_route($this->namespace, '/analytics', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_analytics'),
            'permission_callback' => array($this, 'check_admin_permission'),
            'args' => array(
                'days' => array(
                    'type' => 'integer',
                    'required' => false,
                    'default' => 30,
                    'sanitize_callback' => 'absint'
                )
            )
        ));

        // POST /wp-json/iads/v1/analytics/event - Registra evento
        register_rest_route($this->namespace, '/analytics/event', array(
            'methods' => 'POST',
            'callback' => array($this, 'record_event'),
            'permission_callback' => array($this, 'check_event_permission'),
            'args' => array(
                'event_type' => array(
                    'type' => 'string',
                    'required' => true,
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'event_data' => array(
                    'type' => 'object',
                    'required' => false,
                    'default' => array()
                )
            )
        ));

        // GET /wp-json/iads/v1/status - Status del plugin
        register_rest_route($this->namespace, '/status', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_status'),
            'permission_callback' => '__return_true',
        ));

        // POST /wp-json/iads/v1/donate - Genera URL donazione
        register_rest_route($this->namespace, '/donate', array(
            'methods' => 'POST',
            'callback' => array($this, 'generate_donation_url'),
            'permission_callback' => '__return_true',
            'args' => array(
                'amount' => array(
                    'type' => 'string',
                    'required' => false,
                    'default' => '5.00',
                    'validate_callback' => array($this, 'validate_donation_amount')
                ),
                'currency' => array(
                    'type' => 'string',
                    'required' => false,
                    'default' => 'EUR',
                    'enum' => array('EUR', 'USD', 'GBP', 'CAD', 'AUD')
                )
            )
        ));

        // DELETE /wp-json/iads/v1/user-data - Cancella dati utente (GDPR)
        register_rest_route($this->namespace, '/user-data', array(
            'methods' => 'DELETE',
            'callback' => array($this, 'delete_user_data'),
            'permission_callback' => array($this, 'check_user_data_permission'),
            'args' => array(
                'user_id' => array(
                    'type' => 'integer',
                    'required' => true,
                    'sanitize_callback' => 'absint'
                )
            )
        ));

        // GET /wp-json/iads/v1/export-user-data - Esporta dati utente (GDPR)
        register_rest_route($this->namespace, '/export-user-data', array(
            'methods' => 'GET',
            'callback' => array($this, 'export_user_data'),
            'permission_callback' => array($this, 'check_user_data_permission'),
            'args' => array(
                'user_id' => array(
                    'type' => 'integer',
                    'required' => true,
                    'sanitize_callback' => 'absint'
                )
            )
        ));
    }

    /**
     * Ottieni tema corrente
     */
    public function get_theme($request) {
        $user_id = $request->get_param('user_id') ?: get_current_user_id();
        $session_id = $request->get_param('session_id');

        // Ottieni da database se disponibile
        $db = IADS_Database::getInstance();
        $preference = $db->get_user_preference($user_id, $session_id);

        if ($preference) {
            $theme = $preference->theme_preference;
        } else {
            // Fallback alle impostazioni predefinite
            $theme = 'light';

            // Controlla se il browser supporta dark mode
            if (isset($_SERVER['HTTP_SEC_CH_PREFERS_COLOR_SCHEME'])) {
                $theme = $_SERVER['HTTP_SEC_CH_PREFERS_COLOR_SCHEME'] === 'dark' ? 'dark' : 'light';
            }
        }

        // Registra evento
        $db->record_analytics_event('theme_get', array(
            'theme' => $theme,
            'user_id' => $user_id,
            'session_id' => $session_id
        ));

        return new WP_REST_Response(array(
            'theme' => $theme,
            'timestamp' => current_time('timestamp'),
            'user_id' => $user_id,
            'session_id' => $session_id
        ), 200);
    }

    /**
     * Salva tema
     */
    public function save_theme($request) {
        $theme = $request->get_param('theme');
        $user_id = $request->get_param('user_id') ?: get_current_user_id();
        $session_id = $request->get_param('session_id');

        // Validazione tema
        if (!in_array($theme, array('light', 'dark', 'auto'))) {
            return new WP_Error('invalid_theme', 'Tema non valido', array('status' => 400));
        }

        // Salva nel database
        $db = IADS_Database::getInstance();
        $result = $db->save_user_preference($user_id, $theme, $session_id);

        if (!$result) {
            return new WP_Error('save_failed', 'Errore nel salvataggio del tema', array('status' => 500));
        }

        // Registra evento
        $db->record_analytics_event('theme_save', array(
            'theme' => $theme,
            'user_id' => $user_id,
            'session_id' => $session_id
        ));

        return new WP_REST_Response(array(
            'success' => true,
            'theme' => $theme,
            'message' => __('Tema salvato con successo', 'improve-accessibility-dark-mode-switcher'),
            'timestamp' => current_time('timestamp')
        ), 200);
    }

    /**
     * Ottieni impostazioni plugin
     */
    public function get_settings($request) {
        $options = get_option('iads_options', array());

        // Rimuovi dati sensibili dalla risposta pubblica
        if (isset($options['paypal_email'])) {
            $options['paypal_email'] = $this->mask_email($options['paypal_email']);
        }

        return new WP_REST_Response(array(
            'settings' => $options,
            'version' => IADS_VERSION,
            'timestamp' => current_time('timestamp')
        ), 200);
    }

    /**
     * Salva impostazioni plugin
     */
    public function save_settings($request) {
        $settings = $request->get_param('settings');
        $user_id = get_current_user_id();

        // Ottieni impostazioni attuali per storia
        $old_settings = get_option('iads_options', array());

        // Sanitizza le impostazioni
        $security = IADS_Security::getInstance();
        $sanitized_settings = array();

        foreach ($settings as $key => $value) {
            $sanitized_settings[$key] = $security->sanitize_option($value, $key);
        }

        // Salva le impostazioni
        $result = update_option('iads_options', $sanitized_settings);

        if (!$result && get_option('iads_options') !== $sanitized_settings) {
            return new WP_Error('save_failed', 'Errore nel salvataggio delle impostazioni', array('status' => 500));
        }

        // Salva nella storia
        $db = IADS_Database::getInstance();
        $db->save_settings_history($user_id, 'save', $old_settings, $sanitized_settings);

        // Registra evento
        $db->record_analytics_event('settings_save', array(
            'user_id' => $user_id,
            'settings_count' => count($sanitized_settings)
        ));

        return new WP_REST_Response(array(
            'success' => true,
            'message' => __('Impostazioni salvate con successo', 'improve-accessibility-dark-mode-switcher'),
            'settings' => $sanitized_settings,
            'timestamp' => current_time('timestamp')
        ), 200);
    }

    /**
     * Ottieni statistiche analytics
     */
    public function get_analytics($request) {
        $days = $request->get_param('days') ?: 30;

        $db = IADS_Database::getInstance();
        $stats = $db->get_analytics_stats($days);

        return new WP_REST_Response(array(
            'success' => true,
            'stats' => $stats,
            'period_days' => $days,
            'timestamp' => current_time('timestamp')
        ), 200);
    }

    /**
     * Registra evento analytics
     */
    public function record_event($request) {
        $event_type = $request->get_param('event_type');
        $event_data = $request->get_param('event_data') ?: array();

        // Validazione tipo evento
        $allowed_events = array(
            'theme_toggle',
            'donation_click',
            'toggle_interaction',
            'settings_view',
            'error_occurred'
        );

        if (!in_array($event_type, $allowed_events)) {
            return new WP_Error('invalid_event', 'Tipo evento non valido', array('status' => 400));
        }

        // Registra evento
        $db = IADS_Database::getInstance();
        $result = $db->record_analytics_event($event_type, $event_data);

        if (!$result) {
            return new WP_Error('record_failed', 'Errore nella registrazione evento', array('status' => 500));
        }

        return new WP_REST_Response(array(
            'success' => true,
            'message' => __('Evento registrato con successo', 'improve-accessibility-dark-mode-switcher'),
            'event_type' => $event_type,
            'timestamp' => current_time('timestamp')
        ), 200);
    }

    /**
     * Ottieni status del plugin
     */
    public function get_status($request) {
        // Controlla integrità database
        $db = IADS_Database::getInstance();
        $db_integrity = $db->check_database_integrity();

        // Controlla integrità file
        $security = IADS_Security::getInstance();
        $file_integrity = $security->verify_plugin_integrity();

        // Ottieni statistiche base
        $stats = $db->get_analytics_stats(7); // Ultimi 7 giorni

        return new WP_REST_Response(array(
            'plugin_version' => IADS_VERSION,
            'wordpress_version' => get_bloginfo('version'),
            'php_version' => PHP_VERSION,
            'database_integrity' => is_array($db_integrity) ? false : $db_integrity,
            'file_integrity' => is_array($file_integrity) ? false : $file_integrity,
            'total_users' => $stats['unique_users'] ?? 0,
            'total_events_week' => $stats['total_events'] ?? 0,
            'status' => 'healthy',
            'timestamp' => current_time('timestamp')
        ), 200);
    }

    /**
     * Genera URL donazione PayPal
     */
    public function generate_donation_url($request) {
        $amount = $request->get_param('amount') ?: '5.00';
        $currency = $request->get_param('currency') ?: 'EUR';

        // Ottieni email PayPal dalle impostazioni
        $options = get_option('iads_options', array());
        $paypal_email = $options['paypal_email'] ?? '';

        if (empty($paypal_email)) {
            return new WP_Error('no_paypal_email', 'Email PayPal non configurata', array('status' => 400));
        }

        // Genera URL donazione
        $params = array(
            'business' => $paypal_email,
            'item_name' => __('Dark Mode Switcher Plugin - Caffè per lo sviluppatore', 'improve-accessibility-dark-mode-switcher'),
            'amount' => $amount,
            'currency_code' => $currency,
            'no_shipping' => '1',
            'return' => home_url(),
            'cancel_return' => home_url()
        );

        $donation_url = 'https://www.paypal.com/donate/?' . http_build_query($params);

        // Registra evento
        $db = IADS_Database::getInstance();
        $db->record_analytics_event('donation_url_generated', array(
            'amount' => $amount,
            'currency' => $currency
        ));

        return new WP_REST_Response(array(
            'success' => true,
            'donation_url' => $donation_url,
            'amount' => $amount,
            'currency' => $currency,
            'timestamp' => current_time('timestamp')
        ), 200);
    }

    /**
     * Cancella dati utente (GDPR)
     */
    public function delete_user_data($request) {
        $user_id = $request->get_param('user_id');

        // Verifica che l'utente esista
        $user = get_user_by('id', $user_id);
        if (!$user) {
            return new WP_Error('user_not_found', 'Utente non trovato', array('status' => 404));
        }

        // Cancella dati utente
        $db = IADS_Database::getInstance();
        $db->delete_user_data($user_id);

        // Registra evento
        $db->record_analytics_event('user_data_deleted', array(
            'deleted_user_id' => $user_id,
            'deleted_by' => get_current_user_id()
        ));

        return new WP_REST_Response(array(
            'success' => true,
            'message' => __('Dati utente cancellati con successo', 'improve-accessibility-dark-mode-switcher'),
            'user_id' => $user_id,
            'timestamp' => current_time('timestamp')
        ), 200);
    }

    /**
     * Esporta dati utente (GDPR)
     */
    public function export_user_data($request) {
        $user_id = $request->get_param('user_id');

        // Verifica che l'utente esista
        $user = get_user_by('id', $user_id);
        if (!$user) {
            return new WP_Error('user_not_found', 'Utente non trovato', array('status' => 404));
        }

        // Esporta dati utente
        $db = IADS_Database::getInstance();
        $user_data = $db->export_user_data($user_id);

        // Registra evento
        $db->record_analytics_event('user_data_exported', array(
            'exported_user_id' => $user_id,
            'exported_by' => get_current_user_id()
        ));

        return new WP_REST_Response(array(
            'success' => true,
            'user_data' => $user_data,
            'user_id' => $user_id,
            'export_date' => current_time('mysql'),
            'timestamp' => current_time('timestamp')
        ), 200);
    }

    /**
     * Controllo permessi tema
     */
    public function check_theme_permission($request) {
        // Tutti possono salvare il proprio tema
        return true;
    }

    /**
     * Controllo permessi admin
     */
    public function check_admin_permission($request) {
        return current_user_can('manage_options');
    }

    /**
     * Controllo permessi eventi
     */
    public function check_event_permission($request) {
        // Tutti possono registrare eventi (con rate limiting)
        return true;
    }

    /**
     * Controllo permessi dati utente
     */
    public function check_user_data_permission($request) {
        $user_id = $request->get_param('user_id');
        $current_user_id = get_current_user_id();

        // L'utente può gestire i propri dati o admin può gestire tutti
        return ($current_user_id === $user_id) || current_user_can('manage_options');
    }

    /**
     * Validazione impostazioni
     */
    public function validate_settings($settings, $request, $param) {
        if (!is_array($settings)) {
            return new WP_Error('invalid_settings', 'Le impostazioni devono essere un oggetto', array('status' => 400));
        }

        // Validazioni specifiche
        $allowed_keys = array(
            'toggle_position',
            'toggle_style',
            'paypal_email',
            'show_donation',
            'custom_css'
        );

        foreach ($settings as $key => $value) {
            if (!in_array($key, $allowed_keys)) {
                return new WP_Error('invalid_setting_key', "Chiave impostazione non valida: $key", array('status' => 400));
            }
        }

        return true;
    }

    /**
     * Validazione importo donazione
     */
    public function validate_donation_amount($amount, $request, $param) {
        // Deve essere un numero decimale valido
        if (!preg_match('/^\d+(\.\d{1,2})?$/', $amount)) {
            return new WP_Error('invalid_amount', 'Importo donazione non valido', array('status' => 400));
        }

        $amount_float = floatval($amount);

        // Importo minimo e massimo
        if ($amount_float < 1.00 || $amount_float > 1000.00) {
            return new WP_Error('amount_out_of_range', 'Importo deve essere tra 1.00 e 1000.00', array('status' => 400));
        }

        return true;
    }

    /**
     * Maschere email per sicurezza
     */
    private function mask_email($email) {
        if (empty($email)) {
            return '';
        }

        $parts = explode('@', $email);
        if (count($parts) !== 2) {
            return '***';
        }

        $username = $parts[0];
        $domain = $parts[1];

        $masked_username = substr($username, 0, 2) . str_repeat('*', strlen($username) - 2);

        return $masked_username . '@' . $domain;
    }

    /**
     * Aggiungi header CORS
     */
    public function add_cors_headers($served, $result, $request, $server) {
        // Solo per le nostre API
        if (strpos($request->get_route(), '/iads/v1/') !== false) {
            header('Access-Control-Allow-Origin: ' . home_url());
            header('Access-Control-Allow-Methods: GET, POST, DELETE, OPTIONS');
            header('Access-Control-Allow-Headers: Content-Type, Authorization, X-WP-Nonce');
            header('Access-Control-Allow-Credentials: true');

            // Gestisci richieste OPTIONS (preflight)
            if ($request->get_method() === 'OPTIONS') {
                header('HTTP/1.1 200 OK');
                exit();
            }
        }

        return $served;
    }

    /**
     * Middleware per rate limiting
     */
    public function rate_limiting_middleware($request) {
        $route = $request->get_route();
        $client_ip = $this->get_client_ip();
        $user_id = get_current_user_id();

        // Rate limits per route
        $limits = array(
            '/iads/v1/theme' => array('requests' => 100, 'window' => 3600), // 100/ora
            '/iads/v1/analytics/event' => array('requests' => 200, 'window' => 3600), // 200/ora
            '/iads/v1/settings' => array('requests' => 20, 'window' => 3600), // 20/ora
        );

        foreach ($limits as $pattern => $limit) {
            if (strpos($route, $pattern) !== false) {
                if (!$this->check_rate_limit($client_ip, $user_id, $pattern, $limit)) {
                    return new WP_Error('rate_limit_exceeded', 'Rate limit superato', array('status' => 429));
                }
                break;
            }
        }

        return true;
    }

    /**
     * Controlla rate limit
     */
    private function check_rate_limit($ip, $user_id, $endpoint, $limit) {
        $key = "iads_rate_limit_{$endpoint}_" . ($user_id ?: $ip);
        $current_time = time();

        $requests = get_transient($key) ?: array();

        // Rimuovi richieste vecchie
        $requests = array_filter($requests, function($timestamp) use ($current_time, $limit) {
            return ($current_time - $timestamp) < $limit['window'];
        });

        // Controlla limite
        if (count($requests) >= $limit['requests']) {
            return false;
        }

        // Aggiungi richiesta corrente
        $requests[] = $current_time;
        set_transient($key, $requests, $limit['window']);

        return true;
    }

    /**
     * Ottieni IP client
     */
    private function get_client_ip() {
        $ip_headers = array(
            'HTTP_CF_CONNECTING_IP',
            'HTTP_X_REAL_IP',
            'HTTP_X_FORWARDED_FOR',
            'REMOTE_ADDR'
        );

        foreach ($ip_headers as $header) {
            if (isset($_SERVER[$header]) && !empty($_SERVER[$header])) {
                $header_value = sanitize_text_field(wp_unslash($_SERVER[$header]));
				$ip_list = explode(',', $header_value);
				$ip = isset($ip_list[0]) ? trim($ip_list[0]) : '';
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return $ip;
                }
            }
        }

        //return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
		return isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '0.0.0.0';
    }

    /**
     * Sanitizza parametri API
     */
    public function sanitize_api_params($params) {
        $sanitized = array();

        foreach ($params as $key => $value) {
            switch ($key) {
                case 'theme':
                    $sanitized[$key] = in_array($value, array('light', 'dark', 'auto')) ? $value : 'light';
                    break;

                case 'user_id':
                    $sanitized[$key] = absint($value);
                    break;

                case 'session_id':
                case 'event_type':
                    $sanitized[$key] = sanitize_text_field($value);
                    break;

                case 'event_data':
                case 'settings':
                    $sanitized[$key] = is_array($value) ? array_map('sanitize_text_field', $value) : array();
                    break;

                default:
                    $sanitized[$key] = sanitize_text_field($value);
                    break;
            }
        }

        return $sanitized;
    }
}