<?php
/**
 * Plugin Name: Improve Accessibility - Dark Mode Switcher
 * Plugin URI: https://getwebsolution.it/dark-mode-switcher-universal
 * Description: Plugin universale per aggiungere un toggle dark/light mode elegante e sicuro al tuo sito WordPress con supporto localStorage e possibilità di donazione.
 * Version: 1.0.0
 * Author: Ermald Billa
 * Author URI: https://getwebsolution.it
 * License: GPL v2 or later
 * Text Domain: improve-accessibility-dark-mode-switcher
 * Domain Path: /languages
 */

// Prevengo accesso diretto
if (!defined('ABSPATH')) {
    exit;
}

// Definire costanti del plugin
define('IADS_PLUGIN_URL', plugin_dir_url(__FILE__));
define('IADS_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('IADS_VERSION', '1.0.0');
define('IADS_LOGO_URL', IADS_PLUGIN_URL . 'assets/images/logo.svg');

/**
 * Classe principale del plugin
 */
class IADS_Main {

    /**
     * Istanza singleton
     */
    private static $instance = null;

    /**
     * Ottenere istanza singleton
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Costruttore
     */
    private function __construct() {
        add_action('init', array($this, 'init'));
    }

    /**
     * Inizializzazione del plugin
     */
    public function init() {
        // Caricamento file di classe necessari
        $this->load_includes();

        // Caricamento traduzioni
        load_textdomain('improve-accessibility-dark-mode-switcher', false, dirname(plugin_basename(__FILE__)) . '/languages/');

        // Hook per frontend
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
        add_action('wp_footer', array($this, 'render_dark_mode_toggle'));

        // Hook per admin
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));

        // AJAX handlers con nonce di sicurezza
        add_action('wp_ajax_iads_save_settings', array($this, 'ajax_save_settings'));
        add_action('wp_ajax_iads_reset_settings', array($this, 'ajax_reset_settings'));
        add_action('wp_ajax_nopriv_iads_get_mode', array($this, 'ajax_get_mode'));
        add_action('wp_ajax_iads_get_mode', array($this, 'ajax_get_mode'));

        // Attivazione e disattivazione plugin
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }

    /**
     * Caricamento file includes
     */
    private function load_includes() {
        $includes = array(
            'includes/class-security.php',
            'includes/class-database.php',
            'includes/class-api.php'
        );

        foreach ($includes as $file) {
            $file_path = IADS_PLUGIN_PATH . $file;
            if (file_exists($file_path)) {
                require_once $file_path;
            }
        }

        // Inizializza le classi principali se esistono
        if (class_exists('IADS_Security')) {
            IADS_Security::getInstance();
        }

        if (class_exists('IADS_Database')) {
            IADS_Database::getInstance();
        }

        if (class_exists('IADS_API')) {
            IADS_API::getInstance();
        }
    }

    /**
     * Registrazione delle impostazioni
     */
    public function register_settings() {
		register_setting(
			'iads_settings',
			'iads_options',
			array(
				'type' => 'array',
				'sanitize_callback' => array($this, 'sanitize_options'),
				'default' => array(
					'toggle_position' => 'bottom-right',
					'toggle_style' => 'modern',
					'paypal_email' => '',
					'show_donation' => false,
					'custom_css' => ''
				)
			)
		);
	}

    /**
     * Sanitizzazione delle opzioni
     */
    public function sanitize_options($input) {
		// Aggiungi wp_unslash all'inizio
		if (is_array($input)) {
			$input = wp_unslash($input);
		}

		$sanitized = array();

		if (isset($input['toggle_position'])) {
			$allowed_positions = array('bottom-right', 'bottom-left', 'top-right', 'top-left');
			$sanitized['toggle_position'] = in_array($input['toggle_position'], $allowed_positions) ? 
				sanitize_text_field($input['toggle_position']) : 'bottom-right';
		}

		if (isset($input['toggle_style'])) {
			$allowed_styles = array('modern', 'classic');
			$sanitized['toggle_style'] = in_array($input['toggle_style'], $allowed_styles) ? 
				sanitize_text_field($input['toggle_style']) : 'modern';
		}

		if (isset($input['paypal_email'])) {
			$sanitized['paypal_email'] = sanitize_email($input['paypal_email']);
		}

		if (isset($input['show_donation'])) {
			$sanitized['show_donation'] = (bool) $input['show_donation'];
		}

		if (isset($input['custom_css'])) {
			$sanitized['custom_css'] = wp_strip_all_tags($input['custom_css']);
		}

		return $sanitized;
	}

    /**
     * Caricamento asset frontend
     */
    public function enqueue_frontend_assets() {
        wp_enqueue_style('iads-frontend-css', IADS_PLUGIN_URL . 'assets/css/frontend.css', array(), IADS_VERSION);
        wp_enqueue_script('iads-frontend-js', IADS_PLUGIN_URL . 'assets/js/frontend.js', array('jquery'), IADS_VERSION, true);

        // Aggiungi CSS personalizzato inline SE esiste
        $options = $this->get_options();
        /*if (!empty($options['custom_css'])) {
            wp_add_inline_style('iads-frontend-css', wp_kses_post($options['custom_css']));
        }*/

        // Localizzazione per AJAX sicuro
        wp_localize_script('iads-frontend-js', 'iads_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('iads_nonce'),
            'options' => $this->get_options()
        ));
    }

    /**
     * Caricamento asset admin
     */
    public function enqueue_admin_assets($hook) {
        // CORREZIONE: Hook corretto per la pagina
        if ($hook !== 'settings_page_improve-accessibility-dark-mode-switcher') {
            return;
        }

        wp_enqueue_style('iads-admin-css', IADS_PLUGIN_URL . 'assets/css/admin.css', array(), IADS_VERSION);
        wp_enqueue_script('iads-admin-js', IADS_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), IADS_VERSION, true);

        wp_localize_script('iads-admin-js', 'iads_admin_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('iads_admin_nonce')
        ));
    }

    /**
     * Aggiungere menu admin
     */
    public function add_admin_menu() {
        add_options_page(
            __('Dark Mode Switcher', 'improve-accessibility-dark-mode-switcher'),
            __('Dark Mode Switcher', 'improve-accessibility-dark-mode-switcher'),
            'manage_options',
            'improve-accessibility-dark-mode-switcher',
            array($this, 'admin_page')
        );
    }

    /**
     * Pagina admin
     */
    public function admin_page() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Non hai i permessi per accedere a questa pagina.', 'improve-accessibility-dark-mode-switcher'));
        }

        $options = $this->get_options();
        include IADS_PLUGIN_PATH . 'templates/admin-page.php';
    }

    /**
     * Rendering del toggle (SENZA donazioni per utenti finali)
     */
    public function render_dark_mode_toggle() {
        $options = $this->get_options();
        include IADS_PLUGIN_PATH . 'templates/toggle.php';
    }

    /**
     * Ottenere le opzioni del plugin
     */
    private function get_options() {
        $defaults = array(
            'toggle_position' => 'bottom-right',
            'toggle_style' => 'modern',
            'paypal_email' => '',
            'show_donation' => false, // Default false per evitare donazioni pubbliche
        );

        return wp_parse_args(get_option('iads_options', array()), $defaults);
    }

    /**
     * Handler AJAX per salvare impostazioni
     */
	
    public function ajax_save_settings() {
        
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        
        if (!wp_verify_nonce($nonce, 'iads_admin_nonce')) {
            wp_send_json_error(array('message' => __('Errore di sicurezza', 'improve-accessibility-dark-mode-switcher')));
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permessi insufficienti', 'improve-accessibility-dark-mode-switcher')));
            return;
        }

        // Verifica che options esista e sia un array
        if (!isset($_POST['options']) || !is_array($_POST['options'])) {
            wp_send_json_error(array('message' => __('Dati non validi', 'improve-accessibility-dark-mode-switcher')));
            return;
        }

        // Sanitizza l'array correttamente
        $raw_options = array_map('sanitize_text_field', wp_unslash($_POST['options']));
        
        $sanitized_options = array(
            'toggle_position' => isset($raw_options['toggle_position']) ? $raw_options['toggle_position'] : 'bottom-right',
            'toggle_style' => isset($raw_options['toggle_style']) ? $raw_options['toggle_style'] : 'modern',
            'paypal_email' => '',
            'show_donation' => false
        );

        // Validazione posizione
        $allowed_positions = array('bottom-right', 'bottom-left', 'top-right', 'top-left');
        if (!in_array($sanitized_options['toggle_position'], $allowed_positions, true)) {
            $sanitized_options['toggle_position'] = 'bottom-right';
        }

        // Validazione stile
        $allowed_styles = array('modern', 'classic');
        if (!in_array($sanitized_options['toggle_style'], $allowed_styles, true)) {
            $sanitized_options['toggle_style'] = 'modern';
        }

        $result = update_option('iads_options', $sanitized_options);

        if ($result || get_option('iads_options') === $sanitized_options) {
            wp_send_json_success(array('message' => __('Impostazioni salvate con successo!', 'improve-accessibility-dark-mode-switcher')));
        } else {
            wp_send_json_error(array('message' => __('Errore nel salvataggio delle impostazioni', 'improve-accessibility-dark-mode-switcher')));
        }
    }
	/**
	 * Sanitizza una singola opzione
	 */
	private function sanitize_single_option($key, $value) {
		switch ($key) {
			case 'toggle_position':
				$allowed_positions = array('bottom-right', 'bottom-left', 'top-right', 'top-left');
				return in_array($value, $allowed_positions) ? sanitize_text_field($value) : 'bottom-right';

			case 'toggle_style':
				$allowed_styles = array('modern', 'classic');
				return in_array($value, $allowed_styles) ? sanitize_text_field($value) : 'modern';

			case 'paypal_email':
				return sanitize_email($value);

			case 'show_donation':
				return (bool) $value;

			case 'custom_css':
				return wp_strip_all_tags($value);

			default:
				return sanitize_text_field($value);
		}
	}

    /**
     * Handler AJAX per reset impostazioni
     */
    public function ajax_reset_settings() {
		$nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';

		// USA la variabile sanitizzata, non $_POST['nonce']
		if (!wp_verify_nonce($nonce, 'iads_admin_nonce')) {
			wp_send_json_error(array('message' => __('Errore di sicurezza', 'improve-accessibility-dark-mode-switcher')));
			return;
		}

		if (!current_user_can('manage_options')) {
			wp_send_json_error(array('message' => __('Permessi insufficienti', 'improve-accessibility-dark-mode-switcher')));
			return;
		}

		// Reset alle impostazioni di default
		$defaults = array(
			'toggle_position' => 'bottom-right',
			'toggle_style' => 'modern',
			'paypal_email' => '',
			'show_donation' => false,
			'custom_css' => ''
		);

		$result = update_option('iads_options', $defaults);

		if ($result) {
			wp_send_json_success(array('message' => __('Impostazioni ripristinate!', 'improve-accessibility-dark-mode-switcher')));
		} else {
			wp_send_json_error(array('message' => __('Errore nel reset', 'improve-accessibility-dark-mode-switcher')));
		}
	}

    /**
     * Handler AJAX per ottenere modalità corrente
     */
    public function ajax_get_mode() {
		$nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';

		// USA la variabile sanitizzata, non $_POST['nonce']
		if (!wp_verify_nonce($nonce, 'iads_nonce')) {
			wp_send_json_error(array('message' => __('Errore di sicurezza', 'improve-accessibility-dark-mode-switcher')));
			return;
		}

		wp_send_json_success(array('mode' => 'light')); // Placeholder
	}

    /**
     * Attivazione plugin
     */
    public function activate() {
        // Creare opzioni di default
        $defaults = array(
            'toggle_position' => 'bottom-right',
            'toggle_style' => 'modern',
            'paypal_email' => '',
            'show_donation' => false, // Default false
        );

        add_option('iads_options', $defaults);

        // Creare tabelle se necessario
        $this->create_tables();
    }

    /**
     * Disattivazione plugin
     */
    public function deactivate() {
        // Pulizia se necessario
    }

    /**
     * Creare tabelle database
     */
    private function create_tables() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'iads_user_preferences';
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            user_id bigint(20) NOT NULL,
            dark_mode_preference varchar(10) NOT NULL DEFAULT 'light',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY user_id (user_id)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
}

// Inizializzare il plugin
IADS_Main::getInstance();