/**
 * Improve Accessibility - Dark Mode Switcher for WP - Admin JavaScript
 * Gestione interfaccia amministrazione
 */
(function($) {
    'use strict';

    /**
     * Classe per la gestione dell'interfaccia admin
     */
    class IADSAdmin {
        constructor() {
            this.currentTab = 'general';
            this.isLoading = false;
            this.unsavedChanges = false;

            this.init();
        }

        /**
         * Inizializzazione
         */
        init() {
            this.bindEvents();
            this.initTabs();
            this.initPreview();
            this.watchFormChanges();
            this.initTooltips();
        }

        /**
         * Binding degli eventi
         */
        bindEvents() {
            // Navigation tabs
            $('.iads-nav-tab a').on('click', (e) => {
                e.preventDefault();
                const tab = $(e.currentTarget).data('tab');
                this.switchTab(tab);
            });

            // Form submission
            $('#iads-settings-form').on('submit', (e) => {
                e.preventDefault();
                this.saveSettings();
            });

            // Save button
            $('#iads-save-settings').on('click', (e) => {
                e.preventDefault();
                this.saveSettings();
            });

            // Test toggle button
            $('#iads-test-toggle').on('click', (e) => {
                e.preventDefault();
                this.testToggle();
            });

            // Reset settings
            $('#iads-reset-settings').on('click', (e) => {
                e.preventDefault();
                this.showConfirmModal(
                    'Sei sicuro di voler ripristinare tutte le impostazioni ai valori di default? Questa azione non può essere annullata.',
                    () => this.resetSettings()
                );
            });

            // Modal events
            $('.iads-modal-close, #iads-confirm-no').on('click', () => {
                this.hideModal();
            });

            $('#iads-confirm-yes').on('click', () => {
                if (this.confirmCallback) {
                    this.confirmCallback();
                }
                this.hideModal();
            });

            // Click outside modal to close
            $('.iads-modal').on('click', (e) => {
                if ($(e.target).hasClass('iads-modal')) {
                    this.hideModal();
                }
            });

            // Prevent closing on unsaved changes
            $(window).on('beforeunload', (e) => {
                if (this.unsavedChanges) {
                    const message = 'Hai modifiche non salvate. Vuoi davvero uscire?';
                    e.returnValue = message;
                    return message;
                }
            });

            // Code editor enhancements
            $('.iads-code-editor').on('input', () => {
                this.updatePreview();
                this.markUnsavedChanges();
            });

            // Style changes
            $('input[name="iads_options[toggle_style]"]').on('change', () => {
                this.updatePreview();
                this.markUnsavedChanges();
            });

            // Position changes
            $('select[name="iads_options[toggle_position]"]').on('change', () => {
                this.updatePreview();
                this.markUnsavedChanges();
            });
        }

        /**
         * Inizializzazione tabs
         */
        initTabs() {
            // Se c'è un hash nell'URL, vai a quel tab
            const hash = window.location.hash.substring(1);
            if (hash && $('.iads-nav-tab a[data-tab="' + hash + '"]').length) {
                this.switchTab(hash);
            } else {
                this.switchTab('general');
            }
        }

        /**
         * Switch tra i tabs
         */
        switchTab(tabName) {
            // Update navigation
            $('.iads-nav-tab').removeClass('active');
            $('.iads-nav-tab a[data-tab="' + tabName + '"]').parent().addClass('active');

            // Update content
            $('.iads-tab-panel').removeClass('active');
            $('#' + tabName).addClass('active');

            // Update URL hash
            history.replaceState(null, null, '#' + tabName);

            this.currentTab = tabName;

            // Trigger custom event
            $(document).trigger('iads:tab-changed', [tabName]);
        }

        /**
         * Inizializzazione preview
         */
        initPreview() {
            this.updatePreview();
        }

        /**
         * Aggiornamento preview del toggle
         */
        updatePreview() {
            const position = $('select[name="iads_options[toggle_position]"]').val();
            const style = $('input[name="iads_options[toggle_style]"]:checked').val();
            const customCSS = $('.iads-code-editor').val();

            // Genera HTML preview
            const previewHTML = `
                <div class="iads-toggle-wrapper ${style}" style="position: relative; display: inline-flex;">
                    <button class="iads-toggle-switch" type="button">
                        <span class="iads-icon iads-icon-sun">☀️</span>
                        <span class="iads-icon iads-icon-moon">🌙</span>
                    </button>
                    <span class="iads-toggle-label">Light Mode</span>
                </div>
            `;

            $('#iads-preview').html(previewHTML);

            // Applica CSS personalizzato se presente
            if (customCSS.trim()) {
                let styleElement = $('#iads-custom-preview-styles');
                if (!styleElement.length) {
                    styleElement = $('<style id="iads-custom-preview-styles"></style>').appendTo('head');
                }
                styleElement.text(customCSS);
            }

            // Aggiungi classe posizione per dimostrazione
            $('#iads-preview .iads-toggle-wrapper').attr('data-position', position);
        }

        /**
         * Monitoraggio modifiche form
         */
        watchFormChanges() {
            $('#iads-settings-form input, #iads-settings-form select, #iads-settings-form textarea').on('input change', () => {
                this.markUnsavedChanges();
            });
        }

        /**
         * Marca modifiche non salvate
         */
        markUnsavedChanges() {
            if (!this.unsavedChanges) {
                this.unsavedChanges = true;
                $('#iads-save-settings').addClass('iads-button-warning').text('💾 Salva Modifiche');
            }
        }

        /**
         * Marca modifiche salvate
         */
        markChangesSaved() {
            this.unsavedChanges = false;
            $('#iads-save-settings').removeClass('iads-button-warning').html('<span class="dashicons dashicons-yes"></span> Salva Impostazioni');
        }

        /**
         * Salvataggio impostazioni via AJAX
         */
        saveSettings() {
            if (this.isLoading) return;

            this.setLoading(true);
            this.showNotice('Salvataggio in corso...', 'info');

            const formData = new FormData($('#iads-settings-form')[0]);
            formData.append('action', 'iads_save_settings');
            formData.append('nonce', iads_admin_ajax.nonce);

            // Converti FormData in oggetto per facilità
            const options = {};

            // Raccogli le options
            formData.forEach((value, key) => {
                if (key.startsWith('iads_options[')) {
                    const optionName = key.replace('iads_options[', '').replace(']', '');
                    options[optionName] = value;
                }
            });


            $.ajax({
                url: iads_admin_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'iads_save_settings',
                    nonce: iads_admin_ajax.nonce,
                    'options[toggle_position]': options.toggle_position,
                    'options[toggle_style]': options.toggle_style
                },
                success: (response) => {
                    if (response.success) {
                        this.showNotice('✅ Impostazioni salvate con successo!', 'success');
                        this.markChangesSaved();

                        // Aggiorna preview
                        setTimeout(() => {
                            this.updatePreview();
                        }, 100);

                    } else {
                        this.showNotice('❌ Errore nel salvataggio: ' + (response.data.message || 'Errore sconosciuto'), 'error');
                    }
                },
                error: (xhr, status, error) => {
                    this.showNotice('❌ Errore di connessione: ' + error, 'error');
                },
                complete: () => {
                    this.setLoading(false);
                }
            });
        }

        /**
         * Test del toggle
         */
        testToggle() {
            // Simula il toggle nel preview
            const $preview = $('#iads-preview .iads-toggle-switch');

            if ($preview.hasClass('active')) {
                $preview.removeClass('active');
                $('#iads-preview .iads-toggle-label').text('Light Mode');
            } else {
                $preview.addClass('active');
                $('#iads-preview .iads-toggle-label').text('Dark Mode');
            }

            // Feedback visivo
            this.showNotice('🔄 Toggle testato nel preview!', 'info');

            // Reset dopo 2 secondi
            setTimeout(() => {
                $preview.removeClass('active');
                $('#iads-preview .iads-toggle-label').text('Light Mode');
            }, 2000);
        }

        /**
         * Reset impostazioni
         */
        resetSettings() {
            if (this.isLoading) return;

            this.setLoading(true);
            this.showNotice('Reset in corso...', 'info');

            $.ajax({
                url: iads_admin_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'iads_reset_settings',
                    nonce: iads_admin_ajax.nonce
                },
                success: (response) => {
                    if (response.success) {
                        this.showNotice('✅ Impostazioni ripristinate ai valori predefiniti!', 'success');

                        // Ricarica la pagina per mostrare i nuovi valori
                        setTimeout(() => {
                            window.location.reload();
                        }, 1500);

                    } else {
                        this.showNotice('❌ Errore nel reset: ' + (response.data.message || 'Errore sconosciuto'), 'error');
                    }
                },
                error: (xhr, status, error) => {
                    this.showNotice('❌ Errore di connessione: ' + error, 'error');
                },
                complete: () => {
                    this.setLoading(false);
                }
            });
        }

        /**
         * Mostra modal di conferma
         */
        showConfirmModal(message, callback) {
            $('#iads-confirm-message').text(message);
            $('#iads-confirm-modal').fadeIn(300);
            this.confirmCallback = callback;
        }

        /**
         * Nasconde modal
         */
        hideModal() {
            $('.iads-modal').fadeOut(300);
            this.confirmCallback = null;
        }

        /**
         * Mostra notifica
         */
        showNotice(message, type = 'info') {
            const typeClass = type === 'info' ? 'warning' : type; // WordPress usa 'warning'

            const noticeHTML = `
                <div class="iads-notice ${typeClass}">
                    <span class="dashicons dashicons-${this.getNoticeIcon(type)}"></span>
                    <span>${message}</span>
                </div>
            `;

            $('#iads-admin-notices').html(noticeHTML);

            // Auto-hide per messaggi di successo
            if (type === 'success' || type === 'info') {
                setTimeout(() => {
                    $('#iads-admin-notices').fadeOut(500);
                }, 4000);
            }
        }

        /**
         * Ottiene icona per tipo di notifica
         */
        getNoticeIcon(type) {
            const icons = {
                success: 'yes-alt',
                error: 'dismiss',
                warning: 'warning',
                info: 'info'
            };
            return icons[type] || 'info';
        }

        /**
         * Imposta stato di caricamento
         */
        setLoading(loading) {
            this.isLoading = loading;

            if (loading) {
                $('#iads-save-settings, #iads-test-toggle, #iads-reset-settings').prop('disabled', true).addClass('iads-loading');
                $('body').addClass('iads-admin-loading');
            } else {
                $('#iads-save-settings, #iads-test-toggle, #iads-reset-settings').prop('disabled', false).removeClass('iads-loading');
                $('body').removeClass('iads-admin-loading');
            }
        }

        /**
         * Inizializzazione tooltips
         */
        initTooltips() {
            // Semplici tooltips CSS-only per elementi con title
            $('[title]').each(function() {
                const $this = $(this);
                const title = $this.attr('title');

                $this.removeAttr('title').attr('data-tooltip', title);
            });
        }

        /**
         * Validazione email
         */
        isValidEmail(email) {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailRegex.test(email);
        }

        /**
         * Validazione form prima del salvataggio
         */
        validateForm() {
            const errors = [];

            // Validazione email PayPal
            const paypalEmail = $('input[name="iads_options[paypal_email]"]').val().trim();
            const showDonation = $('input[name="iads_options[show_donation]"]').is(':checked');

            if (showDonation && paypalEmail && !this.isValidEmail(paypalEmail)) {
                errors.push('Email PayPal non valida');
            }

            // Validazione CSS personalizzato (controllo sintassi base)
            const customCSS = $('.iads-code-editor').val().trim();
            if (customCSS) {
                // Controllo bilanciamento parentesi graffe
                const openBraces = (customCSS.match(/\{/g) || []).length;
                const closeBraces = (customCSS.match(/\}/g) || []).length;

                if (openBraces !== closeBraces) {
                    errors.push('CSS personalizzato: parentesi graffe non bilanciate');
                }
            }

            return errors;
        }

        /**
         * Export delle impostazioni
         */
        exportSettings() {
            const formData = new FormData($('#iads-settings-form')[0]);
            const settings = {};

            formData.forEach((value, key) => {
                if (key.startsWith('iads_options[')) {
                    const optionName = key.replace('iads_options[', '').replace(']', '');
                    settings[optionName] = value;
                }
            });

            const dataStr = "data:text/json;charset=utf-8," + encodeURIComponent(JSON.stringify(settings, null, 2));
            const downloadAnchorNode = document.createElement('a');
            downloadAnchorNode.setAttribute("href", dataStr);
            downloadAnchorNode.setAttribute("download", "dark-mode-switcher-settings.json");
            document.body.appendChild(downloadAnchorNode);
            downloadAnchorNode.click();
            downloadAnchorNode.remove();

            this.showNotice('⬇️ Impostazioni esportate!', 'success');
        }

        /**
         * Import delle impostazioni
         */
        importSettings(event) {
            const file = event.target.files[0];
            if (!file) return;

            const reader = new FileReader();
            reader.onload = (e) => {
                try {
                    const settings = JSON.parse(e.target.result);

                    // Popola il form con le impostazioni importate
                    Object.keys(settings).forEach(key => {
                        const input = $(`[name="iads_options[${key}]"]`);
                        if (input.length) {
                            if (input.attr('type') === 'checkbox') {
                                input.prop('checked', !!settings[key]);
                            } else if (input.attr('type') === 'radio') {
                                input.filter(`[value="${settings[key]}"]`).prop('checked', true);
                            } else {
                                input.val(settings[key]);
                            }
                        }
                    });

                    this.markUnsavedChanges();
                    this.updatePreview();
                    this.showNotice('⬆️ Impostazioni importate! Ricordati di salvare.', 'success');

                } catch (error) {
                    this.showNotice('❌ Errore nell\'import: file non valido', 'error');
                }
            };
            reader.readAsText(file);
        }

        /**
         * Copia shortcode negli appunti
         */
        copyToClipboard(text, successMessage) {
            if (navigator.clipboard) {
                navigator.clipboard.writeText(text).then(() => {
                    this.showNotice(successMessage, 'success');
                });
            } else {
                // Fallback per browser più vecchi
                const textArea = document.createElement('textarea');
                textArea.value = text;
                document.body.appendChild(textArea);
                textArea.focus();
                textArea.select();
                try {
                    document.execCommand('copy');
                    this.showNotice(successMessage, 'success');
                } catch (err) {
                    this.showNotice('❌ Impossibile copiare negli appunti', 'error');
                }
                document.body.removeChild(textArea);
            }
        }

        /**
         * Anteprima live del CSS
         */
        initLiveCSSPreview() {
            let cssTimeout;
            $('.iads-code-editor').on('input', () => {
                clearTimeout(cssTimeout);
                cssTimeout = setTimeout(() => {
                    this.updatePreview();
                }, 500); // Debounce di 500ms
            });
        }
    }

    /**
     * Utility per la gestione dei dati
     */
    class IADSDataManager {
        /**
         * Salva bozza automaticamente
         */
        static autoSave() {
            const formData = $('#iads-settings-form').serialize();
            localStorage.setItem('iads_admin_draft', formData);
        }

        /**
         * Carica bozza salvata
         */
        static loadDraft() {
            const draft = localStorage.getItem('iads_admin_draft');
            if (draft) {
                // Implementa logica di caricamento bozza se necessario
            }
        }

        /**
         * Pulisce bozze salvate
         */
        static clearDraft() {
            localStorage.removeItem('iads_admin_draft');
        }
    }

    /**
     * Inizializzazione quando DOM è pronto
     */
    $(document).ready(function() {
        // Verifica che siamo nella pagina corretta
        if (!$('.iads-admin-wrap').length) {
            return;
        }

        // Inizializza l'admin
        window.IADSAdmin = new IADSAdmin();

        // Carica eventuali bozze
        IADSDataManager.loadDraft();

        // Auto-save ogni 30 secondi se ci sono modifiche
        setInterval(() => {
            if (window.IADSAdmin.unsavedChanges) {
                IADSDataManager.autoSave();
            }
        }, 30000);

        // Pulizia bozze al salvataggio
        $(document).on('iads:settings-saved', () => {
            IADSDataManager.clearDraft();
        });

        // Keyboard shortcuts
        $(document).on('keydown', (e) => {
            // Ctrl+S per salvare
            if ((e.ctrlKey || e.metaKey) && e.key === 's') {
                e.preventDefault();
                window.IADSAdmin.saveSettings();
            }

            // Esc per chiudere modal
            if (e.key === 'Escape') {
                window.IADSAdmin.hideModal();
            }
        });

        // Gestione responsive per tabs su mobile
        function handleResponsiveTabs() {
            if ($(window).width() < 768) {
                $('.iads-nav-tabs').addClass('mobile-tabs');
            } else {
                $('.iads-nav-tabs').removeClass('mobile-tabs');
            }
        }

        $(window).on('resize', handleResponsiveTabs);
        handleResponsiveTabs();

        // Animazioni di entrata per elementi
        $('.iads-field-group').each(function(index) {
            $(this).css('animation-delay', (index * 0.1) + 's');
            $(this).addClass('fade-in-up');
        });

        // Supporto per drag & drop per import file
        let dragCounter = 0;
        $(document).on('dragenter', (e) => {
            e.preventDefault();
            dragCounter++;
            $('.iads-admin-wrap').addClass('drag-over');
        });

        $(document).on('dragleave', (e) => {
            e.preventDefault();
            dragCounter--;
            if (dragCounter === 0) {
                $('.iads-admin-wrap').removeClass('drag-over');
            }
        });

        $(document).on('dragover', (e) => {
            e.preventDefault();
        });

        $(document).on('drop', (e) => {
            e.preventDefault();
            dragCounter = 0;
            $('.iads-admin-wrap').removeClass('drag-over');

            const files = e.originalEvent.dataTransfer.files;
            if (files.length && files[0].type === 'application/json') {
                // Simula click su input file per import
                const fakeEvent = { target: { files: files } };
                window.IADSAdmin.importSettings(fakeEvent);
            }
        });

    });

    /**
     * Aggiunge stili per animazioni CSS
     */
    const animationStyles = `
        <style>
        .fade-in-up {
            animation: fadeInUp 0.6s ease-out forwards;
        }
        
        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .drag-over::after {
            content: '📁 Rilascia il file JSON delle impostazioni qui';
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 115, 170, 0.9);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5em;
            z-index: 999999;
            pointer-events: none;
        }
        </style>
    `;

    $('head').append(animationStyles);

})(jQuery);