<?php
/**
 * Classe per la gestione della sicurezza
 *
 * @package DarkModeSwitcherUniversal
 * @since 1.0.0
 */

// Prevengo accesso diretto
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Classe IADS_Security
 * Gestisce tutti gli aspetti di sicurezza del plugin
 */
class IADS_Security {

    /**
     * Istanza singleton
     */
    private static $instance = null;

    /**
     * Nonces attivi
     */
    private $nonces = array();

    /**
     * Rate limiting data
     */
    private $rate_limits = array();

    /**
     * Ottieni istanza singleton
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Costruttore privato per singleton
     */
    private function __construct() {
        $this->init();
    }

    /**
     * Inizializzazione
     */
    public function init() {
        // Hook per sicurezza
        add_action('init', array($this, 'security_headers'));
        add_action('wp_loaded', array($this, 'validate_requests'));

        // Filtri per sanitizzazione
        add_filter('iads_sanitize_option', array($this, 'sanitize_option'), 10, 2);
        add_filter('iads_validate_nonce', array($this, 'validate_nonce'), 10, 2);

        // Rate limiting per AJAX
        add_action('wp_ajax_iads_save_settings', array($this, 'rate_limit_check'), 1);
        add_action('wp_ajax_nopriv_iads_get_mode', array($this, 'rate_limit_check'), 1);
    }

    /**
     * Header di sicurezza
     */
    public function security_headers() {
        if (!headers_sent()) {
            // Solo per le pagine admin del plugin
            if (is_admin() && get_current_screen() && get_current_screen()->id === 'toplevel_page_improve-accessibility-dark-mode-switcher') {
                header('X-Content-Type-Options: nosniff');
                header('X-Frame-Options: SAMEORIGIN');
                header('X-XSS-Protection: 1; mode=block');
            }
        }
    }

    /**
     * Validazione richieste
     */
    public function validate_requests() {
        // Validazione richieste AJAX IADS
        if (defined('DOING_AJAX') && DOING_AJAX) {
            if (isset($_POST['action'], $_POST['_wpnonce']) && 
    wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'iads_action') &&
    strpos(sanitize_text_field(wp_unslash($_POST['action'])), 'iads_') === 0) {
                $this->validate_ajax_request();
            }
        }

        // Validazione richieste admin
        if (is_admin() && isset($_GET['page']) && $_GET['page'] === 'improve-accessibility-dark-mode-switcher') {
            $this->validate_admin_request();
        }
    }

    /**
     * Validazione richieste AJAX
     */
    private function validate_ajax_request() {
        // Verifica referrer
        if ((!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'] ?? '')), 'iads_nonce') &&
     !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'] ?? '')), 'iads_admin_nonce'))) {
    wp_die(esc_html__('Verifica di sicurezza fallita', 'improve-accessibility-dark-mode-switcher'), 403);
        }

        // Verifica capability per azioni admin
        if (isset($_POST['action']) && 
    (strpos(sanitize_text_field(wp_unslash($_POST['action'])), 'iads_save') === 0 ||
     strpos(sanitize_text_field(wp_unslash($_POST['action'])), 'iads_reset') === 0)) {
            if (!current_user_can('manage_options')) {
                wp_die(esc_html__('Permessi insufficienti', 'improve-accessibility-dark-mode-switcher'), 403);
            }
        }
    }

    /**
     * Validazione richieste admin
     */
    private function validate_admin_request() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Non hai i permessi per accedere a questa pagina.', 'improve-accessibility-dark-mode-switcher'));
        }

        // Verifica nonce per POST
        if (isset($_SERVER['REQUEST_METHOD']) === 'POST') { 
    $nonce = sanitize_text_field(wp_unslash($_POST['iads_nonce'] ?? ''));
    if (!wp_verify_nonce($nonce, 'iads_admin_settings')) {
                wp_die(esc_html__('Verifica di sicurezza fallita', 'improve-accessibility-dark-mode-switcher'), 403);
            }
        }
    }

    /**
     * Rate limiting per prevenire abusi
     */
    public function rate_limit_check() {
        $user_id = get_current_user_id();
        $ip = $this->get_client_ip();
        $key = $user_id ? "user_{$user_id}" : "ip_{$ip}";

        $current_time = time();
        $window = 60; // 1 minuto
        $max_requests = 30; // Max 30 richieste per minuto

        // Pulisci vecchi record
        if (isset($this->rate_limits[$key])) {
            $this->rate_limits[$key] = array_filter(
                $this->rate_limits[$key],
                function($timestamp) use ($current_time, $window) {
                    return ($current_time - $timestamp) < $window;
                }
            );
        } else {
            $this->rate_limits[$key] = array();
        }

        // Controlla limite
        if (count($this->rate_limits[$key]) >= $max_requests) {
            wp_die(esc_html__('Troppe richieste. Riprova più tardi.', 'improve-accessibility-dark-mode-switcher'), 429);
        }

        // Aggiungi timestamp corrente
        $this->rate_limits[$key][] = $current_time;
    }

    /**
     * Ottieni IP del client (con supporto proxy)
     */
    private function get_client_ip() {
        $ip_headers = array(
            'HTTP_CF_CONNECTING_IP',     // Cloudflare
            'HTTP_X_REAL_IP',           // Nginx proxy
            'HTTP_X_FORWARDED_FOR',     // Load balancer/proxy
            'HTTP_X_FORWARDED',         // Proxy
            'HTTP_FORWARDED_FOR',       // Proxy
            'HTTP_FORWARDED',           // Proxy
            'REMOTE_ADDR'               // Standard
        );

        foreach ($ip_headers as $header) {
			if (isset($_SERVER[$header]) && !empty($_SERVER[$header])) {
				$header_value = sanitize_text_field(wp_unslash($_SERVER[$header]));
				$ip_list = explode(',', $header_value);
				$ip = trim($ip_list[0]);
				if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
					return $ip;
				}
			}
		}

        return isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '0.0.0.0';
    }

    /**
     * Sanitizza opzioni in base al tipo
     */
    public function sanitize_option($value, $option_name) {
        switch ($option_name) {
            case 'toggle_position':
                $allowed_positions = array('bottom-right', 'bottom-left', 'top-right', 'top-left');
                return in_array($value, $allowed_positions) ? $value : 'bottom-right';

            case 'toggle_style':
                $allowed_styles = array('modern', 'classic');
                return in_array($value, $allowed_styles) ? $value : 'modern';

            case 'paypal_email':
                return sanitize_email($value);

            case 'show_donation':
                return (bool) $value;

            case 'custom_css':
                return $this->sanitize_css($value);

            default:
                return sanitize_text_field($value);
        }
    }

    /**
     * Sanitizza CSS personalizzato
     */
    private function sanitize_css($css) {
        // Rimuovi tag PHP e script
        $css = preg_replace('/<\?php.*?\?>/s', '', $css);
        $css = preg_replace('/<script.*?<\/script>/s', '', $css);

        // Rimuovi proprietà CSS pericolose
        $dangerous_properties = array(
            'expression',
            'javascript:',
            'vbscript:',
            'data:text/html',
            '@import',
            'binding'
        );

        foreach ($dangerous_properties as $property) {
            $css = str_ireplace($property, '', $css);
        }

        // Rimuovi commenti HTML
        $css = preg_replace('/<!--.*?-->/s', '', $css);

        return wp_strip_all_tags($css);
    }

    /**
     * Validazione nonce personalizzata
     */
    public function validate_nonce($nonce, $action) {
        if (!wp_verify_nonce($nonce, $action)) {
            $this->log_security_event('Invalid nonce attempt', array(
                'action' => $action,
                'user_id' => get_current_user_id(),
                'ip' => $this->get_client_ip(),
                'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : ''
            ));
            return false;
        }
        return true;
    }

    /**
     * Genera nonce sicuro con scadenza personalizzata
     */
    public function create_nonce($action, $expiry = 3600) {
        $nonce = wp_create_nonce($action);

        // Salva nonce con timestamp
        $this->nonces[$nonce] = array(
            'action' => $action,
            'created' => time(),
            'expiry' => $expiry
        );

        return $nonce;
    }

    /**
     * Verifica integrità file plugin
     */
    public function verify_plugin_integrity() {
        $core_files = array(
            IADS_PLUGIN_PATH . 'improve-accessibility-dark-mode-switcher.php',
            IADS_PLUGIN_PATH . 'includes/class-security.php',
            IADS_PLUGIN_PATH . 'assets/js/frontend.js',
            IADS_PLUGIN_PATH . 'assets/css/frontend.css'
        );

        $integrity_issues = array();

        foreach ($core_files as $file) {
            if (!file_exists($file)) {
                $integrity_issues[] = "File mancante: " . basename($file);
            } elseif (!is_readable($file)) {
                $integrity_issues[] = "File non leggibile: " . basename($file);
            }
        }

        return empty($integrity_issues) ? true : $integrity_issues;
    }

    /**
     * Logging eventi di sicurezza
     */
    private function log_security_event($event, $data = array()) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
//             error_log(sprintf(
//                 '[IADS Security] %s - %s - %s',
//                 $event,
//                 wp_json_encode($data),
//                 gmtime('Y-m-d H:i:s')
//             ));
        }

        // Opzionalmente salva nel database per audit
        $this->save_security_log($event, $data);
    }

    /**
     * Salva log di sicurezza nel database
     */
    private function save_security_log($event, $data) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'iads_security_log';

        // Crea tabella se non esiste
        $this->create_security_log_table();
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct database access
        $wpdb->insert(
            $table_name,
            array(
                'event_type' => sanitize_text_field($event),
                'event_data' => wp_json_encode($data),
                'user_id' => get_current_user_id(),
                'ip_address' => $this->get_client_ip(),
                'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '',
                'created_at' => current_time('mysql')
            ),
            array('%s', '%s', '%d', '%s', '%s', '%s')
        );
    }

    /**
     * Crea tabella per log di sicurezza
     */
    private function create_security_log_table() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'iads_security_log';

        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            event_type varchar(100) NOT NULL,
            event_data longtext,
            user_id bigint(20) UNSIGNED DEFAULT 0,
            ip_address varchar(45) NOT NULL,
            user_agent text,
            created_at datetime NOT NULL,
            PRIMARY KEY (id),
            KEY event_type (event_type),
            KEY user_id (user_id),
            KEY created_at (created_at)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    /**
     * Pulizia automatica vecchi log
     */
    public function cleanup_old_logs($days = 30) {
		global $wpdb;
		$cutoff_date = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table requires direct database access
		$wpdb->query($wpdb->prepare(
			"DELETE FROM {$wpdb->prefix}iads_security_log WHERE created_at < %s",
			$cutoff_date
		));
	}

    /**
     * Controllo blacklist IP
     */
    public function is_ip_blacklisted($ip = null) {
        if (!$ip) {
            $ip = $this->get_client_ip();
        }

        // Lista IP bloccati (può essere estesa)
        $blacklisted_ips = get_option('iads_blacklisted_ips', array());

        return in_array($ip, $blacklisted_ips);
    }

    /**
     * Aggiungi IP alla blacklist
     */
    public function blacklist_ip($ip, $reason = '') {
        $blacklisted_ips = get_option('iads_blacklisted_ips', array());

        if (!in_array($ip, $blacklisted_ips)) {
            $blacklisted_ips[] = $ip;
            update_option('iads_blacklisted_ips', $blacklisted_ips);

            $this->log_security_event('IP Blacklisted', array(
                'ip' => $ip,
                'reason' => $reason
            ));
        }
    }

    /**
     * Scansione malware di base
     */
    public function basic_malware_scan() {
        $suspicious_patterns = array(
            'eval\(',
            'base64_decode',
            'shell_exec',
            'system\(',
            'exec\(',
            'passthru',
            'file_get_contents\(',
            'curl_exec'
        );

        $plugin_files = array(
            IADS_PLUGIN_PATH . 'includes/',
            IADS_PLUGIN_PATH . 'assets/',
            IADS_PLUGIN_PATH . 'templates/'
        );

        $suspicious_files = array();

        foreach ($plugin_files as $dir) {
            if (is_dir($dir)) {
                $files = new RecursiveIteratorIterator(
                    new RecursiveDirectoryIterator($dir)
                );

                foreach ($files as $file) {
                    if ($file->isFile() && pathinfo($file, PATHINFO_EXTENSION) === 'php') {
                        $content = file_get_contents($file);

                        foreach ($suspicious_patterns as $pattern) {
                            if (preg_match('/' . preg_quote($pattern, '/') . '/i', $content)) {
                                $suspicious_files[] = $file->getPathname();
                                break;
                            }
                        }
                    }
                }
            }
        }

        return $suspicious_files;
    }
}