<?php
/**
 * Classe per la gestione del database
 *
 * @package DarkModeSwitcherUniversal
 * @since 1.0.0
 */

// Prevengo accesso diretto
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Classe IADS_Database
 * Gestisce tutte le operazioni database del plugin
 */
class IADS_Database {

    /**
     * Istanza singleton
     */
    private static $instance = null;

    /**
     * Versione database
     */
    private $db_version = '1.0';

    /**
     * Nome tabelle
     */
    private $tables = array();

    /**
     * Ottieni istanza singleton
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Costruttore privato per singleton
     */
    private function __construct() {
        $this->init();
    }

    /**
     * Inizializzazione
     */
    public function init() {
        global $wpdb;

        // Definisci nomi tabelle
        $this->tables = array(
            'user_preferences' => $wpdb->prefix . 'iads_user_preferences',
            'settings_history' => $wpdb->prefix . 'iads_settings_history',
            'analytics' => $wpdb->prefix . 'iads_analytics',
            'security_log' => $wpdb->prefix . 'iads_security_log'
        );

        // Hook per aggiornamenti database
        add_action('init', array($this, 'check_database_version'));
        add_action('iads_daily_cleanup', array($this, 'daily_cleanup'));

        // Pianifica pulizia automatica se non esiste
        if (!wp_next_scheduled('iads_daily_cleanup')) {
            wp_schedule_event(time(), 'daily', 'iads_daily_cleanup');
        }
    }

    /**
     * Crea tutte le tabelle necessarie
     */
    public function create_tables() {
        $this->create_user_preferences_table();
        $this->create_settings_history_table();
        $this->create_analytics_table();
        $this->create_security_log_table();

        // Salva versione database
        update_option('iads_db_version', $this->db_version);
    }

    /**
     * Controlla versione database e aggiorna se necessario
     */
    public function check_database_version() {
        $installed_version = get_option('iads_db_version', '0');

        if (version_compare($installed_version, $this->db_version, '<')) {
            $this->upgrade_database($installed_version);
        }
    }

    /**
     * Aggiorna database
     */
    private function upgrade_database($from_version) {
        // Backup delle tabelle prima dell'aggiornamento
        $this->backup_tables();

        // Esegui aggiornamenti incrementali
        if (version_compare($from_version, '1.0', '<')) {
            $this->create_tables();
        }

        // Future versioni...
        // if (version_compare($from_version, '1.1', '<')) {
        //     $this->upgrade_to_1_1();
        // }

        update_option('iads_db_version', $this->db_version);
    }

    /**
     * Crea tabella preferenze utente
     */
    private function create_user_preferences_table() {
        global $wpdb;

        $table_name = $this->tables['user_preferences'];
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE $table_name (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            user_id bigint(20) UNSIGNED NOT NULL,
            session_id varchar(255) DEFAULT NULL,
            theme_preference enum('light','dark','auto') NOT NULL DEFAULT 'light',
            last_changed datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            ip_address varchar(45),
            user_agent text,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY user_session (user_id, session_id),
            KEY theme_preference (theme_preference),
            KEY last_changed (last_changed),
            FOREIGN KEY (user_id) REFERENCES {$wpdb->users}(ID) ON DELETE CASCADE
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    /**
     * Crea tabella storia delle impostazioni
     */
    private function create_settings_history_table() {
        global $wpdb;

        $table_name = $this->tables['settings_history'];
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE $table_name (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            user_id bigint(20) UNSIGNED NOT NULL,
            action_type enum('save','reset','import','export') NOT NULL,
            old_settings longtext,
            new_settings longtext,
            ip_address varchar(45),
            user_agent text,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY user_id (user_id),
            KEY action_type (action_type),
            KEY created_at (created_at),
            FOREIGN KEY (user_id) REFERENCES {$wpdb->users}(ID) ON DELETE CASCADE
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    /**
     * Crea tabella analytics
     */
    private function create_analytics_table() {
        global $wpdb;

        $table_name = $this->tables['analytics'];
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE $table_name (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            event_type varchar(50) NOT NULL,
            event_data longtext,
            user_id bigint(20) UNSIGNED DEFAULT NULL,
            session_id varchar(255),
            page_url text,
            referrer text,
            ip_address varchar(45),
            user_agent text,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY event_type (event_type),
            KEY user_id (user_id),
            KEY session_id (session_id),
            KEY created_at (created_at)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    /**
     * Crea tabella log di sicurezza
     */
    private function create_security_log_table() {
        global $wpdb;

        $table_name = $this->tables['security_log'];
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE $table_name (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            event_type varchar(100) NOT NULL,
            severity enum('low','medium','high','critical') NOT NULL DEFAULT 'low',
            event_data longtext,
            user_id bigint(20) UNSIGNED DEFAULT NULL,
            ip_address varchar(45) NOT NULL,
            user_agent text,
            resolved tinyint(1) NOT NULL DEFAULT 0,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY event_type (event_type),
            KEY severity (severity),
            KEY user_id (user_id),
            KEY resolved (resolved),
            KEY created_at (created_at)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    /**
     * Salva preferenza tema utente
     */
    public function save_user_preference($user_id, $theme, $session_id = null) {
		global $wpdb;
		$table_name = $this->tables['user_preferences'];
		$current_time = current_time('mysql');

		// Ottieni dati aggiuntivi
		$ip_address = $this->get_client_ip();
		$user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '';

		// Inserisci o aggiorna preferenza
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
		$result = $wpdb->replace(
			$table_name,
			array(
				'user_id' => absint($user_id),
				'session_id' => sanitize_text_field($session_id),
				'theme_preference' => sanitize_text_field($theme),
				'last_changed' => $current_time,
				'ip_address' => $ip_address,
				'user_agent' => $user_agent,
				'updated_at' => $current_time
			),
			array('%d', '%s', '%s', '%s', '%s', '%s', '%s')
		);

		if ($result === false) {
			//error_log('IADS Database Error: ' . $wpdb->last_error);
			return false;
		}

		// Registra evento analytics
		$this->record_analytics_event('theme_change', array(
			'theme' => $theme,
			'user_id' => $user_id,
			'session_id' => $session_id
		));

		return true;
	}

    /**
     * Ottieni preferenza tema utente
     */
    // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	public function get_user_preference($user_id, $session_id = null) {
		global $wpdb;
		$table_name = $this->tables['user_preferences'];

		if ($user_id && $session_id) {
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
			$result = $wpdb->get_row($wpdb->prepare(
				"SELECT * FROM $table_name WHERE user_id = %d AND session_id = %s ORDER BY updated_at DESC LIMIT 1",
				$user_id,
				$session_id
			));
		} elseif ($user_id) {
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
			$result = $wpdb->get_row($wpdb->prepare(
				"SELECT * FROM $table_name WHERE user_id = %d ORDER BY updated_at DESC LIMIT 1",
				$user_id
			));
		} else {
			return null;
		}

		return $result;
	}

    /**
     * Salva storia delle impostazioni
     */
    public function save_settings_history($user_id, $action_type, $old_settings, $new_settings) {
        global $wpdb;

        $table_name = $this->tables['settings_history'];
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $result = $wpdb->insert(
            $table_name,
            array(
                'user_id' => absint($user_id),
                'action_type' => sanitize_text_field($action_type),
                'old_settings' => wp_json_encode($old_settings),
                'new_settings' => wp_json_encode($new_settings),
                'ip_address' => $this->get_client_ip(),
                'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '',
                'created_at' => current_time('mysql')
            ),
            array('%d', '%s', '%s', '%s', '%s', '%s', '%s')
        );

        return $result !== false;
    }

    /**
     * Ottieni storia delle impostazioni
     */
	public function get_settings_history($user_id = null, $limit = 50) {
		global $wpdb;
		$table_name = $this->tables['settings_history'];

		if ($user_id) {
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
			$results = $wpdb->get_results($wpdb->prepare(
				"SELECT * FROM $table_name WHERE user_id = %d ORDER BY created_at DESC LIMIT %d",
				$user_id,
				$limit
			));
		} else {
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
			$results = $wpdb->get_results($wpdb->prepare(
				"SELECT * FROM $table_name ORDER BY created_at DESC LIMIT %d",
				$limit
			));
		}

		return $results;
	}

    /**
     * Registra evento analytics
     */
    public function record_analytics_event($event_type, $event_data = array(), $user_id = null) {
        global $wpdb;

        $table_name = $this->tables['analytics'];

        // Ottieni dati aggiuntivi
        $session_id = $this->get_session_id();
        $page_url = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        $referrer = isset($_SERVER['HTTP_REFERER']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_REFERER'])) : '';
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
        $result = $wpdb->insert(
            $table_name,
            array(
                'event_type' => sanitize_text_field($event_type),
                'event_data' => wp_json_encode($event_data),
                'user_id' => $user_id ? absint($user_id) : get_current_user_id(),
                'session_id' => $session_id,
                'page_url' => $page_url,
                'referrer' => $referrer,
                'ip_address' => $this->get_client_ip(),
				'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '',
                'created_at' => current_time('mysql')
            ),
            array('%s', '%s', '%d', '%s', '%s', '%s', '%s', '%s', '%s')
        );

        return $result !== false;
    }

    /**
     * Ottieni statistiche analytics
     */
    public function get_analytics_stats($days = 30) {
        global $wpdb;

        $table_name = $this->tables['analytics'];
        $date_from = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));

        // Statistiche generali
        $stats = array();

        // Total eventi
        // // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
        $stats['total_events'] = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table_name WHERE created_at >= %s",
            $date_from
        ));

        // Eventi per tipo
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
        $stats['events_by_type'] = $wpdb->get_results($wpdb->prepare(
            "SELECT event_type, COUNT(*) as count 
             FROM $table_name 
             WHERE created_at >= %s 
             GROUP BY event_type 
             ORDER BY count DESC",
            $date_from
        ), ARRAY_A);

        // Utenti unici
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
        $stats['unique_users'] = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT user_id) FROM $table_name WHERE created_at >= %s AND user_id > 0",
            $date_from
        ));

        // Sessioni uniche
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
        $stats['unique_sessions'] = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT session_id) FROM $table_name WHERE created_at >= %s",
            $date_from
        ));

        // Preferenze tema
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
        $preferences_table = $this->tables['user_preferences'];
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
        $stats['theme_preferences'] = $wpdb->get_results(
            "SELECT theme_preference, COUNT(*) as count 
             FROM $preferences_table 
             GROUP BY theme_preference 
             ORDER BY count DESC",
            ARRAY_A
        );

        // Eventi giornalieri
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
        $stats['daily_events'] = $wpdb->get_results($wpdb->prepare(
            "SELECT DATE(created_at) as date, COUNT(*) as count 
             FROM $table_name 
             WHERE created_at >= %s 
             GROUP BY DATE(created_at) 
             ORDER BY date ASC",
            $date_from
        ), ARRAY_A);

        return $stats;
    }

    /**
     * Backup tabelle
     */
    public function backup_tables() {
        global $wpdb;

        $upload_dir = wp_upload_dir();
        $backup_dir = $upload_dir['basedir'] . '/iads-backups/';

        if (!file_exists($backup_dir)) {
            wp_mkdir_p($backup_dir);
        }

        $timestamp = gmdate('Y-m-d_H-i-s');
        $backup_file = $backup_dir . "iads_backup_{$timestamp}.sql";

        $sql_content = "-- IADS Plugin Database Backup\n";
        $sql_content .= "-- Date: " . gmdate('Y-m-d H:i:s') . "\n\n";

        foreach ($this->tables as $table_key => $table_name) {
            // Controlla se la tabella esiste
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
            if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") === $table_name) {
                // Struttura tabella
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange -- Reading table schema for backup purposes
				$create_table = $wpdb->get_row("SHOW CREATE TABLE $table_name", ARRAY_N);
                if ($create_table) {
                    $sql_content .= "DROP TABLE IF EXISTS `$table_name`;\n";
                    $sql_content .= $create_table[1] . ";\n\n";

                    // Dati tabella
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
                    $rows = $wpdb->get_results("SELECT * FROM $table_name", ARRAY_A);
                    if ($rows) {
                        foreach ($rows as $row) {
                            $values = array();
                            foreach ($row as $value) {
                                $values[] = is_null($value) ? 'NULL' : "'" . esc_sql($value) . "'";
                            }
                            $sql_content .= "INSERT INTO `$table_name` VALUES (" . implode(', ', $values) . ");\n";
                        }
                        $sql_content .= "\n";
                    }
                }
            }
        }

        file_put_contents($backup_file, $sql_content);

        return $backup_file;
    }

    /**
     * Ripristina da backup
     */
    public function restore_from_backup($backup_file) {
		if (!file_exists($backup_file)) {
			return false;
		}

		global $wpdb;
		$sql_content = file_get_contents($backup_file);
		$statements = explode(';', $sql_content);

		foreach ($statements as $statement) {
			$statement = trim($statement);
			if (!empty($statement) && !str_starts_with($statement, '--')) {
				// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared -- Static SQL from backup file
				$result = $wpdb->query($statement);
				if ($result === false) {
					//error_log('IADS Restore Error: ' . $wpdb->last_error);
					return false;
				}
			}
		}

		return true;
	}

    /**
     * Pulizia giornaliera automatica
     */
    public function daily_cleanup() {
        $this->cleanup_old_analytics(90); // 90 giorni
        $this->cleanup_old_security_logs(30); // 30 giorni
        $this->cleanup_old_preferences(365); // 1 anno
        $this->cleanup_old_settings_history(180); // 6 mesi
        $this->optimize_tables();
    }

    /**
     * Pulizia analytics vecchi
     */
    private function cleanup_old_analytics($days) {
        global $wpdb;

        $table_name = $this->tables['analytics'];
        $cutoff_date = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE FROM $table_name WHERE created_at < %s",
            $cutoff_date
        ));

        if ($deleted !== false) {
            //error_log("IADS: Cleanup - Deleted {$deleted} old analytics records");
        }
    }

    /**
     * Pulizia log sicurezza vecchi
     */
    private function cleanup_old_security_logs($days) {
        global $wpdb;

        $table_name = $this->tables['security_log'];
        $cutoff_date = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE FROM $table_name WHERE created_at < %s AND resolved = 1",
            $cutoff_date
        ));

        if ($deleted !== false) {
            //error_log("IADS: Cleanup - Deleted {$deleted} old security log records");
        }
    }

    /**
     * Pulizia preferenze vecchie
     */
    private function cleanup_old_preferences($days) {
        global $wpdb;

        $table_name = $this->tables['user_preferences'];
        $cutoff_date = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE FROM $table_name WHERE updated_at < %s AND user_id = 0",
            $cutoff_date
        ));

        if ($deleted !== false) {
            //error_log("IADS: Cleanup - Deleted {$deleted} old anonymous preferences");
        }
    }

    /**
     * Pulizia storia impostazioni vecchia
     */
    private function cleanup_old_settings_history($days) {
        global $wpdb;

        $table_name = $this->tables['settings_history'];
        $cutoff_date = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE FROM $table_name WHERE created_at < %s",
            $cutoff_date
        ));

        if ($deleted !== false) {
            //error_log("IADS: Cleanup - Deleted {$deleted} old settings history records");
        }
    }

    /**
     * Ottimizza tabelle
     */
    private function optimize_tables() {
        global $wpdb;

        foreach ($this->tables as $table_name) {
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
            if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") === $table_name) {
                $wpdb->query("OPTIMIZE TABLE $table_name");
            }
        }
    }

    /**
     * Ottieni IP client
     */
    private function get_client_ip() {
        $ip_headers = array(
            'HTTP_CF_CONNECTING_IP',
            'HTTP_X_REAL_IP',
            'HTTP_X_FORWARDED_FOR',
            'REMOTE_ADDR'
        );

        foreach ($ip_headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = trim(explode(',', isset($_SERVER[$header]))[0]);
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return $ip;
                }
            }
        }

        return isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '0.0.0.0';
    }

    /**
     * Ottieni ID sessione
     */
    private function get_session_id() {
        if (session_id()) {
            return session_id();
        }

        // Genera ID sessione personalizzato per guest
        return 'guest_' . substr(md5((isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '') . $this->get_client_ip()), 0, 12);
    }

    /**
     * Esporta dati per GDPR
     */
    public function export_user_data($user_id) {
        global $wpdb;

        $data = array();

        // Preferenze utente
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
        $table_name = $wpdb->prefix . 'iads_user_preferences';
        $preferences = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$table_name} WHERE user_id = %d",
            $user_id
        ), ARRAY_A);
        $data['preferences'] = $preferences;

        // Storia impostazioni
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
        $history = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$this->tables['settings_history']} WHERE user_id = %d",
            $user_id
        ), ARRAY_A);
        $data['settings_history'] = $history;

        // Analytics (anonimizzati)
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
        $analytics = $wpdb->get_results($wpdb->prepare(
            "SELECT event_type, event_data, created_at FROM {$this->tables['analytics']} WHERE user_id = %d",
            $user_id
        ), ARRAY_A);
        $data['analytics'] = $analytics;

        return $data;
    }

    /**
     * Cancella dati utente per GDPR
     */
    public function delete_user_data($user_id) {
        global $wpdb;

        $tables_to_clean = array(
            'user_preferences',
            'settings_history',
            'analytics'
        );

        foreach ($tables_to_clean as $table_key) {
            $table_name = $this->tables[$table_key];
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
            $wpdb->delete($table_name, array('user_id' => $user_id), array('%d'));
        }

        // Anonimizza log di sicurezza invece di cancellare
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
        $wpdb->update(
            $this->tables['security_log'],
            array('user_id' => 0),
            array('user_id' => $user_id),
            array('%d'),
            array('%d')
        );
    }

    /**
     * Controlla integrità database
     */
    public function check_database_integrity() {
        global $wpdb;

        $issues = array();

        foreach ($this->tables as $table_key => $table_name) {
            // Controlla se tabella esiste
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
            if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") !== $table_name) {
                $issues[] = "Tabella mancante: $table_name";
                continue;
            }

            // Controlla integrità tabella
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with no applicable WP cache
            $check = $wpdb->get_row("CHECK TABLE $table_name", ARRAY_A);
            if ($check && $check['Msg_type'] === 'error') {
                $issues[] = "Errore integrità tabella $table_name: " . $check['Msg_text'];
            }
        }

        return empty($issues) ? true : $issues;
    }
}