/**
 * Dark Mode Switcher Universal - Frontend JavaScript
 * Plugin sicuro e performante per WordPress
 */
(function($) {
    'use strict';

    /**
     * gestione del Dark Mode
     */
    class DarkModeSwitcher {
        constructor() {
            this.currentTheme = 'light';
            this.storageKey = 'iads_theme_preference';
            this.bodyElement = document.documentElement;
            this.options = iads_ajax.options || {};

            this.init();
        }

        /**
         * Inizializzo plugin
         */
        init() {
            // Caricamento tema salvato
            this.loadSavedTheme();

            // Applicazione immediata del tema per evitare flash
            this.applyTheme(this.currentTheme, false);

            // Binding eventi
            this.bindEvents();

            // Debug info (solo in development)
            this.debugInfo();
        }

        /**
         * Caricamento del tema salvato da localStorage
         */
        loadSavedTheme() {
            try {
                // Controllo supporto localStorage
                if (!this.isLocalStorageAvailable()) {
                    console.warn('IADS: localStorage non disponibile, usando default');
                    return;
                }

                const savedTheme = localStorage.getItem(this.storageKey);

                if (savedTheme && (savedTheme === 'dark' || savedTheme === 'light')) {
                    this.currentTheme = savedTheme;
                } else {
                    // Controllo preferenza di sistema
                    if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
                        this.currentTheme = 'dark';
                    }
                }
            } catch (error) {
                console.warn('IADS: Errore nel caricamento tema salvato:', error);
            }
        }

        /**
         * Controllo disponibilità localStorage
         */
        isLocalStorageAvailable() {
            try {
                const test = '__iads_test__';
                localStorage.setItem(test, 'test');
                localStorage.removeItem(test);
                return true;
            } catch (e) {
                return false;
            }
        }

        /**
         * Binding degli eventi
         */
        bindEvents() {
            // Event listener per il toggle
            $(document).on('click', '.iads-toggle-switch', (e) => {
                e.preventDefault();
                this.toggleTheme();
            });

            // Listener per cambio preferenze di sistema
            if (window.matchMedia) {
                const mediaQuery = window.matchMedia('(prefers-color-scheme: dark)');
                mediaQuery.addListener((e) => {
                    if (!localStorage.getItem(this.storageKey)) {
                        this.currentTheme = e.matches ? 'dark' : 'light';
                        this.applyTheme(this.currentTheme);
                    }
                });
            }

            // Listener per eventi personalizzati
            $(document).on('iads:theme-changed', (e, theme) => {
                this.currentTheme = theme;
                this.applyTheme(theme);
            });

            // Gestione visibilità pagina (per sincronizzazione cross-tab)
            $(document).on('visibilitychange', () => {
                if (!document.hidden) {
                    this.syncTheme();
                }
            });

            // Listener per storage events (sincronizzazione cross-tab)
            $(window).on('storage', (e) => {
                if (e.originalEvent.key === this.storageKey) {
                    const newTheme = e.originalEvent.newValue;
                    if (newTheme && newTheme !== this.currentTheme) {
                        this.currentTheme = newTheme;
                        this.applyTheme(newTheme);
                    }
                }
            });
        }

        /**
         * Toggle del tema
         */
        toggleTheme() {
            const newTheme = this.currentTheme === 'dark' ? 'light' : 'dark';
            this.setTheme(newTheme);
        }

        /**
         * Impostazione del tema
         */
        setTheme(theme) {
            if (theme !== 'dark' && theme !== 'light') {
                console.warn('IADS: Tema non valido:', theme);
                return;
            }

            this.currentTheme = theme;
            this.applyTheme(theme);
            this.saveTheme(theme);

            // Evento personalizzato per sviluppatori
            $(document).trigger('iads:theme-changed', [theme]);
        }

        /**
         * Applicazione del tema
         */
        applyTheme(theme, animate = true) {
            // Rimozione classi esistenti
            this.bodyElement.removeAttribute('data-theme');
            document.body.classList.remove('iads-light-mode', 'iads-dark-mode');

            // Applicazione nuovo tema
            this.bodyElement.setAttribute('data-theme', theme);
            document.body.classList.add(`iads-${theme}-mode`);

            // Update toggle UI
            this.updateToggleUI(theme);

            // Animazione se richiesta
            if (animate) {
                this.animateTransition();
            }

            // Aggiornamento meta theme-color per mobile
            this.updateMetaThemeColor(theme);

            // Callback per tema applicato
            this.onThemeApplied(theme);
        }

        /**
         * Aggiornamento UI del toggle
         */
        updateToggleUI(theme) {
            const $toggle = $('.iads-toggle-switch');
            const $label = $('.iads-toggle-label');

            if (theme === 'dark') {
                $toggle.addClass('active');
                $label.text($label.data('dark-label') || 'Dark Mode');
            } else {
                $toggle.removeClass('active');
                $label.text($label.data('light-label') || 'Light Mode');
            }
        }

        /**
         * Animazione di transizione
         */
        animateTransition() {
            document.body.style.transition = 'background-color 0.3s ease, color 0.3s ease';

            setTimeout(() => {
                document.body.style.transition = '';
            }, 300);
        }

        /**
         * Aggiornamento meta theme-color
         */
        updateMetaThemeColor(theme) {
            let metaThemeColor = document.querySelector('meta[name="theme-color"]');

            if (!metaThemeColor) {
                metaThemeColor = document.createElement('meta');
                metaThemeColor.name = 'theme-color';
                document.getElementsByTagName('head')[0].appendChild(metaThemeColor);
            }

            const color = theme === 'dark' ? '#1a1a1a' : '#ffffff';
            metaThemeColor.setAttribute('content', color);
        }

        /**
         * Salvataggio tema in localStorage
         */
        saveTheme(theme) {
            try {
                if (this.isLocalStorageAvailable()) {
                    localStorage.setItem(this.storageKey, theme);
                }
            } catch (error) {
                console.warn('IADS: Errore nel salvataggio tema:', error);
            }
        }

        /**
         * Sincronizzazione tema cross-tab
         */
        syncTheme() {
            try {
                if (this.isLocalStorageAvailable()) {
                    const savedTheme = localStorage.getItem(this.storageKey);
                    if (savedTheme && savedTheme !== this.currentTheme) {
                        this.currentTheme = savedTheme;
                        this.applyTheme(savedTheme);
                    }
                }
            } catch (error) {
                console.warn('IADS: Errore nella sincronizzazione tema:', error);
            }
        }

        /**
         * Callback quando tema è applicato
         */
        onThemeApplied(theme) {
            // Hook per sviluppatori
            if (typeof window.iadsOnThemeApplied === 'function') {
                window.iadsOnThemeApplied(theme);
            }

            // Analytics (se Google Analytics è presente)
            if (typeof gtag !== 'undefined') {
                gtag('event', 'theme_change', {
                    'theme_type': theme,
                    'event_category': 'user_preferences'
                });
            }
        }

        /**
         * API pubblica per sviluppatori
         */
        getTheme() {
            return this.currentTheme;
        }

        /**
         * Metodo per forzare un tema specifico
         */
        forceTheme(theme) {
            if (theme !== 'dark' && theme !== 'light') {
                console.warn('IADS: Tema non valido per forceTheme:', theme);
                return false;
            }
            this.setTheme(theme);
            return true;
        }

        /**
         * Reset alle impostazioni predefinite
         */
        resetToDefault() {
            try {
                if (this.isLocalStorageAvailable()) {
                    localStorage.removeItem(this.storageKey);
                }

                // Controllo preferenza di sistema
                const systemTheme = window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches ? 'dark' : 'light';
                this.setTheme(systemTheme);
            } catch (error) {
                console.warn('IADS: Errore nel reset:', error);
            }
        }
    }

    /**
     * Utility per la gestione delle donazioni PayPal
     */
    class DonationManager {
        constructor(paypalEmail) {
            this.paypalEmail = paypalEmail;
        }

        /**
         * Generazione URL per donazione PayPal
         */
        generateDonationUrl(amount = '5.00', currency = 'EUR') {
            if (!this.paypalEmail) {
                console.warn('IADS: Email PayPal non configurata');
                return '#';
            }

            const params = new URLSearchParams({
                'business': this.paypalEmail,
                'item_name': 'Dark Mode Switcher Plugin - Caffè per lo sviluppatore',
                'amount': amount,
                'currency_code': currency,
                'no_shipping': '1',
                'return': window.location.href,
                'cancel_return': window.location.href
            });

            return `https://www.paypal.com/donate/?${params.toString()}`;
        }

        /**
         * Apertura finestra donazione
         */
        openDonation(amount = '5.00') {
            const url = this.generateDonationUrl(amount);

            if (url === '#') return;

            // Apertura in nuova finestra
            const popup = window.open(
                url,
                'paypal_donation',
                'width=800,height=600,scrollbars=yes,resizable=yes'
            );

            // Focus sulla finestra se possibile
            if (popup) {
                popup.focus();
            }

            // Analytics
            if (typeof gtag !== 'undefined') {
                gtag('event', 'donation_click', {
                    'amount': amount,
                    'event_category': 'donation'
                });
            }
        }
    }

    /**
     * Inizializzazione quando DOM è pronto
     */
    $(document).ready(function() {
        // Controllo sicurezza - verifica che siamo in un contesto WordPress
        if (typeof iads_ajax === 'undefined') {
            console.warn('IADS: Variabili di configurazione mancanti');
            return;
        }

        // Inizializzazione Dark Mode Switcher
        window.DarkModeSwitcher = new DarkModeSwitcher();

        // Inizializzazione Donation Manager
        const paypalEmail = iads_ajax.options?.paypal_email;
        if (paypalEmail) {
            window.DonationManager = new DonationManager(paypalEmail);

            // Binding click donazione
            $(document).on('click', '.iads-donation-btn', function(e) {
                e.preventDefault();
                const amount = $(this).data('amount') || '5.00';
                window.DonationManager.openDonation(amount);
            });
        }

        // API globale per sviluppatori
        window.iadsAPI = {
            getTheme: () => window.DarkModeSwitcher.getTheme(),
            setTheme: (theme) => window.DarkModeSwitcher.forceTheme(theme),
            toggleTheme: () => window.DarkModeSwitcher.toggleTheme(),
            resetTheme: () => window.DarkModeSwitcher.resetToDefault(),

            // Event listeners personalizzati
            onThemeChange: (callback) => {
                $(document).on('iads:theme-changed', callback);
            }
        };

        // Supporto per framework/librerie comuni
        // Vue.js
        if (typeof window.Vue !== 'undefined') {
            window.Vue.prototype.$iads = window.iadsAPI;
        }

        // React (se presente)
        if (typeof window.React !== 'undefined') {
            window.React.iads = window.iadsAPI;
        }

        // Notifica di inizializzazione completata
        $(document).trigger('iads:initialized');

    });

    /**
     * Gestione errori globali per il plugin
     */
    window.addEventListener('error', function(e) {
        if (e.filename && e.filename.includes('dark-mode-switcher-universal')) {
            console.error('IADS Error:', e.message, e.filename, e.lineno);

            // Report errore (opzionale)
            if (typeof gtag !== 'undefined') {
                gtag('event', 'javascript_error', {
                    'error_message': e.message,
                    'error_filename': e.filename,
                    'error_line': e.lineno,
                    'event_category': 'plugin_error'
                });
            }
        }
    });

    /**
     * Supporto per temi WordPress specifici
     */
    const themeIntegrations = {
        // Twentytwentythree
        twentytwentythree: function() {
            $('[data-theme="dark"] .wp-site-blocks').css('background-color', '#1a1a1a');
        },

        // Astra
        astra: function() {
            $('[data-theme="dark"] .ast-container').css('background-color', 'transparent');
        },

        // GeneratePress
        generatepress: function() {
            $('[data-theme="dark"] .main-navigation').css('background-color', '#222222');
        }
    };

    // Applicazione automatica integrazioni tema
    $(document).on('iads:theme-changed', function(e, theme) {
        const bodyClasses = document.body.className.toLowerCase();

        Object.keys(themeIntegrations).forEach(themeName => {
            if (bodyClasses.includes(themeName)) {
                themeIntegrations[themeName]();
            }
        });
    });

    /**
     * Utility per performance
     */
    const performanceOptimizations = {
        // Throttle function per eventi frequenti
        throttle: function(func, limit) {
            let inThrottle;
            return function() {
                const args = arguments;
                const context = this;
                if (!inThrottle) {
                    func.apply(context, args);
                    inThrottle = true;
                    setTimeout(() => inThrottle = false, limit);
                }
            }
        },

        // Debounce function per input
        debounce: function(func, wait, immediate) {
            let timeout;
            return function() {
                const context = this, args = arguments;
                const later = function() {
                    timeout = null;
                    if (!immediate) func.apply(context, args);
                };
                const callNow = immediate && !timeout;
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
                if (callNow) func.apply(context, args);
            };
        }
    };

    // Esposizione utilities globalmente
    window.iadsUtils = performanceOptimizations;

})(jQuery);