<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
/** @var array $settings */
/** @var array $status */

use Imgspalat\Settings;

$masked_api_key = '';

if ( ! empty( $settings['api_key'] ) ) {
	$length         = strlen( $settings['api_key'] );
	$masked_api_key = str_repeat( '•', max( 0, $length - 4 ) ) . substr( $settings['api_key'], -4 );
} else {
	$masked_api_key = esc_html__( 'Not set', 'imgsmaller' );
}

$connection_status = ! empty( $settings['connected'] ) ? 'connected' : 'disconnected';
$connection_label  = $connection_status === 'connected' ? esc_html__( 'Connected', 'imgsmaller' ) : esc_html__( 'Not Connected', 'imgsmaller' );

$status_json    = esc_attr( wp_json_encode( $status ) );
$settings_json  = esc_attr( wp_json_encode( $settings ) );
$progress_percent   = isset( $status['progress_percent'] ) ? (int) $status['progress_percent'] : 0;
$progress_label     = sprintf(
	/* translators: %d: percent completed */
	esc_html__( '%d%% complete', 'imgsmaller' ),
	$progress_percent
);
$remaining_count    = isset( $status['remaining_count'] ) ? (int) $status['remaining_count'] : 0;
$remaining_label    = $remaining_count === 0
	? esc_html__( 'No images remaining', 'imgsmaller' )
	: sprintf(
		/* translators: %d: number of images remaining */
		_n( '%d image remaining', '%d images remaining', $remaining_count, 'imgsmaller' ),
		$remaining_count
	);
$is_paused          = ! empty( $status['is_paused'] );
$processing_count   = isset( $status['processing_count'] ) ? (int) $status['processing_count'] : 0;
$queued_count       = isset( $status['queued_count'] ) ? (int) $status['queued_count'] : 0;
$in_progress_count  = isset( $status['in_progress_count'] ) ? (int) $status['in_progress_count'] : 0;
$is_running         = $processing_count > 0 || $queued_count > 0 || $in_progress_count > 0;
?>

	<div class="wrap imgsmaller-dashboard" data-status="<?php echo esc_attr( $status_json ); ?>" data-settings="<?php echo esc_attr( $settings_json ); ?>">
	<h1><?php esc_html_e( 'ImgSmaller', 'imgsmaller' ); ?></h1>

	<?php if ( isset( $_GET['imgsmaller-updated'] ) ) : // phpcs:ignore WordPress.Security.NonceVerification.Recommended ?>
		<div class="notice notice-success is-dismissible">
			<p><?php esc_html_e( 'Settings saved successfully.', 'imgsmaller' ); ?></p>
		</div>
	<?php endif; ?>

	<?php if ( isset( $_GET['imgsmaller-restore'] ) && 'done' === sanitize_text_field( wp_unslash( (string) $_GET['imgsmaller-restore'] ) ) ) : // phpcs:ignore WordPress.Security.NonceVerification.Recommended ?>
		<div class="notice notice-info is-dismissible">
			<p>
				<?php
				$restored_count = isset( $_GET['restored'] ) ? absint( wp_unslash( $_GET['restored'] ) ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$restored_label = number_format_i18n( $restored_count );
				/* translators: %s: number of images restored */
				echo esc_html( sprintf( __( 'Restore complete. %s images were reverted to their originals.', 'imgsmaller' ), $restored_label ) );
				?>
			</p>
		</div>
	<?php endif; ?>

	<?php if ( isset( $_GET['imgsmaller-backup-error'] ) ) : // phpcs:ignore WordPress.Security.NonceVerification.Recommended ?>
		<div class="notice notice-error is-dismissible">
			<?php
			// Read, then sanitize display-only message coming from a controlled redirect arg.
			$imgsmaller_backup_error_param = isset( $_GET['imgsmaller-backup-error'] ) ? wp_unslash( $_GET['imgsmaller-backup-error'] ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
			$imgsmaller_backup_error = sanitize_text_field( rawurldecode( (string) $imgsmaller_backup_error_param ) );
			?>
			<p><?php echo esc_html( $imgsmaller_backup_error ); ?></p>
		</div>
	<?php endif; ?>

	<?php if ( isset( $_GET['imgsmaller-backup'] ) ) : // phpcs:ignore WordPress.Security.NonceVerification.Recommended ?>
	<?php $backup_status = sanitize_text_field( wp_unslash( (string) $_GET['imgsmaller-backup'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended ?>
		<?php if ( 'imported' === $backup_status ) : ?>
			<div class="notice notice-success is-dismissible">
				<p><?php esc_html_e( 'Backup archive imported successfully.', 'imgsmaller' ); ?></p>
			</div>
		<?php elseif ( 'disabled' === $backup_status ) : ?>
			<div class="notice notice-warning is-dismissible">
				<p><?php esc_html_e( 'Backups are disabled. Enable backups to keep originals for restore or export.', 'imgsmaller' ); ?></p>
			</div>
		<?php elseif ( 'deleted' === $backup_status ) : ?>
			<?php
			$deleted_files       = isset( $_GET['deleted-files'] ) ? absint( wp_unslash( $_GET['deleted-files'] ) ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$deleted_attachments = isset( $_GET['deleted-attachments'] ) ? absint( wp_unslash( $_GET['deleted-attachments'] ) ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			?>
			<div class="notice notice-warning is-dismissible">
				<p>
					<?php
					$deleted_files_label       = number_format_i18n( $deleted_files );
					$deleted_attachments_label = number_format_i18n( $deleted_attachments );
					/* translators: 1: number of files deleted, 2: number of attachments */
					echo esc_html( sprintf( __( 'Deleted %1$s backup files associated with %2$s attachments.', 'imgsmaller' ), $deleted_files_label, $deleted_attachments_label ) );
					?>
				</p>
			</div>
		<?php endif; ?>
	<?php endif; ?>

	<?php if ( isset( $_GET['imgsmaller-reset'] ) && 'done' === sanitize_text_field( wp_unslash( (string) $_GET['imgsmaller-reset'] ) ) ) : // phpcs:ignore WordPress.Security.NonceVerification.Recommended ?>
		<div class="notice notice-warning is-dismissible">
			<p><?php esc_html_e( 'Plugin has been reset to defaults. You may need to re-enter your API key and settings.', 'imgsmaller' ); ?></p>
		</div>
	<?php endif; ?>

	<!-- Compression Overview moved above Plans -->
	<h2><?php esc_html_e( 'Compression Overview', 'imgsmaller' ); ?></h2>
	<p class="description">
		<?php esc_html_e( 'This section provides an overview of your image compression status and history.', 'imgsmaller' ); ?>
	</p>
	<div class="imgsmaller-status-row" style="align-items:center; gap:20px;">
		<div class="status-card" style="display:flex; align-items:center; gap:16px;">
			<canvas class="js-scan-pie" width="220" height="220" style="max-width:100%;"></canvas>
			<div>
				<p><strong><?php esc_html_e( 'Total images:', 'imgsmaller' ); ?></strong> <span class="js-scan-total-images">0</span></p>
				<p><strong><?php esc_html_e( 'Optimized:', 'imgsmaller' ); ?></strong> <span class="js-scan-optimized">0</span></p>
				<p><strong><?php esc_html_e( 'Not optimized:', 'imgsmaller' ); ?></strong> <span class="js-scan-nonoptimized">0</span></p>
			</div>
		</div>
		<div class="status-card">
			<h3><?php esc_html_e( 'Storage Overview', 'imgsmaller' ); ?></h3>
			<p><strong><?php esc_html_e( 'Total size:', 'imgsmaller' ); ?></strong> <span class="js-scan-total-size">0 KB</span></p>
			<p><strong><?php esc_html_e( 'Estimated potential saving:', 'imgsmaller' ); ?></strong> <span class="js-scan-potential">0 KB</span></p>
			<p><strong><?php esc_html_e( 'You saved total:', 'imgsmaller' ); ?></strong> <span class="js-scan-saved">0 KB</span></p>
			<p class="description"><?php esc_html_e( 'Potential saving is an estimate based on file types; actual results vary.', 'imgsmaller' ); ?></p>
			<p style="margin-top:10px;">
				<button type="button" class="button button-secondary js-scan-media" data-default-label="<?php esc_attr_e( 'Check Now', 'imgsmaller' ); ?>" data-working-label="<?php esc_attr_e( 'Checking…', 'imgsmaller' ); ?>"><?php esc_html_e( 'Check Now', 'imgsmaller' ); ?></button>
			</p>
		</div>
	</div>

	<h2><?php esc_html_e( 'Failed Images (skipped after 3 retries)', 'imgsmaller' ); ?></h2>
	<p class="description"><?php esc_html_e( 'These images were skipped after failing multiple times. Review and restore originals or try again later.', 'imgsmaller' ); ?></p>
	<div class="imgsmaller-exclude-browser imgsmaller-failed-browser">
		<div class="imgsmaller-exclude-toolbar">
			<button type="button" class="button button-secondary js-failed-refresh"><?php esc_html_e( 'Refresh', 'imgsmaller' ); ?></button>
			<span style="flex:1 1 auto"></span>
			<button type="button" class="button button-secondary js-failed-load-more"><?php esc_html_e( 'Load more', 'imgsmaller' ); ?></button>
		</div>
		<div class="imgsmaller-exclude-grid js-failed-grid" tabindex="0" role="region" aria-label="<?php echo esc_attr__( 'Failed images list', 'imgsmaller' ); ?>">
			<!-- thumbnails injected by JS -->
		</div>
		<div class="imgsmaller-exclude-loading js-failed-loading"><?php esc_html_e( 'Loading…', 'imgsmaller' ); ?></div>
	</div>

	<!-- Plans moved to top -->
	<div class="imgsmaller-status-row">
		<div class="status-card" style="min-width:320px;">
			<h2><?php esc_html_e( 'Current Plan & Quota', 'imgsmaller' ); ?></h2>
			<div class="js-plan-box">
				<p class="js-plan-loading"><?php esc_html_e( 'Loading plan…', 'imgsmaller' ); ?></p>
				<div class="js-plan-content" style="display:none;">
					<p><strong><?php esc_html_e( 'Plan:', 'imgsmaller' ); ?></strong> <span class="js-plan-name">—</span></p>
					<p><strong><?php esc_html_e( 'Images Used:', 'imgsmaller' ); ?></strong> <span class="js-plan-usage">—</span> / <span class="js-plan-limit">—</span></p>
					<p><strong><?php esc_html_e( 'Billing Cycle:', 'imgsmaller' ); ?></strong> <span class="js-plan-renews">—</span></p>
					<p><strong><?php esc_html_e( 'Images Remaining:', 'imgsmaller' ); ?></strong> <span class="js-plan-remaining">—</span></p>
				</div>
			</div>
		</div>
		<div class="status-card" style="min-width:480px;">
			<h2><?php esc_html_e( 'Available Plans', 'imgsmaller' ); ?></h2>
			<p class="description"><?php esc_html_e( 'Compare plans and upgrade for higher limits or more allowed domains.', 'imgsmaller' ); ?></p>
			<div class="js-plans-list">
				<p class="js-plans-loading"><?php esc_html_e( 'Loading plans…', 'imgsmaller' ); ?></p>
			</div>
		</div>
	</div>



	<h2><?php esc_html_e( 'Compression Progress', 'imgsmaller' ); ?></h2>
	<div class="imgsmaller-progress-overview" aria-live="polite">
		<div class="imgsmaller-progress-meta">
			<span class="js-progress-label"><?php echo esc_html( $progress_label ); ?></span>
			<span class="js-progress-remaining"><?php echo esc_html( $remaining_label ); ?></span>
		</div>
		<div class="imgsmaller-progress-bar js-progress-bar" role="progressbar" aria-valuemin="0" aria-valuemax="100" aria-valuenow="<?php echo esc_attr( $progress_percent ); ?>" aria-valuetext="<?php echo esc_attr( $progress_label ); ?>">
			<div class="imgsmaller-progress-bar__fill js-progress-fill" style="width: <?php echo esc_attr( (string) $progress_percent ); ?>%;"></div>
		</div>
	</div>
	<div class="imgsmaller-progress-grid">
		<div class="status-card">
			<h3><?php esc_html_e( 'Total Images', 'imgsmaller' ); ?></h3>
			<p class="imgsmaller-metric js-metric" data-key="total_images"><?php echo esc_html( (string) ( $status['total_images'] ?? 0 ) ); ?></p>
		</div>
		<div class="status-card">
			<h3><?php esc_html_e( 'Fetched', 'imgsmaller' ); ?></h3>
			<p class="imgsmaller-metric js-metric" data-key="fetched_count"><?php echo esc_html( (string) ( $status['fetched_count'] ?? 0 ) ); ?></p>
		</div>
		<div class="status-card">
			<h3><?php esc_html_e( 'Sent to Server', 'imgsmaller' ); ?></h3>
			<p class="imgsmaller-metric js-metric" data-key="sent_count"><?php echo esc_html( (string) ( $status['sent_count'] ?? 0 ) ); ?></p>
		</div>
		<div class="status-card">
			<h3><?php esc_html_e( 'Compressed', 'imgsmaller' ); ?></h3>
			<p class="imgsmaller-metric js-metric" data-key="compressed_count"><?php echo esc_html( (string) ( $status['compressed_count'] ?? 0 ) ); ?></p>
		</div>
	</div>

	<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" class="imgsmaller-settings-form">
		<?php wp_nonce_field( 'imgsmaller_save_settings' ); ?>
		<input type="hidden" name="action" value="imgsmaller_save_settings" />
		<table class="form-table" role="presentation">
			<tbody>
			<tr>
				<th scope="row"><label for="imgsmaller-api-key"><?php esc_html_e( 'API Key', 'imgsmaller' ); ?></label></th>
				<td>
					<input type="password" id="imgsmaller-api-key" name="<?php echo esc_attr( Settings::OPTION_KEY ); ?>[api_key]" class="regular-text" value="<?php echo esc_attr( $settings['api_key'] ?? '' ); ?>" placeholder="<?php esc_attr_e( 'Enter your ImgSmaller API key', 'imgsmaller' ); ?>" autocomplete="off" />
					<p class="description" style="margin-top:6px;">
						<a href="https://imgsmaller.com/login" target="_blank" rel="noopener noreferrer" class="button-link">
							<?php esc_html_e( 'Get free API keys – login or register', 'imgsmaller' ); ?>
						</a>
					</p>
					<p>
						<button type="button" class="button button-primary js-test-connection" data-nonce="<?php echo esc_attr( wp_create_nonce( 'imgsmaller_dashboard' ) ); ?>" style="margin-right:8px;">
							<?php esc_html_e( 'Connect', 'imgsmaller' ); ?>
						</button>
						<span class="imgsmaller-status-badge imgsmaller-status-badge--<?php echo esc_attr( $connection_status ); ?> js-connection-status">
							<?php echo esc_html( $connection_label ); ?>
						</span>
					</p>
				</td>
			</tr>
			<tr>
				<th scope="row"><label for="imgsmaller-domain"><?php esc_html_e( 'Allowed Domain', 'imgsmaller' ); ?></label></th>
				<td>
					<div style="display:flex;gap:6px;align-items:center;max-width:520px;">
						<input type="text" id="imgsmaller-domain" class="regular-text js-domain-input" placeholder="example.com" />
						<button type="button" class="button button-secondary js-save-domain"><?php esc_html_e( 'Save Domain', 'imgsmaller' ); ?></button>
					</div>
					<p class="description" style="margin-top:6px;">
						<?php esc_html_e( 'Your domain will be sent as example.com (no https, www, or subdomain).', 'imgsmaller' ); ?>
					</p>
					<p class="js-domain-status" style="margin-top:6px;"></p>
					<p style="margin-top:6px;"><strong><?php esc_html_e( 'Allowed list:', 'imgsmaller' ); ?></strong> <span class="js-domain-list">—</span></p>
				</td>
			</tr>
			<tr>
				<th scope="row"><label for="imgsmaller-output-format"><?php esc_html_e( 'Output Format', 'imgsmaller' ); ?></label></th>
				<td>
					<select id="imgsmaller-output-format" name="<?php echo esc_attr( Settings::OPTION_KEY ); ?>[output_format]">
						<option value="webp" <?php selected( $settings['output_format'] ?? 'webp', 'webp' ); ?>><?php esc_html_e( 'WEBP (recommended)', 'imgsmaller' ); ?></option>
						<option value="avif" <?php selected( $settings['output_format'] ?? 'webp', 'avif' ); ?>><?php esc_html_e( 'AVIF (modern)', 'imgsmaller' ); ?></option>
						<option value="jpg" <?php selected( $settings['output_format'] ?? 'webp', 'jpg' ); ?>><?php esc_html_e( 'JPG', 'imgsmaller' ); ?></option>
						<option value="png" <?php selected( $settings['output_format'] ?? 'webp', 'png' ); ?>><?php esc_html_e( 'PNG', 'imgsmaller' ); ?></option>
						<option value="original" <?php selected( $settings['output_format'] ?? 'webp', 'original' ); ?>><?php esc_html_e( 'Original (keep current type)', 'imgsmaller' ); ?></option>
					</select>
					<p class="description"><?php esc_html_e( 'Default output format is WEBP. Choose AVIF for even better compression, JPG, PNG, or Original to keep the incoming file type.', 'imgsmaller' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row"><label for="imgsmaller-exclude-ids"><?php esc_html_e( 'Exclude Images', 'imgsmaller' ); ?></label></th>
				<td>
					<input type="text" id="imgsmaller-exclude-ids" name="<?php echo esc_attr( Settings::OPTION_KEY ); ?>[exclude_ids]" class="regular-text js-exclude-ids-input" value="<?php echo esc_attr( $settings['exclude_ids'] ?? '' ); ?>" placeholder="e.g. 123, 456 789" />
					<p class="description" style="margin-bottom:8px;">
						<?php esc_html_e( 'Optional: Enter attachment IDs to exclude from compression. Separate by commas or spaces. Or use the browser below to select images visually.', 'imgsmaller' ); ?>
					</p>
					<div class="imgsmaller-exclude-browser">
						<div class="imgsmaller-exclude-toolbar">
							<input type="search" class="regular-text js-exclude-search imgsmaller-exclude-search" placeholder="<?php echo esc_attr__( 'Search images…', 'imgsmaller' ); ?>" />
							<button type="button" class="button button-secondary js-exclude-refresh"><?php esc_html_e( 'Search', 'imgsmaller' ); ?></button>
							<select class="js-exclude-size-filter" style="min-width: 160px;">
								<option value="all"><?php esc_html_e( 'All sizes', 'imgsmaller' ); ?></option>
								<option value="small"><?php esc_html_e( 'Smaller (< 500KB)', 'imgsmaller' ); ?></option>
								<option value="medium"><?php esc_html_e( 'Medium (500KB–1MB)', 'imgsmaller' ); ?></option>
								<option value="large"><?php esc_html_e( 'Larger (> 1MB)', 'imgsmaller' ); ?></option>
							</select>
							<select class="js-exclude-type-filter" style="min-width: 140px;">
								<option value="all"><?php esc_html_e( 'All types', 'imgsmaller' ); ?></option>
								<option value="image/jpeg"><?php esc_html_e( 'JPEG', 'imgsmaller' ); ?></option>
								<option value="image/png"><?php esc_html_e( 'PNG', 'imgsmaller' ); ?></option>
								<option value="image/webp"><?php esc_html_e( 'WebP', 'imgsmaller' ); ?></option>
								<option value="image/avif"><?php esc_html_e( 'AVIF', 'imgsmaller' ); ?></option>
							</select>
							<span style="flex:1 1 auto"></span>
							<button type="button" class="button js-exclude-select-all"><?php esc_html_e( 'Select all on page', 'imgsmaller' ); ?></button>
							<button type="button" class="button js-exclude-clear"><?php esc_html_e( 'Clear selection', 'imgsmaller' ); ?></button>
						</div>
						<div class="imgsmaller-exclude-grid js-exclude-grid" tabindex="0" role="region" aria-label="<?php echo esc_attr__( 'Excluded images selector', 'imgsmaller' ); ?>">
							<!-- thumbnails injected by JS -->
						</div>
						<div class="imgsmaller-exclude-loading js-exclude-loading">
							<?php esc_html_e( 'Loading…', 'imgsmaller' ); ?>
						</div>
						<div class="imgsmaller-exclude-more">
							<button type="button" class="button button-secondary js-exclude-load-more"><?php esc_html_e( 'Load 50 more', 'imgsmaller' ); ?></button>
						</div>
					</div>
				</td>
			</tr>
			<tr>
				<th scope="row"><label for="imgsmaller-target-size"><?php esc_html_e( 'Target Size', 'imgsmaller' ); ?></label></th>
				<td>
					<input type="text" id="imgsmaller-target-size" name="<?php echo esc_attr( Settings::OPTION_KEY ); ?>[target_size]" class="regular-text" value="<?php echo esc_attr( $settings['target_size'] ?? 'auto' ); ?>" placeholder="auto" />
					<p class="description"><?php esc_html_e( 'Set to "auto" or e.g. 80KB.', 'imgsmaller' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row"><?php esc_html_e( 'Keep Backups', 'imgsmaller' ); ?></th>
				<td>
					<label for="imgsmaller-enable-backup">
						<input type="checkbox" id="imgsmaller-enable-backup" name="<?php echo esc_attr( Settings::OPTION_KEY ); ?>[enable_backup]" value="1" <?php checked( ! empty( $settings['enable_backup'] ) ); ?> />
						<?php esc_html_e( 'Store a copy of each original image before compression (recommended).', 'imgsmaller' ); ?>
					</label>
					<p class="description"><?php esc_html_e( 'Disable only if you have another backup strategy. Required for restoring originals or exporting backups.', 'imgsmaller' ); ?></p>
				</td>
			</tr>
			</tbody>
		</table>
		<?php submit_button( __( 'Save Settings', 'imgsmaller' ) ); ?>
	</form>

	<h2><?php esc_html_e( 'Actions', 'imgsmaller' ); ?></h2>
	<div class="imgsmaller-actions">
		<button type="button" class="button button-primary js-start-compression" data-nonce="<?php echo esc_attr( wp_create_nonce( 'imgsmaller_dashboard' ) ); ?>" <?php disabled( $is_paused || $is_running ); ?>>
			<?php esc_html_e( 'Start Compression', 'imgsmaller' ); ?>
		</button>
		<button type="button" class="button js-pause-compression" <?php disabled( $is_paused || ! $is_running ); ?>>
			<?php esc_html_e( 'Pause', 'imgsmaller' ); ?>
		</button>
		<button type="button" class="button js-resume-compression" <?php disabled( ! $is_paused ); ?>>
			<?php esc_html_e( 'Resume', 'imgsmaller' ); ?>
		</button>
		<button type="button" class="button button-secondary js-process-now" data-default-label="<?php esc_attr_e( 'Process Next Batch Now', 'imgsmaller' ); ?>">
			<?php esc_html_e( 'Process Next Batch Now', 'imgsmaller' ); ?>
		</button>
	</div>

	<h2><?php esc_html_e( 'Background Cron (optional but recommended)', 'imgsmaller' ); ?></h2>
	<p class="description">
		<?php esc_html_e( 'Some hosting environments stop WordPress cron when traffic is low. If your compression pauses unexpectedly, set up an external cron to ping this URL every 5-10 minutes.', 'imgsmaller' ); ?>
	</p>
	<?php
		$site_url   = site_url();
		$rest_base  = trailingslashit( rest_url( 'imgsmaller/v1/cron' ) );
		$cron_url   = add_query_arg( 'token', (string) ( $settings['cron_token'] ?? '' ), $rest_base );
	?>
	<div class="notice notice-info inline">
	<?php if ( isset( $_GET['imgsmaller-cron'] ) && 'regenerated' === sanitize_text_field( wp_unslash( (string) $_GET['imgsmaller-cron'] ) ) ) : // phpcs:ignore WordPress.Security.NonceVerification.Recommended ?>
			<div class="notice notice-success is-dismissible" style="margin: 10px 0;">
				<p><?php esc_html_e( 'Cron token regenerated. Your Cron URL has changed.', 'imgsmaller' ); ?></p>
			</div>
		<?php endif; ?>
		<p>
			<strong><?php esc_html_e( 'Cron URL:', 'imgsmaller' ); ?></strong>
			<code class="js-cron-url-text"><?php echo esc_html( $cron_url ); ?></code>
			<button type="button" class="button button-secondary js-copy-cron-url" style="margin-left: 8px;">
				<?php esc_html_e( 'Copy URL', 'imgsmaller' ); ?>
			</button>
		</p>
		<p>
			<?php esc_html_e( 'Use any external cron service to request this URL periodically. We recommend:', 'imgsmaller' ); ?>
			<a href="https://console.cron-job.org/login" target="_blank" rel="noopener noreferrer">console.cron-job.org</a>
			— <?php esc_html_e( "Don't worry, it is free.", 'imgsmaller' ); ?>
		</p>

		<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" class="imgsmaller-inline-form" style="margin-top: 8px;">
			<?php wp_nonce_field( 'imgsmaller_regenerate_cron_token' ); ?>
			<input type="hidden" name="action" value="imgsmaller_regenerate_cron_token" />
			<button type="submit" class="button button-secondary">
				<?php esc_html_e( 'Regenerate Cron Token', 'imgsmaller' ); ?>
			</button>
		</form>
	</div>


	<div class="imgsmaller-logs">
		<h2><?php esc_html_e( 'Recent Activity', 'imgsmaller' ); ?></h2>
		<ul class="js-activity-log">
			<?php if ( ! empty( $logs ) ) : ?>
				<?php foreach ( $logs as $log_entry ) : ?>
					<?php
					$timestamp = isset( $log_entry['time'] ) ? (int) $log_entry['time'] : time();
					$level     = isset( $log_entry['level'] ) ? sanitize_html_class( $log_entry['level'] ) : 'info';
					$message   = isset( $log_entry['message'] ) ? $log_entry['message'] : '';
					?>
					<li class="log-item log-item--<?php echo esc_attr( $level ); ?>">
						<strong><?php echo esc_html( wp_date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $timestamp ) ); ?></strong>
						<span><?php echo esc_html( $message ); ?></span>
					</li>
				<?php endforeach; ?>
			<?php else : ?>
				<li><?php esc_html_e( 'No activity logged yet.', 'imgsmaller' ); ?></li>
			<?php endif; ?>
		</ul>
	</div>
	</div>
